/* @(#)codegen.h	1.9 2/12/90 */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#ifndef _CODEGEN_H
#define _CODEGEN_H

#include "stacks.h"
#include "opcodes.h"

/* stores info about where a label is used */
typedef struct labeldef_s {
    int pos;	/* offset of label reference */
    bcode *ref;	/* operand of a branch to this label */
} LabelDef_t;

/* stores info about where the label is defined and used */
typedef struct label_s {
    List refs;	/* references to label (list of LabelDef_t's) */
    int def;	/* offset of label definition */
} Label_t;

/* optimizer info flags */
typedef struct opt_info_s {
    Boolean tail;	/* set if compiling a tail expression */
    Obj id;		/* id of definition */
    int numargs;	/* number of args for lambda */
    Boolean optarg;	/* whether lambda has optional arguments */
    Boolean constant;	/* whether value of id is constant */
    Label_t *top;	/* entry point of lambda */
    struct opt_info_s *prev;	/* enclosing lambda's opt info */
} optInfo_t;

/* assign default values to opt */
#define DEF_OPT(opt, par) ((opt).tail = FALSE,\
			    (opt).id = (Obj)NULL,\
			    (opt).numargs = 0,\
			    (opt).optarg = FALSE,\
			    (opt).constant = TRUE,\
			    (opt).top = (Label_t *)NULL,\
			    (opt).prev = (par))

/* stores info concerning code generation */
typedef struct codegen_s {
    List insts,		/* list of opcodes and operands */
	 refs,		/* list of Obj references */
	 labels;	/* list of labels */
    int len;		/* length of the code portion of the codevector */
} CodeGen_t, *CodeGen;

/* code-vector type */

/* instance struct */
typedef struct codevecInst_s {
    basicInst_t inherit;
    Obj locals, expr;
    bcode *vec, *refs;
} codevecInst_t, *codevecInst;

extern basicClass_t protoCode;
#define CodeVec ((Class)&protoCode)

extern Obj ccVec; /* constant codevec for call/cc */

#ifdef __STDC__

/* initialize codegen module */
extern void codeInit(void);

/* create a new label for code generator */
extern Label_t *labelNew(CodeGen gen);

/* mark where label is defined */
extern void labelDefine(Label_t *label, CodeGen gen);

/* resolve all labels */
extern void labelResolve(CodeGen gen);

/* Create a code generator
 *
 * cb is used to store info for the callback mechanism to free
 * the generator in case of an error
 */
extern CodeGen codeNew(Callback_t *cb);

/* Free code generator. */
extern void codeFree(CodeGen gen);

/* Convert gen into a codevector
 *  
 * exp is the lambda expression that was compiled
 * fp is a list of parameters and local variables in exp
 * gen is the code generator containing the instructions which are
 * to be placed in the codevector */
extern Obj newCodevec(F_OBJ alloc, CodeGen gen, Obj fp, Obj exp);

/* output inst to code generator */
extern void codeInst(opcode inst, CodeGen gen);

/* output integer operand to code generator */
extern void codeOp(int op, CodeGen gen);

/* output C function pointer operand to code generator */
extern void codeFcnOp(F_OBJ op, CodeGen gen);

/* output Obj operand to code generator */
extern void codeGCOp(Obj op, CodeGen gen);

/* output code to push obj on stack */
extern void codePush(Obj obj, CodeGen gen);

/* output code to pop obj from stack */
extern void codePop(CodeGen gen);

/* output a branch instruction to label to the generator */
extern void codeBranch(opcode branch, Label_t *label, CodeGen gen);

/* Compile expr and output the instructions to gen
 *
 * tail is TRUE if expr is the tail expression of a lambda body
 */
extern void compExpr(Obj expr, CodeGen gen, Boolean tail, optInfo_t *opt);

/* Print the instructions in vec with indentation of depth. */
extern void codevecPrint(Obj cvec, int depth);

#else

extern void codeInit();
extern Label_t *labelNew();
extern void labelDefine();
extern void labelResolve();
extern CodeGen codeNew();
extern void codeFree();
extern Obj newCodevec();
extern void codeInst();
extern void codeOp();
extern void codeFcnOp();
extern void codeGCOp();
extern void codePush();
extern void codePop();
extern void codeBranch();
extern void compExpr();
extern void codevecPrint();

#endif /* __STDC__ */

#endif /* _CODEGEN_H */
