(* The compiler entry points *)

#open "obj";;
#open "misc";;
#open "const";;
#open "lexer";;
#open "parser";;
#open "location";;
#open "syntax";;
#open "builtins";;
#open "hashtbl";;
#open "globals";;
#open "modules";;
#open "types";;
#open "ty_error";;
#open "typing";;
#open "ty_decl";;
#open "pr_decl";;
#open "ty_intf";;
#open "front";;
#open "instruct";;
#open "compiler";;
#open "lam";;
#open "emit_phr";;
#open "printexc";;
#open "prim";;

(* Compiling an implementation *)

let dump_impl_phrase (Impl(desc,loc)) =
  reset_type_expression_vars();
  begin match desc with
    Zexpr expr ->
      let ty = type_expression loc expr in
                  (dump_lambda false (translate_expression expr));
      if !verbose then print_expr ty
  | Zletdef(rec_flag, pat_expr_list) ->
      let env = type_letdef loc rec_flag pat_expr_list in
         (if rec_flag
          then dump_lambda true  (translate_letdef_rec loc pat_expr_list)
          else dump_lambda false (translate_letdef loc pat_expr_list));
      if !verbose then print_valdef env
  | Ztypedef decl ->
      let ty_decl = type_typedecl loc decl in
      if !verbose then print_typedecl ty_decl
  | Zexcdef decl ->
      let ex_decl = type_excdecl loc decl in
      if !verbose then print_excdecl ex_decl
  | Zimpldirective dir ->
      do_directive dir
  end
;;

let dump_imports opened = 
   do_list ( function (name, filename) ->
     print_string "(import "; 
     print_string name;
     print_string " \"";
     print_string filename; print_string "\")\n" )
   opened
;;
     
let dump_impl modname filename =
  let source_name = filename ^ ".ml"
  and obj_name = filename ^ ".bgl" in
  let ic = open_in_bin source_name
  (* The source file must be opened in binary mode, so that the absolute
     seeks in print_location work. The lexer ignores both \n and \r,
     so this is OK on the Mac and on the PC. *)
  in
  let lexbuf = lexing__create_lexer_channel ic in
    input_name := source_name;
    input_chan := ic;
    input_lexbuf := lexbuf;
    try
      while true do
        dump_impl_phrase (parse_impl_phrase lexbuf)
      done
    with End_of_file ->
      dump_imports (modules__get_ever_opened_modules ());
      close_in ic
    | x ->
      close_in ic;
      remove_file obj_name;
      raise x
;;

let dump_syntax_implementation modname filename =
  let f () = 
  external_types := [];
  if file_exists (filename ^ ".mli") then begin
    try
      let intf = read_module (filename ^ ".zi") in
      start_compiling_implementation modname intf;
      enter_interface_definitions intf;
      dump_impl modname filename;
      check_interface intf;
      if !write_extended_zi then begin
        let ext_intf_name = filename ^ ".zix" in
        let oc = open_out_bin ext_intf_name in
        try
          write_compiled_interface oc;
          close_out oc
        with x ->
          close_out oc;
          remove_file (ext_intf_name);
          raise x
      end;
      kill_module modname
    with x ->
      remove_file (filename ^ ".zo");
      raise x
  end else begin
    let intf_name = filename ^ ".zi" in
    let oc = open_out_bin intf_name in
    try
      start_compiling_interface modname;
      dump_impl modname filename;
      write_compiled_interface oc;
      close_out oc
    with x ->
      close_out oc;
      remove_file intf_name;
      raise x
  end
 in
  printexc__f f ()
;;

(* Compiling an interface *)

let mklib = ref false;;

let prims_to_dump = ref( [] : (string * string * int) list );;

let dump_intf_phrase (Intf(desc,loc)) =
  begin match desc with
    Zvaluedecl decl ->
      let dump_one_decl = function 
          (name , _, ValuePrim(_, Pccall(cname, arity))) -> 
            prims_to_dump := (name, cname, arity) :: !prims_to_dump
	| _ -> () in
      type_valuedecl loc decl;
      do_list dump_one_decl decl
  | Ztypedecl decl ->
      let ty_decl = type_typedecl loc decl in
      if !verbose then print_typedecl ty_decl
  | Zexcdecl decl ->
      let ex_decl = type_excdecl loc decl in
      if !verbose then print_excdecl ex_decl
  | Zintfdirective dir ->
      do_directive dir
  end
;;

let write_one_decl oc = function 
  (name, cname, arity) -> 
    output_string oc ("      (obj c-" ^ cname ^ " (");   
    for i = 0 to arity - 1 do output_string oc "obj " done;
    output_string oc (") \"" ^ cname ^ "\")\n")
;;
  
let write_dumped_interface modname filename = 
  if not !mklib then
    let dump_name = filename ^ ".sci" in
    let oc = open_out dump_name in
    try
      output_string oc "(module ";
      output_string oc ("__camli_" ^ filename);
      output_string oc "\n   (foreign\n";
      do_list (write_one_decl oc) !prims_to_dump;
      output_string oc "))\n\n";
      close_out oc
    with x -> 
      close_out oc;
      remove_file dump_name;
      raise x
;;

let dump_interface modname filename =
  let source_name = filename ^ ".mli"
  and intf_name = filename ^ ".zi" in
  let ic = open_in_bin source_name (* See compile_impl *)
  and oc = open_out_bin intf_name in
    try
      start_compiling_interface modname;
      let lexbuf = lexing__create_lexer_channel ic in
      input_name := source_name;
      input_chan := ic;
      input_lexbuf := lexbuf;
      external_types := [];
      while true do
        dump_intf_phrase(parse_intf_phrase lexbuf)
      done
    with End_of_file ->
      close_in ic;
      write_compiled_interface oc;
      write_dumped_interface modname filename;
      close_out oc
    | x ->
      close_in ic;
      close_out oc;
      remove_file intf_name;
      raise x
;;   

