
;;;; Records

;; Record  A = List Binding A
;; Binding A = Name * A

;; Names are compared using eq? and can be anything.

(define make-binding cons)
(define binding-name car)
(define binding-value cdr)
(define binding-find assq)

(define (binding-map f bindings)
  (map (lambda (binding)
	 (f (binding-name binding)
	    (binding-value binding)))
       bindings))

;;; Lists as sets

(define (list-intersection l1 l2)
  (cond 
   ((null? l1) '())
   ((memq (car l1) l2)
    (cons (car l1) (list-intersection (cdr l1) l2)))
   (else (list-intersection (cdr l1) l2))))

(define (list-difference l1 l2)
  (cond 
   ((null? l1) '())
   ((memq (car l1) l2)
    (list-difference (cdr l1) l2))
   (else (cons (car l1) (list-difference (cdr l1) l2)))))

(define (list-duplicates l)
  (cond ((null? l) '())
	((memq (car l) (cdr l))
	 (cons (car l)
	       (list-duplicates (delq (car l) (cdr l)))))
	(else
	 (list-duplicates (cdr l)))))

(define (check list message)
  (if (not (null? list))
      (error message list)))

(define duplicate-error "Duplicate names: ")
(define not-found-error "Not found: ")

;;; Record operators

;; The starred operators ignore missing bindings.

;;; empty-record, empty?, defines?, lookup, names, append

(define record-empty? null?)
(define make-record list)
(define record-map map)
(define record-for-each for-each)
(define record->list identity-procedure)

(define (empty-record)
  (quote ()))

(define (record-defines? name record)
  (not (null? (binding-find name record))))

(define (record-lookup name record)
  (binding-value (binding-find name record)))

(define (record-names record)
  (map binding-name record))

(define (record-values record)
  (map binding-value record))

;;; append, union

(define record-append append)

(define (record-union r1 r2)
  (check (list-intersection (record-names r1) (record-names r2))
	 duplicate-error)
  (record-append r1 r2))

;;; select

(define (record-select name record)
  (if (record-defines? name record)
      (record-lookup name record)
      (error not-found-error name)))

;;; delete, extract

(define (record-delete* names record)
  (list-transform-negative record
    (lambda (binding)
      (memq (binding-name binding) names))))

(define (record-extract* names record)
  (list-transform-positive record
    (lambda (binding)
      (memq (binding-name binding) names))))

(define (record-extract names record)
  (check (list-difference names (record-names record))
	 not-found-error)
  (record-extract* names record))

(define (record-delete names record)
  (check (list-difference names (record-names record))
	 not-found-error)
  (record-delete* names record))

;;; subtract, difference, symmetric-difference

(define (record-subtract r1 r2)
  (record-delete (record-names r2) r1))

(define (record-subtract* r1 r2)
  (record-delete* (record-names r2) r1))

(define record-difference record-subtract*)

(define (record-symmetric-difference r1 r2)
  (record-append (record-difference r1 r2)
		 (record-difference r2 r1)))

;;; update, layer

(define (record-update r1 r2)
  (record-append r1 (record-subtract r2 r1)))

(define (record-update* r1 r2)
  (record-append r1 (record-subtract* r2 r1)))

(define record-layer record-update*)

;;; extract-as, copy-as, rename, extract-as*, copy-as*, rename*
;;; Binding = New * Old
;;; rename can also extract

(define new-name binding-name)
(define old-name binding-value)
(define new-names record-names)
(define old-names record-values)

(define (invert-bindings bindings)
  (binding-map 
   (lambda (new old) (make-binding old new))
   bindings))

(define (record-extract-as-helper bindings record)
  (check (list-duplicates (new-names bindings)) duplicate-error)
  (binding-map
   (lambda (new old)
     (make-binding new (record-lookup old record)))
   bindings))

(define (record-extract-as bindings record)
  (check (list-difference (old-names bindings) (record-names record))
	 not-found-error)
  (record-extract-as-helper bindings record))

(define (record-extract-as* bindings record)
  (record-extract-as-helper 
   (list-transform-positive bindings
     (lambda (binding)
       (record-defines? (old-name binding) record)))
   record))

(define (record-copy-as bindings record)
  (record-union record (record-extract-as bindings record)))

(define (record-copy-as* bindings record)
  (record-union record (record-extract-as* bindings record)))

(define (record-rename bindings record)
  (record-union
   (record-extract-as bindings record)
   (record-delete (old-names bindings) record)))

(define (record-rename* bindings record)
  (record-union
   (record-extract-as* bindings record)
   (record-delete* (old-names bindings) record)))

