;;; -*- Base: 10; Mode: Scheme; Syntax: MIT Scheme; Package: USER -*-
;;
;; LOG-OPER.SCM
;;
;; June 29, 1991
;; Minghsun Liu
;;
;; This file contains some logical operations that are not provided by
;; MIT Scheme but implemented in CL.
;;
;;
;; The following(s) are(is) defined:
;;
;; *MAX-BIT-STRING-LENGTH*
;; (I-LIST->B-LIST ORIGINAL-LIST)
;; (LOGIOR N ...)
;; (LOGAND N ...)
;; (SI->BS N)
;; (LOGANDC2 N1 N2)
;; (INTEGER-LENGTH N)
;; (ASH N COUNT)
;; (LOGCOUNT N)
;;
(declare (usual-integrations))


;;
;; *MAX-BIT-STRING-LENGTH*
;;
;; determines the maximum length of bit-string to create when
;; converting from decimal representation of integers.
;;
(define *max-bit-string-length* 200)


;;
;; (I-LIST->B-LIST ORIGINAL-LIST)
;;
;; returns a list of bitstrings converted from the elements in the
;; list ORIGNAL-LIST which are signed integers.
;;
(define (i-list->b-list original-list)
  (map si->bs original-list))


;;
;; (LOGIOR N ..)
;;
;; returns the bitwise logical 'inclusive or' of its arguments.  0 is
;; the identity for this function.
;;
(define (logior #!rest args)
  (define (logior-aux aux-args)
    (if (null? (cdr aux-args))
        (car aux-args)
        (bit-string-or (car aux-args) (logior-aux (cdr aux-args)))))
  (if (null? args)
      0
      (bit-string->signed-integer (logior-aux (i-list->b-list
					       args)))))


;;
;; (LOGAND N ...)
;;
;; returns the bitwise logical `and' or its arguments.  -1 is the
;; identity of this function.
;;
(define (logand #!rest args)
  (define (logand-aux aux-args)
    (if (null? (cdr aux-args))
        (car aux-args)
        (bit-string-and (car aux-args) (logand-aux (cdr aux-args)))))
  (if (null? args)
      -1
      (bit-string->signed-integer (logand-aux (i-list->b-list
					       args)))))

;;
;; (SI->BS N)
;;
;; converts N into a newly allocated bit string of length
;; *max-bit-string-length*, a global variable.
;;
(define (si->bs n)
  (signed-integer->bit-string *max-bit-string-length* n))


;;
;; (LOGANDC2 N1 N2)
;;
;; returns the bitwise logical `and' of N1 and the complement of N2.
;;
(define (logandc2 b1 b2)
  (bit-string->signed-integer
   (bit-string-andc (si->bs b1) (si->bs b2))))


;;
;; (INTEGER-LENGTH N)
;;
;; get number of bits required to store the absolute magnitude of a
;; given integer N.
;;
(define (integer-length n)
  (let ((leng
	 (inexact->exact (ceiling (/ (log (if (< n 0)
					      (- n)
					      (1+ n)))
				     (log 2))))))
    (if (= (expt 2 (-1+ leng)) (1+ n))
	(-1+ leng)  ;; correction needed - a little fudging to fix the round-off error
	leng)))
	

;;
;; (ASH N COUNT)
;;
;; returns an integer representing the integer N shifted COUNT bits to
;; the left or right, depending if COUNT is positive or negative.  For
;; now, this is done using arithmetic to simulate the logical
;; operations and is expensive.  Further benchmark needed.
;;
(define (ash n count)
  (floor (* n (expt 2 count))))
	

;;
;; (LOGCOUNT N)
;;
;; counts the number of 1 or 0 bits in an integer, depending if the
;; integer is positive or negative.
;;
(define (logcount n)
  (let* ((count-one #t)
         (result 0)
         (bs-size (1+ (integer-length n)))
         (bin-rep (signed-integer->bit-string bs-size n)))
    (if (> 0 n)
        (set! count-one #f))
    (do ((i 0 (1+ i)))
        ((= i bs-size) result)
      (if (eq? (bit-string-ref bin-rep i) count-one)
          (set! result (1+ result))))))




