;;;; -*- Scheme -*-
;;;; $Header: /home/panda/pg/bevan/progs/scheme/misc/RCS/lit.scm,v 1.1 91/05/18 19:03:49 bevan Exp $

;;;+file-summary
;;; A Scheme implementation of some of the functions defined in :-
;;;   On the Power of List Iteration
;;;   M.J.C Gordon
;;;   The Computer Journal 22(4):376-379 1979
;;; Defines a list iterator function and then defines other (usually
;;; recursive) functions in terms of this it.
;;;-

;;; If you Scheme already has an atom? then use it instead of the 
;;; following
(define (lit:atom? x) (not (pair? x)))

;;;+fs
;;; This is the function wich is the subject of the paper.
;;; All other functions will be defined in terms of this one.
;;;-
(define (lit x y f)
  (if (lit:atom? x)
      y
      (f (car x) (lit (cdr x) y f))))

;;; Define sum of a list ... etc. in terms of lit.
(define (lit:sigma x) (lit x 0 +))
(define (lit:prod x) (lit x 1 *))

;;;; Some functions that operate on lists

;;;+fs
;;; return `x' appended to the front of `y'
;;;- +fe
;;; > (lit:append '(1 2 3) '(4 5 6))
;;; (1 2 3 4 5 6)
;;;-
(define (lit:append x y) (lit x y cons))

;;;+fs
;;; returns the reverse of `x'
;;;- +fe
;;; > (lit:rev '(1 2 3))
;;; (3 2 1)
;;;-
(define (lit:rev listToReverse)
  (lit listToReverse '() (lambda (y1 y2) (lit y2 (list y1) cons))))

;;;+fs
;;; returns true if `x' is a member of `y'
;;;- +fe
;;; > (lit:member 2 '(1 2 3))
;;; #t
;;;-
(define (lit:member elementToSearchFor listToSearchIn)
  (lit listToSearchIn #f (lambda (z1 z2)
			   (or (eq? z2 #t) (eq? z1 elementToSearchFor)))))

;;;; These next functions treat the given lists as sets.

;;;+fs
;;; return the a list representing the union of the lists (sets) `x' and `y'
;;;- +fe
;;; > (lit:union '(1 2 3) '(2 4 5))
;;; (1 3 2 4 5)
;;;-
(define (lit:union x y)
  (lit x y (lambda (z1 z2) (if (lit:member z1 y) z2 (cons z1 z2)))))

;;;+fs
;;; returns a list representing the intersection of the lists `x' and `y'
;;;- +fe
;;; > (lit:intersection '(1 2 3) '(1 3 4))
;;; (1 3)
;;;-
(define (lit:intersection x y)
  (lit x '() (lambda (z1 z2) (if (lit:member z1 y) (cons z1 z2) z2))))

;;;+fs
;;; returns a list representing the complement of `x' in `y'
;;;- +fe
;;; > (lit:complement '(1 3) '(1 2 3))
;;; (2)
;;;-
(define (lit:complement x y)
  (lit y '() (lambda (z1 z2) (if (lit:member z1 x) z2 (cons z1 z2)))))

;;;+fs
;;; return the powerset of the the set `x'
;;;- +fe
;;; > (lit:powerset '(1 2 3))
;;; (() (3) (2 3) (2) (1 2) (1 2 3) (1 3) (1))
;;;-
(define (lit:powerset x)
  (lit x (list '())
       (lambda (y1 y2)
	 (lit y2 y2 (lambda (z1 z2)
		      (lit:append z2 (cons (cons y1 z1) '())))))))

;;; this just creates a function with the original name that Gordon used
;;; for powerset
(define lit:boolean lit:powerset)

;;;+fs
;;;- +fe
;;; > (lit:cartesian-product '((1 2) (3 4) (5 6)))
;;; ((1 3 5) (1 3 6) (1 4 5) (1 4 6) (2 3 5) (2 3 6) (2 4 5) (2 4 6))
;;;-
(define (lit:cartesian-product x)
  (lit x (list '())
       (lambda (y1 y2)
	 (lit y1 '() (lambda (z1 z2)
		       (lit y2 z2 (lambda (w1 w2)
				    (cons (cons z1 w1) w2))))))))

;;; Some functions to simulate primitive recursive arithmetic

(define lit:zero (lambda (x) '()))
(define lit:succ (lambda (x) (cons '() x)))
;;; The following is as defined in the article.
;;; I think that it is wrong.
(define lit:pred
  (lambda (y)
    (cdr (lit y (list '()) (lambda (z1 z2)
			     (cons (cons z1 (car z2)) z2))))))
;;; why not just use
;(define lit:pred cdr)
