;;;; -*- Scheme -*-
;;;; $Header: /home/panda/pg/bevan/progs/scheme/misc/RCS/iterators.scm,v 1.1 91/05/26 11:32:26 bevan Exp $

;;;+file-summary
;;; Some of the code given in :-
;;;   Building Abstract Iterators Using Continuations
;;;   Christopher Coyle and Peter Grogono
;;;   SIGPLAN Noticies 26(2):17-24  February 1991
;;; This defines an interator interface to functions that iterate over
;;; structured objects such as lists and trees.
;;; All code should be vanilla Scheme.
;;; The code works on fine on scm (Aubry Jaffer's Scheme)
;;; Examples of how to use the iterators are given at the end of the file.
;;;-

;;; Define a structured type `iterator' and functions to set/access the various
;;; fields of the structure.
;;; You could replace this with some sort of define-structure if you Scheme has
;;; it.
(define (make-iterator) (make-vector 4))
(define (iterator-item iterator) (vector-ref iterator 0))
(define (iterator-item! iterator new-item) (vector-set! iterator 0 new-item))
(define (iterator-dead iterator) (vector-ref iterator 1))
(define (iterator-dead! iterator d) (vector-set! iterator 1 d))
(define (iterator-k1 iterator) (vector-ref iterator 2))
(define (iterator-k1! iterator new-k1) (vector-set! iterator 2 new-k1))
(define (iterator-k2 iterator) (vector-ref iterator 3))
(define (iterator-k2! iterator new-k2) (vector-set! iterator 3 new-k2))

;;;+fs
;;; Returns the current element in the iterator.
;;;-
(define iterator-current iterator-item)

;;;+fs
;;; Returns #t if the end of the object being iterated over has been reached
;;;-
(define iterator-finished? iterator-dead)

;;;+fs
;;; Move to the next element in the item being iterated over.
;;;-
(define iterator-next!
  (lambda (iterator)
    (iterator-k2! iterator
		  (call-with-current-continuation (iterator-k2 iterator)))
    iterator))

;;;+fs
;;; Create a new iterator to iterate over the `collection-to-iterate-over'
;;; using the `iterating-function'.
;;;-
(define iterator-init!
  (lambda (collection-to-iterate-over iterating-function)
    (let ((new-iterator (make-iterator)))
      (iterator-dead! new-iterator #f)
      (iterator-k2!
       new-iterator
       (call-with-current-continuation
	(lambda (k0)
	  (iterator-k1! new-iterator k0)
	  (iterating-function
	   collection-to-iterate-over
	   (lambda (x)
	     (iterator-item! new-iterator x)
	     (iterator-k1! new-iterator
			   (call-with-current-continuation (iterator-k1 new-iterator)))))
	  (iterator-dead! new-iterator #t)
	  ((iterator-k1 new-iterator) 'dead-iterator))))
      new-iterator)))

;;;+file-examples
;;; The rest of this file is examples

;;; A generic function to iterate over a list
;;; This is the standard recursive method for applying
;;; `function-to-apply-to-each-element' to each element in the
;;; `list-to-iterate-over'.
(define iterate-over-list
  (lambda (list-to-iterate-over function-to-apply-to-each-element)
    (if (null? list-to-iterate-over)
	'()
	(begin
	  (function-to-apply-to-each-element (car list-to-iterate-over))
	  (iterate-list (cdr list-to-iterate-over)
			function-to-apply-to-each-element)))))

;;; A generic `iterator' for lists.
(define list-iterator
  (lambda (list-to-iterate-over)
    (iterator-init! list-to-iterate-over iterate-over-list)))

;;; Tests if the two lists are equal.
(define list-equal?
  (lambda (list-a list-b)
    (let loop ((a-i (list-iterator list-a)) (b-i (list-iterator list-b)))
      (cond ((and (iterator-finished? a-i) (iterator-finished? b-i)) #t)
	    ((or (iterator-finished? a-i) (iterator-finished? b-i)) #f)
	    ((not (eq? (iterator-current a-i) (iterator-current b-i))) #f)
	    (else (loop (iterator-next! a-i) (iterator-next! b-i)))))))

;;; If you run this it should return a list '(#t #t #f #f)
(define (list-test)
  (let ((a '(1 2 3))
	(b '(1 2 3))
	(c '(3 4 5))
	(d '(2 3)))
    (list (list-equal? a b)
	  (list-equal? a a)
	  (list-equal? a c)
	  (list-equal? c d))))

;;;-file-examples
