;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;               This is the file COMPILE.SCM                     ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; COMPILER: DESUGARER

(define (desugar exp)
  (cond ((or (number? exp) (boolean? exp) (string? exp) (char? exp))
         exp)
        ((symbol? exp)
         exp)
        ((eq? (car exp) 'quote) exp)
        ((eq? (car exp) 'lambda)
         `(lambda ,(cadr exp) ,(desugar-body (cddr exp))))
        ((eq? (car exp) 'set!)
         `(set! ,(cadr exp) ,(desugar (caddr exp))))
        ((eq? (car exp) 'begin)
         (desugar-body (cdr exp)))
        ((eq? (car exp) 'if)
         (if (= (length exp) 3)
             `(if ,@(map desugar (cdr exp))
                  ',unspecified)
             `(if ,@(map desugar (cdr exp)))))
        ((eq? (car exp) 'letrec)
         `(letrec ,(map (lambda (spec) `(,(car spec) ,(desugar (cadr spec))))
                        (cadr exp))
            ,(desugar-body (cddr exp))))
        ((sugar? exp) (desugar (rewrite exp)))
        (else (map desugar exp))))

(define (desugar-body body)
  (let ((body (map desugar body)))
    (if (null? (cdr body))
        (car body)
        `(begin ,@body))))

(define (sugar? exp)
  (and (pair? exp)
       (member (car exp) '(and cond do let or list))))

(define (rewrite exp)
  (cond ((not (pair? exp)) exp)
        ((eq? (car exp) 'and)        (rewrite-and exp))
        ((eq? (car exp) 'cond)       (rewrite-cond exp))
        ((eq? (car exp) 'do)         (rewrite-do exp))
        ((eq? (car exp) 'let)        (rewrite-let exp))
        ((eq? (car exp) 'or)         (rewrite-or exp))
        ((eq? (car exp) 'list)       (rewrite-list exp))
        (else exp)))



(define (rewrite-and exp)
  (let ((conjuncts (cdr exp)))
    (cond ((null? conjuncts) `#t)
          ((null? (cdr conjuncts)) (car conjuncts))
          (else `(if ,(car conjuncts)
                     (and ,@(cdr conjuncts))
                     #f)))))

(define (rewrite-cond exp)
  (let ((clauses (cdr exp)))
    (cond ((null? clauses) `',unspecified)
          ((null? (cdar clauses))
           `(or ,(caar clauses)
                (cond ,@(cdr clauses))))
          ((eq? (caar clauses) 'else)
           `(begin ,@(cdar clauses)))
          (else `(if ,(caar clauses)
                     (begin ,@(cdar clauses))
                     (cond ,@(cdr clauses)))))))

(define (rewrite-let exp)
  (cond ((symbol? (cadr exp))
         (let ((tag (cadr exp))
               (bindings (caddr exp))
               (body (cdddr exp)))
           `(letrec ((,tag (lambda ,(map car bindings) ,@body)))
              (,tag ,@(map cadr bindings)))))
        (else
         (let ((bindings (cadr exp))
               (body (cddr exp)))
           `((lambda ,(map car bindings) ,@body)
             ,@(map cadr bindings))))))

(define (rewrite-or exp)
  (let ((disjuncts (cdr exp)))
    (cond ((null? disjuncts) `#f)
          ((null? (cdr disjuncts)) (car disjuncts))
          (else `(if ,(car disjuncts)
                     #t
                     (or ,@(cdr disjuncts)))))))

; In Scheme, LIST is supposed to be an n-ary procedure, but MPL Scheme
; doesn't have n-ary procedures, so we implement LIST as a macro.

(define (rewrite-list exp)
  (if (null? (cdr exp))
      ''()
      `(cons ,(cadr exp) (list ,@(cddr exp)))))




;;; COMPILER: CODE GENERATOR

(define (compile lambda-exp)
  (reset-label-counter)
  (assemble (generate-lambda-code (desugar lambda-exp)
                                  initial-c-t-env)))

(define (ctest lambda-exp)                ;test routine
  (generate-lambda-code (desugar lambda-exp) initial-c-t-env))

(define (generate-lambda-code exp c-t-env)
  (let ((formals (lambda-formals exp)))
    `((check-nargs ,(length formals))
      (make-environment ,(length formals))
      ,@(generate (caddr exp)
                  (c-t-bind formals c-t-env)
                  0
                  '((return))))))

(define (generate exp c-t-env depth continue-code)
  (cond ((variable? exp)
         (generate-variable exp c-t-env depth continue-code))
        ((literal? exp)
         (generate-literal exp continue-code))
        ((lambda? exp)
         (generate-lambda exp c-t-env depth continue-code))
        ((if? exp)
         (generate-if exp c-t-env depth continue-code))
        ((begin? exp)
         (generate-begin exp c-t-env depth continue-code))
        ((letrec? exp)
         (generate-letrec exp c-t-env depth continue-code))
        ((application? exp)
         (generate-application exp c-t-env depth continue-code))
        (else (error "unknown expression type" exp))))

; Constant

(define (generate-literal exp continue-code)
  `((load-constant (literal ,(literal-value exp)))
    ,@continue-code))

; Variable reference

(define (generate-variable var c-t-env depth continue-code)
  (let ((info (locate-variable var c-t-env)))
    (if (primitive? info)
        (generate (eta-expand var (primitive-nargs info))
                  initial-c-t-env
                  depth
                  continue-code)
        `((load-variable ,(env-access-back info) ,(env-access-over info))
          ,@continue-code))))

; LAMBDA

(define (generate-lambda exp c-t-env depth continue-code)
  `((make-procedure (code ,(generate-lambda-code exp c-t-env)))
    ,@continue-code))

; IF

(define (generate-if exp c-t-env depth continue-code)
  (let ((alt-label (generate-label 'else)))
    (generate (if-predicate exp)
              c-t-env
              depth
              `((jump-if-false ,alt-label)
                ,@(generate (if-consequent exp)
                            c-t-env
                            depth
                            ;; Never label a jump or return.
                            (if (or (jump-instruction? (car continue-code))
                                    (return-instruction? (car continue-code)))
                                `(,(car continue-code)
                                  ,alt-label
                                  ,@(generate (if-alternate exp)
                                              c-t-env
                                              depth
                                              continue-code))
                                (let ((continue-label
                                        (generate-label 'after-if)))
                                  `((jump ,continue-label)
                                    ,alt-label
                                    ,@(generate (if-alternate exp)
                                                c-t-env
                                                depth
                                                `(,continue-label
                                                  ,@continue-code))))))))))

; BEGIN

(define (generate-begin exp c-t-env depth continue-code)
  (let loop ((exp-list (begin-subexpressions exp)))
    (generate (car exp-list)
              c-t-env
              depth
              (if (null? (cdr exp-list))
                  continue-code
                  (loop (cdr exp-list))))))

; Code for LETREC:
; 1. Push a bunch of unspecifieds.
; 2. Make an environment.
; 3. Evaluate the right-hand sides, storing the results into the
;    environment.
; 4. Evaluate the body.
; 5. Reset environment to prior state.

(define (generate-letrec exp c-t-env depth continue-code)
  (let ((bindings (letrec-bindings exp))
        (body (letrec-body exp)))
    (let ((new-env (c-t-bind (map binding-lhs bindings) c-t-env)))
      (do ((bs bindings (cdr bs))
           (i 1 (+ i 1))
           (code (generate body
                           new-env
                           depth
                           (if (return-instruction? (car continue-code))
                               continue-code
                               `((leave-environment)
                                 ,@continue-code)))
                 (generate (binding-rhs (car bs))
                           new-env
                           depth
                           `((set-variable 0 ,i)
                             ,@code))))
          ((null? bs)
           (do ((bs bindings (cdr bs))
                (code `((make-environment ,(length bindings))
                        ,@code)
                      `((load-constant (literal ,unspecified))
                        (push)
                        ,@code)))
               ((null? bs) code)))))))

; Application

(define (generate-application exp c-t-env depth continue-code)
  (if (variable? (operator exp))
      (let ((info (locate-variable (operator exp) c-t-env)))
        (if (primitive? info)
            (generate-open-application info (operands exp)
                                       c-t-env depth continue-code)
            (generate-closed-application exp c-t-env depth continue-code)))
      (generate-closed-application exp c-t-env depth continue-code)))

(define (generate-open-application info arg-exps c-t-env depth continue-code)
  (if (not (= (length arg-exps) (primitive-nargs info)))
      (error "wrong number of arguments" (primitive-opcode info) arg-exps))
  (let ((call-code `((,(primitive-opcode info))
                     ,@continue-code)))
    (if (null? arg-exps)
        call-code
        (generate-pushes (cdr arg-exps)
                         c-t-env
                         depth
                         (generate (car arg-exps)
                                   c-t-env
                                   (+ depth (length (cdr arg-exps)))
                                   call-code)))))

(define (generate-closed-application exp c-t-env depth continue-code)
  (let ((fun-exp (operator exp))
        (arg-exps (operands exp)))
    (let ((nargs (length arg-exps)))
      (let ((do-it (lambda (call-code)
                     (generate-pushes arg-exps
                                      c-t-env
                                      depth
                                      (generate fun-exp
                                                c-t-env
                                                (+ depth nargs)
                                                call-code)))))
        (cond ((return-instruction? (car continue-code))
               ;; Handle tail recursion
               (do-it `((call ,nargs)
                        ,@(cdr continue-code))))
              ((jump-instruction? (car continue-code))
               ;; Avoid generating a jump to a jump
               `((make-continuation ,(jump-instruction-target
                                       (car continue-code))
                                    ,depth)
                 ,@(do-it `((call ,nargs)
                            ,@(cdr continue-code)))))
              (else
               (let ((return-label (generate-label 'return)))
                 `((make-continuation ,return-label ,depth)
                   ,@(do-it `((call ,nargs)
                              ,return-label
                              ,@continue-code))))))))))

; Push values of expressions so that the first expression's value is
; the last to be pushed.

(define (generate-pushes exp-list c-t-env depth continue-code)
  (if (null? exp-list)
      continue-code
      (generate-pushes (cdr exp-list)
                       c-t-env
                       depth
                       (generate (car exp-list)
                                 c-t-env
                                 (+ depth (length (cdr exp-list)))
                                 `((push) ,@continue-code)))))




;;; COMPILER: ASSEMBLER

; An instruction stream is a list of items.  Each item is either a
; label definition or an instruction.  A label definition is simply a
; label.  An instruction is a list whose car is an opcode and whose
; cdr is a list of operands.

; This is a typical two-pass assembler.  The first pass figures out the
; code offsets for labels; the second pass actually builds the code vector.

(define (assemble code)
  (resolve-labels code
                  (lambda (size labels)
                    (really-assemble code size labels))))

(define (resolve-labels code k)
  (let loop ((offset 0)
             (code code)
             (labels '()))
    (if (null? code)
        (k offset labels)
        (let ((item (car code)) (code (cdr code)))
          (if (pair? item)
              (loop (+ offset (length item)) code labels)
              (loop offset code (cons (list item offset) labels)))))))

(define (really-assemble code size labels)
  (let ((code-vector (make-vector size)))
    (let loop ((offset 0)
               (code code))
      (if (null? code)
          code-vector
          (let ((item (car code)) (code (cdr code)))
            (if (pair? item)
                (begin (vector-set! code-vector
                                    offset
                                    (opcode (car item)))
                       (assemble-operands (cdr item)
                                          (+ offset 1)
                                          labels
                                          code-vector)
                       (loop (+ offset (length item)) code))
                (loop offset code)))))))



(define (assemble-operands operands offset labels code-vector)
  (cond ((not (null? operands))
         (vector-set! code-vector
                      offset
                      (assemble-operand (car operands) labels))
         (assemble-operands (cdr operands)
                            (+ offset 1)
                            labels
                            code-vector))))

(define (assemble-operand operand labels)
  (cond ((symbol? operand)
         (let ((probe (assoc operand labels)))
           (if probe
               (cadr probe)
               (error "undefined label" operand))))
        ((not (pair? operand)) operand)
        ((eq? (car operand) 'literal)
         (cadr operand))
        ((eq? (car operand) 'code)
         (assemble (cadr operand)))
        (else
         (error "illegal operand syntax" operand))))





;;; COMPILER: ENVIRONMENTS

; Compile time environments

(define (c-t-bind vars c-t-env)
  (lambda (var back)
    (let loop ((i 1)
               (vars vars))
      (cond ((null? vars)
             (c-t-env var (+ back 1)))
            ((same-variable? var (car vars))
             (make-env-access back i))
            (else
             (loop (+ i 1) (cdr vars)))))))

(define (locate-variable var c-t-env)
  (c-t-env var 0))

(define initial-c-t-env
  (lambda (var back)
    ;; primitive-opcodes is a list of (name nargs)
    (let ((maybe-primitive (assq var primitive-opcodes)))
      (if maybe-primitive
          (make-primitive (car maybe-primitive) (cadr maybe-primitive))
          (make-env-access back (global-variable-index var))))))

(define *global-variables* '())

(define (global-variable-index var)
  (let loop ((l *global-variables*) (i 0))
    (cond ((null? l)
           (set! *global-variables* (append *global-variables* (list var)))
           i)
          ((same-variable? var (car l)) i)
          (else (loop (cdr l) (+ i 1))))))

(define (make-env-access back over) (list 'env-access back over))
(define (env-access? info) (eq? (car info) 'env-access))
(define env-access-back cadr)
(define env-access-over caddr)

(define (make-primitive name nargs) (list 'primitive name nargs))
(define (primitive? info) (eq? (car info) 'primitive))
(define primitive-opcode cadr)
(define primitive-nargs caddr)




;;; COMPILER: UTILITIES

(define (eta-expand exp nargs)
  (let ((vars (do ((some-vars '(a b c d e f) (cdr some-vars)) ;kludge
                   (vars '() (cons (car some-vars) vars))
                   (i nargs (- i 1)))
                  ((<= i 0) (reverse vars)))))
    `(lambda ,vars
       (,exp ,@vars))))

(define (return-instruction? instruction)
  (and (pair? instruction)
       (eq? (car instruction) 'return)))

(define (jump-instruction? instruction)
  (and (pair? instruction)
       (eq? (car instruction) 'jump)))

(define jump-instruction-target cadr)

(define *label* 0)

(define (reset-label-counter)
  (set! *label* 0))

(define (generate-label prefix)
  (set! *label* (+ *label* 1))
  (string->symbol (string-append (symbol->string prefix)
				 "-"
				 (number->string *label*)))) ;;; '(heur)))))


;;; COMPILER: EXPRESSION ABSTRACTION

(define special-form-predicate
  (lambda (keyword)
    (lambda (exp)
      (and (pair? exp)
           (eq? (car exp) keyword)))))

(define keyword?
  (lambda (x)
    (member x '(quote lambda if begin letrec define))))

(define literal?
  (lambda (exp)
    (or (number? exp)
        (boolean? exp)
        (quotation? exp))))

(define literal-value
  (lambda (exp)
    (cond ((quotation? exp) (cadr exp))
          ;; Hack to distinguish #F from () -- look at COPY-TO-HEAP
          ((eq? exp #f) false)
          ((eq? exp #t) true)
          (else exp))))

(define quotation? (special-form-predicate 'quote))

(define variable?
  (lambda (exp)
    (and (symbol? exp)
         (not (keyword? exp)))))

(define same-variable? eq?)

(define lambda? (special-form-predicate 'lambda))
(define lambda-formals cadr)
(define lambda-body caddr)

(define application?
  (lambda (exp)
    (and (pair? exp)
         (not (keyword? (car exp))))))

(define operator car)
(define operands cdr)

(define if? (special-form-predicate 'if))
(define if-predicate cadr)
(define if-consequent caddr)
(define if-alternate cadddr)

(define begin? (special-form-predicate 'begin))
(define begin-subexpressions cdr)

(define letrec? (special-form-predicate 'letrec))
(define letrec-bindings cadr)
(define letrec-body caddr)

(define binding-lhs car)
(define binding-rhs cadr)

(define definition? (special-form-predicate 'define))
(define (definition-lhs form)
  (let ((pattern (cadr form)))
    (if (pair? pattern) (car pattern) pattern)))
(define (definition-rhs form)
  (let ((pattern (cadr form)))
    (if (pair? pattern)
        `(lambda ,(cdr pattern) ,@(cddr form))
        (caddr form))))
