;;;;
;;;; $Id: special.scm,v 1.12 1992/11/12 12:27:40 pk Exp $
;;;;
;;;; sclint -- a Scheme lint.
;;;;
;;;; Pertti Kellom\"aki, 1992
;;;;
;;;; $Log: special.scm,v $
;;;; Revision 1.12  1992/11/12  12:27:40  pk
;;;; Complains about unused variables.
;;;;
;;;; Revision 1.11  1992/10/30  13:14:08  pk
;;;; PC-Scheme does not like make-environment and sequence as variable
;;;; names. Changed to make-env and seq.
;;;;
;;;; Revision 1.10  1992/10/30  07:35:12  pk
;;;; Checks let*.
;;;;
;;;; Revision 1.9  1992/10/23  13:39:39  pk
;;;; Attach information about a procedure to the variable that names it.
;;;;
;;;; Revision 1.8  1992/10/22  12:57:06  pk
;;;; Fixed a procedure call with wrong number of args. sclint caught this by itself!
;;;; Added support for named let.
;;;;
;;;; Revision 1.7  1992/10/17  11:24:02  pk
;;;; Added checking of an implicit begin.
;;;; Checking of case.
;;;; Checking of cond.
;;;; Corrections of primitive procedure profiles.
;;;;
;;;; Revision 1.6  1992/09/22  08:54:18  pk
;;;; Checks let bindings.
;;;;
;;;; Revision 1.5  1992/09/22  07:37:00  pk
;;;; Checks lambda's.
;;;;
;;;; Revision 1.4  1992/09/22  07:22:52  pk
;;;; Adds parameters of (define (foo x y) ...) to environment.
;;;; Checks set!:s to global variables and malformed set! forms.
;;;;
;;;; Revision 1.3  1992/09/03  13:58:56  pk
;;;; Now pass around environment info. Does not add local variable to it yet.
;;;;
;;;; Revision 1.2  1992/09/02  09:17:50  pk
;;;; Checking of and and or.
;;;;
;;;; Revision 1.1  1992/08/31  12:52:51  pk
;;;; Initial revision
;;;;
;;;;

;;;;
;;;; These procedures check that special forms are well formed.
;;;;

;;;
;;; Dispatch on various special forms.
;;;

(define (check-special-form form environment)
  (case (pexp->sexp (psd-car form))
    ((and) (check-and form environment))
    ((begin) (check-begin form environment))
    ((case) (check-case form environment))
    ((cond) (check-cond form environment))
    ((define) (check-define form environment))
    ((delay) (check-delay form environment))
    ((do) (check-do form environment))
    ((else) (check-else form environment))
    ((if) (check-if form environment))
    ((lambda) (check-lambda form environment))
    ((let) (check-let form environment))
    ((let*) (check-let* form environment))
    ((letrec) (check-letrec form environment))
    ((or) (check-or form environment))
    ((quasiquote) (check-quasiquote form environment))
    ((quote) #f)
    ((set!) (check-set! form environment))
    ((unquote) (check-unquote form environment))
    ((unquote-splicing) (check-unquote-splicing form environment))))

;;;
;;; Definitions.
;;;

(define (check-define form environment)
  (cond ((match '(define) form)
	 (warning form "Missing variable and expression in a define form."))
	((match '(define symbol) form)
	 (warning form "Missing expression in a define form."
		  " This may work, but it is nonportable."))
	((match '(define symbol expr) form)
	 (check-argument-counts (caddr (match-result)) environment))
	((match '(define symbol expr expr expr*) form)
	 (warning form "Too many subexpressions in a define form."))
	((match '(define pair) form)
	 (warning form "Missing procedure body."))
	((match '(define pair expr*) form)
	 (let* ((params (check-parameter-list (psd-cdr (cadr (match-result)))))
		(new-environment (make-env environment
						   params)))
	   (check-sequence (caddr (match-result))
			   new-environment)
	   (map (lambda (var)
		  (if (variable-references var)
		      #f
		      (warning (variable-source-name var)
			       "Variable "
			       (variable-name var)
			       " never used.")))
		params)))
	(else
	 (warning form "Wrong number of subexpressions in define form."))))

;;;
;;; Check that a parameter list is well formed and return the parameters as local variables.
;;;

(define (check-parameter-list params)
  (cond ((psd-null? params) '())
	((psd-symbol? params) (list (make-variable params 'local)))
	((psd-pair? params)
	 (cond ((psd-symbol? (psd-car params))
		(cons (make-variable (psd-car params) 'local)
		      (check-parameter-list (psd-cdr params))))
	       (else
		(warning (psd-car params) "Formal parameter must be a symbol.")
		(check-parameter-list (psd-cdr params)))))
	(else
	 (warning params "Formal parameter must be a symbol.")
	 '())))

;;;
;;; If.
;;;

(define (check-if form environment)
  (check-sequence (psd-cdr form) environment)
  (cond ((match '(if expr expr) form)
	 (warning form "Missing else clause in if. This is nonportable."))
	((match '(if expr expr expr) form))
	(else
	 (warning form "Wrong number of subexpressions in if."))))

;;;
;;; and
;;;

(define (check-and form environment)
  (if (match '(and) form)
      (warning form "No subexpressions in and form. You could replace this by #t.")
      #f)
  (check-sequence (psd-cdr form) environment))

;;;
;;; begin
;;;

(define (check-begin form environment)
  (check-sequence (psd-cdr form) environment))

;;;
;;; Check an expression sequence as in begin, let-body etc.
;;;

(define (check-sequence seq environment)

  ;; collect the internal defines
  (define (internal-defines seq)
    (cond ((psd-null? seq)
	   '())
	  ((and (psd-pair? seq)
		(match '(define symbol expr*) (psd-car seq)))
	   (cons (make-variable (cadr (match-result))
				'local)
		 (internal-defines (psd-cdr seq))))
	  ((and (psd-pair? seq)
		(match '(define (symbol expr*) expr*) (psd-car seq)))
	   (let ((this-proc (make-variable (caadr (match-result))
				'local)))
	     (variable-set-parameter-info! this-proc
					   (parameter-info
					    (psd-cdr (psd-cadr
						      (psd-car seq)))))
	     (cons this-proc
		   (internal-defines (psd-cdr seq)))))
	  ((psd-pair? seq)
	   (internal-defines (psd-cdr seq)))
	  (else '())))
    
  ;; do the actual checking
  (define (check-seq seq environment)
    (cond ((psd-null? seq)
	   #f)
	  ((not (psd-pair? seq))
	   (warning seq "This shoud be a list of expressions."))
	  (else
	   (check-argument-counts (psd-car seq) environment)
	   (check-seq (psd-cdr seq) environment))))

  (check-seq seq (make-env environment
					(internal-defines seq))))

;;;
;;; Check how many parameters this procedure accepts, and wether it
;;; accepts a rest argument. The information is returned as a list of
;;; (param-count rest?)
;;;

(define (parameter-info params)
  (let loop ((params params)
	     (count 0))
    (cond ((psd-null? params) (list count #f))
	  ((psd-symbol? params) (list count #t))
	  ((psd-pair? params)
	   (loop (psd-cdr params)
		 (+ count 1)))
	  (else
	   (list count #f)))))  

;;;
;;; case
;;;

(define (check-case form environment)

  (define (check-case-clause clause)
    (cond ((not (psd-pair? clause))
	   (warning clause "Badly formed case clause."))
	  ((match '((quote expr) expr*) clause)
	   (warning clause
		    "This clause is very likely wrong. The first part of a clause should be "
		    "a list of literals, not a quoted literal.")
	   (check-sequence (psd-cdr clause) environment))
	  ((match '(else expr expr*) clause)
	   (check-sequence (psd-cdr clause) environment))
	  ((not (psd-pair? (psd-car clause)))
	   (warning clause "The first part of a clause should be a list of literals.")
	   (check-sequence (psd-cdr clause) environment))
	  (else (check-sequence (psd-cdr clause) environment))))
		    

  ;; check the key
  (if (psd-pair? (psd-cdr form))
      (check-argument-counts (psd-cadr form) environment)
      #f)

  ;; and the rest of the form
  (cond ((match '(case expr) form)
	 (warning form "No missing clauses in a case form."))
	(else
	 (let loop ((clauses (psd-cddr form)))
	   (cond ((psd-null? clauses)
		  #f)
		 ((psd-pair? clauses)
		  (if (and (psd-null? (psd-cdr clauses))
			   (not (match '(else expr*) (psd-car clauses))))
		      (warning clauses "Missing else part in case.")
		      #f)
		  (check-case-clause (psd-car clauses))
		  (loop (psd-cdr clauses)))
		 (else
		  (warning form "Badly formed case.")))))))


;;;
;;; cond
;;;

(define (check-cond form environment)

  ;; check one clause in cond
  (define (check-cond-clause clause)
    (cond ((not (pair? clause))
	   (warning clause "Badly formed cond clause."))
	  ((match '(else expr*) clause)
	   (check-sequence (psd-cdr clause) environment))
	  (else
	   (check-sequence clause environment))))


  (cond ((match '(cond) form)
	 (warning form "No clauses in cond."))
	(else
	 (let loop ((clauses (psd-cdr form)))
	   (cond ((psd-null? clauses)
		  #f)

		 ((psd-pair? clauses)

		  ;; check for a missing else clause
		  (if (and (psd-null? (psd-cdr clauses))
			   (not (match '(else expr expr*) (psd-car clauses))))
		      (warning (psd-car clauses) "Missing else part in cond.")
		      #f)

		  (check-cond-clause (psd-car clauses))
		  (loop (psd-cdr clauses)))
		 
		 (else
		  (warning form "Badly formed cond.")))))))

;;;
;;; delay
;;;

(define (check-delay form environment)
  (warning form "check-delay not implemented yet."))

;;;
;;; do
;;;

(define (check-do form environment)
  (warning form "check-do not implemented yet."))

;;;
;;; else
;;; The builtin else's of case and cond are handled there, so an else
;;; elsewhere is a stray one.
;;;

(define (check-else form environment)
  (warning form "This else is not part of case or cond. Maybe you closed one too early.")
  (check-sequence (psd-cdr form) environment))

;;;
;;; lambda
;;;

(define (check-lambda form environment)
  (cond ((match '(lambda) form)
	 (warning form "Missing parameter list and body in lambda form."))
	((match '(lambda expr) form)
	 (warning form "Missing body in lambda form."))
	((match '(lambda expr expr*) form)
	 (let* ((params (check-parameter-list (cadr (match-result))))
		(new-environment (make-env environment
						   params)))
	   (check-sequence (caddr (match-result))
			   new-environment)))
	(else
	 (warning form "Sclint internal error. Please report this."))))

;;;
;;; let
;;;

(define (check-let form environment)
  (cond ((match '(let) form)
	 (warning form "Missing binding list and body in let."))
	((match '(let ()) form)
	 (warning form "Empty binding list and missing body in let. This does not make much sense."))
	((match '(let () expr*) form)
	 (warning form "No bindings in a binding list. You should probably use begin instead."))
	((match '(let pair) form)
	 (check-let-bindings (cadr (match-result)) environment)
	 (warning form "Missing body in let."))
	((match '(let pair expr*) form)
	 (let* ((this-match (match-result))
		(variables (check-let-bindings (cadr (match-result)) environment))
		(new-environment (make-env environment
						   variables)))
	   (check-sequence (caddr this-match) new-environment)))
	((match '(let symbol () expr*) form)
	 (let* ((this-match (match-result))
		(new-environment
		 (make-env environment
				   (list (make-variable (cadr this-match) 'local)))))
	   (check-sequence (cadddr this-match) new-environment)))
	((match '(let symbol pair expr*) form)
	 (let* ((this-match (match-result))
		(variables (check-let-bindings (caddr (match-result)) environment))
		(new-environment
		 (make-env environment
				   (cons (make-variable (cadr this-match) 'local)
					 variables))))
	   (check-sequence (cadddr this-match) new-environment)))
	(else
	 (warning form "Badly formed let form."))))

;;;
;;; Check that the binding list is well formed and that the variable
;;; initializations do not refer to illegal variables. Return a list
;;; of the local variables.
;;;

(define (check-let-bindings bindings environment)
  (cond ((psd-null? bindings) '())
	((psd-pair? bindings)
	 (let ((first-var (check-one-let-binding (psd-car bindings) environment)))
	   (if first-var
	       (cons first-var (check-let-bindings (psd-cdr bindings)
						   environment))
	       (check-let-bindings (psd-cdr bindings)
				   environment))))
	(else
	 (warning bindings "Badly formed binding list")
	 '())))

  
(define (check-one-let-binding binding environment)
  (cond ((match '(symbol) binding)
	 (warning binding "Missing initialization of binding.")
	 (make-variable (car (match-result)) 'local))
	((match '(expr) binding)
	 (warning binding "Missing variable name in binding.")
	 #f)
	((match '(symbol expr) binding)
	 (let ((this-match (match-result)))
	   (check-argument-counts (cadr (match-result)) environment)
	   (make-variable (car this-match) 'local)))
	(else
	 (warning binding "Bad binding.")
	 #f)))

;;;
;;; let*
;;;

(define (check-let* form environment)
  (cond ((match '(let*) form)
	 (warning form "Missing binding list and body in let."))
	((match '(let* ()) form)
	 (warning form "Empty binding list and missing body in let. This does not make much sense."))
	((match '(let* () expr*) form)
	 (warning form "No bindings in a binding list. You should probably use begin instead."))
	((match '(let* pair) form)
	 (check-let*-bindings (cadr (match-result)) environment)
	 (warning form "Missing body in let."))
	((match '(let* pair expr*) form)
	 (let* ((this-match (match-result))
		(new-environment (check-let*-bindings (cadr (match-result)) environment)))
	   (check-sequence (caddr this-match) new-environment)))
	(else
	 (warning form "Badly formed let* form."))))

;;;
;;; When checking let* bindings, we must add the variables to the
;;; environment as soon as they are declared. 
;;;

(define (check-let*-bindings bindings environment)
  (let loop ((bindings bindings)
	     (environment environment))
    (cond ((psd-null? bindings)
	   environment)
	  ((psd-pair? bindings)
	   (let ((first-var (check-one-let-binding (psd-car bindings) environment)))
	     (if first-var
		 (loop (psd-cdr bindings)
		       (make-env environment (list first-var)))
		 (loop (psd-cdr bindings)
		       environment))))
	(else
	 (warning bindings "Badly formed binding list")
	 environment))))

;;;

(define (check-letrec form environment)
  (warning form "check-letrec not implemented yet."))

;;;
;;; or
;;;

(define (check-or form environment)
  (if (match '(or) form)
      (warning form "No subexpressions in or form. You could replace this by #f.")
      #f)
  (check-sequence (psd-cdr form) environment))

;;;
;;; quasiquote
;;;

(define (check-quasiquote form environment)
  (warning form "check-quasiquote not implemented yet."))

;;;
;;; set!
;;;

(define (check-set! form environment)
  (cond ((match '(set!) form)
	 (warning form "No variable and expression in set! form."))
	((match '(set! symbol) form)
	 (warning form "No expression in set! form."))
	((match '(set! symbol expr) form)
	 (if (lookup environment (pexp->sexp (cadr (match-result))))
	     (if (variable-global? (lookup environment (pexp->sexp (cadr (match-result)))))
		 (warning form
			  "Assignment to global variable. You better be able to explain this one really well!")
		 #f)
	     (warning form "Assignment to non-existent variable."))
	 (check-argument-counts (caddr (match-result)) environment))
	((match '(set! symbol expr*) form)
	 (warning form "Too many expressions in set! form.")
	 (warning form "Subexpressions not checked."))
	(else
	 (warning form "Missing variable in set! form.")
	 (warning form "Subexpressions not checked."))))

;;;
;;; unquote
;;;

(define (check-unquote form environment)
  (warning form "check-unquote not implemented yet."))

;;;
;;; unquote-splicing
;;;

(define (check-unquote-splicing form environment)
  (warning form "check-unquote-splicing not implemented yet."))


;;;
;;; R4RS procedures.
;;;

(define r4rs-procedures
  '((* &rest)
    (+ &rest)
    (- &rest)
    (- one&rest)
    (/ one&rest)
    (< &rest)
    (<= &rest)
    (= &rest)
    (> &rest)
    (>= &rest)
    (abs 1)
    (acos 1)
    (angle 1)
    (append &rest)
    (apply two&rest)
    (asin 1)
    (assoc 2)
    (assq 2)
    (assv 2)
    (atan one/two)
    (boolean? 1)
    (caaaar 1)
    (caaadr 1)
    (caaar 1)
    (caadar 1)
    (caaddr 1)
    (caadr 1)
    (caar 1)
    (cadaar 1)
    (cadadr 1)
    (cadar 1)
    (caddar 1)
    (cadddr 1)
    (caddr 1)
    (cadr 1)
    (call-with-current-continuation 1)
    (call-with-input-file 2)
    (call-with-output-file 2)
    (car 1)
    (cdaaar 1)
    (cdaadr 1)
    (cdaar 1)
    (cdadar 1)
    (cdaddr 1)
    (cdadr 1)
    (cdar 1)
    (cddaar 1)
    (cddadr 1)
    (cddar 1)
    (cdddar 1)
    (cddddr 1)
    (cdddr 1)
    (cddr 1)
    (cdr 1)
    (ceiling 1)
    (char->integer 1)
    (char-alphabetic? 1)
    (char-ci<=? 2)
    (char-ci<? 2)
    (char-ci=? 2)
    (char-ci>=? 2)
    (char-ci>? 2)
    (char-downcase 1)
    (char-lower-case? 1)
    (char-numeric? 1)
    (char-ready? one/two0)
    (char-upcase 1)
    (char-upper-case? 1)
    (char-whitespace? 1)
    (char<=? 2)
    (char<? 2)
    (char=? 2)
    (char>=? 2)
    (char>? 2)
    (char? 1)
    (close-input-port 1)
    (close-output-port 1)
    (complex? 1)
    (cons 2)
    (cos 1)
    (current-input-port 0)
    (current-output-port 0)
    (denominator 1)
    (display one/two)
    (eof-object? 1)
    (eq? 2)
    (equal? 2)
    (eqv? 2)
    (even? 1)
    (exact->inexact 1)
    (exact? 1)
    (exp 1)
    (expt 2)
    (floor 1)
    (for-each two&rest)
    (force 1)
    (gcd &rest)
    (imag-part 1)
    (inexact->exact 1)
    (inexact? 1)
    (input-port? 1)
    (integer->char 1)
    (integer? 1)
    (lcm &rest)
    (length 1)
    (list &rest)
    (list->string 1)
    (list->vector 1)
    (list-ref 2)
    (list-tail 2)
    (list? 1)
    (load 1)
    (log 1)
    (magnitude 1)
    (make-polar 2)
    (make-rectangular 2)
    (make-string one/two)
    (make-vector one/two)
    (map two&rest)
    (max one&rest)
    (member 2)
    (memq 2)
    (memv 2)
    (min one&rest)
    (modulo 2)
    (negative? 1)
    (newline zero/one)
    (not 1)
    (null? 1)
    (number->string one/two)
    (number? 1)
    (numerator 1)
    (odd? 1)
    (open-input-file 1)
    (open-output-file 1)
    (output-port? 1)
    (pair? 1)
    (peek-char zero/one)
    (positive? 1)
    (procedure? 1)
    (quotient 2)
    (rational? 1)
    (rationalize 2)
    (read zero/one)
    (read-char zero/one)
    (real-part 1)
    (real? 1)
    (remainder 2)
    (reverse 1)
    (round 1)
    (set-car! 2)
    (set-cdr! 2)
    (sin 1)
    (sqrt 1)
    (string &rest)
    (string->list 1)
    (string->number one/two)
    (string->symbol 1)
    (string-append &rest)
    (string-ci<=? 2)
    (string-ci<? 2)
    (string-ci=? 2)
    (string-ci>=? 2)
    (string-ci>? 2)
    (string-copy 1)
    (string-fill! 2)
    (string-length 1)
    (string-ref 2)
    (string-set! 3)
    (string<=? 2)
    (string<? 2)
    (string=? 2)
    (string>=? 2)
    (string>? 2)
    (string? 1)
    (substring 3)
    (symbol->string 1)
    (symbol? 1)
    (tan 1)
    (transcript-off 0)
    (transcript-on 1)
    (truncate 1)
    (vector &rest)
    (vector->list 1)
    (vector-fill! 2)
    (vector-length 1)
    (vector-ref 2)
    (vector-set! 3)
    (vector? 1)
    (with-input-from-file 2)
    (with-output-to-file 2)
    (write one/two)
    (write-char one/two)
    (zero? 1)))

;;;
;;; Is this a primitive?
;;;

(define (primitive-procedure? expr)
  (assoc (pexp->sexp expr) r4rs-procedures))

;;;
;;; Check the number of arguments to a primitive procedure, as defined
;;; in R4RS.
;;;

(define (check-primitive-call call environment)
  (let* ((sexp-call (pexp->sexp call))
	 (entry (assoc (car sexp-call) r4rs-procedures)))
    (if (or

	 ;; fixed number of arguments
	 (and (number? (cadr entry))
	      (not (= (cadr entry)
		      (length (cdr sexp-call)))))

	 ;; zero or one arguments
	 (and (eq? (cadr entry) 'zero/one)
	      (not (or (= (length sexp-call) 1)
		       (= (length sexp-call) 2))))

	 ;; one or two arguments
	 (and (eq? (cadr entry) 'one/two)
	      (not (or (= (length sexp-call) 2)
		       (= (length sexp-call) 3))))

	 ;; one required argument
	 (and (eq? (cadr entry) 'one&rest)
	      (< (length sexp-call) 2))

	 ;; two required arguments
	 (and (eq? (cadr entry) 'two&rest)
	      (< (length sexp-call) 3)))

	;; issue warning if any of the above are true
	(warning call
		 "wrong number of arguments to procedure "
		 (car entry)))))
