;;;;
;;;; $Id: match.scm,v 1.3 1992/09/03 13:56:57 pk Exp $
;;;;
;;;; sclint -- a Scheme lint.
;;;;
;;;; Pertti Kellom\"aki, 1992
;;;;
;;;; $Log: match.scm,v $
;;;; Revision 1.3  1992/09/03  13:56:57  pk
;;;; Rewritten once again.
;;;;
;;;; Revision 1.2  1992/09/03  07:02:19  pk
;;;; Fixed a bug in verbatim match. Return the actual match, not the pattern.
;;;;
;;;; Revision 1.1  1992/08/27  12:22:36  pk
;;;; Initial revision
;;;;
;;;;

;;;;
;;;; This is a simple pattern matcher. It takes a pattern and
;;;; a pexp to which to match the pattern. The pattern is a list,
;;;; and each element can be either
;;;;
;;;;   * the symbol "symbol", which matches any symbol,
;;;;   * the symbol "pair", which matches a nonempty list,
;;;;   * the symbol "list", which matches a pair and the empty list,
;;;;   * the symbol "expr", which matches any expression,
;;;;   * the symbol "expr*", which matches a (possibly empty) sequence of
;;;;     expressions, or
;;;;   * a list, which is matched recursively,
;;;;   * an arbitrary atom, which matches itself.
;;;;
;;;; The matched elements are returned as a list, with each matced
;;;; element in the same place as the element of pattern it matched.
;;;;
;;;; Note that the symbol "expr*" can only occur as the last element
;;;; of the pattern, because it matches the rest of the pexp.
;;;;
;;;; The procedure match-result returns the result of the most recent
;;;; match. 
;;;;

(define match #f)
(define match-result #f)


(letrec
    (

     ;; store the result of the matching here
     (previous-match '())

     ;; returning of the previous match
     (do-match-result (lambda () previous-match))

     ;; this is only used for calls from outside
     (do-match-memoize
      (lambda (pattern expr)
	(let ((result (do-match pattern expr)))
	  (if result
	      (set! previous-match result)
	      (set! previous-match #f)))
	previous-match))
    
     ;; the actual matching
     (do-match
      (lambda (pattern expr)

	(case pattern
	  ((symbol) (if (psd-symbol? expr)
			expr
			#f))
	  ((pair) (if (psd-pair? expr)
		      expr
		      #f))
	  ((list) (if (or (psd-null? expr)
			  (psd-pair? expr))
		      expr
		      #f))
	  ((expr) expr)

	  (else

	   (cond
	    
	    ;; an expr* always matches the rest of pexp
	    ((and (pair? pattern)
		  (equal? (car pattern) 'expr*))
	     (if (null? (cdr pattern))
		 (list expr)
		 (error "do-match: expr* must be the last element of a (sub)pattern.")))

	    ;; match a pair recursively. If car does not match there
	    ;; is no need to try to match cdr. 
	    ((and (pair? pattern)
		  (psd-pair? expr))
	     (let ((car-match (do-match (car pattern) (psd-car expr))))
	       (if car-match
		   (let ((cdr-match (do-match (cdr pattern) (psd-cdr expr))))
		     (if cdr-match
			 (cons car-match cdr-match)
			 #f))
		   #f)))
		   
	    ;; check for verbatim match
	    ((equal? pattern (pexp->sexp expr))
	     expr)

	    ;; matching failed
	    (else #f)))))))

  ;; install the real definitions
  (set! match do-match-memoize)
  (set! match-result do-match-result))
