;;;
;;;   A U G M E N T E D   T R A N S I T I O N
;;;   = = = = = = = = =   = = = = = = = = = =
;;;
;;;   N E T W O R K     T O O L B O X
;;;   = = = = = = =     = = = = = = =
;;;
;;; Depends on the following toolboxes:
;;;    Systems
;;;
;;; Augmented Transition Network (ATN) Toolbox
;;;
;;; This toolbox contains routines to build and use ATN's to
;;; parse natural languages.

; An ATN consists of a finite automata with states conected by arcs
; where the arcs are labelled by category names (these represent a
; class of words such as verb or noun which must appear in the input
; sentence) or the name of another ATN (in which case that ATN is
; called recursively). Also arcs can have tests that must be satisfied
; before the arc is followed and actions to be performed. There is
; also a set of registers that can store information about the parse
; and be referred to by the tests.

; The following data structures are used:
;
; REGISTERBANKs - association list of register value pairs
; PARSEPAIRs    - list of the form (aSentence aRegisterBank)
; TESTS         - either #t or a boolean procedure
; FORMs         - one of #t, #f, **, (GETR ...) or lambda
; ACTIONs       - either (SETR ...) or (TO ...)
; ARCs          - one of (CAT ...), (ANY ..), (PUSH ...) or (POP ...)
; ARCLISTs      - a list of ARCs
; STATEs        - a list of the form (label anArcList)
; ATNs          - a list of states

; ========== fns on REGISTERBANKs ===========================
; The registers are stored in register banks as association lists.

(define MakeRegisterBank (lambda () ()))

(define SetRegister
  (lambda (aRegisterBank aRegister aValue)
    
    (define removeAux
      (lambda (aList aKey)
        (cond ((null? aList) ())
              ((eq? aKey (caar aList)) (cdr aList))
              (else (cons (car aList) (removeAux (cdr aList) aKey))))))
    
    (cons (list aRegister aValue)
          (removeAux aRegisterBank aRegister))))

(define KnownRegister?
  (lambda (aRegisterBank aRegister)
    (if (assoc aRegister aRegisterBank) #t #f)))

(define GetRegister
  (lambda (aRegisterBank aRegister)
    (let ((pair (assoc aRegister aRegisterBank)))
      (if (not pair)
          (Fatal-Error "GetRegister:" "No such register as" aRegister)
          (cadr pair)))))

(define UpdateRegisters
  (lambda (anActionList aRegisterBank)
    (if (EmptyActions? anActionList)
        aRegisterBank
        (let ((action (FirstAction anActionList)))
          (if (SETR? action)
              (UpdateRegisters (RestOfActions anActionList)
                               (SetRegister aRegisterBank
                                            (GetActionRegister action)
                                            (EvalForm (GetActionForm 
						        action)
                                                      aRegisterBank)))
              (Fatal-Error "UpdateRegisters:" "illegal action" action))))))

; ========== fns on PARSEPAIRs ==============================
; list of the form (aSentence aRegisterBank)

(define MakeParsePair (lambda (aSentence aRegisterBank) (list aSentence aRegisterBank)))

(define GetSentence (lambda (aParsePair) (car aParsePair)))

(define GetRegisterBank (lambda (aParsePair) (cadr aParsePair)))

; ========== fns on TESTs ===================================
; Either #t or a boolean procedure

(define CheckTest?
  (lambda (aTest aRegisterBank)
    (cond ((eq? aTest #t) #t)
          ((procedure? aTest) (aTest aRegisterBank))
          (else (Fatal-Error "CheckTest:" "invalid test" aTest)))))

; ========== fns on FORMs ===================================
; one of the following:
;     (GETR aRegister)
;     **
;     #t
;     #f
;     ()
;  or an arbitary lambda expression

(define GETR (lambda (aRegister) (list 'GETR aRegister)))

(define GETR? (lambda (aForm) (and (pair? aForm) (eq? 'GETR (car aForm)))))

(define GetFormRegister (lambda (aForm) (cadr aForm)))

(define ** '**)

(define **?
  (lambda (aForm) (eq? '** aForm)))

(define EvalForm
  (lambda (aForm aRegisterBank)
    (cond ((GETR? aForm)
           (GetRegister aRegisterBank (GetFormRegister aForm)))
          ((**? aForm)
           (GetRegister aRegisterBank '**))
          ((eq? aForm #t) #t)
          ((null? aForm) ())
          ((eq? aForm #f) #f)
          ((procedure? aForm) (aForm aRegisterBank))
          (else (Fatal-Error "EvalForm: illegal form")))))

; ========== fns on ACTIONs =================================
; one of the following:
;     (SETR  aREGISTER aFORM)
;     (TO    label)
;

; (define FirstAction         - See Search Toolbox
;
; (define RestOfActions       - See Search Toolbox
;
; (define EmptyActions?       - See Search Toolbox

(define ActionType (lambda (anAction) (car anAction)))

(define SETR (lambda (aRegister aForm) (list 'SETR aRegister aForm)))

(define SETR? (lambda (anAction) (eq? 'SETR (ActionType anAction))))

(define GetActionRegister (lambda (anAction) (cadr anAction)))

(define GetActionForm (lambda (anAction) (caddr anAction)))

(define TO (lambda (aLabel) (list 'TO aLabel)))

(define TO? (lambda (anAction) (eq? 'TO (ActionType anAction))))

(define CheckActions
  (lambda (anActionList)
    ; check that the last action is (TO ..) and return a list
    ; of the form (otherActions (TO label))
    
    (define checkActionsAux
      (lambda (anActionList otherActions)
        (let ((head (FirstAction anActionList))
              (tail (RestOfActions anActionList)))
          (cond ((EmptyActions? tail)
                 (if (TO? head)
                     (list otherActions head)
                     (Fatal-Error "CheckActions: last action not TO")))
                ((SETR? head)
                 (checkActionsAux tail
                                  (append otherActions (list head))))
                (else
                 (Fatal-Error "CheckActions:" "illegal action" head))))))
    
    (if (null? anActionList)
        (Fatal-Error "CheckActions:"
                     "there must be at least one action")
        (checkActionsAux anActionList ()))))

; ========== fns on ARCs ====================================
; one of the following lists:
;    (CAT  aCATEGORY test actions (TO aLabel))
;    (ANY            test actions (TO aLabel))
;    (PUSH anATN     test actions (TO aLabel))
;    (POP  aFORM     test)

(define ArcType (lambda (anArc) (car anArc)))

(define CAT
  (lambda (aCategory aTest . anActionList)
    (append (list 'CAT aCategory aTest)
            (CheckActions anActionList))))

(define CAT? (lambda (anArc) (eq? 'CAT (ArcType anArc))))

(define MakeCategory (lambda aWordList aWordList))

(define GetCategory
  (lambda (anArc)
    (if (CAT? anArc)
        (cadr anArc)
        (Fatal-Error "GetCategory:" "Arc" anArc "not a CATegory"))))

(define InCategory?
  (lambda (aWord aCategory) (if (member aWord aCategory) #t #f)))

(define ANY
  (lambda (aTest . anActionList)
    (append (list 'ANY aTest) (CheckActions anActionList))))

(define ANY?
  (lambda (anArc) (eq? 'ANY (ArcType anArc))))

(define PUSH
  (lambda (anATN aTest . anActionList)
    (append (list 'PUSH anATN aTest) (CheckActions anActionList))))

(define PUSH? (lambda (anArc) (eq? 'PUSH (ArcType anArc))))

(define GetATN
  (lambda (anArc)
    (if (PUSH? anArc)
        (cadr anArc)
        (Fatal-Error "GetATN:" "Arc" anArc "not a PUSH"))))

(define POP (lambda (aForm aTest) (list 'POP aForm aTest)))

(define POP? (lambda (anArc) (eq? 'POP (ArcType anArc))))

(define GetForm
  (lambda (anArc)
    (if (POP? anArc)
        (cadr anArc)
        (Fatal-Error "GetForm:" "Arc" anArc "not a POP"))))

(define GetTest
  (lambda (anArc)
    (if (ANY? anArc)
        (cadr anArc)
        (caddr anArc))))

(define GetArcActions
  (lambda (anArc)
    (cond ((POP? anArc)
           (Fatal-Error "GetArcActions:"
                        "there are no actions for a POP arc"))
          ((ANY? anArc)
           (caddr anArc))
          (else
           (cadddr anArc)))))

(define GetNextState
  (lambda (anArc)
    (if (POP? anArc)
        (Fatal-Error "GetNextState:"
                     "there is no next state for a POP arc")
        (let ((toarc (if (ANY? anArc)
                         (car (cdddr anArc))
                         (car (cddddr anArc)))))
          (if (TO? toarc)
              (cadr toarc)
              (Fatal-Error "GetNextState:" "bad TO arc" toarc))))))
    
; ========== fns on ARCLISTs ================================
; a list of ARCs

(define CurrentArc (lambda (anArcList) (car anArcList)))

(define RestOfArcs (lambda (anArcList) (cdr anArcList)))

(define EmptyArcList? (lambda (anArcList) (null? anArcList)))

; ========== fns on STATEs ==================================
; a list of the form (label ARCLIST)

(define STATE
  (lambda (aStateLabel . anArcList) (cons aStateLabel anArcList)))

(define GetStateLabel (lambda (aState) (car aState)))

(define GetArcs (lambda (aState) (cdr aState)))

; ========== fns on ATNs ====================================
; an ATN is a list of states.

(define MakeATN (lambda aStateList aStateList))

(define AddState
  (lambda (anATN aState) (append anATN (list aState))))

(define InitialState (lambda (anATN) (car anATN)))

(define FindState
  (lambda (anATN aStateLabel)
    (cond ((null? anATN) ())
          ((eq? aStateLabel (GetStateLabel (car anATN))) (car anATN))
          (else (FindState (cdr anATN) aStateLabel)))))

; ========== fns for Parsing sentences ======================
; In general to parse ATN's a form of Earley's parsing method is
; needed which follows all possible parses of the sentence in a
; breadth first manner. The following routines use a simpler recursive
; descent with backtracking method. This is possible because we have
; disallowed arcs with tests only, so that a word is
; consumed or an ATN called recusively as every arc is followed. Each
; call to the parsing routines will follow each arc in turn for as far
; as possible until either a node with no valid arcs is found
;(failure) or a POP arc is found (success). This differs from the
; general Earley algorithm that would also look at the arcs that
; appear AFTER the successful POP arc for further matches. Some
; measure of control over this is available to the user as the arcs
; are considered in the order they are defined - this suggests that
; POP arcs should usually appear last.

(define ParseAux
  (lambda (anATN aState aRegisterBank restOfSentence aTraceFlag)
    ; Attempt to Parse the 'restOfSentence' beginning from 'aState' of
    ; 'anATN' and with the current register values given in
    ; 'aRegisterBank'. Arcs with successful tests are followed by
    ; calling ParseAux recursively with updated register values. When
    ; a node with no valid arcs is encountered the routine returns #f.
    ; When a valid POP is encountered a PARSEPAIR is returned.
    (if aTraceFlag
        (DisplayLine "PauseAux: state=" (GetStateLabel aState)
                     "sentence=" restOfSentence)
        #f)
    (do ((arcs (GetArcs aState) (RestOfArcs arcs))
         (arc #f)
         (newRegisterBank #f)
         (newerRegisterBank #f)
         (word #f)
         (actions #f)
         (result #f))
        ((or (EmptyArcList? arcs)
             result) (if aTraceFlag
                         (DisplayLine "PauseAux: state="
                                      (GetStateLabel aState)
                                      "-->"
                                      result)
                         #f)
                     result)
        (set! arc (CurrentArc arcs))
        (if (null? restOfSentence)
            (begin (set! word #f)
                   (set! newRegisterBank aRegisterBank))
            (begin (set! word (car restOfSentence))
                   (set! newRegisterBank
                         (SetRegister aRegisterBank '** word))))
        (if (CheckTest? (GetTest arc) newRegisterBank)
            (cond ((CAT? arc)
                   (if (and restOfSentence
                            (InCategory? word (GetCategory arc)))
                       (begin
                        (set! newerRegisterBank
                              (UpdateRegisters (GetArcActions arc)
                                               newRegisterBank))
                        (set! result
                              (ParseAux anATN
                                        (FindState anATN (GetNextState arc))
                                        newerRegisterBank
                                        (cdr restOfSentence)
                                        aTraceFlag)))
                       #f))
                  ((ANY? arc)
                   (if restOfSentence
                       (begin
                        (set! newerRegisterBank
                              (UpdateRegisters (GetArcActions arc)
                                               newRegisterBank))
                        (set! result
                              (ParseAux anATN
                                        (FindState anATN (GetNextState arc))
                                        newerRegisterBank
                                        (cdr restOfSentence)
                                        aTraceFlag)))
                       #f))
                  ((PUSH? arc)
                   (set! result
                         (ParseAux (GetATN arc)
                                   (InitialState (GetATN arc))
                                   newRegisterBank
                                   restOfSentence
                                   aTraceFlag))
                   (if result
                       (begin
                        (set! newerRegisterBank
                              (UpdateRegisters (GetArcActions arc)
					       (GetRegisterBank result)))
                        (set! result
                              (ParseAux anATN
                                        (FindState anATN (GetNextState arc))
                                        newerRegisterBank
                                        (GetSentence result)
                                        aTraceFlag)))
                       #f))
                  ((POP? arc)
                   (set! result
                         (MakeParsePair
                          restOfSentence
                          (SetRegister newRegisterBank
                                       '**
                                       (EvalForm (GetForm arc)
                                                 newRegisterBank)))))
                  (else (Fatal-Error "ParseAux:" "illegal arc type" arc)))
            #f))))

(define Parse
  (lambda (anATN aSentence . aTraceFlag)
    ; Return #f if parse fails. If it succeeds then return the
    ; register bank. If not supplied aTraceFlag defaults to #f
    (let ((result (ParseAux anATN
                            (InitialState anATN)
                            (MakeRegisterBank)
                            aSentence
                            (if (null? aTraceFlag)
                                #f
                                (car aTraceFlag)))))
      (if (or (not result)
              (not (null? (GetSentence result))))
          #f
          (GetRegisterBank result)))))
