; ***** K A L A H ********
;
; This game is for two players, with each player having 6 pits each
; containing 6 stones and a Kalah. These are aranged as follows:

;    Player 1:  kalah  pit6  pit5  pit4  pit3  pit2  pit1
;    stones             6     6     6     6     6     6
;
;    Player 2:         pit1  pit2  pit3  pit4  pit5  pit6  kalah
;    stones             6     6     6     6     6     6
;
; Each move involes taking all the stones out of one of your pits
; and then dropping one in each of your own pits, your kalah and
; your opponents pit (but skipping their kalah) in a counter
; clockwise direction. E.g. if the stones come from player 2,
; pit 3 and there were 11 stones then one stone would be dropped
; in each of: players 2 pits 4, 5, 6 - player 2 kalah - player 1
; pit 1, 2, 3, 4, 5, 6 and the last stone would go into player 2
; pit 1.
;
; There are three possible outcomes:
;    1) if the last stone lands in the player's own kalah they move
;       again.
;    2) if the last stone lands in an empty pit owned by the player
;       and the opponent's hole directly opposite contains stones,
;       then the last stone plus all the stones in the opponent's
;       pit are dropped in the player's kalah
;    3) otherwise the player's turn ends.
;
; The game ends when one player's pits are all empty or more than
; half the stones are in one of the kalahs. The winner is the
; player with the most stones in their kalah at the end of the game.
;

(define KalahGameProblem #f)

(define SetUpKalah
  (lambda ()
    ; The state of the board will be represented as a 14 element
    ; vector of first the machines pits plus kalah and then the
    ; opponent's pits plus kalah. An additional element (the last)
    ; will be used to indicate that this state is one that allows
    ; another move to a player. It will be #f for no extra move,
    ; "machine" if the machine gets another move and "opponent" if
    ; the opponent gets another move
    ; Hence the initial state is:
    
    (define initial
      (vector 6 6 6 6 6 6 0   6 6 6 6 6 6 0 #f))
    
    ; The following functions are used to select and set the values
    ; in the state
    
    (define machinePits 0)
    (define machinePit1 0)
    (define machinePit2 1)
    (define machinePit3 2)
    (define machinePit4 3)
    (define machinePit5 4)
    (define machinePit6 5)
    (define machineKalah 6)
    (define opponentPits 7)
    (define opponentPit1 7)
    (define opponentPit2 8)
    (define opponentPit3 9)
    (define opponentPit4 10)
    (define opponentPit5 11)
    (define opponentPit6 12)
    (define opponentKalah 13)
    (define lastPit 13)
    (define numberPits 14)
    (define extraMove? 14)
    
    (define getStones
      (lambda (aState whichPit) (vector-ref aState whichPit)))

    (define setStones!
      (lambda (aState whichPit stones)
        (vector-set! aState whichPit stones)))

    (define getExtra
      (lambda (aState) (vector-ref aState extraMove?)))

    (define setExtra!
      (lambda (aState) (vector-set! aState extraMove? #t)))

    (define resetExtra!
      (lambda (aState) (vector-set! aState extraMove? #f)))
    
    (define copyState
      (lambda (aState)
        ; return a new vector with same contents and extra move cleared
        (let ((result (list->vector (vector->list aState))))
          (resetExtra! result)
          result)))
    
    (define nextPit
      (lambda (aPit machine?)
        ; wrap around end and skip opponent's kalah
        (let ((n (+ aPit 1)))
          (cond ((> n lastPit) 0)
                ((and (= n machineKalah) (not machine?))
		 opponentPit1)
                ((and (= n opponentKalah) machine?)
		 machinePit1)
                (else n)))))
    
    ; Evaluation function - return the difference between number of
    ; stones in machine's and opponent's kalah.
    
    (define evalFN
      (lambda (aState machineMove?)
        (let* ((machine (getStones aState machineKalah))
               (opponent (getStones aState opponentKalah)))
          (- machine opponent))))
    
    ; Winning positions are if all a players pits empty or one
    ; kalah more than half the stones
    
    (define status
      (lambda (aState machineMove?)
        
        (define moreThanHalf? (lambda (n) (> n 36)))
        
        (define noStones?
          (lambda (aPlayer)
            (do ((number 0)
                 (pit aPlayer (+ pit 1))
                 (i 6 (- i 1)))
                ((zero? i) (zero? number))
                (set! number (+ number (getStones aState pit))))))
        
        (define playerWithMost
          (lambda ()
            (cond ((> (getStones aState machineKalah)
                      (getStones aState opponentKalah)) "machine")
                  ((< (getStones aState machineKalah)
                      (getStones aState opponentKalah)) "opponent")
                  (else "draw"))))
        
        (cond ((moreThanHalf? (getStones aState machineKalah))
               "machine")
              ((moreThanHalf? (getStones aState opponentKalah))
               "opponent")
              ((or (noStones? machinePits)
                   (noStones? opponentPits)) (playerWithMost))
              ((getExtra aState) "again")
              (else #f))))
    
    (define checkForExtraTurn!
      (lambda (newState machine? lastPit)
        ; if last move is into own kalah set extra move
        ; if last move is into pit opposite non-empty pit transfer
        ; stones to kalah
        
        (define oppositePit
          (lambda (aPit)
            (vector-ref (vector 12 11 10 9 8 7 #f 5 4 3 2 1 0 #f)
                        aPit))) 
        
        (cond ((or (= lastPit machineKalah)
                   (= lastPit opponentKalah)) (setExtra! newState))
              ((and machine?
                    (< lastPit machineKalah)
                    (= 1 (getStones newState lastPit))
                    (positive? (getStones newState
                                          (oppositePit lastPit))))
               (begin (setStones! newState
                                  machineKalah
                                  (+ (getStones newState
                                                machineKalah)
				     (+ (getStones newState
						   (oppositePit lastPit))
					1)))
                      (setStones! newState (oppositePit lastPit) 0)
                      (setStones! newState lastPit 0)
                      newState))
              ((and (not machine?)
                    (>= lastPit opponentPits)
                    (= 1 (getStones newState lastPit))
                    (positive? (getStones newState
                                          (oppositePit lastPit))))
               (begin (setStones! newState
                                  opponentKalah
                                  (+ (getStones newState opponentKalah)
				     (+ (getStones newState
                                                (oppositePit lastPit))
					1)))
                      (setStones! newState (oppositePit lastPit) 0)
                      (setStones! newState lastPit 0)
                      newState)))))
    
    (define moveStones
      (lambda (aPit machine?)
        ; Return a function that moves the stones from a given pit
        ; machine? is #t if it is a machine move.
        (lambda (aState)
          (let ((n (getStones aState aPit)))
            (if (zero? n)
                #f
                ; construct a new state with stones in 'aPit'
                ; redistributed
                (let ((newState (copyState aState))
                      (lastPit 0))
                  (setStones! newState aPit 0)
                  (do ((pit (nextPit aPit machine?)
                            (nextPit pit machine?)))
                      ((zero? n))
                      (setStones! newState
                                  pit
                                  (+ 1 (getStones newState pit)))
                      (set! n (- n 1))
                      (set! lastPit pit))
                  (checkForExtraTurn! newState machine? lastPit)
                  newState))))))
    
    (define printKalah
      (lambda (aState)
        ; output in two facing rows. e.g.:
        ; K  6  5  4  3  2  1
        ; 0  6  6  6  6  6  6
        ; ======================
        ;    6  6  6  6  6  6  0
        ;    1  2  3  4  5  6  K
        
        (define myDisplay
          (lambda (n)
            (if (< n 10)
                (display " ")
                #f)
            (display n)
            (display " ")))
        
        (newline)
        (display " K  6  5  4  3  2  1") (newline)
        (myDisplay (getStones aState opponentKalah))
        (do ((i opponentPit6 (- i 1)))
            ((= i machineKalah))
            (myDisplay (getStones aState i)))
        (newline)
        (display "======================") (newline)
        (display "   ")
        (do ((i machinePit1 (+ i 1)))
            ((= i machineKalah))
            (myDisplay (getStones aState i)))
        (myDisplay (getStones aState machineKalah)) (newline)
        (display "    1  2  3  4  5  6  K")))
    
    (define getMove
      (lambda (aState)
        (let ((which 0))
          (if (getExtra aState)
              (begin (newline)
                     (display "Have an extra move!"))
              #f)
          (set! which
                (do ((pos -1))
                    ((and (number? pos)
                          (<= 0 pos)
                          (<= pos 6)
                          (positive? (getStones aState (+ pos machineKalah))))
                     pos)
                    (newline)
                    (display "Give pit (1-6,0=resign): ")
                    (newline)
                    (set! pos (read))))
          (if (zero? which)
              #f
              ((moveStones  (+ which machineKalah) #f) aState)))))
    
    (set! KalahGameProblem
          (MakeGameProblem initial
                           status
                           evalFN
                           (MakeActionList
                            (moveStones machinePit1 #t)
                            (moveStones machinePit2 #t)
                            (moveStones machinePit3 #t)
                            (moveStones machinePit4 #t)
                            (moveStones machinePit5 #t)
                            (moveStones machinePit6 #t))
                           (MakeActionList
                            (moveStones opponentPit1 #f)
                            (moveStones opponentPit2 #f)
                            (moveStones opponentPit3 #f)
                            (moveStones opponentPit4 #f)
                            (moveStones opponentPit5 #f)
                            (moveStones opponentPit6 #f))))
    
    (SetPrintState! KalahGameProblem printKalah)
    
    (SetGetAMove! KalahGameProblem getMove)))
    
(begin (SetUpKalah)
       (DisplayLine "Type (PlayGame KalahGameProblem)"
                    "to play Kalah"))
