; ===== Building Blocks for Queueing Network Scenarios =====

; ----- a Transaction Flavour -----

; Transactions are dynamic workload items. They are modelled by processes
; which carry a timestamp so that durations of relevant sequences of
; activities may be measured (i.e. at sinks)

(defFlavour Transaction (ako Process) (ivars birthTime activeMethod unitsRequested) 
  setivars getivars testivars)

(defMethod (timeStamp! Transaction) (aClockTime)
  (self 'birthTime! aClockTime))

(defMethod (show Transaction) ()   
  (display "a Process of flavour ") (display (self 'flavourName)) 
  (display " with birthTime : ") (display birthTime)) 

; ----- Source Flavour -----

; Sources are processes, which create transactions within intervals 
; determined according to some random distribution.These are then 
; time-stamped and immediately activiated.

(defFlavour Source (ako Process) 
  (ivars monitor terminationTime creationCount 
         processFlavour processParams methodName arrivalDist tracing?)
  setivars getivars testivars)

(defMethod (finished? Source) () (> (monitor 'clockTime) terminationTime))

(defMethod (startUp! Source) (aDuration) 
  (display "--- Source  starts up at: ") 
  (display (monitor 'clockTime))
  (newline)
  (set! terminationTime aDuration)
  (set! creationCount (Count 'new))
  (monitor 'schedule! self 'create! (+ (monitor 'clockTime)
                                       (arrivalDist 'sample))))

(defMethod (reset! Source) () (creationCount 'reset!))

(defCoroutine (create! Source) ()
  (while (>=? terminationTime (monitor 'clockTime))
      (begin (let ((newTA nil) (message nil))
               (set! newTA (processFlavour 'new))
               (creationCount 'update! 1)
               ; set the process' parameters
               (newTA 'monitor! monitor)
               (for-each (lambda (parValuePair)
                              (set! message (string->symbol
                                               (string-append
                                                  (symbol->string 
                                                  (car parValuePair)) "!")))
                              (newTA message (eval (cadr parValuePair))))
                         processParams)
               (newTA 'timeStamp! (monitor 'clockTime))
               (newTA 'activeMethod! methodName)
               (if (self 'tracing??) 
                 (begin (display "+SOURCE: schedules a '")
                        (display (newTa 'flavourName)) 
                        (display "' with method '") (display methodName) 
                        (display "' at ") (display (monitor 'clockTime))
                        (newline)))               
               (monitor 'schedule! newTA methodName (monitor 'clockTime))
               (if (self 'tracing??) 
                   (begin (display "+SOURCE: reschedules itself")                       
                          (display " at ") (display (monitor 'clockTime))
                          (newline)))
               (self 'hold 'create! (arrivalDist 'sample))
                )))
      (self 'terminate!) ) 

(defMethod (terminate! Source) () 
  (display "--- Source terminates at: ")
  (display (monitor 'clocktime)) (newline))

(defMethod (restart Source) (aDuration) (self 'startUp aDuration))

(defMethod (show Source) ()
  (display "--- Source Statistics ---") (newline)
  (display "Time now      : ") (display (monitor 'clockTime)) (newline)
  (display "Terminates at : ") (display terminationTIme) (newline)
  (display "Creations     : ") (display (creationCount 'counter)) (newline) 
  (display "ProcessFlavour: ") 
  (display ((processFlavour 'new) 'flavourName)) (newline)
  (display "Method        : ") (display methodName) (display " with ")
  (display processParams) (display " as parameters ") (newline) 
  (display "- Inter-arrival-time distribution -") (newline)
  (arrivalDist 'show) )

; ----- Sink Flavour -----

; Sinks are simple data collection devices. They measure the time departing
; transactions spent since they were last timestamped (normally at "birth")

(defFlavour Sink (ako Vanilla) (ivars monitor obsTally tracing?) 
  setivars getivars testivars)

(defMethod (depart! Sink) (aTransaction)
  (if (not (self 'obsTally?)) (set! obsTally (Tally 'new)))
  (if (self 'tracing??)
      (begin (display "+SINK: ") (display (aTransaction 'flavourName))
             (display " leaves at ") (display (monitor 'clockTime))
             (display " after ") 
             (display (- ((aTransaction 'monitor) 'clockTime)
                         (aTransaction 'birthTime)))
             (display " time units")))                
  (obsTally 'update! (- ((aTransaction 'monitor) 'clockTime)
                        (aTransaction 'birthTime)))
  (monitor 'continue 'resumeProcesses) )

(defMethod (show Sink) ()
  (display "--- Sink Statistics (TA's flowtimes) ---") (newline)
  (if (self 'obsTally?)
      (obsTally 'show)
      (begin (display "NO observations made !") (newline))) )

; ----- flavour FifoQueue -----

(defFlavour FifoQueue (ako Vanilla) (ivars contents lengthAccum monitor) 
  setivars getivars testivars)

(defMethod (empty? FifoQueue) () (null? contents))

(defMethod (clear! FifoQueue) () (set! contents nil))

(defMethod (length FifoQueue) () (length contents))

(defMethod (queue! FifoQueue) (aProcess) 
  (self 'contents! (append (self 'contents) (list aProcess)))
  (if (null? lengthAccum) (set! lengthAccum (Accumulate 'new)))
  (lengthAccum 'update! (monitor 'clockTime) (self 'length)))

(defMethod (next FifoQueue) () 
  (if (self 'contents?) (car (self 'contents))))

(defMethod (remove! FifoQueue) ()
  (define item (self 'next))
  (if item 
      (self 'contents! (cdr (self 'contents))))
  (if (null? lengthAccum) (set! lengthAccum (Accumulate 'new)))
  (lengthAccum 'update! (monitor 'clockTime) (self 'length))
  item)

(defMethod (show FifoQueue) ()
  (displayLine "   --- Q length: ---")
  (if (null? lengthAccum)
      (displayLine "q was never entered")
      (lengthAccum 'show (monitor 'clockTime)))
  (displayLine "   --- Q contents:---") 
  (if (self 'empty?)
      (displayLine "... Q is empty ...")
      (for-each (lambda (aMember)
                  (if (not (null? aMember))
                      (begin (aMember 'show) (newline))))
                (self 'contents)) ))

; ----- Server Flavours -----

; Server entities model capacity constrained resources. They are associated
; with a specified capacity and own a queue of transactions which need to
; perform some time consuming (sampled from "serviceDist") service activity. 

(defFlavour Server (ako Vanilla) 
  (ivars monitor capacity inUse utilizationAccum Q tracing?)
  setivars getivars testivars)

(defMethod (canAccomodate? Server) (aRequest) 
  (>=? (- capacity inUse) aRequest)) 

(defMethod (reset Server) () (utilizationAccum 'reset!))

(defMethod (acquire! Server) (aTA aRequest)
  (if (null? inUse) (set! inUse 0))
  (if (null? utilizationAccum) 
      (begin (set! utilizationAccum (UtilizationAccumulate 'new))
             (utilizationAccum 'capacity! (self 'capacity))))
  
  (aTA 'unitsRequested! aRequest)
  (if (or (null? aTA) (>? aRequest capacity))
      (begin (display "This request is invalid (> max. capacity) !")
             (display " - TA passivated"))
      (begin (if (self 'tracing??)
                 (begin (self 'announce aTA "arrives with a request of" aRequest)
                        (self 'announce nil "idle units are" (- capacity inUse))))
             (if (self 'canAccomodate? aRequest)
                 (self 'grab! aTa aRequest)
                 (begin (Q 'monitor! monitor)
                        (Q 'queue! aTA)
                        (if (self 'tracing??)
                            (begin (self 'announce aTA "queues" nil)
                                   (Q 'show)))
                        (aTA 'passivate))) )))
                 
(defMethod (grab! Server) (aTA someCapacity)
  (set! inUse (+ inUse someCapacity))  
  (utilizationAccum 'update! (monitor 'clockTime) inUse)
  
  (if (self 'tracing??)
      (begin (self 'announce aTA "grabs units:" someCapacity)
             (self 'announce nil "idle units are" (- capacity inUse)))
       ))

(defMethod (release! Server) (aTA someCapacity)
  (let ((waitingTA nil))
    (if (<? (- inUse someCapacity) 0)
        (self 'announce
              "This request is invalid (can't release what is not taken !)")
        (begin (set! inUse (- inUse someCapacity))
               (utilizationAccum 'update! (monitor 'clockTime) inUse)
               (if (self 'tracing??)
                   (begin (self 'announce aTA "releases units:" someCapacity)
                          (self 'announce nil "idle units are" (- capacity inUse))))
               (if (not (Q 'empty?))
                   (begin (set! waitingTA (Q 'remove!))
                          (if (self 'tracing?)
                              (begin (self 'announce waitingTA 
                                                     "unqueued and rescheduled"
                                                     nil)
                                     (Q 'show)))
                          (self 'grab! waitingTA 1)
                          (monitor 'schedule! waitingTA 
                                              (waitingTA 'activeMethod)
                                              (monitor 'clockTime)) ))
               (if (self 'tracing??) (self 'announce aTA "continues on" nil))
               (aTA 'proceed (aTA 'activeMethod))) )))
                 
(defMethod (show Server) ()
  (display "--- Server Statistics ---") (newline)
  (display "Clock time  : ") (display (monitor 'clockTime)) (newline)
  (display "Capacity    : ") (display capacity) (newline)
  (display "In Use      : ") (display inUse)    
  (newline) (newline)   
  (display "- Utilization    : ...") (newline)
  (if (not utilizationAccum)
      (begin (display "none") (newline))
      (utilizationAccum 'show (monitor 'clockTime)))
  (display "- Q statistics   : ...") (newline)
  (if (not Q) 
      (begin (display "none") (newline))
      (Q 'show)) )

(defMethod (announce Server) (aSubject aMessageString aValue)
    (displayLine self ": (at " (monitor 'clockTime) ") ")
    (displayLine "     " aSubject " " aMessageString " " aValue))
                        

(defFlavour UtilizationAccumulate (ako Accumulate) (ivars capacity)
  getivars setivars testivars)


   (defMethod (show UtilizationAccumulate) (clockTime)
     (let ((duration clockTime))
       (display "==> over ") (display duration)
       (display " time units ") (display obs)
       (display " observations were made.") (newline)
       (display "    Mean: ") 
       (display (if (not (or (zero? obs) (zero? duration)))
                    (/ (self 'finalIntegral duration)
                       duration)
                    0))
       (newline)
       (display "    Min : ") (display min) (newline)
       (display "    Max : ") (display max) (newline) ))
   
  
  
 