;=================== APPLICATIONS - Level ====================

; ***** Implementation of "BEHAVIOUR-TABLE"

(defMethod (showHierarchy System) (aTabLevel)
  
  (define (showSubsystems aSystemList someSpaces aTabLevel)
    (define (printSpaces someSpaces)
       (do ((i 0 (1+ i))) ((=? i someSpaces) nil) (display " ")))
    
    (if (null? aSystemList) 
        nil 
        (begin (printSpaces someSpaces)
               (display ((car aSystemList) 'name)) (newline)
               (showSubsystems ((car aSystemList) 'subSystems) 
                                (+ someSpaces aTabLevel) 
                                aTabLevel)
               (showSubsystems (cdr aSystemList) someSpaces aTabLevel))) )
    
  (display "++++++++++++++++++++++++++++++++++++")   (newline)
  (display "SUBSYSTEMS of: ") (display (self 'name)) (newline)
  (display "++++++++++++++++++++++++++++++++++++")   (newline)
  (if (self 'subsystems?)
      (showSubsystems (self 'subSystems) 0 aTabLevel)
      nil) )

(defMethod (allSubsystems System) ()
; returns a list of all the system's subsystems 
  
  (define (traceAllSubsystems aSystemCollection)
  ; returns a list of all the subsystems of all systems in a collection 
  ; (depth first search)
  
     (define (removeDuplicatesFrom aList)
       (if (member (car aList) (cdr aList))
           (removeDuplicatesFrom (cdr aList)) 
           (begin (cons (car aList)
                        (if (null? aList)
                             nil
                            (removeDuplicatesFrom (cdr aList)))))))
  
   (if (not (null? aSystemCollection))
       (begin (removeDuplicatesFrom 
               (append
                ; find the first system's subsystems 
                ((eval (car aSystemCollection)) 'subsystems)
                ; now find the subsystems of those
                (traceAllSubsystems
                   ((eval (car aSystemCollection)) 'subsystems))
                ; process the rest of the list
                (traceAllSubsystems (cdr aSystemCollection)) )))) ) 
   
  (traceAllSubsystems (list (self 'name))) )

(defMethod (nameAllSubsystems System) ()
  (for-each (lambda (aSystem) (display (aSystem 'name)) (newline))
            (self 'allSubsystems)) )

(defMethod (allStates System) ()
; returns a list of all the system's states 
  
  (define (traceAllStates aSystemCollection)
  ; returns a list of states for all systems in a collection 
  ; AND all their subsystems (depth first search)
    
      (if (null? aSystemCollection)
          nil
          (begin (list
                  ; find the first system's states 
                ((eval (car aSystemCollection)) 'states)
                ; recurse down the subsystems and return their states
                (if (not (null? ((eval (car aSystemCollection)) 'subsystems)))
                    (begin 
                     ((car 
                      ((eval (car aSystemCollection)) 'subsystems)) 'states)
                     (traceAllStates 
                      (cdr ((eval (car aSystemCollection)) 'subsystems))))) 
                ; now recurse down the tails of the collection
                (traceAllStates (cdr aSystemCollection)))) )) 
   
  (traceAllStates (list (self 'name))) )

(defFlavour BehaviourTable (ako Vanilla) 
  (ivars systemMatrix inputDistMatrix measurementMatrix)
  setivars getivars testivars)
; systemMatrix     : n*n coefficients - across all n states
; inputDistMatrix  : n*m coefficients - across all n states  & all m inputs
; measurementMatrix: k*n coefficients - across all k outputs & all n states 

(defMethod (addMatrix! BehaviourTable) (aName aMatrix) 
  (cond ((eq? aName 'system)
         (self 'systemMatrix!  aMatrix))
        ((eq? aName 'inputDist)
         (self 'inputDistMatrix! aMatrix))
        ((eq? aName 'measurement)
         (self 'measurementMatrix! aMatrix))) nil) 
  

(defMethod (show BehaviourTable) () 
  (display "-----      System Matrix        -----") (newline)
  (if (self 'systemMatrix?)
      ((self 'systemMatrix) 'show)
      (display "+++ empty !"))
  (newline) 
  (display "----- Input-Distribution Matrix -----") (newline)
  (if (self 'inputDistMatrix?)
      ((self 'inputDistMatrix) 'show) 
      (display "+++ empty !"))
  (newline)
  (display "-----    Measurement Matrix     -----") (newline)
  (if (self 'measurementMatrix?)
      ((self 'measurementMatrix) 'show) 
      (display "+++ empty !"))
  (newline)  nil)

; ***** Implementation of "COEFFICIENTMATRIX"

(defFlavour CoefficientMatrix (ako Vanilla) (ivars shape) getivars setivars testivars)
; shape: 2 dimensional table of coefficients (lists of expressions or 0)
; note: expressions are not (!) represented as objects
; --- used for all 3 components of BehaviourTables ---

(defMethod (addRow! CoefficientMatrix) (aRowList) 
  (self 'shape! (append (list aRowList) (self 'shape))))

(defMethod (addCoefficient! CoefficientMatrix) (aRow aCoefficient) 
  (append aRow (list aCoefficient)))

(defMethod (showRow CoefficientMatrix) (aList)
  (If (null? aList) 
      (display "--- no coefficients in row ---")
      (for-each (lambda (element) (display element) (display " "))
                 aList)) )

(defMethod (show CoefficientMatrix) () 
  (for-each (lambda (row) (self 'showRow row) (newline)) (self 'shape)) )

(defMethod (linearize System) ()
; linearizes a system's behaviour by partial differentiation across all 
; equations, state, input and output variables. 
; It returns a BehaviourTable object.
    
  (define table  (BehaviourTable 'new))
   
  (define (differentiateAcross someEqns someVars)
 
    (define dummy  (InfixExp          'new))   
    (define matrix (CoefficientMatrix 'new))   
    (define row    '())
    
    (display "*") (newline) 
    (for-each 
       (lambda (nextExp) 
               (for-each 
                  (lambda (nextVar)
                          (set! row (matrix 'addCoefficient! 
                                             row
                                            (dummy 'derive nextExp nextVar)))
                          (display "."))
                  someVars)
               (matrix 'addRow! row) 
               (set! row nil)
               (newline))
       someEqns) matrix)   
  
  ; process the systemMatrix
  (table 'addMatrix! 'system 
                     (differentiateAcross ((self 'behaviour) 'stateEqns) 
                                          ( self 'states)))  
  ; process the inputDistMatrix
  (table 'addMatrix! 'inputDist 
                     (differentiateAcross ((self 'behaviour) 'stateEqns) 
                                          ( self 'inputs)))                             
  ; process the measurementMatrix
  (table 'addMatrix! 'measurement
                     (differentiateAcross ((self 'behaviour) 'outputEqns) 
                                          ( self 'states)))
   table)
  
      