/*
 * gpmacros.c
 * General Purpose macros for use in graph generation and
 * static analysis
 *
 * Only macros to be used by numerous modules in the static-analysis
 * or graph generation should be placed here.
 *
 * CAVEATS:
 * The file pgm_typedefs.h MUST BE included BEFORE this header file
 *
 * AUTHOR:  David Roch
 *
 * CONTRIBUTORS:
 *	DAR - David Roch
 *
 * HISTORY:	06/21/88 - created DAR
 *		07/04/88 - list and comma macros added DAR
 *		07/12/88 - prototype macros added
 *
 * NOTES:
 *	User must declare the following as externals in his/her
 *	source code for some macros to function correctly.
 *		SLOT ListFunc
 *		SLOT NilAtom
 *		SLOT CommaFunc
 */


/* define if compiler supports prototyping */
/* #define PROTOTYPE 1 */

/*
 * UARGS((arguments))
 * Used to pass arguments to compilers that support prototyping
 * e.g. consider a function reverse(a, b) that takes a string
 * a, and reverses it, storing the result in b.
 * Normal prototying would be:
 *	extern void reverse(char *, char *)
 * UARGS lets us specify this regardless of whether or not the
 * compiler supports prototyping:
 *	extern void reverse UARGS((char *, char *))
 *	note the double ().  This is to make the two
 *	char *s one argument.  If PROTOTYPE is defined,
 *	then the char *s are included.  Otherwise a null
 *	substitution is performed.
 */

#ifdef PROTOTYPE
#   define UARGS(a) (a)
#   else
#      define UARGS(a) ()
#endif

/*
 * GETPROC("procedure_name", arity)
 * finds index into procedures for procedure_name of arity
 * Returns -1 if procedure/arity does not exist, does not
 * insert in symbol table.
 */
#define GETPROC(proc, arity) \
  niProcFor( PgmLookupAtom(proc), arity)

/*
 * CONS_CELL(SLOT tail)
 * CONS_CELL_PTR(SLOT *tail)
 * True if tail is the head of a list
 * Use CONS_CELL when tail is a SLOT,
 *     CONS_CELL_PTR when tail is a pointer to a slot
 */
#define CONS_CELL(tail) \
  GET_FUNCTOR((SLOT *) tail) == GET_FUNCTOR(&ListFunc)
#define CONS_CELL_PTR(tail) \
  GET_FUNCTOR(tail) == GET_FUNCTOR(&ListFunc)

/*
 * NEXT_CELL(SLOT tail)
 * NEXT_CELL_PTR(SLOT *tail)
 * Sets tail to the next cons cell in a list
 * Use NEXT_CELL if tail is a SLOT
 *     NEXT_CELL_PTR if tail is a pointer to a SLOT
 */
#define NEXT_CELL(tail)		tail += sizeof(SLOT) << 1
#define	NEXT_CELL_PTR(tail)	tail += 2

/*
 * NIL_ATOM(SLOT tail)
 * NIL_ATOM_PTR(SLOT *tail)
 * Tests whether or not tail is a NilAtom
 * Use NIL_ATOM if tail is a SLOT
 *     NIL_ATOM_PTR if tail is a pointer to a SLOT
 */
#define NIL_ATOM(tail) \
  GET_FUNCTOR((SLOT *) tail) == GET_FUNCTOR(&NilAtom)
#define NIL_ATOM_PTR(tail) \
  GET_FUNCTOR(tail) == GET_FUNCTOR(&NilAtom)

/*
 * COMMA(SLOT comma)
 * COMMA_PTR(SLOT *comma)
 * True if comma points to a comma functor
 * Use COMMA if comma is a SLOT
 *     COMMA_PTR if comma is apointer to a slot
 */
#define COMMA(comma) \
  GET_FUNCTOR((SLOT *) comma) == GET_FUNCTOR(&CommaFunc)
#define COMMA_PTR(comma) \
  GET_FUNCTOR(comma) == GET_FUNCTOR(&CommaFunc)

/*
 * VARIABLE(SLOT var)
 * VARIABLE_PTR(SLOT *var)
 * True if comma points to a variable
 * Use VARIABLE if var is a SLOT
 *     VARIABLE_PTR if var is apointer to a slot
 */
#define VARIABLE(var)		TAG_IS_TUPLE_INDEX_1((SLOT *) var)
#define VARIABLE_PTR(var)	TAG_IS_TUPLE_INDEX_1(var)
