/* File Det.c
 * 
 * Determinacy analysis
 */

#include "typedefs.h"
#include "pgm_typedefs.h"
#include "lattice.h"
#include "macros.c"
#include "pgm_macros.c"
#include "be_macros.c"

/* Maximum number of clauses for a single predicate */
#define MaxClauses 50

extern Procedure procedures[MaxProcs];
extern Procedure *procs;

/* Determinacy_Analysis()
 *
 * Check each predicate for determinacy.  Set predicate flag accordingly.
 * Note: Determinacy Analysis takes three passes over program.  This
 *       simplifies understanding of the algorithm, but will need to be
 *       changed to improve efficiency.
 */

Determinacy_Analysis()
   {
   Procedure *p;

/*   Set_Facts_Det(); */    /* this is incorrect - Ram 7/29/89 */
   PairwiseMutEx();

   /* Test for predicates with clauses mutex and deterministic. */
   for (p = procedures; p<procs; p++)
      {
      if (!Clauses_Det(p))
         {
         continue;
         }
      if (!IN_FLAG(p->flags,CLAUSES_MUT_EX))
         {
         continue;
         }

      /* Get to here?  ==> Predicate is Deterministic */
      SET_FLAG(p->flags,DETERMINISTIC);
      TRACE2(printf("%s/%d is Deterministic.\n",p->head,p->arity));
      }
   }

/* Clauses_Det()
 * 
 * Follow clause chain of a procedure to determine if all clauses
 *   have Deterministic bit set.
 */

Clauses_Det(p)
Procedure *p;
   {
   CLAUSE *c;
   SHORT All_Det;

   All_Det = TRUE;
   c = p->clauses;
   while (c != NULL)
      {
      if (! PgmIsClauseDet(c))
         {
         All_Det = FALSE;
         break;
         }
      c = c->next;
      }
   return All_Det;
   }

/* Set_Facts_Det()
 *
 * Pass over the entire program, setting Deterministic bits of
 *   clauses which are ground facts.  Check calling pattern
 *   to determine if fact is ground.  A fact is ground if all
 *   of the terms in the caling pattern are ground.
 */

Set_Facts_Det()
   {
   Procedure *p;
   CLAUSE *c;

   for (p=procedures; p<procs; p++)
      {
      c = p->clauses;
      while (c != NULL)
         {
         if (PgmIsFact(c))
            {
            PgmSetClauseDet(c);
            TRACE2(printf("Fact is deterministic\n"));
            } /* if (PgmIsFact(c)) */
         c = c->next;
         } /* while */
      } /* for */
   }


/* PairwiseMutEx()
 *
 * Determine, for each predicate, if its clauses are pairwise mutually
 *   exclusive.  If each pair of clauses are incompatible, then we KNOW
 *   that at most one of the clauses will succeed when called as described
 *   in the calling patterns.  If every pair of clauses are compatible,
 *   then we don't know anything, since our rules are not complete.
 */

PairwiseMutEx()
   {
   Procedure *p;
   CLAUSE *c[MaxClauses], *clause;
   SHORT i, j, num, MutEx;

   for (p=procedures; p<procs; p++)
      {
      if ((p->clauses != NULL) && (strcmp(p->head,"main$") != 0))
         {
         if (IN_FLAG(p->flags, CLAUSES_MUT_EX))
            {
            TRACE2(printf("Mutually Exclusive -- user defined\n"));
            continue; /* go to next predicate */
            }

         MutEx = TRUE; /* innocent until proven guilty */

       /* Collect pointers to all clauses for this predicate */
         clause = p->clauses;
         num = 0;
         while (clause != NULL)
            {
            if (num >= MaxClauses)
               {
               fprintf(stderr,"Too many clauses in predicate:  ");
               fprintf(stderr,"Change MaxClauses and re-compile.\n");
               MutEx = FALSE;
               }
            c[num++] = clause;
            clause = clause -> next;
            }

         if ((num > 1) && (MutEx == TRUE))
            {
            for (i=0; (i<num-1) && (MutEx == TRUE); i++)
               {
               for (j=i+1; (j<num) && (MutEx == TRUE); j++)
                  {
                  if (!Incompatible(p, c, i, j))
                     {
                     MutEx = FALSE;
                     }
                  }
               }
            } /* if (num > 1) ... */

         TRACE2(printf("Predicate %s/%d is ",p->head,p->arity));
         if (MutEx == TRUE)
            {
            SET_FLAG(p->flags,CLAUSES_MUT_EX);
            TRACE2(printf("Mutually Exclusive\n"));
            }
         else
            {
            /* Do nothing.  Non-mutually-exclusive is default, but 
             *   user may have specified predicate to be 
             *   mutually exclusive. This takes precedence.
             */
            TRACE2(printf("not Mutually Exclusive\n"));
            }
         } /* if ((p->clauses... */
      } /* for (p=procedures... */
   }

/* Incompatible()
 * 
 * Determine if a pair of clauses are incompatible.
 * Return: TRUE if a rule shows that clauses are incompatible.
 *       : FALSE if no reason to believe that clauses are incompatible.
 */

Incompatible(p, c, i, j)
Procedure *p;
CLAUSE *c[];
SHORT i, j;

{
   BOOLEAN Res, dummy;
   SHORT k, *junk, size1, size2;
   SLOT *head1, *head2, *CT1, *CT2, *tuple1[MaxVarTbl], *tuple2[MaxVarTbl];
   CALLSUCC *cs;
   CALSUCPR *cp;

  /* This may eat up a lot of memory.  tuple1 and tuple2 are 
   *   automatic variables which disappear when procedure 
   *   completes, but "malloced" space is still in existence.
   */
   size1 = ClauseNumVars(c[i]) + 1;
   size2 = ClauseNumVars(c[j]) + 1;
   if (size1 > 1)
      {
      Malloc_Slots(*tuple1, size1);
      INSERT_TUPLE_SIZE(*tuple1, size1);
      }
   else *tuple1 = NULL;
   if (size2 > 1)
      {
      Malloc_Slots(*tuple2, size2);
      INSERT_TUPLE_SIZE(*tuple2, size2);
      }
   else *tuple2 = NULL;

   cs = p->call_succ;
   while (cs != NULL)
      {
      cp = cs->patterns;
      head1 =  c[i]->lit_position[0];
      head2 =  c[j]->lit_position[0];
      DEREFPTR(head1);
      DEREFPTR(head2);
      TRACE2(printf("%s\n",procedures[GET_FUNCTOR(head1)].head));
      TRACE2(printf("%s\n",procedures[GET_FUNCTOR(head2)].head));

   /* Test each term in the head of one clause for incompatibility with 
    *   the corresponding term in the head of the other clause.
    */
      for (k=0; k<p->arity; k++)
         {
         CT1 = ++head1;
         CT2 = ++head2;
         if (lub_callpat(GROUND,cp[k].call) == GROUND)
	 {
            TRACE2(printf("Ground\n"));
            InitTuple((*tuple1 + 1), (size1 - 1));
            InitTuple((*tuple2 + 1), (size2 - 1));
 		/*PrintTerm(CT1,*tuple1);*/
 		/*PrintTerm(CT2,*tuple2);*/
            Res = UnifyPhaseI(CT1,*tuple1,CT2,*tuple2, &dummy, FALSE);
	    TRACE2(PrintTerm(ClauseLitPos(c[i],0), *tuple1));
	    TRACE2(PrintTerm(ClauseLitPos(c[j],0), *tuple2));
            if (Res == FALSE)
	    { /* Incompatible Clauses */
               TRACE2(printf("Does not Unify\n"));
               return TRUE;
	    }
            else TRACE2(printf("Does Unify\n"));
	 }
         else if (lub_callpat(PARTIAL,cp[k].call) == PARTIAL)
	 { 
            TRACE2(printf("Partial\n"));
            /*
            if (!Unify(TOP(CT1), TOP(CT2)))
               { * Incompatible Clauses *
               return 1;
               }
            */
	 }
      }  /* for (k ... */
      cs = cs->next;
   } /* while (cp != NULL) */

  /* As far as we know, clauses are compatible. */
   return FALSE;
}
