/************************************************************************/
/*									*/
/*			Single-Word Bit-Vectors				*/
/*									*/
/************************************************************************/

/* The following macros assume a bit-vector of one LONG word.
 * The word itself (not a pointer) is passed as an argument to the macros.
 */

/* Test vector for a '1' in position 'pos' */
#define BitInVector1W(vector,pos)	((vector) & (1<<(pos))) 

/* Clear the vector to all zeroes */
#define EmptyVector1W(BV)		BV = 0;

/************************************************************************/
/*									*/
/*			Multiple-Word Bit-Vectors			*/
/*									*/
/************************************************************************/

/*
 * list of macros:
 *	VectorAlloc
 *	ArrayAlloc
 *	EmptyVector - set to all zeros
 *	FillVector - set to all ones
 *	BitInVector - test for a set bit
 *	InsInVector - set a bit
 *	DelFromVector - clear a bit
 *	CopyVector - copy a bit
 *	VectorUnion - bit-wise OR
 *	VectorIntersect - bit-wise AND
 *	VectorDifference - set difference
 *	Invert - 1's complement
 *	VectorLength - length of vector in words
 */


/* The following macros assume a vector that could be of many words.
 *   The bit-vector passed as an argument to the macros is actually
 *   a pointer to a contiguous block of LONG words. The first 8 bits 
 *   of the first word pointed to by a bit-vector is a reserved field 
 *   containing the length of the vector-block (in words).  The 
 *   rest of the bits in the first word and all of the bits in the 
 *   remaining words are bit to be used.  VectorAlloc() and ArrayAlloc()
 *   assume that the user has declared a block of LONG words in the
 *   program from which the vectors are to be "allocated." An index
 *   should be declared to indicate how much of the block has already
 *   been allocated. Vectors should be declared as pointers.
 * 
 *   Example:
 *	LONG FreeBlock[MaxBlock];
 *	SHORT BlockTop = 0;
 *	VECTOR *BitVector, *BitVectorArray[ArrayLen];
 */

#define	BVLENMASK	0xFF
/* VectorAlloc()
 *
 * block is a pointer to the vector allocation block. block[top] is the 
 *   first free position in block. MaxBlock is the length of the vector
 *   allocation block.  BV is a pointer to a LONG word.  BV_Len is the 
 *   number of positions (bits) needed in the vector.  This is converted
 *   into the number of words needed and stored in the first 8 bits of 
 *   the vector.  Note: if you want to do vector operations on two vectors 
 *   (i.e. take the intersection, union, or set difference), these vectors 
 *   should be of the same length.  The "allocated" words in the vector
 *   allocation block will be initialized to an empty vector.
 * 
 * Example:
 * 	VectorAlloc(FreeBlock, BlockTop, MaxBlock, BitVector, 50)
 *   will "allocate" a vector within the FreeBlock that has room for 
 *   50 elements.
 */

#define VectorAlloc(block,top,MaxBlock,BV,BV_Len) \
{ SHORT bv_i, bv_WordsInVector;\
bv_WordsInVector = ( ((BV_Len) + 7) >> WSHIFT) + 1; \
if (((top)+bv_WordsInVector)<=MaxBlock) \
  { \
  for (bv_i=(top); bv_i<((top) + bv_WordsInVector); bv_i++) block[bv_i] = 0; \
  (BV) = &(block[top]);  \
  (top) += bv_WordsInVector; \
  *(BV) = bv_WordsInVector; \
  } \
else \
  { \
  fprintf(stderr,"Block Allocation Error -- No space\n"); \
  exit(1); \
  } \
}

/* ArrayAlloc()
 * 
 * block[top] specifies the first free position in the vector allocation
 *   block.  MaxBlock is the length of the allocation block.
 *   BVA (bit-vector array) is an array of pointers to LONG words.
 *   BV_Len is the number of elements in each bit-vector.
 *   Arr_Len is the number of elements in BVA.
 * The vectors will be initialized to be empty.
 * 
 * Example:
 *
 *   LONG FreeBlock[MaxBlock];
 *   SHORT BlockTop = 0;
 *   VECTOR *BitVectorArray[10]; *array of 10 vectors*
 *   
 *   ArrayAlloc(FreeBlock,BlockTop,MaxBlock,BVA,50,10);
 *
 *   This will allocate 10 vectors with 50 elements each.
 */ 

#define ArrayAlloc(block,top,MaxBlock,BVA,BV_Len,Arr_Len) \
{ \
SHORT bv_i, bv_j, bv_WordsInVector; \
bv_WordsInVector = ( ((BV_Len) + 7) >> WSHIFT) + 1; \
if (((top)+(Arr_Len)*(bv_WordsInVector)) <= (MaxBlock)) \
  { \
  for (bv_j=top; bv_j<(top + Arr_Len*bv_WordsInVector); bv_j++) \
     block[bv_j] = 0;\
  for (bv_i=0; bv_i<(Arr_Len); bv_i++) \
    { \
    BVA[bv_i] = &(block[top]); \
    (top) += bv_WordsInVector; \
    *(BVA[bv_i]) = bv_WordsInVector; \
    } \
  } \
else \
  { \
  fprintf(stderr,"Block Allocation Error -- No space\n"); \
  exit(1); \
  } \
}

/* The following macros assume that the vector has been allocated by
 *   using VectorAlloc() or ArrayAlloc().
 */

/* EmptyVector()
 *
 * Clear the vector to all zeroes.
 */

#define EmptyVector(BV) \
{ SHORT _i, _Len; \
_Len = VectorLength(BV); \
for (_i=0; _i<_Len; _i++) \
  { \
  *((BV) + _i) = 0; \
  } \
ReplaceLength(BV, _Len);\
}

/* FillVector()
 *
 * Set all the bits in the vector to 1.
 */

#define FillVector(BV) \
{ SHORT _i, _Len; \
_Len = VectorLength(BV); \
for (_i=0; _i<_Len; _i++) \
  { \
  *(BV + _i) = ~0; \
  } \
ReplaceLength(BV, _Len); \
}

/* BitInVector()
 * 
 * Tests position 'pos' in vector, after doing a vector bounds check.
 */

/* no error messages generated! */
#define BitInVector(BV, pos) \
(WithinVector(pos,VectorLength(BV)) && \
 ( *(BV + ((pos+8) >> WSHIFT)) & \
   (1 << ((pos+8) & WMASK)) \
 ) \
)

/* InsInVector()
 *
 * Insert (set to 1) a single bit in a bit-vector 
 *   after checking for bounds error.
 */ 

#define InsInVector(BV, pos) \
if ((pos+8) < (VectorLength(BV) << WSHIFT) && (pos >= 0) ) \
  (*(BV + ((pos+8) >> WSHIFT)) |= (1<<((pos+8)&WMASK))); \
else {fprintf(stderr,"VECTOR BOUNDS ERROR\n"); exit(1);}

/* DelFromVector()
 *
 * Delete (clear to 0) a single bit in a bit-vector 
 *   after checking for bounds error
 */ 

#define DelFromVector(BV, pos) \
if ((pos+8) < (VectorLength(BV) << WSHIFT) && (pos >= 0) ) \
  (*(BV + ((pos+8) >> WSHIFT)) &= ~(1<<((pos+8)&WMASK))); \
else {fprintf(stderr,"VECTOR BOUNDS ERROR\n"); exit(1);}

/* CopyVector()
 *
 * Copy the contents of one bit-vector to another.  Both vectors
 *   must have been "allocated" using VectorAlloc() or ArrayAlloc().
 */

#define CopyVector(From, To) \
{ SHORT _i; \
if (VectorLength(From) == VectorLength(To)) \
   { \
   for (_i=0; _i<VectorLength(From); _i++) \
     *(From + _i) = *(To + _i); \
   } \
else { \
   fprintf(stderr,"CopyVector: Error -- Vectors not of same length\n"); \
   } \
}

/* VectorUnion()
 *
 * Compute the Set Union of two bit-vectors (bit-wise OR). 
 *   BV = BV1 union BV2 
 */

#define VectorUnion(BV, BV1, BV2) \
{ SHORT _i, _Len; \
  if ( ((_Len=VectorLength(BV)) == VectorLength(BV1)) && \
       (_Len == VectorLength(BV2)) ) \
     { \
     for (_i=0; _i<_Len; _i++) \
       *(BV+_i) = *(BV1+_i) | *(BV2+_i); \
     } \
  else {\
     fprintf(stderr,"VectorUnion: Error -- Vectors not of same length\n"); \
     } \
}

/* VectorIntersect()
 *
 * Compute the intersection (bitwise AND) of two bit-vectors.
 *   BV = BV1 intersect BV2
 */ 

#define VectorIntersect(BV, BV1, BV2) \
{ SHORT _i, _Len; \
  if ( ((_Len=VectorLength(BV)) == VectorLength(BV1)) && \
       (_Len == VectorLength(BV2)) ) \
     { \
     for (_i=0; _i<_Len; _i++) \
       { \
       *(BV+_i) = *(BV1+_i) & *(BV2+_i); \
       } \
     } \
  else {\
     fprintf(stderr,"VectorIntersect: Error -- Vectors not of same length\n"); \
     } \
}

/* VectorDifference()
 *
 * Compute the set difference of two bit-vectors.
 *   BV = BV1 - BV2
 */

#define VectorDifference(BV, BV1, BV2) \
{ SHORT _i, _Len; \
  if ( ((_Len=VectorLength(BV)) == VectorLength(BV1)) && \
       (_Len == VectorLength(BV2)) ) \
     { \
     for (_i=0; _i<_Len; _i++) \
       { \
       *(BV + _i) = *(BV1 + _i) & ~(*(BV2 + _i)); \
       } \
     ReplaceLength(BV, _Len); \
     }\
  else {\
     fprintf(stderr,"VectorDifference: Error -- Vectors not of same length\n"); \
     } \
}

/* Invert()
 *
 * Invert each bit in a bit-vector.
 */

#define Invert(BV) \
{ SHORT _i, _Len; \
  _Len = VectorLength(BV); \
  *(BV) |= BVLENMASK; \
  for (_i=0; _i<_Len; _i++) \
    { \
    *((BV)+_i) = ~(*((BV)+_i)); \
    } \
  ReplaceLength(BV, _Len); \
}

/* Return Length of a Bit-Vector (number of words) */
#define VectorLength(BV) (*(BV) & BVLENMASK)

/* --------------------------------------------------*/
/* Don't call these.  They are used by other macros. */
/*
#define WithinVector(pos,Len) \
(( ((pos) >= 0) && ((pos + 8) < ((Len) << WSHIFT)) ) ? 1 : printf("error\n"),0)
*/

/* Check for vector bounds error */
#define WithinVector(pos,Len) \
( ((pos) >= 0) && ((pos + 8) < ((Len) << WSHIFT)) )

/* Replace the Length field of a bit-vector with Len */
#define ReplaceLength(BV, Len) \
{ \
*(BV) &= ~(BVLENMASK); \
*(BV) |= Len; \
}
