/*--------------------------------------------------------------------------*/
/*        Net-Clause Prolog Interpreter V2.0 (Standard Prolog + NCL)        */
/*--------------------------------------------------------------------------*/
/*    Copyright (C) 1993, Z. Markov, Institute of Informatics,              */
/*                  Bulgarian Academy of Sciences,                          */
/*                  Acad.G.Bonchev St. Block 29A, 1113 Sofia                */
/*                  Email: markov@iinf.bg                                   */
/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <signal.h>

#ifndef true
# define true    1
# define false   0
#endif

#define MaxFrames	10000  /* Max number of active goals */
#define LocSize		50000  /* Max number of variables in active goals */
#define MaxDepth	200
#define ReadSize	500
#define ReadDepth	250
#define WriteDepth	250
#define WriteLength	500
#define StringSpace	30000  /* Max space for atom and var characters */
#define AtomBase	0
#define AtomLimit	29000  /* Max number of characters in all atoms */
#define MaxAtomLength	1000
#define VarBase		29000
#define VarLimit	30000
#define HashSize	100
#define MaxVars		250
#define maxlength	255
#define MaxEvalArity	6
#define MaxPrec		1200
#define SubPrec		999
#define ordminchar	0
#define ordmaxchar	255

typedef unsigned char uchar;
typedef unsigned char boolean;
typedef short env;
typedef enum {funcT, intT, varT, anonT, skelT} nodetag;

typedef struct element {
   struct node *proc;
   struct element *chain;
} element;

typedef struct nodeinfo {
  nodetag tag;
  union {
    struct {
      long arity;
      struct atomentry *name;
      struct node *son;
    } U0;
    long ival;
    struct node *val;
    long offset;
  } UU;
} nodeinfo;

typedef struct net_node_record {
  struct node *count;
  long val;
  struct net_node_record *chain;
} net_node_record;

typedef struct node {
  struct node *brother, *chain;
  enum {globalF, localF, heapF} field;
  env scope;
  net_node_record *node_ptr;
  struct node *default_;
  nodeinfo info;
} node;

typedef long key;
typedef short strindex;

typedef struct clause {
  boolean Denied;
  long Nvars;
  node *head, *body;
  long keyval;
  boolean lazy;
  struct clause *chain;
} clause;

typedef struct net_clause_record {
  clause *cls;
  struct net_clause_record *chain;
} net_clause_record;

typedef struct strin {
  strindex index;
  short length;
} strin;

typedef enum {fxO, fyO, xfO, yfO, xfxO, xfyO, yfxO, nonO} optype;
typedef short prec;

typedef enum {normP, evalP} predtype;
typedef enum {
  callR, cutR, readR, writeR, get0R, putR, nlR, nameR, opR, abortR,
  endR, traceR, notraceR, atomR, integerR, varR, stepR, isP, ltR,
  assertaR, assertzR, assertR, listingR, resR, dprocR, functorR, argR,
  listR, nolistR, debugR, consultR, listallR, clenvR, uclR, advclR,
  advpR, zapR, openR, closeR, defR, shellR, dprogR, failR, errorR,
  nostepR, moreR, nomoreR, nodebugR, topR, trailR, eqvR, nprocR,
  netenvR, advnetR, unetR, genR, netmodeR, dellazyR
} evalpred;

typedef char evalarity;

typedef struct ddr_clause_record {
  clause *cls;
  struct ddr_clause_record *chain;
} ddr_clause_record;

typedef struct queue_record {
  ddr_clause_record *ddr_ptr;
  struct atomentry *name;
  clause *proc;
  struct queue_record *chain;
} queue_record;

typedef struct atomentry {
  strin ident;
  long atomno;
  struct atomentry *chain;
  optype oclass;
  prec oprec;
  boolean QUOTED, sys;
  ddr_clause_record *ddr_ptr;
  predtype pclass;
  union {
    clause *proc;
    struct {
      evalpred routine;
      evalarity arity;
    } U1;
  } UU;
} atomentry;

typedef struct trailentry {
  node *boundvar, *VarVal;
  net_node_record *ptr1, *ptr2;
  struct trailentry *chain;
} trailentry;

typedef enum {sysM, progM, userM} phase;
typedef enum {syntaxZ, abortZ, dieZ} moanaction;
typedef enum {goalD, provedD, redoD, failD} tracemessage;

typedef struct _REC_PVARTABLE {
  strin IDENT;
  node *ROOTVAR;
} _REC_PVARTABLE;

typedef struct _REC_display {
  node *Fcall;
  env Fenv, Fchoice;
  clause *Fclause;
  trailentry *Ftrail;
  node *Fglotop;
  long Fbase;
} _REC_display;

boolean interrupted;

char fstop, spac, first_char;
boolean tbool, SAVEVARS, OUTFLAG, FileEnded;
FILE *PROG;
uchar PVARCOUNT;
_REC_PVARTABLE PVARTABLE[MaxVars];
phase mode;
long netmode;
boolean gencontext, lazycontext, inlazy;
queue_record *qtop, *qbase;
char linebuf[maxlength];
uchar linelength;
uchar charpos;
boolean linelisted;
char stringbuf[StringSpace];
strindex atomhwm, varhwm;
strin newatom, newvar;
long atomcount;
atomentry *hashtable[HashSize];
atomentry *commaA, *nilA, *consA, *cutA, *semiA, *questionA, *arrowA,
		 *fxA, *fyA, *xfA, *yfA, *xfxA, *xfyA, *yfxA, *callA, *endA,
		 *plusA, *minusA, *timesA, *divideA, *modA, *negA, *trueA,
		 *failA, *repeatA, *topA, *debugA, *sinA, *atanA, *lnA, *expA,
		 *sqrtA, *curlyA, *netA, *nodeA, *defaultA, *ddrA, *eqA,
		 *niltA;
clause *andG, *or1G, *or2G;
trailentry *trailend;
net_clause_record *net_clause_base, *net_ptr;
long loctop;
env envtop, choicepoint;
node *glotop;
node *locstack[LocSize];

/* For env -1.. MaxFrames */
_REC_display display1[MaxFrames+2];
_REC_display *display = &display1[1];

boolean haltflag, tracing, listing, debugging, debug_state, step, more;
atomentry *pptr;
long hptr;

enum {funcU, intU, VTbindU, TVbindU, VVbindU, succeedU, failU
} Uaction[(long)anonT - (long)funcT + 1][(long)anonT - (long)funcT + 1];
enum {
  smallC, largeC, digitC, specialC, quoteC, stropC, lparC, rparC, braC,
  ketC, commaC, cutC, semiC, lcurlyC, rcurlyC, vbarC, spaceC, wierdC
} CharClass[256];

/* Function Prototypes */
void Moan();
void KillStacks();
static void setmode();
void doclose();
void TopLevel();
void DTERM();
void Bind();
static void WriteTerm();
void asc();
static node *Skeleton();
boolean Execute();
boolean Unify();
boolean CallEvalPred();
void GOAL();

void intrpt()
{ 
  signal(SIGINT,intrpt);
  interrupted = true;
}


void StartAtom()
{
  newatom.index = atomhwm;
  newatom.length = 0;
}


void AtomChar(c)
char c;
{
  if (newatom.index + newatom.length >= AtomLimit)
    Moan(2L, dieZ);
  newatom.length++;
  stringbuf[newatom.index + newatom.length - 1] = c;
}


boolean SameString(s1, s2)
strin s1, s2;
{
  long j;
  boolean same;

  if (s1.length != s2.length)
    return false;
  else {
    j = 0;
    same = true;
    while (j != s1.length && same) {
      j++;
      same = (stringbuf[s1.index + j - 1] == stringbuf[s2.index + j - 1]);
    }
    return same;
  }
}


atomentry *LookUp(QO)
boolean QO;
{
  char h;
  atomentry *a;
  boolean found;

  if (newatom.length >= 1) {
    h = (stringbuf[newatom.index] * 8 +
	 stringbuf[newatom.index + newatom.length - 1] + newatom.length) %
	HashSize + 1;
  } else
    h = 1;
  a = hashtable[h - 1];
  found = false;
  while (a != NULL && !found) {
    if (SameString(a->ident, newatom))
      found = true;
    else
      a = a->chain;
  }
  if (!found) {
	if ((a = (atomentry *)malloc(sizeof(atomentry))) == NULL)
       { printf("\nOut of Memory in InitAtoms\n"); exit(1); }
    atomcount++;
    a->ident = newatom;
    a->atomno = atomcount;
    a->chain = hashtable[h - 1];
    a->oclass = nonO;
    a->oprec = 0;
    a->sys = false;
    a->pclass = normP;
    a->UU.proc = NULL;
    a->ddr_ptr = NULL;
    atomhwm += newatom.length;
    hashtable[h - 1] = a;
  }
  a->QUOTED = QO;
  return a;
}


static void R(w, a)
char *w;
atomentry **a;
{
  char i;

  StartAtom();
  i = 1;
  while (w[i - 1] != ' ') {
    AtomChar(w[i - 1]);
    i++;
  }
  *a = LookUp(false);
}

static void S(w, m, p)
char *w;
evalarity m;
evalpred p;
{
  atomentry *a;

  R(w, &a);
  a->sys = true;
  a->pclass = evalP;
  a->UU.U1.routine = p;
  a->UU.U1.arity = m;
}


void InitAtoms()
{
  char j;

  atomhwm = AtomBase;
  atomcount = 0;
  for (j = 0; j < HashSize; j++)
    hashtable[j] = NULL;
  R(",       ", &commaA);
  R(";       ", &semiA);
  R(".       ", &consA);
  R("!       ", &cutA);
  R("[]      ", &nilA);
  R(":-      ", &arrowA);
  R("?-      ", &questionA);
  R("call    ", &callA);
  R("fail    ", &failA);
  R("repeat  ", &repeatA);
  R("end     ", &endA);
  R("fx      ", &fxA);
  R("fy      ", &fyA);
  R("xf      ", &xfA);
  R("yf      ", &yfA);
  R("xfx     ", &xfxA);
  R("xfy     ", &xfyA);
  R("yfx     ", &yfxA);
  R("+       ", &plusA);
  R("-       ", &minusA);
  R("*       ", &timesA);
  R("/       ", &divideA);
  R("mod     ", &modA);
  R("~       ", &negA);
  R("$top    ", &topA);
  R("true    ", &trueA);
  R("$debug  ", &debugA);
  R("sin     ", &sinA);
  R("atan    ", &atanA);
  R("exp     ", &expA);
  R("ln      ", &lnA);
  R("sqrt    ", &sqrtA);
  R(":       ", &netA);
  R("node    ", &nodeA);
  R("default ", &defaultA);
  R("=>      ", &ddrA);
  R("{}      ", &curlyA);
  R("=       ", &eqA);
  R("nil     ", &niltA);

  S("read    ", 1, readR);
  S("write   ", 1, writeR);
  S("get0    ", 1, get0R);
  S("put     ", 1, putR);
  S("nl      ", 0, nlR);
  S("op      ", 3, opR);
  S("atom    ", 1, atomR);
  S("integer ", 1, integerR);
  S("var     ", 1, varR);
  S("is      ", 2, isP);
  S("<       ", 2, ltR);
  S("asserta ", 1, assertaR);
  S("assertz ", 1, assertzR);
  S("$assert ", 2, assertR);
  S("functor ", 3, functorR);
  S("arg     ", 3, argR);
  S("abort   ", 0, abortR);
  S("list    ", 0, listR);
  S("nolist  ", 0, nolistR);
  S("trace   ", 0, traceR);
  S("notrace ", 0, notraceR);
  S("debug   ", 0, debugR);
  S("nodebug ", 0, nodebugR);
  S("step    ", 0, stepR);
  S("nostep  ", 0, nostepR);
  S("more    ", 0, moreR);
  S("nomore  ", 0, nomoreR);
  S("listing ", 1, listingR);
  S("close   ", 0, closeR);
  S("def     ", 2, defR);
  S("top     ", 1, topR);
  S("call    ", 1, callR);
  S("!       ", 0, cutR);
  S("end     ", 0, endR);
  S("name    ", 2, nameR);
  S("delete  ", 1, dprocR);
  S("consult ", 1, consultR);
  S("listall ", 0, listallR);
  S("$clenv  ", 2, clenvR);
  S("$ucl    ", 4, uclR);
  S("$advcl  ", 1, advclR);
  S("$zap    ", 1, zapR);
  S("open    ", 1, openR);
  S("$r      ", 0, resR);
  S("fail    ", 0, failR);
  S("dprog   ", 1, dprogR);
  S("$error  ", 1, errorR);
  S("nproc   ", 2, nprocR);
  S("$advp   ", 1, advpR);
  S("$netenv ", 0, netenvR);
  S("$advnet ", 0, advnetR);
  S("$unet   ", 1, unetR);
  S("gen     ", 1, genR);
  S("netmode ", 1, netmodeR);
  S("trail   ", 2, trailR);
  S("dellazy ", 0, dellazyR);
  S("eqv     ", 2, eqvR);
  S("shell   ", 1, shellR);

}


void InitRead()
{
  char c;
  short TEMP;

  fstop = '.';
  spac = ' ';
  for (TEMP = (char)ordminchar; TEMP <= (char)ordmaxchar; TEMP++) {
    c = TEMP;
    CharClass[c] = smallC;
  }
  for (TEMP = 128; TEMP <= 159; TEMP++) {
    c = TEMP;
    CharClass[c] = largeC;
  }
  for (c = 'A'; c <= 'Z'; c++)
  CharClass[c]   = largeC;
  CharClass['_'] = largeC;
  CharClass['0'] = digitC;
  CharClass['1'] = digitC;
  CharClass['2'] = digitC;
  CharClass['3'] = digitC;
  CharClass['4'] = digitC;
  CharClass['5'] = digitC;
  CharClass['6'] = digitC;
  CharClass['7'] = digitC;
  CharClass['8'] = digitC;
  CharClass['9'] = digitC;
  CharClass['+'] = specialC;
  CharClass['-'] = specialC;
  CharClass['*'] = specialC;
  CharClass['/'] = specialC;
  CharClass['<'] = specialC;
  CharClass['>'] = specialC;
  CharClass['='] = specialC;
  CharClass['\\'] = specialC;
  CharClass['@'] = smallC;
  CharClass[':'] = specialC;
  CharClass['.'] = specialC;
  CharClass['?'] = specialC;
  CharClass['~'] = specialC;
  CharClass['#'] = specialC;
  CharClass['$'] = specialC;
  CharClass['&'] = specialC;
  CharClass['%'] = specialC;
  CharClass['\''] = stropC;
  CharClass['"'] = quoteC;
  CharClass['('] = lparC;
  CharClass[')'] = rparC;
  CharClass['['] = braC;
  CharClass[']'] = ketC;
  CharClass[','] = commaC;
  CharClass['!'] = cutC;
  CharClass[';'] = semiC;
  CharClass['{'] = lcurlyC;
  CharClass['}'] = rcurlyC;
  CharClass['|'] = vbarC;
  CharClass[' '] = spaceC;
  CharClass['\t'] = spaceC;
  CharClass['\015'] = spaceC;
  CharClass['\n'] = spaceC;
}


void InitTrail()
{
  trailentry *WITH;

  if ((trailend = (trailentry *)malloc(sizeof(trailentry))) == NULL)
   { printf("\nOut of Memory in InitTrail\n"); exit(1); }
  WITH = trailend;
  WITH->boundvar = NULL;
  WITH->VarVal = NULL;
  WITH->ptr1 = NULL;
  WITH->ptr2 = NULL;
  WITH->chain = NULL;
}


void INITSTACKS()
{
  long N;
  node *V;

  glotop = NULL;
  envtop = 0;
  loctop = 0;
  for (N = 0; N < LocSize; N++) {
	 if ((V = (node *)malloc(sizeof(node))) == NULL)
    { printf("\nOut of Memory in InitStacks\n"); exit(1); }
    V->brother = NULL;
    V->chain = NULL;
    V->field = localF;
    locstack[N] = V;
  }
}


void InitUnify()
{
  Uaction[0][0] = funcU;
  Uaction[(long)intT - (long)funcT][(long)intT - (long)funcT] = intU;
  Uaction[(long)varT - (long)funcT][0] = VTbindU;
  Uaction[(long)varT - (long)funcT][(long)intT - (long)funcT] = VTbindU;
  Uaction[0][(long)varT - (long)funcT] = TVbindU;
  Uaction[(long)intT - (long)funcT][(long)varT - (long)funcT] = TVbindU;
  Uaction[(long)varT - (long)funcT][(long)varT - (long)funcT] = VVbindU;
  Uaction[(long)anonT - (long)funcT][0] = succeedU;
  Uaction[(long)anonT - (long)funcT][(long)intT - (long)funcT] = succeedU;
  Uaction[(long)anonT - (long)funcT][(long)varT - (long)funcT] = succeedU;
  Uaction[0][(long)anonT - (long)funcT] = succeedU;
  Uaction[(long)intT - (long)funcT][(long)anonT - (long)funcT] = succeedU;
  Uaction[(long)varT - (long)funcT][(long)anonT - (long)funcT] = succeedU;
  Uaction[(long)anonT - (long)funcT][(long)anonT - (long)funcT] = succeedU;
  Uaction[0][(long)intT - (long)funcT] = failU;
  Uaction[(long)intT - (long)funcT][0] = failU;
}


void INIT()
{
  InitAtoms();
  InitTrail();
  INITSTACKS();
  InitUnify();
}


void ListLine()
{
  long i, FORLIM;

  FORLIM = linelength-1;
  for (i = 0; i < FORLIM; i++)
    putchar(linebuf[i]);
  putchar('\n');
  linelisted = true;
}


void GetLine(f)
FILE **f;
{
   if (feof(*f)) Moan(13L, abortZ);
   charpos = 0;
   linelength = 1;
   linelisted = false;
   linebuf[0] = fgetc(*f);
   if (linebuf[0] != '\n') {
     do {linebuf[linelength] = fgetc(*f);
       if (feof(*f)) {FileEnded=1; break; }
     } while (!(linebuf[linelength++] == '\n'));
   }
   if (listing) ListLine();
}


void CheckLine()
{
  FILE *TEMP;

  if (charpos < linelength)
    return;
  switch (mode) {

  case sysM:
  case progM:
    GetLine(&PROG);
    break;

  case userM:
	 TEMP = stdin;
	 GetLine(&TEMP);
    break;
  }
}


void GetChar(ch)
char *ch;
{
  CheckLine();
  *ch = linebuf[charpos++];
}


boolean LineEnded()
{
  CheckLine();
  return (charpos == linelength);
}


void Abort()
{
  putchar('\n');
  switch (mode) {

  case sysM:
    exit(10);
	 break;

  case userM:
    KillStacks(0);
    setmode(userM);
    TopLevel();
    break;

  case progM:
    KillStacks(0);
    doclose();
    TopLevel();
    break;
  }
}


void _EscIO(i,name)
int i;
char name[255];
{
  switch (i) {

  case 10: 
    printf("\nI/O ERROR: Fail to open %s for reading\n", name);
    break;

  case 20:
    printf("\nI/O ERROR: Fail to open %s for writing\n", name);
  }
  Abort();
}


void Moan(e, A)
long e;
moanaction A;
{
  char ch, lastch;
  long i, FORLIM;

  if (mode == sysM && !debugging) {
    printf("\nINITIALIZATION ERROR\n");
    exit(10); 
  }
  if (A == syntaxZ) {
    if (!linelisted)
      ListLine();
    FORLIM = charpos - 2;
    for (i = 0; i <= FORLIM; i++) {
      if (linebuf[i] == '\t')
	putchar('\t');
      else
	putchar(' ');
    }
    printf("^\n");
    if (mode == userM)
      charpos = linelength;
    else {
      ch = linebuf[charpos - 1];
      do {
	lastch = ch;
	GetChar(&ch);
      } while (!((lastch == fstop && ch == ' ') | FileEnded));
    }
  }
  printf("ERROR %d: ",e);
  switch (e) {

  case 0:
    printf("Built-in predicate with illegal arity\n");
    break;

  case 1:
    printf("Illegal term is added to the database\n");
    break;

  case 2:
    printf("Atom table overflow. Too many and/or too long atoms\n");
    break;

  case 3:
    printf("Out of range ASCII code for \"put\"\n");
    break;

  case 4:
    printf("Missing \".\"\n");
    break;

  case 5:
    printf("Illegal delimiter (special character)\n");
    break;

  case 6:
    printf("Malformed arithmetic expression\n");
    break;

  case 7:
    printf("Illegal argument of the \"call\"\n");
    break;

  case 8:
    printf("Illegal argument of \"clause\" or \"retract\"\n");
    break;

  case 9:
    printf("Unclosed comment\n");
    break;

  case 10:
    printf("Illegal operator precedence\n");
    break;

  case 11:
    printf("Inadmissible nesting of terms (probably a cyclic term)\n");
    break;

  case 12:
    printf("Division by zero\n");
    break;

  case 13:
    printf("\"End_of_file\" marker before \".\"\n");
    break;

  case 14:
    printf("Inadmissible number of active goals\n");
    break;

  case 15:
    printf("Illegal argument of \"functor\"\n");
    break;

  case 16:
    printf("Using an illegal term as a goal\n");
    break;

  case 17:
    printf("Inadmissible number of variables in the active goals\n");
    break;

  case 18:
    printf("Inadmissible length of input string\n");
    break;

  case 19:
    printf("Illegal argument of the \"name\"\n");
    break;

  case 20:
    printf("Infix or postfix operator expected\n");
    break;

  case 21:
    printf("Closing quote expected\n");
    break;

  case 22:
    printf("Operand or prefix operator expected\n");
    break;

  case 23:
    printf("Too many variable names on reading\n");
    break;

  case 24:
    printf("Illegal argument of \"op\"\n");
    break;

  case 25:
    printf("Precedence violation\n");
    break;

  case 26:
    printf("Failure of a goal during consulting\n");
    break;

  case 27:
    printf("Inadmissible nesting of terms on input\n");
    break;

  case 28:
    printf("Inadmissible number of arguments in the input term\n");
    break;

  case 29:
    printf("An attempt was made to change a built-in predicate\n");
    break;

  case 30:
    printf("Too many and/or too long variables in a clause\n");
    break;

  case 31:
    printf("Inadmissible input symbol\n");
    break;

  case 33:
    printf("Illegal argument of \"consult\", \"open\" or \"shell\". Atom required\n");
    break;

  case 34:
    printf("Inadmissible number of symbols in a file name or shell argument\n");
    break;

  case 35:
    printf("Illegal term\n");
    break;

  case 37:
    printf("Illegal syntax for a spreading activation node\n");
    break;
  }
  if (A == dieZ)
    exit(10);   /* halt */
  else
    Abort();
}


/* For Net Clauses */
void TrailVar(v)
node *v;
{
  trailentry *p;

  if (v->scope >= choicepoint && v->node_ptr == NULL)
    return;
  if ((p = (trailentry *)malloc(sizeof(trailentry))) == NULL)
   { printf("\nOut of Memory in TrailVar\n"); exit(1); }
  p->boundvar = v;
  p->VarVal = NULL;
  p->ptr1 = NULL;
  p->ptr2 = NULL;
  p->chain = NULL;
  trailend->chain = p;
  trailend = p;
}


void TrimTrail(base)
trailentry *base;
{
  trailentry *p, *q;

  p = base;
  q = p->chain;
  while (q != NULL) {
/* For Net Clauses */
    if (q->boundvar->scope >= choicepoint && q->boundvar->node_ptr == NULL &&
	q->ptr1 == NULL && q->ptr2 == NULL) {
      p->chain = q->chain;
		free(q);
    } else
      p = q;
    q = p->chain;
  }
  trailend = p;
}


node *Deref(x, e)
node *x;
env e;
{
  node *y, *z;

  y = x;
  if (y->info.tag == skelT)
    y = locstack[display[e - 1].Fbase + y->info.UU.offset - 1];
  while (y->info.tag == varT) {
    z = y->info.UU.val;
    if (z == NULL)
      goto _L1;
    y = z;
  }
_L1:
  return y;
}


/* For Net Clauses */
void Untrail(newtrail)
trailentry *newtrail;
{
  trailentry *p, *q;
  net_node_record *n;
  node *WITH;
  nodeinfo *WITH2;

  p = newtrail->chain;
  while (p != NULL) {
    WITH = p->boundvar;
    if (WITH->node_ptr != NULL) {
      if (WITH->info.tag != varT) {
	n = WITH->node_ptr;
	while (n != NULL) {
	  n->count->info.UU.ival += n->val;
	  n = n->chain;
	}
	WITH->info.tag = varT;
	WITH->info.UU.val = NULL;
      }
    }
    p = p->chain;
  }
  trailend = newtrail;
  p = trailend->chain;
  trailend->chain = NULL;
  while (p != NULL) {
    WITH2 = &p->boundvar->info;
    if (p->ptr1 != NULL && WITH2->tag == varT) {
      p->boundvar->node_ptr = p->ptr1;
      if (WITH2->UU.val != NULL) {
	if (p->ptr2 == NULL)
	  WITH2->UU.val->node_ptr = NULL;
	else
	  p->ptr2->chain = NULL;
      }
    }
    WITH2->tag = varT;
    WITH2->UU.val = p->VarVal;
    q = p->chain;
	 free(p);
    p = q;
  }
}


void NewGlobal(x)
node **x;
{
  node *WITH;

  if ((*x = (node *)malloc(sizeof(node))) == NULL)
   { printf("\nOut of Memory in NewGlobal\n"); exit(1); }
  WITH = *x;
  WITH->brother = NULL;
  WITH->chain = glotop;
  WITH->field = globalF;
  WITH->node_ptr = NULL;
  WITH->default_ = NULL;
  WITH->scope = envtop;
  glotop = *x;
}


void NewEnv(e, callp, envp, clausep, nvars)
env *e;
node *callp;
env envp;
clause *clausep;
long nvars;
{
  long n;
  _REC_display *WITH;
  long FORLIM;
  node *WITH1;
  nodeinfo *WITH2;

  if (envtop >= MaxFrames)
    Moan(14L, abortZ);
  envtop++;
  *e = envtop;
  WITH = &display[*e - 1];
  WITH->Fcall = callp;
  WITH->Fenv = envp;
  WITH->Fchoice = choicepoint;
  WITH->Fclause = clausep;
  WITH->Ftrail = trailend;
  WITH->Fglotop = glotop;
  WITH->Fbase = loctop;
  if (loctop + nvars > LocSize)
    Moan(17L, abortZ);
  if (nvars > 0) {
    FORLIM = loctop + nvars;
    for (n = loctop; n < FORLIM; n++) {
      WITH1 = locstack[n];
      WITH2 = &WITH1->info;
      WITH1->scope = *e;
      WITH1->node_ptr = NULL;
      WITH1->default_ = NULL;
      WITH2->tag = varT;
      WITH2->UU.val = NULL;
    }
  }
  loctop += nvars;
}


void GetEnv(e, callp, envp, clausep)
env e;
node **callp;
env *envp;
clause **clausep;
{
  _REC_display *WITH;

  WITH = &display[e - 1];
  *callp = WITH->Fcall;
  *envp = WITH->Fenv;
  *clausep = WITH->Fclause;
}


void DisposeEnv()
{
  /* For Net Clauses */
  /* loctop:=display[envtop].Fbase; */
  envtop--;
}


void Cut(e)
env e;
{
  env envp;
  clause *cl;
  _REC_display *WITH;

  envp = e;
  cl = display[envp - 1].Fclause;
  while (display[envp - 1].Fchoice > 0 &&
	 (cl == andG || cl == or1G || cl == or2G || cl == NULL)) {
    envp = display[envp - 1].Fenv;
    cl = display[envp - 1].Fclause;
  }
  WITH = &display[envp - 1];
  choicepoint = WITH->Fchoice;
  TrimTrail(WITH->Ftrail);
  if (envtop > e) {
    /* For Net Clauses */
    /* loctop:=display[e + 1].Fbase; */
    envtop = e;
  }
}


void KillStacks(newtop)
env newtop;
{
  node *p, *q;
  _REC_display *WITH;

  if (envtop <= newtop)
    return;
  p = glotop;
  WITH = &display[newtop];
  Untrail(WITH->Ftrail);
  choicepoint = WITH->Fchoice;
  glotop = WITH->Fglotop;
  loctop = WITH->Fbase;
  while (p != glotop) {
    q = p->chain;
	 free(p);
    p = q;
  }
  envtop = newtop;
}


node *EnvRef(offset, e)
long offset;
env e;
{
  return (locstack[display[e - 1].Fbase + offset - 1]);
}


void DTERM(T)
node **T;
{
  node *Q, *P;

  P = *T;
  while (P != NULL) {
    if (P->info.tag == funcT)
      DTERM(&P->info.UU.U0.son);
    Q = P->brother;
	 free(P);
    P = Q;
  }
}


node *MakeFunc(a, m, s)
atomentry *a;
long m;
node *s;
{
  node *x;
  nodeinfo *WITH;

  NewGlobal(&x);
  WITH = &x->info;
  WITH->tag = funcT;
  WITH->UU.U0.name = a;
  WITH->UU.U0.arity = m;
  WITH->UU.U0.son = s;
  return x;
}


node *MakeInt(i)
long i;
{
  node *x;
  nodeinfo *WITH;

  NewGlobal(&x);
  WITH = &x->info;
  WITH->tag = intT;
  WITH->UU.ival = i;
  return x;
}


node *MakeVar(v)
node *v;
{
  node *x;
  nodeinfo *WITH;

  NewGlobal(&x);
  WITH = &x->info;
  WITH->tag = varT;
  WITH->UU.val = v;
  return x;
}


boolean IsFunc(x, a, m)
node *x;
atomentry *a;
long m;
{
  nodeinfo *WITH;

  WITH = &x->info;
  if (WITH->tag != funcT)
    return false;
  else
    return (WITH->UU.U0.name == a && WITH->UU.U0.arity == m);
}


boolean IsAtom(x)
node *x;
{
  nodeinfo *WITH;

  WITH = &x->info;
  if (WITH->tag != funcT)
    return false;
  else
    return (WITH->UU.U0.arity == 0);
}


/* For Net Clauses */
void BindVars(v1, v2)
node *v1, *v2;
{
  net_node_record *n;

  if (v1 == v2)
    return;
  if ((long)v1->field > (long)v2->field ||
      v1->field == v2->field && v1->scope > v2->scope) {
    v1->info.UU.val = v2;
    TrailVar(v1);
    if (v1->node_ptr == NULL)
      return;
    if (v2->node_ptr == NULL) {
      trailend->ptr2 = NULL;
      v2->node_ptr = v1->node_ptr;
    } else {
      n = v2->node_ptr;
      while (n->chain != NULL)
	n = n->chain;
      trailend->ptr2 = n;
      n->chain = v1->node_ptr;
    }
    trailend->ptr1 = v1->node_ptr;
    v1->node_ptr = NULL;
    return;
  }
  v2->info.UU.val = v1;
  TrailVar(v2);
  if (v2->node_ptr == NULL)
    return;
  if (v1->node_ptr == NULL) {
    trailend->ptr2 = NULL;
    v1->node_ptr = v2->node_ptr;
  } else {
    n = v1->node_ptr;
	 while (n->chain != NULL)
      n = n->chain;
    trailend->ptr2 = n;
    n->chain = v2->node_ptr;
  }
  trailend->ptr1 = v2->node_ptr;
  v2->node_ptr = NULL;
}


struct LOC_Bind {
  env e;
  long depth;
} ;


static node *Copy(x, LINK)
node *x;
struct LOC_Bind *LINK;
{
  node *y, *z;

  y = Deref(x, LINK->e);
  switch (y->info.tag) {

  case funcT:
  case intT:
    NewGlobal(&z);
    Bind(&z->info, y, LINK->e, LINK->depth + 1);
    break;

  case varT:
    z = MakeVar(NULL);
    BindVars(y, z);
    break;

  case anonT:
    z = MakeVar(NULL);
    break;
  }
  return z;
}

static node *CopyArgs(s, LINK)
node *s;
struct LOC_Bind *LINK;
{
  node *t, *u, *v;

  if (s == NULL)
    return NULL;
  else {
    u = Copy(s, LINK);
    t = s->brother;
    v = u;
    while (t != NULL) {
      v->brother = Copy(t, LINK);
      t = t->brother;
      v = v->brother;
    }
    return u;
  }
}


void Bind(v, x, e_, depth_)
nodeinfo *v;
node *x;
env e_;
long depth_;
{
  struct LOC_Bind V;
  node *y;

  V.e = e_;
  V.depth = depth_;
  if (V.depth > MaxDepth)
    Moan(11L, abortZ);
  y = Deref(x, V.e);
  if (y->info.tag != funcT || y->field != heapF) {
    *v = y->info;
    return;
  }
  v->tag = funcT;
  v->UU.U0.name = y->info.UU.U0.name;
  v->UU.U0.arity = y->info.UU.U0.arity;
  v->UU.U0.son = CopyArgs(y->info.UU.U0.son, &V);
}


void GetBody(v, b, e)
nodeinfo *v;
node *b;
env e;
{
  node *l, *r;

  if (b->brother == NULL) {
    Bind(v, b, e, 0L);
    return;
  }
  NewGlobal(&l);
  NewGlobal(&r);
  l->brother = r;
  v->tag = funcT;
  v->UU.U0.name = commaA;
  v->UU.U0.arity = 2;
  v->UU.U0.son = l;
  Bind(&l->info, b, e, 0L);
  GetBody(&r->info, b->brother, e);
}


void WriteAtom(F, a)
FILE **F;
atomentry *a;
{
  strindex n;
  strin *WITH;
  strindex FORLIM;

  WITH = &a->ident;
  if (a->QUOTED && OUTFLAG)
    putc('\'', *F);
  FORLIM = WITH->index + WITH->length;
  for (n = WITH->index; n < FORLIM; n++)
    putc(stringbuf[n], *F);
  if (a->QUOTED && OUTFLAG)
    putc('\'', *F);
}


node *ListRep(s)
strin s;
{
  node *x, *y;
  strindex n;

  x = MakeFunc(nilA, 0L, NULL);
  for (n = s.index + s.length - 1; n >= s.index; n--) {
    y = MakeInt((long)stringbuf[n]);
    y->brother = x;
    x = MakeFunc(consA, 2L, y);
  }
  return x;
}


void StartVar()
{
  newvar.index = varhwm;
  newvar.length = 0;
}


void VarChar(c)
char c;
{
  if (newvar.index + newvar.length >= VarLimit)
    Moan(30L, abortZ);
  newvar.length++;
  stringbuf[newvar.index + newvar.length - 1] = c;
}


void KeepVar()
{
  varhwm += newvar.length;
}


void InitVars()
{
  varhwm = VarBase;
}


prec Lprec(a)
atomentry *a;
{
  return (a->oprec - (((1L << ((long)a->oclass)) & ((1L << ((long)xfO)) |
			 (1L << ((long)xfxO)) | (1L << ((long)xfyO)))) != 0));
}


prec Rprec(a)
atomentry *a;
{
  return (a->oprec - (((1L << ((long)a->oclass)) & ((1L << ((long)fxO)) |
			 (1L << ((long)xfxO)) | (1L << ((long)yfxO)))) != 0));
}


typedef enum {outerK, innerK, funcK, listK, endlistK, curlyK, finalK
} readstate;
typedef long stateset;

typedef enum { termL, opL, funcL, markL} elemtag;
typedef elemtag atomtag;

typedef struct _REC_stack {
  elemtag tag;
  union {
    node *tval;
    atomentry *aval;
  } UU;
} _REC_stack;

typedef struct _REC_statestack {
  readstate Scontext;
  prec Shiprec;
} _REC_statestack;

typedef struct _REC_vartable {
  strin ident;
  node *rootvar;
} _REC_vartable;

struct LOC_ReadIn {
  char ch;
  readstate context;
  enum {
    opX, randX
  } expected;
  prec hiprec, loprec;
  short top;
  _REC_stack stack[ReadSize];
  uchar statetop;
  _REC_statestack statestack[ReadDepth];
  uchar varcount;
  _REC_vartable vartable[MaxVars];
} ;


static void Push(t, LINK)
elemtag t;
struct LOC_ReadIn *LINK;
{
  if (LINK->top >= ReadSize)
    Moan(28L, syntaxZ);
  LINK->top++;
  LINK->stack[LINK->top - 1].tag = t;
}


static void ShiftTerm(x, LINK)
node *x;
struct LOC_ReadIn *LINK;
{
  Push(termL, LINK);
  LINK->stack[LINK->top - 1].UU.tval = x;
}


static void Shift(t, a, LINK)
atomtag t;
atomentry *a;
struct LOC_ReadIn *LINK;
{
  Push(t, LINK);
  LINK->stack[LINK->top - 1].UU.aval = a;
}


static node *Pop(LINK)
struct LOC_ReadIn *LINK;
{
  node *Result;

  Result = LINK->stack[LINK->top - 1].UU.tval;
  LINK->top--;
  return Result;
}


static void Reduce(p, lp, LINK)
prec p, lp;
struct LOC_ReadIn *LINK;
{
  node *x, *y;
  atomentry *a;
  boolean reduced;

  x = Pop(LINK);
  reduced = false;
  while (LINK->stack[LINK->top - 1].tag == opL && !reduced) {
    a = LINK->stack[LINK->top - 1].UU.aval;
    switch ((Rprec(a) >= p) - (a->oprec <= lp)) {

    case 1:
      reduced = true;
      break;

    case 0:
      Moan(10L, syntaxZ);
      break;

    case -1:
      LINK->top--;
      switch (a->oclass) {

      case fxO:
      case fyO:
	x = MakeFunc(a, 1L, x);
	break;

      case xfxO:
      case xfyO:
      case yfxO:
	y = Pop(LINK);
	y->brother = x;
	x = MakeFunc(a, 2L, y);
	break;
      }
      break;
    }
  }
  ShiftTerm(x, LINK);
}


static void CheckDelim(s, LINK)
stateset s;
struct LOC_ReadIn *LINK;
{
  atomentry *a;

  if (LINK->expected == randX) {
    if (LINK->stack[LINK->top - 1].tag != opL)
      Moan(22L, syntaxZ);
    a = LINK->stack[LINK->top - 1].UU.aval;
    if (((1L << ((long)a->oclass)) &
	 ((1L << ((long)fxO)) | (1L << ((long)fyO)))) == 0)
      Moan(22L, syntaxZ);
    LINK->top--;
    ShiftTerm(MakeFunc(a, 0L, NULL), LINK);
  }
  if (((1L << ((long)LINK->context)) & s) == 0)
    Moan(5L, syntaxZ);
  Reduce(MaxPrec, MaxPrec, LINK);
}


static void EnterContext(k, h, LINK)
readstate k;
prec h;
struct LOC_ReadIn *LINK;
{
  _REC_statestack *WITH;

  if (LINK->statetop >= ReadDepth)
    Moan(27L, syntaxZ);
  LINK->statetop++;
  WITH = &LINK->statestack[LINK->statetop - 1];
  WITH->Scontext = LINK->context;
  WITH->Shiprec = LINK->hiprec;
  LINK->context = k;
  LINK->expected = randX;
  LINK->hiprec = h;
}


static void ExitContext(x, LINK)
node *x;
struct LOC_ReadIn *LINK;
{
  _REC_statestack *WITH;

  LINK->top--;
  ShiftTerm(x, LINK);
  WITH = &LINK->statestack[LINK->statetop - 1];
  LINK->context = WITH->Scontext;
  LINK->hiprec = WITH->Shiprec;
  LINK->statetop--;
  LINK->expected = opX;
  LINK->loprec = 0;
}


static node *GetFunc(LINK)
struct LOC_ReadIn *LINK;
{
  node *x, *y;
  long n;

  x = Pop(LINK);
  n = 1;
  while (LINK->stack[LINK->top - 1].tag == termL) {
    y = Pop(LINK);
    y->brother = x;
    x = y;
    n++;
  }
  return (MakeFunc(LINK->stack[LINK->top - 1].UU.aval, n, x));
}


static node *GetList(LINK)
struct LOC_ReadIn *LINK;
{
  node *x, *y;

  x = Pop(LINK);
  do {
    y = Pop(LINK);
    y->brother = x;
    x = MakeFunc(consA, 2L, y);
  } while (LINK->stack[LINK->top - 1].tag == termL);
  return x;
}


struct LOC_STOWATOM {
  struct LOC_ReadIn *LINK;
  atomentry *A;
} ;


static void SQUASHRAND(LINK)
struct LOC_STOWATOM *LINK;
{
  prec p, lp;

  p = LINK->A->oprec;
  lp = Lprec(LINK->A);
  if (lp < LINK->LINK->loprec ||
      (p > SubPrec &&
       ((1L << ((long)LINK->LINK->context)) & ((1L << ((long)outerK)) |
	  (1L << ((long)innerK)) | (1L << ((long)curlyK)))) == 0))
	 Moan(25L, syntaxZ);
  Reduce(p, lp, LINK->LINK);
}


static void STOWATOM(A_, LINK)
atomentry *A_;
struct LOC_ReadIn *LINK;
{
  struct LOC_STOWATOM V;

  V.LINK = LINK;
  V.A = A_;
  switch (LINK->expected) {

  case randX:
    if (LINK->ch == '(') {
      Shift(funcL, V.A, LINK);
      EnterContext(funcK, SubPrec, LINK);
      GetChar(&LINK->ch);
    } else if (((1L << ((long)V.A->oclass)) &
		((1L << ((long)fxO)) | (1L << ((long)fyO)))) != 0) {
      if (V.A->oprec > LINK->hiprec)
	Moan(25L, syntaxZ);
      Shift(opL, V.A, LINK);
      LINK->expected = randX;
      LINK->hiprec = Rprec(V.A);
    } else {
      ShiftTerm(MakeFunc(V.A, 0L, NULL), LINK);
      LINK->expected = opX;
      LINK->loprec = 0;
    }
    break;

  case opX:
    switch (V.A->oclass) {

    case xfO:
    case yfO:
      SQUASHRAND(&V);
      ShiftTerm(MakeFunc(V.A, 1L, Pop(LINK)), LINK);
      LINK->expected = opX;
      LINK->loprec = V.A->oprec;
      break;

    case xfxO:
    case xfyO:
    case yfxO:
      SQUASHRAND(&V);
      Shift(opL, V.A, LINK);
      LINK->expected = randX;
      LINK->hiprec = Rprec(V.A);
      break;

    case fxO:
    case fyO:
    case nonO:
      Moan(20L, syntaxZ);
      break;
    }
    break;
  }
}


static long ScanInt(LINK)
struct LOC_ReadIn *LINK;
{
  long n = 0;

  do {
    n = n * 10 + LINK->ch - '0';
    GetChar(&LINK->ch);
  } while (CharClass[LINK->ch] == digitC);
  return n;
}


static void ScanQuote(q, LINK)
char q;
struct LOC_ReadIn *LINK;
{
  boolean done;

  StartAtom();
  done = false;
  do {
    if (LineEnded())
      Moan(21L, syntaxZ);
    GetChar(&LINK->ch);
    if (LINK->ch == q) {
      GetChar(&LINK->ch);
      done = (LINK->ch != q);
    }
    if (!done)
      AtomChar(LINK->ch);
  } while (!done);
}


static node *EnterVar(LINK)
struct LOC_ReadIn *LINK;
{
  node *v;
  uchar n;
  boolean found;
  _REC_vartable *WITH;

  n = 0;
  found = false;
  while (n != LINK->varcount && !found) {
    n++;
    found = SameString(LINK->vartable[n - 1].ident, newvar);
  }
  if (found)
    return (MakeVar(LINK->vartable[n - 1].rootvar));
  else {
    if (LINK->varcount >= MaxVars)
      Moan(23L, syntaxZ);
    LINK->varcount++;
    KeepVar();
    v = MakeVar(NULL);
    WITH = &LINK->vartable[LINK->varcount - 1];
    WITH->ident = newvar;
    WITH->rootvar = v;
    return v;
  }
}


node *ReadIn()
{
     FILE *TEMP;
  struct LOC_ReadIn V;
  node *Result;
  uchar i;
  char lastch;
  uchar FORLIM;

  V.top = 0;
  V.statetop = 0;
  InitVars();
  V.varcount = 0;
  Push(markL, &V);
  V.context = outerK;
  V.expected = randX;
  V.hiprec = MaxPrec;
  V.ch = spac;
  do {
    if (FileEnded && V.top == 1) {
		/* End of file - represented by ?- end. */
		Result = MakeFunc(questionA, 1L, MakeFunc(endA, 0L, NULL));
      V.context = finalK;
    } else {
      switch (CharClass[V.ch]) {

      case smallC:
	StartAtom();
	do {
	  AtomChar(V.ch);
	  GetChar(&V.ch);
	} while (((1L << ((long)CharClass[V.ch])) & ((1L << ((long)smallC)) |
		    (1L << ((long)largeC)) | (1L << ((long)digitC)))) != 0);
	STOWATOM(LookUp(false), &V);
	break;

      case stropC:
	ScanQuote('\'', &V);
	STOWATOM(LookUp(true), &V);
	break;

      case quoteC:
	if (V.expected == opX)
	  Moan(29L, syntaxZ);
	ScanQuote('"', &V);
	ShiftTerm(ListRep(newatom), &V);
	V.expected = opX;
	V.loprec = 0;
	break;

      case specialC:
	lastch = V.ch;
	GetChar(&V.ch);
	if (lastch == '/' && V.ch == '*') {
	  GetChar(&V.ch);
	  do {
	    lastch = V.ch;
	    GetChar(&V.ch);
	  } while (lastch != '*' || V.ch != '/');
	  GetChar(&V.ch);
	} else if (lastch == '~' && CharClass[V.ch] == digitC) {
	  if (V.expected == opX)
	    Moan(20L, syntaxZ);
	  ShiftTerm(MakeInt(-ScanInt(&V)), &V);
	  V.expected = opX;
	  V.loprec = 0;
	} else if (lastch == fstop && CharClass[V.ch] == spaceC) {
	  CheckDelim(1L << ((long)outerK), &V);
	  Result = Pop(&V);
	  V.context = finalK;
	} else {
	  StartAtom();
	  AtomChar(lastch);
	  while (CharClass[V.ch] == specialC) {
	    AtomChar(V.ch);
	    GetChar(&V.ch);
	  }
	  STOWATOM(LookUp(false), &V);
	}
	break;

      case largeC:
	if (V.expected == opX)
	  Moan(20L, syntaxZ);
	lastch = V.ch;
	GetChar(&V.ch);
	if (lastch == '_' &&
	    ((1L << ((long)CharClass[V.ch])) & ((1L << ((long)smallC)) |
	       (1L << ((long)largeC)) | (1L << ((long)digitC)))) == 0)
	  ShiftTerm(MakeVar(NULL), &V);
	else {
	  StartVar();
	  VarChar(lastch);
	  while (((1L << ((long)CharClass[V.ch])) & ((1L << ((long)smallC)) |
		    (1L << ((long)largeC)) | (1L << ((long)digitC)))) != 0) {
	    VarChar(V.ch);
	    GetChar(&V.ch);
	  }
	  ShiftTerm(EnterVar(&V), &V);
	}
	V.expected = opX;
	V.loprec = 0;
	break;

      case digitC:
	if (V.expected == opX)
	  Moan(20L, syntaxZ);
	ShiftTerm(MakeInt(ScanInt(&V)), &V);
	V.expected = opX;
	V.loprec = 0;
	break;

      case lparC:
	if (V.expected == opX)
	  Moan(20L, syntaxZ);
	Push(markL, &V);
	EnterContext(innerK, MaxPrec, &V);
	GetChar(&V.ch);
	break;

      case rparC:
	CheckDelim((1L << ((long)innerK)) | (1L << ((long)funcK)), &V);
	switch (V.context) {

	case innerK:
	  ExitContext(Pop(&V), &V);
	  break;

	case funcK:
	  ExitContext(GetFunc(&V), &V);
	  break;
	}
	GetChar(&V.ch);
	break;

      case braC:
	GetChar(&V.ch);
	if (V.ch == ']') {
	  GetChar(&V.ch);
	  STOWATOM(nilA, &V);
	} else {
	  if (V.expected == opX)
	    Moan(20L, syntaxZ);
	  Push(markL, &V);
	  EnterContext(listK, SubPrec, &V);
	}
	break;

      case ketC:
	CheckDelim((1L << ((long)listK)) | (1L << ((long)endlistK)), &V);
	if (V.context == listK)
	  ShiftTerm(MakeFunc(nilA, 0L, NULL), &V);

	ExitContext(GetList(&V), &V);
	GetChar(&V.ch);
	break;

      case commaC:
	GetChar(&V.ch);
	if (V.ch == '.') {
	  GetChar(&V.ch);
	  if (V.ch != '.')
	    Moan(4L, syntaxZ);
	  V.ch = ']';
	} else {
	  if (((1L << ((long)V.context)) & ((1L << ((long)outerK)) |
		 (1L << ((long)innerK)) | (1L << ((long)curlyK)))) != 0)
	    STOWATOM(commaA, &V);
	  else {
	    CheckDelim((1L << ((long)funcK)) | (1L << ((long)listK)), &V);
	    V.expected = randX;
	    V.hiprec = SubPrec;
	  }
	}
	break;

      case cutC:
	GetChar(&V.ch);
	STOWATOM(cutA, &V);
	break;

      case semiC:
	GetChar(&V.ch);
	STOWATOM(semiA, &V);
	break;

      case vbarC:
	CheckDelim(1L << ((long)listK), &V);
	V.context = endlistK;
	V.expected = randX;
	V.hiprec = SubPrec;
	GetChar(&V.ch);
	break;

      case spaceC:
	GetChar(&V.ch);
	break;

      case wierdC:
	Moan(31L, syntaxZ);
	break;
      }
    }
  } while (V.context != finalK);
  if (!SAVEVARS)
    return Result;
  SAVEVARS = false;
  PVARCOUNT = V.varcount;
  FORLIM = V.varcount;
  for (i = 0; i < FORLIM; i++) {
    PVARTABLE[i].IDENT = V.vartable[i].ident;
    PVARTABLE[i].ROOTVAR = V.vartable[i].rootvar;
  }
  return Result;
}


struct LOC_WriteOut {
  FILE **F;
  env e;
  node *varmap[MaxVars];
  uchar varcount;
} ;


static void WRITE1(CH, LINK)
char CH;
struct LOC_WriteOut *LINK;
{
  putc(CH, *LINK->F);
}


struct LOC_WriteTerm {
  struct LOC_WriteOut *LINK;
  prec p;
  long depth;
  node *y;
} ;


static void WriteStand(LINK)
struct LOC_WriteTerm *LINK;
{
  node *s;
  nodeinfo *WITH;

  WITH = &LINK->y->info;
  WriteAtom(LINK->LINK->F, WITH->UU.U0.name);
  WRITE1('(', LINK->LINK);
  WriteTerm(WITH->UU.U0.son, SubPrec, LINK->depth + 1, LINK->LINK);
  s = WITH->UU.U0.son->brother;
  while (s != NULL) {
    WRITE1(',', LINK->LINK);
    WriteTerm(s, SubPrec, LINK->depth + 1, LINK->LINK);
    s = s->brother;
  }
  WRITE1(')', LINK->LINK);
}


static void WriteOp(LINK)
struct LOC_WriteTerm *LINK;
{
  nodeinfo *WITH;

  WITH = &LINK->y->info;
  switch (WITH->UU.U0.name->oclass) {

  case fxO:
  case fyO:
    WriteAtom(LINK->LINK->F, WITH->UU.U0.name);
    WRITE1(spac, LINK->LINK);
    WriteTerm(WITH->UU.U0.son, Rprec(WITH->UU.U0.name), LINK->depth + 1,
	      LINK->LINK);
    break;

  case xfO:
  case yfO:
    WriteTerm(WITH->UU.U0.son, Lprec(WITH->UU.U0.name), LINK->depth + 1,
	      LINK->LINK);
    WRITE1(spac, LINK->LINK);
    WriteAtom(LINK->LINK->F, WITH->UU.U0.name);
    break;

  case xfxO:
  case xfyO:
  case yfxO:
    WriteTerm(WITH->UU.U0.son, Lprec(WITH->UU.U0.name), LINK->depth + 1,
	      LINK->LINK);
    if (WITH->UU.U0.name != commaA && WITH->UU.U0.name != semiA)
      WRITE1(spac, LINK->LINK);
    WriteAtom(LINK->LINK->F, WITH->UU.U0.name);
    WRITE1(spac, LINK->LINK);
    WriteTerm(WITH->UU.U0.son->brother, Rprec(WITH->UU.U0.name),
	      LINK->depth + 1, LINK->LINK);
    break;
  }
}


static void WriteExp(LINK)
struct LOC_WriteTerm *LINK;
{
  if (LINK->p >= LINK->y->info.UU.U0.name->oprec) {
    WriteOp(LINK);
    return;
  }
  WRITE1('(', LINK->LINK);
  WriteOp(LINK);
  WRITE1(')', LINK->LINK);
}


static void WriteList(LINK)
struct LOC_WriteTerm *LINK;
{
  long n;
  node *z;

  WRITE1('[', LINK->LINK);
  WriteTerm(LINK->y->info.UU.U0.son, SubPrec, LINK->depth + 1, LINK->LINK);
  n = 1;
  z = Deref(LINK->y->info.UU.U0.son->brother, LINK->LINK->e);
  while ((n != WriteLength) & IsFunc(z, consA, 2L)) {
    WRITE1(',', LINK->LINK);
    WriteTerm(z->info.UU.U0.son, SubPrec, LINK->depth + 1, LINK->LINK);
    z = Deref(z->info.UU.U0.son->brother, LINK->LINK->e);
    n++;
  }
  if (!IsFunc(z, nilA, 0L)) {
    if (n < WriteLength) {
      WRITE1('|', LINK->LINK);
      WriteTerm(z, SubPrec, LINK->depth + 1, LINK->LINK);
    } else
      fprintf(*LINK->LINK->F, " ...");
  }
  WRITE1(']', LINK->LINK);
}


static void WriteFunc(LINK)
struct LOC_WriteTerm *LINK;
{
  nodeinfo *WITH;

  WITH = &LINK->y->info;
  if (WITH->UU.U0.arity > 2) {
    WriteStand(LINK);
    return;
  }
  switch (WITH->UU.U0.arity) {

  case 0:
    WriteAtom(LINK->LINK->F, WITH->UU.U0.name);
    break;

  case 1:
    if (((1L << ((long)WITH->UU.U0.name->oclass)) & ((1L << ((long)fxO)) |
	   (1L << ((long)fyO)) | (1L << ((long)xfO)) | (1L << ((long)yfO)))) != 0)
      WriteExp(LINK);
    else
      WriteStand(LINK);
    break;

  case 2:
    if (WITH->UU.U0.name == consA)
      WriteList(LINK);
    else if (((1L << ((long)WITH->UU.U0.name->oclass)) & ((1L << ((long)xfxO)) |
		(1L << ((long)xfyO)) | (1L << ((long)yfxO)))) != 0)
      WriteExp(LINK);
    else
      WriteStand(LINK);
    break;
  }
}


static void WriteVar(LINK)
struct LOC_WriteTerm *LINK;
{
  uchar n;
  boolean found;

  n = 0;
  found = false;
  while (n != LINK->LINK->varcount && !found) {
    n++;
    found = (LINK->y == LINK->LINK->varmap[n - 1]);
  }
  if (found) {
    fprintf(*LINK->LINK->F, "_%d", n);
    return;
  }
  if (LINK->LINK->varcount == MaxVars) {
    putc('_', *LINK->LINK->F);
    return;
  }
  LINK->LINK->varcount++;
  LINK->LINK->varmap[LINK->LINK->varcount - 1] = LINK->y;
  fprintf(*LINK->LINK->F, "_%d", LINK->LINK->varcount);
}


static void WriteTerm(x, p_, depth_, LINK)
node *x;
prec p_;
long depth_;
struct LOC_WriteOut *LINK;
{
  struct LOC_WriteTerm V;
  node *WITH;
  nodeinfo *WITH1;

  V.LINK = LINK;
  V.p = p_;
  V.depth = depth_;
  if (V.depth == WriteDepth) {
    fprintf(*LINK->F, "...");
    return;
  }
  if (x == NULL)
    Moan(35L, abortZ);
  V.y = Deref(x, LINK->e);
  WITH = V.y;
  WITH1 = &WITH->info;
  switch (WITH1->tag) {

  case funcT:
    WriteFunc(&V);
    break;

  case intT:
    if (WITH1->UU.ival >= 0)
      fprintf(*LINK->F, "%ld", WITH1->UU.ival);
    else
      fprintf(*LINK->F, "~%ld", -WITH1->UU.ival);
    break;

  case varT:
    WriteVar(&V);
    break;

  case anonT:
    WRITE1('_', LINK);
    break;
  }
}


void WriteOut(F_, x, e_)
FILE **F_;
node *x;
env e_;
{
  struct LOC_WriteOut V;

  V.F = F_;
  V.e = e_;
  V.varcount = 0;
  WriteTerm(x, MaxPrec, 0L, &V);
}


void misspelling(x, e)
node *x;
env e;
{
  atomentry *xname;
  long xarity;
  clause *cl;
  long n, i;
  FILE *TEMP;

  i = 0;
  n = 0;
  xname = x->info.UU.U0.name;
  if (xname->pclass != normP)
    return;
  xarity = x->info.UU.U0.arity;
  cl = xname->UU.proc;
  if (cl == NULL) {
    printf("\nPredicate ");
    TEMP = stdout;
	 WriteOut(&TEMP, x, e);
    printf(" is not defined");
    asc();
    goto _L1;
  }
  while (cl != NULL) {
    n++;
    if (cl->Denied) {
      i++;
      cl = cl->chain;
    } else if (cl->head->info.UU.U0.arity == xarity)
      goto _L1;
    else
      cl = cl->chain;
  }
  printf("\nPredicate ");
  TEMP = stdout;
  WriteOut(&TEMP, x, e);
  if (i == 0 || n > i)
    printf(" has improper arity");
  else if (i == n)
    printf(" is retracted");
  asc();
_L1: ;
}


void Trace(m, x, e)
tracemessage m;
node *x;
env e;
{
  node *y;
  FILE *TEMP;

  y = Deref(x, e);
  if (!(tracing && (debugging || !y->info.UU.U0.name->sys)))
    return;
  putchar('\n');
  switch (m) {

  case goalD:
    printf("%2d CALL(%2d):", envtop, e);
    break;

  case provedD:
    printf("%2d EXIT(%2d):", envtop, e);
    break;

  case redoD:
    printf("%2d REDO(%2d):", envtop, e);
    break;

  case failD:
    printf("%2d FAIL(%2d):", envtop, e);
    break;
  }
  TEMP = stdout;
  WriteOut(&TEMP, y, e);
}


void WRCL(F, CL)
FILE **F;
clause *CL;
{
  node *H, *B;
  env E;
  boolean W;
  clause *WITH;

  W = true;
  while (CL != NULL) {
    if (CL->Denied) {
      CL = CL->chain;
      continue;
    }
    if (W && fstop != '\015') {
      putc('\n', *F);
      W = false;
    }
    NewEnv(&E, NULL, 0, CL, CL->Nvars);
    WITH = CL;
    if (WITH->body == NULL)
      WriteOut(F, WITH->head, E);
    else {
      H = MakeVar(NULL);
      B = MakeVar(NULL);
      GetBody(&B->info, WITH->body, E);
      TrailVar(B);
      Bind(&H->info, WITH->head, E, 0L);
      TrailVar(H);
      H->brother = B;
      WriteOut(F, MakeFunc(arrowA, 2L, H), E);
    }
    if (fstop != '\015')
      putc(fstop, *F);
    putc('\n', *F);
    KillStacks(E - 1);
    CL = CL->chain;
  }
}


#define Infinity        859
long Hash(x, e)
node *x;
env e;
{
  long Result;
  node *y, *z;
  nodeinfo *WITH;

  y = Deref(x, e);
  if (y->info.tag == varT)
    return 0;
  if (y->info.UU.U0.arity == 0)
    return Infinity;
  z = Deref(y->info.UU.U0.son, e);
  WITH = &z->info;
  switch (WITH->tag) {

  case funcT:
    Result = Infinity + WITH->UU.U0.name->atomno;
    break;

  case intT:
    Result = WITH->UU.ival - (WITH->UU.ival <= 0);
    break;

  case varT:
  case anonT:
    Result = 0;
    break;
  }   
  return Result;
}
#undef Infinity


boolean FindClause(cl, x, e)
clause **cl;
node *x;
env e;
{
  key k;
  boolean ok;
  clause *WITH;

  k = Hash(x, e);
  ok = false;
  while (*cl != NULL && !ok) {
    WITH = *cl;
    if (WITH->Denied || WITH->keyval != 0 && k != 0 && WITH->keyval != k)
      *cl = WITH->chain;
    else
      ok = true;
  }
  return ok;
}


typedef struct _REC_varmap {
  node *sourcevar, *curvar;
  boolean alloc;
  union {
    node *firstref;
    long address;
  } UU;
} _REC_varmap;

struct LOC_AddClause {
  env e;
  boolean asserta;
  env place;
  net_clause_record *netp;
  clause *cproc;
  boolean glovars;
  node *newhead, *newbody;
  key newkey;
  uchar varcount;
  long framesize;
  _REC_varmap varmap[MaxVars];
} ;


static void NewSkel(x, LINK)
node **x;
struct LOC_AddClause *LINK;
{
  node *WITH;

  if ((*x = (node *)malloc(sizeof(node))) == NULL)
   { printf("\nOut of Memory in AddClause\n"); exit(1); }
  WITH = *x;
  WITH->brother = NULL;
  WITH->chain = NULL;
  WITH->field = heapF;
  WITH->node_ptr = NULL;
  WITH->default_ = NULL;
  if (LINK->glovars)  WITH->scope = -1; else WITH->scope = 0;
}


static node *SkelVar(v, LINK)
node *v;
struct LOC_AddClause *LINK;
{
  uchar n;
  node *w;
  boolean found;
  _REC_varmap *WITH;
  nodeinfo *WITH1;

  n = 0;
  found = false;
  while (n != LINK->varcount && !found) {
    n++;
    found = (LINK->varmap[n - 1].sourcevar == v);
  }
  if (found) {
    WITH = &LINK->varmap[n - 1];
    if (LINK->glovars) {
      WITH1 = &WITH->UU.firstref->info;
      WITH1->tag = varT;
      WITH1->UU.val = NULL;
      NewSkel(&w, LINK);
      w->scope = -1;
      WITH->curvar->chain = w;
      WITH->curvar = w;
      WITH1 = &w->info;
      WITH1->tag = varT;
      WITH1->UU.val = WITH->UU.firstref;
      return w;
    }
    if (!WITH->alloc) {
      LINK->framesize++;
      WITH1 = &WITH->UU.firstref->info;
      WITH1->tag = skelT;
      WITH1->UU.offset = LINK->framesize;
      WITH->alloc = true;
      WITH->UU.address = LINK->framesize;
    }
    NewSkel(&w, LINK);
    WITH1 = &w->info;
    WITH1->tag = skelT;
    WITH1->UU.offset = WITH->UU.address;
    return w;
  }
  if (LINK->varcount >= MaxVars)
    Moan(23L, abortZ);
  LINK->varcount++;
  NewSkel(&w, LINK);
  if (LINK->glovars) {
    WITH1 = &w->info;
    w->scope = -1;
    WITH1->tag = varT;
    WITH1->UU.val = NULL;
  } else
    w->info.tag = anonT;
  WITH = &LINK->varmap[LINK->varcount - 1];
  WITH->sourcevar = v;
  WITH->alloc = false;
  WITH->UU.firstref = w;
  WITH->curvar = w;
  return w;
}


struct LOC_Skeleton {
  struct LOC_AddClause *LINK;
  long depth;
} ;


static node *SkelArgs(s, LINK)
node *s;
struct LOC_Skeleton *LINK;
{
  node *t, *u, *v;

  if (s == NULL)
    return NULL;
  else {
    u = Skeleton(s, LINK->depth + 1, LINK->LINK);
    t = s->brother;
    v = u;
    while (t != NULL) {
      v->brother = Skeleton(t, LINK->depth + 1, LINK->LINK);
      t = t->brother;
      v = v->brother;
    }
    return u;
  }
}


static node *Skeleton(x, depth_, LINK)
node *x;
long depth_;
struct LOC_AddClause *LINK;
{
  struct LOC_Skeleton V;
  node *y, *z;
  nodeinfo *WITH;

  V.LINK = LINK;
  V.depth = depth_;
  if (V.depth > MaxDepth)
    Moan(11L, abortZ);
  y = Deref(x, LINK->e);
  switch (y->info.tag) {

  case funcT:
    NewSkel(&z, LINK);
    WITH = &z->info;
    WITH->tag = funcT;
    WITH->UU.U0.name = y->info.UU.U0.name;
    WITH->UU.U0.arity = y->info.UU.U0.arity;
    WITH->UU.U0.son = SkelArgs(y->info.UU.U0.son, &V);
    break;

  case intT:
    NewSkel(&z, LINK);
    WITH = &z->info;
    WITH->tag = intT;
    WITH->UU.ival = y->info.UU.ival;
    break;

  case varT:
    z = SkelVar(y, LINK);
    break;

  case anonT:
    NewSkel(&z, LINK);
    z->info.tag = anonT;
    break;
  }
  return z;
}


static node *SkelCall(x, LINK)
node *x;
struct LOC_AddClause *LINK;
{
  node *Result, *y, *z;
  nodeinfo *WITH;

  y = Deref(x, LINK->e);
  switch (y->info.tag) {

  case funcT:
    Result = Skeleton(y, 0L, LINK);
    break;

  case varT:
    NewSkel(&z, LINK);
    WITH = &z->info;
    WITH->tag = funcT;
    WITH->UU.U0.name = callA;
    WITH->UU.U0.arity = 1;
    WITH->UU.U0.son = SkelVar(y, LINK);
    Result = z;
    break;

  case intT:
  case anonT:
    Moan(1L, abortZ);
    break;
  }
  return Result;
}


static node *SkelHead(x, LINK)
node *x;
struct LOC_AddClause *LINK;
{
  node *Result, *y;

  y = Deref(x, LINK->e);
  if (y->info.tag != funcT)
    Moan(1L, abortZ);
  Result = Skeleton(y, 0L, LINK);
  LINK->newkey = Hash(y, LINK->e);
  return Result;
}


static node *SkelBody(x, depth, LINK)
node *x;
long depth;
struct LOC_AddClause *LINK;
{
  node *y, *z;

  if (depth > MaxDepth)
    Moan(11L, abortZ);
  y = Deref(x, LINK->e);
  if (IsFunc(y, commaA, 2L)) {
    z = SkelCall(y->info.UU.U0.son, LINK);
    z->brother = SkelBody(y->info.UU.U0.son->brother, depth + 1, LINK);
    return z;
  } else
    return (SkelCall(y, LINK));
}


static void PlugA(cp, LINK)
clause **cp;
struct LOC_AddClause *LINK;
{
  clause *cl;

  if ((cl = (clause *)malloc(sizeof(clause))) == NULL)
   { printf("\nOut of Memory in PlugA1\n"); exit(1); }
  if (LINK->glovars) {
    LINK->netp = net_clause_base;
	 if ((net_clause_base = (net_clause_record *)malloc(sizeof(net_clause_record))) == NULL)
     { printf("\nOut of Memory in PlugA2\n"); exit(1); }
    net_clause_base->cls = cl;
    net_clause_base->chain = LINK->netp;
  }
  cl->head = LINK->newhead;
  cl->body = LINK->newbody;
  cl->Nvars = LINK->framesize;
  cl->Denied = false;
  cl->keyval = LINK->newkey;
  cl->lazy = lazycontext;
  cl->chain = *cp;
  *cp = cl;
  LINK->cproc = *cp;
}


static void PlugZ(cp, LINK)
clause **cp;
struct LOC_AddClause *LINK;
{
  clause *p, *ap;

  if (*cp == NULL) {
    PlugA(cp, LINK);
    return;
  }
  p = *cp;
  ap = display[(LINK->place) - 1].Fclause;
  if (LINK->place == 0) {
    while (p->chain != NULL)
      p = p->chain;
  } else {
    while (p->chain != NULL && p->chain != ap)
      p = p->chain;
  }
  PlugA(&p->chain, LINK);
}


static void set_ddr_ptr(ddr_ptr, LINK)
ddr_clause_record **ddr_ptr;
struct LOC_AddClause *LINK;
{
  ddr_clause_record *ddrp;

  ddrp = *ddr_ptr;
  if ((*ddr_ptr = (ddr_clause_record *)malloc(sizeof(ddr_clause_record))) == NULL)
   { printf("\nOut of Memory in Set_ddr_ptr\n"); exit(1); }
  (*ddr_ptr)->cls = LINK->cproc;
  (*ddr_ptr)->chain = ddrp;
}


boolean OccursOnes(x,y)
atomentry *x;
node *y;
{
  boolean ok;

  if (y->info.UU.U0.name == x) {
    ok = true;
    if (IsFunc(y, netA, 2L))
      y = y->info.UU.U0.son;
    else 
      y = y->brother;
    if (y != NULL) {
      do {
        if (IsFunc(y, netA, 2L)) y = y->info.UU.U0.son;
        if (y->info.UU.U0.name == x) ok = false;
        y = y-> brother;
      } while ((y != NULL) && ok);
    } 
  } else 
    ok=false;
  return ok;
}


static void AddOneClause(z, LINK)
node *z;
struct LOC_AddClause *LINK;
{
  node *x, *y, *c;
  net_node_record *p;
  atomentry *WITH;
  node *WITH1;
  nodeinfo *WITH2;

  if (IsFunc(z, questionA, 1L))
    Moan(1L, abortZ);
  if (IsFunc(z, arrowA, 2L)) {
    LINK->newhead = SkelHead(z->info.UU.U0.son, LINK);
    LINK->newbody = SkelBody(z->info.UU.U0.son->brother, 0L, LINK);
  } else {
    LINK->newhead = SkelHead(z, LINK);
    LINK->newbody = NULL;
  }
  /* For Net Clauses */
  z = Deref(z, LINK->e);
  if (LINK->glovars & IsFunc(z, defaultA, z->info.UU.U0.arity)) {
    x = LINK->newhead->info.UU.U0.son;
    y = Deref(x, 0);
    y->default_ = x->brother;
  }
  if (LINK->glovars & IsFunc(z, nodeA, z->info.UU.U0.arity)) {
    x = LINK->newhead->info.UU.U0.son;
    c = x;
    while ((c->info.tag == varT) | IsFunc(c, negA, 1L))
      c = c->brother;
    if (c->info.tag != intT)
      Moan(37L, abortZ);
    if (c->brother == NULL || c->brother->info.tag != funcT)
      Moan(37L, abortZ);
    while ((x->info.tag == varT) | IsFunc(x, negA, 1L)) {
      if (IsFunc(x, negA, 1L))
	y = Deref(x->info.UU.U0.son, 0);
      else
	y = Deref(x, 0);
      if (y->info.tag != varT)
	Moan(37L, abortZ);
      p = y->node_ptr;
	 if ((y->node_ptr = (net_node_record *)malloc(sizeof(net_node_record))) == NULL)
	  { printf("\nOut of Memory in AddOneClause\n"); exit(1); }
      y->node_ptr->count = c;
      if (IsFunc(x, negA, 1L))
	y->node_ptr->val = -1;
      else
	y->node_ptr->val = 1;
      y->node_ptr->chain = p;
      x = x->brother;
    }
  }
  WITH = LINK->newhead->info.UU.U0.name;
  if (WITH->sys && (mode != sysM || WITH->pclass != normP))
    Moan(29L, abortZ);
  if (mode == sysM)
    WITH->sys = true;
  if (LINK->asserta ||
      LINK->place > 0 && display[LINK->place - 1].Fclause == WITH->UU.proc)
    PlugA(&WITH->UU.proc, LINK);
  else
    PlugZ(&WITH->UU.proc, LINK);
  if (!IsFunc(z, ddrA, 2L))
    return;
  x = z->info.UU.U0.son;
  while (IsFunc(x, netA, 2L)) {
    WITH1 = x->info.UU.U0.son;
    if (WITH1->info.tag == funcT && !WITH1->info.UU.U0.name->sys
        && OccursOnes(WITH1->info.UU.U0.name, WITH1))
      set_ddr_ptr(&WITH1->info.UU.U0.name->ddr_ptr, LINK);
    x = WITH1->brother;
  }
  WITH2 = &x->info;
  if (WITH2->tag == funcT && !WITH2->UU.U0.name->sys)
    set_ddr_ptr(&WITH2->UU.U0.name->ddr_ptr, LINK);
}


void AddClause(p, e_, asserta_, place_)
node *p;
env e_;
boolean asserta_;
env place_;
{
  struct LOC_AddClause V;
  node *q;

  V.e = e_;
  V.asserta = asserta_;
  V.place = place_;
  V.varcount = 0;
  V.framesize = 0;
  V.glovars = false;
  q = Deref(p, V.e);
  while (IsFunc(q, netA, 2L)) {
    V.glovars = true;
    AddOneClause(q->info.UU.U0.son, &V);
    q = q->info.UU.U0.son->brother;
  }
  if (!IsFunc(q, nilA, 0L))
    AddOneClause(q, &V);
}

/* For Net Clauses */
boolean CallNode(n, e)
net_node_record *n;
env e;
{
  boolean Result;
  long i, j;
  net_node_record *np;
  node *WITH;
  env last;

  last = choicepoint;
  i = 0;
  j = 0;
  Result = true;
  np = n;
  while (np != NULL) {
    WITH = np->count;
      WITH->info.UU.ival = WITH->info.UU.ival - np->val;
    np = np->chain;
  }
  while (n != NULL) {
    WITH = n->count;
    if (WITH->info.UU.ival == 0 && netmode > 0) {
      j++;
      if (Execute(WITH->brother, e))
	i++;
        else if (netmode==3) {choicepoint = last; return false; }
    }
    n = n->chain;
  };
  choicepoint = last;
  switch (netmode) {

  case 1:
    Result = true;
    break;

  case 2:
    if (j > 0)
      Result = (i > 0);
    else
      Result = true;
    break;

  case 3:
       Result = true;
    break;
  }
  return Result;
}


boolean Occur(x, t, e)
node *x, *t;
env e;
{
  node *y;
  boolean ok;

  ok = false;
  t = Deref(t, e);
  if (t->info.tag != funcT) {
    if (t->info.tag == varT)
      ok = (x == t);
    return ok;
  }
  y = t->info.UU.U0.son;
  while (y != NULL && !ok) {
    ok = Occur(x, y, e);
    y = y->brother;
  }
  return ok;
}


struct LOC_Unify {
  env e1, e2;
  long depth;
} ;


static boolean UnifyArgs(s1, s2, LINK)
node *s1, *s2;
struct LOC_Unify *LINK;
{
  node *t1, *t2;
  boolean ok;

  t1 = s1;
  t2 = s2;
  ok = true;
  while (t1 != NULL && ok) {
    ok = Unify(t1, t2, LINK->e1, LINK->e2, LINK->depth + 1);
    t1 = t1->brother;
    t2 = t2->brother;
  }
  return ok;
}


boolean Unify(x1, x2, e1_, e2_, depth_)
node *x1, *x2;
env e1_, e2_;
long depth_;
{
  struct LOC_Unify V;
  node *y1, *y2;
  trailentry *t;
  boolean result;
  net_node_record *n;

  V.e1 = e1_;
  V.e2 = e2_;
  V.depth = depth_;
  if (V.depth > MaxDepth)
    Moan(11L, abortZ);
  y1 = Deref(x1, V.e1);
  y2 = Deref(x2, V.e2);
  switch (Uaction[(long)y1->info.tag - (long)funcT]
	  [(long)y2->info.tag - (long)funcT]) {

  case funcU:
    if (y1->info.UU.U0.name == y2->info.UU.U0.name &&
	y1->info.UU.U0.arity == y2->info.UU.U0.arity)
      result = UnifyArgs(y1->info.UU.U0.son, y2->info.UU.U0.son, &V);
    else
      result = false;
    break;

  case intU:
    result = (y1->info.UU.ival == y2->info.UU.ival);
    break;

  case VTbindU:
    if (gencontext | Occur(y1, y2, V.e2))
      result = false;
    else {
      TrailVar(y1);
      Bind(&y1->info, y2, V.e2, 0L);
      n = y1->node_ptr;
      if (n != NULL)
	result = CallNode(n, V.e1);
      else
	result = true;
    }
    break;

  case TVbindU:
    if (gencontext | Occur(y2, y1, V.e1))
      result = false;
    else {
      TrailVar(y2);
      Bind(&y2->info, y1, V.e1, 0L);
      n = y2->node_ptr;
      if (n != NULL)
	result = CallNode(n, V.e2);
      else
	result = true;
    }
    break;

  case VVbindU:
    if (y1->default_ != NULL && !gencontext && V.e1 > 0) {
      x1 = Deref(y1->default_, 0);
      if (y1->default_->brother != NULL && x1->info.tag == varT) {
	t = trailend;
	result = Execute(y1->default_->brother, 0);
	if (!result) {
	  if (t != trailend)
	    Untrail(t);
	}
      }
      result = Unify(y2, y1->default_, V.e2, V.e1, V.depth + 1);
    } else if (y2->default_ != NULL && !gencontext && V.e2 > 0) {
      x2 = Deref(y2->default_, 0);
      if (y2->default_->brother != NULL && x2->info.tag == varT) {
	t = trailend;
	result = Execute(y2->default_->brother, 0);
	if (!result) {
	  if (t != trailend)
	    Untrail(t);
	}
      }
      result = Unify(y1, y2->default_, V.e1, V.e2, V.depth + 1);
    } else if (gencontext)
      result = false;
    else {
      BindVars(y1, y2);
      result = true;
    }
    break;

  case succeedU:
    result = true;
    break;

  case failU:
    result = false;
    break;
  }
  return result;
}


boolean NoNil(x, e)
node *x;
env e;
{
  node *y;
  boolean ok;

  ok = true;
  while (x != NULL && ok) {
    y = Deref(x, e);
    ok = !IsFunc(y, niltA, 0L);
    x = x->brother;
  }
  return ok;
}


struct LOC_LazyUnify {
  ddr_clause_record *ddr;
  clause *xproc;
  atomentry *xname;
  node *g;
  env ge;
} ;


static void BackTrack(z, e1, LINK)
node *z;
env e1;
struct LOC_LazyUnify *LINK;
{
  clause *cl;
  node *next;
  env e2;
  ddr_clause_record *WITH;
  clause *WITH1;
  if (interrupted) { interrupted = false; asc(); }
  if (z == NULL) {
    if (e1 == LINK->ge)
      tbool = Execute(LINK->g, e1);
  } else {
    if (IsFunc(z, netA, 2L)) {
      z = z->info.UU.U0.son;
      next = z->brother;
    } else
      next = NULL;
    if (IsFunc(z, eqA, 2L)) {
      WITH = LINK->ddr;
      z = z->info.UU.U0.son;
      if (!IsFunc(Deref(z, e1), niltA, 0L)) {
	NewEnv(&e2, NULL, 0, WITH->cls, WITH->cls->Nvars);
	if (Unify(z, z->brother, e1, e2, 0L))
	  BackTrack(WITH->cls->head->info.UU.U0.son, e2, LINK);
      }
      BackTrack(next, e1, LINK);
    } else if (OccursOnes(LINK->xname, z)) {
      WITH1 = LINK->xproc;
      if (e1 > LINK->ge)
	BackTrack(next, e1, LINK);
      else {
	NewEnv(&e2, z, e1, LINK->xproc, WITH1->Nvars);
	if (Unify(WITH1->head, z, e2, e1, 0L))
	  BackTrack(next, e1, LINK);
      }
    } else {
      cl = z->info.UU.U0.name->UU.proc;
      while (cl != NULL) {
	WITH1 = cl;
	NewEnv(&e2, z, e1, cl, WITH1->Nvars);
	choicepoint = e2;
	if (Unify(WITH1->head, z, e2, e1, 0L)) {
	  BackTrack(next, e1, LINK);
	  if (!NoNil(z->info.UU.U0.son, e1))
	    goto _L1;
	}
	if (WITH1->chain != NULL)
	  KillStacks(e2 - 1);
	cl = WITH1->chain;
      }
    }
  }
_L1: ;
}


void LazyUnify(ddr_, x, e)
ddr_clause_record *ddr_;
node *x;
env e;
{
  struct LOC_LazyUnify V;
  queue_record *q;
  clause *cp;
  node *y;
  env e1;
  boolean ok;
  clause *WITH;
  queue_record *WITH1;
  ddr_clause_record *WITH2;

  V.ddr = ddr_;
  ok = false;
  cp = x->info.UU.U0.name->UU.proc;
  while (FindClause(&cp, x, e) && !ok) {
    WITH = cp;
    NewEnv(&e1, x, e, cp, WITH->Nvars);
    choicepoint = e1;
    ok = Unify(WITH->head, x, e1, e, 0L);
    if (!ok)
      KillStacks(e1 - 1);
    cp = WITH->chain;
  }
  if (ok)
    return;
  lazycontext = true;
  AddClause(x, e, true, 0);
  lazycontext = false;
  q = qtop;
  if ((qtop = (queue_record *)malloc(sizeof(queue_record))) == NULL)
   { printf("\nOut of Memory in LazyUnify\n"); exit(1); }
  WITH1 = qtop;
  WITH1->ddr_ptr = V.ddr;
  WITH1->name = x->info.UU.U0.name;
  WITH1->proc = WITH1->name->UU.proc;
  WITH1->chain = NULL;
  if (q != NULL)
    q->chain = qtop;
  if (qbase == NULL)
    qbase = qtop;
  if (inlazy)
    return;
  inlazy = true;
  while (qbase != NULL) {
    WITH1 = qbase;
    V.ddr = WITH1->ddr_ptr;
    while (V.ddr != NULL) {
      WITH2 = V.ddr;
      NewEnv(&e1, x, e, WITH2->cls, WITH2->cls->Nvars);
      y = WITH2->cls->head->info.UU.U0.son;
      V.g = y->brother;
      V.xname = WITH1->name;
      V.xproc = WITH1->proc;
      V.ge = e1;
      BackTrack(y, e1, &V);
      KillStacks(e1 - 1);
      V.ddr = WITH2->chain;
    }
    q = qbase;
    qbase = WITH1->chain;
	 free(q);
  }
  qtop = NULL;
  inlazy = false;
}


boolean CallGoal(goal, call, e)
node *goal, *call;
env e;
{
  boolean Result;
  node *x;
  env e1;

  NewEnv(&e1, call, e, NULL, 1L);
  x = EnvRef(1L, e1);
  Bind(&x->info, goal, e, 0L);
  Result = Execute(x, e1);
  if (e1 > choicepoint)
    DisposeEnv();
  return Result;
}


void asc()
{
  char ans;
  env se;
  printf("\nAbort, Trace(%1d), Debug(%1d), Step(%1d), topLevel ? ",
	 tracing, debugging, step);
  setmode(userM);
    GetChar(&ans);
    switch (ans) {

    case 'A':
    case 'a':
      Abort();
      break;

    case 'T':
    case 't':
      tracing = (!tracing == true);
      break;

    case 'D':
    case 'd':
      debugging = (!debugging == true);
      break;

    case 'S':
    case 's':
      step = (!step == true);
      break;

    case 'L':
    case 'l':
      if (!debug_state) {
	debug_state = true;
	ans = '\10';
	do {
	  setmode(userM);
	  SAVEVARS = true;
	  NewEnv(&se, NULL, 0, NULL, 0L);
	  GOAL(MakeFunc(debugA, 0L, NULL), 0);
	  KillStacks(se - 1);
	} while (!haltflag);
	debug_state = false;
      }
      break;
    }
  setmode(userM);
}


boolean Execute(goalp, goalenv)
node *goalp;
env goalenv;
{
  boolean Result;
  node *callp;
  env envp, callenv, baseenv;
  clause *clausep;
  ddr_clause_record *ddr;
  enum {
    callQ, procQ, bodyQ, returnQ, failQ, finalQ
  } state;
  atomentry *WITH;

  callp = Deref(goalp, goalenv);
  callenv = goalenv;
  baseenv = envtop;
  state = callQ;
  do {
    if (interrupted) { interrupted = false; asc(); }
    switch (state) {

    case callQ:
      if (tracing)
	Trace(goalD, callp, callenv);
      if (debugging)
	misspelling(callp, callenv);
      if (step)
	asc();
      WITH = callp->info.UU.U0.name;
      switch (WITH->pclass) {

      case normP:
	clausep = WITH->UU.proc;
	state = procQ;
	break;

      case evalP:
	if (CallEvalPred(callp, callenv, WITH->UU.U1.routine,
			 (long)WITH->UU.U1.arity))
	  state = returnQ;
	else
	  state = failQ;
	break;
      }
      break;

    case procQ:
      ddr = callp->info.UU.U0.name->ddr_ptr;
      if (ddr != NULL) {
	LazyUnify(ddr, callp, callenv);
	state = returnQ;
      } else if (FindClause(&clausep, callp, callenv)) {
	NewEnv(&envp, callp, callenv, clausep, clausep->Nvars);
	if (clausep->chain != NULL || more)
	  choicepoint = envp;
	if (Unify(clausep->head, callp, envp, callenv, 0L)) {
	  callp = clausep->body;
	  callenv = envp;
	  state = bodyQ;
	} else
	  state = failQ;
      } else
	state = failQ;
      break;

    case bodyQ:
      if (callp == NULL) {
	envp = callenv;
	GetEnv(envp, &callp, &callenv, &clausep);
	if (envp > choicepoint)
	  DisposeEnv();
	if (tracing)
	  Trace(provedD, callp, callenv);
	state = returnQ;
      } else
	state = callQ;
      break;

    case returnQ:
      if (callenv > goalenv) {
	callp = callp->brother;
	state = bodyQ;
      } else {
	Result = true;
	state = finalQ;
      }
      break;

    case failQ:
      if (choicepoint > baseenv) {
	if (tracing)
	  Trace(failD, callp, callenv);
	GetEnv(choicepoint, &callp, &callenv, &clausep);
	KillStacks(choicepoint - 1);
	if (tracing)
	  Trace(redoD, callp, callenv);
	clausep = clausep->chain;
	state = procQ;
      } else {
	Result = false;
	state = finalQ;
      }
      break;
    }
  } while (state != finalQ);
  return Result;
}


long Evaluate(x, e, depth)
node *x;
env e;
long depth;
{
  long Result;
  node *y;
  long a, b;
  nodeinfo *WITH;

  if (depth > MaxDepth)
    Moan(11L, abortZ);
  y = Deref(x, e);
  WITH = &y->info;
  switch (WITH->tag) {

  case funcT:
    if (WITH->UU.U0.arity == 2) {
      if (WITH->UU.U0.name == consA)
	Result = Evaluate(WITH->UU.U0.son, e, depth + 1);
      else {
	a = Evaluate(WITH->UU.U0.son, e, depth + 1);
	b = Evaluate(WITH->UU.U0.son->brother, e, depth + 1);
	if (WITH->UU.U0.name == plusA)
	  Result = a + b;
	else if (WITH->UU.U0.name == minusA)
	  Result = a - b;
	else if (WITH->UU.U0.name == timesA)
	  Result = a * b;
	else if (WITH->UU.U0.name == divideA) {
	  if (b == 0)
	    Moan(12L, abortZ);
	  Result = a / b;
	} else if (WITH->UU.U0.name == modA) {
	  if (b == 0)
	    Moan(12L, abortZ);
	  Result = a % b;
	} else
	  Moan(6L, abortZ);
      }
    } else if (WITH->UU.U0.name == negA && WITH->UU.U0.arity == 1)
      Result = -Evaluate(WITH->UU.U0.son, e, depth + 1);
    else
      Moan(6L, abortZ);
    break;

  case intT:
    Result = WITH->UU.ival;
    break;

  case varT:
    if (y->default_ != NULL) {
      if (y->default_->brother != NULL) {
	if (!Execute(y->default_->brother, e))
	  Moan(6L, abortZ);
      }
      Result = Evaluate(y->default_, 0, 0L);
    } else
      Moan(6L, abortZ);
    break;

  case anonT:
    Moan(6L, abortZ);
    break;
  }
  return Result;
}


boolean IntResult(x, e, i)
node *x;
env e;
long i;
{
  boolean Result;
  node *y;
  net_node_record *n;
  long a, b;
  nodeinfo *WITH;
  node *WITH1;

  y = Deref(x, e);
  a = 0;
  b = 0;
  WITH = &y->info;
  switch (WITH->tag) {

  case funcT:
    Result = false;
    break;

  case intT:
    Result = (WITH->UU.ival == i);
    break;

  case varT:
    WITH->tag = intT;
    WITH->UU.ival = i;
    TrailVar(y);
    n = y->node_ptr;
    if (n != NULL) {
      while (n != NULL) {
	WITH1 = n->count;
	WITH1->info.UU.ival -= n->val;
	if (WITH1->info.UU.ival == 0) {
	  a++;
	  if (Execute(WITH1->brother, 0))
	    b++;
	}
	n = n->chain;
      }
      if (a > 0)
	Result = (b > 0);
      else
	Result = true;
    } else
      Result = true;
    break;

  case anonT:
    Result = true;
    break;
  }
  return Result;
}


void DOLIST(F, T, L)
FILE **F;
node *T;
boolean L;
{
  clause *CL, *P;
  env E;

  if (T->info.tag != funcT)
    Moan(8L, abortZ);
  if (T->info.UU.U0.name->sys)
    Moan(29L, abortZ);
  CL = T->info.UU.U0.name->UU.proc;
  if (L) {
    WRCL(F, CL);
    return;
  }
  while (CL != NULL) {
    P = CL;
    if (CL->Denied)
      CL = CL->chain;
    else {
      NewEnv(&E, NULL, 0, CL, 0L);
      CL->Denied = true;
      DTERM(&CL->head);
      DTERM(&CL->body); 
      DisposeEnv();
      CL = CL->chain;
    }
	 free(P);
  }
  T->info.UU.U0.name->UU.proc = NULL;
}


void DelTerm(x)
node *x;
{
  FILE *TEMP;

  if (IsFunc(x, netA, 2L))
    net_clause_base = NULL;
  while (IsFunc(x, netA, 2L)) {
    if (IsFunc(x->info.UU.U0.son, arrowA, 2L)) {
      TEMP = stdout;
		DOLIST(&TEMP, x->info.UU.U0.son->info.UU.U0.son, false);
    } else {
      TEMP = stdout;
		DOLIST(&TEMP, x->info.UU.U0.son, false);
    }
    x = x->info.UU.U0.son->brother;
  }
  if (IsFunc(x, arrowA, 2L)) {
    TEMP = stdout;
	 DOLIST(&TEMP, x->info.UU.U0.son, false);
  }
  if (!IsFunc(x, questionA, 1L)) {
    TEMP = stdout;
	 DOLIST(&TEMP, x, false);
  } else if (IsFunc(x->info.UU.U0.son, endA, 0L))
    haltflag = true;
}


void CallShell(t)
node *t;
{
  long i;
  char s[80];
  strin *WITH;
  long FORLIM;

  if (t->info.tag != funcT)
    Moan(33L, abortZ);
  WITH = &t->info.UU.U0.name->ident;
  if (WITH->length > 79)
    Moan(34L, abortZ);
  else {
    FORLIM = WITH->length;
    for (i = 0; i < FORLIM; i++)
      s[i] = stringbuf[WITH->index + i];
  }
  s[i] = '\0';   /* end of string is "char(0)" */
  system(s);
}


void dofile(t, out, dcon)
node **t;
boolean out, dcon;
{
  long I;
  node *X;
  env E;
  char name[80];
  strin *WITH;
  long FORLIM;
  char STR1[256];

  if ((*t)->info.tag != funcT)
    Moan(33L, abortZ);
  WITH = &(*t)->info.UU.U0.name->ident;
  if (WITH->length > 79)
    Moan(34L, abortZ);
  else {
    FORLIM = WITH->length;
    for (I = 0; I < FORLIM; I++)
      name[I] = stringbuf[WITH->index + I];
  }
  name[I] = '\0';
  if (out) {
    if (PROG != NULL) {
      sprintf(STR1, "%.80s", name);
      PROG = freopen(STR1, "w", PROG);
    } else {
      sprintf(STR1, "%.80s", name);
      PROG = fopen(STR1, "w");
    }
    if (PROG == NULL)
      _EscIO(20,name);
    OUTFLAG = true;
    return;
  }
  if (PROG != NULL) {
    sprintf(STR1, "%.80s", name);
    PROG = freopen(STR1, "r", PROG);
  } else {
    sprintf(STR1, "%.80s", name);
    PROG = fopen(STR1, "r");
  }
  if (PROG == NULL)
    _EscIO(10,name);
  setmode(progM);
  do {
    NewEnv(&E, NULL, 0, NULL, 0L);
    X = ReadIn();
    if (IsFunc(X, questionA, 1L))
      GOAL(X->info.UU.U0.son, E);
    else if (dcon)
      DelTerm(X);
    else
      AddClause(X, E, false, 0);
    KillStacks(E - 1);
  } while (!haltflag);
  if (PROG != NULL)
    fclose(PROG);
  PROG = NULL;  /* Close file */
  setmode(userM);
}


void doclose()
{
  if (PROG != NULL)
    fclose(PROG);
  PROG = NULL;
  setmode(userM);
  OUTFLAG = false;
}


void DOLISTALL(F)
FILE **F;
{
  atomentry *A;
  clause *CL;
  char I;

  for (I = 0; I < HashSize; I++) {
    A = hashtable[I];
    while (A != NULL) {
      if (A->sys || A->UU.proc == NULL)
	A = A->chain;
      else {
	CL = A->UU.proc;
	WRCL(F, CL);
	A = A->chain;
      }
    }
  }
}


void dellazy()
{
  atomentry *a;
  clause *cl, *cp;
  queue_record *q;
  char i;
  clause *WITH;

  inlazy = false;
  qtop = NULL;
  while (qbase != NULL) {
    q = qbase->chain;
	 free(qbase);
    qbase = q;
  }
  qbase = NULL;
  for (i = 0; i < HashSize; i++) {
    a = hashtable[i];
    while (a != NULL) {
      if (a->sys || a->UU.proc == NULL) {
	a = a->chain;
	continue;
      }
      cl = a->UU.proc;
      while (cl != NULL && cl->lazy) {
	WITH = cl;
	DTERM(&WITH->head);
	cp = cl;
	cl = WITH->chain;
	free(cp);
      }
      a->UU.proc = cl;
      a = a->chain;
    }
  }
}


struct LOC_CallEvalPred {
  node *call;
  env e;
  long arity;
  boolean result;
  node *argval[MaxEvalArity];
} ;


static void GetArgs(LINK)
struct LOC_CallEvalPred *LINK;
{
  char i;
  node *x, *a;
  char FORLIM;

  x = Deref(LINK->call, LINK->e);
  if (LINK->arity != x->info.UU.U0.arity)
    Moan(0L, abortZ);
  a = x->info.UU.U0.son;
  if (LINK->arity <= 0)
    return;
  FORLIM = LINK->arity;
  for (i = 0; i < FORLIM; i++) {
    LINK->argval[i] = Deref(a, LINK->e);
    a = a->brother;
  }
}


static void Dodef(LINK)
struct LOC_CallEvalPred *LINK;
{
  char c;

  c = (char)Evaluate(LINK->argval[0], LINK->e, 0L);
  switch (Evaluate(LINK->argval[1], LINK->e, 0L)) {

  case 0:
    CharClass[c] = smallC;
    break;

  case 1:
    CharClass[c] = largeC;
    break;

  case 2:
    CharClass[c] = specialC;
    break;

  case 3:
    CharClass[c] = spaceC;
    spac = c;
    break;

  case 4:
    CharClass[c] = wierdC;
    break;

  case 5:
    fstop = c;
    CharClass[c] = specialC;
    break;
  }
}


static void RES(LINK)
struct LOC_CallEvalPred *LINK;
{
  strindex I, J;
  char ch;
  strindex FORLIM;
  strin *WITH;
  strindex FORLIM1;
  FILE *TEMP;

  FORLIM = PVARCOUNT;
  for (I = 0; I < FORLIM; I++) {
    putchar('\n');
    WITH = &PVARTABLE[I].IDENT;
    FORLIM1 = WITH->index + WITH->length;
    for (J = WITH->index; J < FORLIM1; J++)
      putchar(stringbuf[J]);
    putchar('=');
    TEMP = stdout;
    WriteOut(&TEMP, PVARTABLE[I].ROOTVAR, 0);
  }
  if (PVARCOUNT <= 0)
    return;
  setmode(userM);
  GetChar(&ch);
  if (ch == ';')
    LINK->result = false;

}


static void DoCall(LINK)
struct LOC_CallEvalPred *LINK;
{
  if (LINK->argval[0]->info.tag != funcT)
    Moan(7L, abortZ);
  else
    LINK->result = CallGoal(LINK->argval[0], LINK->call, LINK->e);
}


static void DoGet0(LINK)
struct LOC_CallEvalPred *LINK;
{
  char ch;

  GetChar(&ch);
  LINK->result = IntResult(LINK->argval[0], LINK->e, (long)ch);
}


static void DOPUT(LINK)
struct LOC_CallEvalPred *LINK;
{
  long CH;

  CH = Evaluate(LINK->argval[0], LINK->e, 0L);
  if ((unsigned long)CH > 255) {
    Moan(3L, abortZ);
    return;
  }
  if (OUTFLAG)
	 putc((char)CH, PROG);
  else
	 putchar((char)CH);
}


static void DoOp(LINK)
struct LOC_CallEvalPred *LINK;
{
  long p;
  atomentry *a;
  optype f;
  atomentry *WITH;

  if ((LINK->argval[0]->info.tag != intT) | (!IsAtom(LINK->argval[1])) |
      (!IsAtom(LINK->argval[2])))
    Moan(24L, abortZ);
  p = LINK->argval[0]->info.UU.ival;
  a = LINK->argval[1]->info.UU.U0.name;
  if (p < 1 || p > MaxPrec)
    Moan(24L, abortZ);
  if (a == fxA)
    f = fxO;
  else if (a == fyA)
    f = fyO;
  else if (a == xfA)
    f = xfO;
  else if (a == yfA)
    f = yfO;
  else if (a == xfxA)
    f = xfxO;
  else if (a == xfyA)
    f = xfyO;
  else if (a == yfxA)
    f = yfxO;
  else
    Moan(24L, abortZ);
  WITH = LINK->argval[2]->info.UU.U0.name;
  WITH->oclass = f;
  WITH->oprec = p;
}


static void DoName(LINK)
struct LOC_CallEvalPred *LINK;
{
  node *x, *y;
  long ch;

  if (IsAtom(LINK->argval[0])) {
    LINK->result = Unify(LINK->argval[1],
			 ListRep(LINK->argval[0]->info.UU.U0.name->ident),
			 LINK->e, 0, 0L);
    return;
  }
  StartAtom();
  x = LINK->argval[1];
  while (IsFunc(x, consA, 2L)) {
    y = Deref(x->info.UU.U0.son, LINK->e);
    if (y->info.tag != intT)
      Moan(19L, abortZ);
    ch = y->info.UU.ival;
    if ((unsigned long)ch > ordmaxchar)
      Moan(3L, abortZ);
	 AtomChar((char)ch);
    x = Deref(x->info.UU.U0.son->brother, LINK->e);
  }
  if (!IsFunc(x, nilA, 0L))
    Moan(19L, abortZ);
  LINK->result = Unify(LINK->argval[0], MakeFunc(LookUp(listing), 0L, NULL),
		       LINK->e, 0, 0L);
}


static void DoFunctor(LINK)
struct LOC_CallEvalPred *LINK;
{
  node *x, *y;
  long i, m;

  switch (LINK->argval[0]->info.tag) {

  case funcT:
    if (!IntResult(LINK->argval[2], LINK->e,
		   LINK->argval[0]->info.UU.U0.arity))
      LINK->result = false;
    else
      LINK->result = Unify(LINK->argval[1],
	  MakeFunc(LINK->argval[0]->info.UU.U0.name, 0L, NULL), LINK->e, 0,
	  0L);
    break;

  case intT:
    LINK->result = false;
    break;

  case varT:
  case anonT:
    if (LINK->argval[2]->info.tag != intT)
      Moan(15L, abortZ);
    m = LINK->argval[2]->info.UU.ival;
    if (IsAtom(LINK->argval[1]) && m >= 0) {
      x = NULL;
      for (i = m; i >= 1; i--) {
	y = MakeVar(NULL);
	y->brother = x;
	x = y;
      }
      LINK->result = Unify(LINK->argval[0],
			   MakeFunc(LINK->argval[1]->info.UU.U0.name, m, x),
			   LINK->e, 0, 0L);
    } else if (LINK->argval[1]->info.tag == intT && m == 0)
      LINK->result = IntResult(LINK->argval[0], LINK->e,
			       LINK->argval[1]->info.UU.ival);
    else
      Moan(15L, abortZ);
    break;
  }
}


static void DoArg(LINK)
struct LOC_CallEvalPred *LINK;
{
  node *x;
  long i, n;

  if (LINK->argval[0]->info.tag != intT || LINK->argval[1]->info.tag != funcT) {
    LINK->result = false;
    return;
  }
  n = LINK->argval[0]->info.UU.ival;
  if (n < 1 || n > LINK->argval[1]->info.UU.U0.arity) {
    LINK->result = false;
    return;
  }
  x = LINK->argval[1]->info.UU.U0.son;
  for (i = 2; i <= n; i++)
    x = x->brother;
  LINK->result = Unify(LINK->argval[2], x, LINK->e, LINK->e, 0L);
}


static void DOCLENV(LINK)
struct LOC_CallEvalPred *LINK;
{
  env e1;

  if (LINK->argval[0]->info.tag == funcT) {
    if (LINK->argval[0]->info.UU.U0.name->sys)
      Moan(29L, abortZ);
    NewEnv(&e1, NULL, 0, LINK->argval[0]->info.UU.U0.name->UU.proc, 0L);
    LINK->result = IntResult(LINK->argval[1], LINK->e, (long)e1);
    return;
  }
  for (hptr = 1; hptr <= HashSize; hptr++) {
    pptr = hashtable[hptr - 1];
    while (pptr != NULL) {
      if (!(pptr->sys || pptr->UU.proc == NULL))
	goto _L1;
      pptr = pptr->chain;
    }
  }
  LINK->result = false;
  goto _L2;
_L1:
  NewEnv(&e1, NULL, 0, pptr->UU.proc, 0L);
  LINK->result = IntResult(LINK->argval[1], LINK->e, (long)e1) & Unify(
		   LINK->argval[0], MakeFunc(pptr, 0L, NULL), LINK->e, 0, 0L);
_L2: ;
}


static void DOUCL(LINK)
struct LOC_CallEvalPred *LINK;
{
  env E1;
  node *T;
  clause *WITH;

  if (!FindClause(&display[LINK->argval[2]->info.UU.ival - 1].Fclause,
		  LINK->argval[0], LINK->e)) {
    LINK->result = false;
    return;
  }
  WITH = display[LINK->argval[2]->info.UU.ival - 1].Fclause;
  NewEnv(&E1, NULL, 0, display[LINK->argval[2]->info.UU.ival - 1].Fclause,
	 WITH->Nvars);
  T = MakeVar(NULL);
  if (WITH->body == NULL)
    Bind(&T->info, MakeFunc(trueA, 0L, NULL), 0, 0L);
  else
    GetBody(&T->info, WITH->body, E1);
  LINK->result = Unify(LINK->argval[0], WITH->head, LINK->e, E1, 0L) &
		 Unify(LINK->argval[1], T, E1, E1, 0L);
  if (LINK->result)
    DisposeEnv();
  else
    KillStacks(E1 - 1);
  LINK->result = IntResult(LINK->argval[3], LINK->e, (long)WITH->head->scope)
                 & LINK->result;
}


static void DOADVCL(LINK)
struct LOC_CallEvalPred *LINK;
{
  if (display[LINK->argval[0]->info.UU.ival - 1].Fclause == NULL)
    LINK->result = false;
  else
    display[LINK->argval[0]->info.UU.ival - 1].Fclause =
      display[LINK->argval[0]->info.UU.ival - 1].Fclause->chain;
}


static void DoAdvP(LINK)
struct LOC_CallEvalPred *LINK;
{
  pptr = pptr->chain;
  while (hptr <= HashSize) {
    while (pptr != NULL) {
      if (!(pptr->sys || pptr->UU.proc == NULL))
	goto _L1;
      pptr = pptr->chain;
    }
    hptr++;
    pptr = hashtable[hptr - 1];
  }
  LINK->result = false;
  goto _L2;
_L1:
  display[LINK->argval[0]->info.UU.ival - 1].Fclause = pptr->UU.proc;
_L2: ;
}


static void DOZAP(LINK)
struct LOC_CallEvalPred *LINK;
{
  clause *WITH;

  WITH = display[LINK->argval[0]->info.UU.ival - 1].Fclause;
  WITH->Denied = true;
  DTERM(&WITH->head);
  DTERM(&WITH->body); 
}


static void DoNetEnv(LINK)
struct LOC_CallEvalPred *LINK;
{
  if (net_clause_base == NULL)
    LINK->result = false;
  else
    net_ptr = net_clause_base;
}


static void DoAdvNet(LINK)
struct LOC_CallEvalPred *LINK;
{
  if (net_ptr->chain == NULL)
    LINK->result = false;
  else
    net_ptr = net_ptr->chain;
}


static void DoUnet(LINK)
struct LOC_CallEvalPred *LINK;
{
  clause *WITH;
  env e1;
  if (net_ptr == NULL | net_ptr->cls->Denied)
    LINK->result = false;
  else {
    WITH = net_ptr->cls;
    NewEnv(&e1, NULL, 0, net_ptr->cls, WITH->Nvars);
    LINK->result = Unify(LINK->argval[0], WITH->head,LINK->e, e1, 0);
    if (LINK->result) DisposeEnv(); 
    else KillStacks(e1-1);
  }
}


static node *gettrail(LINK)
struct LOC_CallEvalPred *LINK;
{
  trailentry *t;
  node *x, *v, *WITH;

  t = display[Evaluate(LINK->argval[0], LINK->e, 0L) - 1].Ftrail->chain;
  x = MakeFunc(nilA, 0L, NULL);
  while (t != NULL) {
    WITH = t->boundvar;
    if (WITH->info.tag != varT && (WITH->scope < 0 || WITH->node_ptr != NULL)) {
      v = MakeVar(NULL);
      Bind(&v->info, t->boundvar, 0, 0L);
      v->brother = x;
      x = MakeFunc(consA, 2L, v);
    }
    t = t->chain;
  }
  return x;
}


boolean notmember(x, l)
node *x;
element *l;
{
  boolean result;

  result = true;
  while (l != NULL && result) {
    result = x != l->proc;
    l = l->chain;
  }
  return result;
}


static node *nproc(LINK)
struct LOC_CallEvalPred *LINK;
{
  element *p, *q;
  trailentry *t;
  node *x, *v;
  net_node_record *n;
  node *WITH, *WITH1;

  p = NULL;
  q = NULL;
  t = display[Evaluate(LINK->argval[0], LINK->e, 0L) - 1].Ftrail->chain;
  x = MakeFunc(nilA, 0L, NULL);
  while (t != NULL) {
    WITH = t->boundvar;
    n = WITH->node_ptr;
    if (WITH->info.tag != varT) {
      while (n != NULL) {
	WITH1 = n->count;
	if (WITH1->info.UU.ival <= 0 && notmember(WITH1->brother, p)) {
	  v = MakeVar(NULL);
	  Bind(&v->info, WITH1->brother, 0, 0L);
	  v->brother = x;
	  x = MakeFunc(consA, 2L, v);
          p = (element *)malloc(sizeof(element));
          p->proc = WITH1->brother;
          p->chain = q;
          q = p;
	}
	n = n->chain;
      }
    }
    t = t->chain;
  }
  q = p;
  while (q != NULL) {
    p = q;
    q = q->chain;
    free(p);
  }
  return x;
}


static void Generalize(LINK)
struct LOC_CallEvalPred *LINK;
{
  trailentry *p, *t, *t0;
  net_node_record *n;

  gencontext = true;
  t0 = display[Evaluate(LINK->argval[0], LINK->e, 0L) - 1].Ftrail->chain;
  p = t0;
  while (p != NULL) {
    t = t0;
    while (t != NULL) {
      if (t != p && t->boundvar->scope < 0 && p->boundvar->scope < 0 &&
	  p->VarVal == NULL) {
	if (Unify(t->boundvar, p->boundvar, 0, 0, 0L)) {
	  t->VarVal = p->boundvar;
	  if (t->boundvar->default_ != NULL)
	    p->boundvar->default_ = t->boundvar->default_;
	  if (p->boundvar->node_ptr == NULL)
	    p->boundvar->node_ptr = t->boundvar->node_ptr;
	  else {
	    n = p->boundvar->node_ptr;
	    while (n->chain != NULL)
	      n = n->chain;
	    n->chain = t->boundvar->node_ptr;
	  }
	  t->boundvar->node_ptr = NULL;
	}
      }
      t = t->chain;
    }
    p = p->chain;
  }
  gencontext = false;
}


boolean CallEvalPred(call_, e_, routine, arity_)
node *call_;
env e_;
evalpred routine;
long arity_;
{
  struct LOC_CallEvalPred V;
  FILE *TEMP;

  V.call = call_;
  V.e = e_;
  V.arity = arity_;
  GetArgs(&V);
  V.result = true;
  switch (routine) {

  case cutR:
    Cut(V.e);
    break;

  case callR:
    DoCall(&V);
    break;

  case readR:
    V.result = Unify(ReadIn(), V.argval[0], 0, V.e, 0L);
    break;

  case writeR:
    if (OUTFLAG)
      WriteOut(&PROG, V.argval[0], V.e);
    else {
      TEMP = stdout;
		WriteOut(&TEMP, V.argval[0], V.e);
    }
    break;

  case get0R:
    DoGet0(&V);
    break;

  case putR:
    DOPUT(&V);
    break;

  case nlR:
    if (OUTFLAG)
      putc('\n', PROG);
    else
      putchar('\n');
    break;

  case opR:
    DoOp(&V);
    break;

  case abortR:
    Abort();
    break;

  case endR:
    if (!debug_state && mode == userM)
      exit(10);   /* halt */
    else
      haltflag = true;
    break;

  case atomR:
    V.result = IsAtom(V.argval[0]);
    break;

  case integerR:
    V.result = (V.argval[0]->info.tag == intT);
    break;

  case varR:
    V.result = (((1L << ((long)V.argval[0]->info.tag)) &
		 ((1L << ((long)varT)) | (1L << ((long)anonT)))) != 0);
    break;

  case nameR:
    DoName(&V);
    break;

  case isP:
	 V.result = IntResult(V.argval[0], V.e, Evaluate(V.argval[1], V.e, 0L));
    break;

  case ltR:
    V.result = (Evaluate(V.argval[0], V.e, 0L) < Evaluate(V.argval[1], V.e, 0L));
    break;

  case assertaR:
    AddClause(V.argval[0], V.e, true, 0);
    break;

  case assertzR:
    AddClause(V.argval[0], V.e, false, 0);
    break;

  case assertR:
    AddClause(V.argval[0], V.e, false, (int)V.argval[1]->info.UU.ival);
    break;

  case functorR:
    DoFunctor(&V);
    break;

  case argR:
    DoArg(&V);
    break;

  case listallR:
    if (OUTFLAG)
      DOLISTALL(&PROG);
    else {
      TEMP = stdout;
		DOLISTALL(&TEMP);
    }
    break;

  case listingR:
    if (OUTFLAG)
      DOLIST(&PROG, V.argval[0], true);
    else {
      TEMP = stdout;
		DOLIST(&TEMP, V.argval[0], true);
    }
    break;

  case clenvR:
    DOCLENV(&V);
    break;

  case advclR:
    DOADVCL(&V);
    break;

  case advpR:
    DoAdvP(&V);
    break;

  case uclR:
    DOUCL(&V);
    break;

  case zapR:
    DOZAP(&V);
    break;

  case dprocR:
    TEMP = stdout;
	 DOLIST(&TEMP, V.argval[0], false);
    break;

  case consultR:
    dofile(V.argval, false, false);
    break;

  case openR:
    dofile(V.argval, true, false);
    break;

  case dprogR:
    dofile(V.argval, false, true);
    break;

  case closeR:
    doclose();
    break;

  case resR:
    RES(&V);
    break;

  case defR:
    Dodef(&V);
    break;

  case failR:
    V.result = false;
    break;

  case listR:
    listing = true;
    break;

  case nolistR:
    listing = false;
    break;

  case stepR:
    step = true;
    break;

  case nostepR:
    step = false;
    break;

  case debugR:
    debugging = true;
    break;

  case nodebugR:
    debugging = false;
    break;

  case moreR:
    more = true;
    break;

  case nomoreR:
    more = false;
    break;

  case traceR:
    tracing = true;
    break;

  case notraceR:
    tracing = false;
    break;

  case topR:
    V.result = IntResult(V.argval[0], V.e, (long)V.e);
    break;

  case netenvR:
    DoNetEnv(&V);
    break;

  case advnetR:
    DoAdvNet(&V);
    break;

  case unetR:
    DoUnet(&V);
    break;

  case genR:
    Generalize(&V);
    break;

  case netmodeR:
    if (V.argval[0]->info.tag == intT)
      netmode = Evaluate(V.argval[0], V.e, 0L);
    else
      V.result = IntResult(V.argval[0], V.e, netmode);
    break;

  case trailR:
    V.result = Unify(gettrail(&V), V.argval[1], 0, V.e, 0L);
    break;

  case nprocR:
    V.result = Unify(nproc(&V), V.argval[1], 0, V.e, 0L);
    break;

  case dellazyR:
    dellazy();
    break;

  case eqvR:
    V.result = (V.argval[0] == V.argval[1]);
    break;

  case shellR:
    CallShell(V.argval[0]);
	 break;

  }
  return V.result;
}


static void setmode(m)
phase m;
{
  mode = m;
  listing = false;
  linelength = 0;
  charpos = 0;
  haltflag = false;
  FileEnded = false;
}


void GOAL(X, E)
node *X;
env E;
{
  node *y;
  boolean success;

  y = Deref(X, E);
  if (y->info.tag != funcT)
    Moan(16L, abortZ);
  success = Execute(X, E);
  switch (mode) {

  case sysM:
  case progM:
    if (!success)
      Moan(26L, abortZ);
    break;

  case userM:
    if (!haltflag) {
      putchar('\n');
      if (success)
	printf("yes\n");
      else
	printf("no\n");
      putchar('\n');
    }
    break;
  }
}


void TopLevel()
{
  node *x;
  env e;

  do {
    charpos = 0;
    linelength = 0;
    OUTFLAG = false;
    SAVEVARS = true;
    choicepoint = 0;
    NewEnv(&e, NULL, 0, NULL, 0L);
    if (mode == userM)
      GOAL(MakeFunc(topA, 0L, NULL), 0); 
    else {
      x = ReadIn();
      if (IsFunc(x, questionA, 1L))
	GOAL(x->info.UU.U0.son, 0);
      else
	AddClause(x, 0, false, 0);
    }
    KillStacks(0);
  } while (!haltflag);
}


main(argc, argv)
int argc;
char *argv[];
{
  int i;
  node *X;
  env E;
  interrupted = false;
  signal(SIGINT,intrpt);
  InitRead();
  net_clause_base = NULL;
  netmode = 3;
  gencontext = false;
  lazycontext = false;
  qbase = NULL;
  INIT();
  printf("\nNCL/Prolog V2.0\n");
  printf("(C) 1993 Z. Markov, II-BAS\n");
  printf("--------------------------\n\n");
  setmode(sysM);
  PROG = fopen("ncl.lib", "r");
  if (PROG == NULL)
  _EscIO(10,"ncl.lib");
  debugging = false;
  tracing = false;
  debug_state = false;
  step = false;
  more = true;
  TopLevel();
  if (PROG != NULL)
    fclose(PROG);
  PROG = NULL;
  failA->sys = true;
  repeatA->UU.proc->chain = repeatA->UU.proc;
  andG = commaA->UU.proc;
  or1G = semiA->UU.proc->chain->chain;
  or2G = or1G->chain;
  for (i = 1; i < argc; i++)
  { 
    PROG = fopen(argv[i], "r");
    if (PROG == NULL)
    _EscIO(10,argv[i]);
    setmode(progM);
    do {
      NewEnv(&E, NULL, 0, NULL, 0L);
      X = ReadIn();
      if (IsFunc(X, questionA, 1L))
        GOAL(X->info.UU.U0.son, E);
      else
        AddClause(X, E, false, 0);
      KillStacks(E - 1);
    } while (!haltflag);
    if (PROG != NULL)
    fclose(PROG);
    PROG = NULL;
  }
  setmode(userM);
  TopLevel();
  if (PROG != NULL)
  fclose(PROG);
  exit(10);
}
