;top
;===========================================================================
;
;     PROSIT  --  Programming in SItuation Theory
;
;     Version 0.3
;
;     Written in Ibuki Common Lisp for the HP Bobcat
;
;     History
;     ----------
;     Wednesday, July 19, 1989
;           Started major work on 0.1, creating new sit structure.
;     Friday, July 28, 1989
;           Corrected forward-chaining to ignore backward-chained stuff.
;     Monday, July 31, 1989
;           Fixed unification problems, repeated code eliminated.
;     Tuesday, August 15, 1989
;           Version 0.2 completed, placed in prosit.o.  Unification
;           really fixed this time, and stuff broken in process repaired.
;     Thursday, August 17, 1989
;           Since Tuesday added the Exists predicate, ``No'' answers,
;           improved gensym, a part-of relation, the respects/supports
;           distinction, the carries predicate, and made rules non-
;           inheritable, and implemented setof and bagof.
;     Friday, August 18, 1989
;           Implemented the subtype-like subsituation relation
;           (with constraint inheritance added back in).
;     Thursday, August 24, 1989
;           Implemented prolog-like tracing, and cleaned-up and
;           speeded up stuff relating to the goal stack, like srg.
;     Friday, August 25, 1989
;           Implemented "cut".  Simplified usage of deref. Made contexts
;           into numerals instead of symbols.  Speeded up gencon.
;     Monday, August 28, 1989
;           "in" vs. "!=" distinction.  "defsys" to do special forms.
;           unary "!=" predicate implemented as basis of the binary "!=".
;     Tuesday, August 29, 1989
;           Reinstated <= and => as predicates, changed ROOT to a 
;           floating TOP.  Worked on simple partial compiler.
;     Friday, September 1, 1989
;           Replaces pushs, pops, and pushup with a unary "in" and
;           a 0-ary and a unary "out".
;     Monday, September 18
;           "carries" fixed
;     Wednesday, September 20, 1989
;           Introduction of reserved names (nil, !=, ...)
;     Friday, September 22, 1989
;           'cut' cuts away clauses now, not just literals.
;     Wednesday, September 27, 1989
;           Merging of Situations (function mergesit)
;     Tuesday, October 3, 1989
;           the results of make-functions (make-unify, make-subchunk...)
;           are undone, if the assertion they are part of fails
;     Thursday, October 12, 1989
;           Forward-Chaining is now complete
;     Saturday, August 10, 1991
;           Started implementing default-reasoning: d<
;     add for subdefaults special treatment already present for subtypes/sits
;     do something against loops
; -   "update": default rules not yet used for forward chaining
; -   !=: inferences for default
; -   a predicate (rule) for finding all default rules
;============================================================================

;'
;list has to be extended (rule, resp) or use @syshash
(setq reserved '(nil != <- -> [_ _] ! -! <= =>  ? = !!-  @< +! d<))     


(defstruct (sit (:print-function print-sit))
  subsits supersits 
  abouts owner 
  name 
  param-hash 
  infons 
  resp-fc resp-bc
  non-local-bc
  variable-bc
  subchunks superchunks
  subtypes supertypes
  subdefaults superdefaults
  changed
  )

;;; Print a situation's name in brackets, with situation's owner, if
;;; any, also included.  Used in prompting the user.

(defun print-sit (sit &rest lstr)
  (let ((s (sit-owner sit)) (str (car lstr)))
    (cond (s
	    (princ "<" str) (prin1 (sit-name sit) str)
	    (loop
	      (unless (sit-owner s) (princ ">" str) (return))
	      (princ "." str) (prin1 (sit-name s) str)
	      (setf s (sit-owner s))))
	  (t
	    (princ "<" str) (prin1 (sit-name sit) str) (princ ">" str)))))

(defvar result)
(defvar /i)
(defvar /tail)
(defvar /bagres)
(defvar /bagcon)
(defvar /res)

(proclaim '(special @syshash
                    @lookup
                    reserved
                    /goals /sit /quantifieds /no-mode 
		    /con *contexts* /sysvars /query /cutcon /cutpreds
		    /topsit /duals
		    /quantifiers /trace /goal /mastertrace
                    /tracepreds /traceall
		    /lint /depth /oldoutput
                    /restore
                    /fwc-rule /fwc-sit
                    fastdemop))


(defstruct (cell (:print-function print-cell))
  val					; The value we're bound to
  con					; The contextthe value belongs in
  )

;;; Used in debugging the interpreter to print the cell structure in a more
;;; abbreviated form than the usual #S(CELL VAL XXX CON XXX).  Should
;;; never be displayed to the user, though.

(defun print-cell (cell &rest lstr)
  (let ((str (car lstr)))
    (princ "[" str)
    (prin1 (cell-val cell) str)
    (princ "," str)
    (prin1 (cell-con cell) str)
    (princ "]" str)))

;;; Generates a "system variable" consisting of a "*" followed by a
;;; number.  Used in derive-vague.  System variable bindings should
;;; not be displayed by user-confirm.

(defun genvar () (intern (symbol-name (gensym "*"))))

;;; Get a new context number, usually 1 greater than the current
;;; context.  Sometimes this number is already in use in *contexts*,
;;; in which case we have to move up to higher numbers.

(defun gencon ()  ;get a new context number that's not in *contexts*
  (let ((n (1+ /con)))
    (loop (unless (assoc n *contexts*)
		  (push (list n) *contexts*)
		  (return n))
	  (incf n))))

;;; Macro used to isolate the *contexts* dynamic variable so that
;;; changes to it will be undone during backtracking.

(defmacro context (forms)
  `(let ((*contexts* *contexts*))
     . ,forms))

;;; Macro to get the value associated with a context number in
;;; the *contexts* association list.

(defmacro vl (con)
  `(cdr (assoc ,con *contexts*)))

;;; Like vl above but goes on to get the cell associated with a
;;; particular variable in the retrieved context. 

(defmacro var-cell (var con)
  `(cdr (assoc ,var (cdr (assoc ,con *contexts*)))))

;;; Macro to enter (set /sit to) a new situation, and set up
;;; /goals to return to the old sit when this goal exits.

(defmacro in-sit (sit &rest forms)
  `(let ((/goals (cons (imple &setsit /sit) /goals))
	 (/sit ,sit))
     . ,forms))

;;; Macro to make an implementation goal.  The function-name
;;; "func" is not evaluated, but the argument "arg" is.

(defmacro imple (func arg)
  `(make-imple :func ',func :arg ,arg))

(setf @lisphash (make-hash-table :test 'equal))
(defmacro makeknown (function)
  `(setf (gethash ',function @lisphash) t))
(makeknown print)
(makeknown prin1)
(makeknown random)
(makeknown >=)
(makeknown >)
(makeknown <=)
(makeknown <)
(makeknown +)
(makeknown -)
(makeknown *)
(makeknown /)
(makeknown sit-p)













;--------------------------------------------------------------
; system information hash table
; method of storage
;
; (query <pred> <arity>) -> function
; (assert <pred> <arity>) -> function
; (simple <pred> <arity>) -> t
; (infon <pred> <arity>) -> t
;--------------------------------------------------
(setf @syshash (make-hash-table :test 'equal))
(defmacro defsys (info &rest stuff)
  `(progn
     (setf (gethash ',info @syshash) ,(if stuff (list 'quote (car stuff)) t))
     ,(if (cdr stuff)
	  `(defun . ,stuff))))

(defmacro setf-restore (form1 form2)
`(progn
  (setf /restore (cons (list 'special ',(car form1) ,(cadr form1) ,form1)
                       /restore))
  (setf-symbol ',(car form1) ,(cadr form1) ,form2)))
;'  (print "setf-restore")
;'  (print /restore)))

;;; macro to look up information stored in the @syshash hash table.
;;; the "tag" is not evaluated, but the "args" are.  result is
;;; copied to the global, @lookup.

(defmacro lookup (tag &rest args)
  `(setf @lookup (gethash (list ',tag . ,args) @syshash)))

;;; Asserts subsit relation.  Various failures reported in lint.

(defmacro change-frame (form infon sit)
  `(when (if /restore
	     ,form
	   (if ,form t
		   (and (undo /restore) nil)))
         (fwc-proto ,infon)
;(print "change-frame update")
;(print /goals)
	 (update ,sit)
	 T))

;;; This macro sets up a "catch" to catch throws caused by backtracking
;;; to a cut.  When this happens, it checks to make sure it's the
;;; context that the "cut" wanted to cut back to.  If it's not, we
;;; keep on throwing back.

(defmacro newcatch (tag form)
  `(let ((catchvalue nil))
(catch ,tag (setf catchvalue ,form))
  catchvalue))


(defmacro catchcut (&rest forms)   
;'
`(progn
  (setf /cutcon nil)
  (cond ((newcatch 'cut . ,forms))
	(/cutcon
;           (if (and (print "huch") (print /cutcon)))
	  (unless (>= /cutcon /con) (throw 'cut nil))))))

;--------------------------------------------------------------------
;       MAJOR HIGH-LEVEL CONTROL AND USER-INTERFACE FUNCTIONS

;--------------------------------------------------------------------
; run
;       Start prosit.  Uses no global variables, so may be called
;   from within prosit itself.  (As a lisp function, claro).
;   if input-loop returns "t" to succeed or "nil" to backtrack,
;   that has to meaning here so we just go back into input-loop.
;--------------------------------------------------------------------
(defun run ()
  (print-level 10)
  (print-length 100)
  (let ((/sit (make-sit :name 'top))
	(/lint nil) 
	(/depth 0)
	(/con 0)		;The number of the context.
	(*contexts* '((0)))
	(/sysvars nil)
	(/quantifieds nil)
	(/quantifiers '(E setof))
	(/no-mode nil) (/mastertrace nil) (/tracepreds nil)
	(/duals nil)
	(/oldoutput nil)
	(/cutcon nil)
        (/restore nil)
        (/fwc-rule nil)
        (/fwc-sit nil)
        (fastdemop nil)
	)
    (gensym 0)
    (loop
      (if (eq 'exit (input-loop *standard-input* t nil nil)) (return t)))))

;-----------------------------------------
; cont
;        Continue PROSIT in case of a
;  lisp error.  Fails the current query.
;-----------------------------------------
(defun cont ()
  (throw 'continue nil))

;;; The main input-loop.  verbosep is nil when called by "load",
;;; meaning no output is printed.  demop is t when called by "demo",
;;; meaning we echo the input and wait for a (read-line).
;;; BTW, read-char-no-hang doesn't work right and is only good for
;;; skipping over the chars that have already been entered, as it's
;;; used here.

(defun input-loop (stream verbosep demop fastdemop)
  (if demop (unless fastdemop (read-line)))
  (let ((input nil) (assert-mode nil) (/topsit /sit) (ch nil)
	(/depth (1+ /depth)) (/goal nil))
    (makunbound '/goal)
    (loop
      (setf input nil)
      (when verbosep
            (terpri)
	    (princ (make-string (* 4 (1- /depth))))
	    (prin1 /topsit)
	    ;;(format t "<~s> " (sit-name /topsit))
	    (if assert-mode (princ " ! ") (princ " ? "))
;	    (terpri)
	;    (if assert-mode (princ "asserting ...") (princ "querying ..."))
            (if (and fastdemop (> /depth 1)) (return nil))
	    (unless (or demop (= /depth 1))
		    (loop (unless (read-char-no-hang stream) (return)))
		    (setf ch (read-char stream))
		    (cond ((eq ch #\newline) (setf input 'b))
			  (t (unread-char ch stream)))))
      (if (not input) (setf input (read stream nil 'end-of-stream)))
      (if (eq input 'end-of-stream) (progn (close stream) (return t)))
      (if demop (progn (prin1 input)
		       (if fastdemop (terpri) (read-line))))
      (unless (case input
		    ((t) (if (> /depth 1) (return t)))
		    ((b) (if (> /depth 1) (return nil)))
		    ((!) (setf assert-mode t))
		    ((?) (setf assert-mode nil) t)
		    ((/) (setf assert-mode (not assert-mode)) t)
		    ((exit) (return 'exit)))
	      (if assert-mode
		  (answer (list '! input) verbosep)
		(answer input verbosep))))))

;;; do the given query, trying the dual also if verbosep is t
;;; and /duals is t. 

(defun answer (q verbosep)
  (if verbosep
      (let ((pos (try-polarity q t))
	    (neg (if /duals (try-polarity (neg q) nil))))
	(terpri)
	(write-line
	  (if pos (if neg "yes and no." "yes.")
	    (if neg "no." "unknown."))))
    (let ((/goals (list q (imple &settopsit nil))) (/query q)
	  (/sit /topsit))
      (catchcut (newcatch 'continue (srg))))))

;;; verbosely try one polarity of a goal.  /no-mode is set to
;;; distinguish between "yes, if" and "no, if" answers later.  puts
;;; together a goal-list with the user-confirm goal on it,
;;; and &settopsit also so that (in s) and (out) will change
;;; the top-level situation.

(defun try-polarity (query yes-mode)
  (let ((/no-mode (not yes-mode)) 
	(/goals 
	  (list query (imple &userconfirm nil) (imple &settopsit nil)))
	(/query query) (/sit /topsit)
	(/trace /mastertrace))
    (catchcut (newcatch 'continue (srg)))))
  

;;; given any query, forms its dual.  q --> (no q),  (no q) --> q.

(defun neg (query)
  (if (and (consp query) (eq (car query) 'no) (consp (cdr query)))
      (cadr query)
    (list 'no query)))

;-------------------------------------------------------
; user-confirm
;               an internal goal and function meaning
;   if there are any bindings, show them to the user and
;   allow the user to continue with more goals, succeed,
;   or backtrack.
;        goes to great pains to keep track of what
;   variable bindings have been displayed before, and
;   displays a binding iff there is a change in the
;   flattened form of the binding.
;-------------------------------------------------------
(defun user-confirm ()
  (let ((output nil) (newoutput nil) (b (vl /con)))
    (loop
      (unless b (return))
      (let ((node (pop b)))
	(if (not (member (car node) /sysvars))
	    (setf output 
		  (acons (car node) 
			 (remove-cells 
			   (flatten (cdr node) /con) /con) output)))))
    (cond ((setf newoutput 
		 (set-difference output /oldoutput :test #'equal))
	   (terpri)
	   (if /no-mode (write-line "no, if") (write-line "yes, if"))
	   (loop
	     (unless newoutput (return))
	     (format t "  ~s = ~s~&" (caar newoutput) (cdar newoutput))
	     (setf newoutput (cdr newoutput)))
	   (if /duals (if (let ((/goals nil)) (catchcut (? (neg /query))))
			  (write-line 
			    (if /no-mode "also, yes." "also, no."))))
	   (and (let ((/oldoutput output)) 
		  (if fastdemop (input-loop *standard-input* t t t)
		    (input-loop *standard-input* t nil nil)))
		(srg)))
	  ((srg)))))

;;; flatten recursively reduces all variables in the given expression
;;; in the proper context, to their bindings.  parameters are not
;;; reduced using param-hash.  variables quantified in an expression
;;; like exists or setof have to be displayed as-is.

(defun flatten (expr con)
 ; (print 'flatten)
 ; (print *contexts*)
  (cond ((varp expr)
	 (if (member expr /quantifieds) expr
	   (let ((cell (var-cell expr con)))
	     (if cell
		 (if (and (= (cell-con cell) con) (eq (cell-val cell) expr))
		     cell
		   (flatten cell con))
	       (if (= con /con) expr (make-cell :val expr :con con))))))
	((cell-p expr)
	 (flatten (cell-val expr) (cell-con expr)))
	((consp expr)
	 (flatten-cons expr con))
	(expr)))

;;; flatten-cons flattens cons cells and watches out for "exists" and
;;; "setof", in which case it has to note which variables are quantified
;;; in those expressions.  unfortunately this will also happen if
;;; the cons is inside a list, like (foo bar exists (*x) (p *x)).

(defun flatten-cons (expr con)
  (let ((car (flatten (car expr) con)))
    (if (and (member car /quantifiers) (consp (cdr expr)) 
	     (real-list (cadr expr)))
	(let ((/quantifieds (append (cadr expr) /quantifieds)))
	  (cons car (flatten (cdr expr) con)))
      (cons car (flatten (cdr expr) con)))))

;------------------------------------
; paramval
;           given a parameter, finds
; a value for the parameter in the
; current situation.  returns the
; value, or the parameter itself if failure.
;   is now fast!  because assumes that
; all parameter bindings are directly
; present in a sit instead of inherited.
;------------------------------------
(defun paramval (p)
  (let ((h (sit-param-hash /sit)))
    (if h
	(loop
	  (let ((pv (gethash p h p)))
	    (if (eq p pv) (return p)
	      (setf p pv))))
      p)))


;-----------------------------
; varp  checks if o is the
; symbol _ or any symbol
; starting with *.
;-----------------------------
(defun varp (o)
  (and (symbolp o)
       (not (equal (symbol-name o) ""))
       (or (eq (elt (symbol-name o) 0) #\*)
	   (eq o '_))))


;---------------------------------
; lisp-function
;   if g is not a system pred but
; is a proper lisp function, executes
; it with prosit variable substtutions
; and then srg's iff it returns nonnil.
; assumes that g is a list.
; -------------------------------------
(defun lisp-function (g)
  (when (and (real-list g)
	     (not (or (lookup query (car g))
		      (lookup query (car g) (length (cdr g)))
		      (lookup badlisp (car g))
		      (lookup badlisp (car g) (length (cdr g)))))
	     (newcatch 'abort-lisp (lisp-eval g)))
	(rlint "goal ~s succeeded as a lisp function." g)
;;;        (print 'inlispfunction)
	(srg)))


;;; simple lisp evaluator that, instead of producing an error
;;; when it gets an unbound variable, simply causes the lisp
;;; function as a goal to fail.

(defun lisp-eval (e)
  (cond ((listp e) (setf e (macroexpand e))
	 (cond ((gethash (car e) @lisphash)
		(apply (car e) (mapcar 'lisp-eval (cdr e))))
	       ((eq (car e) 'quote) (cadr e))
	       ((special-form-p (car e)) (eval e))))
	((varp e) (flatten e /con))
	((and (symbolp e) (boundp e)) (symbol-value e))
	((or (numberp e) (stringp e)) e)
	((throw 'abort-lisp nil))))

;;; coverts a variable name to an unused parameter symbol.
;;; *x --> x0, and then *x --> x1, etc.  used in skolemizing,
;;; in asserting "exists" predicates.

(defun varparam (var)
  (smart-gensym 
    (string-left-trim 
      (list (elt (symbol-name var) 0)) 
      (symbol-name var))))

;;; tells prosit not to use <= as a lisp less-than-or-equal.
;;; not needed now that "<=" is a predicate.  doesn't hurt tho.

(setf (get '<= 'bad-lisp) t)

;-----------------------------------
; hash-key-list
;                for any hash table,
;  returns a list of all keys to the
;  table (the domain of the function
;  represented by the table).
;-----------------------------------
(defun hash-key-list (htab)
  (let ((result nil))
    (declare (special result))
    (maphash 'build-hash-key-list htab)
    result))
(defun build-hash-key-list (&rest lk)
  (setf result (cons (car lk) result)))

;;; makes two expressions unify, both in a situation and in all
;;; of its "super" situations of various types.  so that in
;;; unification, the tree of unifiable parameters will all be
;;; together in the current situation.

(defun chain-make-unify (a b)
  (recursive-make-unify (flatten a /con) (flatten b /con)))
(defun recursive-make-unify (a b)
  (and (make-unify a b)
       (let ((sits (make-set (append (sit-supersits /sit)
;;;'				(sit-superchunks /sit)
				    (sit-supertypes /sit)))))
	 (loop
	   (unless sits (return t))
	   (let ((/sit (pop sits)))
	     (unless (recursive-make-unify a b) (return nil)))))))

;'substitute s1 by s2 in all structures of sit
(defun subst-sit (s1 s2 sit)        
  (let ((h (sit-param-hash (sit-owner sit)))
        (name1 nil)
        (name2 (sit-name sit)))
    (loop
      (setf name1 name2)
      (setf name2 (gethash name1 h))
      (when (sit-p name2)
        (setf (gethash name1 h) s2)
        (return))))
  (delete s1 (sit-abouts (sit-owner sit)))
  (let ((abouts (sit-abouts sit)) (s nil))
    (loop
      (unless abouts (return))
      (setf s (pop abouts))
      (setf (sit-owner s) s2)))
  (subst-field-sit s1 s2 'sit-subsits     'sit-supersits   sit)
  (subst-field-sit s1 s2 'sit-supersits   'sit-subsits     sit)
  (subst-field-sit s1 s2 'sit-subtypes    'sit-supertypes  sit)
  (subst-field-sit s1 s2 'sit-supertypes  'sit-subtypes    sit)
  (subst-field-sit s1 s2 'sit-subchunks   'sit-superchunks sit)
  (subst-field-sit s1 s2 'sit-superchunks 'sit-subchunks   sit))

(defun subst-field-sit (s1 s2 field1 field2 sit) 
  (let ((sits (funcall field1 sit)) (s nil))
    (loop
      (unless sits (return))
      (setf s (pop sits))
      (delete s1 (funcall field2 s))
      (setf-symbol field2 s (cons s2 (funcall field2 s))))))

;'unification of parameters of sit1 within /sit is not necessary
(defun newhashtable (sit1 sit2)
  (if (null (sit-param-hash sit2))
      t
    (if (null (sit-param-hash sit1))
	(progn 
	  (setf (sit-param-hash sit1) (make-hash-table))
	  (let ((h1 (sit-param-hash sit1)) 
		(params1 (hash-key-list (sit-param-hash sit1)))
		(h2 (sit-param-hash sit2)) 
		(params2 (hash-key-list (sit-param-hash sit2)))
		(p nil) (pv nil))
	    (and    
	     (loop 
	      (unless params1 (return t))
	      (setf p (pop params1))
	      (setf pv (gethash p h1))
	      (unless (chain-make-unify pv p) 
		      (return nil)))
	     (loop 
	      (unless params2 (return t))
	      (setf p (pop params2))
	      (setf pv (gethash p h2))
	      (unless (chain-make-unify pv p) 
		      (return nil)))))))))
   
;'
;is it a good idea to create a new situation?
;the sit and infon fields have to be created anyway, though
(defun mergesit (a b)
  (unless (or (member a (sit-set b 'sit-subsits nil))
          (member a (sit-set b 'sit-supersits nil))
;'          (member a (sit-set b 'sit-subchunks nil))
;'          (member a (sit-set b 'sit-superchunks nil))
          (member a (sit-set b 'sit-subtypes nil))
          (member a (sit-set b 'sit-supertypes nil)))
          (member a (sit-set b 'sit-subdefaults nil))
          (member a (sit-set b 'sit-superdefaults nil)))
  (let* ((/sit (make-sit :subsits (union (sit-subsits a) (sit-subsits b))
                         :supersits (union (sit-supersits a) (sit-supersits b))
                         :abouts (union (sit-abouts a) (sit-abouts b))
                         :owner (sit-owner a)
                         :name (sit-name a)
                         :param-hash (sit-param-hash a)
                         :infons (union (sit-infons a) (sit-infons b))
           ;' different test for union in merging the resp-fc's and resp-bc's?
                         :resp-fc (union (sit-resp-fc a) (sit-resp-fc b))
                         :resp-bc (union (sit-resp-bc a) (sit-resp-bc b))
                         :subchunks (union (sit-subchunks a) (sit-subchunks b))
                         :superchunks (union (sit-superchunks a) (sit-superchunks b))
                         :subtypes (union (sit-subtypes a) (sit-subtypes b))
                         :supertypes (union (sit-supertypes a) (sit-supertypes b))
                         :subdefaults (union (sit-subdefaults a) (sit-subdefaults b))
                         :superdefaults (union (sit-superdefaults a) (sit-superdefaults b))
                         :changed nil)))
    (if (newhashtable a b) /sit)))
       

         
;;; makes two expressions unify in a situation.  if the two are situation
;;; structures, merges the structures.  actually, makes one sit a
;;; "subtype" of the other, and makes both names point to the bigger one.
;;; bug: this doesn't trigger forward-chaining within the merged situation.
; must return true, if it succeeds
(defun make-unify (a b)
  (cond ((eql a b))
	;'  better use @syshash for it? but nil!
        ((or (member a reserved) (member b reserved))
	 ;' reserved names cannot be bound to parameters
	 (rlint "make ~s = ~s failed in situation ~s (reserved name)." a b /sit))
	((symbolp a)
	 (let ((pva (paramval a)))
	   (if (eq pva a) (assign a (flatten b /con)) 
	     (if (symbolp b)
		 (let ((pvb (paramval b)))
		   (if (eq pvb b) (assign b (flatten a /con))
		     (make-unify pva pvb)))
	       (make-unify pva b)))))
	((symbolp b)
	 (let ((pvb (paramval b)))
	   (if (eq pvb b) (assign b (flatten a /con)) 
	     (make-unify a pvb))))
	((and (consp a) (consp b))
	 (and (make-unify (car a) (car b))
	      (make-unify (cdr a) (cdr b))))
	((and (sit-p a) (sit-p b))
	   (let ((newsit (mergesit a b)))
           (cond (newsit
                  (subst-sit a newsit a)
                  (subst-sit b newsit b)
                  (setf /restore
                        (cons `(subst-sit ',newsit ',a ',a)
                               (cons `(subst-sit ',newsit ',b ',b) /restore)))
                  t)
                 (t (rlint "make ~s = ~s failed in situation ~s." a b /sit)))))
	((rlint "make ~s = ~s failed in situation ~s." a b /sit))))

;;; assigns a parameter to something, in param-hash.  may be another 
;;; parameter.  if it's a sit structure, and the current sit isn't the
;;; sit's owner, actually makes a new situation that inherits
;;; from the given one.  this is to keep different situations'
;;; "perspectives" on a situation distinct.
;;; must return assigned value

(defun assign (param val)
  (let ((h (sit-param-hash /sit)))
    (if (null h) (setf (sit-param-hash /sit)
		       (setf h (make-hash-table))
                       /restore 
                       (cons `(setf (sit-param-hash ',/sit) nil) /restore)))
    (cond 
;' example for this??
;((and (sit-p val) (not (member val (sit-abouts /sit))))
;	   (let ((sit (make-sit :name param :owner /sit :subtypes (list val))))
;	     (setf (sit-supertypes val) (cons sit (sit-supertypes val)))
;	     (setf (sit-abouts /sit) (cons sit (sit-abouts /sit)))
;	     (setf (gethash param h) sit)))
	  (t
(setf /restore (cons `(remhash ',param ',h) /restore))
;'(print "assign")
;'(print /restore)
(setf (gethash param h) val)))))

;;; makes one situation a subtype of another.  
;;; i.e., all information about the sub- is inherited by the super-.
;;; has to copy all parameter unification information from the sub- 
;;; to the super-type.

(defun make-subtype (sub sup)		;assume sub's not already a subsit.
  (setf-restore (sit-subtypes sup) (cons sub (sit-subtypes sup)))
  (setf-restore (sit-supertypes sub) (cons sup (sit-supertypes sub)))
  (let ((h (sit-param-hash sub)))
    (if (null h)
      t
	(context
	  ((let ((params (hash-key-list h)) (/sit sup) (p nil) (pv nil))
	    (loop
	      (unless params (return t))
	      (setf p (pop params))
	      (setf pv (gethash p h))
	      (unless (chain-make-unify pv p) (return nil)) )) )))))

(defun make-subdefault (sub sup)     ;assume sub's not already a subdefault.
  (setf-restore (sit-subdefaults sup) (cons sub (sit-subdefaults sup)))
  (setf-restore (sit-superdefaults sub) (cons sup (sit-superdefaults sub)))
  (let ((h (sit-param-hash sub)))
    (if (null h)
      t
	(context
	  ((let ((params (hash-key-list h)) (/sit sup) (p nil) (pv nil))
	    (loop
	      (unless params (return t))
	      (setf p (pop params))
	      (setf pv (gethash p h))
	      (unless (chain-make-unify pv p) (return nil)) )) )))))

;;;; similar to make-subtype above.

;;; format of entries of restore list
;;; either: (sit slot-access-function old-value)
;;; or:     (hashtable parameter)
;;; or:     ('subst-sit situation-that-was-merged result-of-merge)

(defun setf-symbol (name sit value)
     (cond ((eql name 'sit-subtypes)
	    (setf (sit-subtypes sit) value))
	   ((eql name 'sit-supertypes)
	    (setf (sit-supertypes sit) value))
	   ((eql name 'sit-superdefaults)
	    (setf (sit-superdefaults sit) value))
	   ((eql name 'sit-subdefaults)
	    (setf (sit-subdefaults sit) value))
	   ((eql name 'sit-subchunks)
	    (setf (sit-subchunks sit) value))
	   ((eql name 'sit-superchunks)
	    (setf (sit-superchunks sit) value))
	   ((eql name 'sit-subsits)
	    (setf (sit-subsits sit) value))
	   ((eql name 'sit-supersits)
	    (setf (sit-supersits sit) value))
	   ((eql name 'sit-abouts)
	    (setf (sit-abouts sit) value))
       	   (t (print "error in setf-symbol"))))




(defun undo (restorelist)
  (when restorelist
	(let ((restore-el (car restorelist)))
	  (if (eql (car restore-el) 'special)
	      (setf-symbol (cadr restore-el) (caddr restore-el) (cadddr restore-el))
	    (eval (car restorelist)))
	  (undo (cdr restorelist)))))
            
(defun make-subsit (sub sup)		;assume sub's not already a subsit.
  (setf-restore (sit-subsits sup) (cons sub (sit-subsits sup)))
  (setf-restore (sit-supersits sub) (cons sup (sit-supersits sub)))
  (let ((h (sit-param-hash sub)))
    (if (null h)
	t
      (context
	((let ((params (hash-key-list h)) (/sit sup) (p nil) (pv nil))
	  (loop
	    (unless params (return t))
	    (setf p (pop params))
	    (setf pv (gethash p h))
	    (unless (chain-make-unify pv p) (return nil)))))))))

;;; changes a list of expressions into a set of them, using prosit's
;;; unification to determine identity.  used in setof.

(defun make-set (lis)
  (let ((r nil) (c nil))
    (loop
      (if lis (setf c (pop lis)) (return r))
      (if (not (member c r :test 'prosit-equal)) (setf r (cons c r))))))

;;; sets the "top", here meaning "user-interface level" situation
;;; to the current situation in focus.  needed for (in) and (out) to
;;; have an effect on the user interface.

(defun set-topsit ()
  (setf /topsit /sit)
  (srg))

;;; record lint.  just adds the given formatted output to /lint.

(defun rlint (string &rest args)
  (setf /lint (cons (apply 'format nil string args) /lint))
  nil)

;;; determines if the last cdr in the list is nil.

(defun real-list (x)
  (or (null x) (and (consp x) (real-list (cdr x)))))

;;; used in make-set above

(defun prosit-equal (a b)
  (context ((unify a /con b /con))))

;;; query.  flattens the given goal, and queries it.  adds the &exit
;;; implementation goal if we're in trace mode. 

(defsys (assert ? 1) ?)
(defsys (query ? 1) ?)
(defun ? (g)
  (setf g (flatten g /con))
  ;  (format t "entering ? with goal ~s,~&" g)
  ;  (format t "remaining goals ~s~&" /goals)
  (if /trace (let ((/goals (cons (imple &exit g) /goals)))
	       (trace-display "calling" /con /sit g)
;	       (trace-prompt)
	       (cond ((?-core g))
		     (t (trace-display "failing" /con /sit g) nil)))
    (?-core g)))

;;; routes a query to derive, derive-vague, or a system predicate
;;; function.  for "simple" system preds, does (srg) for the pred.

(defun ?-core (g)
  (cond ((symbolp g)
	 (cond ((get g 'atomfunc) (funcall g))
	       ((free g) (derive-vague g))
	       ((derive g /sit))))
	((listp g)
	 (cond ((and (symbolp (car g)) (real-list g)
		     (or (lookup query (car g) (length (cdr g)))
			 (lookup query (car g))) )
		(if (apply @lookup (cdr g))
		    (if (or (lookup simple (car g))
			    (lookup simple (car g) (length (cdr g))))
			(srg)
		      t)))
	       ((free (car g)) (derive-vague g))
	       ((derive g /sit))
	       ((lisp-function g))))
	((derive g /sit))))

;;; prompts for a trace command.  a plain carriage return is
;;; converted to "s" for "step."

(defun trace-prompt ())
;  (let ((ch #\Newline) (input nil))
;    (loop
;      (princ " ? ")
;      (terpri)
;     (loop (unless (read-char-no-hang) (return)))
;;      (setf ch (read-char))
;      (cond ((eq ch #\Newline) (setf input 's))
;	    (t (unread-char ch) (setf input (read))))
;      (case input
;	    ((a q) (throw 'continue nil))
;	    ((s) (return))
;	    ((c l) (setf /trace nil) (return)))
;      (trace-help))))
;old trace-prompt
;(defun trace-prompt ()
;  (let ((ch) (input))
;    (loop
;      (princ " ? ")
;      (loop (unless (read-char-no-hang) (return)))
;      (setf ch (read-char))
;      (cond ((eq ch #\Newline) (setf input 's))
;	    (t (unread-char ch) (setf input (read))))
;      (case input
;	    ((a q) (throw 'continue nil))
;	    ((s) (return))
;	    ((c l) (setf /trace nil) (return)))
;      (trace-help))))
	
;;; Prints help for the trace commands.

(defun trace-help ()
  (terpri)
  (write-line "a,q = Aborts computation.")
  (write-line "c,l = Continues computation with tracing off.")
  (write-line "h,? = Shows this help.")
  (write-line "s, <RETURN> = Steps forward to next call.")
  (terpri))

;;; Determines if a variable has a binding in the current context.
;;; A variable that is "bound" to an unbound variable is not 
;;; considered to really be bound, in this function's view.

(defun bound (e)
  (Rbound e /con))
(defun Rbound (e con)
  (cond ((varp e)
	 (let ((node (assoc e (vl con))))
	   (and node (Rbound (cdr node) con))))
	((cell-p e)
	 (Rbound (cell-val e) (cell-con e)))
	(e)))

;;; Duh.

(defun free (e)
  (not (bound e)))

;;; Takes a goal g that is an unbound variable or has an unbound car,
;;; and tries to derive it, additionally matching it to the system
;;; predicates that can also serve as real infons.  Currently there
;;; are only 4 of these special forms.

(defun derive-vague (g)
  (or (derive g /sit)
      (let* ((v1 (genvar)) (v2 (genvar)) (/sysvars `(,v1 ,v2 . ,/sysvars)))
	(or (context
	      ((and (unify g /con `([_ ,v1 ,v2) /con)
		   ([_ v1 v2))))
	    (context
	      ((and (unify g /con `(!= ,v1 ,v2) /con)
		   (!= v1 v2))))
	    (context
	      ((and (unify g /con `(@< ,v1 ,v2) /con)
		   (@< v1 v2))))
	    (context
	      ((and (unify g /con `(d< ,v1 ,v2) /con)
		   (d< v1 v2))))
	    (context
	      ((and (unify g /con `(resp ,v1 ,v2) /con)
		   (resp v1 v2))))
	    (context
	      ((and (unify g /con `(<- ,v1 ,v2) /con)
		   (<- v1 v2))))))))


;-----------------------------------------------------------------
; derive
;              Tries to solve/derive the given goal by consulting
;  the infons and backward chaining constraints.  
;-------------------------------------------------------------
(defsys (query derive 2) derive)
(defun derive (g s)
  (cond ((rec-derive g s))
	((and 
	  (sit-subdefaults s)
	  (not (rec-derive (neg g) s)))
	 (default-derive g s))))

;;; Scans through the subsits and subtypes of the current situation,
;;; trying to prove the goal by matching it with infons or using
;;; backward-chaining rules.  When going into a subsit we set
;;; /sit to that situation (using in-sit), because we want the
;;; backward-chaining rules we use there NOT to use infons in
;;; the supersit we were in previously.  On the other hand, when
;;; going into a subtype situation, any bc-rules found there are
;;; also inherited to the supersits, so we can use infons in
;;; the supersit we were in previously when using those rules.
;;; Thus /sit doesn't need to be changed.

(defun rec-derive (g s)
  (or (let ((infons (sit-infons s)) (i nil))
	(loop (if infons (setf i (pop infons)) (return))
	      (if (unify-old-new g i) (return t))))
					;'
      (let ((rules (sit-resp-bc s)) (r nil) (/cutpreds nil))
	(loop (if rules (setf r (pop rules)) (return))
	      (unless (and (listp (cadr r))
			   (member (caadr r) /cutpreds))
		      (if (backward-chain g r) (return t)))))
      (dolist (info (sit-non-local-bc s))
	      (let ((/cutpreds nil))
		(unless (member (caadr (cdr info)) /cutpreds)
			(if (in-sit (car info) (backward-chain g (cdr info)))
			    (return t)))))
      (let ((sits (sit-subsits s)) (s nil))
	(loop (if sits (setf s (pop sits)) (return))
	      (if (in-sit s (rec-derive g s)) (return t))))
      (let ((sits (sit-subtypes s)))
	(loop (if sits (if (rec-derive g (pop sits)) (return t)) (return))))
      ))

(defun default-derive (g s)
 (let ((sits (sit-subdefaults s)))
	(loop (if sits (if (in-sit s (derive g (pop sits))) (return t)) (return)))))

; feature +/- default rules apply 
;             = inherits from default situation
; (duals) applies only to -default situations


;;; Described extensively in ~mpf/prositguide.

(defun backward-chain (goal rule)
  (context
    ((let ((newcon (gencon)))
      (and (unify goal /con (cadr rule) newcon)
	   (let ((/goals
		   `(,(imple &goals (cddr rule))
		     ,(imple &setcon /con)
		     . ,/goals))
		 (/con newcon))
;'
	     (cond ((catchcut (srg)) t)
                   ((if /cutcon 
                      (progn (setf /cutcon nil)
                             (if (listp (cadr rule))
		               (setf /cutpreds (cons (caadr rule) /cutpreds)))))
                    nil))))))))

;;; Displays a line of trace information.

(defun trace-display (s con sit goal)
  (when (or /traceall
	    (member (car goal) /tracepreds))
	(setf goal (remove-cells (flatten goal con) con))
	(format t "~a ~s ~a ~s  " con sit s goal)
        (terpri)))


(defsys (badlisp <=))
(defsys (badlisp =>))

;;; Asserts all the argument infons.  "?" infons are queried instead.
;;; Infons that succeed as queries are not asserted.  Assertable
;;; system predicates are routed to the appropriate functions.

(defsys (query !) !)
(defsys (assert !) !)
(defsys (simple !))
(defun ! (&rest asserts)
  (let ((i nil))
    (loop
      (unless asserts (return t))
      (setf i (pop asserts))
      (cond ((if (and (listp i) (eq (car i) '?) (real-list i) (= (length i) 2))
		 (if (let ((/goals `((! . ,asserts)))) (catchcut (? i)))
		     (return t))
	       (if /trace
		   (let ((/goals (list (imple &tron t))) (/trace nil)) (? i))
		 (let ((/goals nil)) (catchcut (? i))))))
	    ((and (listp i) (symbolp (car i)) (real-list i)
		  (or (lookup assert (car i) (length (cdr i)))
		      (lookup assert (car i))))
	     (let ((/goals nil) (/restore nil)) (catchcut (apply @lookup (cdr i)))))
	    (t (+! i))))))
;;; Changes all cells to variables.  [*x,3] --> *x3.
;;; This is neccessary before presenting something to the user,
;;; or asserting it into the database, since cells mean nothing
;;; outside of the interpreter's main goal-solving recursion.

(defun remove-cells (e con)
  (cond ((consp e) (cons (remove-cells (car e) con) 
			 (remove-cells (cdr e) con)))
	((cell-p e) (remove-cells (cell-val e) (cell-con e)))
	((eq e '_) (intern (concatenate 'string (symbol-name (genvar))
					(prin1-to-string con))))
	((varp e)
	 (let ((cell (var-cell e con)))
	   (if cell (remove-cells (cell-val cell) (cell-con cell))
	     (if (= con /con) e
	       (intern (concatenate 'string 
				    (symbol-name e) (prin1-to-string con)))))))
	(e)))

(defun update-list (sits)
  (loop
    (unless sits (return))
;(print "update-list update")
;(print /goals)
    (update (pop sits))))

(defun create-union (sits)
  (if sits
      (union (sit-resp-fc (car sits)) (create-union (cdr sits)))
    nil))

(defun update (sit)
  (when (and 
	 sit
	 (not (sit-changed sit)))
	(setf (sit-changed sit) t)
;(print "update: sit-changed")
;(print (sit-changed sit))
	(let* ((/sit sit) 
	       (subtypes (sit-set /sit 'sit-subtypes t))
	       (fc-rules (create-union subtypes)))
	  (loop 
					;            (print 'fc-rules)
					;            (print fc-rules)
					;            (print (sit-infons sit))
	   (unless fc-rules (return))
	   (forward-chain-off-rule (pop fc-rules))))
	(update-list (sit-supersits sit))
	(update-list (sit-superchunks sit))
	(update-list (sit-supertypes sit))
	(update (sit-owner sit))
	(setf (sit-changed sit) nil)))
  

(defsys (query +! 1) +!)
(defsys (assert +! 1) +!)
(defsys (simple +! 1))
(defsys (infon +! 1))
(defun +! (infon)
;'
  (setf infon (remove-cells infon /con))
  (setf (sit-infons /sit) (cons infon (sit-infons /sit)))
  (setf (sit-changed /sit) nil)
;(print "+! update")
  (fwc-proto infon)
;(print "+! update")
;(print /goals)
  (update /sit)
  t)


;;; Retracts all infons matching i from the current sit.  Scans through
;;; the infons, copying all those that don't match to a new list.
 
(defsys (query -! 2) sit-!)
(defun sit-! (s i)
  (setf s (flatten s /con))
  (if (symbolp s)
      (let ((/goals (cons (imple &setsit /sit) /goals))
	    (/sit (get-sit-named s)))
;;;	(format t "goals in sit-!: ~a" /goals)
	(-! i))))

(defsys (query -! 1) -!)
(defun -! (i) 
;;;	(format t "goals in sit-!: ~a" /goals)
  (context
   ((let ((matchn 0) (infons (sit-infons /sit)) (okinfons nil)
	  (testcon (gencon)))
      (loop
       (unless infons (return))
       (context
	((if (unify i /con (car infons) testcon)
	     (incf matchn)
	    (setf okinfons (cons (car infons) okinfons)))))
	(setf infons (cdr infons)))
      (if (= matchn 0) nil)
      ;;(rlint "-! : Infon ~s was not found in sit, succeeding anyway." i) 
      (if (> matchn 1)
	  (rlint "-! : ~s infons matching ~s were found, removing all of them." matchn i))
      (setf (sit-infons /sit) (reverse okinfons))
      (srg)))))

;;; Tests subsit.  The "intersection" call ensures that we only obtain
;;; (<- s1 s2) when the current sit "describes" both s1 and s2.  Sometimes
;;; s2 can merely be a subsit because of the way s1 was created.
;;; See the make-unify function.

(defsys (query <- 2) <-)
(defsys (assert <- 2) assert<-)
(defsys (infon <-))
(defun <- (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sits (sit-abouts /sit)))
	(loop
	  (unless sits (return))
	  (context
	       ((and (unify s2 /con (sit-name (car sits)) /con)
		    (prosit-member 
		      s1 (mapcar 'sit-name 
				 (intersection (sit-abouts /sit)
					       (sit-set (car sits)
							'sit-subsits nil))))
		    (return t))))
	  (setf sits (cdr sits))))))


(defun assert<- (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sit1 (get-sit-named s1)) (sit2 (get-sit-named s2)))
	(cond ((eq sit1 sit2)
	       (rlint "Can't assert (<- ~s ~s)." s1 s2)
	       (srg))
	      ((member sit1 (sit-set sit2 'sit-subsits nil))
	       (rlint "Assertion (<- ~s ~s) is already true." s1 s2)
	       (srg))
	      ((member sit2 (sit-set sit1 'sit-subsits nil))
	       (rlint "assert<- : (<- ~s ~s) would form a loop.  Fail." s1 s2))
	      (t
		(change-frame (make-subsit sit1 sit2) `(<- ,s1 ,s2) sit1)
		(srg))))))

(defsys (query -> 2) ->)
(defsys (assert ->  2) assert->)
(defsys (infon ->))
(defun -> (s1 s2) (<- s2 s1))
(defun assert-> (s1 s2) (assert<- s2 s1))

;;; Supports goes into a situation matching s and queries (!= i) there.

(defsys (query != 2) !=)
(defsys (assert != 2) assert!=)
(defsys (infon != 2))
(defun != (s i)
  (cond ((prosit-free s)
	 (or
	  (dolist (sit (sit-abouts /sit))
		  (context ((and (unify s /con (sit-name sit) /con)
				 (in-sit sit (unary!= i))
				 (return t)))))
;	  (let ((sits (sit-abouts /sit)) (sit nil))
;	    (loop (if sits (setf sit (pop sits)) (return))
;		  (context ((and (unify s /con (sit-name sit) /con)
;				 (in-sit sit (unary!= i))
;				 (return t))))))
	  (dolist (sit (sit-subtypes /sit))
		  (if (in-sit sit (!= s i))
		      (return t)))))
	((symbolp s) (in-sit (get-sit-named s) (unary!= i)))
	((sit-p s) (in-sit s (unary!= i)))))

;;; Goes into a situation matching s and asserts (!= i) there.

(defun assert!= (s i)
  (setf s (flatten s /con))
  (if (symbolp s)
      (let ((/goals (cons (imple &setsit /sit) /goals))
	    (/sit (get-sit-named s)))
	(! `(!= ,i)))))

;;; Goes into a sitaution matching s and queries i there.

; (defsys (query in 2) in)
; (defsys (assert in 2) assertin)
; (defun in (s i)
;   (cond ((prosit-free s)
; 	 (let ((sits (sit-abouts /sit)) (sit nil))
; 	   (loop (if sits (setf sit (pop sits)) (return))
; 		 (context ((and (unify s /con (sit-name sit) /con)
; 			       (in-sit sit (? i))
; 			       (return t)))))))
; 	((symbolp s) (in-sit (get-sit-named s) (? i)))
; 	((sit-p s) (in-sit s (? i)))))


;(defsys (assert circular 1) create-circular-sit)
;(defun create-circular-sit (sym)
;  (let ((sit (make-sit :name sym :owner /sit )))
;    (setf (sit-abouts sit) (list sit)
;	  (sit-abouts /sit) (cons sit (sit-abouts /sit)))
;    (let ((/sit sit))
;      (assign sym sit))
;    (assign sym sit)))

;;; Goes into a sitaution matching s and asserts i there.

; (defun assertin (s i)
;   (setf s (flatten s /con))
;   (if (symbolp s)
;       (let ((/goals (cons (imple &setsit /sit) /goals))
; 	    (/sit (get-sit-named s)))
; 	(! i))))
			
;;; Gets a sit structure in the current situation's /abouts list
;;; whose name matches the given symbol.  Creates one if needed.

(defun get-sit-named (sym)
  (let ((val (paramval sym)))
    (if (sit-p val) 
	val
      (let ((sit (make-sit :name sym :owner /sit)))
	(setf (sit-abouts /sit) (cons sit (sit-abouts /sit)))
	(assign sym sit)
        (add-constraints sit (sit-variable-bc /sit))
        sit))))


;;; Starting from situation sit, recurively gets situations in
;;; the given field of the situation structure.  Like "sit-subsit"
;;; or "sit-supertype".  Include-self-p tells whether to include
;;; the given situation in the resulting output list.

(defun sit-set (sit field include-self-p)
  (let ((/res nil)) 
    (declare (special /res))
    (rec-sit-set sit field)
    (if include-self-p /res (delete sit /res))))
(defun rec-sit-set (sit field)
  (when (not (member sit /res))
	(push sit /res)
	(let ((sits (funcall field sit)))
	  (loop (if sits (rec-sit-set (pop sits) field) (return))))))

;;; Given a fc-rule, queries the head, and for all solutions,
;;; asserts the rule's tail infons.  Uses results of backward-
;;; chaining rules, whereas forward-chaining off of only asserted
;;; infons does not.

(defun fwc-proto (added-infon)
  (when /fwc-rule
	(terpri)
	(format t "Forward-chaining-off of   ~a   in   ~a:" /fwc-rule /fwc-sit)
	(terpri)
	(format t "~a   has been added in   ~a." added-infon /sit)))


(defun forward-chain-off-rule (rule)
  (let ((/goals nil)
	(/fwc-rule rule)
	(/fwc-sit /sit)
        (/con 0)
        (*contexts* '((0))))
    (catchcut 
      (foreach (cadr rule) (cons '! (cddr rule))))))
	      

;=============================================================================
;   MISCELLANEOUS SYSTEM PREDICATES
;=============================================================================
; These are for the most part descibed in the documentation.

      
(defsys (query gensym 2) prosit-gensym)
(defun prosit-gensym (s p)
  (if (symbolp p)
      (if (prosit-free p) 
	  (prosit= p (smart-gensym s))
	(srg))))

    
(defsys (query foreach) foreach)
(defsys (simple foreach))
(defun foreach (&rest stuff)
  (let ((/goals `(,(imple &goals stuff) fail)))
    (srg))
  t)


(defsys (query printsit 0) printsit)
(defsys (simple printsit))
(defun printsit ()
  (mapcar 'pr (sit-infons /sit)))

(defsys (query beep 0) beep)
(defsys (simple beep))
(defun beep ()
  (princ ""))

(defun rl () (load "prosit.lsp"))
(defun print-level (x) (if (integerp x) (setf *print-level* x)))
(defun print-length (x) (if (integerp x) (setf *print-length* x)))

(defsys (query lisp 1) lisp)
(defun lisp (s-expr)
  (if (newcatch 'abort-lisp (lisp-eval s-expr))
      (srg)))

(defsys (query not 1) prosit-not)
(defsys (infon not 1))
(defun prosit-not (p)
  (if (let ((/goals nil)) 
	(not (catchcut (? p))))
      (srg)))

(defsys (query no 1) no)
(defsys (infon no 1))
(defun no (p)
  (context
   ((if (eq (car p) 'no)	
	(catchcut (? (cadr p)))
      (catchcut (derive (list 'no p) /sit))))))

(defsys (query or) prosit-or)
(defsys (infon or))
(defun prosit-or (&rest goals)
  (loop
    (unless goals (return))
    (if (let ((/goals (cons (car goals) /goals)))
	  (srg))
	(return t))
    (setf goals (cdr goals))))

(defsys (query lint 0) lint)
(defun lint ()
  (if /lint (mapcar 'write-line (reverse /lint))))

(defsys (query demo 1) demo)
(defun demo (filename)
  (let ((str (open filename :if-does-not-exist nil)))
    (if str 
	(input-loop str t t nil)
      (rlint "Tried to demo the file ~s but it did not exist.  Query failed." filename)
      )))

(defsys (query fastdemo 1) fastdemo)
(defun fastdemo (filename)
  (let ((str (open filename :if-does-not-exist nil)) (/depth 0))
    (if str 
	(input-loop str t t t)
      (rlint "Tried to demo the file ~s but it did not exist.  Query failed." filename)
      )))

(defsys (query bind-lisp 2) bind-lisp)
(defsys (query := 2) bind-lisp)
(defun bind-lisp (var s-expr)
  (context
    ((if (unify var /con (lisp-eval s-expr) /con)
	(srg)))))

(defsys (query load 1) prosit-load)
(defun prosit-load (filename)
  (let ((str (open filename :if-does-not-exist nil)))
    (if str
	(input-loop str nil nil nil)
      (rlint "Tried to load the file ~s but it did not exist.  Query failed." filename)
      )))

(defsys (query and) prosit-and)
(defsys (assert and) !)
(defun prosit-and (&rest goals)
  (let ((/goals (cons (imple &goals goals) /goals)))
    (srg)))

(defsys (query = 2) prosit=)
(defsys (assert = 2) assert=)
(defsys (infon = 2))
(defun prosit= (expr1 expr2)
  (context
       ((and (unify expr1 /con expr2 /con) (srg)))))


(defsys (query me 1) me)
(defsys (assert me 1) me)
(defsys (infon me 1))
(defun me (a)
  (context 
   ((and (unify a /con (sit-name /sit) /con) (srg)))))


(defun assert= (a b)
  (change-frame (chain-make-unify a b) `(= ,a ,b) /sit))

(defun rt () (eval (cons 'trace (trace))))

(defsys (query free 1) prosit-free)
(defsys (simple free 1))
(defun prosit-free (expr)
  (not (prosit-bound expr)))

(defsys (query fail 0) fail)
(defun fail () nil)

(defsys (query succeed 0) succeed)
(defun succeed () t)

(defsys (query repeat 0) repeat)
(defun repeat ()
  (loop 
    (unless (srg)
	    (return t))))

(defsys (query member 2) prosit-member)
(defun prosit-member (expr lis)
  (if (listp lis)
      (if (member expr lis)
	  (srg)
	(loop
	  (if (null lis) (return nil))
	  (if (prosit= expr (car lis)) (return t))
	  (setf lis (cdr lis))))))


(defsys (query bound 1) prosit-bound)
(defsys (simple bound))
(defun prosit-bound (e)
  (if (or (varp e) (cell-p e)) (bound e) t))

(defsys (query pr 1) pr)
(defsys (simple pr 1))
(defun pr (x)
  (princ x) (terpri)
  t)

(setf (get 'true 'atomfunc) t)
(setf (get 'fail 'atomfunc) t)
(setf (get 'beep 'atomfunc) t)

;'
;;; parameters (i.e. symbols) are only bound to things not
;;; containing variables? situations should not contain
;;; variables?
(defun v-occurs-in (var con1 expr2 con2)
  (cond ((and (equal var expr2) (= con1 con2)) 
         t)
	((cell-p expr2)
         (v-occurs-in var con1 (cell-val expr2) (cell-con expr2)))
	((consp expr2)
	 (or (v-occurs-in var con1 (car expr2) con2) 
	     (v-occurs-in var con1 (cdr expr2) con2)))
        (t nil)))

;;; Unify is described in prositguide.

(defun unify (expr1 con1 expr2 con2)
  ;;A con is a number that *contexts* associates with a list of (var . cell)
  ;;pairs.  It's a context number.
  (cond ((and (= con1 con2) (eql expr1 expr2))) ;equal breaks on sit structs
	((eq expr1 '_))
	((eq expr2 '_))
	((varp expr1)
	 (let ((cell (var-cell expr1 con1)))
	   (if cell 
	       (unify (cell-val cell) (cell-con cell) expr2 con2)
;'
             (unless (v-occurs-in expr1 con1 expr2 con2)
	       (push `(,con1 (,expr1 . ,(deref expr2 con2)) 
		  	     . ,(vl con1))
		     *contexts*)))))
	((varp expr2)
	 (let ((cell (var-cell expr2 con2)))
	   (if cell
	       (unify expr1 con1 (cell-val cell) (cell-con cell))
;'
             (unless (v-occurs-in expr2 con2 expr1 con1)
  	       (push `(,con2 (,expr2 . ,(deref expr1 con1))
			     . ,(vl con2))
		     *contexts*)))))
	((cell-p expr1)
	 (unify (cell-val expr1) (cell-con expr1) expr2 con2))
	((cell-p expr2)
	 (unify expr1 con1 (cell-val expr2) (cell-con expr2)))
	((eql expr1 expr2))
((and (stringp expr1) (stringp expr2))
(string= expr1 expr2))
	((and (consp expr1) (consp expr2))
	 (and (unify (car expr1) con1 (car expr2) con2) 
	      (unify (cdr expr1) con1 (cdr expr2) con2)))
	((symbolp expr1)
	 (let ((p (paramval expr1)))
	   (if (not (eq p expr1))
	       (unify p con1 expr2 con2)
	     (if (symbolp expr2)
		 (let ((p (paramval expr2)))
		   (if (not (eq p expr2))
		       (unify expr1 con1 p con2)))))))
	((symbolp expr2)
	 (let ((p (paramval expr2)))
	   (if (not (eq p expr2))
	       (unify expr1 con1 p con2))))))

;;; Deref is like flatten, but restricted to variables -- they are
;;; reduced until they are no longer variables, or until they are
;;; unbound variables.

(defun deref (expr con)
  (cond ((varp expr)
	 (let ((cell (var-cell expr con)))
	   (if cell
	       (let ((v (cell-val cell)) (c (cell-con cell)))
		 (if (and (eq v expr) (eq c con))
		     cell
		   (deref (cell-val cell) (cell-con cell))))
	     (make-cell :val expr :con con))))
	((cell-p expr) (deref (cell-val expr) (cell-con expr)))
	(t (make-cell :val expr :con con))))

;;; A model for how to cleanly use the unify function.  cont is
;;; a 0-argument continuation function.  Note use of "context" macro.

(defun clean-unify (expr1 con1 expr2 con2 cont)
  (context
    ((and (unify expr1 con1 expr2 con2)
	 (funcall cont)))))

;;; Tries to unify a and b, but gives b its own context.  Used in
;;; matching a goal to infons in the database.  See derive.

(defun unify-old-new (a b)		;B will get a new context/scope
  (context
    ((let ((newcon (gencon)))
      (and (unify a /con b newcon)
	   (srg))))))

;;; From the given name string, constructs a new symbol.  Different
;;; from lisp's gensym in that it tries to use as low numbers as
;;; possible. "x"->x0, and then "y"->y0, and then "x"->x1.

(defun smart-gensym (name)
  (let ((n 0) (string nil))
    (loop
      (setf string (concatenate 'string name (prin1-to-string n)))
      (if (not (find-symbol string))
	  (return (intern string)))
      (incf n))))


(defsys (query exists) exists)
(defsys (assert exists) e!)
(defun exists (vars &rest queries)
  (if (varlist-p vars)
      (context 
	((let ((/goals (cons (imple &goals (localize vars queries (gencon)))
			    /goals)))
	  (srg))))
    (rlint "E failed because ~s was not a list of variables." vars)))

;;; Makes sure that occurences of vars in expr are localized to con.

(defun localize (vars expr con)
  (cond ((varp expr)
	 (if (member expr vars) (make-cell :val expr :con con) expr))
	((consp expr)
	 (cons (localize vars (car expr) con)
	       (localize vars (cdr expr) con)))
	(expr)))

(defun E! (vars &rest asserts)
  (if (varlist-p vars)
      (let ((/goals `((! . ,(objectize vars asserts)) . ,/goals)))
	(srg))
    (rlint "E not asserted because ~s was not a list of variables." vars)))

;;; Replaces occurences of vars in asserts with new parameters.

(defun objectize (vars asserts)
  (let ((substlist nil) (v nil))
    (loop
      (if vars (setf v (pop vars)) (return))
      (push (cons v (varparam v))
	    substlist))
    (recursive-objectize substlist asserts)))

(defun recursive-objectize (substs expr)
  (cond ((varp expr)
	 (or (cdr (assoc expr substs)) expr))
	((consp expr)
	 (cons (recursive-objectize substs (car expr))
	       (recursive-objectize substs (cdr expr))))
	(expr)))

(defun listtransform (string table number)
  (if (null string)
      (list nil table number)
    (if (listp string)
	(let* ((headtr (listtransform (car string) table number))
	       (tailtr (listtransform (cdr string) (cadr headtr) (caddr headtr))))
	  (cons (cons (car headtr) (car tailtr)) (cdr tailtr)))
      (atomtransform string table number))))

(defun atomtransform (atom table number)
  (if (varp atom)
      (let ((newvar (cdr (assoc atom table))))
	(if newvar 
	    (list newvar table number)
	  (let ((newsym (format nil "@@@@@~d" number)))
	    (list newsym (acons atom newsym table) (+ number 1)))))
    (list atom table number)))


(defsys (query canonical-form 2) can-form)
(defsys (infon canonical-form 2))
(defun can-form (raw-expr can-expr)
  (context
   ((prosit= can-expr (car (listtransform 
			    (flatten (remove-cells raw-expr /con) /con) nil 0))))))



(defsys (query bagof 3) bagof)
(defsys (infon bagof 3))
(defun bagof (vars form result)
  (if (varlist-p vars)
      (context
	((prosit= result (get-bag vars form))))
    (rlint "bagof failed because ~s was not a list of variables." vars)))

;;; Builds up a list of bindings of vars in the solutions of "form",
;;; by using a special continuation function "get-bag-cont" that
;;; is placed on the goal list.

(defun get-bag (vars form)
  (context
    ((let ((/bagcon (gencon)) (/bagres nil))
      (declare (special /bagcon /bagres))
      (let ((/goals '(get-bag-cont)))
	(catchcut (? (localize vars form /bagcon))))
      /bagres))))

(setf (get 'get-bag-cont 'atomfunc) t)

;;; Takes note of the values of the quantified variables, then fails
;;; to backtrack and get the next solution.

(defun get-bag-cont ()
  (setf /bagres (append (mapcar 'cdr (vl /bagcon)) /bagres))
  nil)

(defsys (query setof 3) setof)
(defun setof (vars form result)
  (if (varlist-p vars)
      (context
	((prosit= result (make-set (get-bag vars form)))))
    (rlint "setof failed because ~s was not a list of variables." vars)))

(defun varlist-p (x)
  (or (null x) (and (consp x) (varp (car x)) (varlist-p (cdr x)))))

(defsys (query [_ 2) [_)
(defsys (query _] 2) _])
(defsys (assert [_ 2) [_!)
(defsys (assert _] 2) _]!)
(defsys (infon [_))
(defsys (infon _]))
(defun [_ (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sits (sit-abouts /sit)) (sit nil))
	(loop
	  (if sits (setf sit (pop sits)) (return))
	  (context
	    ((and (unify s2 /con (sit-name sit) /con)
		 (prosit-member 
		   s1 (mapcar 'sit-name 
			      (intersection (sit-abouts /sit)
					    (sit-set sit
						     'sit-subchunks nil))))
		 (return t))))))))
(defun _] (s1 s2) ([_ s2 s1))

(defun [_! (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sit1 (get-sit-named s1)) (sit2 (get-sit-named s2)))
	(cond 
;             ((eq sit1 sit2)
;	       (rlint "Can't assert ([_ ~s ~s)." s1 s2)
;	       (srg))
;	      ((member sit1 (sit-set sit2 'sit-subchunks nil))
;	       (rlint "Assertion ([_ ~s ~s) is already true." s1 s2)
;	       (srg))
;	      ((member sit2 (sit-set sit1 'sit-subchunks nil))
;	       (rlint "assert<- : (<- ~s ~s) would form a loop.  Fail." s1 s2))
	      (t
		(change-frame (make-subchunk sit1 sit2) `([_ ,s1 ,s2) sit1)
		(srg))))))
(defun _]! (s1 s2) ([_! s2 s1))

;;; Similar to make-subsit and make-subtype.
  
(defun make-subchunk (sub sup)
  (setf-restore (sit-subchunks sup) (cons sub (sit-subchunks sup)))
  (setf-restore (sit-superchunks sub) (cons sup (sit-superchunks sub)))
  (setf-restore (sit-abouts sup) (cons sub (sit-abouts sup)))
  (let ((h (sit-param-hash sup)))
    (if (and h (gethash (sit-name sub) h))
	(let ((/sit sup)) (chain-make-unify sub (sit-name sub)))
      (let ((/sit sup)) (assign (sit-name sub) sub)))))

;'
; why should equalities of the subchunk be transferred to the
; superchunk?
;  (let ((h (sit-param-hash sub)))
;    (if h
;	(context
;	  (let ((params (hash-key-list h)) (/sit sup) (p) (pv))
;	    (loop
;	      (unless params (return))
;	      (setf p (pop params))
;	      (setf pv (gethash p h))
;	      (chain-make-unify pv p)))))))



;(defsys (query -{ 2) -{)
;(defsys (query }- 2) }-)
; (defsys (assert -{ 2) -{!)
; (defsys (assert }- 2) }-!)
; (defsys (infon -{))
; (defsys (infon }-))
;(defun -{ (s1 s2)
;  (if (and (symbolp s1) (symbolp s2))
;      (let ((sits (sit-abouts /sit)) (sit nil))
;	(loop
;	  (if sits (setf sit (pop sits)) (return))
;	  (context
;	    ((and (unify s2 /con (sit-name sit) /con)
;		 (prosit-member 
;		   s1 (mapcar 'sit-name 
;			      (intersection (sit-abouts /sit)
;					    (sit-set sit
;						     'sit-subchunks nil))))
;		 (return t))))))))
;(defun }- (s1 s2) (-{ s2 s1))

; (defun -{! (s1 s2)
;   (if (and (symbolp s1) (symbolp s2))
;       (let ((sit1 (get-sit-named s1)) (sit2 (get-sit-named s2)))
; 	(change-frame (make-subcollection sit1 sit2) `(-{ ,s1 ,s2) sit1)
; 	(srg))))
; (defun }-! (s1 s2) (-{! s2 s1))

;;; makes one situation a subcollection of another.  
;;; i.e., all information about the sub- is inherited by the super-.
;;; has to copy all parameter unification information from the sub- 
;;; to the super-type.
; (defun make-subcollection (sub sup)
;   (setf-restore (sit-subchunks sup) (cons sub (sit-subchunks sup)))
;   (setf-restore (sit-superchunks sub) (cons sup (sit-superchunks sub)))
;   (setf-restore (sit-abouts sup) (cons sub (sit-abouts sup)))
;   (setf-restore (sit-subtypes sup) (cons sub (sit-subtypes sup)))
;   (setf-restore (sit-supertypes sub) (cons sup (sit-supertypes sub)))
;   (and 
;    (let ((h (sit-param-hash sup)))
;      (if (and h (gethash (sit-name sub) h))
; 	 (let ((/sit sup)) (chain-make-unify sub (sit-name sub)))
;        (let ((/sit sup)) (assign (sit-name sub) sub))))
;    (let ((h (sit-param-hash sub)))
;      (if (null h)
; 	 t
;        (context
; 	((let ((params (hash-key-list h)) (/sit sup) (p nil) (pv nil))
; 	   (loop
; 	    (unless params (return t))
; 	    (setf p (pop params))
; 	    (setf pv (gethash p h))
; 	    (unless (chain-make-unify pv p) (return nil))))))))))

(defsys (query resp 2) resp)
(defsys (assert resp 2) resp!)
(defsys (infon resp))
(defun resp (s c)
  (cond ((prosit-free s)
	 (let ((sits (sit-abouts /sit)) (sit nil))
	   (loop (if sits (setf sit (pop sits)) (return))
		 (context ((and (unify s /con (sit-name sit) /con)
			       (in-sit sit (rule c))
			       (return t)))))))
	((symbolp s) (let ((/sit (get-sit-named s))) (rule c)))))
	

(defun resp! (s c)
  (if (symbolp s)
      (let ((/sit (get-sit-named s)))
	(rule! c))))

;'
(defun copyhashtable (oldtable)
  (if (null oldtable)
      (make-hash-table)
    (let ((h (make-hash-table))
          (params (hash-key-list oldtable))
          (p nil))
      (loop
        (unless params (return h))
        (setf p (pop params))
        (setf (gethash p h) (gethash p oldtable))))))  

;;; Carries creates a new sit that has the given sit as a subsit, and
;;; respects the constraints "cs", and then queries i within that sit.

(defsys (query !!- 3) !!-)
(defun !!- (s i cs)
  (if (symbolp cs) (setf cs (paramval cs)))
  (if (and (symbolp s) (real-list cs)
	   (not (member nil (mapcar 'is-constr cs))))
      (let ((sit (get-sit-named s)))
	(in-sit (make-sit :name (sit-name sit)
		          :subsits (list sit)
;'
			  :param-hash (copyhashtable (sit-param-hash sit)))	
;(when (remhash sit (sit-param-hash in-sit))
;(setf (gethash in-sit (sit-param-hash in-sit)) in-sit))
		(mapcar 'constrain-with cs)
		(? i)))))

;;; Returns t iff c is in the form of a constraint. (=>, <=, or <=>).
			      
(defun is-constr (c)
  (and (real-list c) (> (length c) 2)
       (member (car c) '(=> <= <=>))))

(defsys (query if 3) prosit-if)
(defun prosit-if (if then else)
  (if (let ((/goals nil)) (catchcut (? if)))
      (? then)
    (? else)))

(defsys (query true) true)
(defun true (&rest queries)
  (if queries (let ((/goals nil)) (catchcut (apply 'prosit-and queries))))
  (srg))


; (@< s1 s2)
; - means "things true of s1 are true of s2"

;;; This is the subtype predicate.  See subsit for comparison.

(defsys (query @< 2) @<)
(defsys (assert @< 2) @<!)
(defsys (infon @< 2))
(defun @< (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sits (sit-abouts /sit)))
	(loop
	  (unless sits (return))
	  (context
	    ((and (unify s2 /con (sit-name (car sits)) /con)
		 (prosit-member 
		   s1 (mapcar 'sit-name 
			      (intersection (sit-abouts /sit)
					    (sit-set (car sits)
						     'sit-subtypes nil))))
		 (return t))))
	  (setf sits (cdr sits))))))

(defun @<! (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sit1 (get-sit-named s1)) (sit2 (get-sit-named s2)))
	(cond ((eq sit1 sit2)
	       (rlint "Can't assert (@< ~s ~s)." s1 s2)
	       (srg))
	      ((member sit1 (sit-set sit2 'sit-subtypes nil))
	       (rlint "Assertion (@< ~s ~s) is already true." s1 s2)
	       (srg))
	      ((member sit2 (sit-set sit1 'sit-subtypes nil))
	       (rlint "Asserting (@< ~s ~s) would form a loop.  Fail." s1 s2))
	      (t
		(change-frame (make-subtype sit1 sit2) `(@< ,s1 ,s2) sit1)
		(srg))))))

; (d< s1 s2)
; - means "things true of s1 are true of s2 by default"

;;; This is the default subtype predicate.  See subtype for comparison.

(defsys (query d< 2) d<)
(defsys (assert d< 2) d<!)
(defsys (infon d< 2))
(defun d< (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sits (sit-abouts /sit)))
	(loop
	  (unless sits (return))
	  (context
	    ((and (unify s2 /con (sit-name (car sits)) /con)
		 (prosit-member 
		   s1 (mapcar 'sit-name 
			      (intersection (sit-abouts /sit)
					    (sit-set (car sits)
						     'sit-subdefaults nil))))
		 (return t))))
	  (setf sits (cdr sits))))))

(defun d<! (s1 s2)
  (if (and (symbolp s1) (symbolp s2))
      (let ((sit1 (get-sit-named s1)) (sit2 (get-sit-named s2)))
	(cond ((eq sit1 sit2)
	       (rlint "Can't assert (d< ~s ~s)." s1 s2)
	       (srg))
	      ((member sit1 (sit-set sit2 'sit-subdefaults nil))
	       (rlint "Assertion (d< ~s ~s) is already true." s1 s2)
	       (srg))
	      ((member sit2 (sit-set sit1 'sit-subdefaults nil))
	       (rlint "Asserting (d< ~s ~s) would form a loop.  Fail." s1 s2))
	      (t
		(change-frame (make-subdefault sit1 sit2) `(d< ,s1 ,s2) sit1)
		(srg))))))

(defsys (query <=) ?<=)
(defsys (assert <=) !<=)
(defun ?<= (&rest args)
  (rule `(<= . ,args)))

(defun !<= (&rest args)
  (rule! `(<= . ,args)))

(defsys (query =>) ?=>)
(defsys (assert =>) !=>)
(defun ?=> (&rest args)
  (rule `(=> . ,args)))

(defun !=> (&rest args)
  (rule! `(=> . ,args)))


(defsys (query rule 1) rule)
(defsys (assert rule 1) rule!)
(defun rule (r)
  (context
   ((let ((sits (sit-set /sit 'sit-subtypes t)) 
	  (sit nil) (cs nil) (newcon (gencon)))
      (loop
       (cond (sits
	      (setf sit (pop sits))
	      (setf cs (append (sit-resp-bc sit) (sit-resp-fc sit)))
	      (if (loop 
		   (if cs (context ((and (unify r /con (pop cs) newcon) 
					(srg)
					(return t))))
		     (return)))
		  (return t)))
	     ((return))))))))

(defun repl (string variable subst)
  (if (equal string variable)
      subst
    (if (null string)
	nil
      (if (listp string)
	  (cons (repl (car string) variable subst)
		(repl (cdr string) variable subst))
	string))))

(defun add-constraints (sit constraints)
  (dolist (c constraints)
	  (let* ((/sit sit)
		 (constraint-sit (cdr c))
		 (goal (caar c))
		 (body (cdar c))
		 (newgoal (caddr goal))
		 (newbody (repl body (cadr goal) (sit-name sit))))
	    (unless (constrain-with-bc-support newgoal newbody constraint-sit)
		    (push 
		     (cons constraint-sit `(<= ,newgoal . ,newbody))
		     (sit-non-local-bc /sit))))))


(defun constrain-with-bc-support (goal body constraint-sit)
  (if (and (listp goal) (eq (car goal) `!=))
      (if (varp (cadr goal))
	  (progn
	    (push `((,goal . ,body) . ,constraint-sit) (sit-variable-bc /sit))
	    (dolist (s (union (sit-subchunks /sit) (sit-abouts /sit)))
		    (add-constraints s (list `((,goal . ,body) . ,constraint-sit)))))
	(let ((/sit (get-sit-named (cadr goal))))
	  (unless 
	   (constrain-with-bc-support
	    (caddr goal)
	    body 
	    constraint-sit))
	  (push 
	   (cons constraint-sit `(<= ,(caddr goal) . ,body)) 
	   (sit-non-local-bc /sit))))
    nil))

(defun constrain-with-bc (infon)
  (unless 
   (constrain-with-bc-support (cadr infon) (cddr infon) /sit)
   (push infon (sit-resp-bc /sit))
   (fwc-proto infon)
   (update /sit)
   t))  

(defun constrain-with-fc (infon)
  (push infon (sit-resp-fc /sit))
  (fwc-proto infon)
  (let ((supertypes (sit-set /sit 'sit-supertypes t)))
    (loop
     (unless supertypes (return))
     (let ((/sit (pop supertypes)))
       (forward-chain-off-rule infon))))
  (update-list (sit-supertypes /sit))
  (update-list (sit-superchunks /sit))
  (update (sit-owner /sit))
  t)

(defun rule! (r)
  (setf r (remove-cells r /con))
  (cond ((when (listp r)
	       (case (car r)
		     ((=>) (constrain-with-fc r))
		     ((<=) (constrain-with-bc r))
		     ((<=>) (and 
			     (constrain-with-fc r)
			     (constrain-with-bc r))))))
	((rlint "Tried to make ~s respect a non-constraint ~s." /sit r))))

;;;	 (if (sit-owner /sit)
;;;	     (let ((/sit (sit-owner /sit)) (s (sit-name /sit)))
;;;	       (forward-chain-off-infon `(resp ,s ,r)))))

(defsys (query trace 0) prosit-trace)
(defsys (simple trace))
(defun prosit-trace ()
  (setf /mastertrace (setf /trace t)
        /traceall t))

(defsys (query trace 1) prosit-trace-arg)
(defun prosit-trace-arg (tracepred)
  (setf /mastertrace (setf /trace t)
        /tracepreds (cons tracepred /tracepreds)
        /traceall nil)
        t)


(defsys (query untrace 0) prosit-untrace)
(defsys (simple untrace))
(defun prosit-untrace ()
  (setf /mastertrace (setf /trace nil))
  t)


(defstruct (imple (:print-function print-imple))
  func		;symbols &exit, &setcon, &setsit, &goals
  arg)

(defun print-imple (imple &rest lstr)
  (let ((str (car lstr)))
    (princ (imple-func imple) str)
    (princ (imple-arg imple) str)))

;;; srg is described in prositguide

(defun srg ()
;;;  (format t "Entering srg with goals ~s~&" /goals)
  (if /goals
      (or (let ((/goal (car /goals)) (/goals (cdr /goals)))
	    (if (imple-p /goal)
		(funcall (imple-func /goal) (imple-arg /goal))
	      (? /goal)))
	  (when (and /trace (boundp '/goal))
		(trace-display "Back to" /con /sit /goal)
;		(trace-prompt)
		nil))
    t))

;;; Variations on SRG to save a little stack space when calling
;;; imples.  Imples can destructively modify /goals (instead of
;;; the normal "let") because they never succeed more than once.

(defun new-srg ()
  (if (imple-p /goal)
      (funcall (imple-func /goal) (imple-arg /goal))
    (? /goal)))

(defun new-srg2 ()
  (cond (/goals
	  (setf /goal (pop /goals))
	  (if (imple-p /goal)
	      (funcall (imple-func /goal) (imple-arg /goal))
	    (? /goal)))
	(t)))

;;; Displays the trace of exiting a query.

(defun &exit (goal)
  (trace-display "Exiting" /con /sit goal)
  (new-srg2))

;;; Sets the context number.

(defun &setcon (con)
  (let ((/con con))
    (new-srg2)))

;;; Sets the situation.

(defun &setsit (sit)
  (let ((/sit sit))
    (new-srg2)))

;;; Solves a conjunction of goals.

(defun &goals (goals)
  (cond (goals
	  (setf /goal (pop goals))
	  (if goals
	      (if (cdr goals)
		  (push (imple &goals goals) /goals)
		(push (car goals) /goals)))
	  (new-srg))
	(t
	  (new-srg2))))

;;; Sets the user-interface situation.

(defun &settopsit (unused)
  (set-topsit))

;;; Displays bindings to the user and gets confirmation.

(defun &userconfirm (unused)
  (user-confirm))

;;; Turns trace back on in case implementation turned it off (see "!")

(defun &tron (unused)
  (let ((/trace t)) (new-srg2)))

(defsys (query duals 0) duals)
(defsys (simple duals))
(defun duals ()
  (setf /duals t))

(defsys (query noduals 0) noduals)
(defsys (simple noduals))
(defun noduals ()
  (setf /duals nil) t)

;;; When cut is backtracked to, it has to "throw" back to the correct
;;; "catchcut" point.  This is made more complicated by the fact that
;;; the previous catchcut point may have been within a subgoal.  But
;;; we really want to jump back 'till we get to a cut earlier in our
;;; current context.  So we set /cutcon for this purpose.  

(defsys (query cut 0) cut)
(defun cut ()
  (cond ((srg))
	(t (setf /cutcon /con) (throw 'cut nil))))

(defsys (assert != 1) !unary!=)
(defun !unary!= (infon)
  (cond ((and (listp infon)
	      (symbolp (car infon))
	      (or (lookup infon (car infon) (length (cdr infon)))
		  (lookup infon (car infon))))
	 (! infon))
	(t
;'
          (+! infon))))
;;;	  (setf infon (remove-cells infon /con))
;;;	  (setf (sit-infons /sit) (cons infon (sit-infons /sit)))
;;;	  (if (not (sit-owner /sit))
;;;	      (constrain-with infon))
;;;	  (forward-chain-off-infon infon))))

(defsys (query != 1) unary!=)
(defsys (infon != 1))
(defun unary!= (i)
  (cond ((and (listp i)
	      (symbolp (car i))
	      (or (lookup infon (car i) (length (cdr i)))
		  (lookup infon (car i))))
	 (? i))
	(t
	 (if (or (free i) (and (listp i) (free (car i))))
	     (derive-vague i)
	   (derive i /sit)))))
	
;;; A simple lisp print-eval-read interface.

(defsys (query lisp 0) lisp0)
(defun lisp0 ()
  (terpri)
  (princ "Lisp interface under Prosit.")
  (terpri)
  (princ "Enter (cont) to exit and fail.")
  (loop
    (terpri)
    (princ "lisp>")
    (print (eval (read)))))

(defsys (query atom 1) prosit-atom)
(defsys (simple atom 1))
(defun prosit-atom (x)
  (and (not (varp x)) (atom x)))

(defsys (query in 1) in1)
(defun in1 (s)
  (if (bound s)
      (let ((/sit (get-sit-named s)))
	(srg))))

(defsys (query out 0) out0)
(defun out0 ()
  (if (sit-owner /sit)
      (let ((/sit (sit-owner /sit)))
	(srg))
;'
    (rlint "out: Invalid attempt to leave top.")))

(defsys (query out 1) out1)
(defun out1 (rename)
  (if (not (sit-owner /sit))
      (let ((sit (make-sit :name 'top
			   :abouts (list /sit)
			   :param-hash (make-hash-table))))
	(setf (gethash rename (sit-param-hash sit)) /sit)
	(setf (sit-name /sit) rename)
	(setf (sit-owner /sit) sit)
	(let ((/sit sit)) (srg)))
    (if (equal (sit-name (sit-owner /sit)) rename)
        (out0)
      (rlint "out: ~s not owner of ~s. " rename (sit-name /sit)))))

(defun x (a b)
  (* a b))


;--------------------------------------------------------------------
;       UNOFFICIAL FUNCTIONS 
;--------------------------------------------------------------------


;''
;allows to make a new start and demonstrating "work.dem" by typing in (0)
(defsys (query o 0) standard-demo)
(defsys (simple o))
(defun standard-demo ()
  (fastdemo "work.dem"))


;'
(defsys (query showeq) showeq)
(defsys (assert showeq) showeq)
(defsys (simple showeq))
(defun showeq ()
  (let ((h (sit-param-hash /sit)))
     (if h
       (let ((params (hash-key-list h)) (p nil) (pv nil))
          (loop
             (unless params (return))
             (terpri) 
             (setf p (pop params))
             (setf pv (gethash p h))
             (princ "(= ")
             (prin1 p)
             (princ " ")
             (prin1 pv)
             (princ ")"))))
  t))

;'
(defun new ()
  (load "prosit.lsp")
  (run))

(defun mpf ()
  (load "../../mpf/prosit")
  (run))

(defsys (query comment 1) nothing)
(defsys (assert comment 1) nothing)
(defsys (simple comment))
(defun nothing (a) t)


;--------------------------------------------------------------------
;       HACKS FOR DEMO
;--------------------------------------------------------------------


(defsys (query count 3) prosit-count)
(defun prosit-count (expr start stop)
  (if (and (numberp start) (numberp stop))
      (loop
	(if (> start stop) (return nil))
	(if (prosit= expr start) (return t))
	(setf start (+ 1 start)))))

(defsys (query nextprod 3) nextprod)
(defun nextprod (expr prod max)
  (if (numberp prod)
      (if (= 0 prod) 
	  (nullprod expr max)
	(let ((f1 (isqrt prod)) (f2 nil))
	  (loop
	    (if (loop
		  (if (or (= f1 0) (> (/ prod f1) max)) (return t))
		  (if (= (rem prod f1) 0) (return nil))
                  (setf f1 (- f1 1)))
		(return nil))
	    (setf f2 (/ prod f1))
	    (if (prosit= expr `(,f1 ,f2)) (return t))
	    (setf f1 (- f1 1)))))))

(defun nullprod (expr max)
  (let ((f 0))
    (loop 
      (if (> f max) (return nil))
      (if (prosit= expr `(0 ,f)) (return t))
      (setf f (+ f 1)))))


(defsys (query nextsum 3) nextsum)
(defun nextsum (expr sum max)
  (if (numberp sum)
      (let* ((s2 (min max sum)) (s1 (- sum s2)))
	(loop
	  (if (or (> s1 s2) (> s1 max)) (return nil))
	  (if (prosit= expr `(,s1 ,s2)) (return t))
	  (setf s1 (+ s1 1) s2 (- s2 1))))))







;--------------------------------------------------------------------
;       MERGE-SITS
;--------------------------------------------------------------------
;
;
;(defun easychanges (s1 s2)
;   (setf (sit-subsits s1)
;	 (union (sit-subsits s1) (sit-subsits s2)))
;   (setf (sit-supersits s1)
;	 (union (sit-supersits s1) (sit-supersits s2)))
;   (setf (sit-infons s1)
;	 (union (sit-infons s1) (sit-infons s2)))
;   (setf (sit-resp-fc s1)
;	 (union (sit-resp-fc s1) (sit-resp-fc s2)))
;   (setf (sit-resp-bc s1)
;	 (union (sit-resp-bc s1) (sit-resp-bc s2)))
;   (setf (sit-superchunks s1)
;	 (union (sit-superchunks s1) (sit-superchunks s2)))
;   (setf (sit-subtypes s1)
;	 (union (sit-subtypes s1) (sit-subtypes s2)))
;   (setf (sit-supertypes s1)
;	 (union (sit-supertypes s1) (sit-supertypes s2)))
;   (setf (sit-changed s1) nil))
;
;(defmacro namemember (sit1 sitlist)
;  (member-if 
;   `(lambda (sit) (equal (sit-name ,sit1) (sit-name sit)))
;   ,sitlist))
;
;(defmacro namedelete (sit1 sitlist)
;  (delete-if 
;   `(lambda (sit) (equal (sit-name ,sit1) (sit-name sit)))
;   ,sitlist))
;
;;parameters as situations
;;hashtable hat nur noch eintraege: par -> sit
;(defun sit-unify (globalsit localsit doomedsit)
;  (if (eq globalsit doomedsit)
;      (sit-unify localsit globalsit doomedsit)
;    (unless 
;     (or 
;      (member globalsit (sit-set localsit 'sit-subsits nil))
;      (member globalsit (sit-set localsit 'sit-supersits nil))
;      (member globalsit (sit-set localsit 'sit-subtypes nil))
;      (member globalsit (sit-set localsit 'sit-supertypes nil)))
;     (easychanges globalsit localsit)
;     (let
;	 ((currsubchunks (sit-subchunks localsit))
;	  (currabouts (sit-abouts localsit)))
;       (dolist 
;	(currsit currabouts)
;	(let
;	    ((matchsubchunk (namemember currsit (sit-subchunks globalsit)))
;	     (matchabout (namemember currsit (sit-abouts globalsit))))
;	  (when matchsubchunk ;at most one of the matches is non-null
;		(sit-unify (car matchsubchunk) currsit doomedsit))
;	  (when matchabout
;		(sit-unify (car matchabout) currsit nil)))) ;or doomedsit for nil
;       (dolist 
;	(currsit currsubchunks) 
;	(let
;	    ((matchsubchunk (namemember currsit (sit-subchunks globalsit)))
;	     (matchabout (namemember currsit (sit-abouts globalsit))))
;	  (if matchsubchunk
;	      (sit-unify (car matchsubchunk) currsit doomedsit)
;	    (progn 
;	      (setf (sit-subchunks globalsit) (cons currsit (sit-subchunks globalsit)))
;	      (when matchabout
;		    (namedelete currsit (sit-abouts globalsit))
;		    (sit-unify currsit (car matchabout) doomedsit)))))))))
;      
;(defun merge-sits (s1 s2)
;   (sit-unify s1 s2 s2)
;
;  param-hash 
;
;
;
