/*-------------------------------------------------------------------------*/
/* Passe 4: Generation du code.                                            */
/*                                                                         */
/* predicat d'appel: generation_code(T,C,LFdC,NbChunk,Pred,W,LFdCW)        */
/*                                                                         */
/* entree                                                                  */
/* T      : format interne de la tete                                      */
/* C      : format interne du corps                                        */
/* LFdC   : liste des contraintes FD (cf passe 1)                          */
/* NbChunk: nombre de chunks du corps                                      */
/*                                                                         */
/* sortie                                                                  */
/* Pred   : nom du predicat de la tete (nom de la clause)                  */
/* W      : code wam associe a la clause, W = [inst_wam,...]               */
/* LFdCW  : code wam associe aux contraintes  de la clause                 */
/*          LFdCW = [cstr(NoFdC,InstallW,FdCW),...]                        */
/*                  InstallW=[inst_wam,...]    FdCW=[inst_wam,...]         */
/*                                                                         */
/*                                                                         */
/* Principe:                                                               */
/* genere_tete  se charge de generer le code de la tete.                   */
/* genere_corps se charge de generer le code du corps. Remarquer que si le */
/* dernier but est un predicat inline il n'y a pas de execute mais un      */
/* un proceed.                                                             */
/*                                                                         */
/* Les termes structures:                                                  */
/* On entend par terme structure un format interne lst(_,_) ou stc(_,_),   */
/* on designera pas Strc de tels formats.                                  */
/* La gestion des termes structures de la tete et du corps est assuree par */
/* gen_lst_strc(LStrcReg,UL,LSuiv,W)                                       */
/*                                                                         */
/* entree                                                                  */
/* LStrcReg: une liste de Strc/Reg dont il faut generer le code            */
/* UL      : unif/load pour differencier les traitements                   */
/* LSuiv   : liste de codes wam a placer apres le resultat                 */
/*                                                                         */
/* sortie                                                                  */
/* W       : code wam associe a LStrcReg                                   */
/*                                                                         */
/* Un element de LStrcReg (un Strc/Reg) indique que Strc doit etre unifie  */
/* avec (resp. charge dans) le registre Reg si UL=unif (resp. UL=load).    */
/*                                                                         */
/* Pour chaque Strc/Reg on appelle:                                        */
/*                                                                         */
/* gen_strc(Strc,Reg,UL,LSuiv,StrcW,LStrcRegAux) qui calcule le code wam   */
/* associe a Strc/Reg (StrcW) et fournit une liste auxiliaire contenant    */
/* les sous termes structures de Strc (liste LStrcRegAux).                 */
/*                                                                         */
/* Il reste alors a rappeler gen_lst_strc() sur LStrcRegAux et aussi sur   */
/* LStrcReg' (LStrcReg = [Strc/Reg | LStrcReg']).                          */
/* Ce qui differe entre la tete et le corps est la facon dont doivent etre */
/* ajances les differents codes:                                           */
/*                                                                         */
/* en unif (tete) : LStrcRegW = StrcW        + LStrcRegAuxW + LStrcReg'W   */
/* en load (corps): LStrcRegW = LStrcRegAuxW + StrcW        + LStrcReg'W   */
/*                                                                         */
/* Exemples:                                                               */
/*                                                                         */
/* p(f(g(a,b),c)).                                                         */
/*                                                                         */
/* LStrcReg    = [f(g(a,b),c)/0]              (0 car a unifier contre A0)  */
/* StrcW       = [get_structure(f/2,0),                                    */
/*                unify_x_variable(_1),                                    */
/*                unify_constant(c)]                                       */
/*                                                                         */
/* LStrcRegAux = [g(a,b)/_1]                  (_1 temporaire cree)         */
/*                                                                         */
/* puis:                                                                   */
/* LStrcRegAuxW= [get_structure(g/2,_1),                                   */
/*                unify_constant(a),                                       */
/*                unify_constant(b)]                                       */
/*                                                                         */
/* d'ou:                                                                   */
/* LStrcRegW   = [get_structure(f/2,0),                                    */
/*                unify_x_variable(_1),                                    */
/*                unify_constant(c),                                       */
/*                get_structure(g/2,_1),                                   */
/*                unify_constant(a),                                       */
/*                unify_constant(b)]                                       */
/*                                                                         */
/*                                                                         */
/* p:- q(f(g(a,b),c)).                                                     */
/*                                                                         */
/* LStrcReg    = [f(g(a,b),c)/0]              (0 car a charger dans A0)    */
/* StrcW       = [put_structure(f/2,0),                                    */
/*                unify_x_value(_1),                                       */
/*                unify_constant(c)]                                       */
/*                                                                         */
/* LStrcRegAux = [g(a,b)/_1]                  (_1 temporaire cree)         */
/*                                                                         */
/* puis:                                                                   */
/* LStrcRegAuxW= [put_structure(g/2,_1),                                   */
/*                unify_constant(a),                                       */
/*                unify_constant(b)]                                       */
/*                                                                         */
/* d'ou:                                                                   */
/* LStrcRegW   = [put_structure(g/2,_1),                                   */
/*                unify_constant(a),                                       */
/*                unify_constant(b)]                                       */
/*                put_structure(f/2,0),                                    */
/*                unify_x_value(_1),                                       */
/*                unify_constant(c)]                                       */
/*-------------------------------------------------------------------------*/

:- public generation_code/7.

generation_code(T,C,LFdC,NbChunk,Pred,TW1,LFdCW):-
	(g_read(debug,t) -> TW1=[dbg_clause|TW], CW1=[dbg_body|CW]
                         ;  TW1=TW, CW1=CW),
	genere_lst_fd_contrainte(LFdC,LFdCW),         !, % cut important
	genere_tete(T,NbChunk,CW1,TW,Pred),
	genere_corps(C,CW).




genere_tete(p(_,Pred,LstArg,NbY),NbChunk,LSuiv,TW,Pred):-
	gen_tete_lst_arg(LstArg,0,LSuiv,TW1),
	(NbChunk > 1 -> TW=[allocate(NbY)|TW1]
	             ;  TW=TW1).




gen_tete_lst_arg([],_,LSuiv,LSuiv).

gen_tete_lst_arg([Arg|LstArg],Reg,LSuiv,ArgW):-
	gen_unif_arg(Arg,Reg,LstArgW,ArgW),
	!,
	Reg1 is Reg+1,
	gen_tete_lst_arg(LstArg,Reg1,LSuiv,LstArgW).




genere_corps([],[proceed]).                                         % 0 buts

genere_corps([p(_,fail/0,[],_)|_],[fail]).

genere_corps([p(_,false/0,[],_)|_],[fail]).

genere_corps([p(NoPred,Pred/N,LstArg,_)|C],PredW):-
	inline_predicate(Pred,N),
	gen_inline_pred(Pred,N,NoPred,LstArg,CW,PredW),
	(C=[] -> (NoPred>1 -> CW=[deallocate,proceed]
	                   ;  CW=[proceed])
	      ;
	         genere_corps(C,CW)).

genere_corps([p(NoPred,Pred,LstArg,Trimming)|C],PredW):-
	gen_corps_lst_arg(LstArg,0,NoPred,CallExecuteW,PredW),
	(C=[] -> (NoPred>1 -> CallExecuteW=[deallocate,execute(Pred)]
	                   ;  CallExecuteW=[execute(Pred)])
	      ;
	         CallExecuteW=[call(Pred,Trimming)|CW],
	         genere_corps(C,CW)).




gen_corps_lst_arg([],_,_,LSuiv,LSuiv).

gen_corps_lst_arg([Arg|LstArg],Reg,NoPred,LSuiv,ArgW):-
	gen_load_arg(Arg,Reg,NoPred,LstArgW,ArgW),
	Reg1 is Reg+1,
	!,
	gen_corps_lst_arg(LstArg,Reg1,NoPred,LSuiv,LstArgW).




	/* gen_unif_arg(Arg,Reg,LSuiv,W) */

gen_unif_arg(var(igv(x(No),_,_,_,_),PremOcc,_),Reg,LSuiv,[ArgW|LSuiv]):-
	(PremOcc==t -> ArgW=get_x_variable(No,Reg)
	            ;  ArgW=get_x_value(No,Reg)).

gen_unif_arg(var(igv(y(No),_,_,_,_),PremOcc,_),Reg,LSuiv,[ArgW|LSuiv]):-
	(PremOcc==t -> ArgW=get_y_variable(No,Reg)
	            ;  ArgW=get_y_value(No,Reg)).

gen_unif_arg(cst(C),Reg,LSuiv,[get_constant(C,Reg)|LSuiv]).

gen_unif_arg(int(N),Reg,LSuiv,[get_integer(N,Reg)|LSuiv]).

gen_unif_arg(nil,Reg,LSuiv,[get_nil(Reg)|LSuiv]).

gen_unif_arg(Strc,Reg,LSuiv,StrcW):-
	gen_lst_strc([Strc/Reg],unif,LSuiv,StrcW).




	/* gen_load_arg(Arg,Reg,NoPred,LSuiv,W) */

gen_load_arg(var(igv(x(No),_,_,_,_),PremOcc,_),Reg,_,LSuiv,[ArgW|LSuiv]):-
	(PremOcc==t -> ArgW=put_x_variable(No,Reg)
	            ;  ArgW=put_x_value(No,Reg)).

gen_load_arg(var(igv(y(No),_,NoPDOcc,_,Unsafe),PremOcc,_),Reg,NoPred,LSuiv,[ArgW|LSuiv]):-
	(PremOcc==t -> ArgW=put_y_variable(No,Reg)
	            ;  (Unsafe==t, NoPred==NoPDOcc 
	                          -> ArgW=put_y_unsafe_value(No,Reg)
	                          ;  ArgW=put_y_value(No,Reg))).

gen_load_arg(cst(C),Reg,_,LSuiv,[put_constant(C,Reg)|LSuiv]).

gen_load_arg(int(N),Reg,_,LSuiv,[put_integer(N,Reg)|LSuiv]).

gen_load_arg(nil,Reg,_,LSuiv,[put_nil(Reg)|LSuiv]).

gen_load_arg(Strc,Reg,_,LSuiv,StrcW):-
	gen_lst_strc([Strc/Reg],load,LSuiv,StrcW).




	/* gen_lst_strc(LStrcReg,UL,LSuiv,W) */

gen_lst_strc([],_,LSuiv,LSuiv).

gen_lst_strc([Strc/Reg|LStrcReg],UL,LSuiv,W):-
	gen_strc(Strc,Reg,UL,W1,W2,LStrcRegAux),
	gen_lst_strc(LStrcRegAux,UL,W3,W4),
	gen_lst_strc(LStrcReg,UL,LSuiv,W5),
	(UL==unif -> W=W2, W1=W4, W3=W5
	          ;  W=W4, W3=W2, W1=W5).




gen_strc(lst(Car,Cdr),Reg,UL,LSuiv,[W|LstW],LStrcRegAux):-
	(UL==unif -> W=get_list(Reg) 
	          ;  W=put_list(Reg)),
	gen_unify_lst_arg([Car,Cdr],UL,LSuiv,LstW,LStrcRegAux).

gen_strc(stc(F,LstArg),Reg,UL,LSuiv,[W|StcW],LStrcRegAux):-
	(UL==unif -> W=get_structure(F,Reg) 
	          ;  W=put_structure(F,Reg)),
	gen_unify_lst_arg(LstArg,UL,LSuiv,StcW,LStrcRegAux).




gen_unify_lst_arg([],_,LSuiv,LSuiv,[]).

gen_unify_lst_arg([Arg|LstArg],UL,LSuiv,ArgW,LStrcRegAux1):-
	gen_compte_void([Arg|LstArg],0,N,LstArg1),
	(N=0 
	  -> gen_unify_arg(Arg,UL,LstArgW,ArgW,LStrcRegAux,LStrcRegAux1),
	     gen_unify_lst_arg(LstArg,UL,LSuiv,LstArgW,LStrcRegAux)
	  ;
	     ArgW=[unify_void(N)|LstArg1W],
	     gen_unify_lst_arg(LstArg1,UL,LSuiv,LstArg1W,LStrcRegAux1)),
	!.




gen_compte_void([var(_,t,t)|LstArg],N,N2,LstArg1):-
	N1 is N+1,
	gen_compte_void(LstArg,N1,N2,LstArg1).

gen_compte_void(LstArg,N,N,LstArg).

	          


gen_unify_arg(var(igv(x(No),NoPPOcc,_,Stc,_),PremOcc,_),_,LSuiv,[ArgW|LSuiv],
	  LStrcRegAux,LStrcRegAux):-
	(PremOcc==t -> ArgW=unify_x_variable(No)
	            ; (NoPPOcc=0, Stc==f -> ArgW=unify_x_local_value(No)
	                                 ;  ArgW=unify_x_value(No))).

gen_unify_arg(var(igv(y(No),_,_,Stc,_),PremOcc,_),_,LSuiv,[ArgW|LSuiv],
	  LStrcRegAux,LStrcRegAux):-
	(PremOcc==t  -> ArgW=unify_y_variable(No)
	             ;  (Stc==f -> ArgW=unify_y_local_value(No)
	                        ;  ArgW=unify_y_value(No))).

gen_unify_arg(cst(C),_,LSuiv,[unify_constant(C)|LSuiv],LStrcRegAux,LStrcRegAux).

gen_unify_arg(int(N),_,LSuiv,[unify_integer(N)|LSuiv],LStrcRegAux,LStrcRegAux).

gen_unify_arg(nil,_,LSuiv,[unify_nil|LSuiv],LStrcRegAux,LStrcRegAux).

gen_unify_arg(Strc,UL,LSuiv,[ArgW|LSuiv],LStrcRegAux,[Strc/Reg|LStrcRegAux]):-
	(UL==unif -> ArgW=unify_x_variable(Reg)
	          ;  ArgW=unify_x_value(Reg)).



	/* generation de code pour les contraintes FD */

genere_lst_fd_contrainte([],[]).

genere_lst_fd_contrainte([FdC|LFdC],[FdCW|LFdCW]):-
	genere_fd_contrainte(FdC,FdCW),
	!,                                                % cut important
	genere_lst_fd_contrainte(LFdC,LFdCW).




	% genere_fd_contrainte(FdC,FdCW)

genere_fd_contrainte(cstr(NoFdC,Opt2,fv(TellFV),R),cstr(NoFdC,InstallW,XinrW)):-
	genere_fd_top_range(R,DicFV,_,EndW,RW,Interval),
	traite_dicfv(DicFV,TellFV,NoFdC,Opt2,RW,LoadIndW,InstallW,NbVal),
	(NbVal>0 -> EndW=[label(1),fd_proceed]
	         ;  EndW=[fd_proceed]),
	(Interval==t -> XinrW=LoadIndW
	             ;  XinrW=[fd_allocate|LoadIndW]).

genere_fd_contrainte(cstr(NoFdC,_,fv(TellFV),R),_):-
	write('Syntax Error in FD Constraint #'), write(NoFdC),
	write('  '), write(fv(TellFV)), write(' in '), write(R), nl,
	fail.




genere_fd_top_range(R,DicFV,DicFP,LSuiv,RW,t):-
	(R={T};R=T..T),
	!,
	genere_fd_term(T,DicFV,DicFP,RMin,W,RW),
	W=[fd_tell_interval(RMin,RMin)|LSuiv].

genere_fd_top_range(T1..T2,DicFV,DicFP,LSuiv,RW,t):-
	!,
	genere_fd_term(T1,DicFV,DicFP,RMin,T2W,RW),
	genere_fd_term(T2,DicFV,DicFP,RMax,W,T2W),
	W=[fd_tell_interval(RMin,RMax)|LSuiv].

genere_fd_top_range(R,DicFV,DicFP,LSuiv,RW,f):-
	genere_fd_range(R,DicFV,DicFP,RRange,W,RW),
	W=[fd_tell_range(RRange)|LSuiv].




	% genere_fd_range(R,DicFV,DicFP,RRange,LSuiv,RW)

genere_fd_range([fp(FP,r)],_,DicFP,RRange,LSuiv,RW):-
	!,
	ajout_en_dicfp(DicFP,FP,r,Reg),
	(var(Reg) -> RW=[fd_range_parameter(RPA,FP)|RW1]
	          ;  RW=RW1),
	Reg=[RPA],
	RW1=[fd_range_copy(RRange,RPA)|LSuiv].

genere_fd_range(R,DicFV,DicFP,RRange,LSuiv,RW):-
	(R={T};R=T..T),
	!,
	genere_fd_term(T,DicFV,DicFP,RMin,W,RW),
	W=[fd_term_copy(RMax,RMin),
	   fd_interval_range(RRange,RMin,RMax)|LSuiv].

genere_fd_range(T1..T2,DicFV,DicFP,RRange,LSuiv,RW):-
	!,
	genere_fd_term(T1,DicFV,DicFP,RMin,T2W,RW),
	genere_fd_term(T2,DicFV,DicFP,RMax,W,T2W),
	W=[fd_interval_range(RRange,RMin,RMax)|LSuiv].

genere_fd_range(dom(fv(FV)),DicFV,_,RRange,LSuiv,RW):-
	!,
	ajout_en_dicfv(DicFV,FV,dom,[RRange1]),
	RW=[fd_range_copy(RRange,RRange1)|LSuiv].

genere_fd_range(R1 : R2,DicFV,DicFP,RRange,LSuiv,RW):-
	!,
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	W=[fd_union(RRange,RRange1)|LSuiv].

genere_fd_range(R1 & R2,DicFV,DicFP,RRange,LSuiv,RW):-
	!,
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	W=[fd_inter(RRange,RRange1)|LSuiv].

genere_fd_range(-R,DicFV,DicFP,RRange,LSuiv,TW):-
	(R={T};R=T..T),
	!,
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_compl_of_singleton(RRange,RT)|LSuiv].

genere_fd_range(-R,DicFV,DicFP,RRange,LSuiv,RW):-
	!,
	genere_fd_range(R,DicFV,DicFP,RRange,W,RW),
	W=[fd_compl(RRange)|LSuiv].

genere_fd_range(R1 + R2,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	!,
	W=[fd_range_add_range(RRange,RRange1)|LSuiv].

genere_fd_range(R1 - R2,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	!,
	W=[fd_range_sub_range(RRange,RRange1)|LSuiv].

genere_fd_range(R1 * R2,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	!,
	W=[fd_range_mul_range(RRange,RRange1)|LSuiv].

genere_fd_range(R1 / R2,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	!,
	W=[fd_range_div_range(RRange,RRange1)|LSuiv].

genere_fd_range(R1 mod R2,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R1,DicFV,DicFP,RRange,R2W,RW),
	genere_fd_range(R2,DicFV,DicFP,RRange1,W,R2W),
	!,
	W=[fd_range_mod_range(RRange,RRange1)|LSuiv].

genere_fd_range(R + T,DicFV,DicFP,RRange,LSuiv,RW):-
	!,
	genere_fd_range(R,DicFV,DicFP,RRange,TW,RW),
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_range_add_term(RRange,RT)|LSuiv].

genere_fd_range(R - T,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R,DicFV,DicFP,RRange,TW,RW),
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_range_sub_term(RRange,RT)|LSuiv].

genere_fd_range(R * T,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R,DicFV,DicFP,RRange,TW,RW),
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_range_mul_term(RRange,RT)|LSuiv].

genere_fd_range(R / T,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R,DicFV,DicFP,RRange,TW,RW),
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_range_div_term(RRange,RT)|LSuiv].

genere_fd_range(R mod T,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_range(R,DicFV,DicFP,RRange,TW,RW),
	genere_fd_term(T,DicFV,DicFP,RT,W,TW),
	W=[fd_range_mod_term(RRange,RT)|LSuiv].

genere_fd_range(R,DicFV,DicFP,RRange,LSuiv,RW):-
	genere_fd_user_fct(R,range,DicFV,DicFP,RRange,LSuiv,RW).




	% genere_fd_term(T,DicFV,DicFP,RT,LSuiv,TW)

genere_fd_term(fp(FP,t),_,DicFP,RT,LSuiv,TW):-
	ajout_en_dicfp(DicFP,FP,t,Reg),
	(var(Reg) -> TW=[fd_term_parameter(RPA,FP)|TW1]
	          ;  TW=TW1),
	Reg=[RPA],
	TW1=[fd_term_copy(RT,RPA)|LSuiv].

genere_fd_term(N,_,_,RT,LSuiv,TW):-
	 (integer(N); N==infinity),
	 TW=[fd_integer(RT,N)|LSuiv].

genere_fd_term(min(fv(FV)),DicFV,_,RT,LSuiv,TW):-
	ajout_en_dicfv(DicFV,FV,min,[RMin]),
	TW=[fd_term_copy(RT,RMin)|LSuiv].

genere_fd_term(max(fv(FV)),DicFV,_,RT,LSuiv,TW):-
	ajout_en_dicfv(DicFV,FV,max,[RMax]),
	TW=[fd_term_copy(RT,RMax)|LSuiv].

genere_fd_term(val(fv(FV)),DicFV,_,RT,LSuiv,TW):-
	ajout_en_dicfv(DicFV,FV,val,[RVal]),
	TW=[fd_term_copy(RT,RVal)|LSuiv].

genere_fd_term(T1 + T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_add_term(RT,RT1)|LSuiv].

genere_fd_term(T1 - T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_sub_term(RT,RT1)|LSuiv].

genere_fd_term(T1 * T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_mul_term(RT,RT1)|LSuiv].

genere_fd_term(T1 / T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_ceil_div_term(RT,RT1)|LSuiv].

genere_fd_term(T1 // T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_floor_div_term(RT,RT1)|LSuiv].

genere_fd_term(T1 mod T2,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_term(T1,DicFV,DicFP,RT,T2W,TW),
	genere_fd_term(T2,DicFV,DicFP,RT1,W,T2W),
	W=[fd_term_mod_term(RT,RT1)|LSuiv].

genere_fd_term(T,DicFV,DicFP,RT,LSuiv,TW):-
	genere_fd_user_fct(T,t,DicFV,DicFP,RT,LSuiv,TW).




	% genere_fd_user_fct(Fct,range/term,
	%                    DicFV,DicFP,RResult,LSuiv,FctW)

genere_fd_user_fct(Fct,FctType,DicFV,DicFP,RResult,LSuiv,FctW):-
	functor(Fct,FctName,N),
	Fct=..[_|LstArg],
	is_a_fct_name(FctName,CName),
	genere_fd_lst_arg(LstArg,DicFV,DicFP,LReg,W,FctW),
	number_atom(N,AN),
	atom_concat(arg_,AN,ArgFunctor),
	Args=..[ArgFunctor|LReg],
	(FctType==range -> W=[fd_range_fct(CName,RResult,Args)|LSuiv]
	                ;  W=[fd_term_fct(CName,RResult,Args)|LSuiv]).




genere_fd_lst_arg([],_,_,[],LSuiv,LSuiv).

genere_fd_lst_arg([Arg|LstArg],DicFV,DicFP,[Reg|LReg],LSuiv,ArgW):-
	(genere_fd_term(Arg,DicFV,DicFP,RT,LstArgW,ArgW),
	 Reg=tr(RT)
	        ;
	 genere_fd_range(Arg,DicFV,DicFP,RRange,LstArgW,ArgW),
	 Reg=rr(RRange)),
	genere_fd_lst_arg(LstArg,DicFV,DicFP,LReg,LSuiv,LstArgW).




is_a_fct_name(CFct,CFct):-
	atom(CFct),
	CFct\==dom, CFct\==min, CFct\==max, CFct\==val, CFct\==fv, CFct\==fp,
	atom_codes(CFct,[X|String]),
	(X >= 0'A, X =< 0'Z ; 
	 X >= 0'a, X =< 0'z),
	check_string(String).



check_string([]).

check_string([X|String]):-
	(X >= 0'A, X =< 0'Z ; 
	 X >= 0'a, X =< 0'z ; 
	 X >= 0'0, X =< 0'9 ;
	 X == 0'_),
	!,
	check_string(String).




	% Gestion des dictionnaires

ajout_en_dicfv([fv(FV,LRegMin,LRegMax,LRegDom,LRegVal)|_],FV,Type,LReg):-
	(Type==min, LRegMin=LReg                         % trouve ou cree
	         ;
	 Type==max, LRegMax=LReg
	         ;
	 Type==dom, LRegDom=LReg
	         ;
	 Type==val, LRegVal=LReg),
	 !.

ajout_en_dicfv([_|DicFV],FV,Type,LReg):-
	ajout_en_dicfv(DicFV,FV,Type,LReg).




traite_dicfv(DicFV,TellFV,NoFdC,Opt2,LSuiv,LoadIndW,InstallW,NbVal):-
	InstallW=[fd_create_C_frame(NoFdC,TellFV,Opt2)|InstallW1],
	genere_load_et_install(DicFV,LSuiv,LoadIndW,InstallW1,NbVal).




ajout_en_dicfp([fp(FP,FpType,LReg)|_],FP,FpType,LReg):-      % trouve ou cree
	!.

ajout_en_dicfp([_|DicFp],FP,FpType,LReg):-
	ajout_en_dicfp(DicFp,FP,FpType,LReg).




	% code d'installation et de chargement des indexicaux

genere_load_et_install([],LSuiv,LSuiv,[fd_proceed],0).        % unif var fin

genere_load_et_install([Ind|DicFV],LSuiv,LoadIndW,InstallW,NbVal):-
	Ind=fv(FV,LRegMin,LRegMax,LRegDom,LRegVal),
	(nonvar(LRegVal),               /* val used, maybe min/max/dom */
	 LRegVal=[RVal],
	 LRegMin=[RVal],
	 LRegMax=[RVal],
	 InstallW=[fd_install_dly_val(FV)|InstallW1],
	 (nonvar(LRegDom) 
	       -> LRegDom=[RRange],
	          RMax=RVal, 
	          LoadIndW=[fd_dly_val(RMin,FV,1), fd_term_copy(RMax,RMin),
	                    fd_interval_range(RRange)|
	                    LoadIndW1]
	       ;
	          LoadIndW=[fd_dly_val(RVal,FV,1)|LoadIndW1])
	     ;                              /* dom used, maybe min/max */
	 nonvar(LRegDom),
	 LRegDom=[RRange],
	 LRegMin=[RMin],
	 LRegMax=[RMax],
	 InstallW=[fd_install_ind_dom(FV)|InstallW1],
	 LoadIndW=[fd_ind_dom(RRange,FV)|LoadIndW1]
	     ;                                        /* only min used */
	 nonvar(LRegMin),var(LRegMax),
	 LRegMin=[RMin],
	 InstallW=[fd_install_ind_min(FV)|InstallW1],
	 LoadIndW=[fd_ind_min(RMin,FV)|LoadIndW1]
	     ;                                        /* only max used */
	 var(LRegMin),nonvar(LRegMax),
	 LRegMax=[RMax],
	 InstallW=[fd_install_ind_max(FV)|InstallW1],
	 LoadIndW=[fd_ind_max(RMax,FV)|LoadIndW1]
	     ;
	 LRegMin=[RMin],                           /* min and max used */
	 LRegMax=[RMax],
	 InstallW=[fd_install_ind_min_max(FV)|InstallW1],
	 LoadIndW=[fd_ind_min_max(RMin,RMax,FV)|LoadIndW1]),
	genere_load_et_install(DicFV,LSuiv,LoadIndW1,InstallW1,NbVal1),
	(nonvar(LRegVal) -> NbVal is NbVal1+1
	                 ;  NbVal =  NbVal1).





	% Code des predicats inline : 
	%      gen_inline_pred(Pred,Arite,NoPred,LstArg,LSuiv,Code)
	% Tous les predicats definis ici doivent aussi avoir une clause
	% dans le predicat inline_predicate/3 en passe 2 definissant tous
	% les predicats inline.




	/* cut inline ('$get_bc_reg'/1,'$cut'/1) */

gen_inline_pred('$get_bc_reg',1,_,[var(igv(Nom,_,_,_,_),PremOcc,_)],
	    LSuiv,[InstW|LSuiv]):-
	(PremOcc==f -> error('$get_bc_reg used with bound variable...')
	            ;  true),
	(Nom=x(No), InstW=get_x_bc_reg(No)
	      ; 
	 Nom=y(No), InstW=get_y_bc_reg(No)).




gen_inline_pred('$cut',1,_,[var(igv(Nom,_,_,_,_),_,_)],LSuiv,[InstW|LSuiv]):-
	(Nom=x(No), InstW=cut_x(No) 
	      ; 
	 Nom=y(No), InstW=cut_y(No)).




	/* dbg_true */

gen_inline_pred('$dbg_true',0,_,_,LSuiv,LSuiv).




	/* pragma C (pragma_c/1) */

gen_inline_pred(pragma_c,1,_,[cst(Code)],LSuiv,[pragma_c(Code)|LSuiv]).




	/* unification inline (=/2) */

gen_inline_pred((=),2,NoPred,[Arg1,Arg2],LSuiv,UnifW):-
	(Arg1=var(igv(x(Reg),_,_,_,_),PremOcc,_) 
	   -> inline_unif_reg_terme(PremOcc,Reg,Arg2,NoPred,LSuiv,UnifW)
	   ;
	      (Arg2=var(igv(x(Reg),_,_,_,_),PremOcc,_)
	         -> inline_unif_reg_terme(PremOcc,Reg,Arg1,NoPred,
	                                  LSuiv,UnifW)
	         ;
	            gen_load_arg(Arg1,Reg,NoPred,Unif1W,UnifW),
	            inline_unif_reg_terme(PremOcc,Reg,Arg2,NoPred,
	                                  LSuiv,Unif1W))).




inline_unif_reg_terme(f,Reg,Arg,_,LSuiv,UnifW):-
	gen_unif_arg(Arg,Reg,LSuiv,UnifW).

inline_unif_reg_terme(_,Reg,Arg,NoPred,LSuiv,UnifW):-
	gen_load_arg(Arg,Reg,NoPred,LSuiv,UnifW).




          /* term comparison inline */

gen_inline_pred(F,1,_,[Arg],LSuiv,TestW):-
	(F==var; F==nonvar; F==atom; F==integer; F==atomic),
	gen_load_arg(Arg,Reg,_,Test1W,TestW),
	Test1W=[builtin_1(F,Reg)|LSuiv].



gen_inline_pred(F,2,_,[Arg1,Arg2],LSuiv,Comp1W):-
	(F==(==); F==(\==); F==(@<); F==(@=<); F==(@>); F==(@>=)),
	gen_load_arg(Arg1,Reg1,_,Comp2W,Comp1W),
	gen_load_arg(Arg2,Reg2,_,InstW,Comp2W),
	comp_functor_name(F,Name),
	!,
	InstW=[builtin_2(Name,Reg1,Reg2)|LSuiv].




comp_functor_name((==),    term_eq).
comp_functor_name((\==),   term_neq).
comp_functor_name((@<),    term_lt).
comp_functor_name((@=<),   term_lte).
comp_functor_name((@>),    term_gt).
comp_functor_name((@>=),   term_gte).




	/* mathematical inlines (is/2 =:=/2 ...) */

gen_inline_pred(is,2,_,[Arg1,Arg2],LSuiv,MathW):-
        inline_load_math_expr(Arg2,Reg,UnifW,MathW),
        !,
        gen_unif_arg(Arg1,Reg,LSuiv,UnifW).




gen_inline_pred(F,2,_,[Arg1,Arg2],LSuiv,Math1W):-
	(F==(=:=); F==(=\=); F==(<); F==(=<); F==(>); F==(>=)),
	inline_load_math_expr(Arg1,Reg1,Math2W,Math1W),
	inline_load_math_expr(Arg2,Reg2,InstW, Math2W),
	math_functor_name(F,Name),
	!,
	InstW=[builtin_2(Name,Reg1,Reg2)|LSuiv].




inline_load_math_expr(var(igv(Nom,_,_,_,_),PremOcc,_),Reg,LSuiv,MathW):-
	(PremOcc==t 
	    -> error('Illegal arithmetic expression')
	    ;  (Nom=x(No), MathW=[math_load_x_value(No,Reg)|LSuiv]
	           ;
	        Nom=y(No), MathW=[math_load_y_value(No,Reg)|LSuiv])).

inline_load_math_expr(int(N),Reg,LSuiv,MathW):-
	MathW=[put_integer(N,Reg)|LSuiv].

inline_load_math_expr(lst(Arg,nil),Reg,LSuiv,MathW):-
	inline_load_math_expr(Arg,Reg,LSuiv,MathW).

inline_load_math_expr(stc(F/_,[Arg]),Reg,LSuiv,MathW):-
	(F== (-) -> Name=neg
                 ;  math_functor_name(F,Name)),
	inline_load_math_expr(Arg,Reg1,InstW,MathW),
        (Name==add -> Reg=Reg1,
                      InstW=LSuiv
                   ;
		      InstW=[function_1(Name,Reg,Reg1)|LSuiv]).

inline_load_math_expr(stc(F/_,[Arg1,Arg2]),Reg,LSuiv,MathW):-
	math_functor_name(F,Name),
	inline_load_math_expr(Arg1,Reg1,Math1W,MathW),
	(Arg2=int(1), (Name==add, Name1=inc ; Name==sub, Name1=dec) 
	      -> Math1W=InstW,
	         InstW=[function_1(Name1,Reg,Reg1)|LSuiv]
	      ;
	         inline_load_math_expr(Arg2,Reg2,InstW,Math1W),
	         InstW=[function_2(Name,Reg,Reg1,Reg2)|LSuiv]).

inline_load_math_expr(_,_,_,_):-
	error('Illegal arithmetic expression').




math_functor_name((+),    add).
math_functor_name((-),    sub).
math_functor_name((*),    mul).
math_functor_name((//),   div).
math_functor_name(mod,    mod).
math_functor_name((/\),   and).
math_functor_name((\/),   or).
math_functor_name((^),    xor).
math_functor_name((\),    not).
math_functor_name((<<),   shl).
math_functor_name((>>),   shr).


math_functor_name((=:=),  eq).
math_functor_name((=\=),  neq).
math_functor_name((<),    lt).
math_functor_name((=<),   lte).
math_functor_name((>),    gt).
math_functor_name((>=),   gte).




	/* Type tests inline (var/ nonvar/...) */

gen_inline_pred(F,1,_,[Arg],LSuiv,TestW):-
	(F==var; F==nonvar; F==atom; F==integer; 
         F==number; F==atomic; F==compound),
	gen_load_arg(Arg,Reg,_,Test1W,TestW),
	Test1W=[builtin_1(F,Reg)|LSuiv].




          /* Global variables */

gen_inline_pred(F,2,_,[Arg1,Arg2],LSuiv,GVarW):-
        (F==g_assign;F==g_read),
        gen_load_arg(Arg1,Reg1,_,GVar1W,GVarW),
        gen_load_arg(Arg2,Reg2,_,GVar2W,GVar1W),
        GVar2W=[builtin_2(F,Reg1,Reg2)|LSuiv].





	/* FD inlines ('$fd_set_AF'/2,...) */

gen_inline_pred('$fd_set_AF',2,_,[int(N),var(igv(Nom,_,_,_,_),_,_)],LSuiv,[InstW|LSuiv]):-
	(Nom=x(No), InstW=fd_set_x_AF(N,No) 
	      ; 
	 Nom=y(No), InstW=fd_set_y_AF(N,No)).

gen_inline_pred('$fd_fv',1,_,[var(igv(x(No),_,_,_,_),PremOcc,_)],
	    LSuiv,[InstW|LSuiv]):-
	(PremOcc==t -> InstW=fd_x_variable_in_A_frame(No)
	            ;  InstW=fd_x_value_in_A_frame(No)).

gen_inline_pred('$fd_fv',1,_,[var(igv(y(No),_,_,_,_),PremOcc,_)],
	    LSuiv,[InstW|LSuiv]):-
	(PremOcc==t -> InstW=fd_y_variable_in_A_frame(No)
	            ;  InstW=fd_y_value_in_A_frame(No)).

gen_inline_pred('$fd_fp',2,_,[var(igv(x(No),_,_,_,_),PremOcc,_),cst(FpType)],
                LSuiv,[InstW|LSuiv]):-
	(PremOcc==t -> error('Error: Incorrect parameter')
	            ;  (FpType==r, 
		        InstW=fd_x_range_parameter_in_A_frame(No)
		          ;
			InstW=fd_x_term_parameter_in_A_frame(No))).

gen_inline_pred('$fd_fp',2,_,[var(igv(y(No),_,_,_,_),PremOcc,_),cst(FpType)],
                LSuiv,[InstW|LSuiv]):-
	(PremOcc==t -> error('Error: Incorrect parameter')
	            ;  (FpType==r, 
		        InstW=fd_y_range_parameter_in_A_frame(No)
		          ;
			InstW=fd_y_term_parameter_in_A_frame(No))).

gen_inline_pred('$fd_cstr',2,_,[int(NoFdC),var(igv(Nom,_,_,_,_),_,_)],
	    LSuiv,[InstW,fd_call_constraint|LSuiv]):-
	(Nom=x(No), InstW=fd_install_constraint_with_x_AF(NoFdC,No)
	      ;
	 Nom=y(No), InstW=fd_install_constraint_with_y_AF(NoFdC,No)).



