;;  $Id: pce-goodies.lisp,v 1.4 1993/02/26 11:54:40 anjo Exp $
;;  
;;  File	pce-goodies.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Several useful predicates
;;  Works with	PCE 4.4
;;  
;;  Notice	Copyright (c) 1992  University of Amsterdam
;;  
;;  History	10/07/92  (Created)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")


(export '(pce-chain-list	; Chain -> List
	  pce-list-chain		; 
	  pce-make-global
	  pce-class-name	; Object -> ClassName (as a keyword)
	  pce-printed-p		; Object x Object -> { T | NIL }
	  pce-create-attribute	; Object x Attribute x Value -> Object
	  pce-parts		; Object -> Part*
	  pce-whole             ; Object -> Whole
	  pce-hierarchy         ; Object -> 
	  pce-name-reference	; Object x Name -> Object
	  pce-get-list		; Object x Selector -> List

	  pce-funcall		; Function x Args ... -> Message
	  pce-apply		; Function x ArgList -> Message
	  pce-eval		; Expression -> Message

	  pce-check-globals	; ->
	  pce-all-globals	; -> Chain
	  ))


;;  ------------------------------------------------------
;;  Utility functions
;;  ------------------------------------------------------

;;! pce-chain-list chain ==> list
;;
;;  Converts ^chain^ to a list of the elements part of the chain.  A
;;  special case is a non-existing chain (@nil) when the Lisp constant
;;  #NIL# is returned.

(defun pce-chain-list (chain &optional
			     (return-as :unchecked)
			     (free-when-done nil))
  (when (and chain
	     (not (eq chain @nil)))
	(let ((arity (pce-get chain :-arity))
	      result)
	  (dotimes (i arity result)
		   (push (pce-get-as return-as chain :-arg (- arity i))
			 result))
	  (when free-when-done
		(pce-send chain :done))
	  result)))


;;! pce-class-name object ==> class-name
;;
;;  Returns the ^class-name^ of ^object^ as a keyword.

(defun pce-class-name (object)
  (pce-get object :-class-name))


;;! pce-printed-p a b ==> { T | NIL }
;;
;;  Succeeds if ^a^ and ^b^ are the same when printed by Lisp.

(defun pce-printed-p (a b)
  (equal (or (when (pce-object-p a)
		   (eq :string (pce-class-name a))) (symbol-name a))
	 (or (when (pce-object-p b)
		   (eq :string (pce-class-name b))) (symbol-name b))))

	   
;;! pce-create-attribute object attribute value ==> object
;;
;;  Defines ^attribute^ has a new attribute for ^object^.  The initial
;;  value of ^attribute^ is set to ^value^.

(defun pce-create-attribute (object attribute value)
  (pce-send object :attribute (pce-new :attribute attribute value)))


;;! pce-parts object ==> part*
;;
;;  Given a visible ^object^ returns all it's parts.  Works for
;;  graphical objects only.

(defun pce-parts (object)
  (pce-chain-list (pce-get object :contains)))


;;! pce-whole object ==> whole
;;
;;  Given a visible ^object^ returns the ^whole^.  Opposite of
;;  pce-parts/#.

(defun pce-whole (object)
  (pce-get object :contained-in))


(defun pce-parent-p (parent child)
  (let ((whole (pce-whole child)))
    (when whole
	  (or (pce-equal parent (pce-whole child))
	      (pce-parent-p parent whole)))))


;;! pce-hierarchy object [selector] ==>
;;
;;  Prints a hierarchy of the graphical objects contained in ^object^.

(defun pce-hierarchy (object &optional selector)
  (format t "~A~%" object)
  (pce-hierarchy2 object 1 selector))

(defun pce-hierarchy2 (object level selector)
  (let ((subs (pce-parts object)))
    (dolist (s subs)
	    (dotimes (i level) (format t "|   "))
	    (if selector
		(format t "~A [~A]~%" s (pce-get s selector))
	      (format t "~A~%" s))
	    (pce-hierarchy2 s (1+ level) selector))))


;;! pce-name-reference object name ==> object
;;
;;  Renames the object reference for ^object^ to ^name^.  This is a
;;  wrapper around ->name-reference that ensures (a) lowercase is used
;;  and (b) the correct name is stored in the PCE name table.

(defun pce-name-reference (object name)
  (pce-unregister object)
  (pce-send object :name-reference (intern name "KEYWORD"))
  (pce-register (pce-at name)))


;;! pce-get-list object selector ==> list
;;
;;  Returns ^list^ as a list of all values returned by (pce-get object
;;  selector) which should be a chain.

(defun pce-get-list (object selector &rest args)
  (pce-gc
   (let ((chain (apply #'pce-get (cons object (cons selector args)))))
    (when (pce-object-p chain)
	  (pce-chain-list chain)))))


;;  ------------------------------------------------------
;;  Call back
;;  ------------------------------------------------------

(defun pce-funcall (function &rest args)
  (pce-make :message (cons @lisp (cons function args))))


(defun pce-apply (function args)
  (pce-make :message (cons @lisp (cons function args))))


(defun pce-execute-eval (expression)
  (eval (pce-to-lisp expression)))


(defun pce-eval (expression)
  (pce-new :message @lisp 'pce-execute-eval (lisp-to-pce expression)))


;;  ------------------------------------------------------
;;  Checking PCE object-base consistency
;;  ------------------------------------------------------

(defun pce-check-globals (&rest objects)
  (let ((ch (pce-all-globals)))
    (pce-send-list ch :append objects)
    (pce-send ch :-check)
    (pce-send ch :done)))


(defun pce-all-globals ()
  (let ((ch (pce-chain)))
    (pce-send @pce :for-name-reference
	      (pce-funcall 'add-global ch @arg1))
    ch))
  

(defun add-global (ch name)
  (unless (or (eq name :-object-to-itf-table)
	      (eq name :-name-to-itf-table)
	      (eq name :-handle-to-itf-table))
	  (pce-send ch :-append (pce-at name))))


;;! pce-make-global object arguments.. ==> object
;;
;;  Creates a named ^object^ from ^arguments^ if it does not already
;;  exist.  Normally used at top-level to define global PCE objects.
;;  Example:
;;=	(pce-make-global @same-identity :identity :size)

(defun pce-make-global (object &rest arguments)
  (or (pce-exists-p object)
      (apply #'pce-new :name (pce-object-id object) arguments)))


;;! pce-list-chain elements ==> Chain
;;
;;  Returns a PCE chain with all elements on the list appended to it.

(defun pce-list-chain (elements)
  (pce-send-list (pce-chain) :append elements))
