;;  $Id: host-lucid.lisp,v 1.6 1993/01/22 15:40:40 anjo Exp $
;;  
;;  File	host-lucid.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Lucid specific code
;;  Works with	PCE 4.5,  SCL 3.0,  SCL 4.0
;;  
;;  Notice	Copyright (c) 1993  University of Amsterdam
;;  
;;  History	26/11/92  (Created)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")


(export '(pce-reset))			; ->


;;  ------------------------------------------------------
;;  Foreign structure pcelisp-status
;;  ------------------------------------------------------

(def-foreign-struct pcelisp-status
  (arguments :type :signed-32bit)
  (error :type :signed-32bit)
  (name :type :signed-32bit)
  (return-type :type :signed-32bit)
  (result :type :signed-32bit)
  (return-as :type :signed-32bit))	; An object in reality


(defun get-pcelisp-status-arguments (ps)
  (pcelisp-status-arguments ps))

(defun get-pcelisp-status-error (ps)
  (pcelisp-status-error ps))

(defun get-pcelisp-status-name (ps)
  (pcelisp-status-name ps))

(defun get-pcelisp-status-return-type (ps)
  (pcelisp-status-return-type ps))

(defun get-pcelisp-status-result (ps)
  (pcelisp-status-result ps))


;;  ------------------------------------------------------
;;  Control of multi-tasking
;;  ------------------------------------------------------

;;! mt-off
;;
;;  [Lucid Inc.]  Turns the SCL multi-tasking capability off.  This is
;;  necessary to ensure proper loading of the PCE/Lisp interface.

(defun mt-off ()
  (when (and (fboundp 'process-allow-schedule)
             lucid::*io-waits-with-process-wait*)
        (warn "Turning OFF the multi-tasking scheduler")
        (setf lucid::*io-waits-with-process-wait* nil)
        (setf lucid::*scheduling-quantum* most-positive-fixnum)
        (process-allow-schedule)))


;;! mt-on
;;
;;  [Lucid Inc.]  Turns multi-tasking back on.

(defun mt-on ()
  (when (and (fboundp 'process-allow-schedule)
             (not lucid::*io-waits-with-process-wait*))
        (warn "Turning ON the multi-tasking scheduler")
        (setf lucid::*io-waits-with-process-wait* t)
        (setf lucid::*scheduling-quantum* 333)
        (process-allow-schedule)))


;;  ------------------------------------------------------
;;  Foreign function declarations
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-initialise
		       (:return-type :boolean)
		       (:name "_pcelisp_c_initialise")))

;;  ------------------------------------------------------
;;  Low-level memory management
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-mark
		       (:return-type :unsigned-32bit)
		       (:name "_pcelisp_mark")))


(def-foreign-function (pcelisp-c-rewind
		       (:name "_pcelisp_rewind"))
  (mark :unsigned-32bit))


;;  ------------------------------------------------------
;;  Allocation of new symbols in the interface
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-new-name
		       (:return-type :boolean)
		       (:name "_pcelisp_c_new_name"))
  (name :simple-string)
#+STATIC-LISP-SYMBOLS (value :lisp)
#-STATIC-LISP-SYMBOLS (value :unsigned-32bit))


(def-foreign-function (pcelisp-c-new-assoc
		       (:return-type :boolean)
		       (:name "_pcelisp_c_new_assoc"))
  (name :simple-string)
  (value :lisp))


;;  ------------------------------------------------------
;;  Argument pushing
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-int
		       (:return-type :boolean)
		       (:name "_pcelisp_c_int"))
  (value :fixnum))


(def-foreign-function (pcelisp-c-special
		       (:return-type :unsigned-32bit)
		       (:name "_pcelisp_c_special"))
  (value :fixnum))


(def-foreign-function (pcelisp-c-real
		       (:return-type :boolean)
		       (:name "_pcelisp_c_real"))
  (value :single-float))


(def-foreign-function (pcelisp-c-string
		       (:return-type :boolean)
		       (:name "_pcelisp_c_string"))
  (value :simple-string))


(def-foreign-function (pcelisp-c-name
		       (:return-type :boolean)
		       (:name "_pcelisp_c_name"))
#+STATIC-LISP-SYMBOLS (value :lisp)
#-STATIC-LISP-SYMBOLS (value :unsigned-32bit))


(def-foreign-function (pcelisp-c-assoc
		       (:return-type :boolean)
		       (:name "_pcelisp_c_assoc"))
  (value :lisp))


(def-foreign-function (pcelisp-c-ref
		       (:return-type :boolean)
		       (:name "_pcelisp_c_ref"))
  (value :fixnum))


(def-foreign-function (pcelisp-c-symbol
		       (:return-type :boolean)
		       (:name "_pcelisp_c_symbol"))
#+STATIC-LISP-SYMBOLS (value :lisp)
#-STATIC-LISP-SYMBOLS (value :unsigned-32bit))


;;  ------------------------------------------------------
;;  Existence Checks
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-is-assoc
		       (:return-type :boolean)
		       (:name "_pcelisp_c_is_assoc"))
  (value :lisp))


(def-foreign-function (pcelisp-c-is-ref
		       (:return-type :boolean)
		       (:name "_pcelisp_c_is_ref"))
  (value :fixnum))


;;  ------------------------------------------------------
;;  PCE Virtual Machine Instructions
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-send
		       (:return-type :boolean)
		       (:name "_pcelisp_c_send"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-new
		       (:return-type :boolean)
		       (:name "_pcelisp_c_new"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-new-named
		       (:return-type :boolean)
		       (:name "_pcelisp_c_new_named"))
  (name :simple-string))


(def-foreign-function (pcelisp-c-new-symbol
		       (:return-type :boolean)
		       (:name "_pcelisp_c_new_symbol"))
  (name :simple-string)
  (package :simple-string))


(def-foreign-function (pcelisp-c-get
		       (:return-type :boolean)
		       (:name "_pcelisp_c_get"))
  (status (:pointer pcelisp-status)))


;;  ------------------------------------------------------
;;  Returning values
;;  ------------------------------------------------------

(def-foreign-function (pcelisp-c-pull
		       (:return-type :boolean)
		       (:name "_pcelisp_c_pull"))
  (status (:pointer pcelisp-status))
  (vector :signed-32bit)
  (argc :fixnum))


(def-foreign-function (pcelisp-c-fetch-integer
		       (:return-type :fixnum)
		       (:name "_pcelisp_c_fetch_integer"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-real
		       (:return-type :single-float)
		       (:name "_pcelisp_c_fetch_real"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-native
		       (:return-type :lisp)
		       (:name "_pcelisp_c_fetch_native"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-name
		       (:return-type :lisp)
		       (:name "_pcelisp_c_fetch_name"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-assoc
		       (:return-type :lisp)
		       (:name "_pcelisp_c_fetch_assoc"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-string
		       (:return-type :simple-string)
		       (:name "_pcelisp_c_fetch_string"))
  (status (:pointer pcelisp-status)))


(def-foreign-function (pcelisp-c-fetch-assoc-name
		       (:return-type :simple-string)
		       (:name "_pcelisp_c_fetch_assoc_name"))
  (status (:pointer pcelisp-status)))


;;  ------------------------------------------------------
;;  Special functions
;;  ------------------------------------------------------

(def-foreign-function (pce-reset
		       (:name "_pcelisp_reset")))


;;  ------------------------------------------------------
;;  Management of PCELISP-STATUS Pointers
;;  ------------------------------------------------------
;;
;;  Some of the data between Lisp and PCE is passed directly through a
;;  foreign structure called pcelisp-status.  This saves the overhead
;;  of a foreign function call for trivial data (e.g. number of
;;  arguments, error status) and also makes it possible to call PCE
;;  recursively.
;;
;;  Unfortunately, allocating and freeing a pcelisp-status pointer
;;  takes a lot of time (0.5 milli seconds on a SparcStation II),
;;  so we have to set up a small manager for them.  [NB.  I am always
;;  amazed to see how Lisp implementors succeed in making the most
;;  critical operation (memory allocation) as slow as possible.]

(defvar *pcelisp-status-free-list* nil)

(defun alloc-pcelisp-status ()
  (or (pop *pcelisp-status-free-list*)
      (malloc-foreign-pointer :type '(:pointer pcelisp-status))))

(defun free-pcelisp-status (ps)
  (push ps *pcelisp-status-free-list*))


;;  ------------------------------------------------------
;;  Call Back from PCE (messages to @lisp)
;;  ------------------------------------------------------

(def-foreign-callable (pce-call-back (:return-type :signed-32bit))
  ((argc :signed-32bit)
   #+STATIC-LISP-SYMBOLS (real-function :lisp)
   #-STATIC-LISP-SYMBOLS (function :signed-32bit)
   (vector :signed-32bit))
  (let ((argv nil)
	(rval 0)
#-STATIC-LISP-SYMBOLS
        (real-function (pcelisp-find-symbol function))
	(ps (alloc-pcelisp-status)))
    (dotimes (i argc)
	     (pcelisp-c-pull ps vector (- argc (1+ i)))
	     (push (pcelisp-answer ps) argv))
    (if (fboundp real-function)
	(if (apply real-function argv)
	    (setf rval 1)
	  (setf rval 0))
      (progn
	(warn ";;; PCE-CALL-BACK (WARNING): Function ~A (~A) not found"
	      real-function argv)
	(setf rval 0)))
    (free-pcelisp-status ps)
    rval))


;;! pcelisp-call-back
;;
;;  This function is called when the user requests information about
;;  the host language (from the PCE tracer).  The argument is a
;;  keyword indicating the kind of request.
;;
;;  The implementation uses the functions defined in CLtL2 (p. 913).

(def-foreign-callable (pcelisp-call-back (:return-type :signed-32bit))
  ((request :lisp))
  (case request
	(:ABORT
	 (pce-reset)			; Try to reset PCE
	 (abort))			; Abort to Lisp top-level
	(:BACKTRACE
	 (break "Backtrace request from PCE -- Use :C [ONTINUE] to return"))
	(:BREAK
	 (break "Break request from PCE -- Use :C [ONTINUE] to return"))
	(:HALT
	 (quit))
	(:FATAL
	 (format t ";;; Fatal error in PCE -- aborting to Lisp top-level~%")
	 (pce-reset)
	 (abort)))
  1)


;;! pcelisp-define-symbol symbol-name package-name ==> symbol
;;
;;  Defines a Lisp symbol from ^symbol-name^ and ^package-name^.  This
;;  function is called from the C interface when an unknown Lisp
;;  symbol is encountered.  Returns the symbol created.

(def-foreign-callable (pcelisp-define-symbol (:return-type :lisp))
  ((symbol-name :simple-string)
   (package-name :simple-string))
  (intern symbol-name (or (find-package package-name)
			  (make-package package-name))))

