% ptn.nl
%
% ptn2nl(+Basename)
% translates a "ptn" file into a Prolog file
% ptn code is only translatable when in the body of a clause, 
% not in the head.
%
% The grammar ptn_gr.nl is required for parsing
%

ptn2nl( Basename) :-
	name( Basename, Chars),
        append( Chars, ".ptn", InputFile),
	append( Chars, ".nl", OutputFile),
		% tokenize the file and parse it, 
        get_tokens( InputFile, Toks), !,
        ptn$program( Toks, [], Node),!,
		% extract all the clause parse trees
	unify( Node, pplus( T, mult( ptn$clause, _, ClausePts), 1)), !, 
		% translate each clause and return new clause1 parse trees
	translate( ClausePts,  NewClausePts, 1, _), !,
		% replace the clauses
	unify( Newnode, pplus( T, mult( ptn$clause, _, NewClausePts), 1)),
		% print out
	extract_tokens( Newnode, Tokens2),!,
	print_tokens( OutputFile, Tokens2).

translate( [], [], Vin, Vin).
translate( [ ClPt|ClPts], [NClPt|NClPts], Vin, Vout) :-
	trans( ClPt, NClPt, Vin, Vout1),
	translate( ClPts, NClPts, Vout1, Vout).

% trans( +ClausePt, -NewClausePt, +VarNum, -NewVarNum)
%
% unify the clause parse tree with a sequence representing the head
% and body of a clause, if the body is "empty" then fail the first clause
% and return the clause unaltered
% The goals are then extracted from the body, and translated and then
% new goals returned, a new clause1 parse tree is then built and returned 
%
trans( ClausePt, NewClausePt, Vin, Vout) :-
	Plus = pplus( _,  sseq([ sub( ptn$head, _, HeadPt),
			sub( ptn$body, B, BodyPt)]), 2),
	unify( ClausePt, Plus),
	B \== [], !,
	Plus1 = pplus( T1, mult( ptn$goal, _, GPts), 3),
	unify( BodyPt, Plus1),	
	newGoals( GPts, NewGPts, Vin, Vout),
	Plus2 = pplus( T1, mult( ptn$goal, _, NewGPts), 3),
	unify( NewBodyPt, Plus2),
	build_parse( ptn$clause, [HeadPt, NewBodyPt], NewClausePt).
trans( ClausePt, ClausePt, V, V).

newGoals( [], [], Vin, Vin).
newGoals( [ GPt|GPts], [NGPt|NGPts], Vin ,Vout) :-
        transGoal( GPt, NGPt, Vin ,Vout1),
        newGoals( GPts, NGPts, Vout1, Vout).

% transGoal( +GoalPt, -NewGoalPt, +Varnum, -NewVarnum)
%
% goals are unified with a sequence which includes a nodeTerm, followed
% by the string "unify" followed by a plusTerm
% i.e. P unify T ++ Sua
%
% the plusTerm is then translated and returned and the newgoal built.
transGoal( GoalPt, NewGoalPt, Vin, Vout):-
	Plus = pplus( _, sseq([ sub( ptn$nodeTerm, _, NPt),"unify", 
				sub( ptn$plusTerm, _, PPt)]), 4),
	unify( GoalPt, Plus), !,
	transPlus( PPt, NewPPt, Vin, Vout),
	build_parse( ptn$goal, ["unify(", NPt, ',', NewPPt, ')'], 
				NewGoalPt).
transGoal( GoalPt, GoalPt, Vin, Vin).

% transplus( +PlusTermPt, -NewTermPt, +Varnum, -NewVarnum)
%
% plusTerm is unified with a nodeTerm "++" restPlusTerm
% and translated into pplus( nodeTerm, restPlusTerm)
%
transPlus( PPt, NewPPt, Vin, Vout) :-
	Plus = pplus( _, sseq( [ sub( ptn$nodeTerm, _, NPt), "++",
				sub( ptn$restPlusTerm, _, RPt)]), 5),
	unify( PPt, Plus), !,
	transRestPlus( RPt, NewRPt, NumPt, Vin, Vout),
	build_parse( ptn$term, ["pplus(", NPt, ',', NewRPt, ',', NumPt, ')'], 
						NewPPt).
transPlus( PPt, PPt, Vin, Vin).	

% translate the rest of the plusTerm depending upon the sequence
% of tokens
% the first clause checks to see if a number has been supplied
% after the '++', the second clause, supplies a variable in place of
% the number
% T ++1 Subs  =>  pplus( T, Subs, 1)
%
transRestPlus( RPt, NewRPt, NumPt, Vin, Vout) :-
	Plus = pplus( _,  sseq([ sub( number(_),  _, NumPt),	
			sub( ptn$subTerm, _, SPt)]), 6),
	unify( RPt, Plus),!,
	transSubs( SPt, NewRPt, Vin, Vout).	
transRestPlus( RPt, NewRPt, Var, Vin, Vout) :-
	transSubs( RPt, NewRPt, Vin, V1),
	newVar( V1, Vout, Var).

% translate the subsequences:
%
% ( Subs ) => [Subs]
% << Subs >> => sseq( [Subs])
%
transSubs( SPt, NewTerm, Vin, Vout) :-
	Plus = pplus( _, [ " (", sub( ptn$subs, _, SubsPt)], 7),
	unify( SPt, Plus), !,
	Plus1 = pplus( T1, mult( ptn$sub, _, SPts), 8),
	unify( SubsPt, Plus1), 
	subList( SPts, NewSPt, Vin, Vout),
	Plus2 = pplus( T1, mult( ptn$term, _, NewSPt), 8),
	unify( NewSubs, Plus2),
	append( [NewSubs], [']'], L),
	build_parse( ptn$term, ['['|L], NewTerm).
transSubs( SPt, NewTerm, Vin, Vout) :-
 	Plus = pplus( _, [ "<<", sub( ptn$subs, _, SubsPt)], 7),
        unify( SPt, Plus), !,
        Plus1 = pplus( T1, mult( ptn$sub, _, SPts), 8),
        unify( SubsPt, Plus1), 
        subList( SPts, NewSPt, Vin, Vout),
        Plus2 = pplus( T1, mult( ptn$term, _, NewSPt), 8),
        unify( NewSubs, Plus2),
        append( [NewSubs], [']', ')'], L),
        build_parse( ptn$term, [sseq, '(','['|L], NewTerm). 
transSubs( SPt, NewTerm, Vin, Vout) :-
	Plus = pplus( _, sub( ptn$sub, _, SPt1), 7),
	unify( SPt, Plus),!,
	dotSub( SPt1, NewTerm, Vin, Vout).
transSubs( S, S, Vin, Vin).
 
subList( [], [], Vin, Vin).
subList( [S|Ss], [NewS|NewSs], Vin, Vout) :-
	dotSub( S, NewS, Vin, Vout1),
	subList( Ss, NewSs, Vout1, Vout).

% translate single sub
%
% Pt::Grsymbol  =>  sub( Grsymbol, Var, Pt)
% Pt::Grsymbol* => mult( Grsymbol, Var, Pt)
% Pt::Grsymbol(Args) ==> sub( Grsymbol(Args), Var, Pt)
%
% The variable are generated using the Number Vin and returning Vout,
% so that each variable is unique
%
dotSub( S, NewS, Vin, Vout) :-
	Plus = pplus( _, sseq([ sub(ptn$nodeTerm, _, NPt), "::",
				sub(ptn$restSub, _, RPt)]), 8),
	unify( S, Plus), !,
	lastSub( RPt, NPt, NewS, Vin, Vout).
dotSub( S, S, Vin, Vin).

lastSub( RPt, NPt, NewS, Vin, Vout) :-
	Plus = pplus( _, sseq([ sub( ptn$atomVar, _, APt), "*"]), 9),
	unify( RPt, Plus), !,
	newVar( Vin, Vout, Var),
	build_parse( ptn$term, [ "mult(", APt, ',', Var, ',', NPt, ')'], NewS).
lastSub( RPt, NPt, NewS, Vin, Vout) :-
        Plus = pplus( _, sub( ptn$atomVar, _, APt), 9),
        unify( RPt, Plus), !,
        newVar( Vin, Vout, Var),
        build_parse( ptn$term, [ "sub(", APt, ',', Var, ',', NPt, ')'], NewS). 
lastSub(  RPt, NPt, NewS, Vin, Vout) :-
        Plus = pplus( _, sub( ptn$goal, _, APt), 9),
        unify( RPt, Plus), !,
        newVar( Vin, Vout, Var),
        build_parse( ptn$term, [ "sub(", APt, ',', Var, ',', NPt, ')'], NewS).

newVar( Num, Num1, Z):-
	number_chars( Num, NumString),
	append( "_Z", NumString, Z),
	Num1 is Num + 1.  

