% demo.nl
%
% This demo tokenizes and parses the file grades.txt, which is
% in a format consistent with the grammar detailed in grades_gr.gr
%
% Each line parse tree is extracted and manipulated to suit.
%
% this example shows the id and scores extracted and listed, followed
% by the score averages followed by each persons individual average.
%
% the lists passed around are of the form:
%
% Scores  [[101,10,3,5,7],[102,3,4,3,2],[111,2,wh,3,5]..]
%		[Id|Scores]
% Averages  [[101,3.5],[102,4.2],[111,3,2]...]
%		[Id,Aver]
% ScoreAves  [5.5, 4.6, 3.8,..]

demo:-
        get_tokens( 'grades.txt', Toks),
       	parse( grades, Toks, Pt1),
		% unify the grades parse tree Pt1 with a sub representing
		% all lines 
	Plus = pplus( T, mult( line, _, LinePts), 1),
	unify( Pt1, Plus), !,
		% manipulate the line parse trees and return a list of 
		% Ids and Scores
	lines_scores( LinePts, Scores),
	write( 'Id       Scores'), nl,
	write_names(Scores),nl, 
		% calculate the average of each set of scores
	scoreAves( Scores, ScoreAves),
	write( 'Avs  '),
	write_list( ScoreAves), nl, nl,
		% calculate each individual's average
	averages( Scores, Averages),
	write( 'Id  Average'), nl,
	write_names(Averages),nl,nl.

		
lines_scores( [], []).
lines_scores( [L|Lpts], [Score|Scores]) :-
	l_score( L, Score),
	lines_scores( Lpts, Scores).
%
% l_score( +LineParseTree, -Scorelist)
%
% unify the LinePraseTree, Lpt with a sequence of subs representing
% the individual items required
l_score( Lpt, [I|List]) :-
	Plus = pplus( T, [sub( name, [Na], NPt), sub( id, [I], IPt),
			sub( scores(N), Slist, SCPt)], 2),
	unify( Lpt, Plus),
	scoreList( SCPt, List).	
%
% scoreList( +ScoreParseTree, -List)
%
% unify the Parse tree representing all the scores with a sub representing
% each score	
scoreList( ScPt, List) :-
	Plus = pplus( T, mult( score, SList, _), 3),
	unify( ScPt, Plus),
	flatten_list( SList, List).
	
flatten_list( [], []).
flatten_list( [[X]|Rest], [X|List]) :-
	flatten_list(Rest, List).

averages( [], []).
averages( [S|Scores], [A|Averages]):-
	score_average( S, A),
	averages( Scores, Averages).

% work out the average of the scores in the list
score_average( [Id|Ss], [Id,Ave]) :-
	add_scores( Ss, 0, Total),
	length( Ss, L),
	ave( Total, L, Ave).	

add_scores( [], Sofar, Sofar).
add_scores( [wh|Rest], Sofar, Total):-
	add_scores( Rest, Sofar, Total).
add_scores( [S|Rest], Sofar, Total) :-
	number(S),
	Subtotal is Sofar + S,
	add_scores( Rest, Subtotal, Total).

ave( _, 0, 0).
ave( T, N, A):-
	N \== 0,
	A is T / N.

% work out the average of each set of scores
scoreAves( [S|T], ScoreAves) :-
	length( S, L),
	L1 is L - 1,
	zero_list( L1, Zeros),
	scoreTotals( [S|T], Zeros, Totals),
	length( [S|T], L2),
	ave_list( Totals, L2, ScoreAves).

zero_list( L, Zeros):-
	length( List, L),
	z_list( List, Zeros).

z_list( [], []).
z_list([_|T],[0|T1]) :-
	z_list( T, T1).


scoreTotals( [], Sofars, Sofars).
scoreTotals( [[_|X]|Rest], Sofars, Totals) :-
	add_X( X, Sofars, T1s),
	scoreTotals( Rest, T1s, Totals).

add_X( [], Sofars, Sofars).
add_X(	[wh|Ss], [Sofar|Sofars], [Sofar|Totals]):-
	add_X( Ss, Sofars, Totals).
add_X(	[S|Ss], [Sofar|Sofars], [Total|Totals]) :-
	number(S),
	Total is Sofar + S,
	add_X( Ss, Sofars, Totals).

ave_list( [], _, []).
ave_list( [X|Xrest], L, [I|Aves]) :-
	ave( X, L, A),
	R is round( A),
	I is integer( R),
	ave_list( Xrest, L, Aves).


write_names( []).
write_names( [N|Names]) :-
	write_list( N),
	nl,
	write_names( Names).

write_list( []).
write_list( [A|Rest]) :-
	write( A),
	write( '  '),
	write_list( Rest).
write_list([]).
write_list([X|R]) :-
	write(X), nl,
	write_list(R).
