;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, 1994 Mark Tarver          |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun typecheck
       (fp1 fp2)
       (prog (checktype)
             (initstacks)
             (set '*skip* 0)
             (overwrite-signature fp1)
             (setq checktype (catch 'delay
                                    (phi (canonical-form fp1)
                                         fp2
                                         nil
                                         nil)))
             (if (null checktype)
                 (return (typecheck1 fp1 nil)))
             (if (eq checktype 'delayed)
                 (return (typecheck1 fp1 'delayed))
                 (return (typecheck1 fp1
                                     (greek (decompile (deref fp2))))))))

(defun typecheck1
       (fp1 fp2)
       (cond ((null fp2)
              (setq *definitions* (remove fp1
                                          *definitions*
                                          ':test
                                          'equal))
              (type-error fp1))
             ((eq 'delayed fp2) 'delayed)
             (t (prog2 (type-success fp1
                                     (rewrite-to-normal-form fp2))
                       fp2))))

(defun type-error
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'define (car fp1)))
              (type-failure-message (cadr fp1)))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'deftactic (car fp1)))
              (type-failure-message (cadr fp1)))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'defrew (car fp1)))
              (type-failure-message (cadr fp1)))
             (t (raise "code 14: TYPE FAILURE!"))))

(defun type-success
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'define (car fp1)))
              (add-to-xtt (cadr fp1) fp2))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'deftactic (car fp1)))
              (add-to-xtt (cadr fp1) fp2))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'defrew (car fp1)))
              (add-to-xtt (cadr fp1) fp2))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'mutual (car fp1))
                   (and (consp (cadr fp1))
                        (> (length (cadr fp1))
                           1))
                   (consp fp2))
              (type-success (car (cdr fp1))
                            (car fp2))
              (if (eq (caadr fp1) 'deftactic)
                  (add-tactic (cadar (cdr fp1))))
              (if (eq (caadr fp1) 'defrew)
                  (pushnew (cadar (cdr fp1))
                           *rewrites*))
              (type-success (cons 'mutual (cddr fp1))
                            (cdr fp2)))
             (t t)))

(defun type-failure-message
       (fp1)
       (raise (format nil
                      "code 14: function ~A: TYPE FAILURE!"
                      fp1)))

(defun type-var
       (fp1)
       (cond ((and (consp fp1)
                   (eq 'mutual (car fp1)))
              (mapcar 'type-var (cdr fp1)))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq '|{| (caddr fp1)))
              (skolemise-sig (getsig (cdddr fp1))))
             ((and (consp fp1)
                   (eq 'defrew (car fp1)))
              (list (list 't-expr) '-> 't-expr))
             (t (list '$ (gentemp)))))

(defun getsig
       (fp1)
       (cond ((and (consp fp1)
                   (eq '|}| (car fp1)))
              nil)
             ((consp fp1) (cons (car fp1)
                                (getsig (cdr fp1))))
             (t (raise "code 36: missing }"))))

(defun skolemise-sig
       (fp1)
       (skolemise-rule (list (butlast fp1 2)
                             '->
                             (car (last fp1)))))

(defun delay
       (fp1)
       (format t
               ";; ~A is delayed~%"
               (cadr fp1)))

(defun canonical-form
       (fp1)
       (cond ((and (consp fp1)
                   (eq 'mutual (car fp1)))
              (list 'mutual
                    (mapcar 'canonical-form (cdr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (or (eq (car fp1) 'define)
                       (eq (car fp1) 'deftactic)
                       (eq (car fp1) 'defrew)))
              (list 'define
                    (cadr fp1)
                    (mapcar 'iso-pattern
                            (brrl (cons-form (rest (remove-sig (cdr fp1))))
                                  (cadr fp1)))))
             (t (remove-quotes fp1))))

(defun iso-pattern (fp1) (ip1 fp1 nil))

(defun ip1
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '<- (car fp1)))
              (list fp2 '<- (cadr fp1)))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '-> (car fp1)))
              (list fp2 '-> (cadr fp1)))
             ((consp fp1) (ip1 (cdr fp1)
                               (append fp2
                                       (list (car fp1)))))
             (t (raise "code 13: No Patterns have Fired in ip1"))))

(defun cons-form
       (fp1)
       (mapcar 'put-conses (bracket fp1)))

(defun put-conses
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq '<> (cadr fp1)))
              (list 'append
                    (put-conses (car fp1))
                    (put-conses (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (bar (cadr fp1)))
              (list 'cons
                    (put-conses (car fp1))
                    (put-conses (caddr fp1))))
             ((and (consp fp1)
                   (equal (car fp1) '@))
              (mapcar 'put-conses (cdr fp1)))
             ((consp fp1)
              (list 'cons
                    (put-conses (car fp1))
                    (put-conses (cdr fp1))))
             (t fp1)))

(defun store-and-print
       (fp1 fp2)
       (prog2 (add-to-xtt fp1 fp2)
              (sigprint fp1 fp2)))

(defun sigprint
       (fp1 fp2)
       (sequel-print fp1)
       (write-string " * ")
       (write fp2)
       (terpri))

(defun add-to-xtt
       (fp1 fp2)
       (prog2 (compile-sig fp1 fp2)
              (record-typed-def fp1 fp2)))

(defun query-if-abort
       nil
       (if (y-or-n-p "abort to top level?")
           (throw 'abort 'aborted)))

(defun greek
       (fp1)
       (greek1 fp1 fp1 *greek-alphabet*))

(defun greek1
       (fp1 fp2 fp3)
       (cond ((null fp3) (raise "code 28: I've run out of Greek letters!"))
             ((atom fp1) fp2)
             ((and (consp fp1)
                   (consp fp3)
                   (or (logical-var (car fp1))
                       (and (symbolp (car fp1))
                            (arbterm (car fp1)))))
              (greek1 (rremove (car fp1) (cdr fp1))
                      (subst (car fp3)
                             (car fp1)
                             fp2
                             ':test
                             'equal)
                      (cdr fp3)))
             ((and (consp fp1)
                   (atom (car fp1)))
              (greek1 (cdr fp1) fp2 fp3))
             ((and (consp fp1)
                   (consp (car fp1)))
              (greek1 (append (car fp1) (cdr fp1))
                      fp2
                      fp3))
             (t (raise "code 13: No Patterns have Fired in greek1"))))

(defun decompile
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq '-> (cadr fp1))
                   (listp (car fp1))
                   (not (eq (caar fp1) 'list))
                   (not (logical-var (car fp1))))
              (mapcar 'decompile
                      (append (car fp1)
                              (list '-> (caddr fp1)))))
             ((consp fp1) (mapcar 'decompile fp1))
             (t fp1)))

(defun remove-$-if-needed
       (fp1)
       (cond ((atom fp1) fp1)
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '$ (car fp1))
                   (var (cadr fp1)))
              (cadr fp1))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '$ (car fp1)))
              (concat (cadr fp1) '?))
             (t (mapcar 'remove-$-if-needed fp1))))

(defun record-typed-def
       (fp1 fp2)
       (setf (gethash fp1 *sfht*) fp2))

(defun compile-sig
       (fp1 fp2)
       (compile-horn-clauses
         (list
           (cons
             (cons (concat fp1 '&)
                   (append (list (var-times (length (butlast fp2 2))))
                           (list (first (last fp2)))
                           (list 'context?)))
             (cons '<-
                   (phi-times (butlast fp2 2)
                              (var-times (length (butlast fp2 2)))))))))

(defun var-times
       (fp1)
       (cond ((equal 0 fp1) nil)
             (t (cons (concat (concat fp1 '&) '?)
                      (var-times (1- fp1))))))

(defun phi-times
       (fp1 fp2)
       (cond ((and (null fp1) (null fp2)) nil)
             ((and (consp fp1) (consp fp2))
              (cons (list 'phi
                          (car fp2)
                          (car fp1)
                          'context?)
                    (phi-times (cdr fp1) (cdr fp2))))
             (t (raise "code 13: No Patterns have Fired in phi-times"))))

(defun declare-type
       (fp1 fp2)
       (setf (gethash (concat fp1 '&) *sfht*)
             fp2))

(defun lambda-function
       (fp1)
       (embed-op (ppx fp1)))

(defun ttc+ nil (set '*ttc* t))

(defun ttc- nil (set '*ttc* nil))
