; -*- mode:     CL -*- ----------------------------------------------------- ;
; File:         expand-file-name.l
; Description:  expand vars in a file-name-string (Unix or MacIntosh)
; Author:       Joachim H. Laubsch
; Created:      13-Nov-91
; Modified:     Tue Aug 11 12:05:46 1992 (Joachim H. Laubsch)
; Language:     CL
; Package:      CL-USER
;;; *************************************************************************
;;; Copyright (c) 1989, Hewlett-Packard Company
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Hewlett-Packard Company
;;; makes no warranty about the software, its performance or its conformity
;;; to any specification.
;;; 
;;; Suggestions, comments and requests for improvements are welcome
;;; and should be mailed to laubsch@hplabs.com.
;;; *************************************************************************

(in-package "CL-USER")
(provide "expand-file-name")

;--------------------------------------------------------------------------;
; expand-file-name
;-----------------
; expand UNIX environment-vars in a file-name-string
; if they are defined

#+KCL (defvar *logical-pathnames* ())
#+(or LUCID KCL)
(defun EXPAND-FILE-NAME (FILENAME)
  "Convert FILENAME to absolute. Initial ~ is expanded."
  (declare (optimize (safety 3)))
  (typecase FILENAME
    (string)
    (symbol (setq FILENAME (symbol-name FILENAME)))
    (pathname (setq FILENAME (namestring FILENAME)))
    (t (error "~S should be a string naming a pathname" FILENAME)))
  (flet ((strip-slash (s)
	   (let ((ln (length s)))
	     (if (char-equal (elt s (1- ln)) #\/)
		 (subseq s 0 (1- ln))
	       s))))
    (let* ((env0 (position #\$ FILENAME)))
      (if env0
	  (let* ((env1 (or (position #\/ FILENAME :start env0)
			   (length FILENAME)))
		 (vname (subseq FILENAME (1+ env0) env1))
		 (value (#+(or ALLEGRO KCL) SYSTEM::getenv
			   #+ LUCID           SYSTEM::environment-variable
			   #-(or ALLEGRO KCL LUCID) identity
			   vname)))
	    ;; allow local redefinition via define-logical-pathname
	    #+KCL(let ((p (assoc vname *logical-pathnames*
				 :test #'string=)))
		   (when p (setq value (cdr p))))
	    (if value
		(expand-file-name
		 (concatenate 'string
			      (subseq FILENAME 0 env0)
			      (strip-slash value)
			      (subseq FILENAME env1)))
	      FILENAME))
	(if (and (> (length FILENAME) 1) (string= "~/" FILENAME :end2 2))
	    (concatenate 'string
			 (namestring (USER-HOMEDIR-PATHNAME))
			 (subseq FILENAME 2))
	  FILENAME)))))

#+(or CCL MCL)
(defun expand-file-name (FILENAME)
  "Convert FILENAME from Unix Syntax to Mac Syntax, substituting logical directories."
  (declare (optimize (safety 3)))
  (typecase FILENAME
    (string)
    (symbol (setq FILENAME (symbol-name FILENAME)))
    (pathname (setq FILENAME (namestring FILENAME)))
    (t (error "~S should be a string naming a pathname" FILENAME)))
  (flet ((strip-seperator (s)
	   (let ((ln (length s)))
	     (if (char-equal (elt s (1- ln)) #\/)
                 (subseq s 0 (1- ln))
	       s))))
    (setq filename (strip-seperator filename))
    (let* ((env0 (position #\$ FILENAME)))
      (substitute
       #\:
       #\/
       (if env0
           (let ((env1 (position #\/ FILENAME :start env0)))
             (#+MCL identity #-MCL expand-logical-namestring
              (concatenate 'string 
                           (subseq FILENAME (1+ env0) env1)
                           ";"
                           (if env1 (subseq FILENAME (1+ env1)) ""))))
	 FILENAME)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                          End of expand-file-name.l
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
