#include <stdio.h>
#include <assert.h>

#include "lisp.h"

extern char *xmalloc();

lispobj *Qnil;
lispobj *Qquote;
lispobj *Qeof;
#if !defined(MINIMAL)
#if notyet
lispobj *Qfunction;
#endif
#endif
lispobj *Qsexp_read;		/* need to come up with a better name... */

void
Finitialize()
{
    Qnil = (lispobj *)xmalloc(sizeof (lispobj));
    Qnil->lo_type = LispObj_Cons;
    Qnil->lo_car = Qnil;
    Qnil->lo_cdr = Qnil;

    Qquote = (lispobj *)xmalloc(sizeof (lispobj));
    Qquote->lo_type = LispObj_Symbol;
    Qquote->lo_symbol = "quote";

    Qeof = (lispobj *)xmalloc(sizeof (lispobj));
    Qeof->lo_type = LispObj_Symbol;
    Qeof->lo_symbol = "<<eof>>";

#if !defined(MINIMAL)
#if notyet
    Qt = (lispobj *)xmalloc(sizeof (lispobj));
    Qt->lo_type = LispObj_Symbol;
    Qt->lo_symbol = "t";

    Qfunction = (lispobj *)xmalloc(sizeof (lispobj));
    Qfunction->lo_type = LispObj_Symbol;
    Qfunction->lo_symbol = "function";
#endif
#endif
}

void
Frecord(obj)
    lispobj *obj;
{
    Qsexp_read = obj;
}

lispobj *
Fcons(obj1, obj2)
    lispobj *obj1;
    lispobj *obj2;
{
    lispobj *ret = (lispobj *)xmalloc(sizeof (lispobj));

    ret->lo_type = LispObj_Cons;
    ret->lo_car = obj1;
    ret->lo_cdr = obj2;

    return (ret);
}

lispobj *
Fnconc2(obj1, obj2)
    lispobj *obj1;
    lispobj *obj2;
{
    lispobj *ptr;

    assert((obj1->lo_type == LispObj_Cons) && (obj1 != Qnil));

    for (ptr = obj1;
	 (ptr->lo_cdr->lo_type == LispObj_Cons) &&
	 (ptr->lo_cdr != Qnil);
	 ptr = ptr->lo_cdr)
	;

    ptr->lo_cdr = obj2;

    return (obj1);
}

lispobj *
Fnumber(number)
    long number;
{
    lispobj *ret = (lispobj *)xmalloc(sizeof (lispobj));

    ret->lo_type = LispObj_Number;
    ret->lo_number = number;

    return (ret);
}

lispobj *
Fstring(string)
    char *string;
{
    lispobj *ret = (lispobj *)xmalloc(sizeof (lispobj));

    ret->lo_type = LispObj_String;
    ret->lo_string = string;

    return (ret);
}

lispobj *
Fsymbol(symbol)
    char *symbol;
{
    lispobj *ret = (lispobj *)xmalloc(sizeof (lispobj));

    ret->lo_type = LispObj_Symbol;
    ret->lo_symbol = symbol;

    return (ret);
}

void
Fprint(obj)
    lispobj *obj;
{
    Fprint_internal(obj);
    printf("\n");
}

Fprint_internal(obj)
    lispobj *obj;
{
    switch (obj->lo_type) {
    case LispObj_Number:
	printf("%d", obj->lo_number);
	break;

    case LispObj_Symbol:
	printf("%s", obj->lo_symbol);
	break;

    case LispObj_String:
	printf("\"%s\"", obj->lo_string);
	break;

    case LispObj_Cons:
	if (obj == Qnil) {
	    printf("nil");
	    break;
	}
	printf("(");
	Fprint_internal(obj->lo_car);
	while (((obj = obj->lo_cdr) != Qnil) &&
	       (obj->lo_type == LispObj_Cons)) {
	    printf(" ");
	    Fprint_internal(obj->lo_car);
	}
	if (obj != Qnil) {
	    printf(" . ");
	    Fprint_internal(obj);
	}
	printf(")");
	break;
	
    default:
	fprintf(stderr, "Illegal object passed to Fprint_internal, type = %d, address = 0x%x\n",
		obj->lo_type, obj);
	fprintf(stderr, "abort()'ing...\n");
	abort();
    }
}

/*
 * Note: we must take special care to see to it that special objects
 * such as nil and quote do not get freed!
 */
void
Ffree(obj)
    lispobj *obj;
{
    if ((obj == NULL) || (obj == Qnil))
	return;

    switch (obj->lo_type) {
    case LispObj_Number:
	obj->lo_type = LispObj_BadType;
	free((char *)obj);
	break;

    case LispObj_String:
	if (obj->lo_string)
	    free(obj->lo_string);

	obj->lo_string = NULL;
	obj->lo_type = LispObj_BadType;
	free((char *)obj);
	break;

    case LispObj_Symbol:
	if (obj == Qquote)	/* quote is a reserved symbol */
	    break;

	if (obj->lo_symbol)
	    free(obj->lo_symbol);

	obj->lo_type = LispObj_BadType;
	obj->lo_symbol = NULL;
	free((char *)obj);
	break;

    case LispObj_Cons:
	if (obj->lo_car != Qnil)
	    Ffree(obj->lo_car);

	if (obj->lo_cdr != Qnil)
	    Ffree(obj->lo_cdr);

	obj->lo_type = LispObj_BadType;
	free((char *)obj);
	break;

    case LispObj_BadType:
	fprintf(stderr, "Attempt to free already free'd object!, address = 0x%x\n",
		obj);
	abort();

    default:
	fprintf(stderr, "Unknown object type passed to Ffree, type = %d, address = 0x%x\n",
		obj->lo_type, obj);
	abort();
    }
}

char *
xmalloc(size)
    int size;
{
    extern char *malloc();

    char *ret = malloc(size);

    if (ret == NULL) {
	fprintf(stderr, "Unable to malloc %d bytes, aborting...\n", size);
	abort();
    }
    return (ret);
}
