;;;___________________________________________________________________________
;;;
;;;                       System: edit
;;;                       (Version 1.1)
;;;
;;; Copyright (c): Forschungsgruppe INFORM
;;;                Universitaet Stuttgart
;;;
;;; File:		       edit.lsp
;;; Last Modification Time:    Thu Apr  7 11:40:39 1988
;;; Last Modification By:      Andreas Girgensohn 
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;___________________________________________________________________________ 

;;; terminal independent routines of TERMCAP are used

(eval-when (compile load eval)
  (unless (fboundp 'termcapinit)
    (load "/users/andreasg/cl/topreader/termcap")))
(termcapinit)

(eval-when (compile load eval)
  (defmacro tcexe (func &rest rest)
    `(termcapexe ,(string-downcase (symbol-name func)) ,@rest))

  (defconstant $left-paren-char$ #\()
  (defconstant $right-paren-char$ #\))
  (defconstant $double-quote-char$ #\")
  
  (defstruct (buffer-structure
	       (:conc-name buffer-))
	     (cursor-x 0)
	     (cursor-y 0)
	     (lines (list (get-empty-line))))
)

;;; main loop

(defun editor-loop (*buffer* &optional (prompt ""))
  (declare (special *buffer*))
  (check-type prompt string)
  (let ((*indent* (length prompt)))
    (declare (special *indent*))
    (princ prompt)
    (display-buffer *buffer*)
    (do ((*exit-flag* nil))
	(*exit-flag* *buffer*)
	(declare (special *exit-flag*))
	(in-tyimode 1
		    (let ((char (read-char)))
		      (execute-command char))))))

(defun execute-command (char)
  (declare (special *editor-readtable*))
  (funcall (aref *editor-readtable* (char-int char)) char))

;;; readtable

(defun make-editor-readtable ()
  (let ((tr (make-array '(128))))
    (do ((i 0 (1+ i))) ((> i 31))
	(setf (aref tr i) 'illegal-operation))
    (do ((i 32 (1+ i))) ((> i 127))
	(setf (aref tr i) 'self-insert))
    (dolist (l `((#\% exit-editor)	       ; test
		 (#\^A beginning-of-line)
		 (#\^B backward-character)
		 (#\^D delete-next-character)
		 (#\^E end-of-line)
		 (#\^F forward-character)
		 (#\tab indent-line)
		 (#\newline newline-and-indent)
		 (#\^K kill-to-end-of-line)
		 (#\return newline)
		 (#\^N next-line)
		 (#\^P previous-line)
		 (,$right-paren-char$ self-insert) ; right-paren
		 (#\; illegal-operation)
		 (#\rubout delete-previous-character)))
      (setf (aref tr (char-int (car l))) (cadr l)))
    tr))

(locally (declare (special *editor-readtable*))
  (setq *editor-readtable* (make-editor-readtable)))

;;; utility functions

(defun display-buffer (buffer)
  (declare (special *indent*))
  (check-type buffer buffer-structure)
  (princ (car (buffer-lines buffer)))
  (terpri)
  (dolist (line (cdr (buffer-lines buffer)))
    (printblanks *indent*)
    (princ line)
    (terpri))
  (termcap-set-cursor-relative 
    (+ (buffer-cursor-x buffer) *indent*)
    (- (buffer-cursor-y buffer) (length (buffer-lines buffer)))))

(defun set-cursor (buffer x y)
  (setf (buffer-cursor-x buffer) x
	(buffer-cursor-y buffer) y))

(locally (declare (special *line-resources*))
  (setq *line-resources* nil))

(defun dispose-line (line)
  (declare (special *line-resources*))
  (setf (fill-pointer line) 0)
  (push line *line-resources*))

(defun get-empty-line ()
  (declare (special *line-resources*))
  (or (pop *line-resources*)
      (make-array '(80) :element-type 'string-char :fill-pointer 0 :adjustable t)))

(defun printblanks (n &optional stream)
  (let ((easy (member n '( 0  ""
			   1  " "
			   2  "  "
			   3  "   "
			   4  "    "
			   5  "     "
			   6  "      "
			   7  "       "
			   8  "        "))))
    (if easy
	(princ (cadr easy) stream)
	(dotimes (i n)
	  (write-char #\space stream)))))

(defun beep ()
  (write-char #\^G)
  (force-output))

(defun delete-linefeed ()
  (declare (special *buffer*))
  (cond ((eql (buffer-cursor-y *buffer*) (1- (length (buffer-lines *buffer*))))
	 (beep))
	(t (let ((current-line (nth (buffer-cursor-y *buffer*)
				    (buffer-lines *buffer*)))
		 (next-line (pop (cdr (nthcdr (buffer-cursor-y *buffer*)
					      (buffer-lines *buffer*))))))
	     (princ next-line)
	     (dotimes (x (length next-line))
	       (vector-push-extend (aref next-line x) current-line))
	     (termcap-set-cursor-relative (- (length current-line)) +1)
	     (termcap-delete-line)
	     (termcap-set-cursor-relative (buffer-cursor-x *buffer*) -1)))))

(locally (declare (special *main-buffer*))
  (setq *main-buffer* (make-buffer-structure)))

;;; reading a s-expression from an editor buffer

(locally (declare (special *read-buffer*))
  (setq *read-buffer*
	(make-array '(500) :element-type 'string-char :fill-pointer 0)))

(defun read-from-edit-buffer (buffer)
  (declare (special *read-buffer*))
  (setf (fill-pointer *read-buffer*) 0)
  (format *read-buffer* "~A" (car (buffer-lines buffer)))
  (dolist (line (cdr (buffer-lines buffer)))
    (format *read-buffer* "~%~A" line))
  (values (read-from-string *read-buffer*)))

;;; termcap interface

(defun termcap-set-cursor-relative (dx dy)
  (if (minusp dx)
      (dotimes (x (- dx))
	(termcapexe "kl"))
      (dotimes (x dx)
	(termcapexe "kr")))
  (if (minusp dy)
      (dotimes (y (- dy))
	(termcapexe "ku"))
      (dotimes (y dy)
	(termcapexe "kd"))))

(defun termcap-insert-character (char)
  (termcapexe "ic")
  (write-char char)
  (force-output))
  
(defun termcap-delete-character ()
  (termcapexe "dc"))

(defun termcap-clear-to-end-of-line ()
  (termcapexe "ce"))

(defun termcap-add-blank-line ()
  (termcapexe "al"))

(defun termcap-delete-line ()
  (termcapexe "dl"))

;;; commands

(defun illegal-operation (char)
  (declare (ignore char))
  (beep))

(defun self-insert (char)
  (declare (special *buffer*))
  (termcap-insert-character char)
  (force-output)
  (let* ((line (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*)))
	 (line-length (length line))
	 (column (buffer-cursor-x *buffer*)))
    (cond ((eql column line-length)
	   (vector-push-extend char line))
	  (t (vector-push-extend #\space line)
	     (do ((x line-length (1- x)))
		 ((eql x column)
		  (setf (aref line x) char))
		 (setf (aref line x) (aref line (1- x))))))
    (incf (buffer-cursor-x *buffer*))))

(defun exit-editor (char)
  (declare (special *exit-flag*) (ignore char))
  (setq *exit-flag* t))

(defun beginning-of-line (char)
  (declare (special *buffer*) (ignore char))
  (termcap-set-cursor-relative (- (buffer-cursor-x *buffer*)) 0)
  (setf (buffer-cursor-x *buffer*) 0))

(defun backward-character (char)
  (declare (special *buffer*) (ignore char))
  (cond ((plusp (buffer-cursor-x *buffer*))
	 (termcap-set-cursor-relative -1 0)
	 (decf (buffer-cursor-x *buffer*)))
	((zerop (buffer-cursor-y *buffer*))
	 (beep))
	(t (decf (buffer-cursor-y *buffer*))
	   (let ((column (length (nth (buffer-cursor-y *buffer*)
				      (buffer-lines *buffer*)))))
	     (termcap-set-cursor-relative column -1)
	     (setf (buffer-cursor-x *buffer*) column)))))

(defun delete-next-character (char)
  (declare (special *buffer*) (ignore char))
  (let* ((line (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*)))
	 (line-length (length line)))
    (cond ((eql (buffer-cursor-x *buffer*) line-length)
	   (delete-linefeed))
	  (t (do ((x (buffer-cursor-x *buffer*) (1+ x)))
		 ((>= x (1- line-length))
		  (vector-pop line))
		 (setf (aref line x) (aref line (1+ x))))
	     (termcap-delete-character)))))

(defun end-of-line (char)
  (declare (special *buffer*) (ignore char))
  (let ((line-length
	  (length (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*)))))
    (termcap-set-cursor-relative (- line-length (buffer-cursor-x *buffer*)) 0)
    (setf (buffer-cursor-x *buffer*) line-length)))

(defun forward-character (char)
  (declare (special *buffer*) (ignore char))
  (cond ((< (buffer-cursor-x *buffer*)
	    (length (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*))))
	 (termcap-set-cursor-relative +1 0)
	 (incf (buffer-cursor-x *buffer*)))
	((eql (buffer-cursor-y *buffer*) (1- (length (buffer-lines *buffer*))))
	 (beep))
	(t (termcap-set-cursor-relative (- (buffer-cursor-x *buffer*)) +1)
	   (incf (buffer-cursor-y *buffer*))
	   (setf (buffer-cursor-x *buffer*) 0))))

(defun indent-line (char)
  (declare (special *buffer*) (ignore char))
)

(defun newline-and-indent (char)
  (newline char)
  (indent-line char))

(defun kill-to-end-of-line (char)
  (declare (special *buffer* *indent*) (ignore char))
  (let ((line (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*))))
    (cond ((eql (buffer-cursor-x *buffer*) (length line))
	   (delete-linefeed))
	  (t (setf (fill-pointer line) (buffer-cursor-x *buffer*))
	     (termcap-clear-to-end-of-line)
	     (terpri)
	     (termcap-set-cursor-relative
	       (+ (buffer-cursor-x *buffer*) *indent*) -1)))))

(defun newline (char)
  (declare (special *buffer* *indent*) (ignore char))
  (let ((current-line (nth (buffer-cursor-y *buffer*) (buffer-lines *buffer*)))
	(next-line (get-empty-line)))
    (push next-line
	  (cdr (nthcdr (buffer-cursor-y *buffer*) (buffer-lines *buffer*))))
    (let ((column (buffer-cursor-x *buffer*)))
      (when (< column (length current-line))
	(dotimes (x (- (length current-line) column))
	  (vector-push-extend (aref current-line (+ x column)) next-line))
	(setf (fill-pointer current-line) column)
	(termcap-clear-to-end-of-line)))
    ;; terpri must be in last line of buffer iff scrolling is necessary
    (let ((dy (- (length (buffer-lines *buffer*)) (buffer-cursor-y *buffer*) 2)))
      (unless (zerop dy)
	(termcap-set-cursor-relative 0 dy))
      (terpri)
      (unless (zerop dy)
	(termcap-set-cursor-relative 0 (- dy))))
    (termcap-add-blank-line)
    (printblanks *indent*)
    (princ next-line)
    (termcap-set-cursor-relative (- (length next-line)) 0)
    (force-output)
    (incf (buffer-cursor-y *buffer*))
    (setf (buffer-cursor-x *buffer*) 0)))

(defun next-line (char)
  (declare (special *buffer*) (ignore char))
  (cond ((eql (buffer-cursor-y *buffer*) (1- (length (buffer-lines *buffer*))))
	 (beep))
	(t (incf (buffer-cursor-y *buffer*))
	   (let ((line-length (length (nth (buffer-cursor-y *buffer*)
					   (buffer-lines *buffer*))))
		 (dx 0))
	     (when (> (buffer-cursor-x *buffer*) line-length)
	       (setf dx (- line-length (buffer-cursor-x *buffer*))
		     (buffer-cursor-x *buffer*) line-length))
	     (termcap-set-cursor-relative dx +1)))))

(defun previous-line (char)
  (declare (special *buffer*) (ignore char))
  (cond ((zerop (buffer-cursor-y *buffer*))
	 (beep))
	(t (decf (buffer-cursor-y *buffer*))
	   (let ((line-length (length (nth (buffer-cursor-y *buffer*)
					   (buffer-lines *buffer*))))
		 (dx 0))
	     (when (> (buffer-cursor-x *buffer*) line-length)
	       (setf dx (- line-length (buffer-cursor-x *buffer*))
		     (buffer-cursor-x *buffer*) line-length))
	     (termcap-set-cursor-relative dx -1)))))

(defun delete-previous-character (char)
  (declare (special *buffer*))
  (cond ((and (zerop (buffer-cursor-x *buffer*))
	      (zerop (buffer-cursor-y *buffer*)))
	 (beep))
	(t (backward-character char)
	   (delete-next-character char))))

