;;; Tue Dec 21 19:31:46 1993 by Mark Kantrowitz <mkant@cs.cmu.edu>
;;; socket-streams.lisp -- 8170 bytes

;;; Copyright (C) 1993 by Mark Kantrowitz, CMU School of Computer Science.
;;; All rights reserved.
;;;
;;; Use, copying, modification, and distribution are permitted, provided
;;; this copyright notice is retained intact, proper credit is given to
;;; the author in all relevant publications, and copies of any improvements
;;; are sent to the author for inclusion in future versions of this program.
;;;
;;; Written by Mark Kantrowitz, School of Computer Science, Carnegie Mellon
;;; University, 5000 Forbes Avenue, Pittsburgh, PA 15213-3891.
;;; E-mail: mkant@cs.cmu.edu
;;;
;;; Please let us know of any novel uses of this code.

;;; ****************************************************************
;;; Streams-based Socket Interface for Lisp ************************
;;; ****************************************************************

;;; This file provides a stream-based interface to UNIX sockets for Lisp.
;;;
;;; It consists of two files: 
;;;      socket-streams.lisp
;;;      sockets.c
;;; Compile the latter using your favorite C compiler, e.g.,
;;;      cc -c sockets.c
;;; to create sockets.o. The sockets.o file is loaded by this file.
;;;
;;; It has been tested in Lucid and Allegro on a HP 700.
;;;
;;; An example of using the interface to provide a Lisp FINGER function
;;; has been included.
;;;
;;; Our basic approach is to do most of the work in C (opening a connection
;;; and returning a file descriptor) and let the Lisp end handle the streams
;;; interface.
;;;
;;; Ideally a generalization of this code should become a standard part of
;;; most Common Lisp implementations.
;;;
;;; Note that we haven't implemented any part of the server side of
;;; the interface, just the client side. The server side shouldn't be
;;; too difficult to add. We didn't add it because we didn't need it.

(export '(get-service-by-name 
	  open-tcp-stream close-tcp-stream with-open-output-device
	  get-service-by-name
	  finger))

;;; ********************************
;;; Load Foreign Files *************
;;; ********************************

#+:lucid
(def-foreign-function (establish-connection
		       (:language :c)
                        (:name "establish_connection")
                        (:return-type :signed-32bit))
  (host :simple-string)
  (port :simple-string))

#+:lucid
(def-foreign-function (close-connection
		       (:language :c)
                        (:name "close_connection")
                        ; (:return-type )
			)
  (port :unsigned-32bit))

#+:lucid
(def-foreign-function (get-serv-port
		       (:language :c)
		       (:name "get_serv_port")
		       (:return-type :signed-32bit)
		       )
    (service :simple-string)
    (type :simple-string))

#+:lucid
(load-foreign-files "sockets.o")

#+allegro-v3.1 (require :foreign)
#+:allegro
(load "sockets.o")
#+:allegro
(ff:defforeign 'establish-connection
	       :entry-point "establish_connection"
	       :arguments '(string string)
	       :return-type :integer)
#+:allegro
(ff:defforeign 'close-connection
	       :entry-point "close_connection"
	       :arguments '(integer))
#+:allegro
(ff:defforeign 'get-serv-port
    :entry-point "get_serv_port"
    :arguments '(string string)
    :return-type :integer)

;;; ********************************
;;; Allegro Streams Fixes **********
;;; ********************************
	       
#+:allegro
(defclass binary-socket-stream (excl::bidirectional-terminal-stream) ())
#+:allegro
(defmethod stream-read-byte ((s binary-socket-stream))
       (let ((c (stream-read-char s)))
	 (if (eq :eof c)
	     :eof
	   (char-code c))))
#+:allegro
(defmethod stream-write-byte ((s binary-socket-stream) integer)
  (let ((c (code-char integer)))
    (stream-write-char s c)))

;;; ********************************
;;; Decode Device ******************
;;; ********************************

;;; (decode-device "glinda.oz.cs.cmu.edu:3000") returns two strings:
;;; "glinda.oz.cs.cmu.edu"
;;; "3000"

(defun decode-device (device-string)
  (let ((colon-pos (position #\: device-string)))
    (cond (colon-pos
	   (values (subseq device-string 0 colon-pos)
		   (subseq device-string (1+ (the fixnum colon-pos)))))
	  (t
	   device-string))))

;;; ********************************
;;; TCP Streams ********************
;;; ********************************

(defun open-tcp-stream (host port 
			     &key (element-type '(unsigned-byte 8))
;			     #+:allegro (element-size 8)
			     )
  "Given a host and a port, returns a stream whose input and output
   points to a TCP/IP socket on that port. The C code does most of the work,
   establishing a connection and returning a file descriptor. HOST is the
   string host name and PORT is the port number given as a string."
  ;; We don't support integer internet addresses, but that's easy to add.
  ;; For string service names, see GET-SERVICE-BY-NAME.
  (let ((fd (the fixnum (establish-connection host port))))
    (when (minusp fd) 
      (error "Problem establishing connection"))
    #+:lucid
    (make-lisp-stream :input-handle fd :output-handle fd 
		      :element-type element-type
		      :auto-force nil
		      :positionable nil)

    #+allegro-v3.1 (excl::make-buffered-terminal-stream fd fd t t)
    #+(or :allegro-v4.0 :allegro-v4.1 :allegro-v4.2)
    (make-instance 'binary-socket-stream
		   :fn-in  fd
		   :fn-out fd
		   :element-type element-type
;		   :element-size element-size
		   )))

(defun close-tcp-stream (stream)
  "Closes the stream's socket."
  ;; Only seems to be necessary in Allegro.
  #+:lucid
  stream
;  (close-connection (lucid::stream-file-handle 
;		     (lucid::split-stream-output-stream stream)))
  #+:allegro
  ;; (slot-value stream 'excl::fn-out) 
  (close-connection (EXCL::STREAM-OUTPUT-FN stream))
  )

;;; ********************************
;;; With Open Output Device ********
;;; ********************************

(defmacro with-open-output-device ((stream-var device-str &rest options)
				   &body body)
  "Generalization of WITH-OPEN-FILE. If device-str is of the form accepted
   by DECODE-DEVICE, opens a temporary TCP connection to the host-port pair,
   closing it when done."
  (let ((abortp (gensym "ABORTP"))
	(device (gensym "DEVICE"))
	(host (gensym "HOST"))
	(port (gensym "PORT")))
    `(let ((,device ,device-str))
       (multiple-value-bind (,host ,port) (decode-device ,device)
	 (let ((,stream-var (if ,port
				(open-tcp-stream ,host ,port ,@options)
				(open ,device :direction :output
				      :if-does-not-exist :create
				      :if-exists :overwrite
				      ,@options)))
	       (,abortp t))
	   (unwind-protect (multiple-value-prog1 ,@body
						 (setq ,abortp nil))
	     (when (streamp ,stream-var)
	       (when ,port (close-tcp-stream ,stream-var))
	       (close ,stream-var :abort ,abortp))))))))

;;; ********************************
;;; Get Service Port ***************
;;; ********************************
(defun get-service-by-name (name)
  "Translates a service name, like finger, into the port number (79)."
  (let ((port (get-serv-port name "tcp")))
    (if (zerop port)
	(error "Unknown Service")
	port)))

;;; ********************************
;;; For Testing Purposes ***********
;;; ********************************

(defun finger (host &optional (name ""))
  "Fingers NAME on HOST to illustrate the streams interface to sockets.
   Prints the information returned. If NAME isn't given, reports on all
   users, just like Unix finger."
  (with-open-output-device (s (format nil "~A:~A" 
				      host
				      ;; should return 79
				      (get-service-by-name "finger"))
			      :element-type #+:allegro 'character
			                    #+:lucid 'string-char)
    ;; FINGER wants to see name<CR> or just <CR>. The latter reports
    ;; on all users.
    (format s "~a~%" name)
    (force-output s)
    ;; Read and print all the information returned.
    (do ((line (read-line s nil nil) (read-line s nil nil)))
        ((null line) (values))
      (write-line (remove #\return line)))))

;;; *EOF*
