;;;-*- Package: RPC2; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

;;; Copyright (c) 1988, 1989, 1990 Stanford University and Xerox Corporation.  
;;; All Rights Reserved.

;;; Permission is hereby granted to use, reproduce, and prepare derivative
;;; works of this software provided that any derivative work based upon
;;; this software is licensed to Stanford University and Xerox Corporation
;;; at no charge.  Any distribution of this software or derivative works
;;; must comply with all applicable United States export control laws.
;;; Any copy of this software or of any derivative work must include the
;;; above copyright notice of Stanford University and Xerox Corporation
;;; and this paragraph.  This software is made available AS IS, and
;;; STANFORD UNIVERSITY AND XEROX CORPORATION DISCLAIM ALL WARRANTIES,
;;; EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
;;; AND NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY
;;; LIABILITY FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS
;;; EXPRESSLY DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING
;;; NEGLIGENCE) OR STRICT LIABILITY, EVEN IF STANFORD UNIVERSITY AND/OR
;;; XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

(provide "RPCRPC")

(in-package "RPC2")


(eval-when (eval compile)
  (require "RPCDECLS"))

                                           ; Public variables


(defglobalparameter *debug* nil "T for printout, NUMBER for even more.")

(defparameter *msec-until-timeout* 10000 
 "Total time in msec before giving up on UDP exchange with remote host")

(defparameter *msec-between-tries* 500 "Time in msec between UDP retries")

(defparameter *rpc-ok-to-cache* t
  "If NIL, does not attempt to cache socket numbers for non-well-known sockets")

(defvar *rpc-socket-cache* ()
  "A list of (<iphost-address> <remote-program-name> <remote-program-version>
           <protocol> <ipsocket-number>) quintuples.")

(defglobalvar *rpc-well-known-sockets*
       '((* 100000 2 udp 111)
         (* 100000 2 tcp 111))
  "List of well-known RPC programs and their sockets.
Each element is a list:
  (host-address prog-number prog-version protocol socket-number)

Host-address may be *, in which case it matches any host address.
Protocol should be either rpc2:UDP or rpc2:TCP.")

                                           ; Internal


(defglobalparameter *rpc-protocol-types*
    '((udp . rpc-create-some-udp-stream)
      (tcp . rpc-create-tcp-stream))
       "Association list mapping protocol names into functions that create the corresponding type of RPC stream.")

(defglobalvar *rpc-programs* ()
  "A list of RPC-PROGRAM structs.

This list is consulted by various routines to find infomation about known
remote programs.

It is assumed that a given NAME field uniquely identifies a (NUMBER, VERSION, PROTOCOL).
On the other hand, there may be several NAMEs (and hence, several RPC-STRUCTs) for
a given (NUMBER, VERSION, PROTOCOL).
")

(defglobalparameter *rpc-protocols* '((tcp . 6)
                                      (udp . 17))
  "Portmapper codes for various RPC transports")

(defglobalvar *xid-count* 0 
  "Contains the  XID stamp of the next remote procedure call")

(defglobalparameter *xid-max* 2147483647
  "Upper bound on XID's for transactions.  See CREATE-XID for discussion.")

(defvar *stream* nil "The stream on which an RPC is being performed (for debugging and error recovery).")

(defvar *program* nil "The rpc program object involved in current call (for debugging and error recovery).")

(defvar *procedure* nil "The rpc procedure object involved in current call (for debugging and error recovery).")

(defconstant *rpc-accept-stats*
  '((0 . success)
    (1 . program-unavailable)
    (2 . program-mismatch)
    (3 . procedure-unavailable)
    (4 . garbage-arguments)
    (5 . system-error))
       "Assoc list for internal use by PARSE-RPC-REPLY.")

                                           ; Defining RPC Programs


(defmacro define-remote-program
    (name number version protocol &key constants types inherits 
	  procedures)
  "
This macro expands into code to add a new RPC-PROGRAM struct to *RPC-PROGRAMS*.  The generated code checks first to see that there are no name conflicts with existing remote programs and then adds the new structure to *RPC-PROGRAMS*."
  (let ((ename (eval name))
	(enumber (eval number))
	(eversion (eval version))
	(eprotocol (or (eval protocol) 'udp))
	(econstants (eval constants))
	(etypes (eval types))
	(einherits (eval inherits))
	(eprocedures (eval procedures)))
    (check-type ename symbol)
    (check-type enumber number)
    (check-type eversion number)
    (setq eprotocol
	  (cond ((string-equal eprotocol "UDP")
		 'udp)
		((string-equal eprotocol "TCP")
		 (when *use-os-networking*
		   (warn "TCP is an unsupported RPC protocol on this machine."))
		 'tcp)
		((error "~a is an unknown RPC protocol." eprotocol))))
    (let ((rprog (define-remote-prog ename enumber eversion 
				     eprotocol econstants etypes einherits 
				     eprocedures)))
      `(define-program-internal ',ename ,enumber
	 ,eversion
	 ',eprotocol
	 ',(rpc-program-constants rprog)
	 ',(rpc-program-types rprog)
	 ',(rpc-program-inherits rprog)
	 ,(cons-up-rpc-procs (rpc-program-procedures rprog))))))

(defun define-program-internal
    (name number version protocol constants types inherits 
	  procedures)
       
  ;; Called by the macro expansion of DEFINE-REMOTE-PROGRAM,
  ;; arguments having been suitably error checked and the
  ;; procedures turned into code.
  (let ((newprog (make-rpc-program :number number :version version
				   :name name :protocol protocol :types types
				   :constants constants :inherits inherits 
				   :procedures procedures))
	mineisnewest newerversion v oldprog oldprog2)
            
    ;; Note: the functions MAKE-RPC-PROGRAM,
    ;; CLEAR-ANY-NAME-CONFLICTS, UNDEFINE-REMOTE-PROGRAM and
    ;; the variable *RPC-PROGRAMS* are referred to directly in
    ;; the old macro-expansion of DEFINE-REMOTE-PROGRAM, so
    ;; please leave them alone unless/until you are willing to
    ;; refuse to load old code.
    (dolist (pgm *rpc-programs*)
      (when (eq (rpc-program-name pgm) name)
					; Same name--only allowed
					; to have one of these, so
					; will delete this other
					; one
	(setq oldprog pgm))
      (when (and (eql (rpc-program-number pgm) number)
		 (eq (rpc-program-protocol pgm) protocol))
					; Existing program of same
					; number & protocol
	(setq v (rpc-program-version pgm))
	(cond ((eql v version)
					; Same version--overwrite it
	       (return (setq oldprog2 pgm)))
	      ((< v version)
					; Our version is newer than
					; this one
	       (unless newerversion (setq mineisnewest t))
	       (return nil))
	      (t			; Remember a version that
					; is newer than ours
	       (setq newerversion pgm)))))
    (format *debug-io* "~:[D~;Red~]efining remote program ~A, version ~D~@[ (~:[old~;new~] version)~]~%"
	    (or oldprog oldprog2)
	    name version (or mineisnewest newerversion)
	    mineisnewest)
    (when oldprog
      (setq *rpc-programs* (delete oldprog *rpc-programs*)))
    (when (and oldprog2 (not (eq oldprog2 oldprog)))
      (setq *rpc-programs* (delete oldprog2 *rpc-programs*)))
    (if newerversion
	(push newprog (cdr (member newerversion *rpc-programs*)))
	(push newprog *rpc-programs*))
    name))

(defun define-remote-prog (name number version protocol constants types
                                inherits procedures)
       
  ;; This guy does the work, so that DEFINE-REMOTE-PROGRAM can
  ;; cons up the macro easily.
       
  ;; An RPC-PROGRAM struct RPROG is passed back to
  ;; DEFINE-REMOTE-PROGRAM. Its innards are then used by
  ;; DEFINE-REMOTE-PROGRAM to build up the big cons that will
  ;; cons up the proper RPC-PROGRAM later.
  (format-t 
   "Building XDR routines for remote program ~a, version ~a~%"
   name version)
  (let ((rprog (make-rpc-program :number number :version version 
				 :name name :protocol protocol)))
            
    ;; Slightly clumsy structure here, because the def-rpc-xxx
    ;; routines (curently just DEF-RPC-PROCEDURES) might want
    ;; to be able to get at the other pieces of the
    ;; definition.
    (setf (rpc-program-inherits rprog)
	  (def-rpc-inherits rprog inherits))
    (setf (rpc-program-constants rprog)
	  (def-rpc-constants rprog constants))
    (setf (rpc-program-types rprog)
	  (def-rpc-types rprog types))
    (setf (rpc-program-procedures rprog)
	  (def-rpc-procedures rprog procedures))
    rprog))

(defun cons-up-rpc-procs (procs)
  "Given a list of RPC-PROCEDURE structs, conses up code to produce that set of RPC-PROCEDURE structs."
  `(list
    ,@(mapcar #'(lambda (proc)
		  `(make-rpc-procedure :name
				       ',(rpc-procedure-name proc)
				       :procnum
				       ',(rpc-procedure-procnum proc)
				       :argtypes
				       ,(rpc-procedure-argtypes proc)
				       :resulttypes
				       ,(rpc-procedure-resulttypes proc)))
	      procs)))

(defun clear-any-name-conflicts (name number version protocol)
  "Determines whether a proposed (NAME, NUMBER, VERSION, PROTOCOL) would violate
the assumption that a NAME uniquely specifies the other three components.

If there exists a violation, the user is given a chance to remove the old program.

Returns T if no violation of assumption (or violation is resolved by removing old program),
Returns NIL if there is an unresolved violation."
  (let (oldrpc)
    (cond ((and (setq oldrpc (find-rpc-program :name name))
		(or (/= number (rpc-program-number oldrpc))
		    (/= version (rpc-program-version oldrpc))
		    (not (eql protocol (rpc-program-protocol
					oldrpc)))))
	   (format *query-io* "Remote program name conflict with existing program:~%   Name ~a, Protocol ~A, Number ~a, Version ~a~%"
		   name (rpc-program-protocol oldrpc)
		   (rpc-program-number oldrpc)
		   (rpc-program-version oldrpc))
	   (and (yes-or-no-p 
		 "Do you want to remove the old program? "
		 )
		(undefine-remote-program (rpc-program-name
					  oldrpc)
					 (rpc-program-number oldrpc)
					 (rpc-program-version oldrpc)
					 (rpc-program-protocol oldrpc))))
	  (t t))))

(defun def-rpc-types (context typedefs)
  "Essentially a no-op, as typedefs are copied directly from the DEFINE-REMOTE-PROGRAM
into the RPC-PROGRAM struct. Just prints out the name of each type as it is encountered."
  (if typedefs (format-t "    Types~%"))
  (dolist (i typedefs)
    (format-t "        ~A~%" (first i)))
  typedefs)

(defun def-rpc-inherits (context proglist)
  "Checks remote program inherited by this one to make sure that it exists.
Issues a warning if it cannot find the program to be inherited."
  (if proglist (format-t "    Inherits~%"))
  (dolist (prg proglist proglist)
    (format-t "        ~A~%" prg)
    (if (not (and (symbolp prg)
		  (find-rpc-program :name prg)))
	(warn 
	 "Trying to inherit from remote program ~a, but ~a not found.~%"
	 prg prg))))

(defun def-rpc-procedures (context procs)
       "Returns a list of RPC-PROCEDURE structs returned by DEF-RPC-PROCEDURE."
       (check-type procs list "A list of RPC procedure declarations")
       (if procs (format-t "    Procedures~%"))
       (mapcar #'(lambda (proc)
                        (def-rpc-procedure context proc))
              procs))

(defun def-rpc-procedure (context proc)
  "For a procedure specified to DEFINE-REMOTE-PROGRAM's :PROCEDURES argument, creates and returns an RPC-PROCEDURE struct.

XDR procedure code is generated via the call to XDR-GENCODE-MAKEFCN."
  (check-type (first proc)
	      (and symbol (not null))
	      "a non-null symbol naming the RPC procedure.")
  (check-type (second proc)
	      (integer 0 *)
	      "a non-negative integer RPC procedure number")
  (check-type (third proc)
	      list)
  (let
      ((name (first proc))
       (argtypes (third proc))
       (results (fourth proc)))
    (format-t "        ~A~%" name)
    (make-rpc-procedure
     :name name :procnum (second proc)
     :argtypes
     (and argtypes
	  `#'(lambda (xdr-stream args)
	       (prog ((argtail args))
		  ,(def-rpc-arg-tail argtypes context t)
                        
		  ;; Fall thru here if some sort of argument
		  ;; mismatch
		  (rpc-argument-error args ,(length argtypes)))))
     :resulttypes
     (and results (list 'function (xdr-codegen context
					       (if (consp results)
					; Implicitly get a list back
						   (cons ':list results)
						   results)
					       'read))))))

(defun def-rpc-arg-tail (argtypes context &optional firsttime)
       
       ;; Generates code that writes each of the elements of the
       ;; variable ARGTAIL per the types in ARGTYPES, with a final
       ;; RETURN if no arg mismatch.  This is recursive, producing
       ;; fragments that looks like (when (setq argtail (cdr argtail))
       ;; (write-somehow (car argtail)) (when ...) ...).  If
       ;; FIRSTTIME, we don't pop ARGTAIL.
       
       ;; This could have been written non-recursively with rplacd's,
       ;; or by producing code with jumps.  The latter turns out to
       ;; compile worse in xcl.
       (if argtypes
           `(when ,(if firsttime
                       'argtail
                       '(setq argtail (cdr argtail)))
                                           ; Test that we have an
                                           ; argument
                ,(xdr-codegen-1 context (car argtypes)
                        'write
                        `(xdr-stream (car argtail)))
                                           ; Generate code to write it
                ,(def-rpc-arg-tail (cdr argtypes)
                        context)
                                           ; Finally, generate code to
                                           ; do the rest
                )
           '(when (null (cdr argtail))
                                           ; Having written all the
                                           ; args accounted for,
                                           ; verify that there are no
                                           ; more
                (return))))

(defun def-rpc-constants (context pairs)
       "
Checks that constants specified to DEFINE-REMOTE-PROGRAM are syntactically
reasonable.
"
       (if pairs (format-t "    Constants~%"))
       (dolist (pair pairs)
           (check-type (first pair)
                  (and (not null)
                       symbol))
           (check-type (second pair)
                  (and (not null)
                       number))
           (format-t "        ~A~%" (first pair)))
       pairs)

(defun undefine-remote-program (name number version &optional protocol)
       "
If finds NAME-NUMBER-VERSION-PROTOCOL match in *RPC-PROGRAMS*, deletes.
If finds NUMBER-VERSION match with NAME mismatch, asks first.
If deletes something, returns NAME of DELETED program, otherwise NIL."
                                           ; 
       (let ((rpc (find-rpc-program :number number :version version 
                         :name name :protocol protocol)))
            (when (and rpc (or (eql name (rpc-program-name rpc))
                               (yes-or-no-p 
               "Do you really want to remove/overwrite RPC program ~a?"
                                      (rpc-program-name rpc))))
                (setq *rpc-programs* (delete rpc *rpc-programs*))
                (rpc-program-name rpc))))

(defmacro xdr-gencode-inline (context typedef oper &rest vars)
       
       ;; Expands into code that reads/writes the rpc encoding
       ;; following context/typedef.  Note that using a NIL context is
       ;; valid here. It just means that no typedefs from other Remote
       ;; Program Definitions are available.
                                           ; XDR-CODEGEN returns a
                                           ; lambda expression
       `(,(xdr-codegen (if (and context (symbolp context))
                           (rpc-resolve-prog context)
                           context)
                 (eval typedef)
                 (eval oper))
         ,.vars))

                                           ; Utilities


(defmacro format-t (&rest args)
       "
Use in low-level code in place of (FORMAT T ...) to avoid disaster. 

The problem is that Xerox Common Lisp, when given (FORMAT <stream> ...), rebinds
*STANDARD-OUTPUT* to <stream> under the assumption that none of the
implementation of FORMAT will ever use *STANDARD-OUTPUT*.  Thus, if you try to 
write to *STANDARD-OUTPUT* in any code called by COMMON LISP I/O routines writing
to another stream, the output goes into the other stream rather than the original 
*STANDARD-OUTPUT*.  This routine is a quick fix for a lot of (FORMAT T ...) calls to 
send their output to *DEBUG-IO*, which is where the output should have gone in the
 first place.  
"
       `(format *debug-io* ,@args))

(defun find-rpc-typedef (context type)
       "Returns the type definition for TYPE defined in RPC CONTEXT
(CONTEXT may be a name or RPC-PROGRAM structure) if any, or else
returns NIL.
"
       (let ((prgstr (etypecase context
                         (symbol (find-rpc-program :name context))
                         (rpc-program context))))
            (second (assoc type (rpc-program-types prgstr)))))

(defun find-rpc-typename (context type)
       "
Returns TYPE, if TYPE defined in RPC CONTEXT
(CONTEXT may be a name or RPC-PROGRAM structure) if any, or else
returns NIL.
"
       (let ((prgstr (etypecase context
                         (symbol (find-rpc-program :name context))
                         (rpc-program context))))
            (first (assoc type (rpc-program-types prgstr)))))

(defun find-xdr-constant (context constant)
       "Find (and return) the constant definition for symbol CONSTANT among the constants for RPC-PROGRAM structure CONTEXT. "
       (check-type constant symbol)
       (second (assoc constant (rpc-program-constants context))))

                                           ; User entries to data
                                           ; structures


(defun list-remote-programs nil 

       ;; Return list of (name number version protocol)  for each
       ;; defined remote program.
       (mapcar #'(lambda (r)
                        (list (rpc-program-name r)
                              (rpc-program-number r)
                              (rpc-program-version r)
                              (rpc-program-protocol r)))
              *rpc-programs*))

(defun find-rpc-program (&key name number version protocol)
       "Returns the RPC-PROGRAM struct for the given identifiers from among all the remote programs defined.  VERSION defaults to the highest existing version."
       (cond
        (name                              ; Name uniquely identifies
                                           ; it, so if we find one,
                                           ; just check that version,
                                           ; etc match
              (dolist (pgm *rpc-programs*)
                  (when (eq name (rpc-program-name pgm))
                      (return (and (or (null number)
                                       (eql number (rpc-program-number
                                                    pgm)))
                                   (or (null version)
                                       (eql version (
                                                    rpc-program-version
                                                     pgm)))
                                   (or (null protocol)
                                       (eq protocol (
                                                   rpc-program-protocol
                                                     pgm)))
                                   pgm)))))
        (number (let ((bestprotocol (or protocol 'udp))
                      firstonefound)
                                           ; If PROTOCOL not given, we
                                           ; prefer UDP.
                     (dolist (pgm *rpc-programs* firstonefound)
                                           ; Programs are sorted by
                                           ; highest version first, so
                                           ; if VERSION is omitted we
                                           ; can just take the first
                                           ; one
                         (when (and (eql number (rpc-program-number
                                                 pgm))
                                    (or (null version)
                                        (eql version (
                                                    rpc-program-version
                                                      pgm))))
                             (cond ((eq bestprotocol (
                                                   rpc-program-protocol
                                                      pgm))
                                           ; Protocol match, or a UDP
                                           ; program when caller
                                           ; didn't say
                                    (return pgm))
                                   ((and (null protocol)
                                         (null firstonefound))
                                           ; Remember this one in case
                                           ; we don't find a UDP
                                           ; version
                                    (setq firstonefound pgm)))))))
        (t (error "Invalid RPC Program Specifier: ~@[ Name: ~a~]~@[ Number: ~a~]~@[ Version: ~a~]~@[ Protocol: ~a~]"
                  name number version protocol))))

(defun find-rpc-procedure (program procid)
       "Finds (and returns) RPC-PROCEDURE structure specified by PROCID from among the procedures of PROGRAM, an rpc-program object. PROCID may be either an integer or a symbol."
       (let ((procs (if (consp program)
                                           ; Backward compatibility
                                           ; with old def
                        program
                        (rpc-program-procedures program))))
            (ctypecase procid (integer
                               (dolist (p procs)
                                   (when (eql procid (
                                                  rpc-procedure-procnum
                                                      p))
                                         (return p))))
                   (symbol (dolist (p procs)
                               (when (eq procid (rpc-procedure-name
                                                 p))
                                     (return p)))))))

(defun find-rpc-host (destination)
       "Returns an IP host address (a number) for specified DESTINATION, or NIL if not known destination."
       (etypecase destination
           (number destination)
           ((or symbol string) (if *use-os-networking*
                                   (os-resolve-host destination)
                                   (il:iphostaddress destination)))))

                                           ; RPC Streams


(defun open-rpcstream (protocol destaddr destsocket &optional 
                             credentials timeout-handler)
       "Create and return a new RPC-STREAM."
       (let ((handler (cdr (assoc protocol *rpc-protocol-types*)))
             stream)
            (unless handler (error "Not a recognized RPC protocol: ~S"
                                   protocol))
            (setq stream (funcall handler destaddr destsocket))
            (setf (rpc-stream-protocol stream)
                  protocol)
            (setf (rpc-stream-monitorlock stream)
                  (il:create.monitorlock "RPC"))
            (setf (rpc-stream-timeout-handler stream)
                  timeout-handler)
            (setf (rpc-stream-credentials stream)
                  credentials)
            stream))

(defun rpc-create-some-udp-stream (destaddr destsocket)
       (if *use-os-networking*
           (create-os-udp-stream destaddr destsocket)
           (rpc-create-udp-stream destaddr destsocket)))

(defun close-rpcstream (rpcstream)
       "Deallocate an RPC Stream. Tries to cleanup after itself."
       (rpc-call-method close rpcstream))

(defglobalparameter
 *string-rpc-methods*
 (make-rpc-methods :protocol 'string :initialize
        #'(lambda (rpcstream)
                 (let ((s (rpc-stream-outstream rpcstream)))
                                           ; Erase the core stream.
                      (file-position s 0)
                      (il:setfileinfo s 'il:length 0)))
        :putbyte
        #'(lambda (rpcstream value)
                 (write-byte value (rpc-stream-outstream rpcstream)))
        :putcell
        #'(lambda (rpcstream value)
                 (let ((outstream (rpc-stream-outstream rpcstream)))
                      (write-byte (ldb (byte 8 24)
                                       value)
                             outstream)
                      (write-byte (ldb (byte 8 16)
                                       value)
                             outstream)
                      (write-byte (ldb (byte 8 8)
                                       value)
                             outstream)
                      (write-byte (ldb (byte 8 0)
                                       value)
                             outstream)))
        :putrawbytes
        #'(lambda (rpcstream base offset nbytes)
                 (il:\\bouts (rpc-stream-outstream rpcstream)
                        base offset nbytes))
        :zerobytes
        #'(lambda (rpcstream nbytes)
                 (let ((stream (rpc-stream-outstream rpcstream)))
                      (dotimes (i nbytes)
                          (il:\\bout stream 0))))
        :getbyte
        #'(lambda (rpcstream)
                 
                 ;; These input functions assume user has stored some
                 ;; sort of stream in the INSTREAM
                 (read-byte (rpc-stream-instream rpcstream)))
        :getcell
        #'(lambda (rpcstream)
                 (let ((instream (rpc-stream-instream rpcstream)))
                      (integer-from-bytes (il:bin instream)
                             (il:bin instream)
                             (il:bin instream)
                             (il:bin instream))))
        :getunsigned
        #'(lambda (rpcstream)
                 (let ((instream (rpc-stream-instream rpcstream)))
                      (unsigned-from-bytes (il:bin instream)
                             (il:bin instream)
                             (il:bin instream)
                             (il:bin instream))))
        :getrawbytes
        #'(lambda (rpcstream base offset nbytes)
                 (il:\\bins (rpc-stream-instream rpcstream)
                        base offset nbytes))
        :skipbytes
        #'(lambda (rpcstream nbytes)
                 (let ((instream (rpc-stream-instream rpcstream)))
                      (dotimes (i nbytes)
                          (il:bin instream))))
        :close
        #'identity))

(defun create-string-rpc-stream (&optional instring)
       "Create RPC STREAM that writes data to a string, retrievable as (RPC-GET-STRING-RESULT stream).  If you supply INSTRING, or store a string input stream in the INSTREAM slot, it will also read."
       (make-rpc-stream :methods *string-rpc-methods* :instream
              (and instring (make-string-input-stream instring))
              :outstream
              (open "{nodircore}" :direction :io)))

(defun rpc-get-string-result (rpc-stream)
       
       ;; For a string RPC stream, retrieve the result string
       
       ;; We might have wanted to use MAKE-STRING-OUTPUT-STREAM and
       ;; then retrieved it via (get-output-stream-string
       ;; (rpc-stream-outstream rpc-stream)), but in this
       ;; implementation, you can't write such strings with 255's in
       ;; them, since there is no such character.  So instead we use a
       ;; core stream and create the string at the very end.
       (let* ((s (rpc-stream-outstream rpc-stream))
              (nbytes (file-position s))
              (str (make-string nbytes)))
             (file-position s 0)
             (il:\\bins s (vector-base str)
                    0 nbytes)
             str))

                                           ; Debugging


(defglobalparameter
 *tty-rpc-methods*
 (make-rpc-methods
  :putcell
  #'(lambda (rpcstream value)
           (format (rpc-stream-outstream rpcstream)
                  "~D,~D,~D,~D~%"
                  (ldb (byte 8 24)
                       value)
                  (ldb (byte 8 16)
                       value)
                  (ldb (byte 8 8)
                       value)
                  (ldb (byte 8 0)
                       value)))
  :putrawbytes
  #'(lambda (rpcstream base offset nbytes)
           (let ((out (rpc-stream-outstream rpcstream)))
                (dotimes (i nbytes)
                    (format out "~D," (il:\\getbasebyte base
                                             (+ offset i))))
                (terpri out)))
  :putbyte
  #'(lambda (rpcstream val)
           (format (rpc-stream-outstream rpcstream)
                  "~D~%" val))
  :zerobytes
  #'(lambda (rpcstream nbytes)
           (format (rpc-stream-outstream rpcstream)
                  "~D*0~%" nbytes))))

(defun create-tty-rpc-stream nil 
       "For debugging using the TTY as the output device."
       (make-rpc-stream :methods *tty-rpc-methods* :outstream 
              *standard-output*))

                                           ; Bignum support


(defun putbase-bignum (base value)
       
       ;; Store the 32-bit bignum VALUE at BASE
       (unless (typep value 'bignum)
           (error 'type-mismatch :expected-type 'integer :name value 
                  :value value))
       (destructuring-bind (lo mid hi . rest)
              (il:\\getbaseptr (il:\\dtest value 'bignum)
                     0)
              (when (or (null hi)
                        (> hi 15)
                        (< hi 0)
                        (not (null rest)))
                    (error "Unsigned value ~S exceeds 32 bits" value))
              (il:\\putbase base 1 (+ lo (il:llsh (logand mid 3)
                                                14)))
              (il:\\putbase base 0 (+ (il:llsh hi 12)
                                      (il:lrsh mid 2)))))

(defun bignum-make-number (hiword loword)
       
       ;; Create the number HIWORD*2^16 + LOWORD, where HIWORD is
       ;; greater than 2^15-1 and hence will not make a FIXP.
       
       ;; Representation of BIGNUMs in Xerox Lisp: pointer to a list
       ;; of 14-bit integers, least significant first.
       (let ((n (il:ncreate 'bignum)))
            (il:\\rplptr n 0 (list (logand loword 16383)
                                   (+ (il:lrsh loword 14)
                                      (il:llsh (logand hiword 4095)
                                             2))
                                   (il:lrsh hiword 12)))
            n))

                                           ; Remote Procedure Call


(defun remote-procedure-call
    (destination program procid arglist &rest keys &key protocol version 
		 remotesocket credentials timeout-handler dynamic-prognum 
		 dynamic-version errorflg msec-until-timeout msec-between-tries
		 retry-until-success note-address &allow-other-keys)
  "This is the high-level way of making a remote procedure call (PERFORM-RPC is the low-level
way).

REMOTE-PROCEDURE-CALL resolves all the arguments, creates a new RPC-STREAM, makes the call, optionally closes the RPC-STREAM, and returns the results of the call.

The resolution of arguments is designed such that all arguments may be either
unresolved (e.g., a remote host name), or already resolved (e.g., an IP address).
"
  (let (rpcstream)
    (unwind-protect
	 (let* ((program (rpc-resolve-prog program version protocol))
		(procedure (rpc-resolve-proc
			    (cond (dynamic-prognum (setf program
							 (
							  copy-rpc-program
							  program))
						   (setf (rpc-program-number
							  program)
							 dynamic-prognum)
						   (setf (rpc-program-version
							  program)
							 (or dynamic-version 1))
						   program)
				  (t program))
			    procid)))
	   (setq rpcstream (open-rpcstream (rpc-program-protocol
					    program)
					   destination
					   (or remotesocket program)
					   credentials timeout-handler))
	   (apply 'perform-rpc nil nil program procedure rpcstream
		  arglist credentials keys))
      (when rpcstream
					; Discard the stream now
	(close-rpcstream rpcstream)))))

(defun
 call-via-portmapper
 (destination program procedure arglist &optional credentials)
 
 ;; Do a remote procedure call without knowing the port to use.  The
 ;; standard use for this is broadcast for servers (i.e., DESTINATION
 ;; = 0 + some net).  Returns 3 values: the results of the procedure
 ;; call, the socket used, and the address of the server that
 ;; answered.
 (let* ((*program* (if (typep program 'rpc-program)
                       program
                       (rpc-resolve-prog program)))
        (*procedure* (if (typep procedure 'rpc-procedure)
                         procedure
                         (rpc-resolve-proc *program* procedure)))
        (rettypes (rpc-procedure-resulttypes *procedure*))
        (rpc-stream (create-string-rpc-stream)))
       (multiple-value-bind
        (results server-address)
        (let (rpcstream)
             (unwind-protect
                 (let* ((pmapperprog (rpc-resolve-prog 'portmapper))
                        (pmapperproc (rpc-resolve-proc pmapperprog
                                            'indirect)))
                       (setq rpcstream (open-rpcstream (
                                                   rpc-program-protocol
                                                        pmapperprog)
                                              destination 
                                              *portmapper-socket*))
                       (perform-rpc nil nil pmapperprog pmapperproc 
                              rpcstream
                              (list (rpc-program-number *program*)
                                    (rpc-program-version *program*)
                                    (rpc-procedure-procnum *procedure*)
                                    (progn ; Args to INDIRECT are the
                                           ; prog#, version, proc#,
                                           ; and a string encoding the
                                           ; arguments as if we had
                                           ; called it directly.
                                           (encode-rpc-args rpc-stream
                                                  arglist *procedure*)
                                           (rpc-get-string-result
                                            rpc-stream)))
                              credentials :note-address t))
                 (when rpcstream
                                           ; Discard the stream now
                     (close-rpcstream rpcstream))))
        
        ;; RESULTS = (address socket string), where string is an
        ;; encoding of the results of the procedure call.  Address was
        ;; consed on the front by virtue of the :NOTE-ADDRESS key.
        (setf (rpc-stream-instream rpc-stream)
              (make-string-input-stream (third results)))
        (values (decode-rpc-args rpc-stream rettypes)
               (second results)
               (first results)))))

(defun setup-rpc (destination program procid &optional destsocket 
                        version dynamic-prognum dynamic-version 
                        protocol)
       "
Resolves arguments to REMOTE-PROCEDURE-CALL. Takes arguments in more or less
any reasonable form and returns multiple values (destination-address, socket-number,
RPC-PROGRAM struct, RPC-PROCEDURE struct).
 
See individual RPC-RESOLVE-* programs for details on what inputs are acceptable.
"
       (let* ((destaddr (rpc-resolve-host destination))
              (rprog (rpc-resolve-prog program version protocol))
              (rproc (rpc-resolve-proc (cond (dynamic-prognum
                                              (setf rprog (
                                                       copy-rpc-program
                                                           rprog))
                                              (setf (rpc-program-number
                                                     rprog)
                                                    dynamic-prognum)
                                              (setf (
                                                    rpc-program-version
                                                     rprog)
                                                    dynamic-version)
                                              rprog)
                                             (t rprog))
                            procid))
              (socket (or destsocket (rpc-find-socket destaddr rprog
                                            (rpc-program-protocol
                                             rprog)))))
             (values destaddr socket rprog rproc)))

(defun perform-rpc (destaddr destsocket *program* *procedure* *stream*
                          arglist credentials &key (errorflg t)
                          ((:msec-until-timeout *msec-until-timeout*)
                           *msec-until-timeout*)
                          ((:msec-between-tries *msec-between-tries*)
                           *msec-between-tries*)
                          note-address &allow-other-keys)
       "The low-level remote procedure call function.  *STREAM* must be an rpc stream."
       (unless (rpc-program-p *program*)
           (setq *program* (rpc-resolve-prog *program*)))
       (unless (rpc-procedure-p *procedure*)
           (setq *procedure* (rpc-resolve-proc *program* *procedure*)))
       (il:with.monitor (rpc-stream-monitorlock *stream*)
           (macrolet ((putn (n)
                                           ; A small optimization,
                                           ; since we call this so
                                           ; much
                            `(funcall putfn *stream* ,n)))
                  (let ((putfn (rpc-method putcell *stream*))
                        (xid (create-xid))
                        result)
                       (reinitialize-rpcstream *stream* destaddr 
                              destsocket)
                       (putn xid)
                       (putn *rpc-msg-call*)
                       (putn *rpc-version*)
                       (putn (rpc-program-number *program*))
                       (putn (rpc-program-version *program*))
                       (putn (rpc-procedure-procnum *procedure*))
                       (encode-authentication *stream*
                              (or credentials (rpc-stream-credentials
                                               *stream*)))
                       (encode-authentication *stream* 
                              *null-authentication*)
                       (encode-rpc-args *stream* arglist *procedure*)
                       (when *debug*
                           (format *debug-io* "~&RPC[~A]: ~A.~A"
                                  (ipaddress-to-string destaddr)
                                  (rpc-program-name *program*)
                                  (rpc-procedure-name *procedure*))
                           (let ((*print-length* 5)
                                 (*print-level* 2)
                                 (sepr "["))
                                (dolist (a arglist)
                                    (princ sepr *debug-io*)
                                    (prin1 a *debug-io*)
                                    (setq sepr ", ")))
                           (princ "]" *debug-io*))
                       (cond ((eq (setq result
                                        (rpc-call-method exchange 
                                               *stream* errorflg xid))
                                  t)
                                           ; Got a reply
                              (setq result (parse-rpc-reply
                                            *stream*
                                            (rpc-procedure-resulttypes
                                             *procedure*)
                                            errorflg))
                              (when *debug* (format *debug-io* 
                                                   " => ~S~%" result))
                              (when note-address
                                  
                                  ;; Hack for call-via-portmapper
                                  (push (il:|fetch| il:ipsourceaddress
                                               il:|of| (
                                                    rpc-stream-instream
                                                        *stream*))
                                        result))))
                       result))))

(defun rpc-resolve-host (destination)
       "Takes an IPADDRESS, symbol, or string and tries to find an IPADDRESS for a remote host.  Signals an error if it cannot resolve the host."
       (or (find-rpc-host destination)
           (error "Could not find an IP address for destination ~A" 
                  destination)))

(defun rpc-resolve-prog (program &optional version protocol)
       "Takes an RPC-PROGRAM, a number, a symbol, or a string along with an optional VERSION and PROTOCOL and tries to find the matching RPC-PROGRAM.
Signals an error if it cannot find the intended program."
       (or (typecase program
               (rpc-program program)
               (symbol (find-rpc-program :name program :version version
                              :protocol protocol))
               (number (find-rpc-program :number program :version 
                              version :protocol protocol))
               (string (find-rpc-program :name (intern program)
                              :version version :protocol protocol)))
           (error "Could not find definition for program ~A~@[, version ~D~]~@[, protocol ~a~]."
                  program version protocol)))

(defun rpc-resolve-proc (program procid)
       "Given an RPC-PROGRAM struct PROGRAM, tries to find and return an RPC-PROCEDURE in RPROG specified by a number, string,  symbol, or RPC-PROCEDURE.

Signals an error if it cannot find the intended rpc-procedure."
       (cond ((typep procid 'rpc-procedure)
              procid)
             ((find-rpc-procedure program procid))
             (t (error 
           "Could not find definition for procedure ~a of program ~a~%"
                       procid (rpc-program-name program)))))

(defun
 rpc-find-socket
 (destaddr program protocol &optional errorflg)
 "Tries to find and return a remote socket number.

(1) Looks in *RPC-WELL-KNOWN-SOCKETS*,
(2) Looks in *RPC-SOCKET-CACHE*, but only if *RPC-OK-TO-CACHE*,
(3) Requests socket number via remote procedure call to Portmapper on remote machine. If found and *RPC-OK-TO-CACHE*, caches the new socket number on *RPC-SOCKET-CACHE*.
(4) If all the above have failed, signals an error."
 (let
  ((prognum (rpc-program-number program))
   (progvers (rpc-program-version program))
   (debug (and (numberp *debug*)
               (> *debug* 1)))
   skt error)
  (cond
   ((setq skt (find-cached-socket '* prognum progvers protocol 
                     *rpc-well-known-sockets*))
    (when debug
        (format-t "~&Using well-known socket ~D for program ~A~%" skt
               (rpc-program-name program)))
    skt)
   ((and *rpc-ok-to-cache* (setq skt (find-cached-socket destaddr 
                                            prognum progvers protocol 
                                            *rpc-socket-cache*)))
    (when debug
        (format-t "~&Using cached socket ~D for program ~A~%" skt
               (rpc-program-name program)))
    skt)
   ((setq
     error
     (progn
      (when debug
          (format-t "~&Looking up socket for program ~a on ~a..."
                 (rpc-program-name program)
                 (ipaddress-to-string destaddr)))
      (cond
       ((null (setq skt
                    (remote-procedure-call
                     destaddr
                     'portmapper
                     'lookup
                     `(,prognum ,progvers
                             ,(or (cdr (assoc protocol *rpc-protocols*)
                                       )
                                  (error "Unknown protocol ~S" protocol
                                         ))
                             0)
                     :remotesocket *portmapper-socket* :errorflg 
                     errorflg)))
                                           ; No reply?  Or ERRORFLG is
                                           ; :NOERRORS
        '("No reply from Portmapper"))
       ((eq (first skt)
            'error)
                                           ; Error return 
        (cdr skt))
       ((<= (setq skt (first skt))
            0)
                                           ; This is really an error
                                           ; reply
        `(program-unavailable ,program "per portmapper")))))
    (rpc-signal-error errorflg error))
   (t (when debug (format-t "found ~D.~%" skt))
      (when *rpc-ok-to-cache* (cache-socket program destaddr skt))
      skt))))

(defun encode-rpc-args (stream arglist *procedure*)
       "Takes a list of arguments and the corresponding XDR procedure definition and converts the arguments into XDR, writing them into the RPC-STREAM."
       (let ((encodefn (rpc-procedure-argtypes *procedure*)))
            (cond ((null encodefn)
                   (unless (null arglist)
                                           ; Expected no args, got some
                       (rpc-argument-error arglist 0)))
                  ((and (consp encodefn)
                        (not (eq (car encodefn)
                                 'lambda)))
                                           ; Old style, one procedure
                                           ; per arg
                   (do ((xdr-fns encodefn (rest xdr-fns))
                        (args arglist (rest args)))
                       ((or (null args)
                            (null xdr-fns))
                        (if (or xdr-fns args)
                            (rpc-argument-error arglist (length 
                                                               encodefn
                                                               ))))
                     (funcall (first xdr-fns)
                            stream
                            (first args))))
                  (t                       ; New style: call a single
                                           ; function
                     (funcall encodefn stream arglist)))))

(defun
 parse-rpc-reply
 (rpcstream rettypes &optional errorflg)
 "Parses a reply message.  If all goes well, returns a list of the values returned (or T if RETTYPES is NIL).

If RPC was REJECTED, or ACCEPTED but with an ACCEPT-STAT other than SUCCESS, then (Following Courier) the response depends on the value of ERRORFLG:
	If ERRORFLG = 'NOERROR, then returns NIL
	If ERRORFLG = 'RETURNERRORS, then returns a list of the form
		(ERROR reply-stat accept-or-reject-stat otherinfo)
	If ERRORFLG = anything else, signals Lisp error."
 
 ;; Most of what goes on here could be one big XDR-GENCODE-INLINE of
 ;; the reply structure, except that in the normal success case, the
 ;; inside is procedure-dependent.  So this is a hand-coding of the
 ;; reply structure.
 (macrolet
  ((getunsigned nil                        ; Small optimization
          '(funcall ufn rpcstream)))
  (let* ((ufn (rpc-method getunsigned rpcstream))
         msgtype reply-stat)
        (cond
         ((not (eql (setq msgtype (getunsigned))
                    *rpc-msg-reply*))
          (rpc-signal-error errorflg `(not-a-reply ,msgtype)))
         ((eql (setq reply-stat (getunsigned))
               *rpc-reply-accepted*)
          (let ((verf (decode-authentication rpcstream))
                (accept-stat (getunsigned)))
               (cond ((not (eql accept-stat *rpc-accept-success*))
                      (rpc-signal-error
                       errorflg
                       (cons (setq accept-stat
                                   (or (cdr (assoc accept-stat 
                                                   *rpc-accept-stats*))
                                       accept-stat))
                             (case accept-stat
                                 (program-mismatch 
                                           ; Includes high and low
                                           ; accepted versions
                                    (list (getunsigned)
                                          (getunsigned)))))))
                     (t (decode-rpc-args rpcstream rettypes)))))
         ((eql reply-stat *rpc-reply-rejected*)
          (rpc-signal-error
           errorflg
           (xdr-gencode-inline nil
                  '(:union (:enumeration :noerrors (
                                                   rpc-version-mismatch
                                                    0)
                                  (authentication 1))
                          (rpc-version-mismatch (:list :unsigned 
                                                       :unsigned))
                          (authentication (:enumeration :noerrors
                                                 (bad-credentials
                                                  1)
                                                 (rejected-credentials
                                                  2)
                                                 (bad-verifier 3)
                                                 (rejected-verifier
                                                  4)
                                                 (too-weak 5))))
                  'read rpcstream)))
         (t (rpc-signal-error errorflg `(illegal-reply-type
                                         ,reply-stat)))))))

(defmacro decode-rpc-args (rpcstream rettypes)
       
       ;; This encapsulates how we read procedure-dependent reply data
       ;; from RPCSTREAM
       `(cond ((null ,rettypes)
                                           ; Returns nothing.  We
                                           ; return T to distinguish
                                           ; this from a :NOERROR NIL
                                           ; return
               t)
              ((and (consp ,rettypes)
                    (not (eq (car ,rettypes)
                             'lambda)))
                                           ; Old style: Call one
                                           ; function per result type
               (mapcar #'(lambda (fn)
                                (funcall fn ,rpcstream))
                      ,rettypes))
              (t                           ; New style: Call a single
                                           ; function to read all the
                                           ; results.
                 (funcall ,rettypes ,rpcstream))))

(defun rpc-handle-timeout (stream timeout-count errorflg)
       
       ;; Called when an RPC call times out on STREAM.  TIMEOUT-COUNT
       ;; is number from 1... indicating how many times we have timed
       ;; out on this call.  ERRORFLG is the arg to perform-rpc.  If
       ;; this procedure returns, it is either a value to return from
       ;; the call, or :CONTINUE to keep trying.
       (let ((handler (rpc-stream-timeout-handler stream)))
            (case handler
                ((nil :noerrors :returnerrors) (rpc-signal-error
                                                (or handler errorflg)
                                                '(rpc-timeout)))
                (otherwise 
                                           ; Call the handler and
                                           ; return what it returns,
                                           ; or keep trying if it says
                                           ; :CONTINUE
                   (if (eq (setq handler (funcall handler stream 
                                                timeout-count))
                           t)
                                           ; Synonym for :continue --
                                           ; T is "successful return"
                                           ; from exchange, so don't
                                           ; want that confusion
                       :continue
                       handler)))))

(defun rpc-signal-error (errorflg errorform)
       (case errorflg
           (:noerrors nil)
           (:returnerrors (cons 'error errorform))
           (otherwise 
                                           ; Signal the appropriate
                                           ; kind of error.
              (case (car errorform)
                  (rpc-timeout (error 'rpc-timeout))
                  (otherwise (error 'rpc-error-reply :type (car 
                                                              errorform
                                                                )
                                    :args
                                    (cdr errorform)))))))

(defun rpc-argument-error (actuals expected#)
       
       ;; Called when rpc call got wrong number of args
       (error 
  "Wrong number of arguments to procedure ~A.~A:  Expected ~D, got ~D."
              (if *program*
                  (rpc-program-name *program*)
                  "?")
              (if *procedure*
                  (rpc-procedure-name *procedure*)
                  "?")
              expected#
              (length actuals)))

(defun create-xid nil 
       "Returns a number to use as the ID of a given transmisssion." 

       ;; ID's are 32 bits, but we want to stick to less than 2^31 to
       ;; avoid touching bignums.  To be really tense, we could stick
       ;; to less than 2^16, which is probably safe, though at a
       ;; sustained a rate of 100 transactions per second (a wild
       ;; rate), it would take little over 10 minutes to cycle thru
       ;; them all.  At 2^26 it would take a week at that rate. 
       (if (> (incf *xid-count*)
              *xid-max*)
           (setq *xid-count* 1)
           *xid-count*))

                                           ; RPC Utility Functions


(defun find-cached-socket (destaddr prognum progvers protocol cache)
       "Looks up a given (DESTADDR, PROGNUM, PROGVERS, PROTOCOL) in the specified CACHE."
       (dolist (entry cache)
           (when (and (eql (car entry)
                           destaddr)
                      (eql (car (setq entry (cdr entry)))
                           prognum)
                      (eql (car (setq entry (cdr entry)))
                           progvers)
                      (eq (car (setq entry (cdr entry)))
                          protocol))
                                           ; Found it-fifth element is
                                           ; the socket number
               (return (cadr entry)))))

(defun cache-socket (program address skt)
       
       ;; Add this info to socket cache
       (unless (integerp address)
           (setq address (rpc-resolve-host address)))
       (check-type skt integer)
       (push `(,address ,(rpc-program-number program)
                     ,(rpc-program-version program)
                     ,(rpc-program-protocol program)
                     ,skt)
             *rpc-socket-cache*))

(defun
 clear-cache
 (&optional program address)
 
 ;; Clear cache of any info about port for PROGRAM at ADDRESS.  NIL
 ;; for either arg means "any".
 (setq
  *rpc-socket-cache*
  (cond ((or program address)
         (when program
             (setq program (rpc-resolve-prog program)))
         (when address
             (setq address (rpc-resolve-host address)))
         (delete-if
          #'(lambda (entry)
                   (and (or (null address)
                            (eql (first entry)
                                 address))
                        (or (null program)
                            (and (eq (second entry)
                                     (rpc-program-number program))
                                 (eq (third entry)
                                     (rpc-program-version program))))))
          *rpc-socket-cache*))))
 t)

(defun ipaddress-to-string (ipaddress)
       
  "Render IPADDRESS in the canonical printed representation of i.j.k.l"
       (format nil "~D.~D.~D.~D" (ldb (byte 8 24)
                                      ipaddress)
              (ldb (byte 8 16)
                   ipaddress)
              (ldb (byte 8 8)
                   ipaddress)
              (ldb (byte 8 0)
                   ipaddress)))

                                           ; Authentication


(defconstant *authentication-typedef*
       '(:struct authentication (type (:enumeration (:null 0)
                                             (:unix 1)
                                             (:short 2)))
               (string :string)))

(defconstant *null-authentication* (make-authentication :type :null 
                                          :string ""))

(defun create-unix-authentication (stamp machine-name uid gid gids)
       "Given the fields of a Unix authentication, creates an AUTHENTICATION struct with these fields encoded as a string."
       (let ((tempstream (create-string-rpc-stream)))
            (putunsigned tempstream stamp)
            (xdr-string tempstream machine-name)
            (putunsigned tempstream uid)
            (putunsigned tempstream gid)
            (xdr-gencode-inline nil '(:list-of :unsigned)
                   'write tempstream gids)
            (make-authentication :type :unix :string (
                                                  rpc-get-string-result
                                                      tempstream))))

(defun encode-authentication (rpcstream auth)
       "
Given an AUTHENTICATION struct, converts the struct to its XDR encoding and writes it to
the RPC-STREAM specified.
"
       (if (null auth)
           (setq auth *null-authentication*))
       (check-type auth authentication)
       (xdr-gencode-inline nil *authentication-typedef* 'write 
              rpcstream auth))

(defun decode-authentication (rpcstream)
       "Reads an authentication from specified RPC-STREAM and returns it as an AUTHENTICATION struct."
       
       ;; Used to do this as (xdr-gencode-inline nil
       ;; *authentication-typedef* 'read rpcstream), but that conses a
       ;; string and authentication object on every rpc reply, even
       ;; though virtually always it is the null authentication.  We
       ;; expect: authtype (integer) followed by string.
       (let ((type (getcell rpcstream))
             (len (getunsigned rpcstream)))
            (cond ((and (eql type 0)
                        (eql len 0))
                                           ; The null authentication
                   nil)
                  (t                       ; Go ahead and make one,
                                           ; and read the rest of the
                                           ; string
                     (make-authentication :type (case type
                                                    (0 :null)
                                                    (1 :unix)
                                                    (2 :short)
                                                    (otherwise nil))
                            :string
                            (xdr-read-array rpcstream len))))))

                                           ; Interface to RESTART.ETHER


(defun rpc-restart (event)
       
       ;; Called by RESTART.ETHER
       (setq *rpc-socket-cache* nil))


;; Can't parse: (il:appendvars (il:restartetherfns rpc-restart))

