;;;
;;; SK8 (Copyright 1988, 1989, 1990, 1991, 1992 by Apple Computer, Inc.)
;;; Advanced Technology Group
;;; Contributors (in alphabetical order): Adam Chipkin, Ruben Kleiman, Dave Vronay
;;; Special Thanks To:  Mark Miller and  Jim Spohrer
;;;


;;;
;;; Runs on MCL 2.0
;;;

;;;
;;; NOTICE:
;;;
;;;     This software is provided free of charge without any explicit or implicit guarantee
;;;     about its operation.  The user assumes all liabilities, whether direct or indirect,
;;;     physical, emotional or damage to any property, as a consequence of the use or during
;;;     the use of this code.  By using this code, the user acknowledges this statement
;;;     and frees Apple Computer, Inc. or any of its employees to any liabilities as stated herein.
;;;


;;;                                                                                         
;;; X  C  M  D  S
;;; Extension to Macintosh Common Lisp to support calls to HyperTalk XMCDs and XFCNs
;;;                                                                                         

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   -------------------------
;;;    I N S T R U C T I O N S
;;;   -------------------------
;;;
;;;    (1)  A.  Copy 'xcmds.lisp' (or compile it and copy 'xcmds.fasl') into one of the
;;;             folders in MCL's *module-search-path* (the folder your 'MCL' application
;;;             is in is one of these).  Now MCL will recognize 'xcmds' as a module and
;;;             you can simply 'require' the module:
;;;
;;;                  (require 'xcmds)
;;;
;;;         --OR--
;;;
;;;         B.  Just 'eval' this buffer.
;;;
;;;
;;;    (2)  Now the the two entry functions 'get-xcmd-handle' and 'do-xcmd' are interned.
;;;         NOTE:  You don't have to use 'get-xcmd-handle'; it just allows you to enhance
;;;         the performance of 'do-xcmd'.  When you give 'do-xcmd' the string name of the
;;;         XCMD or XFCN resource, it calls 'get-xcmd-handle' for you.  So if you plan to
;;;         call a certain XCMD or XFCN repeatedly, you can avoid redundant calls to
;;;         'get-xcmd-handle' by getting and remembering the handle yourself once at the
;;;         start, and then giving 'do-xcmd' the handle instead of the name each time you
;;;         call it.
;;;
;;;         Using 'do-xcmd' is straightforward: the first argument is the name of the XCMD
;;;         or XFCN (or the handle to its resource) and all subsequent arguments are string
;;;         parameters that are passed to the XCMD or XFCN.  'do-xmcd' returns the string
;;;         returned by the XCMD or XFCN.  See end of this file for examples...
;;;
;;;         Note on finding XCMDs and XFCNs: 'get-xcmd-handle' first looks for the named
;;;         XCMD or XFCN in the currently open resource files.  If it doesn't find it, the
;;;         file given in the :resource-file-path keyword argument is checked, or if no
;;;         file is given, the function first looks in a resource file named <xcmd-name>,
;;;         <xcmd-name> + '.xcmd', or <xcmd-name> + '.xfcn' in the 'ccl;' folder or the
;;;         'xcmd;' folder.  Redefine the 'xcmd;' logical directory as you wish.
;;;
;;;
;;;    (3)  Note that not all call-backs have been implemented.  The code below and these
;;;         instructions should be sufficient for you to implement any that you may need.
;;;         Note that the tracing capability will help you determine which call-backs
;;;         are being made from XCMDs.
;;;
;;;         A special note about the 'SendCardMessage' call-back: when an XCMD or XFCN
;;;         invokes this call-back, it expects the application to process the message.  The
;;;         current implementation of this call-back simply calls the 'execute-message'
;;;         function with a single argument (the string given to the call-back to process).
;;;         The default definition of 'execute-message' just prints out the message; if
;;;         your XCMD or XFCN uses this call-back, you'll probably want to redefine
;;;         the 'execute-message' function to do something meaningful.
;;;


(in-package :ccl)

(export '(do-xcmd
          get-xcmd-handle
          execute-message))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                                         ;;;
;;;                            D  E  F  I  N  I  T  I  O  N  S                              ;;;
;;;                                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Set up logical directories:
(def-logical-directory "ccl;" (truename "ccl:"))
(def-logical-directory "xcmd" "ccl;library:")

(eval-when (eval compile load)
  ;;; Define the XcmdBlock record type for communication to and from XCMDs:
  (defrecord (XCmdBlock :pointer)
    (paramCount  integer)
    (param1      handle)
    (param2      handle)
    (param3      handle)
    (param4      handle)
    (param5      handle)
    (param6      handle)
    (param7      handle)
    (param8      handle)
    (param9      handle)
    (param10     handle)
    (param11     handle)
    (param12     handle)
    (param13     handle)
    (param14     handle)
    (param15     handle)
    (param16     handle)
    (returnValue handle)
    (passFlag    boolean)
    (entryPoint  pointer)
    (request     integer)
    (result      integer)
    (inarg1      pointer) ;; These were 'longint's, but they are sometimes used as pointers;
    (inarg2      pointer) ;; making the field-types 'pointer' stops MCL from fiddling with
    (inarg3      pointer) ;; the high bits.
    (inarg4      pointer)
    (inarg5      pointer)
    (inarg6      pointer)
    (inarg7      pointer)
    (inarg8      pointer)
    (outarg1     pointer)
    (outarg2     pointer)
    (outarg3     pointer)
    (outarg4     pointer)))

(eval-when (eval compile)
  ;; Define the callback request constants:
  (defconstant $xreqSendCardMessage 1)
  (defconstant $xreqEvalExpr        2)
  (defconstant $xreqStringLength    3)
  (defconstant $xreqStringMatch     4)
  (defconstant $xreqSendHCMessage   5)
  (defconstant $xreqZeroBytes       6)
  (defconstant $xreqPasToZero       7)
  (defconstant $xreqZeroToPas       8)
  (defconstant $xreqStrToLong       9)
  (defconstant $xreqStrToNum       10)
  (defconstant $xreqStrToBool      11)
  (defconstant $xreqStrToExt       12)
  (defconstant $xreqLongToStr      13)
  (defconstant $xreqNumToStr       14)
  (defconstant $xreqNumToHex       15)
  (defconstant $xreqBoolToStr      16)
  (defconstant $xreqExtToStr       17)
  (defconstant $xreqGetGlobal      18)
  (defconstant $xreqSetGlobal      19)
  (defconstant $xreqGetFieldByName 20)
  (defconstant $xreqGetFieldByNum  21)
  (defconstant $xreqGetFieldByID   22)
  (defconstant $xreqSetFieldByName 23)
  (defconstant $xreqSetFieldByNum  24)
  (defconstant $xreqSetFieldByID   25)
  (defconstant $xreqStringEqual    26)
  (defconstant $xreqReturnToPas    27)
  (defconstant $xreqScanToReturn   28)
  (defconstant $xreqScanToZero     39)
  (defconstant $xreqSendHCEvent    41)
  
  ;; Define the callback result constants:
  (defconstant $xresSucc            0)
  (defconstant $xresFail            1)
  (defconstant $xresNotImp          2)
  
  ;; Define the operating-system error constant 'noErr':
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                                         ;;;
;;;                                  G  L  O  B  A  L  S                                    ;;;
;;;                                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defparameter *params* (make-record :XCmdBlock))       ; The XCmdBlock must be global to
                                                       ; this code since the callback
                                                       ; routine is not given its pointer

(defvar *handles* nil)         ; Handles needing eventual disposal (e.g.,
                               ; those created by the 'PasToZero' callback)

(defvar *pointers* nil)        ; Pointers needing eventual disposal

(defvar *logstream* nil)       ; A log of XCMD calls and callback activity is written
                               ; to this stream

(defvar *requests*             ; Records the symbols of the existing callback requests
  (vector nil                  ; in a vector (index = requestNumber) for fast lookup
          'x-SendCardMessage   ; by the 'CALLBACK-HANDLER' function
          'x-EvalExpr
          'x-StringLength
          'x-StringMatch
          'x-SendHCMessage
          'x-ZeroBytes
          'x-PasToZero
          'x-ZeroToPas
          'x-StrToLong
          'x-StrToNum
          'x-StrToBool
          'x-StrToExt
          'x-LongToStr
          'x-NumToStr
          'x-NumToHex
          'x-BoolToStr
          'x-ExtToStr
          'x-GetGlobal
          'x-SetGlobal
          'x-GetFieldByName
          'x-GetFieldByNum
          'x-GetFieldByID
          'x-SetFieldByName
          'x-SetFieldByNum
          'x-SetFieldByID
          'x-StringEqual
          'x-ReturnToPas
          'x-ScanToReturn
          nil
          nil
          nil
          nil
          nil
          nil
          nil
          nil
          nil
          nil
          'x-ScanToZero
          nil
          'x-SendHCEvent))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                                         ;;;
;;;                    S  U  P  P  O  R  T      F  U  N  C  T  I  O  N  S                   ;;;
;;;                                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; XCMD callback handler:
;;;
;;; The address of this function is passed, via the XcmdBlock, to the XCMD or XFCN to allow
;;; it to invoke our callback routines.  The routine indicated by the 'request' field of the
;;; XcmdBlock is invoked if it's implemented; if it's not implemented, the function indicates
;;; that in the 'result' field of the XcmdBlock. 
;;;
;;; IMPORTANT NOTE: HyperCard makes available to XCMDs all of the callback routines listed
;;; in above (in the definition of the '*REQUESTS*' variable).  In this code, however, the
;;; majority are NOT implemented -- the callback handler simply ignores requests for
;;; unimplemented routines.  I've provided the log feature so that you can see exactly
;;; which routines a given XCMD expects and only implement them as needed.  The callbacks
;;; below should serve as models for any you'll need to write yourself.  Note that all
;;; callback functions must be defined in the same package 'CALLBACK-HANDLER' is in.
;;; 
;;; For reasonably thorough descriptions of the semantics of each callback routine, see
;;; pages 69-121 of Gary Bond's book 'XCMDs FOR HYPERCARD', MIS: Press, Portland, OR, 1988.
;;;
(defpascal callback-handler ()
  (without-interrupts
   (let ((callback-fcn (aref *requests* (rref *params* :XCmdBlock.request)))
         callback-result)
     
     (if *logstream*
       (format *logstream* "  [callback]: ~A (request code ~D)"
               (subseq (string callback-fcn) 2)
               (rref *params* :XCmdBlock.request)))
     
     (setq callback-result
           (if (setq callback-fcn (fboundp callback-fcn))
             (funcall callback-fcn) ;; If the function's here, call it and return its result.
             $xresNotImp))          ;; Else tell the XCMD that the callback is not implemented.
     
     (if *logstream*
       (format *logstream* "~%")))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Used for logging.
;;;
;;; Writes the given character string to the given stream; 'thestring' must be a pointer
;;; to a Pascal-type string (at most 255 characters, first byte is string's length).
;;;
(defun write-pstring (thestream thestring)
  (let ((len (%get-byte thestring)))
    (do ((offset 1 (+ 1 offset))) ((> offset len))
      (write-char (code-char (%get-byte thestring offset)) thestream))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Used for logging.
;;;
;;; Writes the given character string to the given stream; 'thestring' must be a pointer
;;; to a C-type string (zero-terminated, no length byte).
;;;
(defun write-cstring (thestream thestring)
  (do ((offset 0 (+ 1 offset))) ((= 0 (%get-byte thestring offset)))
    (write-char (code-char (%get-byte thestring offset)) thestream)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Used by the 'equal-pstr-cstr' function.
;;;
;;; CHAR-BYTES= returns true if the ASCII characters represented by the two given integers
;;; are equal and CHAR-BYTES/= returns false if they're equal (both are case insensitive)
;;;
#|
(defmacro char-bytes= (b1 b2)
  `(char-equal (int-char (coerce ,b1 'integer)) (int-char (coerce ,b2 'integer))))
(defmacro char-bytes/= (b1 b2)
  `(char-not-equal (int-char (coerce ,b1 'integer)) (int-char (coerce ,b2 'integer))))

(defmacro char-bytes= (b1 b2)
  `(char-equal (code-char (coerce ,b1 'integer)) (code-char (coerce ,b2 'integer))))
(defmacro char-bytes/= (b1 b2)
  `(char-not-equal (code-char (coerce ,b1 'integer)) (code-char (coerce ,b2 'integer))))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Used by the 'StringMatch' callback function.
;;;
;;; Returns true if the two strings match (case insensitive).  'pstr' must be a pointer to a
;;; Pascal-type string and 'cstr' must be a pointer to a C-type string.
;;;
(defun equal-pstr-cstr (pstr cstr)
  (let* ((len (%get-byte pstr))
         (cstr1 (ccl::%str-from-ptr cstr len))
         (pstr1 (ccl::%str-from-ptr (%incf-ptr pstr 1) len)))
    (string-equal cstr1 pstr1)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'SendCardMessage' callback function:
;;;
;;; In the XCmdBlock, inArg1 points to a Pascal string containing the message.
;;;

(defun execute-message (msg)
  (format t "~%Pretending to execute following message:~%~10T~a" msg))

(defun x-SendCardMessage ()
  (let* ((p (rref *params* :XCmdBlock.inArg1)) ; pointer to Str255, the message
         msg)
    
    (push p *pointers*)
    (setq msg (%get-string p))

    (execute-message msg)
    
    (when *logstream*
      (format *logstream* "MESSAGE = '")
      (write-pstring *logstream* p)
      (format *logstream* "'")))
  
  $xresSucc)




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'StringMatch' callback function:
;;;
;;; In the XcmdBlock, inArg1 points to a Pascal-type string, the pattern, and inArg2 points
;;; to a C-type string, the target.
;;;
;;; Performs a case-insensitive search to locate the pattern within the target.  If the pattern
;;; string is found in the target, a pointer to the first character of the match is returned
;;; in outArg1, otherwise the null-pointer is returned in outArg1.
;;;
(defun x-StringMatch ()
  (let ((pattern (rref *params* :XCmdBlock.inArg1))  ; str255 to locate
        (target (rref *params* :XCmdBlock.inArg2)))  ; c-string in which to look
    (rset *params* :XCmdBlock.outArg1 (%null-ptr))   ; Default to NULL.
    (unless (and (%null-ptr-p pattern)
                 (%null-ptr-p target))
      (do ((subtarget target (%inc-ptr subtarget)))
          ((or (eql 0 (%get-byte subtarget))
               (not (%null-ptr-p subtarget))))
        (when (equal-pstr-cstr pattern subtarget)
          (rset *params* :XCmdBlock.outArg1 subtarget))))
    
    (when *logstream*
      (format *logstream* "  Substring '")
      (write-pstring *logstream* pattern)
      (if (%null-ptr-p (rref *params* :XCmdBlock.outArg1))
        (format *logstream* "' not in '")
        (format *logstream* "' in '"))
      (write-cstring *logstream* target)
      (format *logstream* "'."))
    
    $xresSucc))

    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'ZeroToPas' callback function:
;;;
;;; In the XcmdBlock, inArg1 points to a C-type string, the source, and inArg2 points
;;; to a Pascal-type string, the destination.
;;;
;;; Copies the C-string into the Pascal-string's buffer in the Pascal-string format.
;;;
(defun x-ZeroToPas ()
  (let* ((cstr (rref *params* :XCmdBlock.inArg1))    ;; c-string
         (pstr (rref *params* :XCmdBlock.inArg2))    ;; p-string
         (len 0))
    (loop
      (when (or (= (%get-byte cstr len) 0)
                (= len 255))
        (return))
      (incf len))
    (%put-byte pstr len)
    (dotimes (i len) (%put-byte pstr (%get-byte cstr i) (1+ i)))

    (when *logstream*
      ;;(write-cstring *logstream* cstr)
      (format *logstream* "  STRING AT ~A => '" cstr)
      (write-pstring *logstream* pstr)
      (format *logstream* "'"))
    
    $xresSucc))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; StringEqual callback function:
;;;
(defun x-StringEqual ()
  (let* ((pstr1 (rref *params* :XCmdBlock.inArg1))    ;; str255
         (pstr2 (rref *params* :XCmdBlock.inArg2))    ;; str255
         (result (string-equal (%get-string pstr1) (%get-string pstr2))))

    (rset *params* :XCmdBlock.outArg1 (%int-to-ptr (if result 1 0)))

    (when *logstream*
      (princ "  STRINGS = '" *logstream*)
      (write-pstring *logstream* pstr1)
      (princ "' AND '" *logstream*)
      (write-pstring *logstream* pstr2)
      (princ "'" *logstream*))

    $xresSucc))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'PasToZero' callback function:
;;;
;;; In the XcmdBlock, inArg1 points to a Pascal-type string.
;;;
;;; Allocates a handle to enough memory to hold the given string then copies the string there,
;;; terminated by a zero byte.  The C-string handle is returned in outArg1.
;;; NOTE: The XCMD is not expected to dispose of the C-string created by this callback, so
;;; the handle is remembered in the global list *handles* and is disposed of by the 'do-xcmd'
;;; function when the XCMD finishes executing.
;;;
(defun x-PasToZero ()
  (let* ((pstr (rref *params* :XCmdBlock.inArg1))   ;; str255
         (len (%get-byte pstr))
         (end (1+ len))
         (h (#_NewHandle (1+ len))))
    (unless (handlep h) (error "Out of memory"))
    (setq *handles* (cons h *handles*))
    (rset *params* :XCmdBlock.outArg1 h)
    (with-dereferenced-handles ((p h))
      (setq p (%get-ptr h))
      (do ((offset 1 (1+ offset)))
          ((eql offset end))
        (%put-byte p (%get-byte pstr offset) (1- offset)))
      (%put-byte p 0 len)
      
      (when *logstream*
        (format *logstream* "  STRING = '")
        (write-cstring *logstream* p)
        (format *logstream* "'")))
    
    $xresSucc))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'ZeroBytes' callback function:
;;;
;;; In the XcmdBlock, inArg1 is a pointer and inArg2 gives the (long)
;;; number of zero-bytes to write.
;;;
;;; Writes the specified number of zero-bytes into memory starting at the
;;; memory location given by the pointer.
;;;
(defun x-zeroBytes ()
  (let ((pointer (rref *params* :XCmdBlock.inArg1))                  ; pointer
        (numZeros (%ptr-to-int (rref *params* :XCmdBlock.inArg2))))  ; number of zeros to write
    (dotimes (i numZeros)
      (%put-byte pointer 0 i))
    
    (when *logstream*
      (format *logstream* "  ZEROED ~a BYTES AT ~A" numZeros pointer))
    
    $xresSucc))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'ScanToZero' callback function:
;;;
;;; In the XcmdBlock, inArg1 is a handle to a C-type string.
;;;
;;; Updates the handle's master pointer to point to the next zero byte in the string.
;;;
(defun x-scanToZero ()
  (let ((handle (rref *params* :XCmdBlock.inArg1)) ; handle to c-string
        (offset 0)
        cstring
        the-pointer)
    (with-dereferenced-handles ((pointer handle))
      (setq pointer (%get-ptr handle))
      (setq the-pointer pointer)
      (setq cstring (ccl::%get-cstring pointer))
      (loop
        (when (zerop (%get-byte pointer offset))
          (%put-ptr handle (%inc-ptr pointer offset))
          (return))
        (incf offset)))

    (when *logstream* (format *logstream* "  SCANNED OVER '~a' (~a BYTES) AT ~A"
                              cstring offset the-pointer))
    $xresSucc))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'StringToNum' callback function:
;;;
;;; In the XcmdBlock, inArg1 points to a Pascal-type string.
;;;
;;; Returns a signed long-integer in outArg1 equal to the number represented by the string.
;;;
(defun x-StringToNum ()
  (let* ((str (rref *params* :XCmdBlock.inArg1))   ;; str255
         (num (parse-integer (%get-string str))))

    (if *logstream*
      (format *logstream* "  NUMBER = ~D" num))

    (rset *params* :XCmdBlock.outArg1 num)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The 'LongToStr' callback function:
;;;
;;; In the XcmdBlock, inArg1 points to a longint.
;;;
;;; Returns a Pascal Str31 in outArg1 equal to the number represented in inArg1.
;;;
(defun x-LongToStr ()
  (let* ((num (format nil "~s" (%ptr-to-int (rref *params* :XCmdBlock.inArg1))))   ;; longint as string
         (len (length num))
         (h (#_NewHandle 32)))
    (unless (handlep h) (error "Out of memory"))
    (push h *handles*)
    (rset *params* :XCmdBlock.outArg1 h)
    (with-dereferenced-handles ((p h))
      (setq p (%get-ptr h))
      (%put-byte p len)
      (do ((offset 0 (1+ offset)))
          ((= offset len))
        (%put-byte p (aref num offset) (1+ offset)))
      
      (when *logstream*
        (format *logstream* "  STRING = '")
        (write-pstring *logstream* (%ptr-to-int p))
        (format *logstream* "'"))))
  
  $xresSucc)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Used by the 'do-xcmd' function.
;;;
;;; Given a lisp string, allocates a handle to enough memory to hold the string
;;; then copies the string there, terminated by a zero byte
;;;
(defun make-cstr-handle (lisp-string)
  (let* ((len (length lisp-string))
         (cstr-handle (#_NewHandle (+ 1 len))))
    (unless (handlep cstr-handle)
      (error "Out of memory"))
    (with-dereferenced-handles ((cstr-ptr cstr-handle))
      (setq cstr-ptr (%get-ptr cstr-handle))
      (%put-byte cstr-ptr 0 len)
      (dotimes (i len)
        (%put-byte cstr-ptr (char-int (aref lisp-string i)) i)))
    cstr-handle))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                                         ;;;
;;;                       E  N  T  R  Y     F  U  N  C  T  I  O  N  S                       ;;;
;;;                                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Get-XCMD-Handle:
;;;
;;; Returns a handle to the named 'XCMD' or 'XFCN' resource.  If that resource
;;; is not in the currently open resource file, the function tries to find it
;;; in the file given by 'resource-file-path'.  If 'resource-file-path' is nil,
;;; the function tries to find it in a file named <xcmd-name>, <xcmd-name> + '.xcmd',
;;; or <xcmd-name> + '.xfcn' in the 'ccl;' folder or the 'xcmd;' folder.
;;;
;;; By default, the function first tries to get an XCMD with the given name
;;; and, upon failure, tries to get an XFCN with that name.  To force it to
;;; try in the reverse order, supply T for the optional 'XFCN-P' argument.
;;;
(defun get-xcmd-handle (xcmd-name &key resource-file-path xfcn-p)
  (when *logstream*
    (format *logstream* "~%---> Now fetching XCMD/XFCN named '~A'~%" xcmd-name))
  (with-pstrs ((xcmd-name-pstr xcmd-name))
    (let* ((old-currentResFile (#_CurResFile))
           (xcmd-handle (if xfcn-p
                          (#_GetNamedResource "XFCN" xcmd-name-pstr)
                          (#_GetNamedResource "XCMD" xcmd-name-pstr)))
           (err (#_ResError)))
      (unwind-protect
        (progn
          (unless (eql err #$noErr) ; try it the other way
            (setq xcmd-handle (if xfcn-p
                                (#_GetNamedResource "XCMD" xcmd-name-pstr)
                                (#_GetNamedResource "XFCN" xcmd-name-pstr)))
            (setq err (#_ResError)))
          (unless (eql err #$noErr)
            (when (null resource-file-path)
              (setq resource-file-path
                    (or (probe-file (format nil "ccl;~A" xcmd-name))
                        (probe-file (format nil "ccl;~A.xcmd" xcmd-name))
                        (probe-file (format nil "ccl;~A.xfcn" xcmd-name))
                        (probe-file (format nil "xcmd;~A" xcmd-name))
                        (probe-file (format nil "xcmd;~A.xcmd" xcmd-name))
                        (probe-file (format nil "xcmd;~A.xfcn" xcmd-name)))))
            (cond ((null resource-file-path)
                   (error "Can't find XCMD/XFCN resource '~A'" xcmd-name))
                  ((= -1 (with-pstrs ((path-str (namestring resource-file-path))) ; try to open it
                           (#_OpenResFile path-str)))
                   (#_UseResFile old-currentResFile)
                   (error "Bad resource file '~A'" resource-file-path))
                  (t
                   (setq xcmd-handle (#_GetNamedResource "XCMD" xcmd-name-pstr))
                   (unless (eql (#_ResError) #$noErr)
                     (setq xcmd-handle (#_GetNamedResource "XFCN" xcmd-name-pstr))
                     (unless (eql (#_ResError) #$noErr)
                       (error "Can't find XCMD/XFCN resource '~A' in file '~A'" xcmd-name
                              (namestring resource-file-path))))))
            (when (%null-ptr-p xcmd-handle)
              (error "Bad XCMD/XFCN resource '~A'" xcmd-name))))
        
        (#_UseResFile old-currentResFile)
        (when (handlep xcmd-handle) (#_HNoPurge xcmd-handle)))
      (when *logstream*
        (format *logstream* "---> Resolved XCMD/XFCN named '~A' into ~S~%"
                xcmd-name xcmd-handle))
      xcmd-handle)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Do-XCMD:
;;;
;;; Given the string name or the handle of an XCMD or XFCN resource and up
;;; to 16 string parameters, this invokes the XCMD/XFCN, passing it the given
;;; parameters.  It returns the string result of the call (for an XCMD, the
;;; error string; for an XFCN, the function value).
;;;
(defun do-xcmd (xcmd-handle-or-name &rest paramList)
  (unless (or (handlep xcmd-handle-or-name)
              (and (stringp xcmd-handle-or-name)
                   (setq xcmd-handle-or-name (get-xcmd-handle xcmd-handle-or-name))))
    (error "Invalid XCMD/XFCN"))
  
  (when *logstream*
    (format *logstream* "~%---> Now calling XCMD/XFCN at ~D~%" xcmd-handle-or-name)
    (when paramList
      (princ "     with parameters " *logstream*)
      (prin1 paramList *logstream*)
      (terpri *logstream*))
    (terpri *logstream*))
  
  (let ((old-resource-state (#_HGetState xcmd-handle-or-name))
        (lisp-result-string ""))
    (#_MoveHHi xcmd-handle-or-name)
    (#_HLock xcmd-handle-or-name)
    
    (unwind-protect
      (progn
        ;; --- SET UP XCMD-BLOCK ---
        (rset *params* :XCmdBlock.paramCount (length paramList))
        (do ((plist paramList (cdr plist))
             (param (%inc-ptr *params* 2) (%inc-ptr param 4))
             (numparams 1 (1+ numparams)))
            ((> numparams 16))
          (if (null plist)
            (%put-ptr param (%null-ptr))
            (progn
              (unless (stringp (car plist))
                (error "~:r XCMD parameter (~:r function argument) is not a string"
                       numparams (1+ numparams)))
              (%put-ptr param (make-cstr-handle (car plist)))
              (setq *handles* (cons (%get-ptr param) *handles*)))))
        (rset *params* :XCmdBlock.returnValue (%null-ptr))
        (rset *params* :XCmdBlock.entryPoint callback-handler)

        ;; --- JUMP TO THE XCMD/XFCN RESOURCE ---
        (with-dereferenced-handles ((xcmd-ptr xcmd-handle-or-name))
          (ccl::ff-call xcmd-ptr :ptr *params* :novalue)))
      
      ;; --- RESTORE THE RESOURCE'S ORIGINAL STATE ---
      (#_HSetState xcmd-handle-or-name old-resource-state)
      
      ;; --- CONVERT THE XCMD/XFCN's RESULT C-STRING TO A LISP STRING ---
      (let ((cstr-handle (rref *params* :XCmdBlock.returnValue))
            (count -1))
        (with-dereferenced-handles ((cstr cstr-handle))
          (unless (%null-ptr-p cstr)
            (loop (when (eql 0 (%get-byte cstr (incf count))) (return nil)))
            (unless (eql count 0)
              (setq lisp-result-string (ccl::%str-from-ptr cstr (1+ count)))))))
      
      ;; --- DISPOSE OF MEMORY ALLOCATED BY CALLBACKS (AND PARAMETER STRINGS) ---
      (do () ((null *handles*))
        (#_DisposHandle (car *handles*))
        (setq *handles* (cdr *handles*)))
      
      (do () ((null *pointers*))
        (#_DisposHandle (car *pointers*))
        (setq *pointers* (cdr *pointers*)))
      
      (if *logstream*
        (format *logstream* "~%The XCMD has finished executing.~%~A~%" lisp-result-string)))
    
    ;; --- RETURN "THE RESULT" OF THE XCMD CALL (OR THE FUNCTION VALUE OF THE XFCN CALL) ---
    lisp-result-string))

(provide :xcmds)
(pushnew :xcmds *features*)



#|

EXAMPLES:
--------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; MacroMind PlayMovie Command
;;;

  To use MacroMind's 'PlayMovie' XCMD --

    1. With ResEdit, create a resource file containing the XCMD and place
       it into the folder your 'Macintosh Allegro Lisp' application is in.

    2. Copy the 'MacroMind Player' application, as well as the movies you
       want to play and the sounds they use, into the folder your 'Lisp'
       application is in.

    3. Go!

(defparameter *director-file*
  (with-pstrs ((path-str (namestring (truename "ccl;MM MoviePlayer.rsrc")))
               (#_OpenResFile path-str))))

;;; Without preloading the XCMD or the movie:
(do-xcmd "PlayMovie" "Explosion" "movieNoClear" "movieNoUpdate")


;;; Preloading the XCMD but not the movie:
(setq playMovie (get-xcmd-handle "PlayMovie"))
(do-xcmd playMovie "Explosion" "movieNoClear" "movieNoUpdate")


;;; Preloading the movie but not the XCMD:
(do-xcmd "PlayMovie" "Explosion" "moviePreload")
(do-xcmd "PlayMovie" "movieNoClear" "movieNoUpdate")


;;; Preloading both the XCMD and the movie:
(setq playMovie (get-xcmd-handle "PlayMovie"))
(do-xcmd playMovie "Explosion" "moviePreload")
(do-xcmd playMovie "movieNoClear" "movieNoUpdate")

(#_closeresfile *director-file*)

|#



; (setq *logstream* nil)

; (setq *logstream* *standard-output*)