;; -*- Package: ccl -*-

;; A number of people have asked for this code after I mentioned it in
;; info-macl.  Rather than sending it to each person separately, I am
;; posting it to cambridge.apple.com, where it will be available to
;; anyone who needs it.  If you have a better version of code that
;; does what this tries to do, please let me know of it.  I'm not
;; proud, and would much rather borrow or steal than have to
;; re-invent.  If you use this and make improvements to it, please
;; give them back to me and I will try to post improved versions.

(in-package :ccl)
;; You need to export whatever of this you think you need, to be able to
;; use the code in your own packages.  E.g.,
(export '(movable-view-mixin selectable-view-mixin selected? select deselect))

;; It may also be useful to provide additional functionality to allow the
;; program to figure out what got moved and by how much, etc.  Wouldn't it
;; be nice to be able to take the functionality of MacDraw off the shelf
;; and just be able to use it in your own user interface? !!!


#|

This file implements a basic facility for supporting drawing on a view.

The underlying view being drawn on can be either an ordinary view or a
scrolling view, the items being drawn can be simple things like text
items or more complex objects, and various "gestures" can cause calls
to special purpose functions for creating, moving, linking, etc., of
drawn objects.  (Someday...)  For now, there are facilities for
selecting and for moving/dragging views.

|#

#|
BOILERPLATE:

This code is based on the example files distributed with Macintosh
Allegro Common Lisp from Apple Computer.

These extensions are copyright (C) 1990 by Peter Szolovits, MIT Lab
for Computer Science.

They may be used freely in any derivative code so long as the
following are met:
1)  The original copyright must be retained.
2)  No money other than that reasonably required to reimburse costs of
distribution may be charged without express prior permission of MIT.
3)  The user holds the copyright holders free of liability for any
errors or their consequences.

|#

(require 'QuickDraw)
(require 'scroll-bar-dialog-items)
(require 'scrollers)

;;;;;;;;;;;;;;;;
;;;
;;;  Some useful macros and utilities
;;;
;;;;;;;;;;;;;;;;

; This defn is from Quickdraw.lisp
(defun mode-arg (thing)                 
  (or
   (and (fixnump thing) (<= 0 thing 15) thing)
   (position thing *pen-modes*)
   (error "Unknown pen mode: ~a" thing)))

; The next two defns are from View-Example.lisp
(defun port-set-pen-state (&key location size mode pattern)
  (rlet ((ps :PenState))
    (#_GetPenState :ptr ps)
    (when location
      (rset ps PenState.pnLoc location))
    (when size
      (rset ps PenState.pnSize size))
    (when mode
      (rset ps PenState.pnMode (position mode *pen-modes*)))
    (when pattern
      (rset ps PenState.pnPat pattern))
    (#_SetPenState :ptr ps)))

(defmacro with-pen-state ((&rest states) &body body)
  (let ((ps (gensym)))
    `(rlet ((,ps :PenState))
       (#_GetPenState :ptr ,ps)
       (unwind-protect
         (progn
           (port-set-pen-state ,@states)
           ,@body)
         (#_SetPenState :ptr ,ps)))))

(defun point-max (a b)
  (make-point (max (point-h a) (point-h b))
              (max (point-v a) (point-v b))))

(defun point-min (a b)
  (make-point (min (point-h a) (point-h b))
              (min (point-v a) (point-v b))))


;;;;;;;;;;;;;;;;
;;;
;;;  Definition of a movable-view-mixin
;;;
;;;;;;;;;;;;;;;;

(defclass movable-view-mixin () ()
  (:documentation "A mixin to allow any view to be moved within its container"))

(defmethod view-click-event-handler ((v movable-view-mixin) where)
  (cond ((typep v 'view)
         ;; In this case, the movable view is itself a general view, so we need
         ;; to change focus to its container and convert where to the
	 ;; container's coordinates
         (let ((v-cont (view-container v)))
           (with-focused-view v-cont
             (view-move 
              v v-cont (convert-coordinates where v v-cont)))))
        (t 
         ;; In this case, the view is a simple-view, and we just use the
         ;; focus and coordinates of its container
         (view-move v (view-container v) where))))

(defun view-move (v scroller anchor-point)
  "Move a view V and all other selected subviews of its container, SCROLLER.
ANCHOR-POINT is the original mouse position in the coordinates of
SCROLLER.  If SCROLLER is in fact a scroller, then attempting to move
the mouse outside the visible window will cause scrolling."
  ;; First we create a region that represents an outline of all the
  ;; selected subviews: 
  (let ((rgn nil)
        (selsibs (if (fboundp 'selected-siblings)
                   ;; careful not to rely on selectable-view being loaded!
                   (selected-siblings v)
                   #())))
    (unwind-protect
      (progn
        (setq rgn (union-self-to-region v (new-region)))
        (map nil
             #'(lambda (sv)
                 (union-self-to-region sv rgn))
             selsibs)
        (drag-region v scroller anchor-point rgn selsibs)
        )
      (when rgn (dispose-region rgn)))))

(defmethod union-self-to-region ((v simple-view) rgn)
  (let ((r nil))
    (unwind-protect
      (progn
        (setq r (set-rect-region (new-region) 
                                 (view-position v)
                                 (add-points (view-position v) (view-size v))))
        (union-region r rgn rgn))
      (when r (dispose-region r)))))

(defvar *drag-rbound*)
(defvar *drag-rfast*)
(defvar *drag-rslop*)
(defvar *drag-view*)
(defparameter *drag-fast-margin* 16
  "Number of pixels by which we must move outside the border of a scroller
to cause automatic scrolling to jump by scroll-bar-page-size instead of
scroll-bar-scroll-size.  If NIL, then never.")
(defparameter *drag-slop-margin* 48
  "Number of pixels by which we must move outside the border of a scroller
before automatic scrolling is suspended.  If NIL, then never suspended.")
(defconstant drag-failed-outside-slop #x-7FFF8000
  "Magic MAC constant indicating that DragGrayRgn mouse down was
released outside the slop rect.  No movement should therefore take
place.")

(defmethod offset-view-position ((v simple-view) delta)
  (set-view-position v (add-points (view-position v) delta)))

(defun show-rect (r)
  (format nil "(~d,~d)-(~d,~d)"
          (rref r :rect.left) (rref r :rect.top)
          (rref r :rect.right) (rref r :rect.bottom)))

(defun show-point (p)
  (format nil "(~d, ~d)" (point-h p) (point-v p)))
        
;;; We have tried to implement drag-region using the _DragGrayRgn
;;; trap, but it does not work well with scrolling.  Thus, we do our own:

(defun drag-region (v scroller anchor-point rgn sibs)
  (let* ((scroller-scrollpos (view-scroll-position scroller))
         (scroller-otherend (add-points scroller-scrollpos 
					(view-size scroller)))
         (*drag-view* scroller))
    ;; We now define three rects, each successive one containing the
    ;; previous ones.  rbound is the actual boundary of the visible
    ;; pane; rfast is the region outside whick scrolling speeds up;
    ;; rslop is the region outside which the drag suspends.
    (rlet ((*drag-rbound*
            :rect :topleft scroller-scrollpos :bottomright scroller-otherend)
           (*drag-rfast*
            :rect :topleft scroller-scrollpos :bottomright scroller-otherend)
           (*drag-rslop*
            :rect :topleft scroller-scrollpos :bottomright scroller-otherend))
      (when *drag-fast-margin*
        (Inset-Rect *drag-rfast* 
                    (- *drag-fast-margin*) (- *drag-fast-margin*)))
      (when *drag-slop-margin*
        (Inset-Rect *drag-rslop*
                    (- *drag-slop-margin*) (- *drag-slop-margin*)))
      #|(eval-enqueue `(format t "~%bound=~a, fast=~a, slop=~a"
                             ',(show-rect *drag-rbound*)
                             ',(show-rect *drag-rfast*)
                             ',(show-rect *drag-rslop*)))|#
      (let ((delta 
             (DragTheRegion rgn anchor-point scroller)))
        (eval-enqueue `(format t "~%Dragged ~a" ',(show-point delta)))
        (unless (or (zerop delta) (eql delta drag-failed-outside-slop))
          (unless (find v sibs) (offset-view-position v delta))
          (map nil
               #'(lambda (sv)
                   (offset-view-position sv delta))
               sibs))))))

(defun DragTheRegion (rgn anchor-point scroller)
  ;; We assume that we have already focused on the view of the scroller.
  ;; Returns the offset dragged from anchor-point.
  (let* ((current-mouse anchor-point)
         (new-mouse anchor-point)
         (frame-drawn-once? nil)
         )
    (with-pen-state (:mode :patXor :pattern *gray-pattern*)
      (when (mouse-down-p)              ; be sure mouse is still being pressed.
        (setq frame-drawn-once? t)
        (#_FrameRgn rgn))
      (loop (unless (mouse-down-p) (return))
            (unless (and (= (setq new-mouse (view-mouse-position scroller))
                            current-mouse)
                         (point-in-rect-p *drag-rbound* new-mouse))
              (when frame-drawn-once? (#_FrameRgn rgn))   ; erase old frame
              (cond
               ((cond ((not (typep scroller 'scroller-mixin))
                       ;; Cannot scroll, therefore simply constrain
		       ;; the mouse to lie within the view.
                       (setq new-mouse
                             (point-min (point-max new-mouse 
                                                   (rref *drag-rbound*
							 :rect.topleft))
                                        (rref *drag-rbound*
					      :rect.bottomright))))
                      ((point-in-rect-p *drag-rbound* new-mouse)
                       ;; We are inside the body of the scroller,
		       ;; therefore no scrolling is needed.
                       new-mouse)
                      ((point-in-rect-p *drag-rslop* new-mouse)
                       ;; We are outside the scroller, but within the
		       ;; slop rect.  Therefore, invoke scrolling.
                       (setq new-mouse (constrain-to-view scroller new-mouse)))
                      (t 
                       ;; Else we are outside the slop, and should simply stop
                       ;; showing the frame until/unless we return inside.
                       nil))
                (#_OffsetRgn :ptr rgn :long (subtract-points new-mouse
							     current-mouse))
                (#_FrameRgn rgn)           ; draw new frame
                (setq frame-drawn-once? t)
                (setq current-mouse new-mouse))
               (t ;; In this case, we should stop showing frame.
                (setq frame-drawn-once? nil)))))
      (when frame-drawn-once? (#_FrameRgn rgn))   ; final erase of frame
      (if frame-drawn-once?
        ;; I.e., if, we really did drag things some (maybe 0) distance,
        (subtract-points current-mouse anchor-point)
        ;; Otherwise, we stopped outside slop
        drag-failed-outside-slop))))

#|  
(defmethod constrain-to-view ((v view) new-mouse rbound)
  (point-min (point-max new-mouse (rref rbound :rect.topleft))
             (rref rbound :rect.bottomright)))
|#

(defmethod constrain-to-view ((v scroller) new-mouse)
  (let ((m-h (point-h new-mouse))
        (m-v (point-v new-mouse))
        (scroll-h (h-scroller v))
        (scroll-v (v-scroller v))
        )
    ;; There should be a more elegant implementation of this using 
    ;; _PinRect(rect, point) : point, but what we have here does work.
    (labels ((constrain-to-border 
                (min? scroll-bar fast?)
                (let* ((setting (scroll-bar-setting scroll-bar))
                       (limit (if min? 
                                (scroll-bar-min scroll-bar)
                                (scroll-bar-max scroll-bar)))
                       (inc (if fast? 
                              (scroll-bar-scroll-size scroll-bar) 
                              (scroll-bar-page-size scroll-bar)))
                       (possible-change (- limit setting))
                       (change (if min? 
                                 (max possible-change (- inc))
                                 (min possible-change inc)))
                       (offset (if (eq (scroll-bar-direction scroll-bar)
				       :horizontal) 
                                 (make-point change 0)
                                 (make-point 0 change))))
                  (unless (zerop change)
                    (set-scroll-bar-setting scroll-bar (+ setting change))
                    (scroll-bar-changed v scroll-bar)
                    (#_OffsetRect :ptr *drag-rbound* :long offset)
                    (#_OffsetRect :ptr *drag-rfast* :long offset)
                    (#_OffsetRect :ptr *drag-rslop* :long offset)
                    (setq new-mouse (add-points new-mouse offset))
                    t))))
      (let ((rf *drag-rfast*)
            (rb *drag-rbound*))
        #|(eval-enqueue `(format t "~%~a: bound=~a, fast=~a, slop=~a"
                               ',(show-point new-mouse)
                               ',(show-rect *drag-rbound*)
                               ',(show-rect *drag-rfast*)
                               ',(show-rect *drag-rslop*)))|#
        (cond ((< m-h (rref rb :rect.left))
               (constrain-to-border t scroll-h 
                                    (minusp (- (rref rf :rect.left) m-h))))
              ((> m-h (rref rb :rect.right))
               (constrain-to-border nil scroll-h 
                                    (plusp (- (rref rf :rect.right) m-h)))))
        (cond ((< m-v (rref rb :rect.top))
               (constrain-to-border t scroll-v 
                                    (minusp (- (rref rf :rect.top) m-v))))
              ((> m-v (rref rb :rect.bottom))
               (constrain-to-border nil scroll-v 
                                    (plusp (- (rref rf :rect.bottom) m-v)))))
        #|(eval-enqueue `(format t "~%bound=~a, fast=~a, slop=~a"
                               ',(show-rect *drag-rbound*)
                               ',(show-rect *drag-rfast*)
                               ',(show-rect *drag-rslop*)))|#
        new-mouse))))


#|  Here are some examples of how to make movable views.  The first,
w1, is using a non-scrolling base window.  The second, w2, is in a
scrolling window.

(load "ccl:examples;scrolling-windows")

(setq w1 (make-instance 'window :view-position #@(50 50) :view-size #@(300 230)
                        :window-title "Non-scroll movable test"))

(defclass foobar (selectable-view-mixin movable-view-mixin
					static-text-dialog-item)
  ())

(defclass foozorch (movable-view-mixin check-box-dialog-item) ())

(defclass barfoo (movable-view-mixin view) ())

(defmethod view-draw-contents ((v barfoo))
  #|(eval-enqueue `(format t "~%View-draw-contents ~a" ',v))|#
  (let ((vp (view-position v)))
    ;;(fill-rect v #@(0 0) (view-size v))
    (with-focused-view (view-container v)
      (rlet ((r :rect :topleft vp :bottomright (add-points vp (view-size v))))
        (#_FillRect :ptr r :ptr *gray-pattern*)
        (#_FrameRect :ptr r)))))

(setq f1 (make-instance 'foobar :view-position #@(20 20) 
                        :dialog-item-text "Hello there"))

(setq b1 (make-instance 'barfoo :view-position #@(50 50)
                        :view-size #@(40 40)))

(add-subviews w1 f1)
(add-subviews w1 b1)
;;(view-container f1)
;;(point-string (view-position f1))
;;(typep f1 'simple-view)
;;(typep f1 'view)
;;(set-view-position f1 #@(10 10))
;;(set-view-position b1 #@(50 50))
;;(view-draw-contents b1)
;;(typep b1 'simple-view)
;;(type-of b1)
;;(typep b1 'view)
;;(view-window b1)
;;(view-window f1)
;;(find-class 'scrolling-window)

;; Here we make a scrolling-window:

(defclass scroller-psz (scroller) ())

(defmethod scroll-bar-limits ((scr scroller-psz))
  (normal-scroll-bar-limits scr (make-point 1000 1000)))
(defmethod scroll-bar-page-size ((scr scroller-psz))
  #@(40 40))
(setq w2 (make-instance 'scrolling-window 
                        :scroller-class 'scroller-psz
                        :window-title "Movable Test"
                        :track-thumb-p t))


(setq f2 (make-instance 'foobar :view-position #@(20 20) 
                        :dialog-item-text "Hello there"))

(setq b2 (make-instance 'barfoo :view-position #@(50 50)
                        :view-size #@(40 40)))
(setq n2 (make-instance 'foozorch :view-position #@(30 40)
                        :dialog-item-text "Button Me"))
(setq f3 (make-instance 'foobar :view-position #@(20 20) 
                        :dialog-item-text "Hello there"))
(add-subviews (my-scroller w2) f2 b2)
(add-subviews (my-scroller w2) n2 f3)
;;(point-string (view-position b2))
;;(set-view-position b2 0 0)
;;(set-view-position b2 100 100)
;;(setq s2 (my-scroller w2))
;;(setq h2 (h-scroller s2))
;;(setq v2 (v-scroller s2))
;;(scroll-bar-setting h2)
;;(scroll-bar-max h2)
;;(scroll-bar-min h2)
;;(multiple-value-bind (h v) (scroll-bar-limits s2)
;;  (list (point-string h) (point-string v)))
;;(setf (scroll-bar-max h2) 1000)
;;(point-string (view-scroll-position s2))
;;(point-string (view-size s2))
;;(point-string (view-size (view-window s2)))
;;(set-scroll-bar-setting h2 (+ (scroll-bar-setting h2) 1))
;;(progn (incf (scroll-bar-setting h2) 3) (scroll-bar-changed s2 h2))
;;(setf (view-scroll-position s2) (add-points (view-scroll-position s2) #@(1 1)))
;;(update-thumbs s2)
;;(selected-siblings b2)

|#



;;;;;;;;;;;;;;;;
;;;
;;;  Definition of a movable-view-mixin
;;;
;;;;;;;;;;;;;;;;

;; -*- Package: ccl -*-

(in-package :ccl)

#|

This file implements a basic facility for supporting selection of
objects in a view.  The objects are themselves (sub)views.  Usual
Macintosh conventions apply for selecting: Clicking on an object
selects it and de-selects any others selected.  Shift-clicking simply
toggles the selection state of the target object; thus it can be used
to add to a multiple-selection or to remove from it.  There should
also be a means of de-selecting all selected objects when the user
clicks on the container but not on any object; the implementation of
this is, however, commented out, in case selections should survive
clicking on (say) other objects in the view that do not inherit from
selectable-view-mixin.

The underlying view being drawn on can be either an ordinary view or a
scrolling view, the items being drawn can be simple things like text
items or more complex objects, and various "gestures" can cause calls
to special purpose functions for creating, moving, linking, etc., of
drawn objects.

|#

#|
BOILERPLATE:

This code is based on the example files distributed with Macintosh Allegro
Common Lisp from Apple Computer.

These extensions are copyright (C) 1990 by Peter Szolovits, MIT Lab
for Computer Science.

They may be used freely in any derivative code so long as the
following are met:
1)  The original copyright must be retained.
2)  No money other than that reasonably required to reimburse costs of
distribution may be charged without express prior permission of MIT.
3)  The user holds the copyright holders free of liability for any errors or
their consequences.

|#

(require 'QuickDraw)

;; First we provide innocuous default selection behavior for all views that
;; are not selectable-view-mixin derived.

(defmethod selected? ((v simple-view))
  "By default, no view is selected."
  (declare (ignore v))
  nil)

(defmethod select ((v simple-view))
  "Cannot select ordinary views; only selectable-view-mixin's."
  (declare (ignore v))
  nil)

(defmethod deselect ((v simple-view))
  "Cannot deselect ordinary views; only selectable-view-mixin's."
  (declare (ignore v))
  nil)

;; Now we define selectable-view-mixin:

(defclass selectable-view-mixin ()
  ((selected? :initform nil :accessor selected?)))

(defmethod view-click-event-handler :before ((v selectable-view-mixin) where)
  (declare (ignore where))
  ;; If selected with shift key, just toggle the selection of the clicked item.
  ;; Otherwise, do nothing when selecting an already-selected item, but
  ;; if selecting an unselected item, deselect all and select just this one.
  (cond ((shift-key-p)
         (if (selected? v)
           (deselect v)
           (select v)))
        ((selected? v)
         t)
        (t
         (deselect-selected-siblings v)
         (select v))))

(defmethod select ((v selectable-view-mixin))
  (unless (selected? v)
    (draw-selection-marks v))
  (setf (selected? v) t)
  t)

(defmethod deselect ((v selectable-view-mixin))
  (when (selected? v)
    (draw-selection-marks v))           ; this actually erases them!
  (setf (selected? v) nil)
  nil)

;;; Deal with other selections in a view

(defmethod selected-siblings ((v simple-view))
  "Returns sequence of selected subviews of the container of v, including
(probably) v."
  (selected-subviews (view-container v)))

(defmethod selected-subviews ((v view))
  (remove-if #'(lambda (subview) (not (selected? subview)))
             (view-subviews v)))

(defmethod deselect-selected-siblings ((v simple-view))
  (map nil
       #'deselect
       (selected-siblings v)))

(defmethod deselect-selected-subviews ((v view))
  (map nil 
       #'deselect
       (selected-subviews v)))

;;; The method below would cause any clicks not going to selectable-view-mixin
;;; objects to cause de-selection of all.  Uncomment out it if that behavior is
;;; not desired.  It is left commented out because in general it is quite
;;; costly:  the handler runs on every single click on a view, and typically
;;; decides that there are no relevant selectable-subviews for it to work on.
;;; If selectable-subviews occur only in some specific class of views, it would
;;; be more appropriate to define this method only for those.  Therefore it is
;;; not actually installed here.

#|
(defmethod view-click-event-handler :before ((v view) where)
  (view-deselect-selected-subviews-if-random-clicked-on v where))
|#

(defun view-deselect-selected-subviews-if-random-clicked-on (v where)
  "Meant to be called by view-click-event-handler of view.  If something
other than a selectable-view-mixin was clicked on, then deselect all the
selectable subviews."
  (when (some #'(lambda (view)
                  (typep view 'selectable-view-mixin))
              (view-subviews v))
    ;; If there are any selectable-subviews in v, then we must find out
    ;; which one the click will go to, to see if it is selectable.  If not,
    ;; or if the click is off any subviews, then we deselect all.
    (unless (catch 'decide-deselect
              (map nil 
                   #'(lambda (sv)
                       (when (point-in-click-region-p sv where)
                         (throw 'decide-deselect
                                (if (typep sv 'selectable-view-mixin)
                                  t   ; it will be handled by sv
                                  nil   ; clicked on non-sel-view
                                  ))))
                   ;; Note that we have to reverse to be consistent with
                   ;; MACL's ordering of back-to-front subviews.
                   (reverse (view-subviews v))))
      (with-focused-view v
        (deselect-selected-subviews v)))))

;;; Actual drawing of the marks

(defmethod draw-selection-marks ((v selectable-view-mixin))
  "Draws MacDraw-style corner squares to show that the object is selected.
Draws selection marks without examining selected? state.  Therefore, it is
a toggle that either draws or erases the marks."
  ;; Note that if v is a simple-view, we are properly focussed on its
  ;; container, but if not, then we have to refocus.
  (cond ((typep v 'view)
         (let ((v-cont (view-container v)))
           (with-focused-view v-cont
             (draw-selection-marks-really v))))
        (t (draw-selection-marks-really v))))

(defun draw-selection-marks-really (v)
  (let ((pos (view-position v))
        (siz (view-size v)))
    (labels ((draw-little-square 
              (p)
              (rlet ((little-r :rect
                               :topleft (subtract-points p #@(2 2))
                               :bottomright (add-points p #@(2 2))))
                (#_PaintRect little-r)
                (#_ValidRect little-r))))
      (with-pen-state (:mode :patXor)
        (draw-little-square pos)
        (draw-little-square (add-points pos siz))
        (draw-little-square (add-points pos (make-point (point-h siz) 0)))
        (draw-little-square (add-points pos (make-point 0 (point-v siz))))))))

(defmethod invalidate-view ((v selectable-view-mixin) &optional erase-p)
  "Selection squares may extend 2pts outside the bounds of the view, so
we must inval the larger rect."
  (when (view-container v)
    (rlet ((r :rect 
              :topleft (view-position v)
              :bottomright (add-points (view-position v) (view-size v))))
      (with-focused-view (view-container v)
        (#_InsetRect :ptr r :long #@(-2 -2))
        (when erase-p (#_EraseRect r))
        (#_InvalRect r)))))

(defmethod view-draw-contents :after ((v selectable-view-mixin))
  (when (selected? v)
    (draw-selection-marks v)))

#|  Here are some examples of selectable objects.  Note that this
facility was really designed to support movable-views, but can be used
by itself if needed.

(require 'scrolling-windows "ccl;examples:grapher:scrolling-windows")

(setq wst (make-instance 'window :view-position #@(30 40) :view-size #@(200 200)
                         :window-title "Selection Test"))

(defclass seltext (selectable-view-mixin static-text-dialog-item) ())

(setq st1 (make-instance 'seltext :view-position #@(10 10)
                         :dialog-item-text "Item 1")
      st2 (make-instance 'seltext :view-position #@(70 10)
                         :dialog-item-text "Item 2")
      st3 (make-instance 'seltext :view-position #@(10 40)
                         :dialog-item-text "Item 3")
      st4 (make-instance 'seltext :view-position #@(70 40)
                         :dialog-item-text "Item 4"))

(add-subviews wst st1 st2 st3 st4)

|#
