;;; A metaclass for classes which want to keep track of their instances.
;;; 1992, 1993, Mark Nahabedian, Cambridge MA.  naha@mit.edu

;;; This file implements a metaclass called INSTANCE-INTERNING-CLASS.
;;; All instances which are created for a class whose metaclass is
;;; INSTANCE-INTERNING-CLASS are recorded in an instance registry object
;;; (defined below).

;;; When you make a new instance of a class whose metaclass is
;;; INSTANCE-INTERNING-CLASS, MAKE-INSTANCE first checks to see if an
;;; existing instance matches the initialization arguments based on
;;; criteria which can be controlled by specializing the
;;; INSTANCE-INITARGS-MATCH method.

;;; MCL doesn't implement VALIDATE-SUPERCLASS.  It is therefore up to
;;; the programmer to make sure that all subclasses of a class which is
;;; of metaclass INSTANCE-INTERNING-CLASS are also of metalcass
;;; INSTANCE-INTERNING-CLASS.

;;; Any instance-interning-class can be given its own instance registry.
;;; By default, an instance registry is only created for the outermost
;;; instance-interning-class on a class's precedence list.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; classes of metaclass INSTANCE-INTERNING-CLASS will need to
;;; specialize this generic function
(defgeneric instance-initargs-match (instance initargs)
  (:documentation "compares the instance with the initargs list and returns T if the
 instance matches the initargs.  This function is called by MAKE-INSTANCEto determine 
 if an existing instance is suitable."))

;;; default method.
(defmethod instance-initargs-match (instance initargs)
  (dolist (slotd (class-instance-slots (class-of instance)))
    (let ((slot-name (slot-definition-name slotd))
          (slot-initargs (third slotd))
          (no-initarg '#:no-initarg))
      (dolist (initarg slot-initargs)
        (let ((init-value (getf initargs initarg no-initarg)))
          (unless (eq init-value no-initarg)
            (unless (equal init-value (slot-value instance slot-name))
              (return-from instance-initargs-match nil)))))))
  t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; keeping track of existing instances

(defclass instance-registry ()
  ((for-class :initarg :for-class :reader instance-registry-for-class)))

(defmethod print-object ((object instance-registry) stream)
  (format stream "#<~a ~a>" 
          (class-name (class-of object)) 
          (ignore-errors (class-name (instance-registry-for-class object)))))

(defgeneric register-instance (instance-registry instance))

(defgeneric map-instances (instance-registry function)
  (:documentation "Apply function to each registerred instance in the registry"))

(defclass list-instance-registry (instance-registry)
  ((instances :initform nil)))

(defmethod register-instance ((registry list-instance-registry) instance)
  (unless (typep instance (instance-registry-for-class registry))
    (error "The instance ~a is not appropriate for the registry ~a" instance registry))
  (with-slots (instances) registry
    (push instance instances))
  instance)

(defmethod map-instances ((registry list-instance-registry) function)
  (with-slots (instances) registry
    (dolist (instance instances)
      (funcall function instance))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; a metaclass for classes which keep track of their instances

(defclass instance-interning-class (standard-class)
  ((instance-registry :reader class-instance-registry)))

(defgeneric class-instance-registry (class)
  (:documentation "Return the instance registry which records instances of class"))

(defmethod class-instance-registry ((class t)) nil)

;;; A class which hasn't been given an instance registry of its own will use the one
;;; from a superclass.
(defmethod slot-unbound (class (instance instance-interning-class) (slot (eql 'instance-registry)))
  (declare (ignore class))
  (dolist (superclass (reverse (class-precedence-list instance)))
    (when (typep superclass 'instance-interning-class)
      (unless (slot-boundp superclass 'instance-registry)
        (setf (slot-value superclass 'instance-registry)
              (make-instance 'list-instance-registry :for-class superclass)))
      (setf (slot-value instance 'instance-registry) 
            (class-instance-registry superclass))
      (return)))
  (slot-value instance 'instance-registry))


(defgeneric find-specified-instances (class &rest initargs)
  (:documentation "Find instances of the class which match initargs"))

(defmethod find-specified-instances ((class instance-interning-class) &rest initargs)
  (let ((instances nil))
    (map-instances (class-instance-registry class)
                   #'(lambda (instance)
                       (when (typep instance class)
                         (when (instance-initargs-match instance initargs)
                           (push instance instances)))))
    instances))

(defmethod find-specified-instances ((class symbol) &rest initargs)
  (apply #'find-specified-instances (find-class class) initargs))

(defmethod allocate-instance :around ((class instance-interning-class) &rest initargs)
  (declare (ignore initargs))
  (let ((instance (call-next-method)))
    (register-instance (class-instance-registry class) instance)
    instance))

(defmethod make-instance :around ((class instance-interning-class) &rest initargs)
  (let ((i (apply #'find-specified-instances class initargs)))
    (cond ((null i)
           (call-next-method))
          ((null (cdr i)) 
           (apply #'reinitialize-instance (car i) initargs))
          (t (error "There are several instances which match initargs ~a: ~a" 
                    initargs i)))))


(defmethod map-instances ((class symbol) function)
  (map-instances (find-class class) function))

(defmethod map-instances ((class instance-interning-class) function)
  (map-instances (class-instance-registry class)
                 #'(lambda (instance)
                     (when (typep instance class)
                       (funcall function instance)))))


;;; Use this macro to generate a fasl file that will construct all the instances of the class
;;; when loaded.
(defmacro dump-instances (class)
  `(list ',class
         ,@(let ((instances nil))
             (map-instances (find-class class) #'(lambda (i) 
                                                   (push i instances)))
             instances)))

(provide 'instance-interning-class)

#|

(defclass test-class ()
  ((name :initarg :name))
  (:metaclass instance-interning-class))

(defmethod instance-initargs-match ((instance test-class) initargs)
  (eq (slot-value instance 'name)
      (getf initargs :name '#:no-name)))

(defclass test-subclass1 (test-class) () (:metaclass instance-interning-class))
(defclass test-subclass2 (test-class) () (:metaclass instance-interning-class))

(make-instance 'test-class :name 'a)
(make-instance 'test-subclass1 :name 'b)
(make-instance 'test-subclass2 :name 'c)

|#
