;;; -*- package: inspector -*-

(in-package :inspector)

;;; February 1992, Mark Nahabedian, Cambridge MA, naha@mit.edu

;;; Need to abstract the display mode stuff.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Make up for some MCL deficiencies:

(defmethod class-direct-slots ((class standard-class))
  (append (class-direct-class-slots class)
          (class-direct-instance-slots class)))

(defmethod class-slots ((class standard-class))
  (append (class-class-slots class)
          (class-instance-slots class)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Prettier printing of the methods of generic functions

;;; make sure the generic function is the function definition of its name.
(defun generic-function-proper-name-p (gf)
  (eq gf (symbol-function (function-name gf))))

;;; The class's name is more concise than the printed representation 
;;; of the class.  If the name is accurate, use it.
(defun proper-class-name (class)
  (if (eq class (find-class (class-name class)))
    (class-name class)
    class))

;;; In the generic function inspector, the only thing interesting about a 
;;; method are its specializers and qualifiers.  The rest is visual noise
;;; and needlessly consumes valuable screen real estate.
(defmethod prin1-value ((i gf-inspector) stream (value standard-method) 
                        &optional label type)
  (declare (ignore type label))
  (let ((gf (method-generic-function value)))
    (if (generic-function-proper-name-p gf)
      (let ((specializers (method-specializers value)))
        (format stream "  ~a" 
                (mapcar #'(lambda (specializer)
                            (if (typep specializer 'ccl::class)
                              (proper-class-name specializer)
                              specializer))
                        specializers))
        (when (method-qualifiers value)
          (format stream "~{ ~a~}" (method-qualifiers value)))) 
      (call-next-method))))

#|
;;; Find the methods of the generic function which the user cares about.

(defclass gf-applicable-methods-inspector (inspector)
  ())

(defmethod inspector-commands :around ((i gf-inspector))
  (let ((generic-function (inspector-object i)))
    (flet ((inspect-applicable-methods ()
             (make-inspector-window
              (make-instance 'gf-applicable-methods-inspector 
                             :object generic-function)
              :window-title (format nil "Applicable Methods of ~a"))))
      (cons `("Applicable Methods" #'inspect-applicable-methods)
            (call-next-method)))))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; support for modal displays

;;; Some information can be displayed in several different ways.  This 
;;; provides a command interface for several different display modes.
;;; For example, when inspecting the slots of a class, one might want
;;; only the direct slots or one might want all of the slots.

(defclass modal-display-mixin ()
  ((mode :initarg :mode :accessor inspect-display-mode)))

(defgeneric compute-line-count-for-mode (inspector mode))
(defgeneric line-n-for-mode (inspector n mode))

;;; Force a redisplay after the mode is changed.
(defmethod (setf inspect-display-mode) :after (new-value
                                               (i modal-display-mixin))
  (declare (ignore new-value))
  (when (slot-boundp i 'view)
    (resample (slot-value i 'view))))

(defmethod compute-line-count ((i modal-display-mixin))
  ;; Well, we should have a better way of initializing the display
  ;; mode if it didn't get initialized.
  (unless (slot-boundp i 'mode)
      (inspector-commands i))
  (compute-line-count-for-mode i (inspect-display-mode i)))

(defmethod compute-line-count-for-mode ((i modal-display-mixin) mode)
  ;; mode isn't defined for this inspector, reinitialize it.
  mode
  (slot-makunbound i 'mode)
  (compute-line-count i))

(defmethod line-n ((i modal-display-mixin) n)
  (line-n-for-mode i n (inspect-display-mode i)))

(defmethod inspector-command-name ((i modal-display-mixin) mode)
  (format nil "~:{~s~}" mode))

;;; Construct the list of commands based on EQL specializers of the mode
;;; argument for methods applicabble to this type of inspector.
(defmethod inspector-commands :around ((inspector modal-display-mixin))
  (let ((commands nil))
    (labels ((method-mode (method)
               (let ((i-specializer (car (method-specializers method)))
                     (mode-specializer (mode-specializer method)))
                 (when (and (not (listp i-specializer))
                            ;; I tried to use TYPEP
                            (member i-specializer 
                                    (class-precedence-list (class-of inspector)))
                            (listp mode-specializer)
                            (eq (car mode-specializer) 'eql)
                            (typep (second mode-specializer) 'keyword))
                   (second mode-specializer))))
             (mode-specializer (method)
               (third (method-specializers method))) )
    (dolist (method (generic-function-methods #'line-n-for-mode))
      (let ((mode (method-mode method)))
        (when mode
          (unless (slot-boundp inspector 'mode)
            (setf (slot-value inspector 'mode) mode))
          (push (list (inspector-command-name inspector mode)
                      #'(lambda ()
                          (setf (inspect-display-mode inspector) mode)))
                commands))))
    (append (sort commands #'string-lessp :key #'first)
            (call-next-method)))))

;;; A macro to facilitate the construction of display modes.
;;; INSPECTOR-CLASS is the class of the inspector that the mode is being 
;;; defined for.
;;; MODE is the name of the display mode being defined.
;;; COMMAND-NAME is a string which is how the mode will be displayed in
;;; the command menu.
;;; COMPUTE-LINE-COUNT and LINE-N describe the behaviors of these generic
;;; functions when the inspector is in this mode.  They are in the form
;;; (arglist . body).  For COMPUTE-LINE-COUNT, arglist just contains the
;;; name of the first (and only) argument to the COMPUTE-LINE-COUNT
;;; generic function.  For LINE-N, the arglist contains the names of the two
;;; arguments to the LINE-N generic functioon.
(defmacro define-inspector-display-mode ((inspector-class mode) &key
                                         (command-name nil command-name?)
                                         compute-line-count
                                         line-n)
  ;; compute-line-count and line-n are ((arglist) . body)
  `(progn
     ,@(when command-name?
         `((defmethod inspector-command-name ((i ,inspector-class)
                                              (mode (eql ,mode)))
             ,command-name)))
     ,(let ((i-var (caar compute-line-count))
            (body (cdr compute-line-count)))
        `(defmethod compute-line-count-for-mode ((,i-var ,inspector-class)
                                                 (mode (eql ,mode)))
           ,@body))
     ,(let* ((args (car line-n))
             (i-var (first args))
             (n-var (second args))
             (body (cdr line-n)))
        `(defmethod line-n-for-mode ((,i-var ,inspector-class)
                                     ,n-var
                                     (mode (eql ,mode)))
           ,@body))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; class inspector.  Do the normal thing as for standard-object, but also
;;; provide commands to look at subclasses, superclasses, slots and methods
;;; in inspectors designed for those purposes.

(defclass class-inspector (standard-object-inspector)
  ())

(defclass standard-class-inspector (class-inspector)
  ())

;;; Well, there should be a class names CLASS which is a superclass of both 
;;; BUILT-IN-CLASS and STANDARD-CLASS, but there isn't.
(defmethod inspector-class ((o built-in-class))
  'class-inspector)

(defmethod inspector-class ((o structure-class))
  'class-inspector)

(defmethod inspector-class ((o standard-class))
  'standard-class-inspector)

;;; display classes as concisely as possible
(defun print-class-proper-name (inspector stream value label type)
  (declare (ignore inspector type label))
  (prin1 (proper-class-name value) stream))

;;; All classes have subclasses and superclasses
(defmethod inspector-commands :around ((i class-inspector))
  (let ((class (inspector-object i)))
    (flet ((inspect-subclasses ()
             (make-inspector-window
              (make-instance 'class-subclass-inspector :object class) 
              :window-title (format nil "Subclasses of ~a" 
                                    (proper-class-name class))))
           (inspect-superclasses ()
             (make-inspector-window
              (make-instance 'class-superclass-inspector :object class)
              :window-title (format nil "Superclasses of ~a" 
                                    (proper-class-name class)))))
      (list* `("Subclasses" ,#'inspect-subclasses)
             `("Superclasses" ,#'inspect-superclasses)
             (call-next-method)))))

(defmethod inspector-commands :around ((i standard-class-inspector))
  (let ((class (inspector-object i)))
    (flet ((inspect-slots ()
             (make-inspector-window
              (make-instance 'class-slots-inspector :object class)
              :window-title (format nil "Slots of ~a"
                                    (proper-class-name class)))))
      (list* `("Slots" ,#'inspect-slots)
             (call-next-method)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Show the subclasses of the class.  Mode for direct and all.

(defclass class-subclass-inspector (modal-display-mixin inspector)
  ((tree-info))
  (:default-initargs :mode :direct))

(define-inspector-display-mode 
  (class-subclass-inspector :direct)
  :command-name "Direct Subclasses"
  :compute-line-count
  ((inspector)
   (+ 1 (length (class-direct-subclasses (inspector-object inspector)))))
  :line-n
  ((inspector n)
   (let* ((class (inspector-object inspector))
          (direct-subclasses (class-direct-subclasses class)))
     (cond ((= n 0)
            (if direct-subclasses
              (values "" "Direct Subclasses:" :comment)
              (values "" "No direct subclasses" :comment)))
           ((<= 1 n (length direct-subclasses))
            (values (nth (1- n) direct-subclasses) "" :static
                    #'print-class-proper-name))
           (t (values "" "TOO FAR" :comment))))))

(define-inspector-display-mode
  (class-subclass-inspector :subclass-tree)
  :command-name "Subclass Tree"
  :compute-line-count
  ((i)
   (unless (slot-boundp i 'tree-info)
     (setf (slot-value i 'tree-info)
           (build-tree-info (inspector-object i) #'class-direct-subclasses)))
   (1+ (length (slot-value i 'tree-info))))
  :line-n
  ((i n)
   (with-slots (tree-info) i
     (cond ((zerop n)
            (values "" "Subclasses:" :comment))
           ((<= 1 n (length tree-info))
            (let* ((entry (nth (1- n) tree-info))
                   (class (second entry))
                   (indentation (first entry)))
              (flet ((print-it (inspector stream value label type)
                       (dotimes (j (* 2 indentation))
                         (write-char #\space stream))
                       (print-class-proper-name inspector stream value 
                                                label type)))
                (values class "" :static #'print-it))))
           (t (values "" "TOO FAR" :comment))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; show the superclasses of the class.  mode for direct or all

(defclass class-superclass-inspector (modal-display-mixin inspector) 
  ((tree-info))
  (:default-initargs :mode :direct))

(define-inspector-display-mode 
  (class-superclass-inspector :direct)
  :command-name "Direct Superclasses"
  :compute-line-count
  ((i)
   (1+ (length (class-direct-superclasses (inspector-object i)))))
  :line-n
  ((i n)
   (class-superclass-display-common 
    n (class-direct-superclasses (inspector-object i))
    "Direct Superclasses:"
    "No direct superclasses")))

(define-inspector-display-mode 
  (class-superclass-inspector :precedence)
  :command-name "Precedence List"
  :compute-line-count
  ((i)
   (1+ (length (class-precedence-list (inspector-object i)))))
  :line-n
  ((i n)
   (class-superclass-display-common 
    n (class-precedence-list (inspector-object i))
    "Precedence List:"
    "No Superclasses")))

(define-inspector-display-mode
  (class-superclass-inspector :superclass-tree)
  :command-name "Superclass Tree"
  :compute-line-count
  ((i)
   (unless (slot-boundp i 'tree-info)
     (setf (slot-value i 'tree-info)
           (build-tree-info (inspector-object i) #'class-direct-superclasses)))
   (1+ (length (slot-value i 'tree-info))))
  :line-n
  ((i n)
   (with-slots (tree-info) i
     (cond ((zerop n)
            (values "" "Superclasses:" :comment))
           ((<= 1 n (length tree-info))
            (let* ((entry (nth (1- n) tree-info))
                   (class (second entry))
                   (indentation (first entry)))
              (flet ((print-it (inspector stream value label type)
                       (dotimes (j (* 2 indentation))
                         (write-char #\space stream))
                       (print-class-proper-name inspector stream value 
                                                label type)))
                (values class "" :static #'print-it))))
           (t (values "" "TOO FAR" :comment))))))


(defun class-superclass-display-common (n superclasses
                                          description no-description)
  (cond ((= n 0)
         (if superclasses
           (values "" description :comment)
           (values "" no-description :comment)))
        ((<= 1 n (length superclasses))
         (values (nth (1- n) superclasses) "" :static
                 #'print-class-proper-name))
        (t (values "" "TOO FAR" :comment))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; show class slots.  mode for direct or effective slots

(defclass class-slots-inspector (modal-display-mixin inspector) 
  ()
  (:default-initargs :mode :direct))

(defun real-class-direct-slots (class)
  ;; Why do we need to do this?
  (cdr (class-direct-slots class)))

(define-inspector-display-mode 
  (class-slots-inspector :direct)
  :command-name "Direct Slots"
  :compute-line-count
  ((i)
   (1+ (length (real-class-direct-slots (inspector-object i)))))
  :line-n
  ((i n)
   (class-slot-display-common n
                              (real-class-direct-slots (inspector-object i)) 
                              "No direct slots"
                              "Direct Slots:")))

(define-inspector-display-mode
  (class-slots-inspector :effective)
  :command-name "Effective Slots"
  :compute-line-count
  ((i)
   (1+ (length (class-slots (inspector-object i)))))
  :line-n
  ((i n)
   (class-slot-display-common n
                              (class-slots (inspector-object i))
                              "No slots"
                              "Effective Slots:")))

(defun class-slot-display-common (n slot-descriptions no-description 
                                    description)
  (cond ((= n 0) 
         (values ""
                 (if slot-descriptions
                   description
                   no-description)
                 :comment))
        ((<= 0 (1- n) (length slot-descriptions))
         (values (nth (1- n) slot-descriptions) "" :static
                 #'print-class-slot-description))
        (t (values "" "TOO FAR" :comment))))

(defun print-class-slot-description (inspector stream slotd label type)
  (declare (ignore inspector label type))
  (let ((name (first slotd))
        (initform (second slotd))
        (initargs (third slotd)))
    (format stream "~s:  initargs:~{ ~s~}  initform: ~s"
            name
            initargs
            initform)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; utility for class tree displays
;;; This could also be used to inspect the view hierarchy.
(defun build-tree-info (node branches-function)
  (let ((nodes nil))
    (labels ((do-node (node depth)
               (push (list depth node) nodes)
               (dolist (n (funcall branches-function node))
                 (do-node n (1+ depth)))))
      (do-node node 0))
    (nreverse nodes)))

          
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; new features for method-inspector

(defmethod inspector-commands :around ((i method-inspector))
  (let ((method (inspector-object i)))
    (flet ((kill-this-method ()
             (when (y-or-n-dialog (format nil "Remove the method ~s?" method)
                                  :yes-text "Undefine the method" 
                                  :no-text "Keep it"
                                  :cancel-text nil
                                  ;:help-spec (format nil "Permanently remove the method ~s" method)
                                  )
               (remove-method (method-generic-function method) method))))
      (append (call-next-method)
             `(("Kill Method" ,#'kill-this-method))))))
