;;; -*- Mode: LISP; Package: (LGL); Syntax:Common-Lisp; Lowercase: Yes -*-

;;; LGL.lisp  0.5
;;; Daniel LaLiberte
;;; liberte@ncsa.uiuc.edu

#|
LGL is a Lisp implementation of an IRIS-like Graphics Library.  
For more complete documentation on how to use LGL, consult the GL
manuals.  I only describe some differences here.

LGL supports only the bare minimum of functionality that I required, 
but also modifies and enhances the library so that it is more convenient 
to use from Lisp.  Some functions like "clear" conflict with Lisp
symbols, so I preceed them with "GL".

Since Lisp is interactive and MCL multi-threaded, it
is important to recover from errors that normally would crash a C program.
Therefore, instead of using the pair pushmatrix followed by popmatrix, I
provide a macro pushing-matrix that does pops in an unwind-protect.
Similarly, making-object should be used instead of makeobj and closeobj.

Points are vectors, either 3 or 4 long depending on which is required.  
In addition to functions like draw that take three arguments for x, y,
and z values of the coordinate, I provide vector versions of the functions,
e.g. drawv, that take one point vector.

LGL tries to assume that only floating point values are used, so some
day it could be optimized.  If you don't always provide floating point
values, LGL may give you an error, but currently it won't crash.

Not all of the functions in the export list are 
implemented, but those that are not are highest in the priority list.
If you want to extend LGL, please send your changes back to me so
I may include them.

Actually, I'd rather be using a C library for speed and in fact we are working
on interfacing with VOGL.  But LGL performs pretty well.

This is my first Common Lisp package, so I probably did several things
"wrong".  Please let me know so I can correct it.

|#

;; Package definition of LGL

(in-package :CCL)

(unless (find-package "LGL")
  (defpackage "LGL" 
    (:size 30))
  )


(in-package :LGL)

(export
 '(
   backface
   callobj
   GLclear
   closeobj
   GLcolor
   delobj
   depthcue
   draw
   drawv
   foreground
   genobj
   getdcm
   getgpos
   getmatrix
   invertmatrix
   loadmatrix
   lshaderange
   makeobj
   making-object
   mapcolor
   GLmove
   movev
   multmatrix
   multpoint
   ortho
   ortho2
   perspective
   polf
   GLpoly
   popmatrix
   prefposition
   pushmatrix
   rot-sin-cos
   rot
   rotv
   scale
   scalev
   translate
   translatev
   pushing-matrix
   winclose
   window
   winmove
   winopen
   winposition
   winset
   map-to-world
   ))


(require 'quickdraw)


(defparameter DEGTORAD (/ (acos -1.0) 180.0))



;;==================================================================
;; Basic data structures

(defvar LGL-cw nil
  "Current LGL window.  GL has a notion of a current window,
that is changed with winset.  An OO approach would probably be better.")

(defvar LGL-dw nil
  "Default LGL window - copied when a new window is created.")


(defvar LGL-window-alist nil
  "Alist of all LGL windows.")

(defvar LGL-display-mode nil
  "Int i if in display list mode for object i in current window.")


;; Structures for LGL

(defstruct LGLwin
  "LGL window structure, not the window itself."

  (viewport-tl) ; corners of active viewport, x, y
  (viewport-br)
  (viewport-size) ; size of active viewport
  (viewport-scale) ; viewport scaling
  (viewport-trans) ; viewport translation
  
  (backface-flag)
  (double-buffer-flag)
  
  (depthcue-flag)
  (draw-mode)
  
  (display-mode)   ; i if in display list mode for object i.
  (display-alist)  ; all display lists for this window.  could be hash table
  
;;  (GS-path) ; pipeline path through geometry engines
  (geom-type)
  
  (scs-config)  ; SCS configuration
  
  (vert) ; geometry subsystem vertex list
  (vcoor) ; view-coordinate point corresponding to vert
  
  (matrix-stack)
  )

;; Initialize the default LGL window.
(setf LGL-dw (make-LGLwin))
   

;;==================================================================
;; Object routines.
;; These have to be first since if-display-list is defined here.


(defmacro defdlfun (name args docstring &rest body)
  "Define a function that may be used in a display list.  
Currently depends on only seeing name, args, docstring, and body.
Help me fix this so docstring is optional."
  `(defun ,name ,args ,docstring
     (if LGL-display-mode
       (pushobj (list (quote ,name) ,@args))
       ;; else evaluate the body
       (progn ,@body))))

(defun pushobj (form)
  "Push form on the top of the current display list."
  (push form (cdr LGL-display-mode)))

(defun getobj (i)
  (assoc i (LGLwin-display-alist LGL-cw)))

(defun makeobj (i)
  "Switch to display list mode for the current window
which stores subsequent LGL calls in the display list numbered i."
  (if LGL-display-mode
    (error "makeobj: already in display list mode for object ~s."
           (car LGL-display-mode)))
  
  (delobj i)
  (push (cons i nil) (LGLwin-display-alist LGL-cw))
  (setq LGL-display-mode (assoc i (LGLwin-display-alist LGL-cw)))
  )

(defun closeobj ()
  "Switch display list mode off."
  (if (not LGL-display-mode)
    (error "closeobj: not in display list mode"))

  ;; Reverse the elements of the display list since they were
  ;; added in reverse order.
  (if LGL-display-mode
    (setf (cdr LGL-display-mode) (nreverse (cdr LGL-display-mode)))
    )
  (setq LGL-display-mode nil)
  )

(defun delobj (i)
  "Delete the ith object's display list."
  (let ((display-list-cons (assoc i (LGLwin-display-alist LGL-cw))))
    (if display-list-cons
      (setf (LGLwin-display-alist LGL-cw) 
            (delete display-list-cons (LGLwin-display-alist LGL-cw)))
    )))

(defdlfun callobj (i)
  "Call the ith object's display list."
  (let ((display-list-cons (assoc i (LGLwin-display-alist LGL-cw))))
    (if display-list-cons
      (mapcar 'eval (cdr display-list-cons)))))

(defmacro making-object (i &rest body)
  `(unwind-protect
     (progn
       (makeobj ,i)
       ,@body)
     (closeobj)))

;;==================================================================
;; Matrix routines

#|
The matrix stack is a vector with a fill-pointer.
In order to avoid garbage collection, the stack may grow, but
it does not shrink.
|#

(defun getmatrix ()
  "Returns the top matrix, not a copy of it."
  (let ((m (LGLwin-matrix-stack LGL-cw)))
    (aref m (1- (fill-pointer m)))))

(defun make-4-vector ()
  (make-array '(4) :element-type 'double-float))
  
(defun make-4x4-matrix ()
  (make-array '(4 4) :element-type 'double-float))
  
(defun make-4x4-matrix-with (contents)
  (make-array '(4 4) :element-type 'double-float
              :initial-contents contents))
  
(defun make-identity-matrix ()
  (make-4x4-matrix-with '((1.0 0.0 0.0 0.0)
                          (0.0 1.0 0.0 0.0)
                          (0.0 0.0 1.0 0.0)
                          (0.0 0.0 0.0 1.0))))


(defun make-matrix-stack ()
  (make-array '(1) :element-type '(array double-float (4 4))
              :adjustable t :fill-pointer 1
              :initial-contents (list (make-identity-matrix))))
                        

(defun multpoint (u v)
  "Set and return u to the product of v and the top matrix.
u can be the same as v."
  (let ((m (getmatrix))
        (x (aref v 0))
        (y (aref v 1))
        (z (aref v 2))
        (w (aref v 3)))
    (setf (aref u 0) (+ (* x (aref m 0 0))
                        (* y (aref m 1 0))
                        (* z (aref m 2 0))
                        (* w (aref m 3 0))))
    (setf (aref u 1) (+ (* x (aref m 0 1))
                        (* y (aref m 1 1))
                        (* z (aref m 2 1))
                        (* w (aref m 3 1))))
    (setf (aref u 2) (+ (* x (aref m 0 2))
                        (* y (aref m 1 2))
                        (* z (aref m 2 2))
                        (* w (aref m 3 2))))
    (setf (aref u 3) (+ (* x (aref m 0 3))
                        (* y (aref m 1 3))
                        (* z (aref m 2 3))
                        (* w (aref m 3 3))))
    u))

(defun set-matrix (m1 m2)
  "Set elements of m1 from m2, and return m1."
  (dotimes (i 16)
    (setf (row-major-aref m1 i) (row-major-aref m2 i)))
  m1)

(defun copy-matrix (m)
  "Return a copy of the 4 by 4 matrix M."
  (set-matrix (make-4x4-matrix) m))

(defdlfun loadmatrix (m)
  "Set the top matrix to m."
   (set-matrix (getmatrix) m))

(defdlfun multmatrix (m)
  "Multiply M by the top matrix of the current LGLwin."
   (let ((tmat (make-4x4-matrix)))
    ;; All this displaced array stuff may be slow, but it is less code.
    (multpoint (make-array '(4) :displaced-to tmat :displaced-index-offset 0)
               (make-array '(4) :displaced-to m :displaced-index-offset 0))
    (multpoint (make-array '(4) :displaced-to tmat :displaced-index-offset 4)
               (make-array '(4) :displaced-to m :displaced-index-offset 4))
    (multpoint (make-array '(4) :displaced-to tmat :displaced-index-offset 8)
               (make-array '(4) :displaced-to m :displaced-index-offset 8))
    (multpoint (make-array '(4) :displaced-to tmat :displaced-index-offset 12)
               (make-array '(4) :displaced-to m :displaced-index-offset 12))
    (loadmatrix tmat)))

(defdlfun pushmatrix ()
  "Push a copy of the top matrix onto the top of the matrix stack."
  (vector-push-extend (copy-matrix (getmatrix)) 
                      (LGLwin-matrix-stack LGL-cw)))

(defdlfun popmatrix ()
  "Pop the top matrix off the stack.
Coincidently returns the new top matrix.
The actual vector of matrices is not reduced in size."
  (vector-pop (LGLwin-matrix-stack LGL-cw)))

(defmacro pushing-matrix (&rest body)
  `(unwind-protect
     (progn
       (pushmatrix)
       ,@body)
     (popmatrix)))


;; For use by picking.
(defun invertmatrix ()
  "Return the inverse matrix for the matrix on the top of the stack."
  (invert-4x4-matrix (getmatrix)))


  
;;==================================================================
;; Window routines

(defvar LGL-highest-wid 0)

(defun winopen ()
  "Open an LGL window, which is really just an internal structure here.
Return its window id."
  (let ((new-id (incf LGL-highest-wid)))
    (push (cons new-id (setf LGL-cw (copy-LGLwin LGL-dw))) LGL-window-alist)
    (setf (LGLwin-matrix-stack LGL-cw) (make-matrix-stack))
    (setf (LGLwin-vert LGL-cw) (vector 0 0 0 0))
    new-id))

(defun winset (n)
  "Set the current LGL window to window number n."
  (setf LGL-cw 
        (cdr (assoc n LGL-window-alist))))

(defun winclose (n)
  "Delete the LGL window numbered n."
  (let ((w (assoc n LGL-window-alist)))
    (if w (setf LGL-window-alist (delete w LGL-window-alist))))
  (setf LGL-cw nil))


(defun prefposition (x1 x2 y1 y2)
  "Define the preferred position of a new window, in screen coordinates."
  (if (or (< x1 0)
          (< x2 0)
          (< x2 x1))
    (error "prefposition: bad x values: x1=~s; x2=~s" x1 x2))

  (if (or (< y1 0)
          (< y2 0)
          (< y2 y1))
    (error "prefposition: bad y values: y1=~s; y2=~s" y1 y2))

  (setf (LGLwin-viewport-tl LGL-dw) (vector x1 y1)
        (LGLwin-viewport-br LGL-dw) (vector x2 y2)
        (LGLwin-viewport-size LGL-dw) (vector 
                                       (- x2 x1)  ; + 1?
                                       (- y2 y1)  ; + 1?
                                       )
        (LGLwin-viewport-scale LGL-dw) (vector 
                                       (* (- x2 x1) 0.5)
                                       (* (- y2 y1) 0.5)
                                       )
        (LGLwin-viewport-trans LGL-dw) (vector 
                                       (* (+ x2 x1) 0.5)
                                       (* (+ y2 y1) 0.5)
                                       )
        )
  LGL-dw
  )

;;==================================================================
;; View routines

(defun ortho2 (left right bottom top)
  "Set the top matrix to the 2D orthogonal view transformation."
  (let* ((left-right (- left right))
         (bottom-top (- bottom top))
         (new-matrix 
          (make-4x4-matrix-with 
           (list (list (/ -2.0 left-right) 0.0 0.0 0.0)
                 (list 0.0 (/ -2.0 bottom-top) 0.0 0.0)
                 (list 0.0 0.0 -1.0 0.0)
                 (list (/ (+ right left) left-right)
                       (/ (+ top bottom) bottom-top)
                       0.0 1.0)))))
    (loadmatrix new-matrix)))


(defun ortho (left right bottom top near far)
  "Set the top matrix to the orthogonal view transformation."
  (let* ((left-right (- left right))
         (bottom-top (- bottom top))
         (near-far (- near far))
         (new-matrix 
          (make-4x4-matrix-with 
           (list (list (/ -2.0 left-right) 0.0 0.0 0.0)
                 (list 0.0 (/ -2.0 bottom-top) 0.0 0.0)
                 (list 0.0 0.0 (/ 2.0 near-far) 0.0)
                 (list (/ (+ right left) left-right)
                       (/ (+ top bottom) bottom-top)
                       (/ (+ far near) near-far)
                       1.0)))))
    (loadmatrix new-matrix)))

(defun perspective (fovy aspect near far)
  "Set the top matrix to the perspective transformation."
  (let* ((height (/ 1.0 (tan (/ (* DEGTORAD fovy) 20.0))))
         (near-far (- near far))
         (new-matrix 
          (make-4x4-matrix-with 
           (list (list (/ height aspect) 0.0 0.0 0.0)
                 (list 0.0 height 0.0 0.0)
                 (list 0.0 0.0 (/ (+ far near) near-far) -1.0)
                 (list 0.0 0.0 (/ (* 2.0 far near) near-far) 0.0)))))
    (loadmatrix new-matrix)))


;;==================================================================
;; Transformation routines.

(defdlfun translate (x y z)
  "Translate by X Y Z."
  (let ((u (make-4-vector))
        (v (vector x y z 1.0))
        (m (getmatrix)))
    (multpoint u v)
    ;; Copy u back into m[3;]
    (setf (aref m 3 0) (aref u 0))
    (setf (aref m 3 1) (aref u 1))
    (setf (aref m 3 2) (aref u 2))
    (setf (aref m 3 3) (aref u 3))
    m
    ))

(defdlfun translatev (v)
  "Translate by the vector v."
  (translate (aref v 0) (aref v 1) (aref v 2)))

(defdlfun scale (x y z)
  "Scale by X Y Z."
  (let ((m (getmatrix))
        )
    ;; Scale values in-place.
    (setf (aref m 0 0) (* x (aref m 0 0)))
    (setf (aref m 0 1) (* x (aref m 0 1)))
    (setf (aref m 0 2) (* x (aref m 0 2)))
    (setf (aref m 0 3) (* x (aref m 0 3)))
    (setf (aref m 1 0) (* y (aref m 1 0)))
    (setf (aref m 1 1) (* y (aref m 1 1)))
    (setf (aref m 1 2) (* y (aref m 1 2)))
    (setf (aref m 1 3) (* y (aref m 1 3)))
    (setf (aref m 2 0) (* z (aref m 2 0)))
    (setf (aref m 2 1) (* z (aref m 2 1)))
    (setf (aref m 2 2) (* z (aref m 2 2)))
    (setf (aref m 2 3) (* z (aref m 2 3)))
    m
    ))

(defdlfun scalev (v)
  "Scale by the vector v."
  (scale (aref v 0) (aref v 1) (aref v 2)))


(defvar vi (make-4-vector))
(defvar vj (make-4-vector))
(defvar tvi (make-4-vector))

(defdlfun rot-sin-cos (s c axis)
  "Rotate by angle given the sin and cos of the angle, around AXIS."
  (let* ((i (aref #(1 2 0) axis)) ;0->1 1->2 2->0
         (j (aref #(2 0 1) axis)) ;0->2 1->0 2->1
         (m (getmatrix))
         (ui (adjust-array vi '(4) 
                           :displaced-to m :displaced-index-offset (* i 4)))
         (uj (adjust-array vj '(4) 
                           :displaced-to m :displaced-index-offset (* j 4)))
         )
    ;; Rotate in-place, with temporary vector.
    (setf (aref ui 0) (+ (* c (setf (aref tvi 0) (aref ui 0))) (* s (aref uj 0))))
    (setf (aref ui 1) (+ (* c (setf (aref tvi 1) (aref ui 1))) (* s (aref uj 1))))
    (setf (aref ui 2) (+ (* c (setf (aref tvi 2) (aref ui 2))) (* s (aref uj 2))))
    (setf (aref ui 3) (+ (* c (setf (aref tvi 3) (aref ui 3))) (* s (aref uj 3))))
    
    (setf (aref uj 0) (- (* c (aref uj 0)) (* s (aref tvi 0))))
    (setf (aref uj 1) (- (* c (aref uj 1)) (* s (aref tvi 1))))
    (setf (aref uj 2) (- (* c (aref uj 2)) (* s (aref tvi 2))))
    (setf (aref uj 3) (- (* c (aref uj 3)) (* s (aref tvi 3))))
    m
    ))

(defdlfun rot (angle axis)
  "Rotate by angle on axis.  0==x 1==y 2==z."
  ;; Axis is a bit different from GL, but possibly more useful.
  (let ((theta (* angle DEGTORAD)))
    (rot-sin-cos (sin theta) (cos theta) axis)))


(defdlfun rotv (v)
  "Rotate by the vector v of angles, in the order x y z."
  (rot (aref v 0) 0)
  (rot (aref v 1) 1)
  (rot (aref v 2) 2))


;;==================================================================
;; Drawing routines.

;; Assumes that *current-view* is set to the view to draw in.

(defvar temp-vector (make-4-vector)
  "Temporary vector.")

(defun set-temp-vector (x y z)
  (setf (aref temp-vector 0) x)
  (setf (aref temp-vector 1) y)
  (setf (aref temp-vector 2) z)
  (setf (aref temp-vector 3) 1.0))


(defun current-point ()
  "Return the current graphics position vector."
  (LGLwin-vert LGL-cw))

(defdlfun project (x y z)
  "Return the projection point of (x y z) onto the screen."
  (set-temp-vector x y z)
  (normal-to-screen (do-perspective 
                     (multpoint temp-vector temp-vector))))

(defdlfun projectv (v)
  (project (aref v 0) (aref v 1) (aref v 2)))

(defdlfun GLmove (x y z)
  "Move to point (x y z) using the current transformation matrix."
  (set-temp-vector x y z)
  (multpoint (LGLwin-vert LGL-cw) temp-vector)
  (setf (LGLwin-vcoor LGL-cw)
        (normal-to-screen (do-perspective (current-point)))))

(defdlfun movev (v)
  "Move to point in vector v just like GLmove."
  (GLmove (aref v 0) (aref v 1) (aref v 2)))

(defdlfun draw (x y z)
  "Draw from the current point to point (x y z) 
and leave current point at (x y z)."
  (move-to *current-view* (LGLwin-vcoor LGL-cw))
  (GLmove x y z)
  (line-to *current-view* (LGLwin-vcoor LGL-cw))
  )

(defdlfun drawv (v)
  "Draw to point in vector v just like draw"
  (draw (aref v 0) (aref v 1) (aref v 2)))

(defun do-perspective (v)
  "Apply perspective to v and return v.
This same function undoes the perspective transformation too."
  (let ((w (setf (aref v 3) (/ 1.0 (aref v 3)))))
    (setf (aref v 0) (* w (aref v 0)))
    (setf (aref v 1) (* w (aref v 1)))
    (setf (aref v 2) (* w (aref v 2)))
    )
  v)

(defun normal-to-screen (v)
  "Return a MCL point which results from converting v
from normal to screen coordinates."
  (let ((scale (LGLwin-viewport-scale LGL-cw))
        (trans (LGLwin-viewport-trans LGL-cw))
        (sizey (aref (LGLwin-viewport-size LGL-dw) 1)))
    (make-point (truncate (+ (aref trans 0) (* (aref scale 0) (aref v 0))))
                (truncate 
                 (- sizey ;; on many pc's, y axis is upside-down
                    (+ (aref trans 1) (* (aref scale 1) (aref v 1))))))))
 

(defdlfun polf (points)
  "Draw a filled polygon.  Always uses white pattern for now."
  (let ((length (length points))
        (poly))
    (start-polygon *current-view*)
    (movev (aref points 0))
    (dotimes (i (1- length))
      (drawv (aref points (1+ i))))
    (drawv (aref points 0))
    (setq poly (get-polygon *current-view*))
    (fill-polygon *current-view* *white-pattern* poly)
    (kill-polygon poly)
    ))

(defdlfun GLpoly (points)
  "Draw a polygon frame."
  (let ((length (length points))
        (poly))
    (start-polygon *current-view*)
    (movev (aref points 0))
    (dotimes (i (1- length))
      (drawv (aref points (1+ i))))
    (drawv (aref points 0))
    (setq poly (get-polygon *current-view*))
    (frame-polygon *current-view* poly)
    (kill-polygon poly)
    ))


;;==================================================================
;; Picking routines.
;; Picking is broken into two parts: setup to create an inverse
;; matrix and repeated application of a function to map from
;; screen x y coordinates backward into a line in world coordinates.


(defun screen-to-normal (x &optional y)
  "Return the normalized x and y for screen coordinates (x y).
This assumes you have already inverted the matrix on top of the stack."
  (let ((scale (LGLwin-viewport-scale LGL-cw))
        (trans (LGLwin-viewport-trans LGL-cw))
        (sizey (aref (LGLwin-viewport-size LGL-dw) 1)))
    (unless y
      (setq y (point-v x))
      (setq x (point-h x)))
    ;; x and y are now separate x and y
    (setq x (/ (- x (aref trans 0)) (aref scale 0))
          y (/ (- (- sizey y) (aref trans 1)) (aref scale 1)))
    )
;;  (print (format nil "normal x: ~S y: ~S" x y))
  (values x y))



(defun map-to-world (x &optional y)
  "Return two vectors for the points at each end of the line
through the world at screen coordinates (x y).
This assumes you have already inverted the matrix on top of the stack.
The result vectors are 4 long.
The first result vector is at the front."
  (multiple-value-setq (x y) (screen-to-normal x y))
  (values 
   (do-perspective (multpoint (make-4-vector) (vector x y 1.0 1.0)))
   (do-perspective (multpoint (make-4-vector) (vector x y -1.0 1.0)))))



#|
Matrix Inversion
Algorithm by Richard Carling
from "Graphics Gems", Academic Press, 1990
Translated to CL by Daniel LaLiberte
|#

(defun det2x2 (a b c d)
  "Return the determinent of a 2x2 matrix
in the form

    | a b |
    | c d |
"
  (- (* a d) (* b c)))


(defun det3x3 (a1 a2 a3 b1 b2 b3 c1 c2 c3)
  "Return the determinent of a 3x3 matrix
in the form

     | a1  b1  c1 |
     | a2  b2  c2 |
     | a3  b3  c3 |
"
  
  (+ (- (* a1 (det2x2 b2 b3 c2 c3))
        (* b1 (det2x2 a2 a3 c2 c3)))
     (* c1 (det2x2 a2 a3 b2 b3))))


(defun det4x4 (m)
  "Return the determinent of a 4x4 matrix."
  
  #| assign to individual variable names to aid selecting |#
  #|  correct elements |#
  (let ((a1 (aref m 0 0)) (b1 (aref m 0 1))
        (c1 (aref m 0 2)) (d1 (aref m 0 3))
        
        (a2 (aref m 1 0)) (b2 (aref m 1 1))
        (c2 (aref m 1 2)) (d2 (aref m 1 3))
        
        (a3 (aref m 2 0)) (b3 (aref m 2 1))
        (c3 (aref m 2 2)) (d3 (aref m 2 3))
        
        (a4 (aref m 3 0)) (b4 (aref m 3 1))
        (c4 (aref m 3 2)) (d4 (aref m 3 3)))
    
    (- (+ (* a1 (det3x3 b2 b3 b4 c2 c3 c4 d2 d3 d4))
          (* c1 (det3x3 a2 a3 a4 b2 b3 b4 d2 d3 d4)))
       (* b1 (det3x3 a2 a3 a4 c2 c3 c4 d2 d3 d4))
       (* d1 (det3x3 a2 a3 a4 b2 b3 b4 c2 c3 c4)))
    ))


#|
 *   (matrix-adjoint original_matrix)
 *
 *     calculate the adjoint of a 4x4 matrix
 *
 *      Let  a   denote the minor determinant of matrix A obtained by
 *           ij
 *
 *      deleting the ith row and jth column from A.
 *
 *                    i+j
 *     Let  b   = (-1)    a
 *          ij            ji
 *
 *    The matrix B = (b ) is the adjoint of A
 *                     ij
 |#

(defun matrix-adjoint (m)
  "Create and return the adjoint of the 4x4 matrix IN."
  
  #| assign to individual variable names to aid  |#
  #| selecting correct values  |#
  
  (let ((out (make-array '(4 4) :element-type 'double-float))
        (a1 (aref m 0 0)) (b1 (aref m 0 1))
        (c1 (aref m 0 2)) (d1 (aref m 0 3))
        
        (a2 (aref m 1 0)) (b2 (aref m 1 1))
        (c2 (aref m 1 2)) (d2 (aref m 1 3))
        
        (a3 (aref m 2 0)) (b3 (aref m 2 1))
        (c3 (aref m 2 2)) (d3 (aref m 2 3))
        
        (a4 (aref m 3 0)) (b4 (aref m 3 1))
        (c4 (aref m 3 2)) (d4 (aref m 3 3)))
    
    
    #| row column labeling reversed since we transpose rows & columns |#
    
    (setf (aref out 0 0)     (det3x3 b2 b3 b4 c2 c3 c4 d2 d3 d4))
    (setf (aref out 1 0) ( - (det3x3 a2 a3 a4 c2 c3 c4 d2 d3 d4)))
    (setf (aref out 2 0)     (det3x3 a2 a3 a4 b2 b3 b4 d2 d3 d4))
    (setf (aref out 3 0) ( - (det3x3 a2 a3 a4 b2 b3 b4 c2 c3 c4)))
    
    (setf (aref out 0 1) ( - (det3x3 b1 b3 b4 c1 c3 c4 d1 d3 d4)))
    (setf (aref out 1 1)     (det3x3 a1 a3 a4 c1 c3 c4 d1 d3 d4))
    (setf (aref out 2 1) ( - (det3x3 a1 a3 a4 b1 b3 b4 d1 d3 d4)))
    (setf (aref out 3 1)     (det3x3 a1 a3 a4 b1 b3 b4 c1 c3 c4))
    
    (setf (aref out 0 2)     (det3x3 b1 b2 b4 c1 c2 c4 d1 d2 d4))
    (setf (aref out 1 2) ( - (det3x3 a1 a2 a4 c1 c2 c4 d1 d2 d4)))
    (setf (aref out 2 2)     (det3x3 a1 a2 a4 b1 b2 b4 d1 d2 d4))
    (setf (aref out 3 2) ( - (det3x3 a1 a2 a4 b1 b2 b4 c1 c2 c4)))
    
    (setf (aref out 0 3) ( - (det3x3 b1 b2 b3 c1 c2 c3 d1 d2 d3)))
    (setf (aref out 1 3)     (det3x3 a1 a2 a3 c1 c2 c3 d1 d2 d3))
    (setf (aref out 2 3) ( - (det3x3 a1 a2 a3 b1 b2 b3 d1 d2 d3)))
    (setf (aref out 3 3)     (det3x3 a1 a2 a3 b1 b2 b3 c1 c2 c3))
    out
    ))


#|
 *   (invert-4x4-matrix 4x4-matrix)
 *
 *    calculate the inverse of a 4x4 matrix
 *
 *     -1
 *     A  = ___1__ adjoint A
 *         det A
 |#

(defparameter SMALL_NUMBER 1.0E-19)

(defun invert-4x4-matrix (m)
  "Return the inverse of the 4x4 matrix M."
  
  (let* ((out (matrix-adjoint m))     ; calculate the adjoint matrix
         (det (det4x4 out)))    ; calculate the 4x4 determinent
    
    ;;  if the determinent is zero,
    ;;  then the inverse matrix is not unique.
    (if (< (abs det) SMALL_NUMBER)
      (error "Non-singular matrix with determinent ~S has no inverse!" det)
      )
    
    #| scale the adjoint matrix to get the inverse |#
    
    (dotimes (i 16)
      (setf (row-major-aref out i) (/ (row-major-aref out i) det)))
    out))


