;;; -*- Mode:Common-Lisp; Package:ZWEI; Base:10 -*-

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

(load-tools '(Development-Tool-Consistency-Enhancements))
(require 'Development-Tool-Consistency-Enhancements)

(defvar *Zmacs-Enhancements-to-add* :All
"Can have the value :All, in which case all commands are loaded,
 :Menu in which case the user is prompted, or a list of commands to load.
")

(defparameter *Zmacs-Enhancements-Commands*
  '(("Rotate Buffer" :Value (((#\c-sh-l com-rotate-buffer)) nil nil))
    ("Rotate Buffer Backwards" :Value
     (((#\m-sh-l com-rotate-buffer-backwards)) nil nil)
    )
    ("Show Optimised Code" :Value
     (((#\s-sh-m com-show-optimised-code)) nil nil)
    )
    ("Check Args In Expression" :Value
     (((#\m-sh-a com-check-args-in-expression)) nil nil)
    )
    ("Indent Rigidly" :Value
     (((#\m-c-sh-i com-mouse-indent-rigidly)) nil nil)
    )
   )
"This is a list of the commands which can be added to Zmacs.  The structure
 of this list is as follows :-
 It is made of items.  Each item is a list of the form (Menustring :Value spec)
 the spec is used to determine which commands for which tools is represented
 by the menu item.  The spec is a list of the form
 (zmacs-commands nil nil)
 Each element of zmacs-commands has the form (key command-name).
"
)

(defun select-and-install-zmacs-commands (commands switch)
"Is passed a list of all of the commands that can be added and a switch
 which tells the system what commands to add.  This switch can have the values
 :Menu, :All or it can be a list of items.  Each item can be an element from
 Commands, a subitem from Commands (the bit following the :Value) or a string
 denoting the name of one of the menu items.
" 
  (let ((selected-items
	  (if (equal switch :Menu)
	      (w:multiple-menu-choose commands :Label "Select commands")
	      (if (equal switch :All)
		  (mapcar #'Third commands)
		  (mapcar #'(lambda (command)
			      (if (consp command)
				  (if (stringp (first command))
				      (third command)
				      command
				  )
				  (if (stringp command)
				      (let ((entry (assoc command commands
							  :Test #'String-Equal
						   )
					    )
					   )
					   (if entry
					       (third entry)
					       (ferror nil
 "~S is not a defined command to add." command)
					   )
				      )
				      (ferror nil 
 "~S is not a valid command specifier." command)
				  )
			      )
			    )
			    (if (consp switch)
				switch
				(ferror nil
 "~S is not :Menu :All or a list of commands" switch)
			    )
		  )
	      )
	  )
        )
       )
       (mapcar #'tv:install-command selected-items)
  )
)


(defun install-Zmacs-commands ()
"Installs all of the commands that the user wants.
"
  (select-and-install-zmacs-commands *Zmacs-Enhancements-Commands*
				     *Zmacs-Enhancements-to-add*
  )
)

;-------------------------------------------------------------------------------

;;; Many thanks to Jamie Zawinski for this better version.

(defun compile-macro-callers (macro-name &optional (*interval* *interval*))
  (resectionize-buffer *interval*)
  (let* ((done-any nil))
    (dolist (section (node-inferiors *interval*))
      (let* ((name (section-node-name section)))
	(let ((ok-p
		(catch-error
		  ;;; Some sections may not be compilable for some reason.
		  (multiple-value-bind (fdefinition dbis)
		      (sys:fdefinition-safe name t t)
		    (when fdefinition
		      (let ((macros
			      (and dbis
				   (sys:get-debug-info-field
				     dbis :macros-expanded))))
			(when (member
				macro-name macros :test #'equal
				; equal coz macro could be (:prop...)
				:key #'(lambda (x) (if (consp x) (car x) x)))
			  (setq done-any t)
			  (sys:file-operation-with-warnings
			    ((and (buffer-file-id *interval*)
				  (send (send *interval* :generic-pathname)
					:generic-pathname))
			     :compile nil)
			    (compiler:compiler-warnings-context-bind
			      (compile-print-interval
				section nil t t nil "Compiling" "compiled."
				t t nil t))))))
		    t) nil))) ;; OK
	  (unless ok-p
	    (format t "~&An error occurred during the compilation of ~S"
		    name)))))
    done-any))

;;; JWZ
(defcom com-compile-macro-callers
   "Recompile all functions in this buffer which use a prompted-for macro." ()
  (let* ((spec (read-function-name
		 "Recompiler callers of:"
		 (relevant-function-name (point) nil t t t) t nil
		 #.(string-append
		     "Type the name of the macro or inline function for "
		     "which you wish to recompile all callers in this "
		     "buffer."))))
    (if (compile-macro-callers spec)
	(format t "~&Done.~%")
	(barf "No callers of ~S in this buffer." spec)))
  dis-none)

(set-comtab *standard-comtab* ()
	    (make-command-alist '(com-compile-macro-callers))
)

;-------------------------------------------------------------------------------

;;; JPR modified from JWZ.
(defcom com-tags-compile-macro-callers
   "Recompile all functions in a tag table which use a prompted-for macro." ()
  (let* ((spec (read-function-name
		 "Recompiler callers of:"
		 (relevant-function-name (point) nil t t t) t nil
		 #.(string-append
		     "Type the name of the macro or inline function for "
		     "which you wish to recompile all callers in this "
		     "buffer.")))
	 (found-p nil)
	 (buffer (bp-top-level-node (next-file-bp t))))
         ;;; Force to start of tag table.
    (loop when (compile-macro-callers spec buffer)
	  do (setq found-p t)
	  do (setq buffer (bp-top-level-node (next-file-bp nil)))
	  while buffer)
    (if found-p
	(format t "~&Done.~%")
	(barf "No callers of ~S in this tag table." spec)))
  dis-none)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-tags-compile-macro-callers))
)


;-------------------------------------------------------------------------------

(defparameter *show-optimised-code* nil
"When true the optimised version of the code being compiled is printed out."
)

(defun optimise-1 (form optimisers)
"Takes a form and a list of optimisers for that form.  It calls the
 optimisers on that form successively, catching any errors that occur
 during the optimisation process.
"
  (if optimisers
      (multiple-value-bind (result errorp)
	  (catch-error (funcall (first optimisers) form))
	(optimise-1 (if errorp form result) (rest optimisers))
      )
      form
  )
)

(defun optimise (fname args)
"Returns the optimised version of the call (fname . args) if there are any
 compiler optmisers for fname.
"
  (let ((optimisers (get fname 'compiler:optimizers)))
       (optimise-1 (cons fname args)
		   (if (consp optimisers)
		       optimisers
		       (if optimisers
			   (list optimisers)
			   nil
		       )
		   )
       )
  )
)

(defun get-optimised-code-for (form)
"Returns that code into which form will be optimised by any compiler optimisers,
 which are defined for any of its components.
"
  (if (consp form)
      (if (equal (first form) 'si:dont-optimize)
	  form
	  (let ((args (mapcar #'get-optimised-code-for (rest form))))
	       (if (symbolp (first form))
		   (let ((result (optimise (first form) args)))
			(if (equalp (cons (first form) args) result)
			    result
			    (get-optimised-code-for result)
			)
		   )  
		   (cons (get-optimised-code-for (first form)) args)
	       )
	  )
      )
      form
  )
)

(advise-within compiler:qc-translate-function compiler:qcompile1 :Around
	       :Show-Optimisations nil
  (if *show-optimised-code*
      (pprint (get-optimised-code-for
		(compiler:compiland-definition compiler:*current-compiland*)
	      )
      )
      nil
  )
  :Do-It
)


(DEFCOM COM-Show-optimised-code
"Print optimised lisp code of current definition or selected region.
The result is printed on the screen with PPrint." ()
  (let ((*show-optimised-code* t))
       (Com-Compile-Region)
  )
  dis-none
)


;-------------------------------------------------------------------------------


(DEFCOM COM-MACRO-EXPAND-Into-Buffer
"Print macroexpansion of next s-expression or marked s-expression(s) into the
 current buffer.  The result is printed on the screen with GRIND-TOP-LEVEL.
 If a numeric arg is supplied then that number of macroexpansions is applied." 
 ()
  (let ((form (read-form-or-forms-from-buffer)))
       (and (eq form :Nothing-found) (barf))
       (let ((expanded-form 
	       (if *numeric-arg-p*
		   (expand-n-times (macroexpand form) (- *numeric-arg* 1))
		   (macroexpand form)
	       )
	     )
	    )
	    (let ((stream (interval-stream (point) (point) t)))
		 (unwind-protect
		     (progn (grind-top-level expanded-form 80 stream)
			    (terpri stream)
		     )
		   (close stream)
		 )
	    )
       )
  )
  dis-text
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(COM-MACRO-EXPAND-Into-Buffer))
)

;-------------------------------------------------------------------------------

(defcom com-rotate-buffer
"Select the previously selected buffer and put the current buffer on the
bottom of the buffer list." ()
  (rotate-buffer-history 1))


(defcom com-rotate-buffer-backwards
"Select the buffer on the bottom of the buffer history list and push the
current one down one." ()
  (rotate-buffer-history -1))


;-------------------------------------------------------------------------------

(DEFCOM COM-Add-file-to-tag-table
 "Prompt for a file name and add that file to the current tag table." ()
  (let ((file-list
	  (send (select-tag-table) :get 'zmacs-tag-table-file-symbols)
	)
	(pathname (read-defaulted-pathname "File to add:" (pathname-defaults)
					   nil :newest :new-ok
		  )
	)
       )
       (if (pathnamep pathname)
	   (if (member pathname file-list)
	       (format *query-io*
		       "The file is already in the current tag table.")
	       (if (probe-file pathname)
		   (send (select-tag-table) :Putprop (cons pathname file-list)
			 'zmacs-tag-table-file-symbols
		   )
		   (progn
		     (format *query-io* "The file ~A does not exist." pathname)
		     (beep)
		   )
	       )
	   )
	   (beep)
       )
  )
  Dis-None
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Add-File-To-Tag-Table))
)


;-------------------------------------------------------------------------------


(DEFCOM COM-Remove-file-from-tag-table
 "Prompt for a file name and remove that file from the current tag table." ()
  (let ((file-list
	  (send (select-tag-table) :get 'zmacs-tag-table-file-symbols)
	)
	(pathname (read-defaulted-pathname "File to remove:" (pathname-defaults)
					   nil :newest :new-ok
		  )
	)
       )
       (if (pathnamep pathname)
	   (if (member pathname file-list)
	       (send (select-tag-table) :Putprop (remove pathname file-list)
		     'zmacs-tag-table-file-symbols
	       )
	       (format *query-io* "The file ~A is not in the current tag table."
		       pathname
	       )
	   )
	   (beep)
       )
  )
  Dis-None
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Remove-File-From-Tag-Table))
)

;-------------------------------------------------------------------------------



(defcom com-evaluate-and-insert-into-buffer
" Evaluate the next or selected s-expression and insert the result into the
 buffer before the original expression."
	()
    (let ((form (read-form-or-forms-from-buffer)))
         (if (equal form :Nothing-Found)
	     (barf "Unbalanced parentheses or no Form.")
	 )
	 (setq form (si:eval-abort-trivial-errors form)) ; si:eval1 internally.
	 (let ((stream (rest-of-interval-stream (point))))
	      (format stream "~&~S~&" form)
	      (move-bp (point) (send stream :read-bp))
	 )
    )
    dis-text
)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-evaluate-and-insert-into-buffer))
)


;-------------------------------------------------------------------------------


;;; The following patch allows the modular addition of new items with defined
;;; section names.  The purpose of this is that if you have a macro called, say,
;;; Define-Method, which expands into a DefMethod, then when a file containing
;;; calls to this macro is loaded into ZMacs the macro name will not be
;;; understood by ZMacs and so a section node will be created with the name
;;; (:Property...) instead of (:Method...).  This means that if you do an edit
;;; definition on such a method ZMacs will not be able to find it.  This mod
;;; allows you to get around this problem by defining that the Define-Method
;;; section is like a DefMethod.


(defvar *section-defining-items* nil
"A list of specifiers for determining the name of a section from a line, if the
 system has failed to name it from any of the predefined types.
 The list can have elements of two types: a) a three-list, of the form
 (fixnum string result-type) such as (10 \"(DEFMETHOD\" :Always-Method) in
 this case the normal test is applied using the string and number for
 comparison purposes or b) a function, which is called with the line as
 its argument.  The third item of the list can be any of :Maybe-Method
 :Always-Method :Destruct :Defselect :Method :Handler :Defwrapper
 :Property :Internal.  If the third item is anything other that this (a user-
 defined, non-standard section type then a section node naming function must be
 declared and added to the Alist *Non-Standard-Section-Type-Processors*.
"
  ;;; Declared by JPR.
)

(defun test-section-defining-item (item line delim-idx depackage-p)
  ;;; Declared by JPR.
  (declare (optimize (safety 0)))
  (if (consp item)
      (if (if depackage-p
	      (and (> delim-idx (first item))
		   (let ((colon1 (string-search-set '(#\:) line 1 delim-idx)))
			(let ((colon2
				(and colon1
				     (string-search-set
				       '(#\:) line colon1 delim-idx
				     )
				)
			      )
			     )
			     (and colon1
				  (sys:%string-equal
				    line (+ 1 (or colon2 colon1))
				    (second item) 1 (- (first item) 1)
				  )
			     )
			)
		   )
	      )
	      (and (eql delim-idx (first item))
		   (sys:%string-equal line 0 (second item) 0 (first  item))
	      )
	  )
	  (third item)
	  nil
      )
      (if (functionp item)
	  (funcall item line)
	  (ferror nil
"~S cannot be interpreted as a section defining item." item
	  )
      )
  )
)

(defun test-section-defining-items
       (items line DELIM-IDX &optional (depackage-p nil) (all-items items))
"A function for determining the name of a section from a line, if the
 system has failed to name it from any of the predefined types.
 The items can have elements of two types: a) a three-list, of the form
 (fixnum string result-type) such as (10 \"(DEFMETHOD\" :Always-Method)
 in this case the normal test is applied using the string and number for
 comparison purposes or b) a function, which is called with the line as
 its argument.  The third item of the list can be any of :Maybe-Method
 :Always-Method :Destruct :Defselect :Method :Handler :Defwrapper
 :Property :Internal.  If the third item is anything other that this (a user-
 defined, non-standard section type then a section node naming function must be
 declared and added to the Alist *Non-Standard-Section-Type-Processors*.
 If depackage-p then it tries removing package prefices and starts all over
 looping through the list.
"
  ;;; Declared by JPR.
  (declare (optimize (safety 0)))
  (if items
      (let ((result (test-section-defining-item
		      (first items) line DELIM-IDX depackage-p
		    )
	    )
	   )
	   (if result
	       result
	       (test-section-defining-items
		 (rest items) line delim-IDX depackage-p all-items
	       )
	   )
      )
      (if all-items
	  (test-section-defining-items all-items line delim-IDX t nil)
	  nil
      )
  )
)

;-------------------------------------------------------------------------------

(defvar *Non-Standard-Section-Type-Processors* nil
"An AList, which associates non-standard section node types with functions which
 know how to determine the section node name, given the name data for the
 section line.
"
  ;;; Declared by JPR.
)


;;;Edited by RICE                  16 Jan 90  14:12
(defun section-name-from-non-standard-type (type name-data ok-to-ask old-sym line)
"Takes the type of a section node and the data found in the file for its name
 and returns a section node name for it or nil if it does not understand it.
 It determines whether it can understand it by looking in the AList called
 *Non-Standard-Section-Type-Processors* for the type and, if it finds it,
 calls the function with the name data.
"
  ;;; Declared by JPR.
  (let ((item (assoc type *Non-Standard-Section-Type-Processors*
		     :test #'string-equal
	      )
	)
       )
       (if item
	   (multiple-value-bind (value foundp)
	     (funcall (if (consp (rest item));;; Can be either (type . function)
			  (second item)      ;;; or (type function)
			  (rest item)
		      )
		      name-data
		      ok-to-ask
		      line
	     )
	     (if foundp value old-sym)
	   )
	   old-sym
       )
  )
)


;-------------------------------------------------------------------------------

(defun section-name-for-defwhopper (line)
  (ignore line)
  (if (lisp:search "DEFWHOPPER" line :Test #'char-equal)
      :Whopper
      nil
  )
)

(if (boundp 'zwei:*section-defining-items*)
    (pushnew 'section-name-for-defwhopper zwei:*section-defining-items*)
)

(defun section-name-for-defwhopper-1 (spec ok-to-ask line)
  (ignore ok-to-ask)
  (let ((index (lisp:search "DEFWHOPPER" line :Test #'char-equal)))
       (if (and (consp spec) index)
	   (values `(:method ,(first spec) :Around ,@(rest spec)) t)
	   nil
       )
  )
)


(if (boundp 'zwei:*Non-Standard-Section-Type-Processors*)
    (setq zwei:*Non-Standard-Section-Type-Processors*
	  (cons '(:whopper section-name-for-defwhopper-1)
		 (remove
		   (assoc :Whopper zwei:*Non-Standard-Section-Type-Processors*)
		   zwei:*Non-Standard-Section-Type-Processors*
		 )
	  )
    )
)

;-------------------------------------------------------------------------------

(defun section-name-for-defstruct (line)
  (ignore line)
  (if (lisp:search "DEFSTRUCT" line :Test #'char-equal)
      :Struct
      nil
  )
)

(if (boundp 'zwei:*section-defining-items*)
    (pushnew 'section-name-for-defstruct zwei:*section-defining-items*)
)

(defun section-name-for-defstruct-1 (spec ok-to-ask line)
  (ignore ok-to-ask)
  (let ((index (lisp:search "DEFSTRUCT" line :Test #'char-equal)))
       (if index
	   (values (if (consp spec) (first spec) spec) t)
	   nil
       )
  )
)


(if (boundp 'zwei:*Non-Standard-Section-Type-Processors*)
    (setq zwei:*Non-Standard-Section-Type-Processors*
	  (cons '(:struct section-name-for-defstruct-1)
		 (remove
		  (assoc :struct zwei:*Non-Standard-Section-Type-Processors*)
		  zwei:*Non-Standard-Section-Type-Processors*
		 )
	  )
    )
)

;-------------------------------------------------------------------------------

;;; N.B. the following is a modified version of the original code for this
;;; function.  The modifications have been labeled with my initials (JPR).
;;;Edited by RICE                  16 Jan 90  14:12
;;;Edited by RICE                  16 Jan 90  15:31
;;;Edited by RICE                  16 Jan 90  15:35
#8R
(DEFUN SYMBOL-FROM-STRING (STR &OPTIONAL LINE OK-TO-ASK SYM &optional (bp-line nil) &AUX (EOF '(NIL)) ERROR-P)
  "Given a string STR as found after DEF..., return the name of the object being defined.
LINE is the line that the string was found in.  It is used for
finding the particular defining construct used; this affects the result
since (DEFUN (FOO BAR) defines (:PROPERTY FOO BAR)
while (DEFMETHOD (FOO BAR) defines (:METHOD FOO BAR).
OK-TO-ASK means in certain circumstances
where things are not clear, ask the user.  Otherwise we guess.

The arg can also be an object; then its printed representation is used as the string.

The second value is a canonicalized string for the object
 (maybe the same string specified, maybe not).
The third is T if there was a problem
 in parsing the string (such as unbalanced parens).

You can pass the read-in form of the object as the fourth arg
if you already know it."
  (DECLARE (VALUES SYM STR ERROR-P))
  (IF (ARRAYP STR)
      (UNLESS SYM
	(MULTIPLE-VALUE-SETQ (SYM ERROR-P)
			     (CATCH-ERROR (READ-FROM-STRING STR NIL EOF) NIL))
	(SETQ ERROR-P (AND (NULL SYM) ERROR-P)))	; chk 4 error on multi-val funct
							; patch 94.68 ddd/gsl 2/28/84.
      (SETQ SYM STR
	    STR (FORMAT NIL "~S" STR)))
  (COND (ERROR-P
	 (VALUES NIL NIL ERROR-P))
	((SYMBOLP SYM)
	 (VALUES SYM (SYMBOL-NAME SYM)))
	((OR (ATOM SYM) (EQUAL SYM EOF))
	 (VALUES NIL NIL T))
	(T
	 ;; Here SYM is a list.  Certain types of function specs have two ways to
	 ;; type them, with and without the leading type keyword.  Also certain types
	 ;; of functions and other definitions do not follow the standard form
	 ;; of (DEFxxx name options...).  What we do here is to recognize and
	 ;; standardize those cases.  The variables are:
	 ;;	TYPE - the type of function spec or non-function definition
	 ;;	SYM - the function spec or definition name
	 ;;	SPEC - the variant of SYM which appears in the source code
	 ;;	STR - SPEC converted to a string
	 ;; :HANDLER doesn't appear in source files, but gets translated into
	 ;; an appropriate :METHOD here, by analyzing the combined method.
	 ;; :INTERNAL doesn't appear in source files, but might be given as the argument
	 ;; to M-X Disassemble.  The code here just tries not to destory it.
	 (LET ((TYPE (CAR SYM))
	       DELIM-IDX SPEC)
	   (IF (AND (SYMBOLP TYPE)		; Beware of pathological cases.
		    (GET TYPE 'SYSTEM:FUNCTION-SPEC-HANDLER))
	       (IF (AND (MEMBER TYPE '(SETF LOCF) :TEST #'EQ)
			(SYSTEM:VALIDATE-FUNCTION-SPEC SYM))
		   (RETURN-FROM SYMBOL-FROM-STRING (VALUES SYM STR))
		 (SETQ SPEC (CDR SYM)
		       STR (DEFINITION-NAME-AS-STRING TYPE SPEC)))
	       (SETQ SPEC SYM
		     DELIM-IDX (AND LINE (STRING-SEARCH-SET "( " LINE 1))
		     TYPE (COND ((NULL LINE)
				 (if (assoc
				       (first spec)
				       *Non-Standard-Section-Type-Processors*
				       :test #'string-equal
				     )
				     :handled-separately
				     :MAYBE-METHOD))
				((AND (EQL DELIM-IDX 12)
				      (SYS:%STRING-EQUAL LINE 0 "(DEFMETHOD" 0 12))
				 :ALWAYS-METHOD)
				((AND (EQL DELIM-IDX 13)
				      (SYS:%STRING-EQUAL LINE 0 "(DEFWRAPPER" 0 13))
				 (SETQ SPEC (LIST (CAR SPEC) :WRAPPER (SECOND SPEC)))
				 :ALWAYS-METHOD)
				((AND (EQL DELIM-IDX 12)
				      (SYS:%STRING-EQUAL LINE 0 "(DEFSELECT" 0 12))
				 :DEFSELECT)
				((AND (EQL DELIM-IDX 7)
				      (SYS:%STRING-EQUAL LINE 0 "(DEFINE" 0 7)) ; Scheme DEFINE
				 :DEFSTRUCT) ; use (CAR SPEC)
				((test-section-defining-items *section-defining-items* line DELIM-IDX)) ;;; Added by JPR.
				(T
				 :PROPERTY))))
	   (OR (CASE TYPE
		 (:ALWAYS-METHOD
		  (SETQ SYM (CONS :METHOD SPEC)))
		 ((:METHOD :HANDLER :MAYBE-METHOD)
		  (LET ((FLAVOR (CAR SPEC))
			(MESSAGE (IF (CDDR SPEC)
				     (CADDR SPEC)
				     (CADR SPEC)))
			FL)
		    (COND ((SETQ FL (GET FLAVOR 'SI:FLAVOR)))
			  (OK-TO-ASK
			   (DOLIST (SYMBOL (PACKAGE-LOOKALIKE-SYMBOLS FLAVOR NIL '(SI:FLAVOR)))
			     (IF (FQUERY '(:SELECT T) "Do you mean ~S? "
					 `(:METHOD ,SYMBOL ,@(CDR SPEC)))
				 (RETURN (SETQ FLAVOR SYMBOL
					       SPEC (CONS FLAVOR (CDR SPEC))
					       FL (GET FLAVOR 'SI:FLAVOR)))))))
		    (COND ((SYMBOLP FL)			;T or NIL
			   (AND (EQ TYPE :MAYBE-METHOD)
				(VALIDATE-2-LONG-LIST SPEC)
				(SETQ SYM (CONS :PROPERTY SPEC))))
			  ((FDEFINEDP `(:METHOD . ,SPEC))
			   (SETQ SYM `(:METHOD . ,SPEC)))
			  (OK-TO-ASK
			   (DOLIST (SYMBOL (OR (FIND-COMBINED-METHODS FLAVOR MESSAGE NIL)
					       (SI:FLAVOR-ALL-INHERITABLE-METHODS FLAVOR MESSAGE)))
			     (IF (FQUERY '(:SELECT T) "Do you mean ~S? " SYMBOL)
				 (RETURN (SETQ SYM SYMBOL))))))))
		 (:DEFSTRUCT
		  (SETQ SYM (CAR SPEC))
		  (LOOP WHILE (CONSP SYM) ; this can happen in Scheme DEFINE
			DO (SETQ SYM (CAR SYM)))
		  (SETQ STR (SYMBOL-NAME SYM)))
		 (:DEFSELECT
		  (SETQ SYM (CAR SPEC))
		   (IF (SYMBOLP SYM)
		       (SETQ STR (SYMBOL-NAME SYM))
		       (MULTIPLE-VALUE-SETQ (SYM STR)
			 (SYMBOL-FROM-STRING SYM))))
		 (:PROPERTY
		  (AND (VALIDATE-2-LONG-LIST SPEC)
		       (SETQ SYM (CONS TYPE SPEC))))
		 (:INTERNAL
		  (SETQ SYM (CONS TYPE SPEC))
		  (SETQ STR (DEFINITION-NAME-AS-STRING NIL (CAR SPEC))))
		 ;;; Otherwise part inserted by JPR.
		 (otherwise (setq sym (section-name-from-non-standard-type
					(if (equal type :handled-separately)
					    (first spec)
					    type
					)
					(if (equal type :handled-separately)
					    (rest spec)
					    spec
					)
					ok-to-ask sym bp-line))))
	       ;; Something we don't understand, make a bogus symbol to use as a property
	       ;; list to remember the location of this definition
	       (SETQ SYM (INTERN STR *UTILITY-PACKAGE*))))
	 (IF (NOT (SYSTEM:VALIDATE-FUNCTION-SPEC SYM))
	     (VALUES NIL NIL T)
	     (VALUES SYM STR)))))

;-------------------------------------------------------------------------------


(defun coerce-source-file (spec)
"Given a source-file spec from a function's plist it coerces it into a pathname
 which can be editted.
"
  (let ((path (if (consp spec) (second (assoc 'defun spec)) spec)))
       (if (equal (send path :type) :Unspecific)
	   (send (send path :new-type :Lisp) :New-Version :Newest)
	   (send path :New-Version :Newest)
       )
  )
)


(defun find-fef (function)
"Given a function which is a name, a (:method...) or a compiled function,
 returns the compiled function for the function.
"
  (multiple-value-bind (fef error-p)
      (catch-error (typecase function
		     (symbol (sys:fdefinition function))
		     (cons (sys:fdefinition function))
		     (compiled-function function)
		   )
		   nil
      )
    (ignore error-p)
    fef
  )
)


(defun check-instruction (string initial-index)
"Given an instruction represented as a string and the initial index of a
 function's instructions it returns true if the instruction is either a
 tail recursive funcall or a branch to the initial instruction
 (self tail recursive).
"
  (multiple-value-bind (instruction end-of-inst)
      (multiple-value-bind (token end) (read-from-string string)
	(ignore token)
	;;; Token is the instruction number, which is ignored.
	;;; finds the name of the instruction and the end index in the line
	;;; of that instruction name
	(read-from-string string t nil :start end)
      )
      (if (and (symbolp instruction)
	       (string-equal (symbol-name instruction) "BR")
	       (equal initial-index
		      ;;; It's a branch instruction.  Read the destination.
		      ;;; If same as Initial-Index then it's self tail
		      ;;; recursive.
		      (read-from-string string t nil :start end-of-inst)
	       )
	  )
	  :Self
	  (if (and (symbolp instruction)
		   ;;; This function makes tail recursive funcalls.
		   (string-equal (symbol-name instruction) "TAIL-REC")
	      )
	      t
	      nil
	  )
      )
  )
)


(defun tail-recursive-p (fef)
"Is true if the compiled function object (fef) is tail recursive in any way."
  ;;; Disassembles the function line by line into string.  It then checks these
  ;;; instructions to see if theyindicate tail recursiveness.
  (let ((string (make-array 255 :adjustable t
			        :fill-pointer t
				:element-type 'string-char
		)
	)
       )
       (with-output-to-string (*Standard-Output* string (fill-pointer string))
	 (setf (fill-pointer string) 0)
	 (let ((initial-index (char-int (compiler:fef-initial-pc fef)))
	       (lim-pc (compiler:disassemble-lim-pc fef))
	       (ilen nil)
	       (found-p nil)
	      )
	      (do ((pc (compiler:fef-initial-pc fef)
		       (+ pc ilen)))
		  ((>= pc lim-pc))
		(setf (fill-pointer string) 0)
		(setq ilen (compiler:disassemble-instruction-length fef pc))
		(compiler:disassemble-instruction fef pc nil)
		(let ((result (check-instruction string initial-index)))
		     (if result
			 (progn (setq found-p result)
				(return result)
			 )
			 nil
		     )
		)
	      )
	      found-p
	 )
      )
  )
)


(defun add-safety-0 (form)
"Takes a function definition form and returns the compiled function object
 after taking form with (declare (optimize (safety 0))) stuck in it and
 compiling it.
"
  (multiple-value-bind (body decls docs)
      (parse-body (rest (rest (rest form))) nil)
    (let ((new-form (append (list (first form) (second form) (third form))
			    '((declare (optimize (safety 0))))
			    decls (cons docs body)
		    )
	  )
	 )
	 (find-fef (compiler:compile-form new-form))
    )
  )
)

(defun process-section-for (function-name buffer)
"Is passed the name of a function.  It finds the zmacs buffer which contains
 the function, finds the section with its name and compiles, with optimise
 safety 0, so as to switch on tail recursion.  It then checks to see whether
 the function is in fact tail recursive.
"
  (let ((start (si:function-spec-get function-name 'Zmacs-Buffers)))
       (if start
	   (let ((buffer (first (first start)))
		 (line   (rest  (first start)))
		)
		(let ((section (array-leader line 5)))
		     ;;; The section node for the function.  Founds from the
		     ;;; Zmacs buffers property.
		     (multiple-value-bind (vars vals)
			 (send buffer :attribute-bindings)
		       (progv vars vals
			 (let ((form (read (interval-stream
					     (send section :first-bp)
					     (send section :last-bp)
					   )
				     )
			       )
			      )
			      ;;; Make an interval stream and read from the
			      ;;; stream with the right attribute bindings.
			      (tail-recursive-p (add-safety-0 form))
			      
			 )
		       )
		     )
		)
	   )
	   (ferror nil "Cannot find a definition for ~A in ~A."
		   function-name buffer
	   )
       )
  )
)

(defun check-function-code-for (function-name)
"The function specified is not tail recrsive as yet, so load up its source code,
 compile it with declare optimize safety 0 and check again.
"
  (let ((source-file (si:function-spec-get function-name :Source-File-Name)))
       (if (and source-file (coerce-source-file source-file))
	   (process-section-for function-name
	      (load-file-into-zmacs (coerce-source-file source-file) t t)
	   )
	   (ferror nil "~S has no function source file." function-name)
       )
  )
)

(defun check-function-for-tail-recursivity (function)
"Given a function check to see whether it is tail recursive.  If it is then
 fine.  If it isn't then try to make it so by recompiling it with declare
 optimize safety 0.
"
  (let ((fef (find-fef function)))
       (if fef
	   (let ((result (tail-recursive-p fef)))
	        (if result
		    result
		    (check-function-code-for (function-name function))
		)
	   )
	   (check-function-code-for (function-name function))
       )
  )
)


(defcom check-function-is-tail-recursive
  "Find a function and determine whether it is, or can be, self tail recursive."
  ()
  (multiple-value-bind (function spec string explicit-package-p)
      (read-function-name "Function:"
			  (relevant-function-name (point))
			  'aarray-ok 'multiple-ok
      )
      (ignore spec string explicit-package-p)
      (let ((result (check-function-for-tail-recursivity function)))
	   (if result
	       (if (equal result :Self)
		   (format *Query-IO* "~S is or can be self tail recursive."
			   function
		   )
		   (format *Query-IO* "~S is or can be tail recursive."
			   function
		   )
	       )
	       (format *Query-IO* "~S cannot be tail recursive." function)
	   )
      )
  )
  Dis-Text
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(check-function-is-tail-recursive))
)

;-------------------------------------------------------------------------------



(defparameter *section-name* nil
"The name of the current section being arg checked or nil."
)

(defun arg-warn (result string &rest format-args)
"Warns the user using string and form args and then returns Result."
  (declare (special *check-arglist-steam* *section-name*))
  (if *check-arglist-steam*
      (progn (if *section-name*
		 (if (send *check-arglist-steam* :operation-handled-p :item)
		     (progn (terpri *check-arglist-steam*)
			    (send *check-arglist-steam* :item 'function-name
				  *section-name*
				  "<< While checking args of ~A >>" 
				  *section-name*
			    )
		     )
		     (format *check-arglist-steam*
			     "~&<< While checking args of ~A >>" *section-name*
		     )
		 )
		 nil
	     )
	     (format *check-arglist-steam* "~&")
	     (apply #'format *check-arglist-steam* string format-args)
	     (format *check-arglist-steam* ".")
      )
      nil
  )
  result
)


(defun check-optional-arg-is-legal (arg)
"Is true if arg is a symbol or if it is a cons and its default is a legal form."
  (typecase arg
    (symbol t)
    (cons (check-rest-of-args (rest arg)))
    (otherwise (ferror nil "~S is an illegal binding." arg))
  )
)


(defun check-optional-args-are-legal (args)
"Is true if arg is a symbol or if it is a cons and its default is a legal form."
  (if args
      (and (check-optional-arg-is-legal (first args))
	   (check-optional-args-are-legal (rest args))
      )
      t
  )
)


(defun check-&Optional-arg (function actual-args formal-args)
"Given an &optional arg checks to see whether it, and any subsequent args are
 valid.
"
  (if (equal nil actual-args)
      t
      (if (member (second formal-args) lambda-list-keywords)
	  (is-a-legal-call-to function actual-args (rest formal-args))
	  (and (is-a-legal-call-to function (rest actual-args)
			           (cons '&Optional (rest (rest formal-args)))
	       );;; All other args are optional.
	       (check-optional-arg-is-legal (second formal-args))
	       (is-a-legal-call-internal (first actual-args))
	  )
      )
  )
)

(defun check-rest-of-args (actual-args)
"Checks to see whether the expressions which represent an arglist to some
 function are valid calls or not.
"
  (let ((result-from-args
	   (remove t
		   (mapcar #'is-a-legal-call-internal
			   actual-args
		   )
	   )
	 )
	)
	(declare (unspecial result-from-args))
	(equal nil result-from-args)
   )
)

(defun check-&Rest-arg (function actual-args formal-args)
"Given an &rest arg checks to see whether it, and any subsequent args are
 valid.
"
  (if (equal (second formal-args) '&Key)
      (is-a-legal-call-to function actual-args (rest formal-args))
      (check-rest-of-args actual-args)
  )
)

(defun key-args-are-the-same (a b)
"Is true if the Pname of A matches any of the PNames in B"
  (string-equal (symbol-name a) (symbol-name (if (consp b) (first b) b)))
)

(defun check-&Key-arg (function actual-args formal-args)
"Given an &Key arg checks to see whether it, and any subsequent args are
 valid.
"
  (if (equal nil actual-args)
      t
      (if (equal nil (rest actual-args))
	  (arg-warn nil "Not enough args for ~S for keyword arg list"
		    function
	  )
	  (if (keywordp (first actual-args))
	      (let ((matching-key
		      (member (first actual-args) (rest formal-args)
			      :Test #'key-args-are-the-same
		      )
		    )
		   )
		   (if matching-key
		       (and (is-a-legal-call-to function
						(rest (rest actual-args))
						formal-args
			    )
			    (check-optional-arg-is-legal (second matching-key))
			    (is-a-legal-call-internal
			      (first (rest actual-args))
			    )
		       )
		       (arg-warn nil "~S is not a valid keyword arg for ~S"
				 (first actual-args) function
		       )
		   )
	      )
	      (arg-warn nil "~S should be a keyword arg to ~S"
			(first actual-args) function
	      )
	  )
      )
  )
)

(defun check-&Aux-arg (function actual-args formal-args)
"Given an &aux arg checks to see whether it, and any subsequent args are
 valid.
"
  (ignore formal-args)
  (if (equal nil actual-args)
      (and (is-a-legal-call-to function nil
			       (cons '&Aux (rest (rest formal-args)))
	   )
	   (check-optional-arg-is-legal (second formal-args))
      )
      (arg-warn nil "~S are surplus args to ~S" actual-args function)
  )
)

(defun check-&Surplus-arg (function actual-args formal-args)
"Given an & arg, which can be ignored, checks to see whether it, and any
 subsequent args are valid.
"
  (is-a-legal-call-to function actual-args (rest formal-args))
)

(defun check-&Quote-arg (function actual-args formal-args)
"Given an &quote arg checks to see whether it, and any subsequent args are
 valid.
"
  (if (equal nil actual-args)
      (arg-warn nil "Missing args ~S for call to ~S" formal-args function)
      (if (and (consp (first actual-args))
	       (consp (first (first actual-args)))
	       (not (equal 'lambda (first (first (first actual-args)))))
	  )
	  ;;; Probably a binding list.
	  (and (is-a-legal-call-to function (rest actual-args)
				   (rest formal-args)
	       )
	       (check-optional-args-are-legal (first actual-args))
	  )
	  (if (and (consp (rest formal-args))
		   (equal (second formal-args) '&Rest)
	      )
	      (is-a-legal-call-to function actual-args (rest formal-args))
	      (is-a-legal-call-to function (rest actual-args)
				  (rest (rest formal-args))
	      )
	  )
      )
  )
)

(defun match-up-destructuring-args (function actual-args formal-args)
"Checks some destructuring args for a function to see whether they are valid
 or not.
"
  (if (equal nil actual-args)
      (arg-warn nil "No arg supplied for ~S arg spec ~S" function
		(first formal-args)
      )
      (if (is-a-legal-call-to function (first actual-args) (first formal-args))
	  (is-a-legal-call-to function (rest actual-args) (rest formal-args))
	  (arg-warn nil "Args ~S and ~S do not match in call to ~S"
		    (first actual-args) (first formal-args) function
	  )
      )
  )
)

(defun is-a-legal-call-to (function actual-args formal-args)
"Is true if the actual args and formal args for Function match so that this
 could be said to be a valid call.
"
  (ignore function)
  (if (equal nil formal-args)
      (if (equal nil actual-args)
	  t
	  (arg-warn nil "Too many args passed to ~S.  ~S are surplus"
		    function actual-args
	  )
      )
      (if (symbolp formal-args)
	  t ;;; Dotted destructuring arg.
	  (case (first formal-args)
	    (&Optional (check-&Optional-arg function actual-args formal-args))
	    (&Rest (check-&Rest-arg function actual-args formal-args))
	    (&Body (check-&Rest-arg function actual-args formal-args))
	    (&Whole (check-&Rest-arg function actual-args formal-args))
	    (&Key (check-&Key-arg function actual-args formal-args))
	    (&Aux (check-&Aux-arg function actual-args formal-args))
	    (&Quote (check-&Quote-arg function actual-args formal-args))
	    (&Special (check-&Surplus-arg function actual-args formal-args))
	    (&Local (ferror nil "??? ~A" (first formal-args)))
	    (&Functional (check-&Surplus-arg function actual-args formal-args))
	    (&Eval (check-&Surplus-arg function actual-args formal-args))
	    (&List-of (ferror nil "??? ~A" (first formal-args)))
	    (&Extension (ferror nil "??? ~A" (first formal-args)))
	    (&Environment (ferror nil "??? ~A" (first formal-args)))
	    (&Allow-Other-Keys
	      (check-&Surplus-arg function actual-args formal-args)
	    )
	    (Otherwise
	     (if (consp (first formal-args)) ;destructuring-macro
		 (multiple-value-bind (ignore ignore type) (arglist function)
		   (if (equal type 'Macro)
		       (match-up-destructuring-args function actual-args
						    formal-args
		       )
		       (arg-warn nil
				 "Destructuring arg ~S given to non-macro ~S"
				 (first formal-args) function
		       )
		   )
		 )
		 (if (equal nil actual-args)
		     (arg-warn nil "Missing args ~S for call to ~S"
			       formal-args function
		     )
		     (and (is-a-legal-call-to function (rest actual-args)
					      (rest formal-args)
			  )
			  (is-a-legal-call-internal (first actual-args))
		     )
		 )
	     )
	    )
	  )
      )
  )
)

(defun function-body-from-exception (form)
"Given a form it returns the body of the for which can have its args checked.
 This is because Named-Lambda and such like do not have arglists.  A second
 value returned is the name of the thing (like a function name if is can
 deduce it.
"
  (case (first form)
    (function (list 'progn (second form)))
    ((named-lambda named-subst) (values (nthcdr 3 form) (second form)))
    (lambda (nthcdr 2 form))
    (multiple-value-bind (nthcdr 2 form))
    (otherwise form)
  )
)

(defun get-args (function)
"Like arglist, but if the real args of the function are different from the
 args returned by Arglist then it returns the real ones.
"
  (let ((args (arglist function))
	(real-args (sys:dbis-arglist (sys:get-debug-info-struct function t)))
       )
       (if (equal args real-args)
	   args
	   real-args
       )
  )
)


(defun is-a-legal-call-internal (form)
"True if form has a valid set of arglists."
  (declare (unspecial form))
  (if (consp form)
      (let ((function (first form))
	    (actual-args (rest form))
	   )
	   (declare (unspecial function actual-args))
	   (if (consp function)
	       (if (is-a-legal-call-internal function)
		   (check-rest-of-args actual-args)
		   nil
	       )
	       (multiple-value-bind (formal-args error-p)
		   (catch-error (multiple-value-list (get-args function)) nil)
		   (declare (unspecial formal-args error-p))
		   (multiple-value-bind (body name)
		       (if (symbolp function)
			   (function-body-from-exception form)
			   (values nil nil)
		       )
		     (if error-p
			 (if (symbolp function)
			     (if (not (eq form body))
				 (if name
				     (let ((*section-name* name))
					  (check-rest-of-args body)
				     )
				     (check-rest-of-args body)
				 )
				 (if (or (keywordp function)
					 (type-specifier-p function)
					 (equal (symbol-package function)
						(find-package 'lisp)
					 )
				     )
				     (check-rest-of-args actual-args)
				     (arg-warn nil "Undefined function ~S"
					       function
				     )
				 )
			     )
			     (arg-warn nil "Undefined function ~S" function)
			 )
			 (if (not (eq form body))
			     (if name
				 (let ((*section-name* name))
				      (check-rest-of-args body)
				 )
				 (check-rest-of-args body)
			     )
			     (is-a-legal-call-to function actual-args
						 (first formal-args)
			     )
			 )
		     )
		   )
	       )
	   )
      )
      t
  )
)

(defun is-a-legal-call (form &Optional (stream *standard-output*))
"True if form has a valid set of arglists.  Messages describing the nature of
 the errors are printed to Stream unless it is nil.
"
  (declare (unspecial form stream))
  (let ((*print-length* 2)
	(*print-level*  2)
	(*check-arglist-steam* stream)
       )
       (declare (special *print-level* *print-length* *check-arglist-steam*))
       (is-a-legal-call-internal (macroexpand-all form))
  )
)


(defcom com-check-args-in-expression
" Check all of the arglists in the expressions selected or the enclosing
 expression.  This is not always reliable, because a lot of system functions
 have arglists like &quote &rest (eg. DO), so it's hard to determine whether
 the args are right, but it has a go anyway.  Arg checking is done on the
 MacroExpanded version of the selected code.
" ()
  (format *Query-Io* "~&Checking arglists")
  (multiple-value-bind (form section-name) (read-form-or-forms-from-buffer t)
    (if (eq form :Nothing-found)
	(barf)
        (progn (format *Query-Io* " in ~A" section-name)
	       (let ((*section-name*
		       (if (stringp section-name) nil section-name)
		     )
		    )
		    (declare (special *section-name*))
		    (if (is-a-legal-call form) nil (beep))
		    (format *Query-Io* " - checked")
	       )
	)
    )
  )
  dis-none
)

;-------------------------------------------------------------------------------


(defun check-args-in-buffer ()
  (let ((stream (interval-stream (send *Interval* :first-bp)
				      (send *Interval* :last-bp)
		)
	)
       )
       (unwind-protect
         (loop for expr = (read stream nil :end-of-File)
	       until (equal expr :end-of-file)
	       do (is-a-legal-call expr)
	 )
	 (close stream)
       )
  )
)

(defcom com-tags-check-args
"Check the args of all of the expressions in the files denoted by the
 selected tag table.
"
  ()
  (do ((pt (next-file-bp t)))
      (nil)
; (open foo :kdsjfh 42 :input baz)
    (make-buffer-current (bp-top-level-node pt))
    (format t "~&~A" (send *interval* :name))
    (check-args-in-buffer)
    (setq pt (next-file-bp nil))
  )
  dis-text
)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-tags-check-args))
)

;-------------------------------------------------------------------------------


(defun load-up-files-for-system (system)
  "Visits all the files in specified system."
  (let ((*window* (if (boundp '*window*)
		      *window*
		      (first Zwei::*All-Zmacs-Windows*)
		  )
	)
	(*MODE-LIST-SYNTAX-TABLE* (MAKE-LIST-SYNTAX-TABLE))
       )
    (DOLIST (FILE (SI:SYSTEM-SOURCE-FILES SYSTEM))
      (LET ((SOURCE-PATHNAME (SEND FILE ':SOURCE-PATHNAME)))
	(let ((buffer (FIND-FILE-BUFFER SOURCE-PATHNAME)))
	  (if buffer
	      (if (not (equal (send (probe-file source-pathname) :Version)
			      (send (first (send buffer :File-Id)) :Version)
		       )
		  )
		  (if (y-or-n-p "~&There is already an old version of ~S~
                                 in zmacs.  Revert the buffer?"
				(send source-pathname :String-For-Host)
		      )
		      (revert-buffer buffer source-pathname)
		      nil
		  )
	      )
	      (FIND-FILE source-PATHNAME NIL t))))))
  DIS-TEXT)


;-------------------------------------------------------------------------------

(defun maybe-load-system-declaration (system-name)
  (let ((system-path
	  (probe-file (fs:make-pathname
			:defaults "sys:site;foo.system"
			:name system-name
			:type "SYSTEM"
		      )
	  )
	)
	(translation-path
	  (probe-file (fs:make-pathname
			:defaults "sys:site;foo.translations"
			:name system-name
			:type "TRANSLATIONS"
		      )
	  )
	)
       )
       (if system-path
	   (progn (format *query-io* "~&Loading system definition for ~A."
			  system-name
		  )
		  (if translation-path
		      (load translation-path :verbose nil)
		      nil
		  )
		  (load system-path :verbose nil)
	   )
	   (progn (format *query-io* "~&Cannot find a system called ~A."
			  system-name
		  )
		  (barf)
	   )
       )
       system-name
  )
)

(defun read-maybe-defined-system-name
       (prompt)
  "Read a system name in the mini buffer, defaulting to DEFAULT.
Prompts with PROMPT (which should end with a colon and not mention the default).
DEFAULT defaults to a guess based on the current buffer."
  (let ((system-name (completing-read-from-mini-buffer prompt
		       (si:all-systems-name-alist) t
		     )
	)
       )
       (if (consp system-name)
	   (first system-name)
	   (maybe-load-system-declaration system-name)
       )
  )
)

(defparameter *spawning-enabled* nil)

(defcom com-enable-spawning
	"Switch on background operation for some M-x commands." ()
  (setq *spawning-enabled* t)
  (format *query-io* "~&Spawning Enabled.")
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-enable-spawning))
)

(defcom com-disable-spawning
	"Switch off background operation for some M-x commands." ()
  (setq *spawning-enabled* nil)
  (format *query-io* "~&Spawning Disabled.")
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-disable-spawning))
)

(defun maybe-spawn-function (name function &rest args)
  (if *spawning-enabled*
      (process-run-function
	(list :name name :priority -5)
	#'(lambda ()
	    (apply function args)
	    (tv:notify tv:selected-window "Background process ~A completed Ok."
		       name
	    )
	  )
      )
      (progn (apply function args)
	     (format *query-io* "~&Finished ~A." name)
      )
  )
)


(defun spawn-make-system (name &rest keys)
  (apply #'maybe-spawn-function (format nil "Make-System ~A" name)
	 #'make-system name keys
  )
)

(defcom com-make-system "Make System without compile." ()
  (let ((system-name (read-maybe-defined-system-name "System to make:")))
       (let ((compile-p (y-or-n-p "Compile ~A?" system-name)))
	    (spawn-make-system
	      system-name
	      (if compile-p
		  (if (y-or-n-p "ReCompile ~A?" system-name)
		      :ReCompile
		      :Compile
		  )
		  (if (y-or-n-p "ReLoad ~A?" system-name)
		      :ReLoad
		      :Noop
		  )
	      )
	      (if (and compile-p (y-or-n-p "Safe?")) :Safe :Noop)
	      (if (and compile-p (y-or-n-p "Print Only?"))
		  :Print-Only
		  :Noop
	      )
	      (if (y-or-n-p "NoConfirm?") :Noconfirm :Noop)
	      (if (y-or-n-p "NoWarn?") :NoWarn :Noop)
	    )
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-make-system))
)

;-------------------------------------------------------------------------------

(defcom com-load-system "Make System (Just load)." ()
  (let ((system-name (read-maybe-defined-system-name "System to make:")))
       (spawn-make-system system-name :silent :noconfirm :nowarn)
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-load-system))
)

;-------------------------------------------------------------------------------

(defcom com-compile-system "Make System :compile." ()
  (let ((system-name (read-maybe-defined-system-name "System to compile:")))
       (spawn-make-system system-name :silent :noconfirm :nowarn :compile :safe)
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-compile-system))
)

;-------------------------------------------------------------------------------

(defcom com-recompile-system "Make System :recompile." ()
  (let ((system-name (read-maybe-defined-system-name "System to recompile:")))
       (spawn-make-system system-name :silent :noconfirm :nowarn :recompile)
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-recompile-system))
)

;-------------------------------------------------------------------------------


(DEFUN GET-scr-com-section
       (PROMPT &aux bp1 bp2 defun-name)
  (DECLARE (VALUES FILE NAME TYPE DEFAULT BUFFER))
  (LET* ((BUFFER (READ-BUFFER-NAME (FORMAT NIL "~A ~A" PROMPT "Buffer") *INTERVAL*))
	 (NAME (BUFFER-NAME BUFFER)))
    (make-buffer-current buffer)
    (SEND *TYPEOUT-WINDOW* :MAKE-COMPLETE)
    (REDISPLAY-ALL-WINDOWS)
    (format *query-io* "~&Select a region:")
    (loop for char = (send *window* :any-tyi)
	  do 
	  (if (equal char (char-code #\return))
	      (return nil)
	      (if (and (consp char) (equal :Mouse-button (first char)))
		  (apply #'PROCESS-SPECIAL-COMMAND char)
		  (progn (print char) (beep))
	      )
	  )
    )
    (IF (WINDOW-MARK-P *WINDOW*)
	(PROGN
	  (SETQ BP1 (MARK) BP2 (POINT))
	  (OR (BP-< BP1 BP2) (PSETQ BP1 BP2 BP2 BP1))
	  (IF
	    (BP-= (FORWARD-OVER *WHITESPACE-CHARS* (MARK))	      
		  (FORWARD-OVER *WHITESPACE-CHARS* (POINT)))
	    (SETQ *MARK-STAYS* ())
	    (SETQ DEFUN-NAME "Region"))))
    (COND (DEFUN-NAME)
	  ((SETQ BP1 (DEFUN-INTERVAL (BEG-LINE (POINT)) 1 () ()))
	   (SETQ BP2 (INTERVAL-LAST-BP BP1) BP1 (INTERVAL-FIRST-BP BP1)) (SETQ SI:*FORCE-DEFVAR-INIT* T))
	  (T (BARF "Unbalanced parentheses")))
    (MULTIPLE-VALUE-BIND (VARS VALUES)
	(SEND BUFFER :ATTRIBUTE-BINDINGS)
      (VALUES (SRCCOM:MAKE-FILE
		SRCCOM:FILE-NAME NAME
		SRCCOM:FILE-TYPE "Buffer"
		SRCCOM:FILE-STREAM (INTERVAL-STREAM bp1 bp2);BUFFER)
		SRCCOM::FILE-VARS VARS
		SRCCOM::FILE-VALUES VALUES
		SRCCOM:FILE-MAJOR-MODE
		(INTERN (STRING-UPCASE
			  (SYMBOL-VALUE (BUFFER-SAVED-MAJOR-MODE
					  BUFFER)))
			""))
	      NAME
	      "Buffer"
	      (AND (BUFFER-FILE-ID BUFFER)
		   (BUFFER-PATHNAME BUFFER))
	      BUFFER))
  )
)

(DEFCOM COM-SOURCE-COMPARE-regions "Compare two sections.
The output goes on the screen, and also into a buffer named *Source Compare ...*." ()
  (LET (FILE-1 FILE-2 NAME-1 NAME-2 KIND DEFAULT)
    (UNWIND-PROTECT
	(PROGN
	  (MULTIPLE-VALUE-SETQ (FILE-1 NAME-1 KIND DEFAULT)
	    (GET-scr-com-section "Section from buffer:"))
	  (MULTIPLE-VALUE-SETQ (FILE-2 NAME-2)
	    (GET-scr-com-section (FORMAT NIL "Compare ~A ~A with" KIND NAME-1)))
	  (LET ((*STANDARD-OUTPUT* (MAKE-BUFFER-WINDOW-OR-BROADCAST-STREAM
				     (FORMAT NIL "*Source Compare ~A / ~A*" NAME-1 NAME-2)
				     NIL T)))
	    (SRCCOM::DESCRIBE-SRCCOM-SOURCES FILE-1 FILE-2 *STANDARD-OUTPUT*)
	    (SRCCOM:SOURCE-COMPARE-FILES FILE-1 FILE-2 *STANDARD-OUTPUT* (SRCCOM::QUERY-TYPE)))
	  (FORMAT T "~&Done."))
      (AND FILE-1 (SEND (SRCCOM:FILE-STREAM FILE-1) :CLOSE))
      (AND FILE-2 (SEND (SRCCOM:FILE-STREAM FILE-2) :CLOSE))))
  DIS-NONE)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Source-Compare-Regions))
)

;-------------------------------------------------------------------------------



(defcom com-select-pathname-as-tag-table
"Select a (maybe wildcarded) pathname as denoting all of the files in the table"
  ()
  (let ((path (read-defaulted-pathname
		"Pathname for tag files:"
		(if (send *interval* :pathname)
		    (send (send *interval* :pathname) :new-name :wild)
		    fs:*default-pathname-defaults*
		)
	      )
	)
       )
       (let ((name (send path :string-for-printing))
	     (dir (rest (fs:directory-list path)))
	    )
	    (if dir
		(let ((new-dir (if (equal :newest (send path :version))
				   (mapcar #'(lambda (x)
					       (send x :new-version :newest)
					     )
					     (mapcar #'first dir)
				   )
				   dir
			       )
		      )
		     )
		     (select-file-list-as-tag-table new-dir name)
		)
		(barf "No file match ~A" name)
	    )
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-select-pathname-as-tag-table))
)

;(com-select-pathname-as-tag-table)


(defcom com-make-tag-table "Makes a new empty tag table." ()
  (let ((name (completing-read-from-mini-buffer
		"Name for new tag table:" nil t
	      )
	)
       )
       (if name
	   (select-file-list-as-tag-table nil name)
	   (barf)
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-make-tag-table))
)

;-------------------------------------------------------------------------------

(defun find-file-buffer-or-buffer (name)
  (typecase name
    (pathname (find-file-buffer name))
    (zmacs-buffer name)
    (otherwise (ferror nil "~S cannot name a buffer." name))
  )
)

(DEFUN NEXT-FILE (RESTART &AUX PATHNAME BUFFER)
  "Select the next file in the selected tag table.
RESTART non-NIL means select the first file in the tag table
and reset the list of files to be gone through."
  (AND RESTART
       (SETQ *ZMACS-LAST-TAGS-FILE-LIST*
	     (SEND (SELECT-TAG-TABLE) :GET 'ZMACS-TAG-TABLE-FILE-SYMBOLS)))
  (OR *ZMACS-LAST-TAGS-FILE-LIST*
      (BARF "No more files"))
  (POP *ZMACS-LAST-TAGS-FILE-LIST* PATHNAME)
  (COND ((SETQ BUFFER (FIND-FILE-BUFFER-or-buffer PATHNAME))
	 (FORMAT *QUERY-IO* "~&~A~%" PATHNAME)
	 (MAKE-BUFFER-CURRENT BUFFER)
	 (MOVE-BP (POINT) (INTERVAL-FIRST-BP *INTERVAL*)))
	(T
	 (FIND-FILE PATHNAME)))) 

(DEFUN NEXT-FILE-BP (RESTART &AUX PATHNAME BUFFER)
  "Return BP to start of the next file in the selected tag table.
RESTART non-NIL means start again at first file in tag table."
  (AND RESTART
       (SETQ *ZMACS-LAST-TAGS-FILE-LIST*
	     (SEND (SELECT-TAG-TABLE) :GET 'ZMACS-TAG-TABLE-FILE-SYMBOLS)))
  (OR *ZMACS-LAST-TAGS-FILE-LIST*
      (BARF "No more files"))
  (POP *ZMACS-LAST-TAGS-FILE-LIST* PATHNAME)
  (COND ((SETQ BUFFER (FIND-FILE-BUFFER-or-buffer PATHNAME))
	 (FORMAT *QUERY-IO* "~&~A~%" PATHNAME)
	 (INTERVAL-FIRST-BP BUFFER))
	(T
	 (INTERVAL-FIRST-BP (FIND-FILE PATHNAME NIL))))) 

(DEFUN TAG-TABLE-BUFFERS (READ-IN-ALL-FILES &AUX BUFFER-LIST FILE-LIST)
  "Return a list of all buffers in the selected tag table.
READ-IN-ALL-FILES means visit all the files;
otherwise, we return only the buffers for files already read in."
  (SETQ FILE-LIST (SEND (SELECT-TAG-TABLE) :GET 'ZMACS-TAG-TABLE-FILE-SYMBOLS))
  (DOLIST (FILE FILE-LIST)
    (LET ((BUFFER (FIND-FILE-BUFFER-or-buffer FILE)))
      (IF BUFFER
	  (PUSH BUFFER BUFFER-LIST))
      (IF READ-IN-ALL-FILES
	  (PUSH (FIND-FILE FILE NIL)
		BUFFER-LIST))))
  (NREVERSE BUFFER-LIST)) 

;-------------------------------------------------------------------------------


(defcom com-add-buffer-to-tag-table
 "Prompt for a file name and add that file to the current tag table." ()
  (let ((file-list
	  (send (select-tag-table) :get 'zmacs-tag-table-file-symbols)
	)
	(buffer (read-buffer-name "Buffer to add:" *interval* nil))
       )
       (if (typep buffer 'zmacs-buffer)
	   (if (member buffer file-list)
	       (format *query-io*
		       "The buffer is already in the current tag table."
	       )
	       (send (select-tag-table) :Putprop (cons buffer file-list)
		     'zmacs-tag-table-file-symbols
	       )
		   
	   )
	   (beep)
       )
  )
  Dis-None
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Add-buffer-to-tag-table))
)

;-------------------------------------------------------------------------------

(defcom com-Remove-buffer-from-tag-table
 "Prompt for a buffer name and remove that buffer from the current tag table."
  ()
  (let ((file-list
	  (send (select-tag-table) :get 'zmacs-tag-table-file-symbols)
	)
	(buffer (read-buffer-name "Buffer to remove:" *interval* nil))
       )
       (if (typep buffer 'zmacs-buffer)
	   (if (member buffer file-list)
	       (send (select-tag-table) :Putprop (remove buffer file-list)
		     'zmacs-tag-table-file-symbols
	       )
	       (format *query-io*
		       "The buffer ~A is not in the current tag table."
		       buffer
	       )
	   )
	   (beep)
       )
  )
  Dis-None
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(Com-Remove-Buffer-From-Tag-Table))
)


;-------------------------------------------------------------------------------

(defun SELECT-FILE-and-buffer-LIST-AS-TAG-TABLE (FILE-LIST NAME)
  "Select a phony tag table named NAME consisting of the files in FILE-LIST.
This can be used to control commands such as Tags Search."
  (SETQ *ZMACS-CURRENT-TAG-TABLE*
	(MAKE-INSTANCE 'TAG-TABLE-DUMMY-FILE :NAME NAME)
  )
  (SEND *ZMACS-CURRENT-TAG-TABLE* :PUTPROP
	(MAPCAR #'(LAMBDA (X)
		    (if (typep x 'pathname)
			(FS:MERGE-PATHNAME-DEFAULTS X *PATHNAME-DEFAULTS*)
			x
		    )
		  )
		  FILE-LIST)
	'ZMACS-TAG-TABLE-FILE-SYMBOLS)
  (PUSH (CONS NAME *ZMACS-CURRENT-TAG-TABLE*)
	*ZMACS-TAG-TABLE-ALIST*))


(DEFCOM COM-SELECT-ALL-BUFFERS-AS-TAG-TABLE "Select all files currently read in as a tag table.
This causes commands such as Tags Search, Tags Query Replace, and
Tags Compile Changed Sections to look through all files now visited." ()
  (SELECT-FILE-and-buffer-LIST-AS-TAG-TABLE
    (LOOP FOR BUFFER IN *ZMACS-BUFFER-LIST*
	  AS FILE-ID = (BUFFER-FILE-ID BUFFER)
	  COLLECT BUFFER)
    "All buffers")
  DIS-NONE)


;-------------------------------------------------------------------------------

(DEFCOM COM-SELECT-ALL-File-BUFFERS-AS-TAG-TABLE
"Select all files currently read in as a tag table.
This causes commands such as Tags Search, Tags Query Replace, and
Tags Compile Changed Sections to look through all files now visited." ()
  (SELECT-FILE-LIST-AS-TAG-TABLE
    (LOOP FOR BUFFER IN *ZMACS-BUFFER-LIST*
	  AS FILE-ID = (BUFFER-FILE-ID BUFFER)
	  WHEN (OR (EQ FILE-ID T)
		   (AND FILE-ID
			(CONSP FILE-ID)
			(NOT (NODE-SPECIAL-TYPE BUFFER))))
	  COLLECT (BUFFER-PATHNAME BUFFER))
    "All buffers visiting files")
  DIS-NONE)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(COM-SELECT-ALL-File-Buffers-AS-TAG-TABLE))
)

;-------------------------------------------------------------------------------



(defun select-file-list-as-tag-table (file-list name)
  "Select a phony tag table named NAME consisting of the files in FILE-LIST.
This can be used to control commands such as Tags Search."
  (let ((existing (assoc name *zmacs-tag-table-alist* :test #'string-equal)))
       ;;; Maybe reuse existing tag table.  This stops the AList getting
       ;;; clogged up.
       (setq *zmacs-current-tag-table*
	     (if existing
		 (rest existing)
		 (make-instance 'tag-table-dummy-file :name name)
	     )
       )
       (send *zmacs-current-tag-table* :putprop
	     (mapcar
	       #'(lambda (x)
		   (if (pathnamep x)
		       x
		       (fs:merge-pathname-defaults x *pathname-defaults*)
		   )
		 )
		 (if (and (consp (first file-list))
			  (pathnamep (first (first file-list)))
		     )
		     ;;; Then it's a directory plist.
		     (mapcar #'first file-list)
		     file-list
		 )
	     )
	     'zmacs-tag-table-file-symbols)
       (if existing
	   *zmacs-tag-table-alist*
	   (push (cons name *zmacs-current-tag-table*)
		 *zmacs-tag-table-alist*
	   )
       )
  )
)


(defparameter *select-tag-table-choices*
	     '(((:FILE "Tags File") #\F)
	       ((:ALL-BUFFERS "All Buffers") #\B)
	       ((:Pathname "(Wild) Pathname") #\P)
	       ((:SYSTEM "System") #\S #\D)
	       ((:SYSTEM-version "System Version") #\V)
	      )
)


(defparameter *select-tag-table-choice-operations*
  `((:FILE ,#'(lambda ()
	       (LET ((PATHNAME (READ-DEFAULTED-PATHNAME "Tag table:"
			 (PATHNAME-DEFAULTS) "TAGS" :NEWEST)))
		 (READ-TAG-TABLE PATHNAME)
		 *ZMACS-CURRENT-TAG-TABLE*)))
    (:ALL-BUFFERS ,#'(lambda () (COM-SELECT-ALL-BUFFERS-AS-TAG-TABLE)
			     *ZMACS-CURRENT-TAG-TABLE*))
    (:Pathname ,#'(lambda () (com-select-pathname-as-tag-table)
			  *ZMACS-CURRENT-TAG-TABLE*))
    (:SYSTEM ,#'(lambda () (COM-SELECT-SYSTEM-AS-TAG-TABLE)
			*ZMACS-CURRENT-TAG-TABLE*))
    (:SYSTEM-version ,#'(lambda () (COM-SELECT-SYSTEM-version-AS-TAG-TABLE)
			*ZMACS-CURRENT-TAG-TABLE*))
    (:EXISTING ,#'(lambda ()
		    (LET ((TABLE (COMPLETING-READ-FROM-MINI-BUFFER
				   "Tag table (Ctrl-? for list):"
				   *ZMACS-TAG-TABLE-ALIST* NIL NIL
		   "You are typing the name of an existing tags table.")))
		      (COND ((EQUAL TABLE "")
			     (COND (*ZMACS-CURRENT-TAG-TABLE*)
				   (T (BARF))))
			    (T
			     (CDR TABLE))))))
   )
)

  
(DEFUN SELECT-TAG-TABLE (&OPTIONAL (DEFAULT-P T)) "Read a tag table name and return that tag table.
DEFAULT-P non-NIL (as it is if omitted) means if there is an
obvious default than just return it without asking the user at all."
  (COND ((NULL *ZMACS-TAG-TABLE-ALIST*)
	 (funcall (second (assoc (FQUERY `(:CHOICES ,*select-tag-table-choices*)
					 "Specify tag table how? ")
				 *select-tag-table-choice-operations*))))
	((AND DEFAULT-P *ZMACS-CURRENT-TAG-TABLE*)	;for internal commands who want the current one.
	 *ZMACS-CURRENT-TAG-TABLE*)
	((AND DEFAULT-P (NULL (CDR *ZMACS-TAG-TABLE-ALIST*)))
	 ;;next file in current table?
	 (CDAR *ZMACS-TAG-TABLE-ALIST*))
	(T
	 (funcall
	   (second
	     (assoc
	       (fquery
		 `(:choices ,(append *select-tag-table-choices*
				     '(((:existing "Existing Table") #\E))))
		       "Specify tag table how? ")
	       *select-tag-table-choice-operations*))))))


(defcom com-select-system-version-as-tag-table
" Make the files in a system behave like a tags file.
 File version numbers are specified explicitly." ()
  (let ((system-name (read-system-name "System to select as tag table:"))
	(version (prompt-and-read :string "Version specifier:"))
       )
       (let ((files
	       (si:system-source-files system-name si:*source-file-types* nil t)
	     )
	     (name-string (format nil "~A version: ~A" system-name version))
	    )
	    (if (catch-error (send (first files) :new-version version) nil)
		(format *query-io* " - [~A] selected." name-string)
		(ferror nil "~S is an illegal version spec." version)
	    )
	    (select-file-list-as-tag-table
	      (mapcar #'(lambda (path) (send path :new-version version)) files)
	      name-string
	    )
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
   (make-command-alist '(com-select-system-version-as-tag-table))
)


;-------------------------------------------------------------------------------

(defun compare-these-files (path1 path2 t-or-f)
  (let ((file-1 nil)
	(file-2 nil)
       )
       (unwind-protect
	 (progn (setq file-1 (srccom:create-file path1))
		(setq file-2 (srccom:create-file path2))
		(srccom::describe-srccom-sources
		  file-1 file-2 *standard-output*
		)
		(srccom:source-compare-files file-1 file-2
		  *standard-output* t-or-f
		)
	 )
	 (and file-1 (send (srccom:file-stream file-1) :close)
	 )
	 (and file-2 (send (srccom:file-stream file-2) :close)
	 )
       )
  )
)


srccom:
(DEFUN PRINT-DIFFS-1 (FILE DIFF-LINE-NO SAME-LINE-NO &AUX LABEL)
  (SETQ DIFF-LINE-NO (MAX 0 (- DIFF-LINE-NO *LINES-TO-PRINT-BEFORE*))
	SAME-LINE-NO (+ SAME-LINE-NO *LINES-TO-PRINT-AFTER*))
  (FORMAT *OUTPUT-STREAM* "~&**** ~A ~A, Line #~D" (FILE-TYPE FILE) (FILE-NAME FILE)
	  DIFF-LINE-NO)
  (COND
    ((SETQ LABEL (AND *PRINT-LABELS* (LINE-LAST-LABEL FILE DIFF-LINE-NO)))
     (SEND *OUTPUT-STREAM* :STRING-OUT ", After \"")
     (SEND *OUTPUT-STREAM* :STRING-OUT (STRING-REMOVE-FONTS LABEL) 0
	      (MIN (LENGTH LABEL)
;		   (IF (LET ((WHICH-OPERATIONS (SEND *OUTPUT-STREAM* :WHICH-OPERATIONS)))
;		      (AND (MEMBER :READ-CURSORPOS WHICH-OPERATIONS :TEST #'EQ)
;			 (MEMBER :SIZE-IN-CHARACTERS WHICH-OPERATIONS :TEST #'EQ)))
;		     (- (SEND *OUTPUT-STREAM* :SIZE-IN-CHARACTERS)
;			(SEND *OUTPUT-STREAM* :READ-CURSORPOS :CHARACTER) 1)
;		     31)
		   ))
     (WRITE-CHAR #\" *OUTPUT-STREAM*)))
  (WRITE-CHAR #\NEWLINE *OUTPUT-STREAM*)
  (PRINT-FILE-SEGMENT FILE DIFF-LINE-NO SAME-LINE-NO))


(defun fail-match-for (fail-files files1 files2)
  (if fail-files
      (format t "~&No matching files found for:~&	~A~{,~&	~A~}"
	      (first fail-files) (rest fail-files)
      )
      nil
  )
  (values files1 files2)
)

(defun fail-match-for-and-continue
       (fail-files save-file1 save-file2 rest-files1 rest-files2)
  (multiple-value-bind (rest1 rest2)
      (heuristcally-match-up-file-lists rest-files1 rest-files2 fail-files)
    (values (cons save-file1 rest1) (cons save-file2 rest2))
  )
)

(defun allign-on-file (name files rejects)
  (if files
      (if (or (equalp name (send (first files) :name))
	      (let ((path  (send (fs:default-pathname name) :new-version nil))
		    (path2 (send (first files) :new-version nil))
		   )
		   (equalp path path2)
	      )
	  )
	  (values files (reverse rejects))
	  (allign-on-file
	    name (rest files) (cons (first files) rejects)
	  )
      )
      (values nil (reverse rejects))
  )
)


(defun heuristcally-match-up-file-lists (files1 files2 rejects)
  (if (and files1 files2)
      (let ((file1 (send (first files1) :name))
	    (file2 (send (first files2) :name))
	   )
	   (multiple-value-bind (new-files2 rejects2)
	       (allign-on-file file1 files2 nil)
	     (multiple-value-bind (new-files1 rejects1)
		 (allign-on-file file2 files1 nil)
	       (multiple-value-bind (new-files22 rejects22)
		   (if (rest files1)
		       (allign-on-file
			 (send (second files1) :name) files2
			 (list (first files1))
		       )
		       (values nil files2)
		   )
		 (multiple-value-bind (new-files12 rejects12)
		   (if (rest files2)
		       (allign-on-file
			 (send (second files2) :name) files1
			 (list (first files2))
		       )
		       (values nil files1)
		   )
		   (selector (min (length rejects1)  (length rejects2)
				  (length rejects12) (length rejects22)
			     ) eql
		     (((length rejects1))
		      (fail-match-for-and-continue
			(append rejects rejects1) (first new-files1)
			(first files2) (rest new-files1) (rest files2)
		      )
		     )
		     (((length rejects2))
		      (fail-match-for-and-continue
			(append rejects rejects2) (first files1)
			(first new-files2) (rest files1) (rest new-files2)
		      )
		     )
		     (((length rejects12))
		      (fail-match-for-and-continue
			(append rejects rejects12) (first new-files12)
			(second files2)	(rest new-files12) (rest (rest files2))
		      )
		     )
		     (((length rejects22))
		      (fail-match-for-and-continue
			(append rejects rejects22) (second files1)
			(first new-files22) (rest (rest files1))
			(rest new-files22)
		      )
		     )
		   )
		 )
	       )
	     )
	   )
      )
      (if files1
	  (fail-match-for (append rejects files1) nil nil)
	  (if files2
	      (fail-match-for (append rejects files2) nil nil)
	      (fail-match-for rejects nil nil)
	  )
      )
  )
)


(defcom com-tags-source-compare
" Compare the contents of two tag tables.  The output goes on the screen, and
 also into a buffer named *Source Compare ...*." ()
  (format *query-io* "~&Select first tag table:")
  (let ((table1 (select-tag-table nil)))
       (format *query-io* "~&Compare ~A with:" table1)
       (let ((table2 (select-tag-table nil)))
	    (let ((*standard-output*
		    (make-buffer-window-or-broadcast-stream
		      (format nil "*Source Compare ~A / ~A*"
			      (send table1 :name) (send table2 :name)
		      )
		      nil t
		    )
		  )
		  (t-or-f (srccom::query-type))
		 )
	         (format t "~&*Source Compare ~A / ~A* by ~A"
			 (send table1 :name) (send table2 :name) t-or-f
		 )
		 (multiple-value-bind (files1 files2)
		     (heuristcally-match-up-file-lists
		       (send table1 :get 'zmacs-tag-table-file-symbols)
		       (send table2 :get 'zmacs-tag-table-file-symbols)
		       nil
		     )
		   (loop for path1 in files1
			 for path2 in files2
		       do (format t "~&------ Comparison of ~A with~&~
				       ------               ~A"
				  path1 path2
			  )
		          (catch-error (compare-these-files path1 path2 t-or-f))
			  (format t "~&======================================~
				       ======================================~
				       ===="
			  )
		   )
		 )
		 (format *query-io* "~&Done.")
		 (beep)
	    )
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
   (make-command-alist '(com-tags-source-compare))
)


;-------------------------------------------------------------------------------

;;; The Highlight mode/command.


(defvar *user-highlight-preferences*
       '((:Globals             (:upcase     2) t)
	 (:Defined-In-File     (:capitalize 1) nil)
	 (:Keywords            (:downcase   3) nil)
	 (:Not-Defined-In-File (:capitalize 4) nil)
	)
"An Alist of definition types.  Each type is something like
 (:Global <spec> <function calls only>).  The value of the spec can be either
 a font number (zero indexed), :UpCase or :DownCase :Capitalize or a list or
 any of these, e.g. (:Upcase 3).  The third element of the type is a flag
 which, if true, will cause a call to happen only for function calls.  The
 currently supported types for the first element are:
    :Globals             - all symbols in the Lisp and TICL packages.
    :Defined-In-File     - all symbols defined (defun etc.) in the current file.
    :Keywords            - all symbols in the keyword package.
    :Not-Defined-In-File - all symbols defined in a file other than
                           the current file.
    :Else                - all Things which do not satisfy any of the above.
"
)

(defparameter *highlighters*
	      '((:globals             *uppercase-global-function-p*)
		(:Defined-In-File     *highlight-local-p*)
		(:keywords            *highlight-keyword-p*)
		(:Not-Defined-In-File *highlight-external-p*)
		(:else                *highlight-else-p*)
	       )
)

(defparameter *highlight-local-p*
	      'highlight-local-p
"The function called to check whether a symbol should be highlighted.
The function should accept three arguments:
1) The starting BP for the symbol
2) The ending BP for the symbol (also the current BP)
3) The actual string, the symbol-name.
"
)

(defparameter *highlight-else-p*
	      'highlight-else-p
"The function called to check whether a symbol should be highlighted.
The function should accept three arguments:
1) The starting BP for the symbol
2) The ending BP for the symbol (also the current BP)
3) The actual string, the symbol-name.
"
)

(defparameter *highlight-keyword-p*
	      'highlight-keyword-p
"The function called to check whether a symbol should be highlighted.
The function should accept three arguments:
1) The starting BP for the symbol
2) The ending BP for the symbol (also the current BP)
3) The actual string, the symbol-name.
"
)

(defparameter *highlight-external-p*
	      'highlight-external-p
"The function called to check whether a symbol should be highlighted.
The function should accept three arguments:
1) The starting BP for the symbol
2) The ending BP for the symbol (also the current BP)
3) The actual string, the symbol-name.
"
)

(defvar *only-highlight-function-names* nil
"When true only function names are highlighted in highlight mode."
)


;-------------------------------------------------------------------------------

(defvar *logical-pathname-mappings*)
(defvar *source-pathname-mappings*)

(defwhopper (fs:logical-pathname :translated-pathname)
	    (&optional (closest-match nil))
  (if (and (not closest-match) (boundp '*logical-pathname-mappings*))
      (let ((entry (assoc self *Logical-Pathname-Mappings* :Test #'eq)))
	   (if entry
	       (second entry)
	       (let ((result (Continue-whopper closest-match)))
		    (push (list self result) *Logical-Pathname-Mappings*)
		    result
	       )
	   )
      )
      (Continue-whopper closest-match)
  )
)

(defwhopper (fs:pathname :source-pathname) ()
  (if (boundp '*source-pathname-mappings*)
      (let ((entry (assoc self *Source-Pathname-Mappings* :Test #'eq)))
	   (if entry
	       (second entry)
	       (let ((result (Continue-whopper)))
		    (push (list self result) *Source-Pathname-Mappings*)
		    result
	       )
	   )
      )
      (Continue-whopper)
  )
)

(defminor com-highlight-mode
	  highlight-mode "Highlight" 3
	  "Mode for highlighting things." ()
  (command-hook 'highlight-hook *command-hook*)
) 

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-highlight-mode))
)

(defprop highlight-hook 10 command-hook-priority) 

(defun pathname-equal-safe (a b)
  (and (pathnamep a) (pathnamep b) (fs:pathname-equal a b))
)

(defun highlight-hook (char)
"Highlights the atom preceding point if it needs to."
  (if (not (eq *interval* (window-interval *mini-buffer-window*)))
      (highlight-thing-at-bp (point) char)
      nil
  )
) 

(defun highlight-applyer (highlight bp)
"Applys a highlighter."
  (if (and highlight (consp (second highlight)))
      (let ((result
	      (mapcar #'(lambda (x)
			  (highlight-thing
			    bp (first highlight) x (third highlight)
			  )
			)
			(second highlight)
	      )
	    )
	   )
           (if (remove nil result)
	       (first (remove nil result))
	       nil
	   )
      )
      (if highlight
	  (apply #'highlight-thing bp highlight)
	  nil
      )
  )
)

(defun highlight-thing-at-bp
       (bp &optional (char (bp-char bp))
	(highlights *user-highlight-preferences*)
       )
"Highlights atom before BP if it needs to.
 Conditions: (a) last command char must have been a WAM delimiter
	     (except '-' or ':').
             (b) BP must not be inside a comment, string, or slash context.
"
  (and (not (member (make-char char) '(#\- #\:) :Test #'char=))
       ; don't expand after dashes or colons
       (expand-p char)				; expand after WAM delimiters
       ;; don't expand in comments,comp  strings or slashes
       (not (multiple-value-bind (string slash comment)
		(lisp-bp-syntactic-context bp (forward-defun bp -1 t))
	      (or string slash comment)
	    )
       )	 
       ;; okay, check for function name in global package under bp
       (let ((result
	       (loop for highlight in highlights
		     unless (equal :Else (first highlight))
		     collect (highlight-applyer highlight bp)
	       )
	     )
	    )
	    (if (not (member t result))
		(let ((light (assoc :else highlights)))
		     (highlight-applyer light bp)
		)
		nil
	    )
       )
  )
) 


(defun highlight-thing (bp type highlight functions-only)
  "If the atom under BP is a locally defined name, highlight it."
  (let (bp1 str
	(high (assoc type *highlighters* :test #'eq))
       )
       (setq bp1 (forward-atom bp -1 t)			     ; beginning of atom
	     str (string-upcase (string-interval bp1 bp t t)); string with atom
       )
       (cond ((and high
		   (let ((sym (find-symbol str *package*)))
		        (let ((source (if sym (source-file-names sym nil nil))))
			     (funcall (symbol-value (second high))
				      sym source bp1 bp str functions-only
			     )
			)
		   )
	      )
	      (case highlight
		(:UpCase     (upcase-interval     bp1 bp t))
		(:DownCase   (downcase-interval   bp1 bp t))
		(:Capitalize (capitalize-interval bp1 bp t))
		(otherwise (if (numberp highlight)
			       (if (> (length (window-font-alist *window*))
				      highlight
				   )
				   (change-font-interval bp1 bp t highlight)
				   nil
			       )
			       (ferror nil "~S is not a valid highlight."
				       highlight
			       )
			   )
		)
	      )
	      ;; everything after beginning of atom needs redisplay
	      (must-redisplay *window* dis-line (bp-line bp1) (bp-index bp1))
	      t
	     )
	     (t nil)
       )
  )
)

(defun is-function-call-p (start-bp only-functions-p)
"True if the bp Start-bp is a function call form, or only-function-p is not
 nil.
"
  (or (not only-functions-p)
      (let ((previous-string
	      (string-interval (forward-char start-bp -1 t) start-bp)
	    )
	   )
	   (and (plusp (length previous-string))
		(or (char-equal #\( (aref previous-string 0))
		    (char-equal #\' (aref previous-string 0))
		)
	   )
      )
  )
)

;-------------------------------------------------------------------------------

(defun highlight-local-p
       (symbol-found source-files start-bp current-bp string &optional
	(only-functions-p *only-highlight-function-names*)
       )
  "Tests whether or not the current word should be highlighted."
  (declare (values t-or-nil) (ignore current-bp string))
  (and symbol-found
       source-files
       (typep *interval* 'zmacs-buffer)
       (pathnamep (send *interval* :pathname))
       (member (send *interval* :pathname) source-files
	       :test 'pathname-equal-safe
       )
       (is-function-call-p start-bp only-functions-p)
  )
)

;-------------------------------------------------------------------------------

(defun highlight-external-p
       (symbol-found source-files start-bp current-bp string &optional
	(only-functions-p *only-highlight-function-names*)
       )
  "Tests whether or not the current word should be highlighted."
  (declare (values t-or-nil))
  (and symbol-found
       (not (funcall *uppercase-global-function-p* symbol-found source-files
		     start-bp current-bp string only-functions-p
	    )
       )
       source-files
       (typep *interval* 'zmacs-buffer)
       (not (member (send *interval* :pathname) source-files
		    :test #'pathname-equal-safe
	    )
       )
       (is-function-call-p start-bp only-functions-p)
  )
)

;-------------------------------------------------------------------------------

(defun highlight-else-p
       (symbol-found source-files start-bp current-bp string &optional
	(only-functions-p *only-highlight-function-names*)
       )
  "Tests whether or not the current word should be highlighted."
  (declare (values t-or-nil))
  (ignore current-bp string symbol-found source-files)
  (and t;(print symbol-found)
       (is-function-call-p start-bp only-functions-p)
  )
)

;-------------------------------------------------------------------------------

(defun highlight-keyword-p
       (symbol-found source-files start-bp current-bp string &optional
	(only-functions-p *only-highlight-function-names*)
       )
  "Tests whether or not the current word should be highlighted."
  (declare (values t-or-nil) (ignore symbol-found source-files))
  (and (>= (length (the string string)) 1)
       (char-equal #\: (aref string 0))
       (or (and (/= word-delimiter
		   (char-syntax (bp-char (forward-char current-bp -1 t))
				*atom-word-syntax-table*
		   )
		)
		(= word-delimiter
		   (char-syntax (bp-char (forward-char start-bp -1 t))
				*atom-word-syntax-table*
		   )
		)
	   )
	   (and (find-symbol (subseq string 1) 'keyword)
		(or (not only-functions-p)
		    (and (fboundp (find-symbol (subseq string 1) 'keyword))
			 (is-function-call-p start-bp only-functions-p)
		    )
		)
	   )
       )
  )
)

;-------------------------------------------------------------------------------

(defcom com-highlight-region
"Highlight names of things if it should from point to mark.
If there is no region, the current definition is used.
"
  ()
  (let (bp1 bp2
	(*Logical-Pathname-Mappings* nil)
	(*Source-Pathname-Mappings* nil)
	(*undo-save-small-changes* nil)
       )
       (If (window-mark-p *window*)		  ; if there is a region
	   (setq bp1 (point)
		 bp2 (mark)			  ; grab point and mark
	   )
	   (setq bp1 (forward-defun (point) -1 t) ; else use begin and end 
		 bp2 (forward-defun (point) 1 t)  ; of current DEFUN
	   )
       )
       (order-bps bp1 bp2)			  ; put BPs in order
       (do ((bp-temp (forward-char bp1 -1 t)))	  ; move forward one char so we
						  ; don't catch the atom just
						  ; before BP1
	   ((or (bp-= bp2 bp-temp) (bp-< bp2 bp-temp))) ; stop at BP2
	 (setq bp-temp (forward-atom bp-temp 1 t)); move forward one atom
	 (highlight-thing-at-bp
	   bp-temp (bp-char bp-temp)		  ; and try highlight it
	 )
       )
  )
  dis-text
)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-highlight-region))
)


;-------------------------------------------------------------------------------

(defcom com-highlight-buffer
"Highlight names of things if it should in the buffer."
  ()
  (let ((bp1 (interval-first-bp *interval*))
	(bp2 (interval-last-bp *interval*))
	(*Logical-Pathname-Mappings* nil)
	(*Source-Pathname-Mappings* nil)
	(*undo-save-small-changes* nil)
       )
       (do ((bp-temp (forward-char bp1 -1 t)))	  ; move forward one char so we
						  ; don't catch the atom just
						  ; before BP1
	   ((bp-= bp2 bp-temp)) ; stop at BP2
	 (setq bp-temp (forward-atom bp-temp 1 t)); move forward one atom
	 (highlight-thing-at-bp
	   bp-temp (bp-char bp-temp)		  ; and try highlight it
	 )
       )
  )
  dis-text
)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-highlight-buffer))
)


;-------------------------------------------------------------------------------

(defun capitalize-interval (bp1 bp2 &optional ignore)
"Capitalise the interval from bp1 to bp2."
  (do ((i 0 (1+ i))
       (bp)
      )
      ((not (bp-< bp1 bp2)))           ;;;(>= i 1))
    (or (setq bp (forward-to-word bp1)) (return));(barf))
    (or (setq bp1 (forward-word bp)) (return));(barf))
    (do ((ch))
	(nil)
      (setq ch (bp-ch-char bp))
      (and (or (bp-= bp bp1)
	       (alpha-char-p ch)
	   )
	   (return)
      )
      (ibp bp)
    )
    (downcase-interval bp bp1)
    (upcase-char bp)
  )
)


;-------------------------------------------------------------------------------

;;; Patch.

(defvar *global-packages*
	(list (find-package 'lisp)
	      (find-package 'common-lisp)
	      (find-package 'ticl)
	      (find-package 'zlc)
	)
)

(defun uppercase-global-function-p
       (symbol-found source-files start-bp current-bp string
	&optional (only-functions-p t)
       )
  "Tests whether or not the current word should be capitalized."
  (declare (values t-or-nil) (ignore current-bp source-files string))
  (and symbol-found ;;Check to see if symbol is a global function.
       (member (symbol-package symbol-found) *global-packages* :test #'eq)
       (or (functionp symbol-found)
	   (special-form-p symbol-found)
	   (macro-function symbol-found)
       )
       ;;Check to see if we are typing a function name. (after a paren)
       (is-function-call-p start-bp only-functions-p)
  )
)

(DEFUN UPPERCASE-GLOBAL-FUNCTION (BP &OPTIONAL (LOWERCASE NIL))
  "If the atom under BP is a global function or special form, uppercase it."
  (LET (BP1 STR)
    (SETQ BP1 (FORWARD-ATOM BP -1 T)			 ; beginning of atom  
	  STR (STRING-UPCASE (STRING-INTERVAL BP1 BP)))	 ; string with atom
    (COND ((FUNCALL *uppercase-global-function-p*
		    (or (find-symbol str 'lisp) (find-symbol str 'ticl))
		    nil BP1 BP STR t)
	   (UPCASE-INTERVAL BP1 BP T)
	   ;; everything after beginning of atom needs redisplay
	   (MUST-REDISPLAY *WINDOW* DIS-LINE (BP-LINE BP1) (BP-INDEX BP1)))
	  (LOWERCASE
	   (DOWNCASE-INTERVAL BP1 BP T)
	   (MUST-REDISPLAY *WINDOW* DIS-LINE (BP-LINE BP1) (BP-INDEX BP1))))))

;-------------------------------------------------------------------------------


(defun delete-maybe-symbolics-font-shift (current-bp)
  (let ((next (copy-bp current-bp)))
       (let ((next-char (bp-char (forward-char next 1 t))))
	    (cond ((char= next-char #\)
		   (let ((moved-bp (forward-char next 2 t)))
		        (let ((brand-s-p
				(loop for char being the array-elements
				      of sys:*symbolics-epsilon-font-password*
				      when (not (char= (bp-char moved-bp) char))
				      return nil
				      do (setq moved-bp
					       (forward-char moved-bp 1 t)
					 )
				      finally (return t)
				)
			      )
			     )
			     (setq next
				   (if brand-s-p
				       moved-bp
				       (forward-char next 1 t)
				   )
			     )
			)
		   )
		  )
		  ((char= next-char #\()
		   (setq next (forward-sexp (forward-char next 1 t) 1 t))
		  )
		  (t (move-bp next (forward-char next 2 t)))
	    )
       )
       (delete-interval current-bp next)
  )
)

(defun defontify-region (bp1 bp2)
  (let ((current (copy-bp bp1 :Moves)))
       (loop for char = (bp-char current)
	     until (bp-= current bp2)
	     do ;(list (bp-char current) (print current))
	        (if (char-equal char #\epsilon)
		    (delete-maybe-symbolics-font-shift current)
		    (if (> (char-font char) 0)
			(setf (aref (bp-line current) (bp-index current))
			      (int-char (char-code char))
			)
			nil
		    )
	        )
	        (move-bp current (forward-char current))
       )
  )
)

(defcom com-defontify "Remove font shifts in a region." ()
  (let ((bp1 nil)
	(bp2 nil)
	(defun-name nil)
       )
       (if (window-mark-p *window*)
	   (progn (setq bp1 (mark) bp2 (point))
		  (or (bp-< bp1 bp2) (psetq bp1 bp2 bp2 bp1))
		  (if (bp-= (forward-over *whitespace-chars* (mark))
			    (forward-over *whitespace-chars* (point)))
		      (setq *mark-stays* ())
		      (setq defun-name "region"))))
       (cond (defun-name)
	     ((setq bp1 (defun-interval (beg-line (point)) 1 () ()))
	      (setq bp2 (interval-last-bp bp1) bp1 (interval-first-bp bp1))
	      (setq si:*force-defvar-init* t))
	     (t (barf "Unbalanced parentheses")))
       (defontify-region bp1 bp2)
  )
  dis-all
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(COM-defontify))
)

;-------------------------------------------------------------------------------

;;; Patch ZMacs so that the user can select the sort of pathname that
;;; gets defaulted to in Dired.

(defvar zwei:*dired-default-type* nil
"The default type that dired will default to instead of wild.
e.g. '(or \"LISP\" \"DIRECTORY\")"
)

(defvar zwei:*special-directory-to-type-mappings* nil
"An Alist mapping dir names to types to use in dired instead of wild.
e.g. '((\"SITE\" (or \"LISP\" \"SYSTEM\" \"TRANSLATIONS\")))
"
)

(defvar zwei:*dired-default-version* :wild
"The default file version to use in dired."
)

(defun list-if-not (x)
  (if (listp x) x (list x))
)

(defun zwei:get-dired-path-type (pathname wildp lispm-p)
  (and wildp
       (or (let ((dir (first (last (List-If-Not (send pathname :directory))))))
		(let ((entry (assoc dir
				    zwei:*special-directory-to-type-mappings*
				    :test #'string-equal
			     )
		      )
		     )
		     (if entry (second entry) nil)
		)
	   )
	   (and lispm-p *dired-default-type*)
	   wildp
       )
  )
)

(defun physical-host (host)
  (if (send host :host-translation)
      (physical-host (net:parse-host (send host :host-translation)))
      host
  )
)

(defun reasonable-version (version lispm-p)
  (if lispm-p
      version
      (if (consp version)
	  :wild
	  version
      )
  )
)

;;; TI code.
(DEFUN READ-DIRECTORY-NAME (PROMPT PATHNAME &OPTIONAL (WILDP :WILD)
			    &AUX TYPEIN PROMPT-WITH-DEFAULT)
  "Read a pathname to pass to FS:DIRECTORY-LIST.
Prompt with PROMPT, a string probably ending in a colon.
PATHNAME gives the defaults for host, device, directory.
WILDP gives the default used for the other components;
 normally :WILD, but could be NIL."
  ;;; Modded here by JPR.
  (let ((lispm-p
	  (equal (send (physical-host (send pathname :host)) :system-type)
		 :lispm
	  )
	)
       )
       (SETQ PATHNAME
	     (SEND PATHNAME :NEW-PATHNAME
		   :NAME WILDP
		   :TYPE
		   (zwei:get-dired-path-type pathname wildp lispm-p)
		   :VERSION
		   (and wildp
			(or (reasonable-version
			      zwei:*dired-default-version* lispm-p)
			    wildp))))
  )
  (setq PROMPT-WITH-DEFAULT (FORMAT NIL "~A (Default is ~A)" PROMPT PATHNAME))
  (LET ((*READING-PATHNAME-DEFAULTS* PATHNAME)
	(*READING-PATHNAME-SPECIAL-TYPE* (send pathname :type))
	(*READING-PATHNAME-SPECIAL-VERSION* (send pathname :version))
	(*READING-PATHNAME-DIRECTION* :READ)
	(*MINI-BUFFER-VALUE-HISTORY* *PATHNAME-ARGUMENT-HISTORY*))
    (MULTIPLE-VALUE-BIND (NIL NIL INTERVAL)
	(EDIT-IN-MINI-BUFFER
	  *PATHNAME-READING-COMTAB* NIL NIL
	  (LIST PROMPT-WITH-DEFAULT '(:RIGHT-FLUSH " (Completion)")))
      (SETQ TYPEIN (STRING-INTERVAL INTERVAL))))
  (COND ((EQUAL TYPEIN "")
	 (PUSH-ON-HISTORY PATHNAME *PATHNAME-ARGUMENT-HISTORY*)
	 (PROMPT-LINE "~A ~A" PROMPT PATHNAME)
	 PATHNAME)
	(T
	 (LET ((PATHNAME
		 (FS:MERGE-PATHNAME-DEFAULTS TYPEIN PATHNAME :WILD :WILD)))
	   (PUSH-ON-HISTORY PATHNAME *PATHNAME-ARGUMENT-HISTORY*)
	   (PROMPT-LINE "~A ~A" PROMPT PATHNAME)
	   PATHNAME))))

(defparameter zwei:*open-dired-directories-with-inherited-constraints* t)


;;; TI code.
(DEFUN DIRED-OPEN-LINE-SUBDIRECTORY (LINE)
  "Given a DIRED buffer line:
    If it describes an expanded directory, close its display.
    If it describes an unexpanded directory, expand its display.
    If it describes a normal file, close its parent's directory's display,
      except that top-level displays are left alone.
Returns nothing significant."
  (LET* ((*BATCH-UNDO-SAVE* T)
	 PATHNAME
	 DIRECTORY
	 (LEVEL (DIRED-LINE-LEVEL LINE))
	 (DIRECTORY-P (GETF (LINE-PLIST LINE) :DIRECTORY)))
    (WITH-READ-ONLY-SUPPRESSED (*INTERVAL*)
      (DIRED-LINE-PATHNAME-OR-BARF LINE)	     ; verify that line describes a file
      (IF DIRECTORY-P
	  ;; current line is a directory
	  (IF (GETF (LINE-PLIST LINE) 'CONTENTS-PRESENT)
	      ;; directory already expanded, close it
	      (DIRED-CLOSE-LINE-SUBDIRECTORY LINE)
	      ;; directory unexpanded, open it
	      (PROGN
		(SETQ PATHNAME ;;; JPR
		      (if (and zwei:*open-dired-directories-with-inherited-constraints*
			       (not (tv:key-state :Shift))
			  )
			  (let ((path (buffer-pathname *interval*)))
			       (SEND (SEND (DIRED-LINE-PATHNAME LINE)
					   :PATHNAME-AS-DIRECTORY) :NEW-PATHNAME
				     :NAME (send path :Name)
				     :TYPE (send path :Type)
				     :VERSION (send path :Version)
			       )
			  )
			  (SEND (SEND (DIRED-LINE-PATHNAME LINE) :PATHNAME-AS-DIRECTORY) :NEW-PATHNAME
				:NAME :WILD :TYPE :WILD :VERSION :WILD)
		      )
		)
		(SETQ DIRECTORY (FS:DIRECTORY-LIST PATHNAME :DELETED :SORTED))
		(SETF (GETF (LINE-PLIST LINE) 'CONTENTS-PRESENT) T)
		(LET ((NEXT-PLIST (LINE-PLIST (LINE-NEXT LINE)))
		      (STREAM (INTERVAL-STREAM-INTO-BP (CREATE-BP (LINE-NEXT LINE) 0))))
		  (DIRED-INSERT-DIRECTORY DIRECTORY STREAM (1+ (DIRED-LINE-LEVEL LINE)))
		  ;; Restore the plist, now clobbered, of the following line.
		  (SETF (LINE-PLIST (BP-LINE (SEND STREAM :READ-BP))) NEXT-PLIST))
		(DIRED-REGENERATE-BLOCKS-IN-FILES-LISTED)))
	  ;; current line is a regular file so its directory is expanded, close it
	  (IF (= LEVEL 0)
	      ;; don't close a top-level directory
	      (BARF "Cannot close top-level directory.")
	      ;; look for line containing this directory name and close it
	      (DIRED-CLOSE-LINE-SUBDIRECTORY
		(DIRED-PATHNAME-DIRECTORY-LINE (DIRED-LINE-PATHNAME LINE))))))))



;;; TI code.
(DEFCOM COM-DIRED-EDIT-FILE "Edit the current file; or DIRED it if it's a directory" ()
  (OR (TYPEP *WINDOW* 'ZMACS-WINDOW) (BARF))
  (WHEN (GETF (LINE-PLIST (BP-LINE (POINT))) :DELETED)
    (BARF "Can't edit a deleted file."))
  (LET* ((LINE (BP-LINE (POINT)))
	 (DIR-P (GETF (LINE-PLIST LINE) :DIRECTORY))
	 (PATHNAME (DIRED-LINE-PATHNAME-OR-BARF LINE)))
    (WHEN (AND (NOT DIR-P)			; Reversion doesn't make sense for recursive Dired
	       (NOT (GETF (LINE-PLIST LINE) :LINK-TO))	; Links may point to other directories.
	       (GETF (LINE-PLIST LINE) :NEWEST))
      (SETQ PATHNAME (NEED-TO-REVERT-BUFFER PATHNAME))
      (WHEN (NULL PATHNAME)
	(BARF "Re-enter edit command.")))
    (W:WITH-MOUSE-GRABBED			; ignore mouse clicks while we find the file
      (IF DIR-P ;;; JPR
	  (if (and zwei:*open-dired-directories-with-inherited-constraints*
		   (not (tv:key-state :Shift))
	      )
	      (let ((path (buffer-pathname *interval*)))
		   (DIRECTORY-EDIT (SEND (SEND PATHNAME :PATHNAME-AS-DIRECTORY)
					 :New-Pathname
					 :NAME (send path :Name)
					 :TYPE (send path :Type)
					 :VERSION (send path :Version)
				   )
		   )
	      )
	      (DIRECTORY-EDIT (SEND (SEND PATHNAME :PATHNAME-AS-DIRECTORY)
				    :NEW-PATHNAME :NAME :WILD :TYPE :Wild
				    :VERSION :WILD))
	  )
	  (PROGN
	    (FIND-FILE PATHNAME)
	    (LET ((BLURB (KEY-FOR-COMMAND 'COM-SELECT-PREVIOUS-BUFFER
					  *COMTAB* NIL NIL #\c-m-L)))
	      (AND (NULL BLURB)
		   (SETQ BLURB (KEY-FOR-COMMAND 'COM-SELECT-BUFFER))
		   (SETQ BLURB (STRING-APPEND BLURB " Return")))
	      (AND BLURB
		   (FORMAT *QUERY-IO* "~&Type ~A to return to DIRED" BLURB))
	      DIS-TEXT))))))

;-------------------------------------------------------------------------------

;;; TICLOS related fixes...

;;; The following is a patched version of TI's patch file.

#+CLOS
(defparameter *package-to-method-section-name-mappings*
	      '((pcl "PCL:METHOD")
		(lisp "TICLOS:METHOD")
	       )
)

#+CLOS
(defun method-section-name-for-buffer ()
  (let ((entry (find-if #'(lambda (x)
			    (and (find-package (first x))
				 (or (equal *package* (find-package (first x)))
				     (member (find-package (first x))
					     (si:package-use-list *package*)
				     )
				 )
			    )
			  )
			  *package-to-method-section-name-mappings*
	       )
	)
       )
       (if entry
	   (read-from-string (second entry))
	   'ticlos:method
       )
  )
)

	       
#+CLOS
#!C
; From file ZMACS-CHANGES.LISP.NEWEST >CLOS>DEV> SYMA:
#8R ZWEI#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "ZWEI"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* SYS:COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "TICLOS:TICLOS;ZMACS-CHANGES.."


;; 10/17/88 DNG Changed 'GLOBAL:STRING to 'STRING to enable optimization of CONCATENATE.
;;  1/06/89 DNG Fixed DEFMETHOD handling to work for both CLOS and Flavor methods.
;;;Edited by RICE                  16 Jan 90  15:35
(DEFUN (:PROPERTY :LISP GET-SECTION-NAME) (LINE BP &AUX STR SYM ERROR-P
					   IDX TEMP TYPE END-IDX (EOF "") NON-FONT-LINE )
  (IF (NOT (AND (> (LENGTH LINE) 1)
		(CHAR= (MAKE-CHAR (AREF LINE 0)) #\()))
      (VALUES NIL NIL T)
      (PROGN
	(SETQ ERROR-P T)
	(WHEN (or (AND (SYS:%STRING-EQUAL LINE 0 "(DEF" 0 4)
		       (NOT (SYS:%STRING-EQUAL LINE 0 "(DEFPROP " 0 11))
		       (SETQ IDX (STRING-SEARCH-SET *WHITESPACE-CHARS* LINE))
		       (SETQ IDX (STRING-SEARCH-NOT-SET *WHITESPACE-CHARS* LINE IDX)))
		  ;;; JPR code.
		  (and *section-defining-items*
		       (progn (SETQ TEMP (STRING-SEARCH-SET *WHITESPACE-CHARS* LINE))
			      (SETQ IDX (STRING-SEARCH-NOT-SET *WHITESPACE-CHARS* LINE (or TEMP 0)))
			      idx)
		       temp
		       (setq TYPE (test-section-defining-items *section-defining-items* line temp))))
	  (SETQ ERROR-P NIL)
	  (SETQ NON-FONT-LINE (STRING-REMOVE-FONTS LINE))
	  (CONDITION-CASE () (SETF (VALUES SYM END-IDX)
				   (READ-FROM-STRING NON-FONT-LINE NIL EOF :START IDX))
	    (:NO-ERROR
	     (IF (EQUAL SYM EOF)
		 (SETQ ERROR-P T)
		 (SETQ STR (SUBSEQ NON-FONT-LINE IDX (MIN (LENGTH LINE) END-IDX)))))
	    (SYSTEM:READ-ERROR
	     (SETQ STR (GET-DEFUN-NAME (MOVE-BP BP LINE 0)))))
	  (UNLESS ERROR-P
	    (if (and (or (sys:%string-equal NON-FONT-LINE 0 "(DEFMETHOD " 0 13)
			 (eq :always-clos-method type))
		     (not (ticlos:flavor-method-spec-p sym)))
		;; Looks like a CLOS method definition.
		(progn 
		  (setf (values sym str error-p)
			(GET-DEFmethod-arglist (MOVE-BP BP LINE 0)))
		  (and (and sym (null error-p))
		       (progn
			 (setf (car (last sym)) (arglist-type-specifiers (car (last sym))))
			 (setf sym `(,(Method-Section-Name-For-Buffer) ,@sym)))))
		(MULTIPLE-VALUE-SETQ (SYM NIL ERROR-P)
		  (SYMBOL-FROM-STRING STR NON-FONT-LINE NIL SYM line)))))
	(WHEN ERROR-P
	  (SETQ SYM (CONCATENATE 'STRING
				 (THE STRING
				      (LET ((BUFFER (NODE-TOP-LEVEL-NODE (LINE-NODE LINE))))
					(IF (BUFFER-PATHNAME BUFFER)
					    (LET ((NAME (PATHNAME-NAME (BUFFER-PATHNAME BUFFER))))
					      (IF (CONSP NAME)
						  (APPLY 'STRING-APPEND
							 (MAPCAR #'(LAMBDA (NAME-ELT) (IF (CONSP NAME-ELT)
											  (CAR NAME-ELT)
											  NAME-ELT))
								 NAME))
						  (STRING NAME)))
					    (BUFFER-NAME BUFFER))))
				 "-"
				 (THE STRING
				      (LET ((START-INDEX (POSITION #\(
								   (THE STRING (STRING LINE))
								   :TEST-NOT #'CHAR-EQUAL)))
					(SUBSEQ LINE START-INDEX
						(AND START-INDEX
						     (STRING-SEARCH-SET *WHITESPACE-CHARS*
									LINE
									START-INDEX)))))
				 "-"
				 (PRIN1-TO-STRING (INCF *SECTION-COUNT*)))
		STR SYM))
	(VALUES SYM STR NIL))))
)) 


#!C
; From file zmacs-changes.LISP.NEWEST >CLOS>DEV> SYMA:
#8R ZWEI#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "ZWEI"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* SYS:COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "TICLOS:TICLOS;ZMACS-CHANGES.#"

(DEFUN GET-DEFmethod-arglist (BP &AUX BP1 str idx arglist x)
  (declare (values list string error-p))
  ;; The first value returned should be: (function-spec {qualifier}* arglist)
  ;;  3/9/89 DNG - Fixed to not error if the arglist can't be found.
  (AND (SETQ BP (FORWARD-ATOM BP))		;skip defmethod
       (SETQ BP (FORWARD-OVER *WHITESPACE-CHARS* BP))
       (setq bp1 bp)
       (loop
	 (unless (and (SETQ BP1 (forward-sexp bp1))
		      (setq bp1 (SKIP-OVER-BLANK-LINES-AND-COMMENTS Bp1))
		      (setq bp1 (forward-over *blanks* bp1))
		      (not (char= (BP-CH-CHAR BP1) #\()))
	   (if  (and BP1 (char= (BP-CH-CHAR BP1) #\())
		(progn
		  (setq bp1 (FORWARD-SEXP BP1)
			idx 0)
		  (or (and bp1
			    (setf str (STRING-REMOVE-FONTS (STRING-INTERVAL BP BP1)))
			    (ignore-errors
			       (return
				 (values
				   (loop (setf (values x idx)
					       (read-from-string str nil 'eof :start idx))
					 (if (neq x 'eof)
					     (push (dequote-eql-args x) arglist)
					   (if (listp (first arglist))
					        (return (nreverse arglist))
					      ;; else not a valid arglist
					      (return-from get-defmethod-arglist (values nil nil t)))
					      ))
				   str nil))))
		       (return (values nil nil t))))
	     (return (values nil nil t)))))))))


(defun dequote-eql-args (args)
  (if (consp args)
      (mapcar #'(lambda (arg)
		  (if (and (consp arg)
			   (consp (second arg))
			   (equal 'eql (first (second arg)))
			   (consp (second (second arg)))
			   (equal 'quote (first (second (second arg))))
		      )
		     `(,(first arg) (eql ,(second (second (second arg))))
		       ,@(rest (rest arg))
		      )
		      arg
		  )
		)
		args
      )
      args
  )
)


;;; JPR TICLOS/ZMACS code.

#+CLOS
(pushnew '(ticlos:method section-name-for-ticlos-method)
	  *Non-Standard-Section-Type-Processors*
	  :test #'equalp
)

#+CLOS
(let ((string "(DEFGENERIC"))
     (pushnew (list (length string) string :Always-CLOS-Method)
	      zwei:*section-defining-items* :test #'equalp
     )
)
#+CLOS
(let ((string "(TICLOS:DEFMETHOD"))
     (pushnew (list (length string) string :Always-CLOS-Method)
	      zwei:*section-defining-items* :test #'equalp
     )
)
#+CLOS
(let ((string "(TICLOS::DEFMETHOD"))
     (pushnew (list (length string) string :Always-CLOS-Method)
	      zwei:*section-defining-items* :test #'equalp
     )
)
#+CLOS
(let ((string "(P-DEFGENERIC-OPTIONS"))
     (pushnew (list (length string) string :Always-CLOS-Method)
	      zwei:*section-defining-items* :test #'equalp
     )
)

;-------------------------------------------------------------------------------

;;; More JPR TICLOS/ZMACS code.

#+CLOS
(defun find-actual-class (name)
  (if (ticlos:class-named name t)
      name
      (first (package-lookalike-symbols name nil '(ticlos:class-def)))
  )
)

(defun second-if-list (x)
  (if (consp x) (second x) x)
)

#+CLOS
(defun specializers-match (spec specializers)
  (if spec
      (if specializers
	  (if (subtypep (ticlos:class-name (first specializers))
			(second-if-list (find-actual-class (first spec)))
	      )
	      (specializers-match (rest spec) (rest specializers))
	      (specializers-match spec (rest specializers))
	  )
	  (values nil nil)
      )
      :maybe
  )
)

#+CLOS
(defun find-matching-method (spec meths ok-to-ask)
  (if meths
      (let ((specializers (ticlos:method-parameter-specializers (first meths))))
	   (let ((match?
		   (if (and (not (consp (first spec)))
			    (not (member (first spec)
				   (function-name
				     (ticlos:method-function (first meths))
				   )
				 )
			    )
		       )
		       nil
		       (specializers-match
			 (if (consp (first spec))
			     (first spec)
			     (second spec)
			 )
			 specializers
		       )
		   )
		 )
		)
	        (if (or (equal match? t)
			(and (equal match? :maybe)
			     ok-to-ask
			     (fquery '(:select t) "do you mean ~S? "
				     (function-name
				       (ticlos:method-function (first meths))
				     )
			     )
			)
		    )
		    (values
		      (function-name (ticlos:method-function (first meths))) t
		    )
		    (find-matching-method spec (rest meths) ok-to-ask)
		)
	   )
      )
      (values nil nil)
  )
)

;;;Edited by RICE                  16 Jan 90  14:12
#+CLOS
(defun section-name-for-ticlos-method (spec ok-to-ask line)
  (ignore line)
  (if (si:fdefinition-safe (cons 'ticlos:method spec))
      (values spec t)
      (let ((gf-name (first spec)))
	   (multiple-value-bind (gf found-p)
	      (if (and (fboundp gf-name)
		       (ticlos:generic-function-p (symbol-function gf-name))
		  )
		  (values gf-name t)
		  (if ok-to-ask
		      (dolist (symbol (package-lookalike-symbols gf-name))
			(if (and (fboundp symbol)
				 (ticlos:generic-function-p
				   (symbol-function symbol)
				 )
				 (fquery '(:select t) "do you mean ~S? "
					 symbol
				 )
			    )
			    (return symbol t)
			)
		      )
		      nil
		  )
	      )
	     (if found-p
		 (if (or (rest spec) ok-to-ask)
		     (let ((meths
			     (if (fboundp 'tv:generic-function-methods-safe)
				 (tv:generic-function-methods-safe
				   (tv:function-generic-function-safe
				     (symbol-function gf)
				   )
				 )
				 nil
			     )
			   )
			  )
			  (find-matching-method (rest spec) meths ok-to-ask)
		     )
		     (values gf t)
		 )
		 (values nil nil)
	     )
	   )
      )
  )
)


;-------------------------------------------------------------------------------


;;; Make sure we get reasonable behaviour for M-. on generic functions.

(DEFUN EDIT-DEFINITION-1 (OBJECT &OPTIONAL (OBJECTS (LIST OBJECT))
			  STRING DEFINITION-GENERIC-PATHNAME DEFINITION-TYPE
			  &AUX DEF)
  "Visit the definition(s) of OBJECT, or OBJECTS.
STRING will be used eventually to look for look-alike objects,
so it should be a printed representation of OBJECT.
If OBJECTS is T, only OBJECT precisely is used, no matter how desperate the user gets.
Then STRING is not needed.
DEFINITION-GENERIC-PATHNAME restricts to definitions in that file,
and DEFINITION-TYPE restricts to that type of definition.
DEFINITION-TYPE should be something like DEFUN, DEFVAR, etc., or NIL."
  (when (and (symbolp object)
	     (fboundp object)
	     (fboundp 'tv:generic-function-p-safe)
	     (tv:generic-function-p-safe (symbol-function object)))
    (let ((choice (tv:menu-of-methods object)))
      (if choice
	  (setq object choice)
	  (if (zwei:source-file-names object 'clos:defgeneric)
	      nil
	      (setq object nil))))
    (setq objects (list object)))
  (when object ;; if they didn't choose, just quit.
  (SETQ OBJECT (FLUSH-INTERNAL-SPEC OBJECT))
  (IF (CONSP OBJECTS)
      (SETQ OBJECTS (MAPCAR 'FLUSH-INTERNAL-SPEC OBJECTS)))
  (AND (OR (EQ OBJECTS T)
	   (AND (NULL (CDR OBJECTS))
		(EQUAL (CAR OBJECTS) OBJECT)))
       (SETQ DEF (ONLY-DEFINITION-OR-NIL OBJECT DEFINITION-TYPE)))
  (IF DEF
      ;; If there is only one definition of this object that could possibly be meant,
      ;; and it is in a buffer and still real, just go there.
      ;; Don't do any hacking with the possibilities buffer.
      ;; This is probably the most common case, so it should be fast.
      ;; If there is only one definition but it is in a file,
      ;; that is going to be slow enough anyway so no need to special case it.
      (PROGN
	(SOURCE-FILE-NAMES OBJECT T DEFINITION-TYPE)	;Print names of any patch files.
	(POINT-PDL-PUSH (POINT) *WINDOW* T)
	(MAKE-BUFFER-CURRENT (CAR DEF))
	(MOVE-BP (POINT) (CDR DEF) 0)
	(RECENTER-WINDOW *WINDOW* :START
			 (BACKWARD-OVER-COMMENT-LINES (OR (BACKWARD-OVER-PACKAGE-PREFIX (POINT))
							  (POINT))
						      NIL)))
      (INSERT-EDIT-DEFINITION-POSSIBILITY OBJECT OBJECTS STRING
					  DEFINITION-GENERIC-PATHNAME
					  DEFINITION-TYPE))))


;-------------------------------------------------------------------------------

zwei:
(advise dired-close-line-subdirectory :around :check-for-updates nil
  (let ((line (first arglist)))
    (do ((line (line-next line) (line-next line))
	 (thislevel (dired-line-level line))
	 delete-files
	 undelete-files
	 find-files
	 print-files
	 apply-files)
	((let ((linelevel (dired-line-level line)))
	   (or (null linelevel)
	       (<= linelevel thislevel)))
	 (if (or delete-files undelete-files find-files print-files apply-files)
	     (or (y-or-n-p "There are marked lines.  Proceed?")
		 (barf))
	     t))
      (cond ((dired-line-pathname line)
	     (case (aref line 0)
	       (#\D (push line delete-files))
	       (#\U (push line undelete-files))
	       (#\F (push line find-files))
	       (#\P (push line print-files))
	       (#\A (push line apply-files)))))))
  :Do-It)

;-------------------------------------------------------------------------------

(defvar *namespace* (first (remove "BOOT" (name::list-known-namespaces))))

(defun get-host-string (host)
  (let ((real-host
	  (typecase host
	    (string (net:parse-host host t))
	    (pathname (send host :host))
	    (otherwise host)
	  )
	)
	(namespace (string-append *namespace* "|"))
       )
       (if real-host
	   (let ((name-string (if (send real-host :host-translation)
				  (send real-host :host-translation)
				  (send real-host :name)
			      )
		 )
		)
		(subseq name-string
			(if (lisp:search namespace name-string
					 :test #'char-equal
			    )
			    (+  (length namespace)
				(lisp:search namespace name-string
					     :test #'char-equal
				)
			    )
			    0
			)
		)
	   )
	   nil
       )
  )
)

(defvar *make-files-that-are-not-mine-read-only* nil
"When true Zmacs will make any buffer read-only by default unless it
either comes from a host whose is the same as the current user id, or
if the user's name is in the directory path.  For example, if your name is FOO
any you try to edit the file bar:foo;frob.lisp then this will not be made
read-only, but sys:debug-tools;inspect.lisp will be unless there is a logical
host called FOO that points to SYS."
)

(defvar *directories-that-I-own* nil
"A list of directories that the read-only buffer feature recognises as belonging
to you and thus will not cause files to be read-only."
)

;;;Edited by James Rice            6 Feb 90  11:45
(advise zwei:find-file :around :Maybe-Check-Read-Only nil
  (let ((results :Do-it))
       (loop for result in (if (listp results) results (list results)) do
	     (if (and *make-files-that-are-not-mine-read-only*
		      (boundp 'zwei:*interval*)
		      zwei:*interval*
		      result
		      (send result :pathname)
		 )
		 (let ((host      (get-host-string (send result :pathname)))
		       (directory (and (probe-file (send result :pathname))
				       (send (send (send result :pathname)
						   :Truename
					     )
					     :Directory
				       )
				  )
		       )
		       (me        (get-host-string fs:user-id))
		      )
		      (if (and (not zwei:*edit-definition-is-read-only*)
			       (not (string-equal host me))
			       (not (member (string fs:user-id) directory
					    :Test #'string-equal
				    )
			       )
			       (loop for dir in *Directories-That-I-Own*
				     when (member (string dir) directory
						  :Test #'string-equal
					  )
				     return nil
				     finally (return t)
			       )
			  )
			  (Zwei:make-buffer-read-only result)
			  nil
		      )
		 )
		 nil
	     )
       )
       results
  )
)

;-------------------------------------------------------------------------------

(defmethod (zwei :after :mouse-select) (&rest ignore)
  "When mouse-selecting a Zmacs frame, make sure that the frame's *WINDOW*
  becomes the TV:SELECTED-WINDOW.  If we don't do this, then the cursor of the
  window in which the user clicked will be blinking, which is not always right,
  because that is not always the listening window."
  (let* ((selected-editor-window
	   (funcall editor-closure 'symbol-value '*window*)))
    (unless (eq self selected-editor-window)
      (send selected-editor-window :select))))


;-------------------------------------------------------------------------------

(defvar *definitions-edited* nil
"A list of all of the definitions that have been edited."
)

(defvar *sort-completions-p* t
  "When true, minibuffer completions are sorted
with string-lessp before they are displayed."
)

(defvar *select-previous-definitions-for-c-u-m-.-p* nil
"When true will give a completing minibuffer of the previous definitions."
)

;;; Record all of the definitions we make.
(advise edit-definition-1 :Around :Keep-History nil
  (let ((spec (first arglist)))
       (setq *Definitions-Edited*
	     (Cons spec (remove spec *Definitions-Edited* :Test #'equalp))
       )
  )
  :Do-It
)

;;; Put in a hook so that c-U M-. will let us choose from past definitions.
(advise com-edit-definition :Around :keep-history nil
  (if (and *Select-Previous-Definitions-For-C-U-M-.-P*
	   (Equal *numeric-arg-p* :control-u)
      )
      (let ((*numeric-arg-p* nil)
	    (*numeric-arg* nil)
	    (*sort-completions-p* nil)
	   )
	   (letf ((#'read-function-name
		   #'(lambda (prompt &rest ignore)
		       (let ((name (zwei:completing-read-from-mini-buffer
				     prompt
				     (mapcar #'(lambda (x)
						 (list (format nil "~S" x) x)
					       )
					     *Definitions-Edited*
				     )
				     nil
				   )
			     )
			    )
			    (values (second name)
				    (let ((*package* nil))
				         (format nil "~S" (second name))
				    )
				    t
			    )
		       )
		   )
		  )
		 )
		 :Do-It
	   )
      )
      :Do-It
  )
)


;;; TI code modified so that completions are only sorted by default.  We
;;; want to switchthis off for c-U M-. so that we get the temporal sequence
;;; in which we M-.ed.
(DEFUN LIST-COMPLETIONS-INTERNAL (POSS &AUX LEN)
  (SETQ LEN (LENGTH POSS))
  (COND ((ZEROP LEN)
	 (FORMAT T
	   "~&There are no known completions of the text you have typed.~%"))
	((= LEN 1)
	 (FORMAT T "~&The only known completion of the text you have typed is ")
	 (SEND *STANDARD-OUTPUT* :ITEM 'COMPLETION (CAAR POSS)) (FORMAT T ":~%")
	 (COND (*COMPLETING-DOCUMENTER*
		(TERPRI *STANDARD-OUTPUT*)
		(FUNCALL *COMPLETING-DOCUMENTER* (CAR POSS)))))
	((OR (< LEN 62)
	     (LET ((*QUERY-IO* *STANDARD-OUTPUT*))
	       (FQUERY NIL "There are ~D known possibilities, do you really want to see them all? "
		       LEN)))
	 (FORMAT T "~&These are the known completions of the text you have typed:~2%")
	 (SEND *STANDARD-OUTPUT* :ITEM-LIST 'COMPLETION
	       (if *Sort-Completions-P* ;;; JPR.
		   (Sort (MAPCAR #'CAR POSS) #'STRING-LESSP)
		   (MAPCAR #'CAR POSS)))
	 (TERPRI *STANDARD-OUTPUT*))))


;;; TI code modified by JPR so that the items presented are ordered
;;; like in a phone book, not horizontally.
tv:
(DEFMETHOD (tv:BASIC-MOUSE-SENSITIVE-ITEMS :ITEM-LIST)
	   (TYPE LIST &AUX (MAXL 0) N (INSIDE-WIDTH (SHEET-INSIDE-WIDTH)))
  "Type out list of item as many as will fit on each line, centered."
  (SEND SELF :FRESH-LINE)
  (COND
    (LIST					;Do nothing if empty list
     ;; Compute the maximum width of any item, in dots (MAXL).
     (DOLIST (ITEM LIST)
       (LET ((STRING (STRING (IF (CONSP ITEM)
				 (CAR ITEM)
				 ITEM))))
	 (SETQ MAXL (MAX (SHEET-STRING-LENGTH SELF STRING) MAXL))))
     ;; How many items go on each line (except the last)?
     (SETQ N
	   (MAX
	     (MIN (TRUNCATE INSIDE-WIDTH (+ MAXL (FONT-CHAR-WIDTH CURRENT-FONT))) (LENGTH LIST))
	     1))				;Always print something,
						; even if continuation.
     ;; JPR. Reorder the items so that they are vertically ordered, not
     ;; horizontally.
     (setq list (remove '_remove_me_
			(w:reorder-into-n-columns list n '_remove_me_)
		)
     )
     ;; Now print the items and store the data in the table.
     ;; Move to a new line when we exhaust a line, and at the end.
     ;; I counts from 1 thru N on each line.
     (DO ((I 1 (1+ I))
	  (LIST LIST (CDR LIST))
	  (WIDTH-PER (TRUNCATE INSIDE-WIDTH N)))
	 ((NULL LIST))
       ;; Actually make this item.
       (IF (CONSP (CAR LIST))
	   (SEND SELF :ITEM TYPE (CDAR LIST) "~A" (CAAR LIST))
	   (SEND SELF :ITEM TYPE (CAR LIST)))
       ;; Space out for next item, or move to new line.
       (IF (AND (NOT (= I N))
		(CDR LIST))
	   ;; Not end of line, space out for next item.
	   (MULTIPLE-VALUE-BIND (X Y)
	       (SHEET-READ-CURSORPOS SELF)
	     (SHEET-SET-CURSORPOS SELF (* WIDTH-PER (TRUNCATE (+ (1- WIDTH-PER) X) WIDTH-PER)) Y))
	   ;; else end of line
	   (PROGN
	     (SEND SELF :TYO #\NEWLINE)
	     (SETQ I 0))))))
  (MOUSE-WAKEUP)
  NIL)

;-------------------------------------------------------------------------------

(defflavor file-pipe-stream
	   ((pathname nil))
	   (tv:pipe-stream)
  :Initable-Instance-Variables
  (:Default-Init-Plist :Buffer (make-string 2000))
)

(defmethod (file-pipe-stream :Pathname) ()
  pathname
)

(defmethod (file-pipe-stream :line-in) (arg)
  (sys:stream-default-handler self :Line-In arg nil)
)

(defcom com-unfasl "UnFASLs a file." (km r)
  (let ((path (read-defaulted-pathname
		"UnFASL file:"
		(send (fs:default-pathname (pathname-defaults))
		      :new-pathname :type :xld
		      :version nil)
		nil nil :read t nil)))
    (let ((pipe-stream
	    (make-instance 'file-pipe-stream :Function
			   #'(lambda (stream)
			       (let ((*standard-output* stream))
				    (sys:unfasl-print path)
			       )
			     )
			   :Pathname path
	    )
	  )
	 )
         (letf ((#'fs:extract-attribute-list #'(lambda (&rest ignore) nil)))
	       (unwind-protect (view-stream pipe-stream)
		 (send pipe-stream :Close)
	       )
	 )
    )
    dis-none
  )
)


(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-unfasl))
)

;-------------------------------------------------------------------------------

(defcom Com-Move-To-Menu-Selected-Section-From-Point-Pdl
   "Give a completing minibuffer of section names of points to go to and go to the selected point." ()
  (let ((all-points (WINDOW-POINT-PDL *WINDOW*)))
    (let ((sections (loop for point in all-points
			  for (bp) = point
			  for node = (line-node (bp-line bp))
			  for node-name = (section-node-name node)
			  for string-node-name =
			    (cond ((or (symbolp node-name)
				       (stringp node-name)
				   )
				   (string node-name)
				  )
				  (t (format nil "~S" node-name))
			    )
			  collect (cons string-node-name point)
		    )
	  )
	 )
         (let ((filtered-sections
		 (remove-duplicates sections :Key #'first)
	       )
	      )
	      (let ((choice (completing-read-from-mini-buffer
			      "Go to section:" filtered-sections nil
			    )
		    )
		   )
		   (if choice
		       (destructuring-bind (bp pline) (rest choice)
			 (point-pdl-move bp pline)
		       )
		       (beep)
		   )
	      )
	 )
     )
  )
  dis-bps
)

(tv:install-zmacs-commmand
  '(#\s-c-space com-move-to-menu-selected-section-from-point-pdl))

;-------------------------------------------------------------------------------
;;; TI code.
;;; Patched by JPR so that c-m-y works (a bit) with m-. etc.
(DEFUN zwei:READ-FUNCTION-NAME (PROMPT &OPTIONAL DEFAULT MUST-BE-DEFINED STRINGP HELP ;;gsl 3-14-85
			   &AUX EXPLICIT-PACKAGE-P PROMPT-WITHOUT-DEFAULT
			   (*MINI-BUFFER-DEFAULT-STRING*
			     (and default (prin1-to-string 	;; may 05/09/90
					    DEFAULT)))
			   (READ-FUNCTION-NAME-MUST-BE-DEFINED
			     MUST-BE-DEFINED)
			   (READ-FUNCTION-NAME-OLD-GLOBAL-MOUSE-CHAR-BLINKER-HANDLER
			     *GLOBAL-MOUSE-CHAR-BLINKER-HANDLER*)
			   (READ-FUNCTION-NAME-OLD-GLOBAL-MOUSE-CHAR-BLINKER-DOCUMENTATION-STRING
			     *GLOBAL-MOUSE-CHAR-BLINKER-DOCUMENTATION-STRING*)
			   (READ-FUNCTION-NAME-OLD-MOUSE-FONT-CHAR *MOUSE-FONT-CHAR*)
			   (READ-FUNCTION-NAME-OLD-MOUSE-X-OFFSET *MOUSE-X-OFFSET*)
			   (READ-FUNCTION-NAME-OLD-MOUSE-Y-OFFSET *MOUSE-Y-OFFSET*))
  "Read a function name using mini buffer or mouse.
PROMPT is a string that goes in the mode line.
DEFAULT is a function spec to return if the user types just Return.
MUST-BE-DEFINED can be T (allow only defined functions), NIL (allow anything)
 or AARRAY-OK (allow anything either defined as a function
 or known as a section by the editor).
STRINGP can be T, NIL, ALWAYS-READ or MULTIPLE-OK.
 T means if user types text, just return a string; don't try to intern it.
 ALWAYS-READ means intern the user's string afresh now;
  don't use the symbol or list recorded in the completion aarray.
 MULTIPLE-OK means it is ok to return more than one possible function
  the user could have meant, if they differ only in their package.

The first value is a list of function specs (only one, unless STRINGP is MULTIPLE-OK).
 If STRINGP is T, this is NIL.
The second value is the string the user typed, sans package prefix.
The third value is T if the user typed a package prefix."
  (DECLARE (VALUES COMPLETIONS STRING EXPLICIT-PACKAGE-P))
  (DECLARE (SPECIAL READ-FUNCTION-NAME-MUST-BE-DEFINED
		    READ-FUNCTION-NAME-OLD-GLOBAL-MOUSE-CHAR-BLINKER-HANDLER
		    READ-FUNCTION-NAME-OLD-GLOBAL-MOUSE-CHAR-BLINKER-DOCUMENTATION-STRING
		    READ-FUNCTION-NAME-OLD-MOUSE-FONT-CHAR
		    READ-FUNCTION-NAME-OLD-MOUSE-X-OFFSET
		    READ-FUNCTION-NAME-OLD-MOUSE-Y-OFFSET))
  (SYS:WITH-SUGGESTIONS-MENUS-FOR ZWEI:READ-FUNCTION-NAME
    (WHEN (EQ MUST-BE-DEFINED T)
      (SETQ STRINGP 'ALWAYS-READ))
    (SETQ PROMPT-WITHOUT-DEFAULT PROMPT
	  PROMPT (FORMAT NIL "~A~:[:~; (Default: ~S)~]" PROMPT DEFAULT DEFAULT))
    (LET ((NAME (LET ((*POST-COMMAND-HOOK* (APPEND *POST-COMMAND-HOOK* '(READ-FUNCTION-NAME-COMMAND-HOOK)))
		      (*MINI-BUFFER-VALUE-HISTORY* *DEFINITION-NAME-HISTORY*))
		  (LET ((*BATCH-UNDO-SAVE* T))
		    (DELETE-INTERVAL (WINDOW-INTERVAL *MINI-BUFFER-WINDOW*)))
		  (UNWIND-PROTECT
		      (PROGN
			(READ-FUNCTION-NAME-COMMAND-HOOK NIL)
			(COMPLETING-READ-FROM-MINI-BUFFER PROMPT *ZMACS-COMPLETION-AARRAY*
							  (OR (NEQ STRINGP 'ALWAYS-READ)
							      'ALWAYS-STRING)
							  NIL HELP)) ;;gsl 3-14-85
		    (READ-FUNCTION-NAME-COMMAND-HOOK T))))
	  SYM
	  ERROR-P)
      (COND ((EQUAL NAME "")
	     (UNLESS DEFAULT (BARF))
	     ;;; Next two lines by JPR.  Overrides the saved command with
	     ;;; the defaulted arg.
	     (setq *mini-buffer-command* nil)
	     (RECORD-MINI-BUFFER-VALUE t (format nil "~S" default))
	     (SETQ SYM DEFAULT
		   NAME (IF (SYMBOLP NAME)
			    (STRING NAME)
			    (WITH-OUTPUT-TO-STRING (*STANDARD-OUTPUT*) (PRINC DEFAULT)))))
	    ((CONSP NAME)
	     (SETQ SYM (CDR NAME)
		   NAME (CAR NAME))
	     (WHEN (AND (CONSP SYM)
			(NEQ STRINGP 'MULTIPLE-OK))
	       (SETQ SYM (CAR SYM))))
	    ((EQ STRINGP T)			;If returning a string, don't intern it
	     (SETQ SYM NAME))
	    (T
	     ;; If the string that was specified started with a package prefix,
	     ;; return a flag saying so.
	     ;; SYMBOL-FROM-STRING will flush the prefix from NAME.
	     (LET ((NON-LETTER-INDEX (STRING-SEARCH-NOT-SET " ABCDEFGHIJKLMNOPQRSTUVWXYZ-" NAME)))
	       (COND ((AND NON-LETTER-INDEX
			   (plusp non-letter-index) ;; may 05/08/89 Allow keywords with no package prefix
			   (CHAR= (AREF NAME NON-LETTER-INDEX) #\:))
		      (SETQ EXPLICIT-PACKAGE-P T))))
	     (MULTIPLE-VALUE-SETQ (SYM NAME ERROR-P)
	       (SYMBOL-FROM-STRING NAME NIL T))
	     (COND ((AND (CONSP SYM)
			 (EQ STRINGP 'MULTIPLE-OK))
		    (SETQ SYM (CONS SYM NIL))))
	     (COND (ERROR-P
		    (BARF "Read error")))))
      (COND ((AND (EQ MUST-BE-DEFINED T)
		  (NOT (OR (FDEFINEDP SYM)
			   (AND (SYMBOLP SYM)
				(SI:MEMQ-ALTERNATED 'ARGLIST (SYMBOL-PLIST SYM))))))
	     (COND ((NOT (DOLIST (SPEC (PACKAGE-LOOKALIKE-SYMBOLS SYM))
			   (COND ((FQUERY '(:SELECT T)
					  ;; Always print prefix
					  ;; Don't leave PACKAGE in keyword during query.
					  (LET ((*PACKAGE* PKG-KEYWORD-PACKAGE))
					    (FORMAT NIL "Do you mean ~S? " SPEC)))
				  (RETURN (SETQ SYM SPEC))))))
		    (BARF "~S is not defined" SYM)))))
      (PUSH-ON-HISTORY SYM *DEFINITION-NAME-HISTORY*)
      (PROMPT-LINE "~A ~S" PROMPT-WITHOUT-DEFAULT
		   (IF (AND (EQ STRINGP 'MULTIPLE-OK)
			    (NOT (ATOM SYM)))
		       (FIRST SYM)
		       SYM))
      (VALUES SYM NAME EXPLICIT-PACKAGE-P))))

;-------------------------------------------------------------------------------

;;; TI code.
;;; Patched by JPR so that when you turn off a sticky minor mode explicitly
;;; it stays off.

(DEFUN zwei:TURN-ON-MODE (MODE-SYMBOL)
  "Turn on mode MODE-SYMBOL.  Mode symbol is, for example, LISP-MODE.
Does nothing if the mode is already on."
  (SYS:WITH-SUGGESTIONS-MENUS-FOR ZWEI:TURN-ON-MODE
    (COND ((NULL (ASSOC MODE-SYMBOL *MODE-LIST* :TEST #'EQ))
	   (COND ((GET MODE-SYMBOL 'MAJOR-MODE-P)
		  (SETQ *MAJOR-MODE* MODE-SYMBOL)))
	   (PUSH (LIST MODE-SYMBOL (EVALUATE-FORMING-UNDO-LIST (GET MODE-SYMBOL 'MODE)))
		 *MODE-LIST*)))
    (COND ((NULL (MEMBER MODE-SYMBOL *MODE-NAME-LIST* :TEST #'EQ))
	   (PUSH MODE-SYMBOL *MODE-NAME-LIST*)))
    (COND ((GET MODE-SYMBOL 'MAJOR-MODE-P)
	   ;;; Fix by JPR.  Only turn on a mode if the user has not specifically
	   ;;; turned it off.
	   (let ((saved (send *interval* :Send-If-Handles :Saved-Mode-List)))
	     (if (remove (assoc mode-symbol saved) saved)
		 (DOLIST (MINOR *INITIAL-MINOR-MODES*)
		   (IF (ATOM MINOR)
		       (and (assoc minor saved) (TURN-ON-MODE MINOR))
		       (IF (MEMBER MODE-SYMBOL (CAR MINOR) :TEST #'EQ)
			   (DOLIST (MINOR (CDR MINOR))
			     (and (assoc minor saved) (TURN-ON-MODE MINOR))))))
		 (DOLIST (MINOR *INITIAL-MINOR-MODES*)
		   (IF (ATOM MINOR)
		       (TURN-ON-MODE MINOR)
		       (IF (MEMBER MODE-SYMBOL (CAR MINOR) :TEST #'EQ)
			   (DOLIST (MINOR (CDR MINOR))
			     (TURN-ON-MODE MINOR)))))))))
    (LET ((HOOK (GET MODE-SYMBOL 'MODE-HOOK-SYMBOL)))
      (AND HOOK
	   (BOUNDP HOOK)
	   (FUNCALL (SYMBOL-VALUE HOOK))))
    (SORT *MODE-NAME-LIST* #'(LAMBDA (X Y) (< (GET X 'MODE-LINE-POSITION)
					      (GET Y 'MODE-LINE-POSITION))))))

;-------------------------------------------------------------------------------

;;; Improve M-, so that it works on things other than just functions.
;;; Also improves the way it prints out the file names so that they
;;; light up.
(DEFCOM zwei:COM-FIND-SOURCE-FILE-NAME "Find the source file pathname for the function at point." (KM)
  (LET ((DEFINITION-TYPE NIL)
	(OBJECT (READ-FUNCTION-NAME "Find Pathname For Object"
						  (RELEVANT-FUNCTION-NAME (POINT))
						  'AARRAY-OK 'MULTIPLE-OK))
	(paths nil))
    (letf ((#'ed #'(lambda (x)
		     (setq paths
			   (append paths (SOURCE-FILE-NAMES x
							    T
							    DEFINITION-TYPE)))))
	   (#'tv:edit-a-path #'(lambda (x) (print x) (push x paths)))
	  )
	  (tv:edit object))
    (loop for path in paths
	  do (fs:maybe-add-item-type)
	     (send *standard-output* :item 'fs:edit-this-file path "~A"
		   (send path :String-For-Printing)
		   )
	     (terpri *standard-output*)))
  DIS-NONE)

;-------------------------------------------------------------------------------

;;; Originally defined in Sys: Zmacs; Font.Lisp#>
;;; Added optional nobarf argument. (from Jamie Zawinski).
;;;

(DEFUN SET-BUFFER-FONTS (BUFFER &OPTIONAL (FONTS (SEND BUFFER :GET-ATTRIBUTE :FONTS)) (nobarf t))
  "Set the fonts of BUFFER according to its attribute list.
We assume that the buffer's attribute list has been read in and stored.
If NOBARF is non-NIL, then a warning will be printed when we encounter an unknown font."
  (COND ((AND FONTS (SYMBOLP FONTS))
	 (SETQ FONTS (INTERN (SYMBOL-NAME FONTS) "FONTS"))
	 (SETQ FONTS (AND (BOUNDP FONTS) (LIST (CONS (SYMBOL-NAME FONTS) (SYMBOL-VALUE FONTS))))))
	(T
	 (DO ((FL FONTS (CDR FL))
	      (L NIL)
	      (F))
	     ((NULL FL)
	      (SETQ FONTS (NREVERSE L)))
	   (SETQ F (INTERN (SYMBOL-NAME (CAR FL)) "FONTS"))
	   (COND ((NOT (BOUNDP F))
		  ;; jwz: added condition trapping.
		  (condition-call-if nobarf (c)
		      (send (send (send *window* :superior) :superior) :parse-font-descriptor f)
		    ((errorp c)
		     (warn "~A.  Using ~A instead."
			   (send c :report-string)
			   (W:FONT-NAME (W:SCREEN-DEFAULT-FONT W:DEFAULT-SCREEN))
			   )))
		  (OR (BOUNDP F)
		      ;;If font not loaded, substitute default to keep font numbers ok
		      (SETQ F (W:FONT-NAME (W:SCREEN-DEFAULT-FONT W:DEFAULT-SCREEN))))))
	   (PUSH (CONS (SYMBOL-NAME F) (SYMBOL-VALUE F)) L))))
  (SEND BUFFER :SET-SAVED-FONT-ALIST FONTS)
  FONTS)


;-------------------------------------------------------------------------------

(Provide 'Zmacs-Enhancements)

(install-Zmacs-commands)
