;;; -*- Mode:Common-Lisp; Package:IMAP; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------
;;; Fixed version of patch-6-37.  TI code.

#!C
; From file CONDITIONALS-MACROS.LISP#> KERNEL; sys:
#10R SYSTEM#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "SYSTEM"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* *COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: KERNEL; CONDITIONALS-MACROS.#"


(defmacro select-memq (test-list &body clauses)
  "Execute the first clause that matches some element of TEST-LIST.
The first element of each clause is a match value or a list of match values.
Each match value is compare with each element of TEST-LIST, using EQ.
When a match-value matches, the rest of that clause is executed
and the value of the last thing in the clause is the value of the SELECT-MEMQ.
T or :OTHERWISE as the first element of a clause matches any test object."
  (let (test-exp cond-exp)
    (setq test-exp
	  ;; If TEST-LIST is an eval-at-load-time,
	  ;; we will treat it as a random expression, which is right.
	  (cond ((or (atom test-list)
		     (and (member (car test-list) '(car cdr caar cadr cdar cddr) :test #'eq)
			  (atom (cadr test-list))))
		 test-list)
		(t '.case.item.)))
    (setq cond-exp
	  (cons 'cond
		(mapcar #'(lambda (clause)
			    (macro-type-check-warning 'select-memq (car clause))
			    (cond ((member (car clause) '(otherwise :otherwise t) :test #'eq)
				   (list* t nil (cdr clause)))
				  ((atom (car clause))
				   `((member ',(car clause) ,test-exp :test #'eq) nil . ,(cdr clause)))
				  (t
				   `((or . ,(mapcar #'(lambda (match-value)
							`(member ',match-value ,test-exp :test #'eq))
						    (car clause)))
				     nil . ,(or (cdr clause) '(T)) ; DAB 05-24-90
				     ))))
			clauses)))
    (dead-clauses-warning (cdr cond-exp) 'select-memq)
    (cond ((eq test-exp test-list) cond-exp)
	  (t
	   `(let ((.case.item. ,test-list))
	      ,cond-exp)))))
))

;-------------------------------------------------------------------------------

;;; We get PDL overflow if we don't do this during net:define-service compile.
(eval-when (compile) (eh:require-pdl-room 16000 6000))

(defun generic-imap-server ()
"A generic top level for IMAP server startup."
  (let ((net:*medium-stream-type-alist*
	  (cons `(:Eol-Sequencing-Ascii-Translating-Character-Stream
		  :Tcp-Stream (:Bidirectional :Eol-Sequencing-Ascii)
		  :Chaos-Stream
		  Eol-Sequencing-Chaos-Ascii-Translating-Character-Stream
		  ,@(rest (assoc :Ascii-Translating-Character-Stream
				 net:*medium-stream-type-alist*
				 :Test #'eq
			  )
		    )
		 )
		 net:*medium-stream-type-alist*
	  )
	)
       )
       (with-open-stream
	 (stream (net:listen-for-connection-on-medium
		   :Byte-Stream
		   "Generic-IMAP"
		   :Stream-Type
		     :Eol-Sequencing-Ascii-Translating-Character-Stream
		 )
	 )
	 (assert stream () "IMAP stream didn't open properly.")
	 (imap-server-top-level stream)
       )
  )
)


;;; Define the IMAP service.
(net:define-service :Imap (host) (host))

;;; Define logical contacts under CHAOS and IP.
(net:define-logical-contact-name "Generic-IMAP"
				 `((:Chaos "IMAP")
				   (:Tcp ,*IMAP.Port*)
				  )
)

;;; Patch in the server on this host.
(net:add-server-for-medium
  :Byte-Stream "Generic-IMAP"
  '(process-run-function "IMAP Server" 'Generic-Imap-Server)
)



;-------------------------------------------------------------------------------

(defflavor eol-sequencing-ip-ascii-translating-character-stream
	   ((eol-sequence '(13. 10.)))
	   (ip:ascii-translating-character-stream)
  :Settable-Instance-Variables
)

;;; Modified TI code from sys:ip;tcp-stream.lisp
ip:
(DEFMETHOD (imap:eol-sequencing-ip-ascii-translating-character-stream :send-output-buffer)
	   (buffer character-index &aux segment j)
  "RFC854 pp. 11-12 contains the Netascii definition.  On output, CR -> CR NUL  and #\newline -> CR LF."
  ;; find segment to receive translated data
  (WHEN (AND (SETF segment (SEND connection :send-q-end))
	     (OR (= (segment-index segment) (segment-size segment)) (tcp-header-push-p segment)))
    (SEND self :flush-buffer))
  (WHEN (NOT (SETF segment (SEND connection :send-q-end)))
    (SETF segment (ALLOCATE-RESOURCE 'tcp-segment (SEND connection :maximum-send-size)))
    (SETF (tcp-header-ack-p segment) t)
    (WITHOUT-INTERRUPTS
      (IF (SEND connection :send-q-end)
	  (SETF (segment-link (SEND connection :send-q-end)) segment)
	  (SETF (SEND connection :send-q) segment))
      (SETF (SEND connection :send-q-end) segment)))
  (SETF j (segment-index segment))
  ;; translate entire buffer
  (DO ((i 0 (1+ i)))
      (nil)
    (WHEN (OR (= i character-index) (= j (segment-size segment)))
      ;; complete current segment
      (SETF (segment-index segment) j)
      (COND
	((OR send-urgent-mode-p (AND urgent-output-index (< i urgent-output-index)))
	 (SETF (tcp-header-urgent-p segment) t)
	 (SETF (tcp-header-urgent-pointer segment)
	       (- j (* 4. (tcp-header-data-offset segment)))))
	(urgent-output-index (SETF (tcp-header-urgent-p segment) t)
			     (SETF (tcp-header-urgent-pointer segment)
				   (- (- j (* 4 (tcp-header-data-offset segment))) (- i urgent-output-index)))
			     (SETF urgent-output-index ())))
      (WHEN (= i character-index)
	(RETURN))
      ;; allocate new segment
      (SETF segment (ALLOCATE-RESOURCE 'tcp-segment (SEND connection :maximum-send-size)))
      (SETF (tcp-header-ack-p segment) t)
      (WITHOUT-INTERRUPTS
	(IF (SEND connection :send-q-end)
	    (SETF (segment-link (SEND connection :send-q-end)) segment)
	    (SETF (SEND connection :send-q) segment))
	(SETF (SEND connection :send-q-end) segment))
      (SETF j (segment-index segment)))
    ;; translate one character
    (CASE (AREF buffer i)
      (13
       ;; insure room for <cr>/<nul>
       (COND ((= (1+ j) (segment-size segment))
	      (DECF (segment-size segment))
	      (DECF i)
	      (DECF j))
	     (t (SETF (AREF segment j) 13)
		(SETF (AREF segment (INCF j)) 0))))
      (#.(CHAR-CODE #\Backspace) (SETF (AREF segment j) 8))
      (#.(CHAR-CODE #\Tab) (SETF (AREF segment j) 9))
      (#.(CHAR-CODE #\Linefeed) (SETF (AREF segment j) 10))
      (#.(CHAR-CODE #\Page) (SETF (AREF segment j) 12))
      (#.(CHAR-CODE #\Newline)
	 ;;; Changed here by JPR so as to allow arbitrary char sequences to
	 ;;; represent #\Newline
	 ;; insure room for <cr>/<lf>.
	 ;;; It seems to do this by denying room for the coming char sequence
	 ;;; and making it be handled by the next segment.
	 (COND ((>= (+ j (length imap:eol-sequence) -1) (segment-size segment))
		(DECF (segment-size segment) (- (length imap:eol-sequence) 1))
		(DECF i)
		(DECF j))
	       (t (DECF j)
		  (loop for char in imap:eol-sequence do
			(INCF j)
			(SETF (AREF segment j) char)))))
      (#.(CHAR-CODE #\Rubout) (SETF (AREF segment j) 127))
      (otherwise (SETF (AREF segment j) (AREF buffer i))))
    (INCF j))
  (WHEN (buffer-push-p buffer)
    (SETF (tcp-header-push-p (SEND connection :send-q-end)) t))
  (DEALLOCATE-RESOURCE 'tcp-stream-ascii-buffer buffer)
  (SEND connection :external-drive-connection))

;;; This is a gray area.  Does set.eol deal with input as well???
;;;; Modified TI code from sys:ip;tcp-stream.lisp
;ip:
;(DEFMETHOD (imap:eol-sequencing-ip-ascii-translating-character-stream :around :next-input-buffer) (cont mt args ignore)
;  "RFC854 pp. 11-12 contains the Netascii definition.  On input, CR NUL -> CR and CR LF -> #\newline.
;CR should *not* appear in any other context, i.e. not followed by either NUL or LF."
;  (MULTIPLE-VALUE-BIND (buffer start end) (AROUND-METHOD-CONTINUE cont mt args)
;    (WHEN buffer
;      (DO ((i start (1+ i))
;	   (j start (1+ j))
;	   (old-last-char-cr-p (PROG1
;				 last-char-cr-p
;				 (SETF last-char-cr-p ())))
;	   look)
;	  ((EQL i end)
;	   (SETF end j))
;	(SETF (AREF buffer j)
;	      (CASE (AREF buffer i)
;		(0 (IF (AND (EQL i start) old-last-char-cr-p)
;		       13
;		       0))
;		(8 (CHAR-CODE #\Backspace))
;		(9 (CHAR-CODE #\Tab))
;		(10 (IF (AND (EQL i start) old-last-char-cr-p)
;			(CHAR-CODE #\Newline)
;			(CHAR-CODE #\Linefeed)))
;		(12 (CHAR-CODE #\Page))
;		(13 (COND ((EQL (SETF look (1+ i)) end)
;			   (SETF last-char-cr-p t)
;			   (DECF j)
;			   ;; dummy value to insert in array.  Translation not completed until next buffer.
;			   (CHAR-CODE #\Newline))
;			  (t (SETF i look)
;			     (IF (EQL (AREF buffer look) 10)
;				 (CHAR-CODE #\Newline)
;				 13))))
;		(127 (CHAR-CODE #\Rubout))
;		(t (AREF buffer i))))))
;    (VALUES buffer start end)))


(defflavor eol-sequencing-chaos-ascii-translating-character-stream
	   ((eol-sequence '(13. 10.)))
	   (chaos:ascii-translating-character-stream)
  :Settable-Instance-Variables
)


(defwhopper (eol-sequencing-chaos-ascii-translating-character-stream :tyo) (ch)
  (if (eql (char-code ch) #.(char-int #\newline))
      (loop for char in eol-sequence do (send self :tyo char))
      (continue-whopper ch)
  )
)


;sys:
;(DEFWRAPPER (ASCII-TRANSLATING-INPUT-STREAM-MIXIN :TYI) (IGNORE . BODY)
;   `(PROGN
;      .DAEMON-CALLER-ARGS.
;      (TYI-FROM-ASCII-STREAM
;       #'(LAMBDA (&REST .DAEMON-CALLER-ARGS. &AUX (.DAEMON-MAPPING-TABLE. SELF-MAPPING-TABLE))
;	   .DAEMON-MAPPING-TABLE.
;	   ,@BODY))))

;sys:
;(DEFUN-method TYI-FROM-ASCII-STREAM ASCII-TRANSLATING-INPUT-STREAM-MIXIN (ASCII-STREAM &AUX CH) ;03.05.87 DAB
;  (setf CRFF nil)
;    (CASE (SETQ CH (FUNCALL ASCII-STREAM :TYI))
;	  (8. #.(char-int #\BACKSPACE))
;	  (9. #.(char-int #\TAB))
;	  (10. #.(char-int #\LINEFEED))
;	  (12. #.(char-int #\PAGE))
;	  (13.
;	   (LET ((CH1 (FUNCALL ASCII-STREAM :TYI)))
;	     (OR (and (= CH1 10.) (setf CRFF t)) (FUNCALL ASCII-STREAM :UNTYI CH1)))
;	   #.(char-int #\NEWLINE))
;	  (127. #.(char-int #\RUBOUT))
;	  (nil nil)
;	  (T (char-int CH))))


;sys:
;(DEFWRAPPER (ASCII-TRANSLATING-INPUT-STREAM-MIXIN :UNTYI) (ch . BODY)
;   `(PROGN
;      .DAEMON-CALLER-ARGS.
;      (UNTYI-FROM-ASCII-STREAM
;	ch
;       #'(LAMBDA (&REST .DAEMON-CALLER-ARGS. &AUX (.DAEMON-MAPPING-TABLE. SELF-MAPPING-TABLE))
;	   .DAEMON-MAPPING-TABLE.
;	   ,@BODY))))

;sys:
;(DEFun-method UNTYI-FROM-ASCII-STREAM ASCII-TRANSLATING-INPUT-STREAM-MIXIN (CH ASCII-STREAM)
;    (setf ch (case (car ch)
;	     (136. (int-char 8.))   ;#\backspace 
;	     (137. (int-char 9.))   ;#\TAB
;	     (138. (int-char 10.))  ;#\LINEFEED
;	     (140. (int-char 12.))  ;#\PAGE
;	     (141. (prog1 (int-char 13.)
;			  (when CRFF (FUNCALL ASCII-STREAM :UNTYI (int-char 10.)))))  ;#\NEWLINE
;	     (135. (int-char 127.)) ;#\RUBOUT
;	     (t (int-char (car ch)))))
;  (FUNCALL ASCII-STREAM :UNTYI CH)
;  )
