;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defun shortest-of (names best)
"Given a list of names, returns the shortest of them.  Best is
the accumulating result.
"
  (if names
      (if (< (length (first names)) (length best))
	  (shortest-of (rest names) (first names))
	  (shortest-of (rest names) best)
      )
      best
  )
)

(defun find-shortest-name (command)
"Finds the shortest of the names of a command for that command."
  (if (typep command 'ucl:command)
      (let ((names (append
		     (get command :Short-forms)
		     (if yw:*always-complete-commands-to-shortest-available*
			 (mapcar
			   #'first
			   (remove-if-not #'(lambda (x) (get x :Typein-Name?))
					  (Send command :Names)
			   )
			 )
		         nil
		     )
		   )
	    )
	   )
	   (if names
	       (Shortest-Of (rest names) (first names))
	       (values nil t)
	   )
      )
      (values command t)
  )
)

(defun find-shortest-command (commands best best-name)
"Given a list of commands and accumulating best command and best name, returns
the command with the shortesdt name.
"
  (if (and commands (typep best 'ucl:command))
      (multiple-value-bind (new-name failed-p)
	  (find-shortest-name (first commands))
        (if failed-p
	    (Find-Shortest-Command (rest commands) best best-name)
	    (if (or (not best-name) (< (length new-name) (length best-name)))
		(Find-Shortest-Command
		  (rest commands) (first commands) new-name
		)
		(Find-Shortest-Command (rest commands) best best-name)
	    )
	)
      )
      (if best-name best nil)
  )
)

(defun find-matching-short-form (string commands)
"Given a list of commands and a string, returns a command with a matching
short form if there is one.
"
  (find-if #'(lambda (command)
	       (let ((names (get command :Short-forms)))
		    (find-if #'(lambda (name) (string-equal string name)) names)
	       )
	     )
	     commands
  )
)

(defun-rh substitute-command (old-string new-string)
"Substitutes New-string for old-string in the rubout handler buffer."
  (rh-delete-string (- (rhb-typein-pointer) (length old-string))
		    (rhb-typein-pointer) nil
  )
  (rh-insert-string new-string 0 nil nil nil)
  (if yw:*End-Of-Line-Found-P*
      (rh-insert-string (string #\newline) 0 nil nil nil)
;      (rh-insert-string " " 0 nil nil nil);;; This screws up the typin pointer.
  )
  (setf (rhb-scan-pointer) 0)
  (throw 'rubout-handler t)
)

(defun tv:yw-rh-complete-word (arg1 &rest args)
  "Find the word at the cursor position and do completion on it Returns NIL
 if no completion was done"
  (if (keywordp arg1)
      (apply 'tv:yw-rh-complete-word-internal *standard-input* arg1 args)
      (apply 'tv:yw-rh-complete-word-internal arg1 args)
  )
)

(defun-rh tv:yw-rh-complete-word-internal
 (stream &optional type
	 (complete-function (list yw:*completion-function* :auto-complete-p))
;	       (completion-function rh-completion-handler))
	 no-error-p
	 (dont-execute nil)
 )
  "Find the word at the cursor position and do completion on it Returns NIL
 if no completion was done"
  (ignore dont-execute)
  (if user:auto-complete?
      (multiple-value-bind (result completed-p completions-found string)
	  (if (typep stream 'yw:string-parsing-stream)
	      (Yw-Rh-Complete-Word-1-from-string
		stream type complete-function no-error-p
	      )
	      (tv:yw-rh-complete-word-1 type complete-function no-error-p)
	  )
	(let ((commands
		(loop for completion in completions-found
		      for command =
			  (if (send self :Typein-Modes)
			      (send (symbol-value
				      (first (send self :Typein-Modes))
				    )
				    :Get-Command-For-String completion
			      )
			  )
		      when command collect command
		)
	      )
	     )
	     (if completed-p
		 (values result completed-p completions-found string)
		 (let ((short (Find-Matching-Short-Form string commands)))
		      (if short
			  (if (typep stream 'yw:string-parsing-stream)
			      (tv:string-substitute-command
				string (send short :Name)
				(send stream :String) stream
			      )
			      (Substitute-Command string (send short :Name))
			  )
			  (if
			yw:*always-complete-commands-to-shortest-available*
			      (let ((selected
				      (if commands
					  (Find-Shortest-Command
					    (rest commands) (first commands)
					    (Find-Shortest-Name
					      (first commands)
					    )
					  )
					  nil
				      )
				    )
				   )
				   (if selected
				       (values (send selected :Name) t
					       (list (send selected :Name))
					       string
				       )
				       (values result completed-p
					       completions-found string
				       )
				   )
			      )
			      (values result completed-p
				      completions-found string
			      )
			  )
		      )
		 )
	     )
	)
      )
      (if (typep stream 'yw:string-parsing-stream)
	  (Yw-Rh-Complete-Word-1-from-string
	    stream type complete-function no-error-p
	  )
	  (tv:yw-rh-complete-word-1 type complete-function no-error-p)
      )
  )
)


(defun-rh back-to-non-whitespace (from)
"Given a pointer into the rubout-handler-buffer, backup up that pointer
until it finds a non-whitespace char i.e. the end of the last word.
"
  (if (< from 0)
      -1
      (if (or (equal from (rhb-fill-pointer))
	      (equal from (rhb-scan-pointer))
	      (yw:whitespace-p (aref rubout-handler-buffer from))
	  )
	  (back-to-non-whitespace (- from 1))
	  (values from (yw:simple-char (aref rubout-handler-buffer from)))
      )
  )
)
 
(defun-rh Yw-Rh-Complete-Word-1
	  (type &optional
		(complete-function (completion-function rh-completion-handler))
		no-error-p
	  )
  "Find the word at the cursor position and do completion on it Returns NIL
 if no completion was done"
  (declare
    (values result completed-p completions string-which-caused-completion))
  (setf (completions rh-completion-handler) nil)
  (let* ((start-pos
	   (rh-word-start
	     (Back-To-Non-Whitespace
	       (min (rh-scan-pointer) (rh-typein-pointer)))))
	   ;;Find start & end of word
	 (end-pos (rh-word-end start-pos)))
    (if (or (null end-pos)
	    (>= start-pos end-pos)
	    (rh-chars-between
	      end-pos (min (rh-scan-pointer) (rh-typein-pointer))))
	(when (not no-error-p)
	  (send-or-beep self :rh-error "Nothing to complete on"))
        (let ((string (rh-substring-of-buffer start-pos (1- end-pos))))
	     (alter-rh-completion-handler rh-completion-handler
		completion-word string
		completion-syntax
		  (funcall (word-syntax-function rh-completion-handler)
			   start-pos)
		completion-type type)
	     (multiple-value-bind
	       (completed-word completions completed-p completion-message
		completion-continuations
	       )
		 (apply (car complete-function)
			(completion-word rh-completion-handler)
			type nil (cdr complete-function)
		 )
	       (alter-rh-completion-handler rh-completion-handler
		  completed-word completed-word
		  completions completions
		  completion-message completion-message
		  completion-continuations completion-continuations)
	       (send self :send-if-handles :refresh-help)
	       (if (not (consp completions))         ;If no completion
		   (when (not no-error-p)
		     (send-or-beep self :rh-error
				   (format nil "No completions for ~a"
				     (completion-word rh-completion-handler)
				   )
		     )
		     (values nil nil nil string)
		   )
		   ;Replace word if completed
		   (if (string= completed-word
				(rh-substring-of-buffer start-pos (1- end-pos)))
		       (values completed-word t completions string)
		       (progn (rh-delete-string
;			        start-pos (max (rh-typein-pointer) end-pos) nil
				start-pos end-pos nil
			      )
			      (rh-insert-string completed-word 0 nil t t)
			      (values 
				      string
;				      (when (and completed-p (not no-error-p))
;					(beep)
;					(rh-insert-string " " 0 nil t nil))
				      completed-p
				      completions string))))))))
)


;-------------------------------------------------------------------------------

;;; The following functions are invoked when we are doing command input
;;; from a string.

(defun back-to-non-whitespace-in-string (from string)
"Given a pointer into String (from), backs up that pointer until it
finds a non-whitespace char.
"
  (if (< from 0)
      -1
      (if (or (equal from (length string))
	      (yw:whitespace-p (aref string from))
	  )
	  (back-to-non-whitespace-in-string (- from 1) string)
	  (values from (yw:simple-char (aref string from)))
      )
  )
)


(defun string-find-quote (start string)
  "Return the position of the next double quote after START"
  (DO ((i start (1+ i))
       (MAX (1- (LENGTH string))))
      ((OR (> i MAX) (CHAR= (AREF string i) #\")) i)))


(defun string-alphabetic? (i string)
"Is true is the char ar index I in string is an alphabetic."
  (declare (special zwei:*word-syntax-table* zwei:word-alphabetic))
  (let ((c (aref string i)))
    (if (boundp 'zwei:*word-syntax-table*)
	(= (zwei:char-syntax c zwei:*word-syntax-table*) zwei:word-alphabetic)
      (alpha-char-p c))))


(defun string-search-forward-alphabetic (pos string)
"Searches forward from Pos in string for an alphachar."
  (do ((fill-pointer (length string))
       (i pos (1+ i)))
      ((= i fill-pointer) nil)
    (if (string-alphabetic? i string) (return i))))


(defun string-search-forward-non-alphabetic (pos string)
"Searches forward from Pos in string for a non alphachar."
  (do ((fill-pointer (length string))
       (i pos (1+ i)))
      ((= i fill-pointer) nil)
    (if (not (string-alphabetic? i string)) (return i))))


(defun string-search-backward-alphabetic (pos string)
"Searches backwards from Pos in string for an alphachar."
  (do ((i (1- pos) (1- i)))
      ((= i -1) nil)
    (if (string-alphabetic? i string) (return i))))


(defun string-search-backward-non-alphabetic (pos string)
"Searches backwards from Pos in string for a non alphachar."
  (do ((i (1- pos) (1- i)))
      ((= i -1) nil)
    (if (not (string-alphabetic? i string)) (return i))))


(defun string-quoted-string-p (start &optional end string)
  "Return the position after the quote if POS is within a quoted
   string otherwise return nil"
  (let ((max (1- (length string))))
    (when (or (null end) (> end max)) (setq end max)))
  (do ((i start (1+ i))
       (pos nil))
      ((or (> i end) (and pos (= i end)))
       pos)
    (if (char= (aref string i) #\")
	(setq pos
	      (if pos nil (1+ i))))))


(defun-rh string-search-forward-word (n pos string)
"Searches forwards by N words from index Pos in string."
  (do ((search-pos)
       (i 0 (1+ i)))
      ((= i n) pos)
    (cond ((string-alphabetic? pos string)
	   (setq pos (string-search-forward-non-alphabetic pos string)))
	  (t (setq search-pos (string-search-forward-alphabetic pos string))
	     (if (not search-pos) (return pos))
	     (setq pos
		   (string-search-forward-non-alphabetic search-pos string))))
    ;;If within a word and can't find whitespace, leave at right end.
    (if (not pos) (return (length string)))))


(defun string-word-start (pos next-p string)
"Return the starting position of a word in the string
If NEXT-P is non-nil, get the next word if pos is between words,
otherwise get the previous word"
  ;;Fool the rubout-handler search routines into using the ZWEI
  ;;atom syntax table instead of the word syntax table
  (declare (special zwei:*word-syntax-table* zwei:*atom-word-syntax-table*))
  (let-if (boundp ' zwei:*atom-word-syntax-table*)
          ((zwei:*word-syntax-table* zwei:*atom-word-syntax-table*))
    ;(declare (special zwei:*word-syntax-table* zwei:*atom-word-syntax-table*))
    ;;Prevent RH-ALPHABETIC from returning T at the end of the buffer
    (when (eq pos (length string)) (decf pos))
    ;;POS is inside a word - get that word
    (cond ((minusp pos) nil)
	  ((string-alphabetic? pos string)
	   (or (string-quoted-string-p 0 pos string)
	       (1+ (or (string-search-backward-non-alphabetic pos string) -1))))
	  ;;POS is between words - get the next word
	  (next-p
	   (when (setq pos (string-search-forward-alphabetic pos string))
	     (or (string-quoted-string-p 0 pos string) pos)))
	  ;;POS is between words - get the previous word
	  ((setq pos (string-search-backward-alphabetic pos string))
	   (or (string-quoted-string-p 0 (1- pos) string)
	       (1+ (or (string-search-backward-non-alphabetic pos string)
		       -1)))))))


(defun string-word-end (pos string)
"Return the ending position of a word in the string given the starting position"
  (declare (special zwei:*word-syntax-table* zwei:*atom-word-syntax-table*))
  (when pos
    (if (and (plusp pos) (char= (aref string (1- pos)) #\"))
	(string-find-quote pos string)
        (let-if (boundp ' zwei:*atom-word-syntax-table*)
		((zwei:*word-syntax-table* zwei:*atom-word-syntax-table*))
	  (string-search-forward-word 1 pos string)
	)
    )
  )
)

(defun string-word-syntax (word-start string)
  "Looks in the string and looks at the symbol that the cursor is on.
  (If the cursor is between symbol, take the previous symbol.)  Returns one
  of the following syntax symbols:
    :first-function - The word is the first object on the line and is preceeded
                      by a #\(
    :first-atom	- The word is the first object on the line and does not have
                  a #\( before it. 
    function	- The word is in the middle of the line and is preceeded by
                  a #\(
    atom	- The word is in the middle of the line and is not preceeded
                  by a #\(
    string - The word is within a double-quoted string. 
    :empty	- Nothing has been typed in yet"
  (if (not word-start)
      :empty
    (let ((line-start (do ((fill-pointer (fill-pointer string))
			   (i 0 (1+ i)))
			  ((= i fill-pointer) 0)
			(unless (char= (aref string i) #\Space)
			  (return i))))
	  (list-p (and (plusp word-start)
		       (char= (aref string (1- word-start)) #\())))
      (cond
	((and (plusp word-start) (char= (aref string (1- word-start)) #\"))
	 'string)
	((= word-start line-start)
	 (if list-p :first-function :first-atom))
	(t (if list-p  'function 'atom))))))

(defun string-chars-between (pos1 pos2 string)
  "Returns NIL if nothing but spaces occur in the string between pos1
  and pos2, exclusive."
  (loop for pos from (1+ pos1) to (1- pos2)
	unless (char= (aref string pos) #\Space)
	return t))


(defun string-substring-of-buffer (start-pos end-pos string)
"Returns the substring of the buffer string String from start-pos to end-pos."
  (let* ((string-length (1+ (- end-pos start-pos)))
	 (new-string (make-string string-length)))
    (copy-array-portion string start-pos (1+ end-pos)
			new-string 0 string-length)
    new-string))


(defun string-delete-string (begin end string stream)
"Deletes the text in string from begin to end.  Stream is passed in so that
we can reset the read pointer.
"
  (let ((new-length (- (length string) (- end begin))))
       (copy-array-portion string end (length string)
			   string begin (- (length string) (- end begin))
       )
       (setf (fill-pointer string) new-length)
       (send stream :Set-Char-Pointer
	     (- (send stream :Char-Pointer) (- end begin))
       )
       (values string (send stream :Char-Pointer))
  )
)


(defun string-insert-string (new-string string stream position)
"Inserts a new string into the buffer string String at Position.  Stream is
supplied so that we can reset the read pointer.
"
  (let ((new-length (+ (length string) (length new-string))))
       (setf (fill-pointer string) new-length)
       (copy-array-portion (subseq string position (length string)) 0
			   (- (length string) position)
			   string (+ position (length new-string)) new-length
       )
       (copy-array-portion new-string 0 (length new-string) 
			   string position (+ position (length new-string))
       )
       (send stream :Set-Char-Pointer (+ 1 position (length new-string)))
       (values (send stream :String) (send stream :Char-Pointer))
  )
)

(defun yw:insert-string-in-string (new-string stream &optional (backup-p nil))
"Inserts String into the rubout handler of Stream at the current typein point.
If backup-p is true then we back up by one char.
"
  (if backup-p
      (send stream :Set-Char-Pointer (- (send stream :Char-Pointer) backup-p))
      nil
  )
  (String-Insert-String new-string (send stream :String)
			stream (send stream :Char-Pointer)
  )
  new-string
)


(defun-rh Yw-Rh-Complete-Word-1-from-string
	  (stream type &optional
		(complete-function (completion-function rh-completion-handler))
		no-error-p
	  )
"Find the word at the cursor position and do completion on it Returns NIL
if no completion was done.  Operates when we are reading from a string.
"
  (declare
    (values result completed-p completions string-which-caused-completion))
  (setf (completions rh-completion-handler) nil)
  (let* ((start-pos
	   (string-word-start
	     (Back-To-Non-Whitespace-in-string
	       (- (send stream :Char-Pointer) 1)
	       (send stream :String)
	     )
	     nil
	     (send stream :String)
	   )
	 )
	 (end-pos (string-word-end start-pos (send stream :String))))
    (if (or (null end-pos)
	    (>= start-pos end-pos)
;	    (string-chars-between
;	      end-pos (- (send stream :Char-Pointer) 1)
;	      (send stream :String))
	)
	(beep)
        (let ((string
		(string-substring-of-buffer start-pos (1- end-pos)
					    (send stream :String))))
	     (alter-rh-completion-handler rh-completion-handler
		completion-word string
		completion-syntax
		  (String-Word-Syntax start-pos (send stream :String))
		completion-type type)
	     (multiple-value-bind
	       (completed-word completions completed-p completion-message
		completion-continuations
	       )
		 (apply (car complete-function)
			(completion-word rh-completion-handler)
			type nil (cdr complete-function)
		 )
	       (alter-rh-completion-handler rh-completion-handler
		  completed-word completed-word
		  completions completions
		  completion-message completion-message
		  completion-continuations completion-continuations)
	       (if (not (consp completions))         ;If no completion
		   (when (not no-error-p)
		     (send-or-beep self :rh-error
				   (format nil "No completions for ~a"
				     (completion-word rh-completion-handler)
				   )
		     )
		     (values nil nil nil string)
		   )
		   ;Replace word if completed
		   (if (string= completed-word
				(string-substring-of-buffer
				  start-pos (1- end-pos) (send stream :String)))
		       (values completed-word t completions string)
		       (progn (string-delete-string
				start-pos end-pos (send stream :String) stream
			      )
			      (string-insert-string
				completed-word (send stream :String) stream
				(- (send stream :Char-Pointer) 1)
			      )
			      (values 
				      string
				      completed-p
				      completions string))))))))
)


(defun string-substitute-command (old-string new-string string stream)
"Substitutes the command New-string for Old-String in the buffer string String.
Stream is provided so that we can reset the read pointer.
"
  (string-delete-string
    (- (send stream :Char-Pointer) 1 (length old-string))
    (- (send stream :Char-Pointer) 1)
    string stream
  )
  (String-Insert-String
    new-string string stream (- (send stream :Char-Pointer) 1)
  )
  (throw 'string-rubout-handler
	 (list 'string-rubout-handler (send stream :String))
  )
)


;-------------------------------------------------------------------------------

