;;; -*- Mode:Common-Lisp; Package:YW-ZWEI; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defun read-a-char (&rest args)
"Just like read-char."
  (let ((char (apply #'read-char args)))
; (princ char)
       char
  )
)


(defflavor copying-stream
	   (istream ostream)
	   (si:input-stream)
  :Initable-Instance-Variables
  (:Documentation "A flavor of stream that copies from istream to ostream
whenever a char is tyied."
  )
)

(defmethod (copying-stream :tyi) (&rest args)
"Reads a char from istream and copies it to ostream."
  (let ((char (yw:safe-lexpr-send istream :tyi args)))
       (send ostream :tyo char)
       char
  )
)

(defmethod (copying-stream :untyi) (&rest args)
"Unreads a char from istream and unwrites it from ostream."
  (yw:safe-lexpr-send ostream :untyo args)
  (yw:safe-lexpr-send istream :untyi args)
)

(defflavor untyoing-stream
	   (stream
	    (last-char nil) ; the last char written.
	   )
	   (si:output-stream)
  :Initable-Instance-Variables
  (:Documentation "A flavor os stream that knows how to untyo chars.")
)

(defmethod (untyoing-stream :tyo) (char)
"Writes a char out to stream.  Remembers the last char written so that
we can untyi it.
"
  (if last-char
      (send stream :tyo last-char)
      nil
  )
  (setq last-char char)
)

(defmethod (untyoing-stream :flush) ()
"Flushes self's remembered last char."
  (send self :tyo nil)
)

(defmethod (untyoing-stream :untyo) (char)
"Untyos a char."
  (if (equal char last-char)
      (setq last-char nil)
      (yw:yw-error "~S is not the last char, which was ~S." char last-char)
  )
)

(defun copy-read-form (char istream ground-stream ostream)
"Copies a form from ground-stream to ostream.  Ground stream is used to
specify the carriage control that we have to copy to get to this point.
Finally reads a form from istream.
"
  (unread-char char istream)
  (copy-carriage-control ground-stream ostream)
  (let ((new-ostream (make-instance 'untyoing-stream :stream ostream)))
       (let ((new-istream (make-instance 'copying-stream
					 :istream ground-stream
					 :ostream new-ostream
			  )
	     )
	    )
	    (read new-istream)
	    (send new-ostream :flush)
       )
  )
  (read istream)
)

(defun copy-carriage-control (istream ostream)
"Copies any carriage control chars from istream to ostream."
  (let ((read-p nil))
       (loop for char = (read-a-char istream nil :eof)
	     until (or (equal char :eof) (not (yw:whitespace-p char)))
	     do (princ char ostream)
	        (setq read-p t)
	     finally (if (not (equal char :eof)) (unread-char char istream) nil)
       )
       read-p
  )
)

(defun copy-until-carriage-control (istream ostream)
"Copies chars from istream onto ostream until we hit a carriage control char."
  (loop for char = (read-a-char istream nil :eof)
	until (or (equal char :eof)
		  (yw:whitespace-p char)
	      )
	do (princ char ostream)
	finally (if (not (equal char :eof)) (unread-char char istream) nil)
  )
)

(defun space-to (pos ostream)
"Tabs ostream out with Pos spaces."
  (loop for i from 1 to pos do (princ " " ostream))
)

(defun output-comment (istream ostream pos width)
"Outputs a comment which we copy from istream to ostream.  Pos is the
horizontal position to place the comment.  Width is the width od the output
stream at which we want to throw a newline.
"
  (let ((real-pos (if (equal 1 (send istream :read-cursorpos :character))
		      (progn (terpri ostream) 0)
		      pos
		  )
	)
       )
       (princ ";" ostream)
       (loop for i = (read-a-char istream nil :eof)
	     until (or (equal i :eof) (equal i #\newline))
	     do (cond ((and (>= (send ostream :read-cursorpos :character)
				(- width 5)
			    )
			    (yw:whitespace-p i)
		       )
		       (terpri ostream)
		       (space-to real-pos ostream)
		       (princ ";" ostream)
		       (princ i ostream)
		      )
		      ((equal (+ 1 (send ostream :read-cursorpos :character))
			      width
		       )
		       (princ "-" ostream)
		       (terpri ostream)
		       (space-to real-pos ostream)
		       (princ "; " ostream)
		       (princ i ostream)
		      )
		      (t (princ i ostream))
		)
	     finally (if (equal i #\newline)
			 (progn (space-to real-pos ostream)
				(terpri ostream)
			        nil
			 )
			 nil
		     )
       )
  )
)

(defun make-a-string (&optional (size 200))
"Makes a fat string of size Size."
  (make-array size :element-type 'sys:fat-char :fill-pointer t :adjustable t)
)

(defflavor corsorposing-stream
	   ((stream nil) ; the stream we represent
	    (cursorpos 0) ; our current cursorpos.
	    (blank-line t) ; true when we are on a blank line.
	    (buffer (make-a-string)) ; a buffer to use for the current line.
	   )
	   (si:output-stream)
  :Initable-Instance-Variables
  (:Documentation "A flavor of stream that knows how to compute its cursorpos.")
)

(defmethod (corsorposing-stream :after :init) (ignore)
"Sets the buffer's fill pointer."
  (setf (fill-pointer buffer) 0)
)

(defmethod (corsorposing-stream :flush) ()
"Flushes the buffer to the output stream."
  (setq cursorpos 0)
  (if (not blank-line)
      (loop for i from 1 to (fill-pointer buffer)
	    do (princ (aref buffer (- i 1)) stream)
      )
      nil
  )
; (dbg); (describe stream)
  (setf (fill-pointer buffer) 0)
  (setq blank-line t)
)

(defmethod (corsorposing-stream :tyo) (char)
"Outpus a char on a cursorposing stream.  Chars are remembered until we get to
an EOL and then we flush.
"
  (let ((char (if (numberp char)
		  (int-char char)
		  char
	      )
	)
       )
       (if (equal #\newline char)
	   (send self :flush)
	   (progn (setq cursorpos (+ 1 cursorpos))
		  (if (not (yw:whitespace-p char))
		      (setq blank-line nil)
		      nil
		  )
	   )
       )
       (vector-push-extend char buffer 100)
  )
)

(defmethod (corsorposing-stream :close) (&rest args)
"Closes self.  Makes sure that we flush the buffer."
  (send self :flush)
  (yw:safe-lexpr-send stream :close args)
)

(defmethod (corsorposing-stream :read-cursorpos) (type)
"Returns the index into the current buffer as appropriate."
  (if (equal type :character)
      cursorpos
      (* cursorpos (tv:font-char-width (send *window* :current-font)))
  )
)

(defun copy-other-char (char istream ground-stream ostream)
  "Processes the copying from Istream to Ostream using Ground stream as the
primary source of characters unless we become misaligned and then we go
back to using istream.
"
  (copy-carriage-control ground-stream ostream)
  (let ((ground-char (read-a-char ground-stream nil :eof)))
    (if (and (characterp ground-char)
	     (string-equal (int-char (char-code char))
			   (int-char (char-code ground-char))
			   )
	     )
	(princ char ostream)
						; misaligned, so try to reallign.
	;;RDA: Could be a QUOTE
	(if (and (characterp ground-char)
		 (string-equal (int-char (char-code ground-char)) #\')
		 (string-equal #\( (int-char (char-code char))))
	    (progn
	      (princ ground-char ostream)
	      (copy-QUOTE istream 'si:null-stream)
	      (copy-carriage-control istream 'si:null-stream)
	      )
	    (if (string-equal #\) (int-char (char-code char)))
		;;RDA Assume this is a leftover paren from a QUOTE
		(when (characterp ground-char)
		  (unread-char ground-char ground-stream))
		(progn
		  (princ char ostream)
		  (if (copy-carriage-control istream 'si:null-stream)
		      (princ #\space ostream)
		      nil
		      )
		  (copy-until-carriage-control istream ostream)
		  (copy-carriage-control ground-stream ostream)
		  )
		)
	    )
	)
    )
  )

;;;RDA
(defun copy-QUOTE (istream ostream)
  (unless (loop for char being the array-elements of "quote" as ichar =
      (read-a-char istream nil :eof) always
      (prog1
	(string-equal char (int-char (char-code ichar)))
	(princ ichar ostream)))
    (print "Expected QUOTE but didn't see it.")))

(defun copy-using-ground-definition (istream ground-stream ostream width)
"Copies characters from istream to ostream using an output stream width of
Width.  Ground stream contains ground out definitions of the forms in istream.
We copy from istream to ostream and refer to ground stream to provide the
carriage control we want.
"
  (loop for char = (read-a-char istream nil :eof)
        until (equal char :eof)
        do (case (int-char (char-code char))
             (#\; (let ((pos (send ostream :read-cursorpos :character)))
                       (output-comment istream ostream pos width)
                  )
             )
             (#\" (copy-read-form char istream ground-stream ostream))
             ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
              (copy-read-form char istream ground-stream ostream)
             )
             (#\# (copy-carriage-control ground-stream ostream)
                  (princ char ostream)
                  (let ((char (read-a-char istream nil :eof)))
                       (if (equal #\\ (char-int (char-code char)))
                           (progn (princ char ostream)
                                  (princ (read-a-char istream nil :eof))
                                  (read-a-char ground-stream)
                                  (read-a-char ground-stream)
                                  (read-a-char ground-stream)
                           )
                           (progn (princ char ostream)
                                  (read-a-char ground-stream)
                                  (read-a-char ground-stream)
                           )
                       )
                  )
             )
             (#\space)
             (#\tab)
             (#\newline)
             (otherwise (copy-other-char char istream ground-stream ostream))
           )
  )
  (send ostream :flush)
)

(defun copy-string-to (string stream)
"Copies String to the stream Stream."
  (loop for i from 0 to (- (length (the string string)) 1)
        do (princ (if (numberp (aref string i))
                      (int-char (aref string i))
                      (aref string i)
                  )
                  stream
           )
  )
)

(defun replace-text-at (bp1 bp2 new-text left-margin)
"Replaces the text between bp1 and bp2 with the text specified by New-text.
Left margin is the number of spaces to tab over.
"
  (move-bp (point) bp1)
  (move-bp (mark)  bp2)
  (letf (((window-mark-p *window*) t))
        (com-kill-region)
  )
  (let ((new-bp1 (copy-bp (point))))
       (insert-string new-text left-margin)
       (values new-bp1 (point))
  )
)

(defun stringise (fat-string)
"Turns a fat string into a thin string."
  (let ((string (make-string (length fat-string))))
       (loop for i from 0 to (- (length fat-string) 1)
             do (setf (aref string i) (aref fat-string i))
       )
       string
  )
)

(defun backquoted-p (a b)
"Is true if it thinks that a and b are really backquoted forms.  It deduces
this on the basis of them beginning with * type macro functions.
"
  (and (symbolp (first a)) (symbolp (first b))
       (equal (find-package "SYS") (symbol-package (first a)))
       (equal (find-package "SYS") (symbol-package (first b)))
       (or (string-equal (string-append (symbol-name (first a)) "*")
                         (symbol-name (first b))
           )
           (string-equal (string-append (symbol-name (first b)) "*")
                         (symbol-name (first a))
           )
           (and (macro-function (first a))
                (macro-function (first b))
                (Test-Equal (macroexpand a) (macroexpand b))
           )
       )
  )
)

(defun stop-here ()
  (cerror "Go ahead" "Stop here")
)

(defun test-equal (a b)
"Is true if forms a and b are the same thing.  If they aren't then it stops at
the point at which it found the mismatch.
"
  (if (consp a)
      (if (consp b)
          (or (and (test-equal (first a) (first b))
                   (test-equal (rest a) (rest b))
              )
              (Backquoted-P a b)
              (stop-here)
          )
          (stop-here)
      )
      (if (equalp a b)
          t
          nil
      )
  )
)

(defun grind-form-2 (bp1 bp2 outstream stream2 string left-margin width forms)
"Grinds a form between bp1 and bp2 into outstream.  Stream2 is the input stream
for reading this form.  String is the string denoting the form to be ground.
LEft margin is the left margin to tab the new stuff over to.  Width is the max
width of the output stream and forms is the list of forms that were read in
from the region.
"
  (let ((result (tv:with-output-to-fat-string (ostream)
                  (with-input-from-string (ground-stream string)
                    (copy-using-ground-definition
                      stream2 ground-stream
                      (make-instance 'corsorposing-stream :stream ostream)
                      width
                    )
                  )
                )
        )
       )
; (print (stringise result))
       (let ((test (with-input-from-string (in result) (read-forms in))))
            (if (or (equalp test forms)
                    (Test-Equal test forms)
                    (stop-here)
                    (tv:mouse-confirm
                      "The grinding didn't seem to work.  Go ahead?"
                    )
                )
                (if (and bp1 bp2)
                    (replace-text-at bp1 bp2 result left-margin)
                    (copy-string-to result outstream)
                )
                (barf)
            )
       )
  )
)


(defun set-up-by-untabifying (bp1 bp2)
"Prepares a region for grinding by untabifying it."
  (move-bp (point) bp1)
  (com-beginning-of-line)
  (move-bp (mark) bp2)
  (com-untabify)
  (move-bp (point) bp1)
  (loop while (or (equal "" (bp-line (point)))
                  (yw:whitespace-p (aref (bp-line (point)) (bp-index (point))))
              )
        do (com-forward)
  )
)

(defun read-forms (stream)
"Reads all of the forms from Stream into a list."
  (let ((form (read stream nil :.eof.)))
       (if (equal form :.eof.)
           nil
           (cons form (read-forms stream))
       )
  )
)

(defun grind-form (bp1 bp2 outstream &optional (width nil))
"Grinds a form found between BP1 and BP2 into the output stream outstream.
Width is the max width for the output stream.
"
 (with-function-trapped (barf)
  (let ((bp2* (copy-bp bp2)))
;    (print bp1)
;    (print bp2)
       (set-up-by-untabifying bp1 bp2)
;    (print bp1)
;    (print bp2)
       (let ((stream1 (interval-stream bp1 bp2* t))
             (stream2 (interval-stream bp1 bp2* t :read-char))
            )
            (unwind-protect
;;RDA Does this really want to be here?
;              (catch-error
                (let ((forms (read-forms stream1))
                      (real-width (floor (/ width (tv:font-char-width
                                                    (send *window*
                                                          :current-font
                                                    )
                                                  )
                                         )
                                  )
                      )
                     )
                     (let ((string (with-output-to-string (*standard-output*)
                                     (loop for form in forms do
                                           (grind-top-level
                                             form (- real-width (bp-index bp1))
                                           )
                                     )
                                   )
                           )
                           (left-margin (bp-index bp1))
                          )
; (princ string)
;                         (zwei:with-undo-save ("Regrind region." bp1 bp2* t)
                            (grind-form-2 bp1 bp2* outstream stream2 string
                                          left-margin (- real-width left-margin)
                                          forms
                            )
;                         )
                     )
                )
;               nil
;              )
              (close stream1)
              (close stream2)  ;RDA: changed "stream1" to "stream2"
            )
       )
  )
 )
)


(defcom com-grind-region "Grinds the selected region or section." ()
  (let ((bp1 nil)
        (bp2 nil)
        (success nil)
        (defun-name nil)
       )
       (if (window-mark-p *window*)
           (progn (setq bp1 (mark) bp2 (point))
                  (or (bp-< bp1 bp2) (psetq bp1 bp2 bp2 bp1))
                  (if (bp-= (forward-over *whitespace-chars* (mark))
                            (forward-over *whitespace-chars* (point))
                      )
                      (setq *mark-stays* ())
                      (setq defun-name "Region")
                  )
           )
       )
       (cond (defun-name)
             ((setq bp1 (defun-interval (beg-line (point)) 1 () ()))
              (setq bp2 (interval-last-bp bp1) bp1 (interval-first-bp bp1))
              (setq si:*force-defvar-init* t)
             )
             (t (barf "Unbalanced parentheses"))
       )
       (format *query-io* "~&Grinding ~A"
               (or defun-name
                   (if (typep (bp-node bp1) 'zmacs-buffer)
                       "Region"
                       (section-node-name (bp-node bp1))
                   )
               )
       )
       (unwind-protect
           (progn (grind-form bp1 bp2 *standard-output* *fill-column*)
                  (setq success t)
           )
         (if success
             (format *query-io* " - ground")
             (format *query-io* " - aborted")
         )
       )
  )
  dis-all
)


(set-comtab *standard-comtab* nil (make-command-alist '(com-grind-region)))

;-------------------------------------------------------------------------------
