;;; -*- Mode:Common-LISP; Package: TV; Base:10.; Fonts:TVFONT,HL10B,TR10I; Patch-File: t -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;;; TI code.
;;; Changed to make these named defstructs, not just arrays.
(DEFSTRUCT (SCROLL-ENTRY :Named
			 (:CONC-NAME NIL)  (:ALTERANT ALTER-SCROLL-ENTRY)
			 (:PREDICATE NIL) (:COPIER NIL))
  SCROLL-ENTRY-FUNCTION				;Function to call to hack entry
  SCROLL-ENTRY-RECOMPUTE-FUNCTION		;Function called to recompute item
						; (sometimes unused)
  SCROLL-ENTRY-ARGS				;Args to above, also included is data
  SCROLL-ENTRY-LINES				;Number of lines entry spanned last time
  SCROLL-ENTRY-FINAL-X				;Final X position of cursor after this item
  SCROLL-ENTRY-FINAL-PRINTING-X			;Final X position after item was printed
						; (may be different from final-x if fixed
						; width item)
  SCROLL-ENTRY-WIDTH				;Width of entry, or last width if 
						; variable width
  SCROLL-ENTRY-VARIABLE-WIDTH-P			;T if entry is variable width, else NIL
  SCROLL-ENTRY-DATA				;Data to be printed
  SCROLL-ENTRY-PRINTED-FORM			;The data stored in its printed form in
						; case the data isn't a string, and if
						; the data is variable width -- this
						; makes outputting a bit more efficient
						; (Note: this is only used when the item
						; is variable width)
  SCROLL-ENTRY-PRINT-FORMAT			;Specification of how to print data
						; List of (prin1-or-princ base)
  SCROLL-ENTRY-MOUSE-INFO			;Mouse data if item is mouse sensitive
  )

(defun (:Property scroll-entry named-structure-invoke)
       (message-name thing &Rest arguments)
"A message handler form scroll entries."
  (ecase message-name
    (:Print-Self
     (1if* (1equal* '(nil) (scroll-entry-data thing))
	 (format (first arguments) "#<Scroll-Entry {Not Computed}>")
	 (format (first arguments) "#<Scroll-Entry ~>"
		 (list (scroll-entry-data thing) t (scroll-entry-data thing))
	 )
     )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

(DEFUN SCROLL-MAKE-ENTRY (&REST ELTS &AUX ENTRY)
  "2Create an entry and initialize its contents from ELTS.*"
  (SETQ ENTRY (MAKE-SCROLL-ENTRY))
  ;1; We go through this contortion to implement the FILLARRAY function.*
  ;1; What we are doing is filling the ENTRY to have an item from ELTS in*
  ;1; each element.*
  (LOOP FOR INDEX FROM 0 BELOW (LENGTH ENTRY)
	FOR ELEMENT IN ELTS
	DO (SETF (AREF ENTRY INDEX) ELEMENT)
	FINALLY (WHEN (< INDEX (LENGTH ENTRY))
		  ;1; There weren't enough items in ELTS.  Extend ENTRY*
		  ;1; out to contain the last item in ELTS for the*
		  ;1; remaining elements.*
		  (LOOP FOR NEW-INDEX FROM INDEX BELOW (LENGTH ENTRY)
			DO (SETF (AREF ENTRY NEW-INDEX) ELEMENT))))
  ENTRY)

(defadvise scroll-redisplay-display-item
	      (:bind-up-compute-motion-round-down) ()
;;; Force the rounding down in sheet compute motion.
  (1let* ((compute-motion-round-down t))
       :Do-It
  )
)

;;; TI code.
(DEFUN SCROLL-REDISPLAY-DISPLAY-ITEM
       (ITEM CURRENT-COUNT
	&OPTIONAL DONT-SET-BOTTOM-ITEM &AUX CURRENT-LINE CURRENT-ITEM-LINE
	FIRST-LINE FORCE-UPDATE OLD-LINE ENTRY-NEEDS-UPDATING END-OF-ITEM)
  "2Display item ITEM with item number CURRENT-COUNT, if it belongs
on the screen.  CURSOR-Y should be set up to the place to display the
next item that should be displayed (this item, if this one should be
displayed).  The idea is that you call this for successive items until
the cursor gets to the end of the screen.
BOTTOM-ITEM is set to CURRENT-COUNT unless
DONT-SET-BOTTOM-ITEM is non-NIL.*"
  (DECLARE (:SELF-FLAVOR BASIC-SCROLL-WINDOW))
  (COND
    ((>= CURRENT-COUNT TARGET-TOP-ITEM)
     ;1; We wanna be on the screen*
     (SETQ FIRST-LINE (IF SCROLL-NEW-X
			  (TRUNCATE SCROLL-NEW-Y LINE-HEIGHT)
			  (SCROLL-LINE)))
     (OR DONT-SET-BOTTOM-ITEM
	 (SETQ BOTTOM-ITEM CURRENT-COUNT))
     (COND
       ((AND
	  (SETQ OLD-LINE
		(DO ((I FIRST-LINE (1+ I)))
		    ((>= I SCREEN-LINES)
		     NIL)
		  (AND (EQ ITEM (AREF SCREEN-IMAGE I 0))
		       (ZEROP (AREF SCREEN-IMAGE I 1))
		       ;1; If first line of this item on screen anywhere,*
		       ;1; then can move it to current line.*
		       (RETURN I))))
	  (NOT (= OLD-LINE FIRST-LINE)))
	(AND SCROLL-NEW-X
	     (SEND SELF :SET-CURSORPOS SCROLL-NEW-X SCROLL-NEW-Y))
	(SETQ SCROLL-NEW-X NIL
	      SCROLL-NEW-Y NIL)
	;1; On screen but not in same position, move it up*
	(SEND SELF :DELETE-LINE (- OLD-LINE FIRST-LINE))))
     ;1; Now redisplay the item.*
     (SETQ CURRENT-LINE FIRST-LINE
	   CURRENT-ITEM-LINE 0)
     (UNWIND-PROTECT
	 (PROGN
	   (DOTIMES (I (SCROLL-ITEM-SIZE ITEM))
	     (LET ((ENTRY (AREF ITEM I)) (WID) (CHANGED-P))
	       ;1; Loop over all elements of the item*
	       (SETQ ENTRY-NEEDS-UPDATING
		     (OR FORCE-UPDATE (NEQ ITEM (AREF SCREEN-IMAGE CURRENT-LINE 0))
			 (NOT (= (AREF SCREEN-IMAGE CURRENT-LINE 1) CURRENT-ITEM-LINE))))
	       (COND
		 ((NOT
		    (OR
		      (SETQ CHANGED-P
			    (FUNCALL (SCROLL-ENTRY-FUNCTION ENTRY) :CHANGED-P ENTRY
				     (SCROLL-ENTRY-FUNCTION ENTRY)))
		      ENTRY-NEEDS-UPDATING))
		  ;1; Entry didn't change, but take into account how many*
		  ;1; lines it takes up.*
		  (LET ((SEL (SCROLL-ENTRY-LINES ENTRY)))
		    (IF (AND TRUNCATION (> SEL 0))
			;1; Spans more than one line, and truncating -- punt*
			(SETQ END-OF-ITEM T)
			(PROGN
			  (SETQ CURRENT-ITEM-LINE (+ SEL CURRENT-ITEM-LINE)
				CURRENT-LINE (+ SEL CURRENT-LINE))
			  (SETQ SCROLL-NEW-X
				(- (SCROLL-ENTRY-FINAL-X ENTRY) (SHEET-INSIDE-LEFT))
				SCROLL-NEW-Y
				(+ (OR SCROLL-NEW-Y (- CURSOR-Y (SHEET-INSIDE-TOP)))
				   (* LINE-HEIGHT SEL)))
			  (AND (>= (TRUNCATE SCROLL-NEW-Y LINE-HEIGHT) SCREEN-LINES)
			       (THROW 'END-OF-PAGE
				      T))))))
		 ;1; Set cursor to correct place, and continue with COND*
		 ((PROG1
		    ()
		    (AND SCROLL-NEW-X
			 (SEND SELF :SET-CURSORPOS SCROLL-NEW-X SCROLL-NEW-Y))
		    (SETQ SCROLL-NEW-X NIL
			  SCROLL-NEW-Y NIL)))
		 ;1; Entry needs updating, decide whether variable width or not.*
		 ((AND CHANGED-P
		       (SCROLL-ENTRY-VARIABLE-WIDTH-P ENTRY)
		       (NOT
			 (= (SCROLL-ENTRY-WIDTH ENTRY)
			    (SETQ WID
				  (FUNCALL (SCROLL-ENTRY-FUNCTION ENTRY) :WIDTH ENTRY
					   (SCROLL-ENTRY-FUNCTION ENTRY))))))
		  ;1; Going to kill line, move it down if it*
		  ;1; belongs below here anyway.*
		  (AND (AREF SCREEN-IMAGE CURRENT-LINE 0)
		       (NEQ (AREF SCREEN-IMAGE CURRENT-LINE 0) ITEM)
		       (SEND SELF :INSERT-LINE 1))
		  ;1; Variable width entry, and the width changed, force*
		  ;1; complete update of rest of item*
		  (SETQ FORCE-UPDATE T)
		  (SETF (SCROLL-ENTRY-WIDTH ENTRY) WID)
		  (SEND SELF :CLEAR-EOL)
		  (SCROLL-FLUSH-ITEM-FROM-SCREEN-IMAGE ITEM)
		  (LET ((*SCROLL-CURRENT-ITEM* ITEM)
			(*SCROLL-CURRENT-ITEM-LINE* CURRENT-ITEM-LINE))
                    (DECLARE (SPECIAL *SCROLL-CURRENT-ITEM* *SCROLL-CURRENT-ITEM-LINE*))
		    (SETQ END-OF-ITEM
			  (CATCH 'END-OF-LINE
			    (CATCH-ERROR
			      (PROGN
				(FUNCALL (SCROLL-ENTRY-FUNCTION ENTRY) :PRINT ENTRY)
				NIL)
			      T))))
		  (SETF (SCROLL-ENTRY-FINAL-X ENTRY) CURSOR-X)
		  (SETF (SCROLL-ENTRY-FINAL-PRINTING-X ENTRY) CURSOR-X)
		  (SETF (SCROLL-ENTRY-LINES ENTRY)
			(- (MIN (SCROLL-LINE) (1- SCREEN-LINES)) CURRENT-LINE))
		  (SETQ CURRENT-LINE (SCROLL-LINE)
			CURRENT-ITEM-LINE
			(+ (SCROLL-ENTRY-LINES ENTRY) CURRENT-ITEM-LINE)))
		 (T
		  ;1; Fixed width entry, or variable width entry*
		  ;1; and width hasn't changed.  Using the width,*
		  ;1; figure out the cursor motion and erase area.*
		  (MULTIPLE-VALUE-BIND (FINAL-X FINAL-Y FINAL-COUNT)
		      (SHEET-COMPUTE-MOTION SELF NIL NIL SCROLL-SPACES 0
					    (SCROLL-ENTRY-WIDTH ENTRY) NIL
					    (IF TRUNCATION
						(- (SHEET-INSIDE-RIGHT) CHAR-WIDTH)
						0)
					    (IF TRUNCATION
						(- CURSOR-Y (SHEET-INSIDE-TOP))
						NIL)
					    #o1000000)
		    (SETQ FINAL-X (+ FINAL-X (SHEET-INSIDE-LEFT))
			  FINAL-Y (+ FINAL-Y (SHEET-INSIDE-TOP))
			  END-OF-ITEM
			  (AND (NUMBERP FINAL-COUNT)
			       (NOT (= FINAL-COUNT (SCROLL-ENTRY-WIDTH ENTRY)))))
		    ;1; Don't let the FINAL-Y be past the last screen line.*
		    (AND (> FINAL-Y (+ TOP-MARGIN-SIZE (* (1- SCREEN-LINES) LINE-HEIGHT)))
			 (SETQ FINAL-X (SHEET-INSIDE-RIGHT)
			       FINAL-Y (+ TOP-MARGIN-SIZE (* (1- SCREEN-LINES) LINE-HEIGHT))))
		    ;;; These two forms put in by JPR.
		    (1setq* final-x
			   (1if* (1and* (si:mx-p)
				    truncation)
			       (1if* (1>* (- (SHEET-LINE-NO NIL FINAL-Y) CURRENT-LINE) 0)
				   (sheet-inside-width)
				   final-x)
			       final-x
			   )
		    )
		    (1setq* final-y
			  (1if* (1and* (si:mx-p)
				   truncation)
			      (sheet-cursor-y self)
			      final-y
			  )
		    )
		    (SETF (SCROLL-ENTRY-FINAL-X ENTRY) FINAL-X)
		    (SETF (SCROLL-ENTRY-LINES ENTRY)
			  (- (SHEET-LINE-NO NIL FINAL-Y) CURRENT-LINE))
		    ;1; Zero the area*
		    (PREPARE-SHEET (SELF)
		      (DO ((Y CURSOR-Y (+ Y LINE-HEIGHT))
			   (LINE 0 (1+ LINE))
			   (X CURSOR-X (SHEET-INSIDE-LEFT))
			   (LE)
			   (DELTA-ITEMS))
			  ((> Y FINAL-Y))
			(SETQ LE (AREF SCREEN-IMAGE (+ CURRENT-LINE LINE) 0))
			(COND
			  ((OR
			     (AND (EQ LE ITEM)
				  (= (AREF SCREEN-IMAGE (+ CURRENT-LINE LINE) 1)
				     (+ CURRENT-ITEM-LINE LINE)))
			     (NULL LE))
			   ;1; We know about this line so just clear*
			   ;1; the area.*
			   (%DRAW-RECTANGLE
			     (- (IF (= Y FINAL-Y)
				    FINAL-X
				    (SHEET-INSIDE-RIGHT))
				X)
			     LINE-HEIGHT X Y erase-aluf SELF))
			  ((EQ LE ITEM)
			   ;1; We own line, but it is wrong number.*
			   ;1; Clear the line and flush all knowledge.*
			   (%DRAW-RECTANGLE (- (SHEET-INSIDE-RIGHT) X) LINE-HEIGHT X
					    Y erase-aluf SELF)
			   (SCROLL-FLUSH-ITEM-FROM-SCREEN-IMAGE ITEM))
			  (T
			   ;1; Make room for remaining number of*
			   ;1; lines and return.*
			   (SETQ DELTA-ITEMS
				 (- (AREF SCREEN-IMAGE (+ CURRENT-LINE LINE) 2)
				    CURRENT-COUNT))
			   ;1; DELTA-ITEMS is a guess as to the*
			   ;1; number of items in between this*
			   ;1; and the line it collided with.*
			   ;1; Assuming one line per item, this*
			   ;1; is a good guess as to the number*
			   ;1; of additional lines to insert.*
			   (LET-GLOBALLY ((CURSOR-Y Y))
			     (SEND SELF
				   :INSERT-LINE
				   ;1; If we are past the item*
				   ;1; that's on this line, it*
				   ;1; can't possibly appear on the*
				   ;1; screen -- insert enough*
				   ;1; lines to make it go off the*
				   ;1; screen.*
				   (MAX 1
					(MIN
					  (+ (TRUNCATE (- FINAL-Y Y) LINE-HEIGHT)
					     (ABS DELTA-ITEMS))
					  (- SCREEN-LINES (SCROLL-LINE))))))
			   (RETURN T)))))
		    (LET ((*SCROLL-CURRENT-ITEM* ITEM)
			  (*SCROLL-CURRENT-ITEM-LINE* CURRENT-ITEM-LINE))
                      (DECLARE (SPECIAL *SCROLL-CURRENT-ITEM* *SCROLL-CURRENT-ITEM-LINE*))
		      (COND
			((CATCH 'END-OF-LINE
			   (PROGN
			     (FUNCALL (SCROLL-ENTRY-FUNCTION ENTRY) :PRINT ENTRY)
			     (SETQ CURRENT-ITEM-LINE
				   (+ (SCROLL-ENTRY-LINES ENTRY) CURRENT-ITEM-LINE)
				   CURRENT-LINE
				   (+ (SCROLL-ENTRY-LINES ENTRY) CURRENT-LINE))
			     (SETF (SCROLL-ENTRY-FINAL-PRINTING-X ENTRY) CURSOR-X)
			     (SETQ SCROLL-NEW-X (- FINAL-X (SHEET-INSIDE-LEFT))
				   SCROLL-NEW-Y (- FINAL-Y (SHEET-INSIDE-TOP)))
			     (SEND SELF :HANDLE-EXCEPTIONS)
			     NIL))
			 (SETF (SCROLL-ENTRY-FINAL-PRINTING-X ENTRY) CURSOR-X)
			 (SETQ END-OF-ITEM T))))))))
	     (AND END-OF-ITEM
		  (RETURN T)))
	   (SETQ SCROLL-NEW-X 0
		 SCROLL-NEW-Y
		 (+ (OR SCROLL-NEW-Y (- CURSOR-Y (SHEET-INSIDE-TOP))) LINE-HEIGHT))
	   (AND (>= (1+ CURRENT-LINE) SCREEN-LINES)
		(THROW 'END-OF-PAGE T)))
       (SETQ CURRENT-LINE (MIN CURRENT-LINE (1- SCREEN-LINES)))
       (DO ((L FIRST-LINE (1+ L)))
	   ((> L CURRENT-LINE))
	 (SETF (AREF SCREEN-IMAGE L 0) ITEM)
	 (SETF (AREF SCREEN-IMAGE L 1) (- L FIRST-LINE))
	 (SETF (AREF SCREEN-IMAGE L 2) CURRENT-COUNT))))))


;(DEFWRAPPER (AUTO-SCROLLING-MIXIN :NOTICE) ((EVENT &REST IGNORE) . BODY)
;  `(IF
;     (AND (NOT (EQl SCROLL-PIXEL-INCREMENT 0))
;	  (EQ EVENT :INPUT-WAIT)
;	  MORE-VPOS
;	  (SETQ MORE-VPOS (- (SHEET-INSIDE-HEIGHT SELF)
;			     (- (SHEET-INSIDE-BOTTOM) CURSOR-Y))))
;     T
;     . ,BODY))