;;; -*- Mode:Common-Lisp; Package:W; Base:10; Patch-file:T -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; Rich Acuff, Stanford KSL, Nov '87

;;; Code to make it easier to move and resize windows without going
;;; through menus.

;;;----------------------------------------------------------------------

;;; Interface to call window accelerators.
;;; :DISPATCH-ACCELERATED-OPERATION Usually called from :MOUSE-CLICK
;;; methods or some such.

(defflavor window-accelerator-mixin
	   ()
	   ()
  (:documentation :mixin
   "Provides a default definition for :DISPATCH-ACCELERATED-OPERATION
and :<mouse-char>-ACCELERATED-OPERATION, which, by default, do:

   #\Mouse-L-1   Drag-Move the window
   #\Mouse-L-2   Drag-Reshape the window
   #\Mouse-M-1   Standard-Reshape the window
   #\Mouse-M-2   Shrink the window if possible
   #\Mouse-R-1   Zoom the window size if possible, else System Menu
   #\Mouse-R-2   System Menu
"
   )
  )

(defmethod (window-accelerator-mixin :accelerator-who-line-doc) ()
  `(
    :mouse-L-1 "Drag Move"
    :mouse-L-2 "Drag Reshape"
    :mouse-M-1 "Reshape"
    :mouse-M-2 "Shrink"
    ,@(if (send self :operation-handled-p :zoom-size)
	  '(:mouse-R-1 "System Menu" :mouse-R-2 "Zoom Size")
	  '(:mouse-R "System Menu")
	  )
    )
  )

(defmethod (window-accelerator-mixin :dispatch-accelerated-operation)
	   (mx my button)
  "Call :<mouse-char>-ACCELERATED-OPERATION, where <mouse-char> is
   MOUSE-L-1, MOUSE-L-2, MOUSE-M-1, MOUSE-M-2, MOUSE-R-1, or MOUSE-R-2,
   passing MX and MY."
  (block nil
    (process-run-function '(:name "Window Accelerator" :priority 2) self
			  (case button
			    (#\Mouse-L-1 :mouse-l-1-accelerated-operation)
			    (#\Mouse-L-2 :mouse-l-2-accelerated-operation)
			    (#\Mouse-M-1 :mouse-M-1-accelerated-operation)
			    (#\Mouse-M-2 :mouse-M-2-accelerated-operation)
			    (#\Mouse-R-1 :mouse-R-1-accelerated-operation)
			    (#\Mouse-R-2 :mouse-R-2-accelerated-operation)
			    (otherwise
			     (cerror "Ignore the button"
				     "Unknown mouse button: ~S" button)
			     (return)
			     )
			    )
			  mx my
			  )
    )
  )

(defmethod (window-accelerator-mixin :mouse-l-1-accelerated-operation)
	   (mx my)
  "Drag-Move SELF"
  (drag-window-from self mx my)
  )

;;;Edited by Acuff                 28 Nov 87  16:35
(defmethod (window-accelerator-mixin :mouse-l-2-accelerated-operation)
	   (mx my)
  "Drag-Reshape SELF"
  (declare (ignore mx my))
  (drag-reshape self)
  )

(defmethod (window-accelerator-mixin :mouse-m-1-accelerated-operation)
	   (mx my)
  (declare (ignore mx my))
  (mouse-set-window-size self)
  )

(defmethod (window-accelerator-mixin :mouse-m-2-accelerated-operation)
	   (mx my)
  "Shrink SELF."
  (declare (ignore mx my))
  (shrink-window self)
  )

;;;Edited by Acuff                 28 Nov 87  16:35
(defmethod (window-accelerator-mixin :mouse-r-1-accelerated-operation)
	   (mx my)
  "Zoom-size SELF if we have a :ZOOM-SIZE method handler, else call
   System Menu"
  (declare (ignore mx my))
  (if (send self :operation-handled-p :zoom-size)
      (send self :zoom-size)
      (mouse-call-system-menu)
      )
  )

(defmethod (window-accelerator-mixin :mouse-r-2-accelerated-operation)
	   (mx my)
  "Call up a System Menu."
  (declare (ignore mx my))
  (mouse-call-system-menu)
  )

;;;----------------------------------------------------------------------

;;; Label version

(defflavor label-accelerator-mixin
	   ()
	   (window-accelerator-mixin)
  (:required-flavors tv:essential-label-mixin)
  (:documentation :mixin
   "Allow menu-less window operations by clicking on a window's label."
   )
  )

(defmethod (label-accelerator-mixin :around :mouse-buttons)
	   (continuation mapping-table orig-args bd x y)
  "Call :DISPATCH-ACCELERATED-OPERATION if the click was over SELF's
   label, else carry on."
  (if (send self :in-label? x y)
      ;; In region, so dispatch to acclerators
      (send self :dispatch-accelerated-operation
	    x y (mouse-character-button-encode bd))
      ;; Not in region so proceed
      (lexpr-funcall-with-mapping-table
	continuation mapping-table orig-args)
      )
  )

(defmethod (label-accelerator-mixin :around :who-line-documentation-string)
	   (continuation mapping-table orig-args)
  "Call :ACCELERATOR-WHO-LINE-DOC if mouse is in label, else carry on."
  (multiple-value-bind (xoff yoff)
      (sheet-calculate-offsets self mouse-sheet)
    (if (send self :in-label? (- mouse-x xoff) (- mouse-y yoff))
	;; In region, so dispatch to acclerators
	(send self :accelerator-who-line-doc)
	;; Not in region so proceed
	(lexpr-funcall-with-mapping-table
	  continuation mapping-table orig-args)
	)
    )
  )

(defmethod (tv:essential-label-mixin :in-label?) (x y)
  "T iff (X, Y) is a position inside of SELF's label.  X and Y are in
   SELF's coords."
  ;; Macros to make offsets positive; negative ones are relative to
  ;; bottom or right.
  (macrolet ((horiz-pos (pos)
			`(if (minusp ,pos) (+ ,pos (send self :width)) ,pos))
	     (vert-pos (pos)
		       `(if (minusp ,pos) (+ ,pos (send self :height)) ,pos))
	     )
    (let ((spec (send self :label)))
      (and spec
	   (<= (real-label-left self) x)
	   (>= (real-label-right self) x)
	   (<= (real-label-top self) y)
	   (>= (real-label-bottom self) y)
	   )
      )
    )
  )

;;;Edited by Acuff                 30 Nov 87  19:00
(defun real-label-left (window)
  "Returns the coord of left edge of the label of WINDOW."
  (let ((val (label-left (send window :label))))
    (if val
	;; Make sure it's positive
	(if (< val 0)
	    (+ val (send window :width))
	    val
	    )
	;; No label
	nil
	)
    )
  )

;;;Edited by Acuff                 30 Nov 87  19:00
(defun real-label-right (window)
  "Returns the coord of right edge of the label of WINDOW."
  (let ((val (label-right (send window :label))))
    (if val
	;; Make sure it's positive
	(if (<= val 0)
	    (+ val (send window :width))
	    val
	    )
	;; No label
	nil
	)
    )
  )

;;;Edited by Acuff                 30 Nov 87  19:00
(defun real-label-top (window)
  "Returns the coord of top edge of the label of WINDOW."
  (let ((val (label-top (send window :label))))
    (if val
	;; Make sure it's positive
	(if (< val 0)
	    (+ val (send window :height))
	    val
	    )
	;; No label
	nil
	)
    )
  )

;;;Edited by Acuff                 30 Nov 87  19:00
(defun real-label-bottom (window)
  "Returns the coord of bottom edge of the label of WINDOW."
  (let ((val (label-bottom (send window :label))))
    (if val
	;; Make sure it's positive
	(if (<= val 0)
	    (+ val (send window :height))
	    val
	    )
	;; No label
	nil
	)
    )
  )

;;;----------------------------------------------------------------------

;;; For when the label is in an inferior

;;;Edited by Acuff                 30 Nov 87  19:00
(defflavor label-accelerators-for-superior
	   ()
	   ()
  (:documentation :mixin
   "Mixin for a window Y used to get accelerators for a window X
    when Y is a direct inferior of X.  X should have
    WINDOW-ACCELERATOR-MIXIN in it."
   )
  )

(defmethod (label-accelerators-for-superior :around :mouse-buttons)
	   (continuation mapping-table orig-args bd x y)
  "Call :DISPATCH-ACCELERATED-OPERATION if the click was over SELF's
   supierior's label, else carry on."
  (let ((sup (send self :superior)))
    (if (and (send sup :operation-handled-p :dispatch-accelerated-operation)
	     (send self :in-label? x y))
	;; In region, so dispatch to acclerators
	(send sup :dispatch-accelerated-operation
	      x y (mouse-character-button-encode bd))
	;; Not in region so proceed
	(lexpr-funcall-with-mapping-table
	  continuation mapping-table orig-args)
	)
    )
  )

;;;Edited by Acuff                 30 Nov 87  19:00
(defmethod (label-accelerators-for-superior
	     :around :who-line-documentation-string)
	   (continuation mapping-table orig-args)
  "Call :ACCELERATOR-WHO-LINE-DOC if mouse is in label, else carry on."
  (let ((sup (send self :superior)))
    (multiple-value-bind (xoff yoff)
	(sheet-calculate-offsets self mouse-sheet)
      (if (send self :in-label? (- mouse-x xoff) (- mouse-y yoff))
	  ;; In region, so dispatch to acclerators
	  (send sup :accelerator-who-line-doc)
	  ;; Not in region so proceed
	  (lexpr-funcall-with-mapping-table
	    continuation mapping-table orig-args)
	  )
      )
    )
  )
;;;----------------------------------------------------------------------

;;; Zoom sizing

(defvar *default-zoom-edges* `(0 71
				 ,(send main-screen :width)
				 ,(send main-screen :height))
  "Edges used be default for the 'other' size of a window")

(defflavor zoom-sizing-mixin
	   ((other-edges *default-zoom-edges*))
	   ()
  (:documentation :mixin
   "Mixin to allow windows to have two separate sets of edges, and to
    'Zoom' between them.")
  :initable-instance-variables
  :settable-instance-variables
  :gettable-instance-variables
  )

;;;Edited by Acuff                 28 Nov 87  16:35
(defmethod (zoom-sizing-mixin :zoom-size) ()
  "Change between current edges and the other ones."
  (let ((e (multiple-value-list (send self :edges))))
    (apply self :set-edges other-edges)
    (setq other-edges e)
    )
  )

;;;Edited by Acuff                 4 Dec 87  18:45
(defun zoom-window (window)
  "Change (zoom) WINDOW to it's other size if possible."
  (send window :send-if-handles :zoom-size))

;;;Edited by Acuff                 4 Dec 87  18:45
(defun zoom-window-op (window m-x m-y)
  (declare (ignore m-x m-y))
  (zoom-window (find-right-sheet window))
  )

;;;----------------------------------------------------------------------

;;; Drag shaping

;;;Edited by Acuff                 28 Nov 87  16:35
(defun drag-reshape (window)
  "Rubber band WINDOW."
  (multiple-value-bind (left top right bottom)
      (send window :edges)
    (setf (values left top right bottom)
	  (drag-rectangle left top right bottom (send window :superior)))
    (when (numberp left)
      (send window :set-edges left top right bottom))
    )
  )

;;;Edited by Acuff                 28 Nov 87  16:35
;;;Edited by Acuff                 28 Nov 87  16:44
;;;Edited by Acuff                 3 Dec 87  10:14
(defun drag-rectangle (left top right bottom sheet
		       &optional (minimum-width 0) (minimum-height 0))
  "Use the mouse to get a new position for a rectangle who's edges are
   LEFT, TOP, RIGHT, and BOTTOM on SHEET."
  (with-mouse-sheet (sheet)
    (with-mouse-grabbed
      (setf  who-line-mouse-grabbed-documentation
	    "Release Left and Middle buttons to finish; Hold Right button to switch corners")
      (rubber-band left top right bottom
		   minimum-width minimum-height nil nil nil)
      )
    )
  )

;;;----------------------------------------------------------------------

;;; Redefs of system windows to use accelerators

;;;TI Code: Add zooming and accelerators
(DEFFLAVOR lisp-listener ()
	   (ucl:command-loop-mixin
	    notification-mixin
	    listener-mixin
	    ucl:command-and-lisp-typein-window
	    zoom-sizing-mixin label-accelerator-mixin	;RDA
	    window)
  (:default-init-plist
    ;;Use *terminal-io* for typein, instead of a pop-up window which is right out!
    :typein-handler ':HANDLE-TYPEIN-INPUT
    :basic-help '(basic-help-on-listener)
    :save-bits t
    :deexposed-typeout-action ':permit)
  (:documentation :combination "
A Lisp interpreter which povides documentation in the who-line to help the user type and complete on
lisp.  The Lisp Listener uses the Universal Command Loop for the lisp type-in features it provides
\(such as implicit message sending, who-line help, error catching on typos) plus rubout handler command
help.  The Lisp Listener is basically a UCL application which has no commands.  Compare it to the
RTMS, which is a lisp interpreter augmented with commands which provide a nice interface to the
RTMS."))

;;;Add zooming and accelerators to Zmacs by using the mode line window
;;;TI Code: 
Zwei:
(DEFFLAVOR ZMACS-FRAME
	   ((STANDARD-INPUT-FOR-PANES NIL))
	   (TV::LISP-HELP-MIXIN
	    w:zoom-sizing-mixin w:window-accelerator-mixin	;RDA
	    W:PROCESS-MIXIN
	    W:STREAM-MIXIN ZWEI-FRAME
;;Not patched in 93.  Instead, this mixin's methods are patched into ZMACS-FRAME.
	    ZWEI-MACRO-MIXIN)
  :SETTABLE-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST
    :MODE-LINE-WINDOW '(MODE-LINE-WINDOW :RECALCULATE-FUNCTION ZMACS-MODE-LINE-RECALCULATE-FUNCTION)
    :SAVE-BITS T))

zwei:
(defmethod (mode-line-window :who-line-documentation-string) ()
  "Call :ACCELERATOR-WHO-LINE-DOC."
  (when (typep (send self :superior) 'w:window-accelerator-mixin)
    (send (send self :superior) :accelerator-who-line-doc)
    )
  )

;;;TI Code:
zwei:
(defmethod (mode-line-window :mouse-buttons)
	   (bd x y)
  "Call :DISPATCH-ACCELERATED-OPERATION."
  (when (typep (send self :superior) 'w:window-accelerator-mixin)
    (send (send self :superior) :dispatch-accelerated-operation
	  x y (tv:mouse-character-button-encode bd))
    )
  )

;;;TI Code: Add zooming and accelerators
tv:
(DEFFLAVOR peek-frame
	   ((peek-pane nil)			;These variables can be used in the commands to
	    (typeout-pane nil)			;access the respective panes.
	    (mode-pane nil)
	    (cmds-pane nil))
	   (UCL:COMMAND-LOOP-MIXIN
	    TV:STREAM-MIXIN
	    w:zoom-sizing-mixin w:window-accelerator-mixin	;RDA
	    FRAME-DONT-SELECT-INFERIORS-WITH-MOUSE-MIXIN
	    BORDERED-CONSTRAINT-FRAME-WITH-SHARED-IO-BUFFER)
  :GETTABLE-INSTANCE-VARIABLES
  :SETTABLE-INSTANCE-VARIABLES
  :INITTABLE-INSTANCE-VARIABLES
  :SPECIAL-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST
    :active-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :all-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :menu-panes '((menu ucl-peek-menu)
		  (cmds ucl-peek-cmd-menu))
    :blip-alist '((:MENU :handle-menu-input)
		  (:DIRECT-COMMAND-ENTRY :handle-direct-command-input)
		  (:MOUSE-BUTTON :peek-handle-blip)
                  ;(SUPDUP        :peek-handle-blip)        ;!not supported
	          (TELNET        :peek-handle-blip)
	          (QSEND         :peek-handle-blip)
	          (EH            :peek-handle-blip)
	          (INSPECT       :peek-handle-blip)
	          (DESCRIBE      :peek-handle-blip)
		  (HOST:HOST-STATUS   :peek-handle-blip))       ;; LS 11/06/86
    :save-bits :delayed
    :typein-handler nil
    :basic-help '(peek-documentation-cmd)))

;;;TI Code: Add accelerators
tv:
(DEFFLAVOR BASIC-PEEK ((NEEDS-REDISPLAY NIL))
   (SCROLL-MOUSE-MIXIN
;    MARGIN-SCROLLING-WITH-FLASHY-SCROLLING-MIXIN ;x
;    FLASHY-SCROLLING-MIXIN ;x
    BORDERS-MIXIN
;    MARGIN-SCROLL-MIXIN ;x
;    MARGIN-REGION-MIXIN ;x
    scroll-bar-mixin
    w:label-accelerators-for-superior		;RDA
    SCROLL-WINDOW-WITH-TYPEOUT FULL-SCREEN-HACK-MIXIN)
  :SETTABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST ;:SAVE-BITS T
;                       :MARGIN-SCROLL-REGIONS `((:TOP ;x
;						  ,*NO-MORE-BACKWARD-STANDARD-MESSAGE* ;x
;						  ,*MORE-BACKWARD-STANDARD-MESSAGE* ;x
;						  FONTS:TR8B) ;x
;						(:BOTTOM ;x
;						  ,*NO-MORE-FORWARD-STANDARD-MESSAGE* ;x
;						  ,*MORE-FORWARD-STANDARD-MESSAGE* ;x
;						  FONTS:TR8B)) ;x
;		       :FLASHY-SCROLLING-REGION '((20 0.40s0 0.60s0) ;x
;						  (20 0.40s0 0.60s0)) ;x
;                       :scroll-bar-on-off t
     		       :LABEL "Peek"
		       :TRUNCATION NIL)
  (:DOCUMENTATION :SPECIAL-PURPOSE "The actual peek window.  This has the capability
to display in a PEEK display mode."))

;;;TI Code: Add zooming and accelerators
tv:
(DEFFLAVOR basic-inspect-frame                        ;fi
	   ((inspectors nil)
	    (typeout-window nil)
	    (menu nil)
	    (user nil)
	    (frame nil)
	    (history nil)
            (INSPECTOR-TYPEIN-P NIL)
            (*inspect-print-array*  *PRINT-ARRAY*)
            (*inspect-print-circle* *print-circle*)	
            (*inspect-print-radix*  *print-radix*)
            (*inspect-nopoint       *nopoint)
            (*inspect-print-base*   *print-base*)
            (*inspect-read-base*    *read-base*)
            (*inspect-print-level*  *print-level*)
            (*inspect-print-length* *print-length*)
	    (inspection-data-active? NIL)) ;;See documentation string below.
	   (ucl:basic-command-loop
 	    PROCESS-MIXIN
	    FULL-SCREEN-HACK-MIXIN
	    FRAME-DONT-SELECT-INFERIORS-WITH-MOUSE-MIXIN
	    BORDERED-CONSTRAINT-FRAME-WITH-SHARED-IO-BUFFER
	    CONSTRAINT-FRAME-FORWARDING-MIXIN
	    BORDERS-MIXIN
	    w:zoom-sizing-mixin w:label-accelerator-mixin	;RDA
	    essential-mouse			;RDA
	    LABEL-MIXIN BASIC-FRAME)
  (:DEFAULT-INIT-PLIST
    :active-command-tables '(inspector-menu-cmd-table inspector-other-cmd-table)
    :all-command-tables '(inspector-menu-cmd-table inspector-other-cmd-table)
    :menu-panes '((menu ucl-inspector-menu))
    :typein-handler :handle-typein-input
    :prompt "Inspect: "
    ;;This predicate tells UCL to print command execution results only
    ;;on typed Lisp forms during our Lisp Evaluation mode.  It also inspects results when not in that mode.
    :print-results? 'inspector-print-values?
    :basic-help '(documentation-cmd)
    :SAVE-BITS :DELAYED
    :PROCESS '(INSPECT-TOP-LEVEL
		:SPECIAL-PDL-SIZE 4000
		:REGULAR-PDL-SIZE 10000))
  :GETTABLE-INSTANCE-VARIABLES
  :SETTABLE-INSTANCE-VARIABLES
  :INITTABLE-INSTANCE-VARIABLES
  :SPECIAL-INSTANCE-VARIABLES
  (:INIT-KEYWORDS :NUMBER-OF-INSPECTORS)
  (:DOCUMENTATION :MIXIN
    "Basic flavor used to build the inspector and other inspector-like applications.
This flavor may also be used as a mixin to construct special-purpose inspection windows,
such as the Flavor Inspector (TV:FLAVOR-INSPECTOR).  Most any information can be presented using 
it, as long as the Inspector paradigm is appropriate for your application.  Use the code for flavor
TV:FLAVOR-INSPECTOR as a guide for constructing your own inspector.  Basic points to keep in mind
are the following:

1. The inspector command interface makes use of the Universal Command Loop.  Therefore, UCL instance
   variables control much of the command interaction.  Important initializations are:

   :ACTIVE-COMMAND-TABLES --the set of commands your inspector accepts.  Can be NIL.
   :ALL-COMMAND-TABLES    --probably EQ to :ACTIVE-COMMAND-TABLES.  The set of all command tables used in your inspector.
   :MENU-PANES            --an alist which should be ((TV:MENU <your symbol>)).  <your symbol> is a menu symbol used
                            to build your permanent command menu using UCL's BUILD-MENU function.  If for some reason
                            you have arranged your constraint frame to not include a permanent command menu, ommitt this option.
   :TYPEIN-MODES          --the list of symbols set to UCL:TYPEIN-MODE instances which control the processing of typed
                            expressions in the interaction window.   The default just interprets Lisp forms and typed
                            command names; in a special inspector, you might want to design special typein-modes which
                            allow the user to type an expression of some kind to inspect some data.  For instance, in
                            the Flavor Inspector, the user can type flavor names and method specs to inspect them.
 2.  As with the Flavor Inspector, you'll be defining flavors built on TV:INSPECTION-DATA for displaying your various types of data
     in the inspection panes.  Pattern them off of the flavor inspector's.  Use method :INSPECT-THING to create and explicitly inspect 
     a TV:INSPECTION-DATA instance.
 3.  To specify your own window panes and constraints, give your flavor a :BEFORE :INIT method
     similar to (:METHOD TV:FLAVOR-INSPECTOR :BEFORE :INIT).
  3.  If you don't want to handle typed expressions in your inspector,
initialize :TYPEIN-HANDLER to NIL.  
 4.  Your inspector should initialize :INSPECTION-DATA-ACTIVE? to T.  Since the regular inspector just inspects Lisp,
     it turns this off so that programmers are able to inspect instances of TV:INSPECTION-DATA in the normal manner.
     Any special inspectors such as the Flavor Inspector have to set this to T in order to activate the special inspection
     features of TV:INSPECTION-DATA instances.  If you have an inspector which mixes the function of Lisp inspection with
     special data inspection, you can flip this variable on and off when switching between Lisp inspection and
     TV:INSPECTION-DATA inspection.
Constructing your own inspector isn't automatic, but if you use the flavor inspector as a model, it'll be pretty easy."))

;;;TI Code: Add zooming and accelerators
eh:
(DEFFLAVOR DEBUGGER-FRAME
           (INSPECT-WINDOW		    ;Where the disassembled code goes
            INSPECT-HISTORY-WINDOW	    ;History for the inspector
            ARGS-WINDOW			    ;The arguments
            LOCALS-WINDOW		    ;The locals
            STACK-WINDOW		    ;Backtrace
            LISP-WINDOW			    ;read-eval-print loop window
            FRAME-ALIST			    ;Saved frame layout
            TV-CURRENT-WINDOW              ;window which called the w-d, for selection substitutes
            TV-OLD-SUBST                   ;the original selection substitute of the window which called the w-d
	    (*inspect-print-array*  *PRINT-ARRAY*)     ;!
            (*inspect-print-circle* *print-circle*)    ;!
            (*inspect-print-radix*  *print-radix*)     ;!
            (*inspect-nopoint       *nopoint)          ;!
            (*inspect-print-base*   *print-base*)      ;!
            (*inspect-read-base*    *read-base*)       ;!
            (*inspect-print-level*  *print-level*)     ;!
            (*inspect-print-length* *print-length*)    ;!
            (inspection-data-active? nil)
            (DOING-TYPEIN-P NIL)
            (WHO-LINE-DOC-STRING-OVERIDE NIL))
           (UCL::BASIC-COMMAND-LOOP TV:FRAME-DONT-SELECT-INFERIORS-WITH-MOUSE-MIXIN
                                    TV:BORDERED-CONSTRAINT-FRAME
				    w:zoom-sizing-mixin w:label-accelerator-mixin	;RDA
                                    TV:LABEL-MIXIN)
  (:GETTABLE-INSTANCE-VARIABLES LISP-WINDOW
                                         DOING-TYPEIN-P
                                         WHO-LINE-DOC-STRING-OVERIDE
                                         inspection-data-active? TV-CURRENT-WINDOW TV-OLD-SUBST
                                         *inspect-print-array* *inspect-print-circle* *inspect-print-radix*   ;! 
                                         *inspect-nopoint *inspect-print-base* *inspect-read-base*            ;!
                                         *inspect-print-level* *inspect-print-length*)                        ;!
           (:SETTABLE-INSTANCE-VARIABLES DOING-TYPEIN-P WHO-LINE-DOC-STRING-OVERIDE TV-CURRENT-WINDOW TV-OLD-SUBST
                                         inspection-data-active? TV-CURRENT-WINDOW TV-OLD-SUBST
                                         *inspect-print-array* *inspect-print-circle* *inspect-print-radix*   ;!
                                         *inspect-nopoint *inspect-print-base* *inspect-read-base*            ;!
                                         *inspect-print-level* *inspect-print-length*)                        ;!
           (:DEFAULT-INIT-PLIST :SAVE-BITS
                                T
                                :DEEXPOSED-TYPEOUT-ACTION
                                :PERMIT
                                :ACTIVE-COMMAND-TABLES
                                '(WINDOW-DEBUGGER-GENERAL-CMD-TABLE 
                                  WINDOW-DEBUGGER-STACK-CMD-TABLE 
                                  WINDOW-DEBUGGER-STEP-CMD-TABLE 
                                  WINDOW-DEBUGGER-RESUME-CMD-TABLE)
                                :ALL-COMMAND-TABLES
                                '(WINDOW-DEBUGGER-GENERAL-CMD-TABLE 
                                  WINDOW-DEBUGGER-STACK-CMD-TABLE 
                                  WINDOW-DEBUGGER-STEP-CMD-TABLE 
                                  WINDOW-DEBUGGER-RESUME-CMD-TABLE)
                                :MENU-PANES
                                '((COMMAND-MENU-WINDOW UCL-WINDOW-DEBUGGER-EXAMINE-MENU) 
                                  (RESUME-MENU-WINDOW UCL-WINDOW-DEBUGGER-RESUME-MENU))
                                :TYPEIN-HANDLER
                                :HANDLE-TYPEIN-INPUT
                                :BASIC-HELP
                                '(WINDOW-DEBUGGER-HELP-CMD))
           (:DOCUMENTATION :SPECIAL-PURPOSE "Controls layout of
window-based debugger panes"))

tv:
(defflavor basic-x-y-scrolling-window ()
	   (basic-x-y-scrolling-mixin
	    overview-mixin
	    mouse-sensitivity-for-instances-mixin
	    ver-auto-scrolling-mixin
	    basic-mouse-sensitive-items-compatibility-mixin
	    fancy-drag-scrolling-mixin 
	    borders-mixin
	    w:zoom-sizing-mixin w:label-accelerator-mixin	;RDA
	    label-mixin
	    graphics-mixin
	    stream-mixin
	    minimum-window
	    ))

(compile-flavor-methods lisp-listener zwei:zmacs-frame
			tv:peek-frame
			tv:inspect-frame
			tv:basic-x-y-scrolling-window
			)

;;;----------------------------------------------------------------------

#|

;;; Testing code

;;;Edited by Acuff                 30 Nov 87  19:00
(defflavor acc-test
	   ()
	   (tv:minimum-window borders-mixin top-label-mixin
	    zoom-sizing-mixin
	    label-accelerator-mixin)
  )

(setq w (make-instance 'acc-test :edges-from :mouse :label "Test"
   :expose-p t))
|#
