;;; -*- Mode:Common-Lisp; Package:TOOLS; Base:10; Patch-FIle:t -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	Rich Acuff
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1985-1989.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15
;;;	DARPA Grant N00039-86- C-0033

;;;  A package for loading useful tools.  Depends on "TOOLS:TOOLS;DEFSYSTEMS.LISP"

(export '(load-tools add-tool) 'ticl)
(export '(load-tools *known-tools* add-tool))

;;;----------------------------------------------------------------------

;;; Patch file attribute added by JPR on 4/03/89

;;; Written by JPR.
sys:
(define-make-system-special-variable sys:*really-nowarn* t)

;;; Written by JPR.
(defun (:property :really-nowarn si:make-system-keyword) (&rest ignore)
"Inhibits fdefine warnings completely when make-systeming."
  (setq si:inhibit-fdefine-warnings t
	tv:more-processing-global-enable nil
	si:*batch-mode-p* t
	si:*query-type* :noconfirm
	sys:*really-nowarn* t
  )
)


;;; TI Code.
sys:
(DEFUN sys:RECORD-FILE-DEFINITIONS (ACCESS-PATHNAME DEFINITIONS &OPTIONAL (WHOLE-FILE T)
				GENERIC-PATHNAME)
  "Update the list of functions defined in the file ACCESS-PATHNAME.
DEFINITIONS is a list of new definitions.  WHOLE-FILE-P says flush any old ones.
If any methods used to be defined in that file but aren't any longer,
offer to undefine them.  You can specify GENERIC-PATHNAME to save time,
or let it be computed from ACCESS-PATHNAME.

The elements of DEFINITIONS look like (OBJECT-DEFINED . DEFINITION-TYPE).
Usually DEFINITION-TYPE is DEFUN and OBJECT-DEFINED is a function spec."
  (let ((DEFAULT-CONS-AREA BACKGROUND-CONS-AREA))
    (UNLESS GENERIC-PATHNAME
      (SETQ GENERIC-PATHNAME
	    (IF (TYPEP ACCESS-PATHNAME 'INSTANCE)
		(FUNCALL ACCESS-PATHNAME :GENERIC-PATHNAME)
		(MAKE-INSTANCE 'PROPERTY-LIST-MIXIN))))		;Create a generic plist receiver
    (LET* ((ALIST-ELEM (ASSOC *PACKAGE*
			      (FUNCALL GENERIC-PATHNAME :GET :DEFINITIONS)
			      :TEST #'EQ))
	   (OLD-DEFINITIONS (CDR ALIST-ELEM))
	   OLD-FUN)
      (LOCALLY (DECLARE (SPECIAL UNANALYZED-FILES))
	       (IF (AND (VARIABLE-BOUNDP UNANALYZED-FILES)
			(NOT (MEMBER GENERIC-PATHNAME
				     UNANALYZED-FILES
				     :TEST #'EQ)))
		   (SETQ UNANALYZED-FILES (COPY-LIST (CONS GENERIC-PATHNAME UNANALYZED-FILES)))))
      (IF (NOT WHOLE-FILE)
	  (SETQ DEFINITIONS (NUNION OLD-DEFINITIONS DEFINITIONS :TEST #'EQUAL)) ;replace nunion-equal dkm 7/31/86
	  ;; Make the data structure compact for paging efficiency.
	  (SETQ DEFINITIONS (COPY-TREE DEFINITIONS)))
      (IF ALIST-ELEM
	  (RPLACD ALIST-ELEM DEFINITIONS)
	  (FUNCALL GENERIC-PATHNAME :PUSH-PROPERTY (CONS *PACKAGE* DEFINITIONS)
		   :DEFINITIONS))
      (IF (NOT WHOLE-FILE)
	  NIL
	  ;; If we are doing the whole file, offer to undefine any methods deleted from the file.
	  (PROGN (DO ((DEFS DEFINITIONS (CDR DEFS)))
		     ((NULL DEFS))
		   (SETF (CAR DEFS) (COPY-LIST (CAR DEFS))))
		 (OR (FUNCALL GENERIC-PATHNAME :GET :PATCH-FILE)
		     (DOLIST (OLD-DEF OLD-DEFINITIONS)
		       (AND (CONSP OLD-DEF)
			    (EQ (CDR OLD-DEF) 'DEFUN)
			    (SETQ OLD-FUN (CAR OLD-DEF))
			    (CONSP OLD-FUN)
			    (EQ (CAR OLD-FUN) :METHOD)
			    ;; Leave out combined methods, which may have been present
			    ;; due to COMPILE-FLAVOR-METHODS.  They are handled automatically.
			    (OR (= (LENGTH OLD-FUN) 3)
				(NOT (MEMBER (CADDR OLD-FUN)
					     '(:COMBINED FASLOAD-COMBINED)
					     :TEST #'EQ)))
			    (NOT (MEMBER OLD-DEF DEFINITIONS :TEST #'EQUAL))
			    (FDEFINEDP OLD-FUN)
			    ;; Detect automatic methods defined by a DEFFLAVOR that is still present.
			    (MULTIPLE-VALUE-BIND (NAME TYPE)
				(FUNCTION-PARENT OLD-FUN)
			      (NOT (MEMBER (CONS NAME TYPE) DEFINITIONS :TEST #'EQUAL)))
			    (LET* ((FILES (CDR (ASSOC 'DEFUN
						      (GET-ALL-SOURCE-FILE-NAMES OLD-FUN)
						      :TEST #'EQ)))
				   (FILES-1 FILES))
			      (DO () ((NOT (AND FILES-1 (FUNCALL (CAR FILES-1) :GET :PATCH-FILE))))
				(POP FILES-1))
			      (AND (EQ (CAR FILES-1) GENERIC-PATHNAME)
				   (PROGN
				     (IF (EQ FILES FILES-1)
					 (FORMAT *QUERY-IO*
						 "~&File ~A no longer contains a definition of ~S.~%"
						 ACCESS-PATHNAME OLD-FUN)
					 (FORMAT *QUERY-IO* "~&File ~A no longer contains a definition of ~S.
It was more recently redefined by patch file ~A, but no other non-patch file.~%"
						 ACCESS-PATHNAME OLD-FUN
						 (SEND (CAR FILES) :SOURCE-PATHNAME)))
				     ;;; Patched here by JPR so that we don't
				     ;;; get long timeouts when loading tools.
				     ;;; This is useful for when tools change
				     ;;; and we reload a tool which has had a
				     ;;; change of flavor name.
				     (PROG1 (if (and (boundp '*really-nowarn*)
						     *really-nowarn*
						)
						t
						(WITH-TIMEOUT ((* 60. 60.) (FORMAT *QUERY-IO* " ... Yes by timeout.") T)
						  (Y-OR-N-P "Undefine it? (60 sec timeout for Yes) ")))
						(TERPRI *QUERY-IO*)))))
			    (FUNDEFINE OLD-FUN)))))))))


;;; JPR.
(defvar *load-tools-make-system-keywords*
	'(:noconfirm :silent :really-nowarn #+TI :safe)
  "A list of make-system option keywords used by Load-Tools."
)

;;; Functions and variables of intererest to users

(defvar *known-tools* nil
  "List of lists, one for each tool system, of the form (NAME MOUSE-DOC
   FULL-DOC) where NAME is a symbol in the keyword package and MOUSE-DOC and
   FULL-DOC are both strings.")

(defun load-tools (&optional (tool-names :by-menu))
  "Function to take care of loading defined tools.  TOOL-NAMES is a
   list of tool names (system names), the keyword :ALL (to load all
   available tools), or the keyword :BY-MENU, which causes a menu of all
   known tools to be presented, allowing the user to select the desired
   tools.  If the menu item `Document Tools' is selected, then
   documentation is printed instead of loading the tools."
  (let ((tool-list (canonicalize-tool-list tool-names)))
    (if (member :document tool-list)
	(document-tool-list tool-list)
	(load-tool-list tool-list)
	)
    )
  (values)
  )

(defun add-tool (system-name mouse-doc long-doc)
  "Add the system called SYSTEM-NAME to the set of known tools.
   MOUSE-DOC will be displayed when this tool is used in a menu.
   LONG-DOC will be displayed when LOAD-TOOLS is used to print
   documentation."
  (setf system-name (intern (string system-name) 'keyword))
  (setf *known-tools*
	(cl:delete-if
	  #'(lambda (x)
	      (string-equal (string (first x)) (string system-name)))
	  *known-tools*))
  (push (list system-name mouse-doc long-doc) *known-tools*)
  (setf *known-tools* (sort *known-tools* #'string-lessp :key #'first))
  )

;;;----------------------------------------------------------------------

(defun do-nothing-to-keys (system keywords)
  (ignore system)
  keywords
)

(defvar *tool-keyword-modify-function*
	'do-nothing-to-keys
"We need this for the recompile process.  It would be easy to
use LETF here but it hasn't been loaded yet."
)

(defvar *after-load-tool-hook* 'identity
"Call this when a tool is loaded."
)

(defun load-tool-internal (tool)
  (let ((tool (first (canonicalize-tool-list (list tool)))))
    (condition-resume
      `(sys:abort
	 :skip-tool
	 t
	 ("Skip loading the ~A tool" ,(first tool))
	 (lambda (ignore) nil)
	 )
      (apply #'make-system
	     (first tool)
	     (funcall *tool-keyword-modify-function*
		      (first tool)
		      *load-tools-make-system-keywords*
	     )
      )
      (funcall *after-load-tool-hook* (first tool))
    )
  )
)
  
(defun load-tool-list (tool-list)
  "Load the tools named in TOOL-LIST (format like *KNOWN-TOOLS*).
   Not meant to be called directly--use LOAD-TOOLS."
  (loop for tool in tool-list do (load-tool-internal (first tool))))

;;; JPR.
;;;Edited by rice                  28 Nov 89  16:54
(defun tool-documentation-1 (doc-spec)
  (typecase doc-spec
    (cons (mapc 'tool-documentation-1 doc-spec))
    (string (format t "~A" doc-spec))
    (otherwise (if (functionp doc-spec)
		   (funcall doc-spec)
		   (format t "~S" (or doc-spec ""))
	       )
    )
  )
)

(defun tool-documentation (doc-spec)
  (with-output-to-string (*standard-output*)
    (tool-documentation-1 doc-spec)
  )
)

(defun document-tool-list (tool-list)
  "Document the tools in TOOL-LIST, which is like *KNOWN-TOOLS*.  This
   is not meant to be used directly--use LOAD-TOOLS."
  (using-resource (output w:pop-up-text-window)
    (send output :set-label "Tool Documentation")
    (send output :expose)
    (w:window-call (output :deactivate)
      (let ((*terminal-io* output))   ;In case of [Abort] printout and the like
	(loop for tool in tool-list doing
	      (when (typep tool 'cons)
		;;; Modded here by JPR.
		(if (third tool)
		    (format output "~2&Documentation for tool ~A:~2&~A~&"
			    (first tool)
			    ;;; Modded here by JPR.
			    (Tool-Documentation (third tool)))
		    (format output "~2&There is no documentation for tool ~A.~&"
			    (first tool))
		    )
		)
	      )
	(w:await-user-typeahead output))
      )
    )
  )

(defun canonicalize-tool-list (tool-names)
  "TOOL-LIST is :ALL, :BY-MENU, or a list of names.  Returns a list of
  tools like *KNOWN-TOOLS*."
  (typecase tool-names
    (list
     (delete
       nil
       (loop for tool-name in tool-names
	     with defs-loaded? = nil
	     as tool-sym = (intern (string tool-name) 'keyword)
	     when (or
		    ;;; String-equal test put in by JPR because sometimes
		    ;;; tool-sym is in a different package.
		    (assoc tool-sym *known-tools* :test #'string-equal)
		    (progn
		      ;; Make sure we've got the latest info.
		      (unless defs-loaded?
			(load "TOOLS:TOOLS;DEFSYSTEMS" #+TI :verbose #+TI nil)
			(setf defs-loaded? t)
			)
		      ;;; String-equal test put in by JPR because sometimes
		      ;;; tool-sym is in a different package.
		      (or (assoc tool-sym *known-tools* :test #'string-equal)
			  (progn (cerror "Skip loading this tool" "Unknown tool: ~A." tool-name)
				 nil))
		      )
		    )
	     collect it)
       )
     )
    (symbol
     (case tool-names
       (:all *known-tools*)
       (:by-menu (tool-list-from-menu *known-tools*))
       (otherwise (error
		    "The symbol ~S is not the keyword :ALL or :BY-MENU, which is all LOAD-TOOLS knows how to deal with"
		    tool-names))))
    (t (error "The argument TOOL-NAMES, ~S, was not :ALL, :BY-MENU, or a list of tool names." tool-names)
       )
    )
  )

(defun tool-list-from-menu (tool-list)
  "Present a menu of tool names from TOOL-LIST (format like
   *KNOWN-TOOLS*) and return a sublist of TOOL-LIST, or NIL if none are
   selected. Not meant to be called directly--use LOAD-TOOLS."
  (multiple-value-bind (selected-tools selection-made?)
      (choose-some-of (build-tool-menu-items tool-list))
    (when selection-made?
      selected-tools)
    )
  )

#+TI
(defun choose-some-of (items)
  (w:multiple-menu-choose items))

#+Symbolics
(defun choose-some-of (items)
  (let ((menu (tv:make-window
		'tv:momentary-multiple-menu
		:item-list items
		:font-map '(fonts:jess14 fonts:hl12i fonts:medfnb))
	      )
	result)
    (setq result (send menu :choose))
    (if result
	(values result t)
	(values nil nil)
	)
    )
  )

(defun build-tool-menu-items (tool-list)
  "TOOL-LIST is a list of tools like *KNOWN-TOOLS*.  Return a list of
   menu items such that for each system there is an item such that the
   short doc string is the mouse documentation, the NAME of the system
   is the menu string, and the item from TOOL-LIST is returned when the
   item is selected."
  (let ((l (loop for tool in tool-list collecting
		 `(,(string-capitalize-words (string (first tool)))
		   :value ,tool
		   :documentation ,(second tool))
		 )
	   )
	)
    (setf (rest (last l))
	  (list
	    '("Document Selected Tools" :value :document
	      :documentation "Document any seletected tools instead of loading them"
	      :font fonts:tr12bi)
	    )
	  )
    l
    )
  )

#+TI(tv:add-to-system-menu-column :user-aids "Load Tools" '(load-tools)
	 "Present a menu of useful tools which can then be loaded and used.")

#+Symbolics(tv:add-to-system-menu-programs-column "Load Tools" '(load-tools)
	 "Present a menu of useful tools which can then be loaded and used.")

;-------------------------------------------------------------------------------

;;; The following was added by JPR on 16 Nov 87 to define a couple of Zmacs
;;; commands.


;-------------------------------------------------------------------------------

;;; The following defines the Zmacs command M-X Load Tools

(defun make-tool-alist-item (tool)
"Makes an alist item for the completing minibuffer read thing from the known
 tools list.
"
  (list (string-capitalize-words (string (first tool))) (first tool))
)


(defun read-tool-name (prompt)
"Reads the name of a tool from the user with a prompt."
  (let ((tool-name (zwei:completing-read-from-mini-buffer prompt
			 (mapcar #'make-tool-alist-item
				 *known-tools*
			 )
		   )
	)
       )
       (if (consp tool-name) (second tool-name) (zwei:barf))
  )
)

(defun zwei:load-tool (a-tool)
"Loads a tool."
  (load-tools (list a-tool))
)

(zwei:defcom zwei:com-load-tool "Load a tool" ()
  (let ((sys:inhibit-fdefine-warnings t))
       (zwei:load-tool (read-tool-name "Tool to load:"))
  )
  zwei:dis-none
)

;;; Make the system know about the new Load Tools command.
(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-load-tool))
)

;-------------------------------------------------------------------------------

;;; The following defines the Zmacs command M-X Document Tools


(defun itemise-symbol (stream symbol symbol-string)
"Given a symbol, which was extracted from Symbol-String by read, print it to
 Stream as a mouse sensitive item.
"
  (declare (type string symbol-string))
  (let ((print-string (string-trim '(#\Space #\Tab #\Newline) symbol-string)))
       (let ((start (search (the string print-string) symbol-string)))
            (let ((first (subseq symbol-string 0 start))
		  (rest (subseq symbol-string
				(+ start (length (the string print-string)))
			)
		  )
		 )
	         (format stream "~A" first)
		 (send stream :item 'zwei:edit-definition symbol "~A"
		       print-string
		 )
		 (format stream "~A" rest)
	    )
       )
  )
)

(defun source-file-ok (source reference)
"True is the source file is all right to make a mouse sensitive item."
  (and source reference
;       (eq (send source :host) (send reference :host))
;       (equalp (send source :directory) (send reference :directory))
;       (or (beep) t)
  )
)

(defun ignore-char (stream char)
"A reader function, which simply ignores a char."
  (ignore stream char)
  nil
)

(defun make-macro-char-free-readtable ()
"Make a readtable, which doesn't contain any macro characters.  Replace the
 macro chars with a dummy reference.  This will mean that one won't get any
 comma inside a backquote errors whilst reading.
"
  (let ((a-readtable (copy-readtable nil)))
       (let ((macro-table (sys:character-macro-table a-readtable)))
	    (loop for i from 0 to (- (array-total-size macro-table) 1) do
		  (if (equal (aref macro-table i) 'sys:read-token)
		      nil
		      (setf (aref macro-table i) 'ignore-char)
		  )
	    )
       )
       a-readtable
  )
)

(defparameter *macro-char-free-readtable* (make-macro-char-free-readtable)
"A readtable, which has no macrochars in it.  The normal CL macro chars are
 replaced with dummy macrochar processors.
"
)

(defun read-something (line index)
"Reads a token from line starting at Index.  Returns the values
 (the things read, the end index, error-p).
"
  (multiple-value-bind (results error-p)
      (catch-error (multiple-value-list
		     (read-from-string line nil :.end-of-line. :start index)
		   )
		   nil
      )
    (if error-p
	(values nil (+ 1 index) t)
	(values (first results) (second results) nil)
    )
  )
)

(defun itemise-interesting-symbols (string stream source-path)
"Given a string, read from it and print it out to *standard-output*, making any
 interesting looking symbols mouse sensitive.
"
  (declare (type string string))
  (let ((index 0)
	(*readtable* *macro-char-free-readtable*)
       )
       (loop until (equal index  (length string)) do
	     (multiple-value-bind (something end-index error-p)
		 (read-something string index)
	       (if (and (not error-p)
			(symbolp something)
			(not (equal (find-package "LISP")
				    (symbol-package something)
			     )
			)
			(source-file-ok
			  (first (zwei:source-file-names something))
			  source-path
		        )
		   )
		   (itemise-symbol stream something
				   (subseq string index end-index)
		   )
		   (format stream "~A" (subseq string index end-index))
	       )
	       (setq index end-index)
	     )
       )
  )
)

(defun get-first-file-from-system (tool)
"Given the name of a tool, returns one of the files in the system that defines
 that tool.
"
  (let ((system (sys:find-system-named tool t t)))
       (if system
	   (let ((module (first (sys:system-modules system))))
	        (let ((path (first (first (sys:module-components module)))))
		     (fs:merge-pathnames path '((:Type :Lisp)))
		)
	   )
	   nil
       )
  )
)

(defun print-tool-documentation (tool stream)
"Prints the docs for a tool to a given stream.  If the stream understands the
 :Item message then the printing is done in such a way as to make any
 interesting looking symbols mouse sensitive
."
  ;;; String-equal test put in by JPR because sometimes
  ;;; tool-sym is in a different package.
  (let ((docs (Tool-Documentation
		(third (assoc tool tools:*known-tools* :test #'string-equal))
	      )
	)
       )
       (if (send stream :operation-handled-p :item)
	   (let ((file (get-first-file-from-system tool)))
	        ;;; Find a file in the defining system.  Then try to find a
	        ;;; package attribute in it, so as to be able to bind *package*
	        ;;; during the attempted reads.
	        (if file
		    (let ((attributes (fs:file-attribute-list file)))
			 (let ((package-key
				 (get (cons nil attributes) :package)
			       )
			      )
			      (let ((*package* (if package-key
						   (find-package package-key)
						   *package*
					       )
				    )
				   )
				   (format stream "~&Documentation for ")
				   (send stream :Item 'zwei:tool tool
					"~A" (string-capitalize-words
					       (string tool)
					     )
				   )
				   (format stream ":~%")
				   (Itemise-Interesting-Symbols
				     docs stream file
				   )
			      )
			 )
		    )
		    (progn (format stream "~&Documentation for ~A:~%"
				   (string-capitalize-words (string tool))
			   )
			   (format stream docs)
		    )
		)
	   )
	   (format stream docs)
       )
  )
)

(zwei:defcom zwei:com-document-tool "Document a tool" ()
  (let ((tool (read-tool-name "Tool to document:")))
       (print-tool-documentation tool *standard-output*)
  )
  zwei:dis-none
)

;;; Make the system know about the new Document Tools command.
(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-document-tool))
)

(defun add-to-item-type-alist (item)
  (let ((match (assoc (first item) zwei:*typeout-command-alist*)))
       (if match
	   (setf (nth (position match zwei:*typeout-command-alist*)
		      zwei:*typeout-command-alist*
		 )
		 item
	   )
	   (nconc zwei:*typeout-command-alist* (list item))
       )
  )
  nil
)

(add-to-item-type-alist '(zwei:tool zwei:load-tool
				    "L: Load this tool, R: Menu of Load tool."
				    ("Load Tool" :Value zwei:load-tool
				     :Documentation "Load this tool"
				    )
			 )
)

(add-to-item-type-alist '(zwei:edit-definition
			   zwei:edit-definition-for-mouse
			   "L: Edit definition, R: Menu of edit definition."
			   ("Edit Definition" :value
			    zwei:edit-definition-for-mouse
			    :Documentation "Edit the definition of this name."
			   )
			 )
)

;-------------------------------------------------------------------------------

(defun uniqueise-tools (tools)
  (if tools
      (cons (first tools)
	    (remove-if #'(lambda (tool) (eq (first tool) (first (first tools))))
		       (rest tools)
	    )
      )
      nil
  )
)

(defun shift-font-to (to stream)
  (if (send stream :operation-handled-p :set-current-font)
      (send stream :set-current-font to)
      (princ (if (equal 0 to) "*" (format nil "~A" to)) stream)
  )
)

(defun new-page (stream)
  (terpri stream)
  (terpri stream)
  (terpri stream)
  (terpri stream)
  (if (send stream :operation-handled-p :set-current-font)
      ;;; This is a window so just throw a few blank lines."
      nil
      (princ "" stream)
  )
)

(defun print-out-tool (ostream tool fonts)
  (format ostream "~&")
  (shift-font-to 2 ostream)
  (princ (string-capitalize (first tool) :spaces t) ostream)
  (shift-font-to 0 ostream)
  (terpri ostream)
  (terpri ostream)
  (shift-font-to 1 ostream)
  (princ (second tool) ostream)
  (shift-font-to 0 ostream)
  (terpri ostream)
  (terpri ostream)
  (princ (third tool) ostream)
  (new-page ostream)
)

(defun print-out-tool-docstrings-1 (ostream fonts)
  (let ((tools (uniqueise-tools *known-tools*)))
       (mapcar #'(lambda (tool) (print-out-tool ostream tool fonts)) tools)
  )
)

(defun print-mode-line (ostream fonts)
  (format ostream
	  ";;; -*- Mode:Text; Package:User; Base:10; Fonts:~A -*-"
	  fonts
  )
)

(defun print-out-tool-docstrings
       (to &optional (fonts '(cptfont cptfontb metsbi)))
  (let ((real-fonts (mapcar #'string-capitalize (mapcar #'symbol-name fonts))))
       ;;; make it package independent.
       (cond ((typep to 'stream)
	      (print-out-tool-docstrings-1 to real-fonts)
	     )
	     ((equal nil to)
	      (with-output-to-string (*standard-output*)
		(print-out-tool-docstrings-1 *standard-output* real-fonts)
	      )
	     )
	     ((equal t to)
	      (print-out-tool-docstrings-1 *standard-output* real-fonts)
	     )
	     (t (with-open-file (ostream to :direction :output)
		  (print-mode-line ostream fonts)
		  (print-out-tool-docstrings-1 ostream real-fonts)
		)
	     )
       )
  )
  nil
)


;-------------------------------------------------------------------------------

;;;Edited by RICE                  12 Feb 90  12:06
;;;Edited by RICE                  12 Feb 90  12:08
(defun tool-should-be-compiled-p (sys &optional (or-loaded-p nil))
"True if the system SYS has files that need to be compiled 
 (or loaded if or-loaded-p)."
  (declare (values files-to-compile-or-load-string-or-nil
		   compile-or-load-required-p
           )
  )
  (let ((str (with-output-to-string (*standard-output*)
	       (make-system sys :Compile :Safe :Noconfirm :Nowarn :Print-Only)
	     )
	)
       )
       (if (search "COMPILED" str :Test #'char-equal)
	   (values str :compilation-required)
	   (if (and or-loaded-p (search "LOADED" str :Test #'char-equal))
	       (values str :loading-required)
	       nil
	   )
       )
  )
)


(defun maybe-compile-tool (sys confirm-p load-p)
"Make-system compiles the system SYS if there are any files than should be
compiled, otherwise does nothing.
"
  (let ((result (tool-should-be-compiled-p sys))
	(compiler:*output-version-behavior* :Newest)
       )
       (if (and result (or (not confirm-p) (y-or-n-p "Compile ~A?" sys)))
	   (progn
	     (format t "~&~%~%Making System :- ~A~%~A~%~%" sys result)
	     (make-system sys :Compile :Safe :Noconfirm :Really-Nowarn)
	   )
	   (if load-p
	       (progn (format t "~&Loading tool :- ~A" sys) (load-tools (list sys)))
	       (format t "~&NOT Making System :- ~A" sys)
	   )
       )
  )
)

(defvar *tools-not-to-compile* '(:Dvi-Previewer :Explorer-36xx :Ksl-Patches)
"A list of tool names not to compile when maybe-compile-tools is called."
)

(defvar *tools-to-try-at-the-end* '(:genera-to-explorer)
"A list of tools that tend to corrupt the environment in a way that
would make you want to try doing their compiles after those of all
of the other tools.
"
)

(defun tools-to-compile ()
  (append (set-difference
	    (set-difference (mapcar 'first *known-tools*)
			    *Tools-Not-To-Compile*
	    )
	    *tools-to-try-at-the-end*
	  )
	  *tools-to-try-at-the-end*
  )
)

(defun maybe-compile-tools (&optional (confirm-p nil) (load-p nil))
"Loops through all of the tools and make-system compiles any tools which have
files that need to be compiled.
"
  (make-system :ip          :noconfirm :nowarn :silent :safe)
  (make-system :rpc         :noconfirm :nowarn :silent :safe)
  (if (not (sys:mx-p)) 
      (make-system :nfs         :noconfirm :nowarn :silent :safe)
      nil
  )
  (make-system :ksl-patches :noconfirm :nowarn :silent :safe)
  (if (not (fboundp 'add:find-channel))
      (setf (symbol-function 'add:find-channel)
	    #'(lambda (&rest ignore))
      )
      nil
  )
  (if (not (fboundp 'add:wait-port-ready))
      (setf (symbol-function 'add:wait-port-ready)
	    #'(lambda (&rest ignore))
      )
      nil
  )
  (mapc #'(lambda (tool) (maybe-compile-tool tool confirm-p load-p)) (tools-to-compile))
)

(defvar *tools-already-recompiled* nil
"A list of all of the tools that we've recompiled during a global recompile.
This allows us to prevent multiple recompiles.
"
)

(defun recompile-tools ()
"Recompiles all reasonable tools.  If a tool has calls to
LOAD-TOOLS in it then these are recompiled recursively, but
only once.
"
  (let ((tools (tools-to-compile))
	(*tool-keyword-modify-function*
	  #'(lambda (system keys)
	      (if (member system *tools-already-recompiled*
			  :test #'string-equal
		  )
		  (subst :noop :recompile keys)
		  keys
	      )
	    )
	)
	(*after-load-tool-hook*
	  #'(lambda (system)
	      (pushnew system *tools-already-recompiled* :test #'string-equal)
	    )
	)
	(*load-tools-make-system-keywords*
	  '(:recompile :noconfirm :nowarn :safe)
	)
       )
       (load-tools tools)
  )
)

;-------------------------------------------------------------------------------
