;;; -*- Mode:Common-Lisp; Package:FILE-SYSTEM; Base:10; Fonts:(TVFONT) -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in May '86.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;-------------------------------------------------------------------------------
;
;				Map-Over-Files
;
;-------------------------------------------------------------------------------

;;; Author: J P Rice
;;; Date  : 24 May 86

;;; This file contains the definition of the function Global:Map-Over-Files,
;;; A higher order function for the application of functions to complex
;;; filespecs.

;-------------------------------------------------------------------------------

(eval-when (compile load eval) (load-tools '(:36Xx-Explorer)))

(let ((*standard-output* 'si:null-stream))
     (declare (special *standard-output*))
     (mapcar #'(lambda (pkg)
		 (unintern 'ticl:map-over-files (symbol-package pkg))
	       )
	       (where-is "MAP-OVER-FILES")
     )
)

(defun ticl:map-over-files
    (a-function paths &Key (Accumulate nil) (Exclude nil)
		    	   (Log nil)        (Arguments nil)
			   (Confirm nil)    (Bottom-First nil)
			   (Initial-Value nil) (remote-host nil)
			   (remote-bindings nil)
    )
"This is a higher order function which allows you to map a given function over
 a collection of files in, hopefully quite a sophisticated way.  It takes two
 mandatory arguments and a number of optional keyword arguments.  The two
 mandatory arguments are :-

	a-function    - This is the function which will be applied to the
			files selected.  For a specification of the
			arguments to this function see the optional arguments
			below.  The pathname denoting the file being mapped
			over it always the first argument.
	paths	      - This item can be either an individual item or a list
			of items.  Each item can be one of three things; a
			string denoting a path, a pathname object denoting the
			desired path or a two-list of the form (:Wild-Inferiors
			<path>).  The pathnames given for this argument can be
			arbitarily wildcarded.  The list form for an item
			should be used if the host type being refered to in the
			path does not support the :Wild-Inferiors pathname
			option.  In this case the expression
			(:Wild-Inferiors 'host:>foo>bar>baz.lisp') is
			equivalent to 'host:>foo>bar>**>baz.lisp'.

The optional (keyword) arguments are as follows :-

	Accumulate    - This argument allows you to accumulate the results of
			the calls to the a-function argument.  If this option
			is selected to be non-nil then the Second argument to
			the a-function argument will be the currently
			accumulated result.  If this option is not specified
			then no accumulation argument will be supplied to the
			a-function argument when it is called.  An initial value
			is needed for the accumulation process.  The default
			value for the initial value is nil.  If you want
			something different from this then you should use the
			Initial-Value keyword argument (see below).
	Exclude	      - This option allows you to supply an item or a list of
			items, like for the paths parameter.  in this case,
			however all files whose paths match any of the paths in
			the exclude component do not have the a-function
			argument applied to them.  Like the paths argument this
			argument can be arbirarily wildcarded.
	Log	      - This argument, if non-nil will cause the name of the
			file being processed to be printed out just before it
			is processed.  If the value of Log is a stream then the
			output will be printed onto that stream, otherwise it
			will come out on Standard-Output.
	Arguments     - This argument must be a list of extra arguments that are
			to be passed to the a-function argument, when it is
			called.  The arguments are placed after the path to be
			operated on and after the accumulation argument if it
			has been specified.
	Confirm	      - This argument, if supplied will cause the user to be
			prompted with the name of the file to be operated on
			before it is operated on with a y-or-n-p.  If the user
			selects N then the operation will not be performed.  If
			the value of the Confirm argument is a stream then the
			query will come out on this stream, otherwise it will
			come out, as usual, on *Query-io*.
	Bottom-First -  This argument is useful if the order in which you
			traverse the directory structure is of importance.
			Normally the traversal is from top to bottom.  If,
			however it is of importance that the directories should
			be traversed fomr bottom up then this option should be
			non-nil.  This is particularly useful, for instance, if
			you are trying to do a delete and expunge of a tree of
			directories.  On some file systems, directories can only
			be deleteed if their contents have been deleted and
			expunged.  Thus you have to start at the bottom.
	Initial-Value - This argument has to be used in conjunction with
			Accumulate.  A value specified for this argument will
			be used as the initial argument for the accumulation of
			results.  For more information please see above under
			Accumulate.
        Remote-Host   - If supplied this must be the name of an explorer host
                        which is known to be running the map-over-files system
                        and must have loaded the function you want to execute.
                        All arguments that are passed to the remote host must
                        be evaluable on this host.  This operation is worthwhile
                        because file operations are often much cheaper on the
                        local file system.
                        e.g.: (search-and-replace
                                 \"foo\" \"sys:windows;*.lisp#>\"
                                 :Remote-Host \"sys\")
        Remote-Bindings - A binding list which is used at the server when a map
                        is being evaluated remotely.  This allows the server to
                        run in a special environment.
"
  (if (and remote-host
	   (let ((host (net:parse-host remote-host t)))
	        (and host
		     (let ((trans (or (send host :Host-Translation) host)))
		          (equal (send (net:parse-host trans) :System-Type)
				 :Lispm
			  )
		     )
		)
	   )
      )
      (Invoke-Map-Over-Files-On-Host
	(let ((host (net:parse-host remote-host)))
	     (net:parse-host (or (send host :Host-Translation) host))
	)
	Remote-Bindings
	(cond ((and confirm (typep (sys:follow-syn-stream confirm) 'tv:sheet))
	       (sys:follow-syn-stream confirm)
	      )
	      ((typep (sys:follow-syn-stream *standard-output*) 'tv:sheet)
	       (sys:follow-syn-stream *standard-output*)
	      )
	      ((typep (sys:follow-syn-stream *query-io*) 'tv:sheet)
	       (sys:follow-syn-stream *query-io*)
	      )
	      (t tv:selected-window)
        )
	a-function
	(loop for path in (if (listp paths) paths (list paths))
	      collect (send (send (fs:default-pathname path)
				  :Translated-Pathname
			    )
			    :String-For-Printing
		      )
	)
	:Accumulate accumulate
	:Exclude exclude
	:Log log
	:Arguments arguments
	:Confirm confirm
	:Bottom-First bottom-first
	:Initial-Value initial-value
      )
      (let ((all-files (filter-files (get-files-from-filespecs paths)
				     (if (listp exclude) exclude (List exclude))
				     Bottom-first
		       )
	    )
	   )
	   (funcall (if Accumulate
			#'Actually-Map-and-accumulate-over-files
			#'Actually-Map-over-files
		    )
		    a-function all-files Arguments Initial-Value log Confirm
	   )
      )
  )
)


(Export 'ticl:Map-Over-Files 'TICL)


(defvar *end-of-file-hook* nil
"Is non nil then it's a function to call at the end of processing a file."
)

(defvar *start-of-file-hook* nil
"Is non nil then it's a function to call at the start of processing a file."
)

(defun confirm-if-you-must (file confirm)
"If you must confirm then the user is asked whether the operation is to be
 performed on the file.
"
  (declare (special *Query-io*))
  (or (not Confirm)
      (let ((*Query-IO* (if (streamp Confirm) Confirm *Query-io*)))
	   (declare (special *Query-IO*))
           (y-or-n-p "~A ?" (Send File :String-for-Printing))
      )
  )
)

(defvar *log-print-function* 'format
"The function to use when printing out log entries.  Requires an arglist
like format."
)

(defun Actually-Map-over-files (a-function files other-arguments
				initial-value Log Confirm
			       )
"This is the function that actually does the mapping over a collection of
 files, without accumulating the results of each computation.  It takes a
 function to map, a list of all of the files that are to have the function
 mapped over them, an accumulating result initial value argument which it
 ignores and two flags, one to tell it whether it should be logging the events
 and one to tell it whether it should prompt the user before any action is
 taken on each file.  These arguments are constructed suitably by
 Map-Over-files.  The semantics of this function are very simple.  All that
 they do is implement the definition specified in the documentation for
 Map-Over-Files.
"
  (declare (special *Start-Of-File-Hook* *End-Of-File-Hook*))
  (Ignore initial-value)
  (Loop for file in files
    Do (if Log
	   (if (streamp Log)
	       (funcall *log-print-function*
			log "~&~A" (Send File :String-For-Printing)
	       )
	       (funcall *log-print-function*
			t   "~&~A" (Send File :String-For-Printing)
	       )
	   )
	   nil
       )
    Collect
       (if (confirm-if-you-must file confirm)			
	   (prog2 (if *Start-Of-File-Hook*
		      (funcall *Start-Of-File-Hook* file)
		      nil
		  )
		  (Apply a-function File other-arguments)
	          (if *End-Of-File-Hook*
		      (Funcall *End-Of-File-Hook* file)
		      nil
		  )
	   )
	   nil
	)
  )
)


(defun Actually-Map-and-accumulate-over-files (a-function files
					       other-arguments initial-value
					       Log Confirm
					      )
"This is the function that actually does the mapping over a collecxtion of
 files.  It takes a function to map, a list of all of the files that are to
 have the function mapped over them, an accumulating result initial value
 argument and two flags, one to tell it whether it should be logging the events
 and one to tell it whether it should prompt the user before any action is
 taken on each file.  These arguments are constructed suitably by
 Map-Over-files.  The semantics of this function are very simple.  All that
 they do is implement the definition specified in the documentation for
 Map-Over-Files.
"
  (let ((result initial-value))
       (Loop For file in files
	 Do (if Log
		(if (streamp Log)
		    (funcall *log-print-function*
			     log "~&~A" (Send File :String-For-Printing)
		    )
	            (funcall *log-print-function*
			     t   "~&~A" (Send File :String-For-Printing)
		    )
		)
		nil
	    )
	 Do (Setq result
		  (if (confirm-if-you-must file confirm)			
		      (Apply a-function File Result other-arguments)
		      nil
		   )
	    )
       )
       Result
  )
)


(defun map-with-args (a-function over-a-list &Rest other-arguments)
"This function is much like mapcar only more useful.  It takes a function and a
 list to map the function over and an &Rest arguments feature.  It applies the
 function to each element in the list, with the element being the first
 argument and any subsequent arguments being taken from the &Rest paremeter. 
 The value of a call to this function is a list of values from this function
 call, one element for each element in the source list.
"
  (Loop For element In over-a-list
        Collect (apply a-function element other-arguments)
  )
)


(defun get-files-from-filespecs (filespecs)
"This function takes either a filespec in the form of a string or a pathname or
 a list of filespecs in the same form.  It returns the list of pathnames that
 match the filspecs that have been provided.  The filespecs that are provided
 are merged with the pathname 'a:>b>*.lisp', where a and b represent the user's
 homedir and host.  This should allow the user to specify shorter pathnames for
 his own files.
 The function works by making a list of all of the files defined by all of the
 filespecs.  It finds all of the filespecs by mapping a merge pathname over the
 filspecs provided.  It then maps fs:directory list over this list of
 filespecs.  This returns a list of plists; one for each filespec.  It squashes
 these together, using append, and then takes the head of each plist, which is
 the pathname of the file in question.  It deletes any nils from this list,
 since directories generate plists with a nil for the pathname of the file.
"
  (let ((mapping-default-pathname
	    (fs:Make-Pathname :Host      (Send (fs:User-Homedir) :Host)
			      :Directory (Send (fs:User-Homedir) :Directory)
			      :Name      :Wild
			      :Type      :Lisp
	    )
	)
       )
       (Delete nil
	(Mapcar #'First
	  (Apply #'Append
	   (mapcar #'find-directory-list
		   (Mapcar #'(Lambda (a-filespec)
			       (if (consp a-filespec)
				   (list (first a-filespec)
					 (fs:Merge-Pathnames (second a-filespec)
						   mapping-default-pathname
				         )
				    )
				   (fs:Merge-Pathnames a-filespec
						       mapping-default-pathname
				   )
			       )
			   )
			   (if (and (consp filespecs)
				    (not (equal :Wild-Inferiors
						(first filespecs)
					 )
				    )
			       )
			       filespecs
			       (List filespecs)
			   )
		   )
	   )
	  )
	)
       )
  )
)


(defun get-directories (path)
"This function takes a pathname object or a string denoting a pathname for a
 directory and returns a list of all directory files, which match that
 specification.  This is used by those functions that generate menus for the
 user, which contain a collection of directory names to chose from.
"
  (mapcar #'First
          (remove-if-not #'(Lambda (file-data) (Get file-data :Directory))
			 (the list
			      (fs:directory-list
				(send (send (fs:default-pathname path)
					    :New-Name :Wild
				      )
				      :New-Type :Wild
				)
				:Sorted
			      )
			 )
          )
  )
)



(defun find-inferior (directory filespec)
"This function, when passed a pathname denoting a directory (e.g.
 a:b.c;d.directory) and a filespec, which is being used for file mapping,
 delivers a new filespec, rather like the original filespec but with the
 dirspec extended down to include Directory.  This if the args are :-
 #<a:b.c;d.directory> and (:Wild-Inferiors 'a:~;*.lisp') the result will be
 (:Wild-Inferiors #<a:b.c.d;*.lisp>).
"
  (List :Wild-Inferiors
        (Send (fs:default-pathname (second filespec)) :New-Directory
	      (if (equal (send directory :Directory) :Root)
		  (List (Send directory :Name))
		  (Append (send directory :Directory)
			  (List (Send directory :Name))
		  )
	      )
	)
  )
)

(defun find-directory-list (filespec)
"This function, when passed a filespec returns the directory list associated
 with it.  This is, therefore, rather like fs:directory-list only it allows the
 specification of wild-inferiors for systems that do not support the
 :Wild-Inferiors directory component.  It is passed either a pathname or a
 string denoting a pathname, such as 'lm:foo.bar;*.xfasl' or a list of two
 elements in this following form; (:Wild-Inferiors  'lm:foo.bar;*.xfasl').  The
 latter will return a directory list for all files in the .bar directory and
 any directories hanging from it.
"
  (if (and (consp filespec) (equal (first filespec) :Wild-Inferiors))
      (let ((files-at-this-level (Fs:Directory-List (second filespec) :Sorted))
	    (directories (get-directories (second filespec)))
	   )
	   (Append files-at-this-level
		   (Apply #'Append (mapcar #'find-directory-list
					   (map-with-args #'find-inferior
							  directories filespec
					   )
				   )
		   )
	   )
      )
      (Fs:Directory-List filespec :Sorted)
  )
)



(defun Should-a-file-be-excluded (an-exclusion a-file a-default-pathname)
"This is a predicate, which determines whether a given file should be excluded
 from the list of things over which to map.  An-Exclusion is an exclusion taken
 from the list of exclusions.  A-file is a pathname of a real file. 
 A-default-pathname is a pathname, which is merged with the exclusion in order
 to make it a bit more flexible.
 If the file matches the exclusion, merged with the default pathname, then the
 file is excluded.
 If the paths don't match then there is more of a problem.  The wildcard match
 facility doesn't do the 'right thing' as far as :Oldest and :Newest is
 concerned so the fact that you didn't get a match could well have been due to
 the fact that it didn't do the right thing with these.  Thus the function
 checks the exclusion to see if it has an oldest or a newest in it.  If it
 hasn't then the file is rejected.  If it has then the function looks for the
 file with the :Oldest or :Newest version number as appropriate.  It then
 attempts the original merge again, only with the version number of the
 defaulted, merged pathname suitably changed to the real version number of the
 :Oldest or :Newest file.
"
  (if (Send (fs:Merge-pathnames an-exclusion a-default-pathname :Wild)
	    :Pathname-Match a-file  nil
      )
      t
      (if (or (search ".NEWEST" an-exclusion :Test #'String-Equal)
	      (search ".OLDEST" an-exclusion :Test #'String-Equal)
	  )
	  (let ((file-and-version
		    (probe-file (send a-file :new-version
				      (if (search ".NEWEST" an-exclusion
						  :Test #'String-Equal
					  )
					  :Newest
					  :Oldest
				      )
			        )
		    )
		)
	       )
	       (Send (Send (fs:Merge-pathnames an-exclusion
					       a-default-pathname :Wild
			   )
			   :new-version
			   (Send file-and-version :Version)
		     )
		     :Pathname-Match a-file  nil
	       )
	   )
	   nil
      )
  )
)


(defun filter-files (all-files exclude bottom-first)
"This function takes a list of all of the files that match the original
 filespecs for Map-Over-Files, a list of exclude components and a flag that
 specifies whether the traversal of the list is to be bottom-first or not. 
 This function returns a list of files that has been filtered and suitably
 ordered so that the files that match the exclusion specification have been
 removed and the list has been order in a top-first or bottom-first manner. 
 The first thing that the function does is to make a reduced list of all of the
 files.  This is done by mapping a check of all of the exclusions over all of
 the files.  When the test for exclusion is performed on a file a list of test
 results is returned.  If this list contains a single non-nil entry then the
 file is rejected.  The result of this map, therefore is a list of files, with
 nils denoting each of the files that has been rejected.  These nils are
 removed.  This completes the generation of the reduced list of files.  The
 list to return is then generated quite simply.  If the list is not to be a
 bottom first one, then the result is simply the original list, if there are no
 exclusions or the reduced list.  If is is to be a bottom up list then the list
 is sorted according to the length of the directory components.  Tis ensures
 that the bottom files will be processed first.
"
  (let ((a-default-pathname
	  (fs:Make-Pathname
		:Host (Send (fs:User-Homedir) :Host)
		:Directory :Wild
		:Name      :Wild
		:Type      :Lisp
		:Version   :Wild
	  )
	)
       )
       (let ((reduced-files
	       (delete nil
		  (mapcar
		   #'(Lambda (a-file)
		      (if (delete nil
			   (map-with-args #'Should-a-file-be-excluded
				     exclude a-file a-default-pathname
			   )
			  )
			  nil
			  a-file
		       )
		      )
		      all-files
		   )
	       )
	     )
	    )
	    (if bottom-first
	        (Sort (if (equal nil exclude) all-files reduced-files)
		    #'(Lambda (file-1 file-2)
			      (> (Length (Send file-1 :Directory))
				 (Length (Send file-2 :Directory))
			      )
		      )
		)
		(if (equal nil exclude) all-files reduced-files)
	    )
       )
  )
)


;-------------------------------------------------------------------------------
;		    Some functions that use Map-Over-Files.
;-------------------------------------------------------------------------------



(defun set-retention-count (a-file to)
"Sets the generation retention count of a file to the value specified."
  (let ((host (send a-file :host)))
       (let ((system-type (send host :system-type)))
	    (case system-type
	      (:LMFS ;;; This is a symbolics.
	       (fs:change-file-properties
		 a-file t :Generation-retention-count to
	       )
	      )
	      (:LispM (fs:change-file-properties a-file t :Version-Limit to))
	      (otherwise
	       (ferror nil
	       "I don't know how to deal with this host / system type [~A / ~S]"
	               host system-type
	       )
	      )
	    )
       )
  )
)


(defun set-all-retention-counts (path &optional (to 3))
"Sets the generation retention counts for all files in Path to To."
  (map-over-files #'set-retention-count path
		  :confirm nil
		  :log t
		  :arguments (list to)
  )
)

(defun delete-older-files (path)
"Deletes all files that are not the newest in Path."
  (map-over-files #'(lambda (file)
		      (let ((new (probe-file (send file :New-Version :Newest))))
		           (if (< (send file :Version) (send new :Version))
			       (progn (delete-file file)
				      (format t "~&~A deleted." file)
			       )
			       nil
			   )
		      )
		    )
		  path
		  :confirm nil
		  :log t
  )
)


(defun accumulate-the-size-of-files-in-bytes (a-file size-so-far)
"Given a pathname of a file and the size of all of the files that have been
 counted so far this function delivers an integer, which is the old size
 plus the size of the file specified.  The size of the file is determined
 by extracting it from the property list for that file returned by calling
 fs:directory-list for that file.  Because :Bin files have 16 bit bytes
 and not 8 bit bytes this function normalises the sizes of files into 8 bit
 bytes by inspecting their byte-size properties.  The value of the call to
 fs:directory-list is in fact a list of property lists.  In this list there
 will be one element that refers to the file in question and one that refers
 to the directory itself.  The :Length-in-bytes property of the latter is nil.
 Thus when you map a get for this property over the list of lists the result
 will be a list with a nil in it from the directory component and a number,
 from the file. Thus if you delete all of the nils and take the head of this
 list then that will deliver the length.  There is an exception to this in the
 case of directory files, whicih for some reason have a :Length-in-Blocks
 property but not a :Length-in-bytes property.  The map in this case will
 return a list of nils.
" 
  (let ((size-of-file (mapcar #'(lambda (an-item)
				   (if (get an-item :Length-in-bytes)
				       (/ (* (get an-item :Length-in-bytes)
					     (get an-item :Byte-size)
				           )
				           8
				       )
				       nil
				   )
				)
				(fs:directory-list a-file)
		      )
	)
       )
       (+ (if (equal nil (delete nil size-of-file))
	      0
	      (First (delete nil size-of-file))
	  )
          size-so-far
       )
  )
)

(defun space-used-by (path)
"Given a pathname or a list of pathnames this function maps a function that
 will accumulate the lengths of files in bytes over all of the files defined by
 the pathnames given as an argument.
"
 (map-over-files #'accumulate-the-size-of-files-in-bytes
		 path :Log t :Accumulate t :Initial-Value 0
 )
)


(Defun Accumulate-the-size-of-files-in-records (a-file size-so-far)
"Given a pathname of a file and the size of all of the files that have been
 counted so far this function delivers an integer, which is the old size
 plus the size of the file specified.  The size of the file is determined
 by extracting it from the property list for that file returned by calling
 fs:directory-list for that file.  The value of the call to fs:directory-list
 is in fact a list of property lists.  In this list there will be one element
 that refers to the file in question and one that refers to the directory
 itself.  The :Length-in-blocks property of the latter is nil.  Thus when you
 map a get for this property over the list of lists the result will be a list
 with a nil in it from the directory component and a number, from the file.
 Thus if you delete all of the nils and take the head of this list then that
 will deliver the length.  This is returned as the accumulated result.
" 
  (let ((size-of-file (mapcar #'(lambda (an-item)
				  (get an-item :Length-in-blocks)
				)
				(fs:directory-list a-file)
		      )
	)
       )
       (+ (first (delete nil size-of-file)) size-so-far)
  )
)


(defun files-in-records (path)
"Given a pathname or a list of pathnames this function maps a function that
 will accumulate the lengths of files in records over all of the files defined
 by the pathnames given as an argument.
"
 (map-over-files #'accumulate-the-size-of-files-in-records
		 path :Log t :Accumulate t :Initial-Value 0
 )
)

;-------------------------------------------------------------------------------



(eval-when (compile) (eh:require-pdl-room 16000 6000))



(defun generic-query-server ()
"Starts up a server for y-or-n-ps and such like." 
  (with-open-stream (stream (net:listen-for-connection-on-medium
			      :Byte-Stream
			      "Generic-Query"
			      :Stream-Type :Ascii-Translating-Character-Stream
			    )
		    )
    (query-server-top-level stream)
  )
)


(net:define-service :Query (host) (host))

(defvar *Query-Port* 144 "The TCP port for search and replace.")

(net:define-logical-contact-name "Generic-Query"
				 `((:Chaos "Query")
				   (:Tcp ,*Query-Port*)
				  )
)

(net:add-server-for-medium
  :Byte-Stream "Generic-Query"
  '(process-run-function "Query Server" 'Generic-Query-Server)
)


(defun find-window-with-name (name)
"Finds a window with a given name."
  (loop for screen in tv:all-the-screens
	for result = (find-window-with-name-1 name screen)
	when result
	return result
	finally (return nil)
  )
)

(defun find-window-with-name-1 (name window)
  (if (equal name (send window :Name))
      window
      (loop for inferior in (send window :Inferiors)
	    for result = (find-window-with-name-1 name inferior)
	    when result
	    return result
	    finally (return nil)
      )
  )
)

(defun query-server-top-level (stream)
"Implements query service.  Is passed two args, the window name onto which to
do the query and a form to eval which will perform the query.
"
  (let ((window-name (With-Standard-Io-Environment (read stream)))
	(query-form (With-Standard-Io-Environment (read stream)))
       )
       (let ((window (find-window-with-name window-name)))
	    (if window
		(let ((process (if (typep window 'tv:process-mixin)
				   (send window :Process)
				   nil
			       )
		      )
		     )
		     (unwind-protect
			 (progn (if process
				    ;; Lock out any contending process waiting
				    ;; for input on this window.
				    (send process :Arrest-Reason :query-service)
				    nil
				)
				(let ((*query-io* window)
				      (*standard-output* window)
				      (*standard-input* window)
				      (*trace-output* window)
				      (*error-output* window)
				      (*debug-io* window)
				      (*terminal-io* window)
				     )
				     (let ((result (eval query-form)))
				          (With-Standard-Io-Environment
					    (format stream "~S~%" result)
					  )
					  (force-output stream)
				     )
				)
			 )
		       (if process
			   (send process :Revoke-Arrest-Reason :query-service)
			   nil
		       )
		     )
		)
		(progn (With-Standard-Io-Environment
			 (format stream "~S~%" :error-window-not-found)
		       )
		       (force-output stream)
		)
	    )
       )
  )
)


(defun Invoke-query-on-host (host form window-name)
"Performs a query denoted by Form on Host host using the window named
Window-Name for *query-io*.  E.g.:
   (Invoke-query-on-host \"x6\" '(y-or-n-p \"Ok?\") \"Lisp-Listener 1\")
"
  (with-open-stream (stream (net:open-connection-on-medium
			      (net:parse-host host) :Byte-Stream
			      "Generic-Query"
			      :Stream-Type
			      :Ascii-Translating-Character-Stream
			    )
		    )
    (With-Standard-Io-Environment (format stream "~S~%~S~%" window-name form))
    (force-output stream)
    (let ((result (With-Standard-Io-Environment (read stream))))
         (case result
	   (:Error-Window-Not-Found (ferror nil "Window could not be found."))
	   (otherwise result)
	 )
    )
  )
)


;-------------------------------------------------------------------------------


(defun generic-map-over-files-server ()
"Top level for a map over files server."
  (with-open-stream (stream (net:listen-for-connection-on-medium
			      :Byte-Stream
			      "Generic-Map-over-files"
			      :Stream-Type :Ascii-Translating-Character-Stream
			    )
		    )
    (map-over-files-server-top-level stream)
  )
)


(net:define-service :Map-over-files (host) (host))

(defvar *Map-over-files-Port* 145
  "The TCP port for map over files service"
)

(net:define-logical-contact-name "Generic-Map-over-files"
				 `((:Chaos "Map-over-files")
				   (:Tcp ,*Map-over-files-Port*)
				  )
)

(net:add-server-for-medium
  :Byte-Stream "Generic-Map-over-files"
  '(process-run-function "Map over Files Server"
			 'Generic-Map-over-files-Server
   )
)


(defun check-for-abort-from-client (file)
"An end of file hook to check for aborts from the client."
   (declare (special *stream*))
   (ignore file)
   (let ((char (with-timeout (1 0)
		 (send *stream* :any-tyi-no-hang)
	       )
	 )
	)
	(if (char= char 255)
	    (signal 'sys:abort "Abort")
	    nil
	)
   )
)

(defun map-over-files-server-top-level (stream)
"Implements remote map over files.  Is passed the name of the window that
 invoked this service on the client machine, the name of the client machine and
 the args to be passed to Map-over-files.  It then performs the map
 using query service for any y-or-n-ps that may come up.
"
  (if (catch-error
       (let ((window-name (With-Standard-Io-Environment (read stream)))
	     (from-host (With-Standard-Io-Environment (read stream)))
	     (remote-bindings (With-Standard-Io-Environment (read stream)))
	     (map-over-files-args (With-Standard-Io-Environment (read stream)))
	    )
	    (letf ((#'y-or-n-p
		    #'(lambda (&rest args)
			(let ((string (apply #'format nil args)))
			     (force-output stream)
			     (Invoke-Query-On-Host
			       from-host (list 'y-or-n-p string) window-name
			     )
			)
		      )
		   )
		  )
		  (let ((*standard-output* stream)
			(*stream* stream)
			(*End-Of-File-Hook* 'Check-For-Abort-From-Client)
			(*Start-Of-File-Hook* 'Check-For-Abort-From-Client)
			(*Log-Print-Function* 'server-log-print-function)
		       )
		       (Declare (special *stream* *Start-Of-File-Hook*
					 *End-Of-File-Hook*
					 *Log-Print-Function*
				)
		       )
		       (progw remote-bindings
			 (Multiple-value-bind (result error-p)
			     (catch-error
			       (values (apply #'map-over-files
					      map-over-files-args
				       )
				       nil
			       )
			       nil
			      )
			   (With-Standard-Io-Environment
			     (format stream "~%***END***~%")
			   )
			   (force-output stream)
			   (With-Standard-Io-Environment
			     (format stream "~S" (if error-p :Error result))
			   )
			 )
		       )
		  )
	    )
	    t ;; for catch-error
       )
       nil
      )
      nil
      (progn (with-standard-io-environment (format stream "~S" :error))
	     (force-output stream)
	     (send sys:current-process :Kill)
      )
  )
)

(defun ensure-self-evaluating (thing)
"Returns something that's self evaluating to transmit to a remote host."
  (etypecase thing
    (symbol thing)
    (cons (if (equal (first thing) 'quote)
	      thing
	      (cons (ensure-self-evaluating (first thing))
		    (ensure-self-evaluating (rest  thing))
	      )
	  )
    )
    (string (let ((new (make-string (length thing))))
	         (loop for i below (length thing)
		       do (setf (aref new i)
				(code-char (char-code (aref thing i)))
			  )
		 )
		 new
	    )
    )
    (function (function-name thing))
    (pathname (send thing :string-for-printing))
    ((or number character rational) thing)
  )
)

(defun server-form-reply (stream function args)
  (with-standard-io-environment (format stream "~&***FORM COMING UP***~%"))
  (force-output stream)
  (with-standard-io-environment
    (format stream "~S ~S~%" function args)
  )
  (force-output stream)
)

(defun server-log-print-function (stream &rest args)
  (Server-Form-Reply stream 'fs:*Log-Print-Function* args)
)

(defun invoke-map-over-files-on-host
       (host Remote-Bindings from-window &rest map-over-files-args)
"Performs a map-over-files on a remote-host Host using the window
 From-window on the local machine for any y-or-n-ps that may happen. 
 Otherwise it performs just like Map-Over-Files.
"
  (with-open-stream (stream (net:open-connection-on-medium
			      (net:parse-host host) :Byte-Stream
			      "Generic-Map-over-files"
			      :Stream-Type
			      :Ascii-Translating-Character-Stream
			    )
		    )
    (unwind-protect
	(progn
	  (With-Standard-Io-Environment
	    (format stream "~S~%~S~%~S~%~S~%"
		    (send from-window :Name) (send si:local-host :Name)
		    (ensure-self-evaluating Remote-Bindings)
		    (Ensure-Self-Evaluating map-over-files-args)
	    )
	  )
	  (force-output stream)
	  (loop for line = (With-Standard-Io-Environment (read-line stream))
		until (equal line "***END***")
		do (if (equal line "***FORM COMING UP***")
		       (let ((func (with-standard-io-environment
				     (read stream)
				   )
			     )
			     (form (with-standard-io-environment
				     (read stream)
				   )
			     )
			    )
			    (apply (symbol-value func) from-window form)
		       )
		       (Progn (princ line from-window)
			      (terpri from-window)
		       )
		   )
	  )
	  (let ((result (With-Standard-Io-Environment (read stream))))
	       (case result
		 (:Error (ferror nil "Error on server."))
		 (otherwise result)
	       )
	       (setq sys:.file-aborted-flag. nil)
	       result
	  )
	)
      (if (equal :Abort sys:.file-aborted-flag.)
	  (catch-error (progn (send stream :Tyo 255) (force-output stream)))
	  nil
      )
    )
  )
)

;-------------------------------------------------------------------------------
