;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(TVFONT) -*-

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;; This file contains a collection of new commands that can be added to
;;; the inspector.

(load-tools '(Development-Tool-Consistency-Enhancements))

(require 'Development-Tool-Consistency-Enhancements)
(require 'Window-Debugger-Enhancements)


(defvar *Inspector-Enhancements-to-add* :All
"Can have the value :All, in which case all commands are loaded,
 :Menu in which case the user is prompted, or a list of commands to load.
")

(defparameter *Inspector-Enhancements-Commands*
  '(("Debug Stack Group" :Value (nil ((DBG-Sg-CMD t)) nil)))
"This is a list of the commands which can be added to the Inspector.  The
 structure of this list is as follows :-
 It is made of items.  Each item is a list of the form (Menustring :Value spec)
 the spec is used to determine which commands for which tools is represented
 by the menu item.  The spec is a list of the form
 (nil inspector-commands nil)
 Each element of inspector-commands has the form
 (command-name put-in-frames-menu-p).
"
)


(defun install-Inspector-commands ()
"Installs all of the commands that the user wants.
"
  (select-and-install-commands *Inspector-Enhancements-Commands*
			       *Inspector-Enhancements-to-add*
  )
)


;-------------------------------------------------------------------------------

;;; Define a debug stack group command.  JPR 21 Nov 86.

(DEFCOMMAND DBG-Sg-CMD NIL			
            '(:DESCRIPTION 
             "Debug a stack group."
              :NAMES ("Dbg SG") :KEYS (#\h-S))
            (DECLARE (SPECIAL USER HISTORY = INSPECTORS FRAME)) (SEND USER :CLEAR-SCREEN)
            (FORMAT USER "~&Type or mouse a process or stack group to debug:")
            (MULTIPLE-VALUE-BIND (VALUE PUNT-P) (INSPECT-GET-VALUE-FROM-USER USER HISTORY INSPECTORS)
              (OR PUNT-P (Process-Run-Function "Window Debugger from Inspector" #'eh:debug-a-stack-group VALUE)))
            (SEND FRAME :HANDLE-PROMPT))

;-------------------------------------------------------------------------------


(Provide 'Inspector-Enhancements)

(install-Inspector-commands)


;-------------------------------------------------------------------------------

;;; Mouse sensitive printing in Inspector panes...


;-------------------------------------------------------------------------------

;;; James Rice, Stanford KSL, 1986

;;;  Portions of the code in this file are copyright Texas Instruments,
;;;  and may only be used under the terms and conditions of their
;;;  licensing agreements.


;-------------------------------------------------------------------------------

;;; This file contains a modification to the Inspector to allow the user to
;;; define the way in which things are inspected just like print methods.
;;; This can be done by using a special, which is bound when printing
;;; concisely.  This is :-
;;; *printing-mouse-sensitively*         - true whilst printing concisely
;;;                                        (i.e. will accept item1s)
;;; This can be used within print methods or format directives to switch on
;;; the sending of item1 messages rather then just printing the object.


(defparameter *saved-pp-objects* nil
"A list of objects saved whilst pretty printing.  This is used for itemisation
 of things.
"
)

(defun print-concisely-as (something stream print-as depth)
"Prints something concisely on stream, whose printed representation is
 Print-As.
"
  (ignore something)
  (Print-Item-Concisely print-as stream depth)
)


(defun princ-concisely-as (something stream princ-as depth)
"Princs something concisely on stream, whose printed representation is
 Princ-As.
"
  (ignore something depth)
  (Princ princ-as stream)
)

(defflavor item-saving-stream () (sys:output-stream)
 (:Documentation "A flavor of output stream which saves items when things
 are pretty-printed.")
)

(defun safe-string (x)
  (if (typep x 'character)
      (values (make-array 1 :element-type 'sys:fat-char :initial-value x))
      (string x)
  )
)

(defmethod (item-saving-stream :tyo) (char &rest ignore)
"Coerces char into a character (possibly fontified) and then makes a string
 pp-object of it, saving it in the list of saved pp objects.
"
  (let ((real-char (if (and *fontify-this-region*
			    (not (eql (char-code #\newline) (char-code char)))
		       )
		       (fontify-char char)
		       (if (integerp char) (code-char char) char)
		   )
	)
       )
       (let ((object
	       (sys:make-pp-obj :type 'string
				:length 1
				:object (safe-string real-char)
	       )
	     )
	    )
	    (push object *saved-pp-objects*)
	    object
       )
  )
)

;-------------------------------------------------------------------------------

(defun fontify-string (string font)
"
 Given a normal string and a font it returns a fat string, whose chars are in
 the font Font.
"
  (let ((length (length (the string string))))
       (let ((fat-string (make-array length :element-type 'sys:fat-char)))
	    (loop for i from 0 to (- length 1)
		  for char = (aref string i)
		  do (setf (aref fat-string i)
			   (if (sys:whitespacep char)
			       char
			       (code-char char 0 font)
			   )
		     )
	    )
	    fat-string
       )
  )
)

(defflavor shifting-stream
	   (stream)
	   (si:output-stream)
  :Initable-instance-variables
  (:Documentation "A stream which font-shifts all of its characters if
 necessary.")
)

(defmethod (shifting-stream :tyo) (char &rest args)
"Prints char to the stream, fontifying it if *fontify-this-region* is true."
  (if (and *fontify-this-region*
	   (not (eql (char-code #\newline) (char-code char)))
      )
      (lexpr-send stream :tyo (fontify-char char) args)
      (lexpr-send stream :tyo char args)
  )
)

(defvar *dont-shift-string-streams* nil)

(let ((compiler:compile-encapsulations-flag t))
     (advise sys:pp-objify :around :source-code-debugging nil
       ;;; Checks to see that the object is fontified if we are source code
       ;;; debugging.
       (if (and (typep *standard-output* 'closure)
		(not *Dont-Shift-String-Streams*)
	   )
	   (Let ((*standard-output*
		   (make-instance 'shifting-stream :stream *standard-output*)
		 )
		)
		:Do-It
	   )
	   (let ()
		(declare (special eh::*grinding-debugged-source-code*))
		(if (and (boundp 'eh::*grinding-debugged-source-code*)
			 eh::*grinding-debugged-source-code*
			 (eq (first arglist)
			     (second eh::*grinding-debugged-source-code*)
     ;			(funcall 'eh::numbered-component-code
     ;			     eh::*grinding-debugged-source-code*
     ;			   )
			 )
		    )
		    (let ((*fontify-this-region* t))
			 :Do-It
		    )
		    :Do-It
		)
	   )
       )
     )
)

;-------------------------------------------------------------------------------

(defparameter sys:*dont-break* nil
"When true the pprinter is not allowed to break lines."
)

(defparameter sys:*unbreakable-pp-objects*
	      '(sys:complex-atom)
"A list of all of the PP-object types which are not allowed to break lines."
)

(defparameter sys:*breakable-pp-objects*
	      '(sys:complex-breakable)
"A list of all of the PP-object types which are allowed to break lines."
)

sys:
(defun sys:basically-grind (object break-always many-on-a-line indent-style
			       charpos)
  (declare (fixnum charpos))
  "Prints out an object constructed by PP-Objify."
  (cond ((member (pp-obj-type object) *breakable-pp-objects*)
	 (let ((*dont-break* nil))
	      (cond ((or break-always
			 (and (null many-on-a-line)
			      (> (+ (pp-obj-length object) charpos)
				 pp-line-length)))
		     (break-always-grind (pp-obj-object object)
					 indent-style charpos))
		    (t (break-sometimes-grind (pp-obj-object object)
					      indent-style charpos)))
	 )
	)
	(*dont-break*
	 (break-never-grind (pp-obj-object object) indent-style charpos)
	)
	((member (pp-obj-type object) *unbreakable-pp-objects*)
	 (let ((*dont-break* t))
	      (break-never-grind (pp-obj-object object) indent-style charpos)
	 )
	)
	((or break-always
	     (and (null many-on-a-line)
		  (> (+ (pp-obj-length object) charpos) pp-line-length)))
	 (break-always-grind (pp-obj-object object)
			     indent-style charpos))
	(t (break-sometimes-grind (pp-obj-object object)
				  indent-style charpos))))

sys:
(defun sys:break-never-grind (object indent-style charpos)
  (declare (fixnum charpos))
  "Prints as many components as possible on each line."
    (do* ((components object (cdr components))
	  (early-indent-pos (indentation object 1 charpos))
	  (late-indent-pos (indentation object indent-style charpos))
	  (indent-pos early-indent-pos)
	  break-occured)
	 ((null components) break-occured)
      (declare (fixnum indent-pos))
      (cond ((eq (pp-obj-type (car components)) 'space)
	     (setq indent-pos late-indent-pos)
	     (write-char #\space)
	     (setq charpos (1+ charpos)))
	    (t (when (eq break-occured 'partial)
		 (tab-over indent-pos))
	       (setf break-occured (master-grind (car components) charpos))
	       (setq charpos (+ charpos (pp-obj-length (car components))))))))


(defparameter *allow-breaks-in-unbreakables* nil
"If true then the user is allowing line breaks whilst pprinting atoms."
)

(defparameter *inside-atom* nil
"True when inside an atomic unbreakable(?) print object."
)

(let ((compiler:compile-encapsulations-flag t))
     (advise sys:pp-objify-atom :around :record-saved-objects nil
       ;;; If the generation of the item for this atom caused any saved objects
       ;;; to be generated then a complex item is generated for those items.
       ;;; This is returned instead of the atomic item for the atom.
       (let ((*saved-pp-objects* nil)
	     (*inside-atom* t)
	     (type (if *allow-breaks-in-unbreakables*
		       'sys:complex-breakable
		       'sys:complex-atom
		   )
	     )
	    )
	    (declare (special *saved-pp-objects* *inside-atom*))
	    (let ((result :Do-it))
		 (if *saved-pp-objects*
		     (let ((save #'format:format-ctl-ascii))
			  (declare (special save))
			  (setq *saved-pp-objects* nil)
			  (let ((*standard-output*
				  (make-instance 'item-saving-stream)
				)
			       )
			       :Do-It
			  )
			  (sys:make-pp-obj :type type;'sys:complex-atom;type
					   :length (length *saved-pp-objects*)
					   :object (reverse *saved-pp-objects*)
					   :location (second arglist)
			  )
		     )
		     result
		 )
	    )
       )
     )
)


(defun pp-objify-atom-different-name (object location name)
"Makes a PP-Obj for an atom with a different printed representation than the
 original object.
"
  (declare (special sys:pprint-string))
  (ignore object)
  (let ((start (length sys:pprint-string)))
       (print-object name 0 *standard-output* '(:string-out))
       (let ((result
	       (sys:make-pp-obj :length   (- (length sys:pprint-string) start)
				:object   start
				:location location
	       )
	     )
	    )
	    result
       )
  )
)

format:
(defun string-of-pads (count char)
  (with-output-to-string (*standard-output*)
    (format-ctl-repeat-char count char)
    *standard-output*
  )
)

format:
(defun string-of-justify (width size &optional (char #\SPACE))
  (with-output-to-string (*standard-output*)
    (format-ctl-justify width size char)
    *standard-output*
  )
)


(defun process-args (stream args font)
"Given a set of args for the stream, which are an item, and a font, send the
 item to the stream, if provided, in the defined font, if provided.  Returns
 the agrs/font as appropriate.
"
  (if font
      (if stream
	  (let ((old-font (send stream :send-if-handles :current-font)))
	       (unwind-protect
		   (progn (send stream :send-if-handles :set-current-font font)
			  (if (send stream :operation-handled-p :item1)
			      (lexpr-send stream args)
			      nil
			  )
		   )
		 (send stream :send-if-handles :set-current-font old-font)
		 (list :font font args)
	       )
	  )
	  (list :font font args)
      )
      (progn (if (send stream :operation-handled-p :item1)
		 (lexpr-send stream args)
		 nil
	     )
	     args
      )
  )
)

(defmethod (mouse-sensitive-text-scroll-window-without-click :compound)
	   (some-items)
  (loop for item in some-items do
	(if (consp item)
	    (lexpr-send self item)
	    (format self "~A" item))))


format:
(defun tv:process-params (unencapsulated arg params &optional prin1p)
  (let ((edge (car params))
	(period (cadr params))
	(min (caddr params))
	(padchar (cadddr params))
	(result nil))
    (cond
      ((null padchar) (setq padchar #\SPACE))
      ((not (numberp padchar)) (setq padchar (character padchar))))
    (if atsign-flag
	nil
	(push arg result))
    (cond
      ((not (null edge))
       (let ((width
	      (funcall
	       (cond
		 (prin1p (function flatsize))
		 ((stringp unencapsulated) (function length))
		 (t (function flatc)))
	       unencapsulated)))
	 (cond
	   ((not (null min))
	    (push (string-of-pads min padchar) result)
	    (setq width (+ width min))))
	 (cond
	   (period
	    (push (string-of-pads
		    (* (ceiling (- edge width) period) period)
		    padchar) result))
	   (t (push (string-of-justify edge width padchar) result))))))
    (if (not atsign-flag)
	nil
	(push arg result))
    (list :compound (reverse result))))

(defun actually-itemize-for-list
       (something stream print-as princ-as font)
  (let ((*print-escape* print-as)
	(*standard-output* stream)
	(*fontify-this-region* (if font font *fontify-this-region*))
       )
       (declare (special *print-escape*))
       (if (or (eq something print-as)
	       (eq something princ-as)
	   )
	   (let ((result
		   (sys:pp-objify something (cons something something))
		 )
		)
		(push result *saved-pp-objects*)
		result
	   )
	   (let ((result
		   (pp-objify-atom-different-name
		     something (cons something something)
		     (or print-as princ-as)
		   )
		 )
		)
		(push result *saved-pp-objects*)
		result
	   )
       )
  )
)

format:
(defun tv:itemize-for-list
       (something stream print-as princ-as font params &optional prin1p)
"Itemises something to stream in font either princed or printed, whilst
 grinding out a list as the top level object.
"
  (let ((edge (car params))
	(period (cadr params))
	(min (caddr params))
	(padchar (cadddr params))
       )
    (cond
      ((null padchar) (setq padchar #\SPACE))
      ((not (numberp padchar)) (setq padchar (character padchar))))
    (if atsign-flag
	nil
	(tv:actually-itemize-for-list something stream print-as princ-as font))
    (cond
      ((not (null edge))
       (let ((width
	      (funcall
	       (cond
		 (prin1p (function flatsize))
		 ((stringp something) (function length))
		 (t (function flatc)))
	       something)))
	 (cond
	   ((not (null min))
	    (format-ctl-repeat-char min padchar)
	    (setq width (+ width min))))
	 (cond
	   (period
	    (format-ctl-repeat-char
	      (* (ceiling (- edge width) period) period)
	      padchar))
	   (t (format-ctl-justify edge width padchar))))))
    (if (not atsign-flag)
	nil
	(tv:actually-itemize-for-list something stream print-as princ-as font))

  )
)


(defun itemize-element
       (something stream depth &Key print-as princ-as font params)
"Given something to print it displays it as a mouse sensitive item, which looks
 like Print-as or Princ-as in stream. Depth is the depth of mouse sensitive
 items displayed.  If neither Princ-as nor Print-as are supplied then it looks
 like Something.  If stream is nil it returns the item args that would be sent
 to a stream.
"
  (declare (special sys:pprint-string))
  (if (and (boundp 'grind-into-list-string)
	   grind-into-list-string
	   (boundp 'sys:pprint-string)
	   sys:pprint-string
      )
      (itemize-for-list something stream print-as princ-as font params)
      (let ((args (process-params
		    something
		    (if print-as
			(list :Item1 something
			      :Value #'print-concisely-as
			      print-as (+ 1 depth)
			)
			(list :Item1 something
			      :Value #'princ-concisely-as
			      princ-as (+ 1 depth)
			)
		    )
		    params (not princ-as)
		  )
	    )
	   )
	   (process-args stream args font)
      )
  )
)


(defvar *printing-mouse-sensitively* nil
"True when the things that are being printed should be turned into mouse
 sensitive items if you know how to.
"
)

;-------------------------------------------------------------------------------

(let ((compiler:compile-encapsulations-flag t))
     (advise Print-Item-Concisely :Around :Mouse-Sensitivity-Addition nil
       ;;; Bind *printing-mouse-sensitively* because this is a good place to
       ;;; have mouse sensitive objects.
       (let ((*printing-mouse-sensitively* t))
	    (declare (special *printing-mouse-sensitively*))
	    :Do-It
       )
     )
)

;-------------------------------------------------------------------------------

(let ((compiler:compile-encapsulations-flag t))
     (advise Grind-Top-Level :Around :Mouse-Sensitivity-Addition nil
       ;;; Bind *printing-mouse-sensitively* because this is a good place to
       ;;; have mouse sensitive objects as long as you are grinding into an
       ;;; inspect pane.
       (if (and (seventh arglist)
		(or (equal (seventh arglist)  'grind-into-list-make-item)
		    (equal (seventh arglist) #'grind-into-list-make-item)
		)
		(not grind-into-list-string)
	   )
	   (let ((*printing-mouse-sensitively* t))
		(declare (special *printing-mouse-sensitively*))
		:Do-It
	   )
	   :Do-It
       )
     )
)

;-------------------------------------------------------------------------------


(defun format-list (collection slashify)
"Is passed a collection and a flag which denotes slashification.  It returns a
 list of pairs, whose first elements are the elements of the collection and
 the second elements are all the slashify flag.
"
  (loop for i in collection
	collect (list i slashify)
  )
)

;-------------------------------------------------------------------------------

(defvar *inspect-details* (boundp '*printing-mouse-sensitively*)
"A global which is true if the user wants the contents of structures
 to be mouse sensitive in the inspector.
"
)

;;; Define some format directives for the Mouse system.


;;; Formats something using two args; the first is the thing to format
;;; the second it slashify.

(format:defformat format: (:Multi-Arg) (args params)
"Formats something, taking two arguments.  The first is the thing to format and
 the second is the slashification.
 Thus (format stream 'A structure :- ~' a-struct slashify) will print out
 a-struct slashifiedly if slashify is non-nil, otherwise unslashifiedly.
 If the thing being printed is being printed in an inspector pane then the
 thing to be printed is printed out mouse sensitively as long as the
 *inspect-details* flag is true.
"
   (let ((thing-to-print (first args))
	(slashify (second args))
       )
       (if (and *inspect-details*
		(boundp '*printing-mouse-sensitively*)
		*printing-mouse-sensitively*
	   )
	   (itemize-element thing-to-print *standard-output* 0
			    (if slashify :Print-as :Princ-as)
			    thing-to-print :params params
	   )
	   (format:format-ctl-ascii (first args) params (second args))
       )
       (rest (rest args))
  )
)

(defprop format: 2 format:number-of-arguments)

;;; Format something using one arg, which is a list.  The first is the thing
;;; to print and the second is slashify.  If there is a third then this is used
;;; as the thing to print, with the first being the internal representation
;;; for the inspector.

(format:defformat format: (:One-Arg) (arg params)
"Formats something, taking one argument, which is a two/three/four list.
 This list is as follows :-
   The first is the thing to format
   The second is the slashification
   The third, if provided is the printed representation of the first (i.e the
       thing to print).
   The font in which to print it.
 Thus (format stream 'A structure :- ~' (list a-struct nil '#<struct>') will
 print out a-struct as the string '#<struct>', which will be princed.
 If the thing being printed is being printed in an inspector pane then the
 thing to be printed is printed out mouse sensitively as long as the
 *inspect-details* flag is true.
"
  (let ((thing-to-print (if (consp arg) (first arg) arg))
	(slashify (if (consp arg) (second arg) t))
	(display-as (if (consp arg)
			(if (< (length arg) 3)
			    (first arg)
			    (third arg)
			)
		    )
        )
	(font (if (consp arg) (fourth arg) nil))
       )
       (if (and *inspect-details*
		(boundp '*printing-mouse-sensitively*)
		*printing-mouse-sensitively*
		(> (length (format nil "~A" display-as)) 0)
	   )
	   (itemize-element thing-to-print *standard-output* 0
			    (if slashify :Print-as :Princ-as)
			    display-as :font font :params params
	   )
	   (format:format-ctl-ascii display-as params slashify)
       )
  )
)


(format:defformat format: (:One-Arg) (arg params)
"Formats something, taking one argument, which is a two/three list.  This list
 is as follows :-
   The first is the thing to format
   The second is the slashification
   The third, if provided is the printed representation of the first (i.e the
       thing to print).
 Thus (format stream 'A structure :- ~' (list a-struct nil '#<struct>') will
 print out a-struct as the string '#<struct>', which will be princed.
 This format directive does not provide mouse sensitivity.  The reason for it
 is that it is compatible with ~, which does.  This means that you can make
 alternate elements in lists, for instance, mouse sensitive, whilst still
 formatting the list as for ~.
"
  (let ((slashify (if (consp arg) (second arg) t))
	(display-as (if (consp arg)
			(if (< (length arg) 3)
			    (first arg)
			    (third arg)
			)
		    )
        )
       )
       (format:format-ctl-ascii display-as params slashify)
  )
)


(defun print-out-with-separator-1
       (list-to-print slashify whole-thing separator-string left-bracket
	right-bracket alternate-p truncated
       )
  (if alternate-p
      (progn
	(format *standard-output* "~~~A~"
		(list whole-thing nil left-bracket)
		(list (first list-to-print) slashify)
		separator-string
		(list (second list-to-print) slashify)
	)
	(loop for (a b) on (format-list (rest (rest list-to-print)) slashify)
	      by #'cddr
	      do (format *standard-output* "~A~~A~"
			 separator-string a separator-string b
		 )
	)
	(format *standard-output* "~A~A~"
		(if truncated separator-string "")
		(if truncated "..." "")
		(list whole-thing nil right-bracket)
	)
      )
      (progn
	(format *standard-output* "~~"
		(list whole-thing nil left-bracket)
		(list (first list-to-print) slashify)
	)
	(loop for a in (format-list (rest list-to-print) slashify)
	      do (format *standard-output* "~A~" separator-string a)
	)
	(format *standard-output* "~A~A~"
		(if truncated separator-string "")
		(if truncated "..." "")
		(list whole-thing nil right-bracket)
	)
      )
  )
)


(defun print-out-with-separator
  (list &key (slashify *print-escape*)
             (whole-thing list)
	     (separator " ")
	     (left-bracket "")
	     (right-bracket "")
	     (depth 0)
	     (alternate-p nil)
  )
"
 Formats a list of things, taking three arguments.  The first argument is the
 list of things to print out.  The second is the slashification to use for all
 of them and the third is something which can be converted into a string to use
 as a separator.  The list of things is printed out with the separator princed
 between them, using the ~ format directive.  This means that the elements
 will come out mouse sensitive if they are printed out by in an inspector pane
 then the thing to be printed is printed out mouse sensitively as long as the
 *inspect-details* flag is true.
 Thus (print-out-with-separator '(a b c) slashify \", \" nil nil 0)
 will print out
 A list :- foo:a, foo:b, foo:c
 if slashify is true and
 A list :- a, b, c
 if it is not.
 If separator is nil then a space is printed instead.
"
  (declare (unspecial list slashify whole-thing separator brackets)
	   (optimize (safety 0) (speed 3))
  )
  (let ((too-deep (and depth *print-level* (> depth *print-level*))))
       (if too-deep
	    (format *standard-output* "~" (list whole-thing nil "#"))
	    (let ((truncated (and *print-length*
				  (consp (nthcdr *print-length* list))
			     )
		  )
		  (separator-string
		    (if (equal nil separator) " " (string separator))
		  )
		 )
		 (let ((list-to-print
			 (if truncated (firstn *print-length* list) list)
		       )
		      )
		      (if (equal nil list-to-print)
			  nil
			  (print-out-with-separator-1 list-to-print slashify
			    whole-thing separator-string left-bracket
			    right-bracket alternate-p truncated
			  )
		      )
		 )
	    )
	)
  )
)


;;; Formats a list of things with spaces between them.  It takes two args.
;;; The first is a list of items to print.  The second is slashify.

(format:defformat format:	 (:Multi-Arg) (args params)
"Formats a list of things, taking two arguments.  The first argument is the
 list of things to print out.  The second is the slashification to use for all
 of them.  The list of things is printed out with spaces between them, using
 the ~ format directive.  This means that the elements will come out mouse
 sensitive if they are printed out in an inspector pane then the
 thing to be printed is printed out mouse sensitively as long as the
 *inspect-details* flag is true.
 Thus (format stream 'A list :- ~	' '(a b c) slashify) will print out
 A list :- foo:a foo:b foo:c
 if slashify is true and
 A list :- a b c
 if it is not.
"
  (ignore params)
  (print-out-with-separator (first args)
    :Slashify (second args)
    :Whole-Thing (first args)
    :Separator " "
    :Depth 1
  )
  (rest (rest args))
)

(defprop format:	 2 format:number-of-arguments)


(format:defformat format:  (:Multi-Arg) (args params)
"Formats a list of things, taking two arguments.  The first argument is the
 list of things to print out.  The second is the slashification to use for all
 of them.  The alternate elements in the list of things are printed out using
 the ~ format directive.  This means that these elements will come out mouse
 sensitive if they are printed out in an inspector pane then the thing to be
 printed is printed out mouse sensitively as long as the *inspect-details*
 flag is true.
 Thus (format stream 'A list :- ~	' '(a b c) slashify) will print out
 A list :- foo:a foo:b foo:c
 if slashify is true and
 A list :- a b c
 if it is not.
"
  (ignore params)
  (print-out-with-separator (first args)
    :Slashify (second args)
    :Whole-Thing (first args)
    :Separator " "
    :Depth 1
    :Alternate-p t
  )
  (rest (rest args))
)

(defprop format:  2 format:number-of-arguments)

;;; Formats a list of things with spaces between them.  It takes two args.
;;; The first is a list of items to print.  The second is slashify.

(format:defformat format:
 (:Multi-Arg) (args params)
" Formats a list of things, taking seven arguments.
 i)   the list of things to print out.
 ii)  the slashification to use for all of them.
 iii) The whole structure of which i) is a part.
 iv)  something which can be converted into a string to use as a separator.
 v)   a string which is used as the left bracket for i).  This could be,
      for instance, \"[\".
 vi)  a string which is used as the right bracket for i).  This could be,
      for instance, \"]\".
 vii)  the current print depth level.

 The list of things is printed out with the separator princed between them,
 using the ~ format directive and with the brackets princed at each end. 
 This means that the elements will come out mouse sensitive if they are
 printed out in an inspector pane then the thing to be printed is printed
 out mouse sensitively as long as the *inspect-details* flag is true.
 Thus
 (format stream \"A list :- ~
\" '(a b c) slashify fred \", \" \"{\" \"}\" 0)
 will print out as
 A list :- {foo:a, foo:b, foo:c}
 if slashify is true and
 A list :- {a, b, c}
 if it is not.
"
  (ignore params)
  (print-out-with-separator (first args)
    :Slashify      (second args)
    :Whole-Thing   (third  args)
    :Separator     (fourth args)
    :Left-Bracket  (fifth  args)
    :Right-Bracket (sixth  args)
    :Depth   (+ 1 (seventh args))
  )
  (nthcdr 7 args)
)

(defprop format:
 7 format:number-of-arguments)

;-------------------------------------------------------------------------------

;;; Extend the definition for the printing of functions.

(let ((compiler:compile-encapsulations-flag t))
     (advise print-fef-instruction :around :show-dbis nil
       ;;; Spots non instruction type items and prints them with the normal
       ;;; inspect printer.
       (if (consp (first arglist))
	   (inspect-printer (first arglist) nil (third arglist) nil)
	   :Do-It
       )
     )
)

(defvar *show-dbis-for-fefs-in-inspector* t
"When true the name and DBIS for functions is printed in the inspector when
 you inspect a fef.
"
)


(let ((compiler:compile-encapsulations-flag t))
     (advise fef-display-list :around :show-dbis nil
       ;;; Print the function name and DBIS for the FEF.  Do this by putting an
       ;;; extra item onto the list of items.  This item is of a different
       ;;; format from the others but wil be handled by the advice above.
       (let ((results (multiple-value-list :do-it)))
	    (let ((struct (catch-error
			    (sys:get-debug-info-struct (first arglist))
			    nil
			  )
		  )
		  (name (catch-error (function-name (first arglist)) nil))
		 )
		 (if (and *show-dbis-for-fefs-in-inspector* name struct)
		     (progn (setf (first results)
				  (cons (list "Function name: "
					      `(:item1 named-structure-p ,name)
					      ", Debug info struct: "
					      `(:item1 named-structure-p
						       ,struct)
					)
					;;; Remove TI's DBIS thing.
					(rest (first results))
				  )
			    )
			    (values-list results)
		     )
		     (values-list results)
		 )
	    )
       )
     )
)

;-------------------------------------------------------------------------------

;;; By JPR.
(defvar *default-inspect-fonts* '(fonts:cptfont fonts:hl10b fonts:tr10bi)
"A list of fonts to use if the inspector tries to print out a fontified object
 and finds that the font in the font map is the same as the base font.
"
)

(defun set-font-map-if-you-must (sheet)
"Sets the font map of the sheet to something from *default-inspect-fonts*."
  (let ((zero-font (font-name (aref (send sheet :font-map) 0))))
       (send sheet :set-font-map
	     (cons (aref (send sheet :font-map) 0)
		   (remove-if #'(lambda (element)
				  (or (search (symbol-name element)
					      (symbol-name zero-font)
				      )
				      (search (symbol-name zero-font)
					      (symbol-name element)
				      )
				  )
				)
			        *default-inspect-fonts*
		   )
	     )
       )
  )
)

(defmethod (inspect-window :around :tyo)
  (continuation mapping-table original-args char &rest args)
"Prints any fontified chars specially, otherwise just prints as normal."
  (if (> (char-font char) 0)
      (let ((old-font (send self :current-font)))
           (unwind-protect
	       (progn (if (equal (aref (send self :font-map) (char-font char))
				 (aref (send self :font-map) 0)
		          )
			  (set-font-map-if-you-must self)
			  nil
		      )
		      (send self :set-current-font (char-font char))
		      (lexpr-funcall-with-mapping-table
			continuation mapping-table :tyo char args
		      )
	       )
	     (send self :set-current-font old-font)
	   )
      )
      (lexpr-funcall-with-mapping-table
	continuation mapping-table original-args
      )
  )
)

(defmethod (inspect-window :string-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i
	     from start
	     to (if end (- end 1) (- (array-active-length string) 1))
	     do (send self :tyo (aref tem i))
       )
  )
)

(defmethod (inspect-window :line-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i from start to (if end end (- (array-active-length string) 1))
	     do (send self :tyo (aref tem i))
       )
       (send self :tyo #\newline)
  )
)

;-------------------------------------------------------------------------------


(defmethod (inspect-history-window :around :tyo)
  (continuation mapping-table original-args char &rest args)
"Prints any fontified chars specially, otherwise just prints as normal."
  (if (> (char-font char) 0)
      (let ((old-font (send self :current-font)))
           (unwind-protect
	       (progn (if (equal (aref (send self :font-map) (char-font char))
				 (aref (send self :font-map) 0)
		          )
			  (set-font-map-if-you-must self)
			  nil
		      )
		      (send self :set-current-font (char-font char))
		      (lexpr-funcall-with-mapping-table
			continuation mapping-table :tyo char args
		      )
	       )
	     (send self :set-current-font old-font)
	   )
      )
      (lexpr-funcall-with-mapping-table
	continuation mapping-table original-args
      )
  )
)

(defmethod (inspect-history-window :string-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i
	     from start
	     to (if end (- end 1) (- (array-active-length string) 1))
	     do (send self :tyo (aref tem i))
       )
  )
)

(defmethod (inspect-history-window :line-out)
	   (string &optional (start 0) (end nil))
"Prints out the string with any fonts in it."
  (let ((tem (if (typep string 'array) string (string string))))
       (loop for i from start to (if end end (- (array-active-length string) 1))
	     do (send self :tyo (aref tem i))
       )
       (send self :tyo #\newline)
  )
)

;-------------------------------------------------------------------------------

;;; The following code written by JPR to give better inspecting of locatives.
;;; It tries to find the object that contains the locative and give a reasonable
;;; display of it.

(defun array-leader-matching-slot (loc array)
"Try to find the locative Loc in the array leader of Array.  Returns either the
 array leader index or nil.
"
  (let ((leader-length (array-leader-length array)))
       (if leader-length
	   (loop for index from 0 to (- leader-length 1)
		 when (eq loc (locf (array-leader array index)))
		 return index
		 finally (return nil)
	   )
	   nil
       )
  )
)

(defun array-matching-slot (loc array dimensions indices)
"Try to find the locative Loc in the array Array.  Returns either the list of
 indices or nil.  Dimensions is the reversed list of Array-Dimensions.  Indices
 is an accumulating parameter of indices, initially nil.
"
  (if (rest dimensions)
      (loop for index from 0 to (- (first dimensions) 1) do
	    (let ((result (array-matching-slot
			    loc array (rest dimensions)
			    (cons index indices)
			  )
		  )
		 )
	         (if result (return (cons index result)) nil)
	    )
	    finally (return nil)
      )
      (loop for i from 0 to (- (first dimensions) 1) do
	    (if (eq loc (apply #'aloc array i indices))
		(return (list i))
		nil
	    )
	    finally (return nil)
      )
  )
)

(defun array-slot-matching (loc array)
"Try to find the locative Loc in the array Array.  Returns either the list
 (:Leader index) if Loc if in the array leader or (:Body (indices)) if it is
 in the body of the array.
"
  (let ((foundp (array-leader-matching-slot loc array)))
       (if foundp
	   (list :leader foundp)
	   (list :body
		 (reverse (array-matching-slot
			    loc array (reverse (array-dimensions array)) nil
			  )
		 )
	   )
       )
  )
)

(defmethod (basic-inspect :object-locative-array) (loc array)
"Make Items for a locative which points to an array element."
  (let ((the-items (multiple-value-list
		     (send self :object-array array)
		   )
	)
	(title (array-slot-matching loc array))
       )
       (let ((new-item
	       (if (equal (first title) :leader)
		   (list (format nil "Locative to element ~S in the array ~
                                      leader of " (second title)
			 )
			 `(:item1 named-structure-p ,array)
			 "."
		   )
		   (list (format nil "Locative to element [~a~{, ~a~}] of "
				 (first (second title)) (rest (second title))
			 )
			 `(:item1 named-structure-p ,array)
			 "."
		   )
	       )
	     )
	    )
	    (apply #'values nil
		   (list (first (second the-items))
			 (second (second the-items))
			 (cons new-item (third (second the-items)))
		   )
		   (rest (rest the-items))
	    )
       )
  )
)

(defun instance-slot-matching (loc instance)
"Try to find the locative Loc in the instance Instance.  Returns either the name
 of the IV which matches Loc or Nil.
"
  (let ((slots (sys:flavor-all-instance-variables
		 (get (type-of instance) 'sys:flavor)
	       )
	)
       )
       (if slots
	   (loop for slot in slots
		 when (eq loc (locate-in-instance instance slot))
		 return slot
		 finally (return nil)
	   )
	   nil
       )
  )
)


(defmethod (basic-inspect :object-locative-instance) (loc instance)
"Make Items for a locative which points to a Instance's IV."
  (let ((the-items (multiple-value-list
		     (send self :object-instance instance)
		   )
	)
	(title (instance-slot-matching loc instance))
       )
       (let ((new-item (list (format nil "Locative to the ~S slot of " title)
			     `(:item1 named-structure-p ,instance)
			     "."
		       )
	     )
	    )
	    (values-list
	      (cons (cons new-item (first the-items)) (rest the-items))
	    )
       )
  )
)

(defmethod (basic-inspect :object-locative-standard-object) (loc instance)
"Make Items for a locative which points to a Instance's IV."
  (let ((the-items (multiple-value-list
		     (send self :object-clos-instance instance)
		   )
	)
	(title (instance-slot-matching loc instance))
       )
       (let ((new-item (list (format nil "Locative to the ~S slot of " title)
			     `(:item1 named-structure-p ,instance)
			     "."
		       )
	     )
	    )
	    (values-list
	      (cons (cons new-item (first the-items)) (rest the-items))
	    )
       )
  )
)

(defun slot-matching (loc structure)
"Try to find the locative Loc in the defstruct instance Structure.  Returns
 either the name of the Slot which matches Loc or Nil.
"
  (let ((slots (fourth (get (named-structure-p structure)
			    'sys:defstruct-description
		       )
	       )
	)
       )
       (if slots
	   (loop for slot in slots
		 when (eq loc (eval `(locf (,(seventh slot) ,structure))))
		 return (first slot)
		 finally (return nil)
	   )
	   nil
       )
  )
)

(defmethod (basic-inspect :object-locative-named-structure) (loc structure)
"Make Items for a locative which points to a Defstruct's Slot."
  (let ((the-items (multiple-value-list
		     (send self :object-named-structure structure)
		   )
	)
	(title (slot-matching loc structure))
       )
       (let ((new-item (list (format nil "Locative to the ~S slot of " title)
			     `(:item1 named-structure-p ,structure)
			     "."
		       )
	     )
	    )
	    (if (and (arrayp structure) (array-has-leader-p structure))
		(apply #'values nil
		       (list (first (second the-items))
			     (second (second the-items))
			     (cons new-item (third (second the-items)))
		       )
		       (rest (rest the-items))
		)
	        (values-list
		  (cons (cons new-item (first the-items)) (rest the-items))
		)
	    )
       )
  )
)

(defmethod (basic-inspect :object-locative-symbol) (loc symbol)
"Make Items for a locative which points to some cell of a symbol."
  (let ((the-items (send self :object-symbol symbol))
	(title (cond ((eq loc (value-cell-location    symbol)) "Value"
		     )
		     ((eq loc (function-cell-location symbol)) "Function"
		     )
		     ((eq loc (property-cell-location symbol)) "Property"
		     )
		     (t "Some")
	       )
	)
       )
       (cons (list "Locative to "
		  `(:item1 named-structure-p ,symbol)
		   (format nil "'s ~A Cell." title)
	     )
	     the-items
       )
  )
)

(defun headered-inspect-list-printer (item arg stream item-no)
"Like inspect-list-printer only it prints out a header if it finds one.
 Headers are always lists with the first element = :Header.  The second it the
 item to inspect normally.
"
  (declare (:self-flavor basic-inspect))
  (if (and (consp item)
	   (equal (first item) :Header)
      )
      (inspect-printer (second item) arg stream item-no)
      (progn (setf (aref displayed-items (- item-no top-item)) (third item))
	     (send stream :string-out (second item))
      )
  )
)

(defmethod (basic-inspect :object-list-with-header) (header list)
"Generates Items for List with a header as well."
  (let ((the-items (multiple-value-list (send self :object-list list))))
       (values (cons (list :Header header) (first the-items))
	       :list-structure 'headered-inspect-list-printer
       )
  )
)

(defmethod (basic-inspect :object-locative-cons) (loc cons)
"Generates Items for a list, which has Cons as a locative to it."
  (ignore loc)
  (send self :object-list-with-header (list "Locative to the list:") cons)
)

(defun fail-locative ()
"Called when some locative is unprintable for some reason."
  (beep)
  (format t "Sorry, there appears to be something wrong with the ~
             contents of this location!~%"
  )
)


;;; This is a redefinition of the TI version.  TI's version is given below,
;;; commented out.

;(defmethod (basic-inspect :object-locative) (loc)
;  (send self :object-list (list (car loc))))


(defparameter *type-to-locative-method-mappings*
	      '((symbol :object-locative-symbol)
		(named-structure :object-locative-named-structure)
		(clos:standard-object :object-locative-standard-object)
		(instance :object-locative-instance)
		(cons :object-locative-cons)
		(array :object-locative-array)
		(t :object-locative-generic)
	       )
"A list of mappings from to type to method.  If a locative points to a structure
 whose type is the first of one of the elements then it invokes that method.
"
)

(defmethod (basic-inspect :object-locative-generic) (loc structure)
"Tries to inspect Loc in a simplistic manner, because it doesn't known what
 better to do.
"
  (ignore structure)
  (if (%p-contents-safe-p loc)
      (send self :object-list (list (first loc)))
      (fail-locative)
  )
)


(defmethod (basic-inspect :object-locative) (loc)
"Generates items for the locative Loc as best it can.  If Loc is a pointer to
 a slot in a defstruct, for instance, it will say which slot it is and inspect
 the defstruct.
"
  (multiple-value-bind (structure error-p)
      (catch-error (sys:find-structure-header loc) nil)
    (if (or error-p (not structure))
	(send self :object-locative-generic loc structure)
	(loop for (type method) in *type-to-locative-method-mappings*
	      when (typep structure type)
	      return (send self method loc structure)
        )
    )
  )
)

;-------------------------------------------------------------------------------

;;; Patches
;;; The following are all modified versions of TI source code unless otherwise
;;; stated.


;;; Variable added by JPR.
sys:
(defvar sys:*top-level-p* t
"True for top level printing.  If false then a new print circle hash table will
 not be allocated unless necessary.
"
)

sys:
(defmacro sys:print-circle (&body body)
  `(if (not *print-circle*) (progn
                              . ,body)
       ;;; Modded here by JPR to check for non top level printing.
       (if (or *top-level-p*
	       (not (boundp 'print-hash-table))
	       (not print-hash-table))
	   (let ((print-label-number 0)
		 (print-hash-table NIL))
	     (unwind-protect (progn
			       (setq print-hash-table (get-print-hash-table))
			       (clrhash print-hash-table)
			       (print-record-occurrences object)
			       . ,body)
	       (when print-hash-table
		 (setq reusable-print-hash-table print-hash-table))))
	   ;;; This bit added by JPR.
	   (progn ,@body))))

;;; The following functions only included so as to recompile them in the context
;;; of the macro above.

#-CLOS
sys:
(defun sys:print (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed, with a Return before and a Space after."
  (setq stream (decode-print-arg stream))
  (funcall stream :tyo (pttbl-newline *readtable*))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprint object stream)
	(progn 
	  (print-circle (print-object object 0 stream))
	  (funcall stream :tyo (pttbl-space *readtable*)))))
  object)

#+CLOS
sys:
(defun sys:print (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed, with a Return before and a Space after."
  (setq stream (decode-print-arg stream))
  (funcall stream :tyo (pttbl-newline *readtable*))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprint object stream)
	;(progn
	(let ((*prindepth* 0))
	  (print-circle (ticlos:print-object object stream))
	  (funcall stream :tyo (pttbl-space *readtable*)))))
  object)

#-CLOS
sys:
(defun sys:prin1 (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprin1 object stream)
	(print-circle (print-object object 0 stream ))))
  object) 

#+CLOS
sys:
(defun sys:prin1 (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprin1 object stream)
	(let ((*prindepth* 0))
	      (print-circle (ticlos:print-object object stream )))))
  object)

#-CLOS
sys:
(defun sys:write (object &key &optional (stream *standard-output*) ((:escape *print-escape*) *print-escape*)
  ((:radix *print-radix*) *print-radix*) ((:base *print-base*) *print-base*)
  ((:circle *print-circle*) *print-circle*) ((:pretty *print-pretty*) *print-pretty*)
  ((:level *print-level*) *print-level*) ((:length *print-length*) *print-length*)
  ((:case *print-case*) *print-case*) ((:gensym *print-gensym*) *print-gensym*)
  ((:array *print-array*) *print-array*))
  "Print OBJECT on STREAM.  Keyword args control parameters affecting printing.
The argument ESCAPE specifies the value for the flag *PRINT-ESCAPE*, and so on.
For any flags not specified by keyword arguments, the current special binding is used."
  (let ((character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(let ((*standard-output* (decode-print-arg stream)))
	  (output-pretty-object object))
	(print-circle (print-object object 0 (decode-print-arg stream))))
  object))

#+CLOS
sys:
(defun sys:write (object &key &optional (stream *standard-output*) ((:escape *print-escape*) *print-escape*)
  ((:radix *print-radix*) *print-radix*) ((:base *print-base*) *print-base*)
  ((:circle *print-circle*) *print-circle*) ((:pretty *print-pretty*) *print-pretty*)
  ((:level *print-level*) *print-level*) ((:length *print-length*) *print-length*)
  ((:case *print-case*) *print-case*) ((:gensym *print-gensym*) *print-gensym*)
  ((:array *print-array*) *print-array*))
  "Print OBJECT on STREAM.  Keyword args control parameters affecting printing.
The argument ESCAPE specifies the value for the flag *PRINT-ESCAPE*, and so on.
For any flags not specified by keyword arguments, the current special binding is used."
  (let ((character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(let ((*standard-output* (decode-print-arg stream)))
	  (print-circle (output-pretty-object object)))
	(let ((*prindepth* 0))
	  (print-circle (ticlos:print-object object (decode-print-arg stream))) ) )
  object))

#-CLOS
sys:
(defun sys:prin1-then-space (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed, followed by a Space character."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprin1 object stream)
	(print-circle (print-object object 0 stream)))
  (funcall stream :tyo (pttbl-space *readtable*)))
  object)

#+CLOS
sys:
(defun sys:prin1-then-space (object &optional stream)
  "Print OBJECT on STREAM with quoting if needed, followed by a Space character."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* t)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprin1 object stream)
	(let ((*prindepth* 0))
	  (print-circle (ticlos:print-object object stream))))
  (funcall stream :tyo (pttbl-space *readtable*)))
  object)

#-CLOS
sys:
(defun sys:princ (object &optional stream)
  "Print OBJECT with no quoting, on STREAM.
Strings and characters print just their contents with no delimiters or quoting.
Pathnames, editor buffers, host objects, and many other hairy things
 print as their names with no delimiters."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* NIL)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprinc object stream)
	(print-circle (print-object object 0 stream))))
  object)


#+CLOS
sys:
(defun sys:princ (object &optional stream)
  "Print OBJECT with no quoting, on STREAM.
Strings and characters print just their contents with no delimiters or quoting.
Pathnames, editor buffers, host objects, and many other hairy things
 print as their names with no delimiters."
  (setq stream (decode-print-arg stream))
  (let ((*print-escape* NIL)
	(character-attribute-table (character-attribute-table *readtable*)))
    (if *print-pretty*
	(pprinc object stream)
	(let ((*prindepth* 0))
	  (print-circle (ticlos:print-object object stream)))))
  object)

(let ((compiler:compile-encapsulations-flag t))
     (advise sys:print-object :around :bind-top-level nil
       (let ((sys:*top-level-p* nil))
            (declare (special sys:*top-level-p*))
            :Do-It
       )
     )
)

;;; By JPR.
(defvar sys:*make-structure-instances-mouse-sensitive* t
"When true Defstruct instances will be printed mouse sensitively in the
 inspector.
"
)

sys:
(defun sys:print-named-structure (nss exp i-prindepth stream which-operations)
  (declare (special *print-structure*))
;  (declare (optimize (safety 0) (speed 3))) ;;; Fix this when TI does its thing. !!!!!!!!!!!!!!!! JPR.
  (let ((description (get nss 'defstruct-description)))
    (if (or (not description)
	    (if (boundp '*print-structure* )
		 (null *print-structure*)
		 (null *print-array*)))
	(printing-random-object (exp stream :typep))
        (progn
          (funcall stream :string-out "#S")
          (let ((slot-alist (defstruct-description-slot-alist))
                (l (list nss)))
            (dolist (s slot-alist)
	      (unless (defstruct-slot-description-name-slot-p (cdr s))
		(let* ((kwd (intern (symbol-name (car s)) pkg-keyword-package))
		       (fun (defstruct-slot-description-ref-macro-name (cdr s)))
		       (init (defstruct-slot-description-init-code (cdr s)))
		       (val (eval1 `(,fun ,exp))));watch out for macros!
		  
		  (unless (equal val init)
		    (push kwd l)
		    (push val l)))))
	    ;;; The next expression used to be
	    ;;; (print-object body i-prindepth stream which-operations)
	    ;;; Changed by JPR.
	    (let ((body (nreverse l)))
		 (if *make-structure-instances-mouse-sensitive*
		     (format stream "(~ ~ )" (first body) *print-escape*
			     (rest body) *print-escape*
		     )
		     (print-object body i-prindepth stream which-operations)
		 )
	    )
	  )
	)
    )
  )
)

;-------------------------------------------------------------------------------

(DEFUN INSPECT-PRINTER (LINE ARG STREAM ITEM-NO)      ;fi
  ;;;Make sure base is consistent since sometimes this is called from
  ;;; the mouse process.
    (DOLIST (ELT LINE)
      (COND
        ((NUMBERP ELT) (FORMAT STREAM "~VT" ELT))
        ((STRINGP ELT) (PRINC ELT STREAM))
        ((NOT (LISTP ELT)) (FERROR NIL "Unknown element type: ~S" ELT))
        ((STRINGP (CAR ELT)) (APPLY #'FORMAT STREAM ELT))
        (T
         (CASE (FIRST ELT)
               (:FUNCTION (APPLY (SECOND ELT) ARG STREAM ITEM-NO (CDDR ELT)))
               (:COLON (FORMAT STREAM ":~VT " (SECOND ELT)))
               ;;Provides a mechanism for inserting bold or italicized text (for column headers).
               (:FONT
                (UNWIND-PROTECT
		    (PROGN 
		      (SEND STREAM :SET-CURRENT-FONT (SECOND ELT) t)
		      ;;; Modified here by JPR.  It used simply to :String out
		      ;;; (THIRD ELT), which is no good if you want to have
		      ;;; items such as (:font 2 (:item1 ...))
		      (inspect-printer (list (THIRD ELT)) arg stream item-no))
                  (SEND STREAM :SET-CURRENT-FONT 0)))
	       (:Compound
		;;; added by JPR to support compoud items.
		(mapcar #'(lambda (item)
			    (inspect-printer (list item) arg stream item-no))
			(third elt)))
               (:ITEM1
                (LEXPR-SEND STREAM :ITEM1 ELT (SECOND ELT)
                            #'(LAMBDA (ELT &REST ARGS)
                                (APPLY (OR (FOURTH ELT) #'INSPECTION-DATA-PRINT-ITEM-CONCISELY)
                                       (THIRD ELT) ARGS))
                            (NTHCDR 4 elt)))
               (OTHERWISE (FERROR () "Unknown item type ~A" (FIRST ELT))))))))


;-------------------------------------------------------------------------------

(DEFUN GRIND-INTO-LIST-IO (OP &OPTIONAL ARG1 &REST REST)
  (COND
    ((EQ OP :WHICH-OPERATIONS) '(:TYO :line-out :string-out))
    ((EQ OP :TYO)
     (COND
       ((= ARG1 #\NEWLINE)
	(COND
	  (GRIND-INTO-LIST-STRING
	   (PUSH GRIND-INTO-LIST-STRING GRIND-INTO-LIST-LIST)
	   (SETQ GRIND-INTO-LIST-LINE (1+ GRIND-INTO-LIST-LINE))
	   (AND GRIND-INTO-LIST-ITEMS-P
		(PUSH () GRIND-INTO-LIST-ITEMS))))
	;;; Modified here by JPR to use a fat string not the standard one.  This
	;;; stops if from throwing away font information.
	(SETQ GRIND-INTO-LIST-STRING
	      ;;; 'string-char changed to fat-char by JPR.
; :element-type 'fat-char
	      (MAKE-ARRAY 50 :element-type 'sys:fat-char :LEADER-LIST '(0))))
       (T (VECTOR-PUSH-EXTEND ARG1 GRIND-INTO-LIST-STRING))))
    ;;; String-out and Line-out methods provided by JPR because the
    ;;; STREAM-DEFAULT-HANDLER throws away font information.
    ((or (eq op :string-out) (eq op :line-out))
     (let ((tem (if (typep arg1 'array) arg1 (string arg1)))
	   (fctn 'grind-into-list-io))
          (do ((len (cond ((second rest) (second rest))
			  (t (length tem))))
	       (i (cond ((first rest)) (t 0))
		  (1+ i)))
	      ((>= i len) nil)
	    (send fctn :tyo (aref tem i)))
	  (and (eq op :line-out)
	       (send fctn :tyo #\newline))
     )
    )
    (T (STREAM-DEFAULT-HANDLER 'GRIND-INTO-LIST-IO OP ARG1 REST))))

;-------------------------------------------------------------------------------

;;; String printer patches...

;;; by JPR.
sys:
(defun sys:string-and-stream-ok (string stream)
"True if the string is an art-string or if the string is fat and the stream
 knows how to cope with printing fat strings in fonts.
"
  (or (eq (array-type string) 'art-string)
      (and (eq (array-type string) 'art-fat-string)
	   (or (typep stream 'tv:inspect-window)
	       (typep stream 'tv:inspect-history-window)
	       (typep stream 'tv:shifting-stream)
	   )
      )
  )
)

sys:
(defun sys:print-quoted-string
       (string stream fastp &aux tem char (slash (pttbl-slash *readtable*)))
    (declare (ignore fastp))
    (cond
      ((not *print-escape*) (print-raw-string string stream t))
      (t (funcall stream :tyo (pttbl-open-quote-string *readtable*))
	 (setq tem (length string))
	 (cond
	   ((and (string-and-stream-ok string stream)
		 ;;; Modded here by JPR to accept fonted strings.
		 (do ((i 0 (1+ i))
		      (ch))
		     ((>= i tem) t)
		   (And (or (char= (setq ch (aref string i)) slash)
			    (char= ch #\"))
			(return ()))))
	    ;; There are no double quotes, and so no slashifying.
	    
	    (funcall stream :string-out string))
	   ;;; Modded here by JPR to accept fonted strings.
	   ((string-and-stream-ok string stream)
	    (do ((i 0 (1+ i)))
		((>= i tem) NIL)
	      (setq char (ldb %%ch-char (aref string i)))
	      (when 
		(or (char= char slash) (char= char #\"))
		(funcall stream :tyo slash))
	      (funcall stream :tyo (aref string i))))
	   (t
	    (do ((i 0 (1+ i)))
		((>= i tem) NIL)
	      (setq char (ldb %%ch-char (aref string i)))
	      (when 
		(or (char= char slash) (char= char #\"))
		(funcall stream :tyo slash))
	      (funcall stream :tyo char))))
	 (funcall stream :tyo (pttbl-close-quote-string *readtable*)))))

sys:
(defun sys:print-raw-string (string stream fastp &aux tem)
    (declare (ignore fastp))
    (cond
      ;;; Modded here by JPR to accept fonted strings.
      ((and t (string-and-stream-ok string stream))
       (funcall stream :string-out string))
      (t (setq tem (array-active-length string))
	 (do ((i 0 (1+ i)))
	     ((>= i tem) NIL)
	   (funcall stream :tyo (ldb %%ch-char (aref string i)))))))

;-------------------------------------------------------------------------------

;;; By JPR.
(defun debug-source-for-function (function)
"Returns the source code debugging source for Function if it can."
  (catch-error
    (let ((name (function-name function)))
         (if (and (consp name)
		  (equal (first name) :internal)
	     )
	     (debug-source-for-function (second name))
	     (let ((dbis (sys:get-debug-info-struct function t)))
	          (let ((encapsulated-p
			  (not (eq (sys:get-debug-info-struct function) dbis))
			)
		       )
		       (if encapsulated-p
			   (getf (sys:dbis-plist dbis)
				 :Source-Code-Debugger-Numbered-Source
			   )
			   (sys:function-spec-get
			     name :Source-Code-Debugger-Numbered-Source
			   )
		       )
		  )
	     )
	 )
    )
    nil
  )
)


(DEFMETHOD (basic-inspect :OBJECT-STACK-FRAME) (SF)
  (LET* ((sg (STACK-FRAME-STACK-GROUP SF))
	 (RP (SG-REGULAR-PDL sg))
	 (FRAME (STACK-FRAME-FRAME-NUMBER SF))
	 (FUNCTION (RP-FUNCTION-WORD RP FRAME)))
    (COND ((CONSP FUNCTION)
	   (SEND SELF :OBJECT-LIST FUNCTION))
	  ((TYPEP function 'compiled-function)
	   ;;; Modded here by JPR to account for source code debugging.
	   (let ((debugs (debug-source-for-function function)))
	        (if (and debugs
			 (fboundp 'eh:source-code-debugging-enabled)
			 (funcall 'eh:source-code-debugging-enabled)
		    )
		    (multiple-value-bind (code marked-region matched-code)
			(funcall 'eh:body-from-numbered-form
			  debugs (funcall 'current-pc sg frame)
			)
		      (let ((eh:*grinding-debugged-source-code*
			      (list marked-region matched-code)
			    )
			    (eh:*current-pc* (funcall 'current-pc sg frame))
			   )
			   (declare (special eh:*grinding-debugged-source-code*
					     eh:*current-pc*
				    )
			   )
		           (SEND SELF :OBJECT-LIST code)
		      )
		    )
		    (FEF-DISPLAY-LIST
		      FUNCTION SELF (RP-EXIT-PC RP FRAME)
		      (LIST NIL NIL NIL NIL FONTS:HL12B 
			    (STACK-FRAME-FUNCTION-NAME SF)))))))))

;-------------------------------------------------------------------------------

;;; By JPR
(defun sys:make-fat-output-string-stream ()
"Returns a fat-string string output stream."
  (make-string-output-stream
    ;; Changed by RDA to be :TYPE 'ART-FAT-STRING as :ELEMENT-TYPE 'FAT-CHAR doens't work.
    (MAKE-ARRAY 50 :element-type 'sys:fat-char :LEADER-LIST '(0))
  )
)



sys:
(defresource sys:pprint-resource ()
  :constructor (let ((print-structure (make-array (* 100 (size-of-pp-obj))
			   :fill-pointer 0))
		     (pprint-buffer-stream
		       ;;; Modded by JPR to use fat strings.  This lets font
		       ;;; info be printed properly.
		       (sys:make-fat-output-string-stream)))
		 (pprint-init)
		 (cons print-structure pprint-buffer-stream))
  :deallocator pprint-resource-deallocator 
  :initial-copies 0)

;;; CLear the resource, since there may be some old thin strings in it.
(clear-resource 'sys:pprint-resource)

;-------------------------------------------------------------------------------

;;; I've had to patch this because the function in the band is not the compiled
;;; version of this one.

(DEFUN tv:GRIND-INTO-LIST-MAKE-ITEM (THING LOC ATOM-P LEN)
  (LET ((IDX (IF GRIND-INTO-LIST-STRING
	       (ARRAY-ACTIVE-LENGTH GRIND-INTO-LIST-STRING)
	       0)))
    (COND
      (ATOM-P
         ;; An atom -- make an item for it.
       (PUSH (LIST LOC :LOCATIVE IDX
		   (+ IDX  (if (stringp (IF (CONSP loc) (CAR loc) loc))
			       (+ 1 (or (position #\cr (IF (CONSP loc) (CAR loc) loc))
					(1- (flatsize (IF (CONSP loc) (CAR loc) loc)))))
			       len)))
	     (CAR GRIND-INTO-LIST-ITEMS)))
      (T
       ;; Printing an interesting character
       (CASE THING
	 ('sys:start-of-object
	  ;; Start of a list.
	  (PUSH (LIST LOC IDX GRIND-INTO-LIST-LINE () ()) GRIND-INTO-LIST-LIST-ITEM-STACK))
	 ('sys:end-of-object
	  ;; Closing a list.
	  (LET ((ITEM (POP GRIND-INTO-LIST-LIST-ITEM-STACK)))
		;; 1+ is to account for close-paren which hasn't been
		;; typed yet. in rel2 next line was (1+ idx)
	    (SETF (FOURTH ITEM) IDX)
	    (SETF (FIFTH ITEM) GRIND-INTO-LIST-LINE)
	    (PUSH ITEM GRIND-INTO-LIST-LIST-ITEMS))))))))


;-------------------------------------------------------------------------------

(defun (:property fef-function value-function) (thing)
  ;;; patched by JPR.  Was (rest (second (second thing)))
  ;;; sometimes (second (second thing)) is a bad locative
  ;;; so (rest of it is a bad thing to do.
  (if (catch-error (rest (second (second thing))) nil)
      (rest (second (second thing)))
      (second (second thing))))


;;; Patch this so that PDLs only have the number of allocated slots.  Stuff
;;; above the current stack pointer can have bad pointers.
(DEFUN INSPECT-ARRAY-ITEM-GENERATOR (msg &optional arg1)
  (DECLARE (:SELF-FLAVOR BASIC-INSPECT))
  (CASE msg
    (:NUMBER-OF-ITEMS
     (+ (IF (CADR PRINT-FUNCTION-ARG)
	    (OR (ARRAY-LEADER-LENGTH (CAR PRINT-FUNCTION-ARG)) 0)
	    0)
	(LENGTH (CADDR PRINT-FUNCTION-ARG))
	(let ((str (format nil "~S" (CAR PRINT-FUNCTION-ARG))))
	     (let ((pdl-p (search "PDL" str :test #'string-equal))
		   (array (CAR PRINT-FUNCTION-ARG))
		  )
	          (if pdl-p
		      (let ((sg (array-leader array 0)))
			   (if (search "SPECIAL" str :test #'string-equal)
			       (without-interrupts
				 (catch-error
				   (si:sg-special-pdl-pointer sg) nil
				 )
			       )
			       (without-interrupts
				 (catch-error
				   (si:sg-regular-pdl-pointer sg) nil
				 )
			       )
			   )
		      )
		      (ARRAY-TOTAL-SIZE (CAR PRINT-FUNCTION-ARG)))))))
    (:NUMBER-OF-ITEM
     (IF (NUMBERP arg1)
	 (+ arg1 (LENGTH (CADDR PRINT-FUNCTION-ARG)))
	 (POSITION arg1 (THE list (CADDR PRINT-FUNCTION-ARG)) :test #'EQ)))
    (:ITEM-OF-NUMBER
     (IF (< arg1 (LENGTH (CADDR PRINT-FUNCTION-ARG)))
	 (NTH arg1 (CADDR PRINT-FUNCTION-ARG))
	 (- arg1 (LENGTH (CADDR PRINT-FUNCTION-ARG)))))))


;-------------------------------------------------------------------------------

;;; Patch
;;; Old TI code as follows:
;(defmethod  ( standard-method :print-self) (stream &rest ignore)
;  (declare (type stream stream))
;  (let ((name (function-name function)))
;    (format stream "#<method ~s" name)
;    (unless (and (sys:validate-function-spec name)
;		 (eq (si:fdefinition-safe name t) function))
;      ;; Not the currently installed method; show the address to avoid ambiguity.
;      (format stream " ~o" (sys:%pointer self))))
;  (write-char #\> stream))

;;; This is patched for two reasons:
;;; a) to get mouse sensitivity in the inspector
;;; b) to fix a bug.  In the original version it would use the function name
;;;    of the method function to print itself.  This is bogus because some
;;;    methods share method-functions with other methods.  The printed
;;;    representation of the method-function therefore doesn't necessarily
;;;    match with the paramater specializers of the method you're interested in.

ticlos:
(defmethod (ticlos:standard-method :print-self) (stream &rest ignore)
  (declare (type stream stream))
  (let ((name (function-name function)))
    (if parameter-specializers
        (progn (format stream "#<method (~S ~~{ ~s~} ("
		       (first name)
		       (list (sys:fdefinition-safe (second name))
			     t (second name)
		       )
		       (rest (rest (butlast name))))
	       (loop for spec in parameter-specializers
		     for rest on parameter-specializers
		     when (not (eq rest parameter-specializers))
		     do (format stream " ")
		     if (typep spec 'class)
		     do (format stream "~" (list spec t (class-name spec)))
		     else do (format stream "~" (list spec t spec)))
	       (format stream "))")
	)
	(format stream "#<method ~s" name))
    (unless (and (sys:validate-function-spec name)
		 (eq (si:fdefinition-safe name t) function))
      ;; Not the currently installed method; show the address to avoid ambiguity.
      (format stream " ~o" (sys:%pointer self))))
  (write-char #\> stream))

