;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(TVFONT TR10 HL10 TR10I TR10B) -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

(1defvar* 2*Default-Call-Graph-Depth** 2
"The default depth for call graphs."
)

(1deftype* real-compiled-function ()
"This type is true for any sort of compiled function, be it a ucode entry
 or a normal compiled function.
"
  '(1or* compiled-function microcode-function)
)

(1defun* the-fef (x &optional (no-error-p t))
"Given something tries to coerce it into a fef.  If it is a symbol then it tries
 to do clever things to get a fef that denotes it, e.g. its macro function.
 If it's something it knows about, but can't get a fef for it, then it just
 returns X.  If it doesn't know about its type at all then it barfs.
 The types it knows about are:  Symbols, Compiled functions, ucode functions,
 clos methods, clos generic functions and grapher vertices.
"
  (typecase x
    (symbol (1if* (1fboundp* x)
	        (1let* ((1def* (4fdefinition-safe* x t)))
		    (1if* (1consp* def)
		        (1if* (typep (1rest* def) 'real-compiled-function)
			    (1rest* def)
			    x
			)
			(1if* (1typep* def 'real-compiled-function)
			    def
			    x
			)
		    )
	       )
	       x
	    )
    )
    (compiled-function x)
    (microcode-function x)
    (any-type-of-clos-method (get-fef-from-object x))
    (any-type-of-clos-gf (get-fef-from-object x))
    (1closure* (first (si:convert-closure-to-list x)))
    (4vertex* x)
    (cons (if (equal (first x) 'macro)
	      (rest x)
	      x
	  )
    )
    (otherwise (1or* (4fdefinition-safe* x t)
		   (1if* no-error-p
		       nil
		       (ferror nil "~S is not a function." x)
		   )
	       )
    )
  )
)

(1defvar* 2*Selected-Filters** Nil
"The list of filter functions currently selected by the user.  This is a useful
 variable to set in your init file so that you prune call graphs quite a bit
 by default.
"
)

(1defun* maybe-filter (filters fun as child-of window)
"Is passed a list of filter functions, function (probably a fef) a comparison
 function and the function of which this is a child.  The comparison function
 is the function that was recorded if filters were selected by mousing on a
 function.  This allows filters such as \"same package as this function\". 
 If any of the functions in the list of funtions returns true for the function
 when funcalled as (filter-fun fun as child-of) then the function is filtered
 out.
"
  (1declare* (optimize (safety 0)))
  (1if* filters
      (1if* (1funcall* (1first* filters) fun as child-of window)
	  nil
	  (maybe-filter (1rest* filters) fun as child-of window)
      )
      fun
  )
)

(1defvar* 2*As-Function** Nil
"A place where a comparison function is stored.  This is the function on which
 the user last right clicked and put up the filter menu.  This allows filters
 like \"same package as this function\" to work.
"
)

(1defun* filter-called-functions (functions child-of window)
"Given the functions (Functions) that are called by Child-Of, filter out all of
 those that should be filtered out according to the user's currently selected
 set of filter functions.
"
  (1declare* (1special* 2*As-Function** 2*Selected-Filters**))
  (1If* 2*Selected-Filters**
     (1remove-if-not*
       #'(lambda (fun)
	   (maybe-filter 2*Selected-Filters** fun
			 (1if* (1boundp* '2*As-Function**)
			     2*As-Function**
			     nil
			 )
			 Child-of window
           )
	 )
         functions
     )
     functions
  )
)

(defparameter *function-details-cache* (make-hash-table :test #'equal)
"A hash table that maps functions to the 4ivars-and-messages-in-method*
 data for that function.  This is used to speed up call graphing for non-first
 time graphs.
"
)

(1defun* 2Maybe-Cache-Function-Values* (name fef function)
"Looks in the cache for the function Fef, whose name is Name and if it
 finds values then returns these, otherwise it calls Function on Fef and caches
 the results returning the values of the function as its values.
" 
  (1let* ((cache (gethash name *function-details-cache*)))
      (1if* cache
	  (1values-list* cache)
          (1let* ((results
		  (1multiple-value-list* (1funcall* function fef))
	       )
	      )
	      (setf (gethash name *function-details-cache*)
		    results
	      )
	      (1values-list* results)
	  )
      )
  )
)

(1defun* caching-ivars-and-messages-in-method (fef)
"Calls 4ivars-and-messages-in-method* on Fef and caches the results as
 appropriate.  Otherwise it's just like a call to
 4ivars-and-messages-in-method*.
"
  (2Maybe-Cache-Function-Values*
    (1function-name* fef) fef '4ivars-and-messages-in-method*
  )
)

(defun functions-called-from-interpretted-definition-1 (function)
"Given an interpretted function it grovels over the function and records all
 of the functions and generic functions called.  These are collected in the
 specials *all-functions-called* *all-gfs-called*.
"
  (declare (special *all-functions-called* *all-gfs-called*))
  (declare (optimize (safety 0)))
  (typecase function
    (cons (functions-called-from-interpretted-definition-1 (first function))
	  (functions-called-from-interpretted-definition-1 (rest function))
    )
    (symbol (if (fboundp function)
		(1if* (4generic-function-p-safe* function)
		    (pushnew function *all-gfs-called*)
		    (pushnew function *all-functions-called*)
		)
		nil
	    )
    )
    (otherwise nil)
  )
)

(defun 2Functions-Called-From-Interpretted-Definition* (function)
"Given an interpretted function or the name of an interpretted function it
 returns values as if 4ivars-and-messages-in-method* had been called for
 a fef, i.e. it returns sundry null values for the things it can't figure out
 and values for the functions called and the generic functions called.
"
  (1declare* (1values* ignore ignore ignore 2Functions-Called*
		    generic-functions-called ignore ignore ignore ignore ignore
           )
  )
  (declare (optimize (safety 0)))
  (let ((*all-functions-called* nil)
	(*all-gfs-called* nil)
       )
       (declare (special *all-functions-called* *all-gfs-called*))
       (functions-called-from-interpretted-definition-1
	 (if (and (symbolp function) (fboundp function))
	     (symbol-function function)
	     function
	 )
       )
       (1values* nil nil nil (nreverse *all-functions-called*)
	        (1nreverse* *all-gfs-called*) nil nil nil nil nil
       )
  )
)

(1defun* 2Caching-Functions-Called-From-Interpretted-Definition* (name)
"Just like 2Functions-Called-From-Interpretted-Definition*, only the results
 of calling this function are cached.  It returns, amongst a number of null
 values the functions called by Name and the generic functions called by Name.
"
  (1declare* (1values* ignore ignore ignore 2Functions-Called*
		    generic-functions-called ignore ignore ignore ignore ignore
           )
  )
  (2Maybe-Cache-Function-Values*
    name name
    #'2Functions-Called-From-Interpretted-Definition*
  )
)

(1defun* get-function-call-data (function)
"Is called with a function, which can be either a compiled function, an
 interpretted function or the name of an interpretted fucntion.  It returns
 three values:  the non-generic functions called by Function (including the
 methods related to Function if Function is a generic function), the generic
 functions called by Function and the macros that were expanded in processing
 Function.
"
  (1declare* (1values* functions-called-including-methods-for-gfs
		   generic-functions-called 4macros-expanded*
           )
  )
  (multiple-value-bind
    (referenced-ivars referenced-keywords problem referenced-functions
     referenced-generic-functions args returned-values locals
     specials-referenced specials-bound
    )
      (1if* (1compiled-function-p* function)
	  (2Caching-Ivars-And-Messages-In-Method* function)
	  (2Caching-Functions-Called-From-Interpretted-Definition* function)
      )
    (ignore referenced-ivars referenced-keywords problem
     args returned-values locals specials-referenced specials-bound
    )
    (1values* (1append* (1if* (4generic-function-p-safe* function)
			 (4generic-function-methods-safe*
			   (4function-generic-function-safe* function)
			 )
			 nil
		     )
		     referenced-functions
	    )
	    referenced-generic-functions
	    (1mapcar* #'ucl:first-if-list
		     (1if* (1and* (1functionp* function)
			       (sys:get-debug-info-struct function)
			 )
			 (1let* ((dbis (sys:get-debug-info-struct function)))
			     (1if* (1typep* dbis 'sys:debug-info-struct)
				 (1getf* (sys:dbis-plist dbis)
				       3:Macros-Expanded*
				 )
				 nil
			     )
			 )
			 nil
		     )
	    )
    )
  )
)

(defun functions-called (by-function)
"Returns two values: the functions called by By-Function and a flag which is
 true if By-Function is recursive, i.e. By-Function is in the first value.
 The functions returned have been passed through the function filter, so only
 those functions not excluded by the filter will be returned.  The functions
 returned are returned as fefs where possible.
"
  (1declare* (1values* 2Functions-Called* by-function-is-recursive-p)
	   (1special* 2*Graph*-window*)
  )
  (let ((fef (the-fef by-function)))
       (1if* fef
	  (multiple-value-bind
	    (referenced-functions referenced-generic-functions macros)
	      (get-function-call-data fef)
	    (let ((kids (mapcar 'the-fef
			    (1append* referenced-functions
				     referenced-generic-functions
			    )
			)
		  )
		 )
		 (values (2Filter-Called-Functions*
			   (1append*
			     (remove fef kids)
			     (1mapcar* '2The-Fef* macros)
			   )
			   fef *graph-window*
			 )
			 (member fef kids)
		 )
	    )
	  )
	  nil
       )
  )
)

(1defvar* 2*Graph-Calls-Print-Package** "User"
"The default package to bind to during printing when doing a graph calls."
)
(1defvar* 2*Printing-Bindings** nil
"A list of (*special* value) pairs.  bindings to these specials are made
 with these values whilst printing functions in the call grapher.  e.g.
 `((*print-case* :capitalize) (*package* ,(find-package \"TV\")))
"
)

(1defun* function-calls-others (name)
"Is true if the function named by Name calls any other functions.  This allows
 non-terminal functions to be labeled as such.
" 
  (1let* ((fef (2The-Fef* name)))
      (1if* fef
	 (1multiple-value-bind*
	   (referenced-functions referenced-generic-functions macros)
		 (get-function-call-data fef)
	   (1or* referenced-functions referenced-generic-functions macros)
	 )
	 nil
      )
  )
)

(1defun* 2More-Kids-Indicator* (kids)
"Returns a  or a  if there are kids, otherwise the null string."
  (1let* ((4window* (1if* (1typep* self '4vertex*)
		      (1send* self 3:Window*)
		      (1if* (typep self '4basic-x-y-scrolling-window*)
			 self
			 nil
		      )
		  )
       )
      )
      (1if* (1and* 4window* kids)
	 (if (and (mx-p)
		  (or (not (typep self 'vertex))
		      (not (equal fonts:cptfont (send self :font)))
		  )
	     )
	     (1if* (1equal* 3:Vertical* (1send* 4window* 3:Orientation*)) "" "->")
	     (1if* (1equal* 3:Vertical* (1send* 4window* 3:Orientation*)) "" "")
	 )
	  ""
      )
  )
)

(1def*var 2*Fef-Print-Functions**
	`(,#'(Lambda (name)
	       (1if* (1consp* name) (1format* t "~	" name t) (1format* t "~S" name))
	     )
	  ,#'(lambda (name fef)
	      (1princ* (2More-Kids-Indicator*
		       (1if* (1typep* fef '4vertex*)
			  (1not* (1send* fef 3:Child-Edges*))
			  (2Function-Calls-Others* name)
		       )
		     )
	      )
	     )
	 )
"Functions to call during the displaying of functions in the call grapher.
 the functions are called in sequence in the list each with the NAME of the
 function being displayed and optionally the vertex node.  These functions
 should print what they think is important about the function to
 *standard-output*.  An example of such a function that puts an arrow at
 the end of a display to indicate that there are children could be as follows:
  #'(lambda (name)
     (1if* (2Function-Calls-Others* name)
	 (1if* (1equal* 3:Vertical* (1send* (1send* self 3:Window*)3:Orientation*))
	     (1format* t \"\")
	     (1format* t \"\")
	 )
	 nil
     )
    )
"
)

(1defun* 2Simple-Print-Fef* (fef)
"A simple print function to print fefs.  Fef is actually either a fef or the
 name of a function.  It returns a string that will be displayed for the
 function.  It does this by calling the functions in 2*Fef-Print-Functions**.
"
  (1Let* ((4*package**
	  (1find-package*
	    (1string-upcase*
	      (1string* (1if* (1typep* self '2Stand-Alone-Grapher*)
			 (get self '3Print-Package*)
			 2*Graph-Calls-Print-Package**
		     )
	      )
	    )
	  )
	)
	(4*print-pretty** nil)
       )
       (1Progv* (1mapcar* #'1first* 2*Printing-Bindings**)
	      (1mapcar* #'1second* 2*Printing-Bindings**)
         (1with-output-to-string* (4*standard-output**)
	   (1let* ((name (1function-name* (4coerce-to-node* fef))))
	       (1loop* for fn in 2*Fef-Print-Functions** do
		     (1if* (1>* (1length* (1arglist* fn)) 1)
			 (1funcall* fn name fef)
			 (1funcall* fn name)
		     )
	       )
	   )
	 )
       )
  )
)

;-------------------------------------------------------------------------------

(defflavor 2Predicated-Boxing-And-Filling-Mixin*
	   (boxing-selections) ()
  3:Initable-Instance-Variables*
  (3:Required-Flavors* 4boxed-filled-vertex*)
  (3:Default-Init-Plist* 3:Box-Myself-P* nil 3:Fill-Myself-P* nil)
  (3:Documentation* "A mixin that allows predicated boxing and filling.")
)

(defmethod (2Predicated-Boxing-And-Filling-Mixin* :after :init) (ignore)
"Makes sure that each vertex in the graph is set up so that it's box/fill
 status is right.  It does this by searching though the list
 boxing-selections until it gets a match and using the
 values it finds there.
" 
  (let ((fn (1function-name* (coerce-to-node item))))
       (1loop* for (1function* (1fill* box dash))
	     in boxing-selections
	     when (1funcall* function fn)
	     do (1progn* (1set-in-instance* self 'dashed-p dash)
		       (1set-in-instance* self 'fill-myself-p fill)
		       (set-in-instance self 'box-myself-p box)
		       (1return* nil)
	        )
       )
  )
)

;-------------------------------------------------------------------------------

(1Defun* 2Interpretted-Function-P* (1function*)
"Is true if Function is an interpretted function."
  (1equal* (2The-Fef* function) function)
)

(1def*var 2*Function-Vertex-Boxing-Selections**
	`((,#'1macro-function* (t nil))
	  (,#'2Interpretted-Function-P* (nil t))
	  (,#'(lambda (1function*) (1ignore* function) t) (nil nil))
	 )
"A list of pairs, each element of which is of the form:
  (<predicate> (<fill-p> <box-p> <dashed-p>))
 The predicate functions are called successively on functions until one of them
 is true.  When a match is found them the second value is used to set the
 filledness, the boxedness and the dashedness of the vertex on the graph.
"
)

(defflavor function-vertex ()
	   (2Predicated-Boxing-And-Filling-Mixin* boxed-filled-vertex)
  (3:Documentation* "The flavor of grapher vertex to use for functions.")
  (3:Default-Init-Plist* 3:Boxing-Selections* 2*Function-Vertex-Boxing-Selections**)
)

;-------------------------------------------------------------------------------

(defmethod (function-vertex :print-self) (stream &rest ignore)
"Prints out a function vertex."
  (catch-error
    (format stream "#<Function Vertex ~A>"
	    (function-name (coerce-to-node item))
    )
    nil
  )
)

(1defmethod* (4function-vertex* :get-real-function) (from)
  (1function-name* (coerce-to-node from))
)
  
(defmethod (function-vertex :after :init) (ignore)
"Makes sure that each vertex in the graph is set up so that it's box/fill
 status is right.  It does this by searching though the list
 2*Function-Vertex-Boxing-Selections** until it gets a match and using the
 values it finds there.
" 
  (let ((fn (1send* self 3:Get-Real-Function* item)))
       (1loop* for (1function* (1fill* box dash))
	     in 2*Function-Vertex-Boxing-Selections**
	     when (1funcall* function fn)
	     do (1progn* (1set-in-instance* self 'dashed-p dash)
		       (1set-in-instance* self 'fill-myself-p fill)
		       (set-in-instance self 'box-myself-p box)
		       (1return* nil)
	        )
       )
  )
)

(1setf* (1get* '4functions-called* :right-button-menu-items)
     '4*call-grapher-mouse-r-menu-items**
)

(1defmethod* (2Stand-Alone-Grapher* 3:Mouse-R*) (x y)
"Is called when the user right clicks on the background of the call grapher."
  (1ignore* x y)
  (1let* ((entry
	  (1let* ((4vertex* (1find-if* #'(lambda (x) (1typep* x '4basic-vertex*))
				 (1send* self 3:Item-List*)
			)
	       )
	      )
	      (1if* 4vertex*
		  (1get* (1send* 4vertex* 3:Child-Function*)
		       3:Right-Button-Menu-Items*
		  )
		  nil
	      )
	  )
	)
       )
      (1let* ((choice (w:menu-choose
		      (1if* entry
			  (1symbol-value* entry)
			  4*call-grapher-mouse-r-menu-items**
		      )
		    )
	   )
	  )
	  (1if* choice
	     (1send* self choice x y)
	     nil
	  )
      )
  )
)

;-------------------------------------------------------------------------------

(1defvar* 2*Graph-Calls-Item-Type-Alist**
  '((3:Vertex* 2Edit-Graphed-Function*
	     "L: Edit this Function; M: Drag object; Sh-M/M2: Overview; R: Menu of operations"
	     ("Graph this Node" 3:Value* 2Maybe-Graph-Function* 3:Documentation*
	      "Start a new grapher whose top node is this function."
	     )
	     ("Extend this Node" 3:Value* Grapher-Extend-Graph
	      3:Documentation* "Extend the graph below this node."
	     )
	     ("Inspect" 3:Value* 2Graph-Calls-Inspect* 3:Documentation*
	      "Inspect this function."
	     )
	     ("Edit" 3:Value* 2Edit-Graphed-Function* 3:Documentation*
	      "Edit the source code definition of this function."
	     )
	     ("Trace" 3:Value* Trace-function-command 3:Documentation*
	      "Trace this function."
	     )
	     ("Graph Callers" 3:Value* 2Graph*-callers-command 3:Documentation*
	      "Graph the callers of this function."
	     )
	     ("Elide"   3:Value* 4collapse* 3:Documentation*
	      "Elide this function."
	     )
	     ("Elide Children"   3:Value* 4collapse-children*
	      3:Documentation* "Elide the children of this function."
	     )
	     ("UnElide Children" 3:Value* 4expand-children*
	      3:Documentation* "UnElide the children of this function."
	     )
	     ("Filter" 3:Value* 2Select-Grapher-Filters* 3:Documentation*
	      "Select filtration for what functions to show on the graph."
	     )
	     ("Variables" 3:Value* 2Set-Graph-Calls-Variables* 3:Documentation*
	      "Set sundry variables for the call tree grapher."
	     )
    )
    (3:Edge* nil
     "R: Menu of operations"
     ("Elide"   3:Value* remove-edge 3:Documentation* "Elide this edge.")
     ("Elide Edge and Children"   3:Value* 4collapse*
      3:Documentation* "Elide this edge and its children."
     )
    )
   )
"An item type alist for call tree grapher.  To add new right button menu
entries simply add the relevant menu item to the list after the third element of
the list for the relevant item type.
"
)

(1defun* 2Graph-Calls-Inspect* (1function*)
"Inspects a fucntion.  This gets called from the call grapher."
  (1inspect* function)
)

(1defun* 2Grapher-Extend-Graph* (1function* 4window* vertex)
"Extends the graph for the current node."
  (1ignore* 4vertex*)
  (2Extend-Graph-For* function window)
)

(1defvar* 2*Call-Grapher-Mouse-R-Menu-Items**
  '(("Variables" 3:Value* 2Set-Graph-Calls-Variables* 3:Documentation*
     "Set sundry variables for the call tree grapher."
    )
    ("Filter" 3:Value* 2Select-Grapher-Filters*-1 3:Documentation*
     "Select filtration for what functions to show on the graph."
    )
    ("Flush Cache" 3:Value* 2Flush-Function-Details-Cache*
     3:Documentation* "Flush the functions called cache."
    )
    ("3Create ZMACS Diagram*" 3:Value* 2Grapher-Create-*A-2Zmacs-Diagram*
     3:Documentation*
     "3Create a Zmacs Diagram Line representing this graph in an editor buffer.*"
    )
   )
"The menu items that are used for the menu that gets popped up by right
 clicking on the background of the call grapher.
"
)

(1defun* 2Grapher-Create-*A-2Zmacs-Diagram* (4window* x y)
"Creates a zmacs diagram line for the current display.."
  (1ignore* x y)
  (2Grapher-Create-Zmacs-Diagram* nil window nil)
)

(1defun* 2Flush-Function-Details-Cache* (4window* x y)
"Flushes the cache of functions called by other functions."
  (1ignore* 4window* x y)
  (1clrhash* 2*Function-Details-Cache**)
)

(1defvar* 2*Filter-Items**
 '(("Same Package" 3:Value* 2Same-Package* 3:Documentation*
    "Only those functions in the same package as selected function"
   )
   ("Not Sys:" 3:Value* 2Not-Sys-Functions* 3:Documentation*
    "Not the functions in the Sys: package."
   )
   ("No macros" 3:Value* 2No-Macros* 3:Documentation*
    "Only those functions which are not (macro-function x)."
   )
   ("No Special Forms" 3:Value* 2No-*special-forms 3:Documentation*
    "Only those functions which are not special-form-p."
   )
   ("Not called by Sys:" 3:Value* 2Not-Called-By-Sys-Functions* 3:Documentation*
    "Not the functions called by functions in the Sys: package."
   )
   ("Not called by Lisp:" 3:Value* 2Not-Called-By-Lisp-Functions* 3:Documentation*
    "Not the functions called by functions in the Lisp: package."
   )
   ("Not called by Macros" 3:Value* 2Not-Called-By-Macros* 3:Documentation*
    "Filter out functions that are called by macros, but not the macros themselves."
   )
   ("Not called by Special Forms" 3:Value* 2Not-Called-By-*Special-Forms
    3:Documentation*
    "Filter out functions that are called by special forms, but not the special forms themselves."
   )
   ("Not called by Interpretted Functions" 3:Value*
    2Not-Called-By-Interpretted-Functions* 3:Documentation*
    "Filter out functions that are called by interpretted functions, but not the interpretted functions themselves."
   )
   ("Not % Functions" 3:Value* 2Not-%-Functions* 3:Documentation*
    "Not any functions that have a \"%\" in their names."
   )
   ("Not * Functions" 3:Value* 2Not-*-Functions* 3:Documentation*
    "Not any functions that have a \"*\" in their names."
   )
   ("Only methods below generic functions" 3:Value*
    2Only-Methods-Below-Generic-Functions* 3:Documentation*
    "Only the methods for generic functions, not the functions called by the generic function itself during method selection."
   )
   ("Not methods below generic functions" 3:Value*
    2Not-Methods-Below-Generic-Functions* 3:Documentation*
    "Not the methods for generic functions, but the functions called by the generic function itself during method selection."
   )
   ("Same source file" 3:Value* 2Same-Source-File* 3:Documentation*
    "Only those functions that are in the same source file as the selected function."
   )
  )
"This is a list of menu items that is called when the user wants to select some
 call grapher filter functions.  The items in the list are normal menu items,
 the value of the :value element must be a function with the following contract:
   arglist = (function as-function child-of)
   where Function is the function that we're trying to filter,
         As-Function is a function with which we may be comparing Function.
           This may be nil if there is no such function and is set when the
           user selects filters by right buttoning on a function and filtering
           from the menu or explicitly setting tv:2*As-Function**.
         Child-Of is the parent node that called this function, if known (nil
           for the root node).
   Returned value = true if the Function arg is to be filtered out, nil
     otherwise.
 The function cannot guarantee that any of its args will be functions.  They
 should always call coerce-to-node if in doubt to make sure that it isn't a
 grapher internal data structure.  This may result in either function or a
 function name, so the filter function should do the righ thing with either. 
 Any of the functions in this list already will serve as examples.
"
)

(1defun* 2Select-Grapher-Fil*t2ers*-1 (4window* x y)
"Pops up a menu of available filter functions and gets the user to select some.
 These are used in future to remove functions from the call graph.
"
  (1ignore* x y)
  (multiple-value-bind (choices chose-p)
      (w:multiple-menu-choose 2*Filter-Items**
	3:Label* "Filter Graphing"
	3:Highlighted-Items*
	(1remove-if-not*
	  #'(lambda (x) (1member* (1third* x) 2*Selected-Filters**))
	  2*Filter-Items**
	)
      )
      (1if* chose-p
	  (1progn* (1Setq* 2*Selected-Filters** Choices)
		 (1setf* (1get* 4window* '3Set-Graph-Behaviour-Changed*) t)
	  )
	  nil
      )
  )
)

(1defun* call-grapher-background-menu (4window* x y &rest args)
  (1ignore* x y args)
  (1let* ((choice (w:menu-choose 2*Call-Grapher-Mouse-R-Menu-Items**)))
      (1if* choice
	 (funcall choice window x y)
	 nil
      )
  )
)

(1Defun* 2Make-Call-Grapher-Window* (&rest inits)
"Makes a stand-alone call grapher window."
  (1apply* #'1make-instance* '2Stand-Alone-Grapher*
	  3:Edges* (4offsetting-stand-alone-grapher-edges*
		    '2Stand-Alone*-2Graphers*
		  )
	  inits
  )
)

(1defun* maybe-initialize-window-properties (4window* properties)
  (1loop* for (prop-name default-value) in properties
        when (1equal* (1get* 4window* prop-name '--not-there) '--not-there)
	do (1setf* (1get* 4window* prop-name) (1symbol-value* default-value))
  )
)

(1defvar* 2*Graph-Calls-Orientation** 3:Horizontal*
"The default orientation for the call grapher."
)

(1defvar* 2*Graph-Calls-Plot-Trees-P** Nil
"The default for whether a new call grapher should plot trees or graphs."
)

(1defvar* 2*Grapher-Selection-Mode** 3:Existing*
"The default for the way that new graphers are selected."
)

(1defun* 2Graph-Calls* (1function* &optional (on 3:From-Resource*) (label nil))
"Graphs the functions called by Function on grapher window On (when defaulted
 it allocates a new window from a resource).  Label is printed at the bottom
 of the window.  Function can be either the name of a function or a Fef.  If
 the display gets too complicated then it can be filtered by sundry filter
 functions.  These can be selected either by using the right button menus in
 grapher windows or by calling the function tv:2Select-Grapher-Filters*.  In
 your login file you can set the variable tv:2*Selected-Filters** to a list of
 the names of the filter functions that you want.  For information on writing
 your own filter functions please look at the documentation/definition of
 tv:2*Filter-Items**.
"
  (1let* ((fef (2The-Fef* function)))
      (1if* (1or* (1not* fef)
	      (1and* (1typep* function '4vertex*)
		    (1send* function 3:Child-Edges*)
	      )
	 )
	 fef
	 (1let* ((4window* (1if* (1equal* 3:From-Resource* on)
			     (1allocate-resource* '2Stand-Alone-Graphers*)
			     on
			)
	       )
	      )
	      (2Maybe-Initialize-Window-Properties* 4window*
		`((print-package		2*Graph-Calls-Print-Package**)
		  (depth			2*Default-Call-Graph-Depth**)
		  (Orientation			2*Graph-Calls-Orientation**)
		  (tree-p			2*Graph-Calls-Plot-Trees-P**)
		  (grapher-selection-mode	2*Grapher-Selection-Mode**)
		 )
	      )
	      (4plot-a-graph*
		fef
		3:On* window
		3:Child-Function* '2Functions-Called*
		3:Print-Function* '2Simple-Print-Fef*
		3:Vertex-Flavor* '2Function-Vertex*
		3:Item-Type-Alist* 2*Graph-Calls-Item-Type-Alist**
		3:Background-Mouse-Blip-Functions*
		  '((#\mouse-r
		     "Menu of operations on the call grapher as a whole"
		     2Call-Grapher-Background-Menu*
		    )
		   )
		3:Depth* (1if* (1typep* on '2Stand-Alone-Grapher*)
			   (or (get on '3Depth*) 2*Default-Call-Graph-Depth**)
			   2*Default-Call-Graph-Depth**
		       )
		3:Orientation* (1if* (1typep* on '2Stand-Alone-Grapher*)
				(or (1and* (get on '3Orientation*)
					  (1member* (get on '3Orientation*)
						   '(3:Vertical* 3:Horizontal*)
					  )
					  (get on '3Orientation*)
				    )
				    2*Graph-Calls-Orientation**
				)
				2*Graph-Calls-Orientation**
			    )
		3:Tree-P* (1if* (1typep* on '2Stand-Alone-Grapher*)
			    (or (get on '3Tree-P*) 2*Graph-Calls-Plot-Trees-P**)
			    2*Graph-Calls-Plot-Trees-P**
			)
		3:Dont-Graph-If-Already-Graphing-This-Node* t
		3:Label* (1or* label
			   (1format* nil "~S's call graph."
				    (1function-name*
				      (1function-name* fef)
				    )
			   )
		       )
	      )
	      4window*
	 )
     )
  )
)

(1Defun* 2Extend-Graph-For* (1function* &optional (on 3:From-Resource*))
"Extends the graph on window On below the function Function."
  (1let* ((2*Default-Call-Graph-Depth** 1)
        (fef (2The-Fef* function))
       )
       (1if* fef (2Graph-Calls* (4find-vertex-for* fef on) on 3:Default*) nil)
  )
)


(defun 2Get-A-Grapher-Window*
       (&optional (how 3:Existing*) (resource 'Stand-Alone-Graphers))
"Gets a grapher window.  How can be :Existing, :New or :Ask.  This will cause
 either an existing window to be used, a new one to be allocated or you to
 be prompted about what to do.  New windows are allocated out of the resource
 Resource.
"
  (let ((result nil))
       (map-resource #'(lambda (obj used-p ignore)
			 (if used-p (push obj result)))
		     resource
       )
       (1let* ((window
	      (1case* how
		(3:Existing*
		  (if result
		      (loop for win in result
			    when (not (send win :exposed-p))
			    return win
			    finally (return (ucl:first-if-list win))
		      )
		      (allocate-resource resource)
		  )
		)
		(3:New* (1allocate-resource* resource))
		(3:Ask*
		 (1if* (1and* result (4mouse-y-or-n-p* "Use existing window?"))
		     (2Get-A-Grapher-Window* 3:Existing* resource)
		     (2Get-A-Grapher-Window* 3:New* resource)
		 )
		)
		(otherwise (1ferror* nil "Unknown selection type ~S." how))
	      )
	    )
	   )
	   4window*
      )
  )
)

(4add-system-key*
  #\G '(4find-window-of-flavor* '4stand-alone-grapher* selected-window)
  "Grapher Window." nil
)

;-------------------------------------------------------------------------------

(defun inspect-graph-function-p (name &optional (thing nil))
"Is true if Name names a function which we can graph.  Thing is ignored."
  (ignore thing)
  (or (and (symbolp name) (fboundp name)
	   (or (compiled-function-p (symbol-function name))
	       (consp (symbol-function name))
	   )
      )
      (compiled-function-p name)
  )
)

(1defun* 2Inspect-Graph-Function* (1function* &optional (window nil))
"Graphs a function.  This is called from the inspector and such-like.
 A background process is spun off to do this so that selection
 problems are avoided.
"
  (1declare* (1special* 4*force-exposure**))
  (1ignore* window)
  (1process-run-function*
    '(3:Name* "Expose Grapher" 3:Priority* -1)
    #'(lambda (1function* expose-p)
	(1let* ((4*force-exposure** expose-p))
	    (1declare* (1special* 4*force-exposure**))
	    (2Graph-Calls* function
			  (or window
			      (2Get-A-Grapher-Window*
				2*Grapher-Selection-Mode**
				'2Stand-Alone-Graphers*
			      )
			  )
	    )
	)
      )
      Function 4*force-exposure**
  )
)

(1defun* maybe-graph-function (1function* &optional (thing nil) &rest ignore)
"Graphs Function in a grapher if it knows how to."
  (1if* (inspect-graph-function-p function thing)
      (2Inspect-Graph-Function* function thing)
      (tv:notify tv:selected-window "Could not graph ~S." function)
  )
)

;-------------------------------------------------------------------------------

(1defun* Edit-Graphed-Function (1function*)
"Invokes Zmacs to edit the source of a function that has been graphed."
  (4try-and-edit* (4coerce-to-node* function))
)

(1defun* 2Graph*-callers-command (1function* window ignore)
"Graph the callers of this function."
  (graph-callers (4coerce-to-node* function) window)
)

(1defun* Trace-function-command (1function*)
"Invokes Zmacs to edit the source of a function that has been graphed."
  (zwei:just-trace-name (function-name (4coerce-to-node* function)))
)

;-------------------------------------------------------------------------------
;;; Filter function definitions.
;-------------------------------------------------------------------------------

(1defun* 2Same-Package* (1function* as child-of window)
"Will filter out Function if it is not in the same package as As."
  (1ignore* child-of window)
  (1if* as
      (1let* ((fun (1function-name* (2The-Fef* function)))
	    (fas (1function-name* (2The-Fef* as)))
	  )
	  (1and* fun fas
	       (1not* (1and* (1symbolp* fun) (1symbolp* fas)
			  (1equal* (1symbol-package* fun) (1symbol-package* fas))
		    )
	       )
	  )
      )
      nil
  )
)

(1defun* 2Not-Sys-Functions* (1function* as child-of window)
"Will filter out Function if it is in the Sys package."
  (1ignore* as child-of window)
  (1let* ((fun (1function-name* (2The-Fef* function))))
      (1and* fun (1symbolp* fun)
	    (1equal* (1find-package* "SYS") (1symbol-package* fun))
      )
  )
)

(1defun* 2Not-Called-By-Sys-Functions* (1function* as child-of window)
"Will filter out Function if it has been called by a function in the
 Sys package.
"
  (1ignore* function as window)
  (1let* ((fun (1function-name* (2The-Fef* child-of))))
      (1and* fun (1symbolp* fun)
	    (1equal* (1find-package* "SYS") (1symbol-package* fun))
      )
  )
)

(1defun* 2Not-Called-By-Lisp-Functions* (1function* as child-of window)
"Will filter out Function if it has been called by a function in the
 Lisp package.
"
  (1ignore* function as window)
  (1let* ((fun (1function-name* (2The-Fef* child-of))))
      (1and* fun (1symbolp* fun)
	    (1equal* (1find-package* "LISP") (1symbol-package* fun))
      )
  )
)

(1defun* 2Not-%-Functions* (1function* as child-of window)
"Will filter out Function if it has a name with a % in it."
  (1ignore* as child-of window)
  (1let* ((fun (1function-name* (2The-Fef* function))))
      (1and* fun (1symbolp* fun)
	    (find #\% (1the* string (1symbol-name* fun)) 3:Test* #'1char-equal*)
      )
  )
)

(1defun* 2Not-*-Functions* (1function* as child-of window)
"Will filter out Function if it has a name with a % in it.  Such functions are
 often generated by compiler optimisers for system functions.
"
  (1ignore* as child-of window)
  (1let* ((fun (1function-name* (2The-Fef* function))))
      (1and* fun
	    (1symbolp* fun)
	    (1find* #\* (1symbol-name* fun) 3:Test* #'1char-equal*)
	    (1>* (1length* (1the* string (1symbol-name* fun))) 1)
      )
  )
)

(1defun* 2Only-Methods-Below-Generic-Functions* (1function* as child-of window)
"Filters out Function if it is called by a generic function but is not a method
 of that generic function.
"
  (1ignore* as window)
  (and (4generic-function-p-safe* child-of)
       (1let* ((name (1function-name* (2The-Fef* function))))
	   (1and* name
		 (1not* (1and* (1consp* name) (1equal* 'ticlos:method (1first* name))))
	   )
       )
  )
)

(1defun* 2Not-Methods-Below-Generic-Functions* (1function* as child-of window)
"Filters out Function if it is a method on a generic function but is not
 called directly by it.
"
  (1ignore* as window)
  (and (4generic-function-p-safe* child-of)
       (1let* ((name (1function-name* (2The-Fef* function))))
	   (1and* name (1consp* name) (1equal* 'ticlos:method (1first* name)))
       )
  )
)

(1defun* same-source-file (1function* as child-of window)
"Filters out Filter unless it is defined in the same source file as As."
  (1ignore* child-of window)
  (1if* as
      (1let* ((fun (1function-name* (2The-Fef* function)))
	    (fas (1function-name* (2The-Fef* as)))
	  )
	  (1not* (1and* fun fas (1symbolp* fun) (1symbolp* fas)
		     (get fun :source-file-name)
		     (get fas :source-file-name)
		     (fs:pathname-equal
		       (fs:default-pathname (path-string fun 'defun))
		       (fs:default-pathname (path-string fas 'defun))
		     )
	       )
	  )
      )
      nil
  )
)

(1defun* 2No-Macros* (1function* as child-of window)
"Filters out Function if it is a macro."
  (1ignore* as child-of window)
  (1let* ((name (1function-name* (2The-Fef* function))))
       (1and* name (macro-function name) (1not* (1special-form-p* name)))
  )
)

(1defun* 2No-Special-Forms* (1function* as child-of window)
"Filters out Function if it is a special form."
  (1ignore* as child-of window)
  (1special-form-p* (1function-name* (2The-Fef* function)))
)

(1defun* 2Not-Called-By-Macros* (1function* as child-of window)
"Will filter out Function if it has been called by a macro."
  (1ignore* function as window)
  (1let* ((name (1function-name* (2The-Fef* child-of))))
       (1and* name (1symbolp* name) (1macro-function* name)
	     (1not* (1special-form-p* name))
       )
  )
)

(1defun* 2Not-Called-By-Special-Forms* (1function* as child-of window)
"Will filter out Function if it has been called by a special form."
  (1ignore* function as window)
  (1let* ((name (1function-name* (2The-Fef* child-of))))
      (1and* (1symbolp* name) (1special-form-p* name))
  )
)

(1defun* 2Not-Called-By-Interpretted-Functions* (1function* as child-of window)
"Will filter out Function if it has been called by an interpretted function."
  (1ignore* function as window)
  (1equal* child-of (2The-Fef* child-of))
)

(1defun* 2Select-Grapher-Fil*t2ers* (1function* 4window* &rest ignore)
"Is called when the user right buttons on a function vertex.  Pops up a menu
 olf filters to select.  If the user selects any then these are recorded, as is
 the currently selected function.
"
  (multiple-value-bind (choices chose-p)
        (w:multiple-menu-choose 2*Filter-Items**
	  3:Label* (1format* nil "Filter Graphing for ~S"
			  (1function-name* (2The-Fef* function))
		 )
	  3:Highlighted-Items*
	  (1remove-if-not*
	    #'(lambda (x) (1member* (1third* x) 2*Selected-Filters**))
	    2*Filter-Items**
	  )
	)
    (1if* chose-p
        (1progn* (1setq* 2*As-Function** (2The-Fef* function))
	        (1Setq* 2*Selected-Filters** Choices)
		(1send* window 3:Set-Graph-Behaviour-Changed* t)
       )
       nil
    )
  )
)

(1defun* 2Set-Graph-Calls-Variables* (4window* x y)
"Sets up interesting variables to do with the function that has been
 clicked on.
"
  (1ignore* x y)
  (4assign-using-menu*
    (((1get* 4window* 'depth) "Depth" 3:Fixnum*)
     ((1get* 4window* '2Print-Package*) "Package" 3:String-Or-Nil*)
     ((get 4window* '2Orientation*)
      "Orientation" 3:Assoc*
		  (1list* (1cons* "Horizontal" 3:Horizontal*)
		       (1cons* "Vertical"   3:Vertical*)
		  )
     )
     ((1get* 4window* '2Tree-P*) "Tree or Graph plot" 3:Assoc*
	       (1list* (1cons* "Tree" t) (1cons* "Graph" nil))
     )
     ((1get* 4window* '2Grapher-Selection-Mode*) "Grapher selection mode." 3:Assoc*
	(1list* (1cons* "Use an existing one" 3:Existing*)
	     (1cons* "Create a new one" 3:New*)
	     (1cons* "Ask" 3:Ask*)
	)
     )
    )
    3:Label* "Set a grapher variables."
  )
)


;-------------------------------------------------------------------------------

(4defun* 2Graph-Something* (name)
"Is called when the user has selected the Graph command from some tool.  It
 tries to find a way to graph Name and, if it can find such a thing, will
 graph it.  If there are multiple options than the user will be prompted with
 a menu.
"
  (4multiple-value-bind* (object inspect-p name perspective)
      (4map-into-show-x* name nil '2Grapher-Perspective*)
    (1ignore* perspective)
    (4if* object
        (4if* inspect-p
	    (4progn* (1format* 4*query-io** "~&~S graphed in the inspector." name)
		   (1inspect* object name)
	    )
	    (1format* 4*query-io** "~&Graphed ~S." object)
	)
	(1format* 4*query-io** "~&~S not graphed." name)
    )
  )
)

Zwei:
(defcom zwei:com-graph
"Prompt for something and graphs it." ()
  (let ((fcn (zwei:read-function-name "Graph"
				 (relevant-function-name (point) nil t t t)
				 'zwei:aarray-ok 'zwei:multiple-ok
	     )
	)
	(*print-case* :Capitalize)
       )
       (tv:graph-something fcn)
  )
  dis-none
)


;;; Record the new zmacs commands.
(zwei:set-comtab zwei:*standard-comtab* '(#\c-sh-g zwei:com-graph)
		 (zwei:make-command-alist '(zwei:com-graph))
)

(zwei:set-comtab zwei:*standard-comtab* '(#\m-sh-g zwei:com-graph)
		 (zwei:make-command-alist '(zwei:com-graph))
)

;-------------------------------------------------------------------------------

;;; Record Graph as a consistency command.
(1Pushnew* '("Graph" :Value
	     (((#\c-sh-g zwei:Com-graph)
	       (#\m-sh-g zwei:Com-graph)
	       )
	      ((graph-cmd nil))
	      ((eh:Comw-graph nil))
	      )
	     )
	    4*all-consistancy-commands**
	    3:Test* #'equalp
)


(defcommand graph-cmd nil			
  '(:description "Graph something."
    :Names ("Graph")
    :Keys (#\m-sh-g #\c-sh-g)
   )
   (declare (special user history = inspectors frame))
   (send user :clear-SCREEN)
   (format user "~&Something to graph:")
   (multiple-value-bind (value punt-p)
       (inspect-get-value-from-user user history inspectors)
     (or punt-p (tv:graph-something value))
   )
   (send frame :handle-prompt)
)

eh:
(defcommand eh:Comw-graph ()
            '(:description "Graph something."
              :names "Graph"
	      :keys (#\m-sh-g #\c-sh-g)
	     ) 
   (send *window-debugger* :Set-Who-Line-Doc-String-Overide
	 "Select something to graph."
   )
   (1unwind-protect*
       (tv:graph-something (eh:window-read-thing "to graph"))
      (send *window-debugger* :set-who-line-doc-string-overide nil)
      (send *window-debugger* :handle-prompt)
   )
)

;-------------------------------------------------------------------------------


(defun the-class (the-class)
"Coerces The-Class into a class or a flavor object."
  (typecase the-class
    (symbol (the-class (or (class-named-safe the-class t)
			   (get the-class 'si:flavor)
			   (ferror nil "~S is not a class." the-class)
		       )
	    )
    )
    (otherwise (if (or (class-p-safe the-class)
		       (typep the-class 'si:flavor)
		   )
		   the-class
		   (if (any-sort-of-clos-instance-p the-class)
		       (class-named-safe (type-of the-class))
		       (if (instancep the-class)
			   (type-of the-class)
			   (ferror nil "~S is not a class." the-class)
		       )
		   )
	       )
    )
  )
)

(defun class-kids (class)
"Returns a list of the direct superclasses or local component flavors of class."
  (if (class-p-safe (coerce-to-node class))
      (class-local-supers-safe (coerce-to-node class))
      (if (typep (coerce-to-node class) 'si:flavor)
	  (mapcar #'(lambda (x) (get x 'si:flavor))
		  (si:flavor-depends-on (coerce-to-node class))
	  )
	  nil
      )
  )
)

(defun class-parents (class)
"Returns a list of the direct subclasses or local dependent flavors of class."
  (if (class-p-safe (coerce-to-node class))
      (class-direct-subclasses-safe (coerce-to-node class))
      (if (typep (coerce-to-node class) 'si:flavor)
	  (mapcar #'(lambda (x) (get x 'si:flavor))
		  (si:flavor-depended-on-by (coerce-to-node class))
	  )
	  nil
      )
  )
)

(1defun* 2Class-Print-Fun* (class)
"Prints out Class neatly as a graph node." 
  (1let* ((2The-Class* (4coerce-to-node* class))
        (4*package** nil)
	(4*print-pretty** nil)
       )
      (1format* nil "~S~A"
	      (1if* (4class-p-safe* 2The-Class*)
		  (4class-name-safe* 2The-Class*)
		  (1if* (1typep* 2The-Class* 'si:flavor)
		      (si:flavor-name 2The-Class*)
		      2The-Class*
		  )
	      )
	      (2More-Kids-Indicator*
		(1if* (1typep* class '4vertex*)
		    (and (not (1send* class 3:Child-Edges*))
			 (2Class-Kids* (coerce-to-node 2The-Class*))
		    )
		    (2Class-Kids* 2The-Class*)
		)
	      )
      )
  )
)


(defun 2Edit-Graphed-Thing* (thing)
"Edits the source of a thing that has been graphed."
  (try-and-edit (coerce-to-node thing))
)

(1defun* 2Inspect-A-Graphed-Class* (class)
  (1inspect* (4allocate-data*
	     (1if* (4class-p-safe* class)
		 '4show-clos-class*
		 '4show-flavor*
	     )
	     (4coerce-to-node* class)
	   )
  )
)

(1defvar* 2*Graph-Class-Item-Type-Alist**
  '((3:Vertex* graph-class
     "L: Graph this Class; M: Drag object; Sh-M/M2: Overview; R: Menu of operations"
     ("Inspect" 3:Value* 2Inspect-A-Graphed-Class* 3:Documentation*
      "Inspect this class."
     )
     ("Edit"    3:Value* 2Edit-Graphed-Thing* 3:Documentation* "Edit this class.")
     ("Elide"   3:Value* 4collapse* 3:Documentation* "Elide this class.")
     ("Elide Children"   3:Value* 4collapse-children*
      3:Documentation* "Elide the children of this class."
     )
     ("UnElide Children" 3:Value* 4expand-children*
      3:Documentation* "UnElide the children of this class."
     )
    )
    (3:Edge* nil
     "R: Menu of operations"
     ("Elide"   3:Value* remove-edge 3:Documentation* "Elide this edge.")
     ("Elide Edge and Children"   3:Value* 4collapse*
      3:Documentation* "Elide this edge and its children."
     )
    )
   )
"The item type AList for the class grapher.  New menu items should be added to
 the end.
"
)

(1defvar* 2*Default-Class-Graph-Depth** nil
"The default depth of class graphs."
)

(1defflavor* 2Class-Vertex* () (4boxed-filled-vertex*)
  (3:Documentation*
    "A type of vertex for class graphs.  Classes will be filled flavors won't."
  )
)

(1defflavor* dynamic-anchors-edge () (4edge*)
  (3:Default-Init-Plist* 3:Dynamic-Anchors-P* t)
  (3:Documentation* "Edges with dynamic anchors.")
)

(defmethod (2Class-Vertex* :after :init) (ignore)
"Makes sure that each vertex in the graph is set up so that it's box/fill
 status is right.  Classes are filled.  Anything else is boxed.
"
  (1let* ((class (2The-Class* (coerce-to-node item))))
      (1let* ((class-p (4class-p-safe* class)))
	  (1if* (1and* class-p
		    (4class-name-safe* class)
		    (1symbolp* (4class-name-safe* class))
		    (1not* (1get* (4class-name-safe* class) 'si:flavor))
	     )
	     (1progn* (1set-in-instance* self 'fill-myself-p t)
		    (1set-in-instance* self 'box-myself-p nil)
	     )
	     (1progn* (1set-in-instance* self 'fill-myself-p nil)
		    (1set-in-instance* self 'box-myself-p t)
	     )
	  )
      )
  )
)

(1defvar* 2*Class-Grapher-Mouse-R-Menu-Items**
  '(("3Create ZMACS Diagram*" :value grapher-background-create-zmacs-diagram
     3:Documentation*
     "3Create a Zmacs Diagram Line representing this graph in an editor buffer.*"
    )
   )
"The menu items that are used for the menu that gets popped up by right
 clicking on the background of the call grapher.
"
)

(defun grapher-background-create-zmacs-diagram (4window* x y &rest ignore)
"Create a zmacs diagram window from a background mouse blip."
  (1ignore* x y)
  (2Grapher-Create-Zmacs-Diagram* nil 4window* nil)
)

(1defun* 2Class-Grapher-Background-Menu* (4window* x y &rest args)
  "Pops up the background menu for the class grapher."
  (1ignore* x y)
  (1let* ((choice (w:menu-choose 2*Class-Grapher-Mouse-R-Menu-Items**
			       3:Label* "Pick an operation"
	        )
       )
      )
      (1If* choice
	 (apply choice 4window* x y args)
	 nil
      )
  )
)

(1defun* 2Graph-Class*
        (class &optional (on 3:From-Resource*) (label nil) &rest ignore)
"Graphs Class, which is a class or a flavor on the window On.  If On is
 defaulted then a window is selected from a resource.  The returned value is
 the window that was used.
"
  (1let* ((class (2The-Class* class)))
       (4plot-a-graph* class
	  3:On* on
	  3:Child-Function* '2Class-Kids*
;	  :parent-function 'class-parents
	  3:Print-Function* '2Class-Print-Fun*
	  3:Vertex-Flavor* '2Class-Vertex*
	  3:Edge-Flavor* '2Dynamic-Anchors-Edge*
	  3:Depth* 2*Default-Class-Graph-Depth**
	  3:Tree-P* nil
	  3:Item-Type-Alist* 2*Graph-Class-Item-Type-Alist**
	  3:Dont-Graph-If-Already-Graphing-This-Node* t
	  3:Background-Mouse-Blip-Functions*
              '((#\mouse-r "Menu of Operations"
		 2Class-Grapher-Background-Menu*
		)
	       )
	  3:Label*
	    (1or* label
	        (if (class-p-safe class)
		    (format nil "~A's superclasses and supclasses"
			    (class-name-safe class)
		    )
		    (format nil "~A's components and dependants"
			    (typecase class
			      (si:flavor (si:flavor-name class))
			      (symbol (si:flavor-name (get class 'si:flavor)))
			      (otherwise (ferror "~S is not a class or flavor."
						 class
					 )
			      )
			    )
		    )
		)
	    )
       )
  )
)

;(defparameter *window* nil)
;(setq *window* (graph-class 'bottle-flav))
;(compile-if-you-have-to "x6:clos;clos-test")
;(setq *window* (graph-class 'inspect-frame *window*))

(defun inspect-graph-class (class)
"Graphs a class from the inspector.  Spins off a process to do it so
 that we don't have selection/exposure problems.
"
  (1declare* (1special* 4*force-exposure**))
  (process-run-function
    '(:Name "Expose Grapher" :Priority -1)
    #'(lambda (class expose-p)
	(1let* ((4*force-exposure** expose-p))
	    (1declare* (1special* 4*force-exposure**))
	    (graph-class class (2Get-A-Grapher-Window*))
	)
      )
    class
    4*force-exposure**
  )
)

(defun inspect-graph-class-p (class &optional (thing nil))
"Is true if Class is a class or flavor that we can graph, be it a class or
 the name of one.
"
  (ignore thing)
  (1or* (1and* (1symbolp* class) (class-named-safe class t))
      (1and* (1symbolp* class) (1get* class 'si:flavor))
      (4class-p-safe* class)
      (1typep* class 'si:flavor)
  )
)

;-------------------------------------------------------------------------------

;;; Maybe wire M-. to the grapher.

(1defvar* zwei:*wire-grapher-to-m-.* nil
"When true the grapher will be invoked whenever zmacs goes to a new possibility
 if it can be.
"
)

(1let* ((compiler:compile-encapsulations-flag t))
     (1advise* zwei:execute-possibility 3:Around* :wire-to-grapher nil
       (1locally* (1declare* (1special* *inside-execute-possibility*))
	  (1let* ((results (1let-if* (1not* (1boundp* '*inside-execute-possibility*))
			       ((*inside-execute-possibility* t))
			   ;;; Bind *inside-execute-possibility* so that we
			   ;;; only graph once.  Zwei:Execute-Possibility is
			   ;;; called recursively.
			   (1multiple-value-list* 3:Do-It*)
			)
		)
	       )
	       (1if* (1or* (1boundp* '*inside-execute-possibility*)
		      (1not* zwei:*wire-grapher-to-m-.*)
		   )
		   nil
		   (1let* ((4*force-exposure** nil))
		       (1declare* (1special* 4*force-exposure**))
		       (map-into-show-x (1second* (1second* arglist)) t
					'2Grapher-Perspective*
		       )
		   )
	       )
	       (1values-list* results)
	  )
       )
     )
)

(1let* ((compiler:compile-encapsulations-flag t))
     (1advise* zwei:edit-definition-1 3:Around* :wire-to-grapher nil
       (1locally* (1declare* (1special* *inside-execute-possibility*))
	  (1let* ((results (1let-if* (1not* (1boundp* '*inside-execute-possibility*))
			       ((*inside-execute-possibility* t))
			   ;;; Bind *inside-execute-possibility* so that we
			   ;;; only graph once.  Zwei:Execute-Possibility is
			   ;;; called recursively.
			   (1multiple-value-list* 3:Do-It*)
			)
		)
	       )
	       (1if* (1or* (1boundp* '*inside-execute-possibility*)
		      (1not* zwei:*wire-grapher-to-m-.*)
		   )
		   nil
		   (1let* ((4*force-exposure** nil))
		       (1declare* (1special* 4*force-exposure**))
		       (map-into-show-x (first arglist) t
					'2Grapher-Perspective*
		       )
		   )
	       )
	       (1values-list* results)
	  )
       )
     )
)


;-------------------------------------------------------------------------------
;;; From Jamie Zawinski.

(defun 4grapher-create-zmacs-diagram *(ignore window ignore)
  "2Prompt the user for some parameters, and dump this graph to an editor buffer or file as a Diagram Line.
  The diagram line will be the same width and height as the window; only the part of the graph visible in the window
 will be visible in the editor.*"
  (let* ((buffer-p t)
	 (draw-box-p t)
	 (draw-label-p nil)
	 (label-text (or (tv:label-string (send window :label)) (send window :label)))
	 (x-off 100)
	 (name nil)
	 (select-editor-p nil))
    (declare (special buffer-p name draw-box-p x-off draw-label-p label-text select-editor-p))
    (1let* ((result
	   (1catch* :abort-menu
	     (tv:choose-variable-values
	       '((buffer-p "3  Output Diagram to*" :assoc (("3Buffer*" . t) ("3File*" . nil)))
		 (name "3  File Name*" :pathname-or-nil)
		 ""
		 (draw-box-p "3  Enclose Graph in a Box*" :boolean)
		 (x-off "3  Horizontal Offset (pixels)*" :fixnum)
		 (select-editor-p "3  Select Buffer after Writing*" :boolean)
		 ""
		 (draw-label-p "3  Label the Graph*" :boolean)
		 (label-text "3  Label*" :string))
	       3:Margin-Choices* ; JPR.
		(List (List "Abort []" '(Throw :Abort-Menu :abort))
		      "Do it []"
	   	)
	       :label "2Create Diagram Line*"))))
      (1if* (1equal* 3:Abort* result) ;JPR.
	 :ignore
	 (let ((4*print-pretty** nil)
	       (4*print-circle** nil)
	       (output (if buffer-p (zwei:create-one-buffer-to-go) (pathname name))))
	   (when buffer-p
	     (when name (zwei:set-buffer-pathname (pathname name) output))
	     (send output :set-major-mode 'ZWEI:TEXT-MODE))
	   (with-open-stream (stream (if buffer-p
					 (zwei:interval-stream output nil nil t)
					 (open name :direction :output)))
	     (format stream "3-*- Mode:Text; Fonts:(CPTFONT CPTFONT) -*-~2%~A~%----------~%*"
		     (or (tv:label-string (send window :label)) (send window :name)))
	     (grapher-create-zmacs-diagram-1 stream window draw-box-p x-off (and draw-label-p label-text))
	     (format stream "3~&----------~2%*"))
	   (when select-editor-p (ed output))
	   output)))))


(defun 4grapher-create-zmacs-diagram-1 *(stream window draw-box-p x-offset label)
  "2Dump an Epsilon-format representation of the graph drawn on WINDOW to STREAM.
  DRAW-BOX-P: whether to surround the graph with a box.
  X-OFFSET: how much to shift the diagram right, in pixels.
  LABEL: a string or NIL, which will be drawn below the diagram if present.*"
  (format stream "3~&~C# 1 ZWEI::DOX-DIAGRAM~%(*" #\Epsilon)
  (format stream "3(:OPEN ~A ~A ~A ~A)*"		  ;1 Left Top Width Height Color*
	  x-offset 0
	  (send window :inside-width) ; JPR.
	  (+ (send window :inside-height)
	     (if label
		 (max (send window :top-margin-size)
		      (send window :bottom-margin-size))
		 0)))
  (when draw-box-p
    (format stream "3(:BOX ~A ~A ~A ~A 2)*" 0 0 (send window :inside-width) (send window :inside-height)))
  (when label
    (format stream "3(:BOX ~A ~A ~A ~A 2)*" 0 (1- (send window :inside-height))
	    (send window :inside-width)
	    (1- (max (send window :top-margin-size) (send window :bottom-margin-size))))
    (format stream "3(:TEXT ~S ~A ~A ~S)*"
	    (tv:font-name (tv:font-evaluate (or (tv:label-font (send window :label)) fonts:cptfont)))
	    (tv:label-left (send window :label)) (+ 3 (send window :inside-height))
	    label))
  (let* ((x-off (- ;(send window :left-margin-size)
		   (send window :x-pl-offset)))
	 (y-off (- ;(send window :top-margin-size)
		   (send window :y-pl-offset))))
    (dolist (item (send window :item-list))
     (1if* (1send* item 3:Visible-P*)
      (typecase item
      (TV:EDGE (format stream "3(:LINE ~A ~A ~A ~A 1 1) *"	;1 Start-X Start-Y End-X End-Y Depth Line-Width*
		       (+ (send item :from-x) x-off) (+ (send item :from-y) y-off)
		       (+ (send item :to-x) x-off) (+ (send item :to-y) y-off))
	       (when (1and* (send item :arrowhead-p) (1send* item 3:On-Screen-P*))
		 (let* ((rev-p (send item :send-if-handles :reverse-arrowhead-p))
			(to-x (+ x-off (if rev-p (send item :from-x) (send item :to-x))))
			(to-y (+ y-off (if rev-p (send item :from-y) (send item :to-y)))))
		   (format stream "3(:LINE ~A ~A ~A ~A 1 1) *"	;1 Start-X Start-Y End-X End-Y Depth Line-Width*
			   (+ x-off (symeval-in-instance item 'tv:tip1-x)) (+ y-off (symeval-in-instance item 'tv:tip1-y))
			   to-x to-y)
		 (format stream "3(:LINE ~A ~A ~A ~A 1 1) *"	;1 Start-X Start-Y End-X End-Y Depth Line-Width*
			 (+ x-off (symeval-in-instance item 'tv:tip2-x)) (+ y-off (symeval-in-instance item 'tv:tip2-y))
			 to-x to-y)))
	       (when (send item :label)
		 (format stream "3(:TEXT ~S ~A ~A ~S)*"	;1 Font X Y String*
			 (tv:font-name (tv:font-evaluate (or (send item :label-font) fonts:cptfont)))
			 (+ x-off (first (send item :label-position))) (+ y-off (second (send item :label-position)))
			 (send item :label)))
	       )
      (TV:VERTEX (format stream "3(:TEXT ~S ~A ~A ~S)*"	;1 Font X Y String*
			 (tv:font-name (tv:font-evaluate (or (send item :font) fonts:cptfont)))
			 (+ x-off (send item :left-edge)) (+ y-off (send item :top-edge))
			 (funcall (tv:vertex-node-print-function item) (send item :item)))
		 (when (1or* (send item 3:Send-If-Handles* 3:Box-Myself-P*)
			   ;;; Note fill-myself-p is here because I don't know how
			   ;;; to do a filled box.  JPR.  These vertices might not
			   ;;; be boxed or filled vertices so :send-if-handles.
			   (send item 3:Send-If-Handles* 3:Fill-Myself-P*))
		   (let* ((off (send item :boxing-offset)))
		     (format stream "3(:BOX ~A ~A ~A ~A 1)*"	;1 Left Top Width Height Depth Line-Width Fill-Color Line-Color*
			     (- (+ x-off (send item :left-edge)) off)
			     (- (+ y-off (send item :top-edge)) off)
			     (+ off off (- (send item :right-edge) (send item :left-edge)))
			     (+ off off (- (send item :bottom-edge) (send item :top-edge)))))))
      (t (warn "3Don't know what to do with ~A*" item))))))
  (format stream "3)~2%*"))


