;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(TVFONT tr10i tr10b hl10) -*-

;;; ***** Copyright (c) 1987 Texas Instruments.  All rights reserved.
;;; ***** Portions of this file contain code belonging to TI.

;;; This file written by James Rice of the Stanford University
;;; Knowledge Systems Laboratory (Rice@Sumex-Aim.Stanford.Edu)
;;; Some of this code was written by modifying existing
;;; code belonging to TI.

;;; In this file there are a lot of functions called foo-Safe.  This is
;;; a bit of a strange name, that exists for historical reasons.  Maybe I'll
;;; change them all sometime.  Anyway the reason that they are "SAFE" is not
;;; that they won't barf horribly if you give them the wrong args or something
;;; goes wrong, but rather that foo-safe should work and do the right thing
;;; whether it is given a TICLOS thing or a PCL thing.  This is crucial to
;;; my intent of supporting bith TICLOS and PCL in the Inspector.

;-------------------------------------------------------------------------------
;;; Just define a dummy package if CLOS is not loaded.  This should mean
;;; That these patches can be loaded onto a system which does not have CLOS
;;; loaded.

(eval-when (compile load)
  (let ((si:inhibit-fdefine-warnings t))
       (if (find-package 'ticlos)
	   nil
	   (defpackage ticl (:use lisp ticl))
       )
       (if (not (sys:find-system-named 'pcl t t))
	   (defpackage pcl (:use lisp))
	   nil
       )
  )
)

(defun ticlos-p ()
"True if TICLOS is loaded."
  (member :clos *features*)
)

(defun pcl-p ()
"True if PCL is loaded."
  ;;; Both of these are kludges.
  (or (member :Portable-Commonloops *features*)
      (fboundp 'pcl:defmethod)
  )
)

;-------------------------------------------------------------------------------


(defun clos-p ()
"True if some sort of CLOS is loaded."
  (or (Ticlos-P) (Pcl-P))
)

;;; Warning this is very system dependent !!!!!!!! JPR.
(defun ticlos-instance-p (instance)
"This should return T if Instance is an instance of a TICLOS class.
 Whenever a TICLOS class is instantiated it has a cdr code bit set, which
 distinguishes it from flavors instances.  This is shoetimes not the case,
 however, for standard methods and such like, which are flavor class instances
 for some reason.
"
  (declare (special sys:%%clos-instance-header-flag))
  (and (boundp 'sys:%%clos-instance-header-flag)
       (instancep instance)
           ;;; definitely.
       (or (eql (sys:%p-cdr-code instance)
		'#.(ldb sys:%%q-cdr-code
			(dpb 1 sys:%%clos-instance-header-flag 0)
		   )
	   )
	   ;;; Somewhat kludgy, but standard classes seem not to have the
	   ;;; above property.
	   (1let* ((1type* (type-of instance)))
	        (and (symbolp type)
		     (get type 'ticlos:class-def)
		     (or (member type
				 '(ticlos:standard-writer-method
				   ticlos:standard-reader-method
				   ticlos:combined-method
				  )
			 )
			 (not (typep (get type 'ticlos:class-def)
				     'ticlos:flavor-class
			      )
			 )
		     )
		)
	   )
       )
  )
)

(defmacro defsafe
      (fname (discriminator &rest args) &key (documentation nil)
       (pcl nil) (ticlos nil) (error-p t) (declarations nil)
      )
"Defines a function called FName, which should execute PCL if Discriminator
 is a PCL thing or TICLOS if it is a TICLOS instance.  If Error-p is true
 and Discriminator is neither a PCL or TICLOS thing then it generates an error
 otherwise it returns nil.
"
  `(defun ,fname (,discriminator ,@args)
     ,documentation
     ,declarations
     (if (and (pcl-p) (iwmc-class-p-safe ,discriminator))
	,(if pcl
	     pcl
	     (if error-p '(ferror nil "Don't know how to do this.") nil)
	 )
	 (if (and (ticlos-p) (ticlos-instance-p ,discriminator))
	    ,(if ticlos
		 ticlos
		 (if error-p '(ferror nil "Don't know how to do this.") nil)
	     )
	    ,(if error-p
		`(ferror nil "Something wrong with ~S." ,discriminator)
		 nil
	     )
	 )
     )
   )
)

(defmacro defsafe-gf
      (fname (discriminator &rest args) &key (documentation nil)
       (pcl nil) (ticlos nil) (error-p t) (declarations nil)
      )
"Defines a function called FName, which should execute PCL if Discriminator
 is a PCL GF or TICLOS if it is a TICLOS GF.  If Error-p is true
 and Discriminator is neither a PCL or TICLOS thing then it generates an error
 otherwise it returns nil.
"
  `(defun ,fname (,discriminator ,@args)
     ,documentation
     ,declarations
     (if (and (pcl-p) (pcl:generic-function-p ,discriminator))
	,(if pcl
	     pcl
	     (if error-p '(ferror nil "Don't know how to do this.") nil)
	 )
	 (if (and (ticlos-p) (ticlos:generic-function-p ,discriminator))
	    ,(if ticlos
		 ticlos
		 (if error-p '(ferror nil "Don't know how to do this.") nil)
	     )
	    ,(if error-p
		`(ferror nil "Something wrong with ~S." ,discriminator)
		 nil
	     )
	 )
     )
   )
)

(defmacro defsafe-simple
     (fname (discriminator &rest args)
      &key (documentation nil) (both nil) (pcl nil) (ticlos nil) (error-p t)
      (declarations nil)
     )
"Defines a function called FName, which should execute
 (PCL:PCL discriminator ,@args) if Discriminator
 is a PCL thing or (TICLOS:TICLOS discriminator ,@args)
 if it is a TICLOS instance.  If Both is specified instead
 of either of the above, then both functions will have the
 same name, though different packages.  If Error-p is true
 and Discriminator is neither a PCL or TICLOS thing then
 it generates an error otherwise it returns nil.
 (defsafe-simple class-named-safe (x) :both class-named)
 ~=
 (defsafe class-named-safe (x)
   :pcl (pcl:class-named x) 
   :ticlos (ticlos:class-named x))
"
  (let ((real-pcl (intern (symbol-name
			    (or both pcl (and (not error-p) 'identity)))
			  'pcl))
	(real-ti  (intern (symbol-name
			    (or both ticlos (and (not error-p) 'identity)))
			  'ticlos))
       )
      `(defun ,fname (,discriminator ,@args)
	 ,documentation
	 ,declarations
	  (if ,(if (or pcl both (not error-p))
		  `(and (pcl-p) (iwmc-class-p-safe ,discriminator))
		   nil
	       )
	       (,real-pcl ,discriminator ,@args)
	       (if ,(if (or ticlos both (not error-p))
		       `(and (ticlos-p) (ticlos-instance-p ,discriminator))
		        nil
		    )
		    (,real-ti ,discriminator ,@args)
		   ,(if error-p
		       `(ferror nil "Something wrong with ~S." ,discriminator)
			nil
		    )
	       )
	  )
       )
  )
)

(defmacro defsafe-gf-simple
     (fname (discriminator &rest args)
      &key (documentation nil) (both nil) (pcl nil) (ticlos nil) (error-p t)
      (declarations nil)
     )
"Defines a function called FName, which should execute
 (PCL:PCL discriminator ,@args) if Discriminator
 is a PCL GF or (TICLOS:TICLOS discriminator ,@args)
 if it is a TICLOS GF.  If Both is specified instead
 of either of the above, then both functions will have the
 same name, though different packages.  If Error-p is true
 and Discriminator is neither a PCL or TICLOS thing then
 it generates an error otherwise it returns nil.
 (defsafe-gf-simple class-named-safe (x) :both class-named)
 ~=
 (defsafe-gf generic-function-name-safe (x)
   :pcl (pcl:generic-function-name x) 
   :ticlos (ticlos:generic-function-name x))
"
  (let ((real-pcl (intern (symbol-name
			    (or both pcl (and (not error-p) 'identity)))
			  'pcl))
	(real-ti  (intern (symbol-name
			    (or both ticlos (and (not error-p) 'identity)))
			  'ticlos))
       )
      `(defun ,fname (,discriminator ,@args)
	 ,documentation
	 ,declarations
	  (if ,(if (or pcl both (not error-p))
		  `(and (pcl-p) (pcl:generic-function-p ,discriminator))
		   nil
	       )
	       (,real-pcl ,discriminator ,@args)
	       (if ,(if (or ticlos both (not error-p))
		       `(and (ticlos-p)
			     (ticlos:generic-function-p ,discriminator)
			)
		        nil
		    )
		    (,real-ti ,discriminator ,@args)
		   ,(if error-p
		       `(ferror nil "~S is not a generic function."
				,discriminator
			)
			nil
		    )
	       )
	  )
       )
  )
)

(defmacro defsafe-slotd
     (fname (discriminator &rest args)
      &key (documentation nil) (both nil) (pcl nil) (ticlos nil) (error-p t)
      (declarations nil)
     )
"Defines a function called FName, which should execute
 (PCL:PCL discriminator ,@args) if Discriminator
 is a PCL slot-descriptor or (TICLOS:TICLOS discriminator ,@args)
 if it is a TICLOS slot-descriptor.  If Both is specified instead
 of either of the above, then both functions will have the
 same name, though different packages.  If Error-p is true
 and Discriminator is neither a PCL or TICLOS thing then
 it generates an error otherwise it returns nil.
 eg: (defsafe-slotd slotd-name-safe (x)
       :pcl slotd-name
       :ticlos slot-description-name)
"
  (let ((real-pcl (intern (symbol-name
			    (or both pcl (and (not error-p) 'identity)))
			  'pcl))
	(real-ti  (intern (symbol-name
			    (or both ticlos (and (not error-p) 'identity)))
			  'ticlos))
       )
      `(defun ,fname (,discriminator ,@args)
	 ,documentation
	 ,declarations
	  (if ,(if (or pcl both (not error-p))
		  `(and (pcl-p)
			(typep ,discriminator 'pcl:standard-slot-description)
		   )
		   nil
	       )
	       (,real-pcl ,discriminator ,@args)
	       (if ,(if (or ticlos both (not error-p))
		       `(and (ticlos-p)
			     (typep ,discriminator 'ticlos:slot-description)
			)
		        nil
		    )
		    (,real-ti ,discriminator ,@args)
		   ,(if error-p
		       `(ferror nil "Something wrong with ~S." ,discriminator)
			nil
		    )
	       )
	  )
       )
  )
)

(defun get-fef-from-object (x)
"Tries to extract a fef from an object.  X could be a method or a
 generic function, which is not itself a fef, but points to one.
 This is used by code that grovles over the fef for one reason or other.
"
  (flet ((pcl ()
	   (typecase x
	     (pcl:standard-method
	      (get-fef-from-object (method-function-safe x))
	     )
	     (pcl:standard-generic-function
	      (first (si:convert-closure-to-list x))
	     )
	     (compiled-function x)
	     (closure (first (si:convert-closure-to-list x)))
	     (otherwise (sys:fdefinition-safe x))
	   )
	 )
	 (ticlos ()
	   (typecase x
	     (compiled-function x)
	     (ticlos:standard-method (method-function-safe x))
	     (ticlos:standard-generic-function
	      (ticlos:generic-function-discriminator-code x)
	     )
	     (closure (first (si:convert-closure-to-list x)))
	     (otherwise (sys:fdefinition-safe x))
	   )
	 )
	)
    (if (pcl-p)
	(or (pcl) (if (ticlos-p) (or (ticlos) (list nil)) (list nil)))
	(if (ticlos-p) (or (ticlos) (list nil)) (list nil))
    )
  )
)

(defun slotd-p-safe (slotd)
"Is true if SlotD is a slot descriptor object."
  (or (and (pcl-p)
	   (typep slotd 'pcl:standard-slot-description)
      )
      (and (ticlos-p)
	   (typep slotd 'ticlos:slot-description)
      )
  )
)

(defsafe-slotd slotd-name-safe (slotd)
  :documentation "Returns the name of the slot described by SlotD."
  :ticlos slot-definition-name
  :pcl slotd-name
)

(defsafe-simple slot-value-safe (instance slot)
  :documentation
  "Returns the value of the slot named Slot in the instance Instance."
  :both slot-value
)

(defsafe-simple class-of-safe (thing)
  :documentation "Returns the class of Thing."
  :both class-of
)

(defun class-of-gf-safe (thing)
"Returns the class of Thing."
  (if (and (ticlos-p) (ticlos:generic-function-p thing))
      (ticlos:class-of thing)
      (if (and (pcl-p) (pcl:generic-function-p thing))
	  (pcl:class-of-generic-function thing)
	  (ferror nil "~S is not a generic function." thing)
      )
  )
)

(defsafe-simple method-function-safe (Method)
  :documentation "Returns the method function of Method."
  :both method-function
)

(defsafe-slotd slotd-initform-safe (slotd)
  :documentation "Returns the initform of the slot descriptor SlotD."
  :pcl slotd-initform
  :ticlos slot-definition-initform
)

(defsafe-slotd slotd-accessors-safe (slotd)
  :documentation "Returns the accessors of the slot descriptor SlotD."
  :pcl slotd-accessors ;;; !!!!! JPR.
  :ticlos slot-definition-writers
)

(defsafe-slotd slotd-readers-safe (slotd)
  :documentation "Returns the readers of the slot descriptor SlotD."
  :pcl slotd-readers
  :ticlos slot-definition-readers
)

(defsafe-slotd slotd-type-safe (slotd)
  :documentation "Returns the type of the slot descriptor SlotD."
  :pcl slotd-type
  :ticlos slot-definition-type
)

(defsafe-slotd slotd-allocation-safe (slotd)
  :documentation "Returns the allocation of the slot descriptor SlotD."
  :pcl slotd-allocation
  :ticlos slot-definition-allocation
)

(defsafe-slotd slotd-initargs-safe (slotd)
  :documentation "Returns the initargs of the slot descriptor SlotD."
  :pcl slotd-initargs
  :ticlos slot-definition-initargs
)

(defsafe class-local-slots-safe (class)
  :documentation "Returns the local slots of the class Class."
  :ticlos (ticlos:class-direct-slots class)
  :pcl (let ((others (mapcar #'pcl:slotd-name
			     (apply #'append
				      (mapcar #'pcl:class-direct-slots
					      (rest (class-precedence-list-safe
						      class
						    )
					      )
				      )
			     )
		     )
	     )
	    )
	    (remove-if #'(lambda (slotd) (member (pcl:slotd-name slotd) others))
		       (pcl:class-direct-slots class)
	    )
       )
)

(defun all-shared-slots (class)
  (let ((cpl (class-precedence-list-safe class t)))
       (1let* ((slotds
	      (mapcar #'(lambda (a-class)
			  (mapcar #'first
				  (getf (ticlos:class-description-plist
					  (ticlos:class-description a-class)
					)
					'ticlos:class-slot-alist
				  )
			  )
			)
		        cpl
	      )
	     )
	    )
	    (1values* (apply #'1append* slotds) slotds cpl)
       )
  )
)

(defun class-instance-slots-safe (class)
"Returns a list of the Local slots posessed by instances of the class CLASS."
  (if (typep class 'ticlos-instance)
      (let ((descr (ticlos:class-description class)))
	   (let ((shared (all-shared-slots class)))
	        (remove-if
		  #'(lambda (slotd)
		      (member (ticlos:slot-name slotd) shared)
		    )
	            (getf (ticlos:class-description-plist descr)
			  'ticlos:all-slots
		    )
		)
	   )
      )
      (let ((slots (pcl:class-slots class)))
	   (remove-if
	     #'(lambda (slotd) (equal (slotd-allocation-safe slotd) :class))
	       slots
	   )
      )
  )
)

(1defun* group-into-classes (pairs)
  (1let* ((classes nil))
       (1loop* for (class slotd) in pairs do
	     (1let* ((entry (1or* (1assoc* class classes)
			      (1let* ((new (1list* class)))
				   (1push* new classes)
				   new
			      )
			  )
		  )
		 )
	         (1push* slotd (1rest* entry))
	     )
       )
       classes
  )
)

(defun class-instance-slots-grouped-safe (class)
"Returns a list of the Local slots posessed by instances of the class CLASS."
  (2Class-Slots-Grouped-Safe-1* class '1remove-if*)
)

(defun class-non-instance-slots-grouped-safe (class)
"Returns a list of the Local slots posessed by instances of the class CLASS."
  (2Class-Slots-Grouped-Safe-1* class '1remove-if-not*)
)

(defun class-slots-grouped-safe-1 (class function)
"Returns a list of the Local slots posessed by instances of the class CLASS."
  (if (typep class 'ticlos-instance)
      (let ((descr (ticlos:class-description class)))
	   (multiple-value-bind (shared ignore cpl) (all-shared-slots class)
	     (1let* ((slots
		     (1funcall* function
		       #'(lambda (slotd)
			   (member (ticlos:slot-name slotd) shared)
			 )
		       (getf (ticlos:class-description-plist descr)
			     'ticlos:all-slots
		       )
		     )
		   )
		  )
	          (1let* ((ungrouped
			  (1loop* for slotd in slots collect
			    (1multiple-value-bind* (1ignore* cl)
				(find-slot-in-classes
				  (2Slotd-Name-Safe* slotd)
				  cpl
				)
			      (1list* cl slotd)
			    )
			  )
			)
		       )
		       (group-into-classes ungrouped)
		  )
	     )
	   )
      )
      (let ((slots (pcl:class-slots class)))
	   (1list* (1list* class
		      (1funcall* function
			#'(lambda (slotd)
			    (equal (slotd-allocation-safe slotd) :class)
			  )
			slots
		      )
		 )
	   )
      )
  )
)

(defun class-non-instance-slots-safe (class)
"Returns a list of the Shared slots posessed by instances of the class CLASS."
  (if (typep class 'ticlos-instance)
      (let ((descr (ticlos:class-description class)))
	   (let ((shared (all-shared-slots class)))
	        (remove-if-not
		  #'(lambda (slotd)
		      (member (ticlos:slot-name slotd) shared)
		    )
	            (getf (ticlos:class-description-plist descr)
			  'ticlos:all-slots
		    )
		)
	   )
      )
      (let ((slots (pcl:class-slots class)))
	   (remove-if-not
	     #'(lambda (slotd) (equal (slotd-allocation-safe slotd) :class))
	       slots
	   )
      )
  )
)

(defsafe-simple class-finalized-p-safe (class)
  :documentation "Is true if Class has been finalized."
  :both class-finalized-p
)

(defsafe-simple class-precedence-list-safe-1 (class)
  :documentation "Returns the class precedence list of a finalized class."
  :both class-precedence-list
)

(defun class-precedence-list-safe (class &optional (undefined-ok nil))
"Returns the class precedence list of a class.  If undefined-ok is true then
 it's alright to return just the names of undefined classes, rather than the
 class objects themselves.  The class does not have to be finalized for this
 function to work.
"
  (1if* (1typep* class 'ticlos:flavor-class)
      (1loop* for class in (sys:flavor-depends-on-all
			   (1get* (clos:class-name class) 'sys:flavor)
			 )
	    for cl = (2Class-Named-Safe* class t)
	    when (1and* (1not* undefined-ok) (1not* cl))
	    do (1ferror* nil "Undefined component class ~S in ~S" cl class)
	    when cl
	    collect cl
      )
      (if (class-finalized-p-safe class)
	  (class-precedence-list-safe-1 class)
	  (remove nil (mapcar #'(lambda (a-class)
				  (or (and (class-p-safe a-class)
					   a-class
				      )
				      (class-named-safe a-class t)
				      (and undefined-ok a-class)
				  )
				)
				(cons class (class-local-supers-safe class))
		      )
	  )
      )
  )
)

(1defun* class-precedence-list-1 (class undefined-ok)
  (1cons* class 
        (1apply* #'1append*
		(1mapcar* #'(lambda (class)
			     (1if* (2Class-P-Safe* class)
				 (class-precedence-list-1
				   class undefined-ok
				 )
				 class
			     )
			   )
			   (mapcar #'(lambda (a-class)
				       (or (and (class-p-safe a-class)
						a-class
					   )
					   (class-named-safe a-class t)
					   (and undefined-ok a-class)
				       )
				     )
				     (class-local-supers-safe class)
			   )
		)
	)
  )
)

(defun class-precedence-list-safe (class &optional (undefined-ok nil))
"Returns the class precedence list of a class.  If undefined-ok is true then
 it's alright to return just the names of undefined classes, rather than the
 class objects themselves.  The class does not have to be finalized for this
 function to work.
" 
  (if (class-finalized-p-safe class)
      (class-precedence-list-safe-1 class)
      (uniquify (class-precedence-list-1 class undefined-ok) nil)
  )
)

(defun CLOS-method-name (method)
"Returns the name of a clos method."
  (if (typep method 'ticlos:null-combined-method)
      (function-name
	(ticlos:*method-function (ticlos:method-forwarded-method method))
      )
      (function-name (method-function-safe method))
  )
)


(defun coerce-to-class (class)
"Makes sure that class is a class."
  (if (class-p-safe class)
      class
      (if (consp class)
	  (if (equal 'eql (first class))
	      class
	      (list (first class) (coerce-to-class (second class)))
	  )
	  (coerce-to-class (class-named-safe class))
      )
  )
)

(defsafe method-parameter-specializers-safe (method)
  :documentation "Returns the parameter specializers of method Method."
  :ticlos (mapcar #'coerce-to-class
		  (ticlos:method-parameter-specializers method)
	  )
  :pcl    (mapcar #'coerce-to-class (pcl:method-type-specifiers method))
)

(defun method-primary-p-safe (method)
"Is true if Method is a primary method."
  (or (equal nil (method-qualifiers-safe method))
      ;;;; !!!! Allow for PCL reader/writer methods.
      (equal :internal
	     (first (function-name (method-function-safe method)))
      )
  )
)

(defun is-specialised-by (method class looking-for-class)
"Is true if Method is specialized by Class, i.e. Class is something like
 bottle, where bottle is built on container and method has a name something like
 (method fill-me ((me container) (with t))).
 Looking for Class is the class we are really interested in.  This means that,
 although container is built on T, we are not interested in methods that have
 t as specializers.
"
  (and (member class (method-parameter-specializers-safe method))
       (remove nil (mapcar #'(lambda (x)
			       (and (1not* (1consp* x)) ;;; eql specialized.
				    (not (equal t (Class-Name-Safe x)))
				    (my-subtypep looking-for-class x)
			       )
			     )
			     (method-parameter-specializers-safe method)
		   )
       )
  )
)

(defun is-specialised-by-components (method class)
"Is true if Method is specialized by Class, i.e. Class is something like
 bottle, where bottle is built on container and method has a name something like
 (method fill-me ((me container) (with t))).  If it is true then it returns
 the method, otherwise nil.
"
  (let ((matches
	  (remove-if-not #'(lambda (cl) (is-specialised-by method cl class))
			 (class-precedence-list-safe class)
	  )
	)
       )
       (if matches method nil)
  )
)

(defun actual-method-entry (x)
"Returns the method entry we are interested in."
  (if (consp (first x))
      (third x)
      (actual-method-entry (second x))
  )
)

(defsafe-gf generic-function-methods-safe (generic-function)
  :documentation "Returns the methods associated with a generic function."
  :pcl (pcl:generic-function-methods generic-function)
  ;;; Use this for because ticlos:generic-function-methods doesn't get
  ;;; set methods.
  :ticlos (let ((list (clos:slot-value generic-function 'ticlos:method-list)))
	       ;;; Put in catch error to oprotect from During Transport
	       ;;; of Self-Ref-Pointer error.
	       (apply #'append (mapcar 'get-method-from-spec list))
	  )
;  :ticlos (ticlos:generic-function-methods generic-function)
)


(defun ticlos-class-direct-generic-functions
       (a-class &optional (top-class a-class))
"Returns the list of generic functions that name the methods directly
 specialized by a-class.  Top-class is the class we're actually interested in
 since this might be different from a-class (see ticlos-class-direct-methods).
"
  (let ((methods (ticlos-class-direct-methods a-class top-class)))
       (uniquify (mapcar #'method-generic-function-safe methods) nil)
  )
)

(defun get-method-from-spec (spec)
"Given a method spec returns the actual method."
  (declare (optimize (safety 0)))
  (if (and (consp spec) (consp (first spec))
	   (not (equal 'eql (first (first spec))))
      )
      (if (equal 'ticlos:method (first (first spec)))
	  (if (equal :Combined (third (first spec)))
	      nil
	      (list (third spec))
	  )
	  (Get-Method-From-Spec (rest (rest (rest spec))))
      )
      (if (consp spec)
	  (remove nil (apply #'append (mapcar 'get-method-from-spec spec)))
	  nil
      )
  )
)

(defun class-all-slots-safe (class)
"Returns a list of the slotds for all of the slots in Class."
  (let ((slots (mapcar #'class-local-slots-safe
		       (class-precedence-list-safe class)
	       )
	)
       )
       (let ((result nil))
	    (loop for slotd in (apply #'append slots)
		  unless (member slotd result
				 :test #'(lambda (x y)
					   (eq (slotd-name-safe x)
					       (slotd-name-safe y)
					   )
					 )
			 )
		  do (push slotd result)
	    )
	    result
       )
  )
)

(defun get-gfs-for-slot (slotd)
  (append (slotd-accessors-safe slotd)
	  (slotd-readers-safe slotd)
  )
)

(defun get-gfs-for-slots-of (class)
  (let ((slotds (class-all-slots-safe class)))
       (mapcar #'get-gf-for-slot-function
	       (uniquify (apply #'append (mapcar 'get-gfs-for-slot slotds)) nil)
       )
  )
)

(defun ticlos-class-direct-methods (a-class &optional (top-class a-class))
"Returns the list of methods directly specialized by a-class.  Top-class
 is the class we're actually interested in since this might be different
 from a-class during checks of subclasses (e.g. show-all-clos-methods).
 Top-class allows is-specialised-by to know which class we are actually
 interested in.
"
  (let ((fns (append (getf (ticlos:class-description-plist
			     (ticlos:class-description a-class)
			   )
			   'ticlos:direct-generic-functions
		     )
		     (get-gfs-for-slots-of a-class)
	     )
	)
       )
       (let ((locals nil))
            (loop for fn in (mapcar #'ucl:first-if-list fns) do
		  (loop for meth in (generic-function-methods-safe fn)
			when (and meth
				  (is-specialised-by meth a-class top-class)
			     )
			do (push meth locals)
		  )
	    )
	    locals
       )
  )
)


(defsafe method-lambda-list-safe (method)
  :documentation "Returns the method lambda list of a method."
  ;;; Protect agains self-ref-transport bug.
  :ticlos (catch-error (ticlos:method-lambda-list method) nil)
  :pcl (pcl:method-arglist method)
)

(defun method-type-specifiers-safe (method)
  "Returns the type specifiers of a method."
  (method-lambda-list-safe method)
)

(defsafe method-docs-safe (method)
  :documentation "Returns the documentation of a method."
  :pcl (slot-value-safe method 'documentation)
  :ticlos (documentation method)
)

(defun path-string-1 (path types)
"Is given a spec that might have been got from the :source-file-name
 property of something and the type of definition path to look for (e.g. defun).
 It returns either the string-for-printing of the pathname or the string
 \"Not Defined\".
"
  (if path
      (format nil "~a"
        (send (if (consp path)
		  (loop for type in (if (listp types) types (list types))
			when (assoc type path :test #'eq)
			return (second (assoc type path :test #'eq))
		  )
		  path
	      )
	      :string-for-printing
	)
      )
      '(3:Font* 2 "Not Defined")
  )
)

(defun path-string (name type)
"Returns the string for the source file pathname of Name of type Type.  Type
 is something like Defun or Defflavor.
"
  (let ((path (si:function-spec-get name :source-file-name)))
       (Path-String-1 path type)
  )
)

(defun path-string-safe (class)
"Returns a string for the source file name of a class."
  (Path-String (class-name-safe class)
	       (if (iwmc-class-p-safe class)
		   'pcl:class
		   '(ticlos:defclass ticl:defflavor)
	       )
  )
)

(defun is-for (entry method)
"Is true if the method table entry Entry is the entry for Method."
  (if (consp (first entry))
      (member method entry)
      (Is-For (second entry) method)
  )
)

(defun ticlos-method-plist (method)
"Gets the plist from the method table entry for Method."
  (let ((gf (Method-Generic-Function-Safe method)))
       (let ((entry (clos:slot-value gf 'ticlos:method-list)))
	    (loop for meth in entry
		  when (is-for meth method)
		  return (is-for meth method)
	    )
       )
  )
)


(defsafe method-path-string-safe (method)
  :documentation "Returns the pathname string for the source file of Method."
  :ticlos (let ((path (getf (Ticlos-Method-Plist method) :Source-File-Name)))
	       (Path-String-1 path 'defun)
	  )
  :pcl (1if* (1compiled-function-p* (method-function-safe method))
	   (Path-String-1
	     (si:function-spec-get
	       (function-name (method-function-safe method)) :source-file-name
	     )
	     'defun
	   )
	   '(3:Font* 2 "Not Defined")
       )
)

(defsafe class-direct-methods-safe (class &optional (top-class class))
  :documentation 
"Returns the list of methods that have been defined to specialise Class
 directly.
"
  :pcl (pcl:class-direct-methods class)
  :ticlos (Ticlos-Class-Direct-Methods class top-class)
)


(defsafe class-default-initargs-safe (class)
  :documentation "Returns the default initargs of the class Class."
  :pcl (pcl:class-default-initargs class)
  :ticlos
      (mapcar #'(lambda (x) (list (first x) (third x)))
	     (getf (ticlos:class-description-plist
		     (ticlos:class-description class)
		   )
		   :Default-Initargs
	     )
      )
)


(defsafe class-direct-generic-functions-safe (class &optional (top-class class))
  :documentation 
"Gets a list of all of the generic functions that name methods, which have been
 defined directly on this class.  Top class is the class we are actually
 interested in.
"
  :pcl    (pcl:class-direct-generic-functions    class)
  :ticlos (Ticlos-Class-Direct-Generic-Functions class top-class)
)

(defun method-arglist-safe (method)
"Returns the method arglist of a method."
  (1let* ((args (method-lambda-list-safe method))
       (values (catch-error
		 (second (multiple-value-list
			   (arglist (method-function-safe method))
			 )
		 )
		 nil
	       )
       )
      )
      (values args values)
  )
)

(defsafe-simple method-qualifiers-safe (method)
  :documentation "Returns the qualifiers (e.g. :Before) of Method."
  :both method-qualifiers
)

(defsafe unparse-specializers-safe (method)
  :documentation 
"Unparses the specializers of Method.  Returns the specializers list and T
 if the method is combined.
"
  :declarations (declare (values specializers combined-method-p))
  :pcl (values (pcl:unparse-specializers method)
	       (method-qualifiers-safe method)
       )
  :ticlos (let (;(name (function-name (ticlos:method-function method)))
		(specializers (ticlos:method-parameter-specializers method))
	       )
	       (values ;;;(if (fourth name) (fourth name) (third name))
		       (loop for spec in specializers
			     collect (if (Class-P-Safe spec)
					 (Class-Name-Safe spec)
					 spec
				     )
		       )
		       (method-qualifiers-safe method)
	       )
          )
)

(defun class-named-safe (class &optional (noerrorp nil))
"Returns the class named by the name Class.  If noerrorp is true then no error
 is signaled and nil is returned if Class is does not, in fact, name a class.
"
  (1if* (1and* (2Ticlos-P*) (ticlos:class-named class t))
      (ticlos:class-named class t)
      (1if* (1and* (2Pcl-P*) (pcl:find-class class nil))
	  (pcl:find-class class nil)
	  (1or* (and (2Ticlos-P*) (ticlos:class-named class noerrorp))
	      (1and* (2Pcl-P*)   (pcl:find-class class (1not* noerrorp)))
	      (1and* (1not* noerrorp)
		    (1ferror* nil "~S does not name a class." class)
	      )
	  )
      )
  )
)

(defsafe-simple class-name-safe (class)
  :documentation "Returns the name of the class Class."
  :both class-name
)

(defsafe-simple subclassp-safe (x y)
  :documentation "Is true if x is a subclass of y"
  :both subclassp
  :error-p nil
)

(1defun* pcl-class-p (x)
"True if x is a pcl class object."
  (or (typep x 'pcl:standard-class)
;      (typep x 'pcl:structure-class)
      (typep x 'pcl:built-in-class)
  )
)

(defsafe class-p-safe (x)
  :documentation
"Is true if x is some sort of class object.  This could be a built-in class
 standard or structure class.
"
  3:Pcl* (2Pcl-Class-P* x)
  :ticlos (typep x 'clos:class)
  :error-p nil
)

(defun generic-function-p-safe (x)
"Is true if x is a generic function."
  (or (and (pcl-p) (pcl:generic-function-p x))
      (and (ticlos-p) (ticlos:generic-function-p x))
  )
)

(defun function-generic-function-safe (function)
"Returns the generic function object for the function function, where
 function is a fef.
"
  (if (and (pcl-p) (pcl::generic-function-p function))
      function
      (if (and (ticlos-p) (ticlos:generic-function-p function))
	  (if (typep function 'ticlos:generic-function)
	      function
	      ;;; It is a fef.
	      (ticlos:get-generic-function-object function)
	  )
	  nil
      )
  )
)

(defun name-safe (x)
"Reads the name slot of x if it has one.  This returns nil if X either isn't
 an instance or doesn't have a Name slot.
"
  (or (catch-error (Slot-Value-Safe x 'pcl:name)    nil)
      (catch-error (Slot-Value-Safe x 'ticlos:name) nil)
      (catch-error (generic-function-name-safe x) nil)
  )
)

(defsafe-gf-simple argument-precedence-order-safe (gf)
  :documentation
    "Returns the argument precedence order of the generic-function GF."
  :both generic-function-argument-precedence-order
)

(defsafe-gf-simple generic-function-method-class-safe (gf)
  :documentation "Returns the method class of a generic function."
  :both generic-function-method-class
)

(defsafe-gf-simple generic-function-method-combination-safe (gf)
  :documentation "Returns the method combination type of a generic function."
  :both generic-function-method-combination
)

(defsafe-gf-simple generic-function-declarations-safe (gf)
  :documentation "Returns any declarations made for the generic function GF."
  :both generic-function-declare
)

(defsafe class-local-supers-safe (class)
  :documentation
"Returns a list of the local superclasses of the class class.  If a class is
 undefined then it returns a symbol that names the undefined class.
"
  :pcl (pcl:class-local-supers class)
  :ticlos
  ;;; This should be fixed when Forward-Referenced-Class is impemented.
    (mapcar #'(lambda (x) (or (class-named-safe x t) x))
	    (ticlos:class-direct-supers class)
    )
;    (let ((old #'ticlos:class-named))
;	 (letf ((#'ticlos:class-named
;		 #'(lambda (name &optional no-error-p environment)
;		     (ignore no-error-p)
;		     (or (funcall old name t environment) name)
;		   )
;		)
;	       )
;	       (ticlos:class-direct-superclasses class)
;	 )
;    )
)

(defsafe-simple method-generic-function-safe (method)
  :documentation "Returns the generic function that names the method Method."
  :both method-generic-function
)

(defsafe-simple class-direct-subclasses-safe (class)
  :documentation
  "Returns a list of the classes that are direct subclasses of the class Class."
  :both class-direct-subclasses
)

(defsafe-simple class-direct-superclasses-safe (class)
  :documentation
  "Returns a list of the classes that are direct superclasses
   of the class Class."
  :both class-direct-superclasses
)

(defun iwmc-class-p-safe (something)
"Is true if something is iwmc-class-p (instance-with-metaclass-p)."
  (if (fboundp 'pcl:iwmc-class-p)
      (pcl:iwmc-class-p something)
      nil
  )
)


(defparameter *cached-ticlos-class-names* nil
"A list of the ticlos class names of the classes that have been found.  This
 Gets set if the user opts to do an exhaustive search for all ticlos classes.
"
)

(defun all-class-names (&optional (cheep-p nil))
"Returns a list of all of the class names.  This is easy in the PCL case,
since all defined classes are cached in a hash table.  TICLOS, however,
does not have a global list of all classes lso they have to be found by
doing a do-all-symbols, which is fantastically expensive.  Thus, before the
user opts for this he is prompted and, if he ever does it the result is
cached so that this computation can be avoided a second time.
"
  (declare (special pcl:*class-name-hash-table*))
  ;;; There doesn't seem to be a way to get all class names in TICLOS.
  (append (if (boundp 'pcl:*class-name-hash-table*)
	      (maphash-return #'(lambda (key &rest ignore) key)
			      pcl:*class-name-hash-table*
	      )
	      nil
	  )
	  (if (and *Cached-Ticlos-Class-Names*
		   (or cheep-p
		       (mouse-y-or-n-p
			 "Use previously cached list of TICLOS classes?"
		       )
		   )
	      )
	      (maphash-return #'(lambda (key &rest ignore) key)
			      *cached-ticlos-class-names*
	      )
	      (if (And (Ticlos-P)
		       (not cheep-p)
		       (mouse-y-or-n-p
			 (format nil "Getting all named TICLOS classes ~
                                      involves looking at all symbols. ~&~
                                      This may take quite a while.  ~
                                      Go ahead anyway?"
			 )
		       )
		  )
		  (progn
		    (if (not *Cached-Ticlos-Class-Names*)
			(setq *Cached-Ticlos-Class-Names*
			      (make-hash-table)
			)
			nil
		    )
		    (Do-all-symbols (sym nil)
		      (if (get sym 'ticlos:class-def)
			  (setf (gethash
				  (Class-Name-Safe (get sym 'ticlos:class-def))
				  *Cached-Ticlos-Class-Names*
				)
				(get sym 'ticlos:class-def)
			  )
			  nil
		      )
		    )
		    (maphash-return #'(lambda (key &rest ignore) key)
				    *cached-ticlos-class-names*
	            )
		  )
		  nil
	      )
	  )
  )
)

(Defun generic-function-name-safe (gf)
  :documentation "Returns the name of the generic function GF."
  (or (catch-error (clos:generic-function-name gf) nil)
      (catch-error (pcl:generic-function-name gf) nil)
      (clos:generic-function-name gf)
      (pcl:generic-function-name gf)
  )
)

;-------------------------------------------------------------------------------

(deftype ticlos-instance ()
 "The type which is true for instances of TICLOS classes." 
 `(satisfies ticlos-instance-p)
)

(defun any-sort-of-clos-instance-p (x)
"Is true if x is any sort of clos instance, be it PCL or TICLOS."
  (or (ticlos-instance-p x)
      (iwmc-class-p-safe x)
  )
)

(deftype any-sort-of-clos-instance ()
  "The type which is true for instances of either TICLOS or PCL classes."
 `(satisfies any-sort-of-clos-instance-p)
)

(defsafe standard-method-p-safe (x)
  :documentation "Is true if x is a standard method."
  :pcl    (typep x 'pcl:standard-method)
  :ticlos (Typep x 'ticlos:standard-method)
)

(defsafe standard-generic-function-p-safe (x)
  :documentation "Is true if x is a standard generic function."
  :pcl    (typep x 'pcl:standard-generic-function)
  :ticlos (Typep x 'ticlos:standard-generic-function)
)

(defsafe directly-standard-generic-function-p-safe (x)
  :documentation
"Is true if x is a standard generic function and not an instance of some
 specialisation of standard generic function.
"
  :pcl    (and (closurep x) (typep x 'pcl:standard-generic-function))
  :ticlos (equal (Type-of x) 'ticlos:standard-generic-function)
)


(defun any-sort-of-clos-method-p (x)
"Is true if x is either a PCL or a TICLOS method object."
  (if (iwmc-class-p-safe x)
      (typep x 'pcl:standard-method) ;;; JPR !!!
      (if (Ticlos-P)
	  (Typep x 'ticlos:method)
	  nil
      )
  )
)

(deftype any-type-of-clos-method ()
"The type that defines either PCL or TICLOS method objects."
  `(satisfies any-sort-of-clos-method-p)
)

(defun any-sort-of-clos-gf-p (x)
"Is true if x is either a PCL or a TICLOS generic function object."
  (or (and (Pcl-P)
	   (typep x 'pcl:standard-generic-function)
      )
      (and (Ticlos-P)
	   (Typep x 'ticlos:generic-function)
      )
  )
)

(deftype any-type-of-clos-gf ()
"The type that defines either PCL or TICLOS generic function objects."
  `(satisfies any-sort-of-clos-gf-p)
)

(defun fef-of-gf-p (x)
"Is true if x is the fef that defines a generic function."
  (or (and (Pcl-P) (typep x 'pcl:standard-generic-function))
      (and (Ticlos-P) (Function-Generic-Function-Safe x))
  )
)

(deftype fef-of-gf ()
"The type that is true for fefs that are for generic functions."
  `(satisfies fef-of-gf-p)
)

(deftype any-sort-of-clos-class ()
 "The type which is true for all CLOS classes." 
 `(satisfies class-p-safe)
)

(deftype any-sort-of-clos-slotd ()
 "The type which is true for all CLOS slot descriptors." 
 `(satisfies slotd-p-safe)
)

(defun is-a-class-name-not-flavor-class (name)
"Is true if Name is a symbol that names a class that does not have a flavor
 class declaration.  This is useful because it lets us view flavors as flavors
 even though they have flavor-class declarations.
"
  (and (typep name 'symbol)
       (fboundp 'class-named-safe)
       (funcall 'class-named-safe name t)
       (or (not (get name 'si:flavor))
	   (not (type-specifier-p 'ticlos:flavor-class))
	   (and (type-specifier-p 'ticlos:flavor-class)
		(not (typep (funcall 'class-named-safe
				     name t
			    )
			    'ticlos:flavor-class
		     )
		)
	   )
       )
  )
)

(defsafe-simple method-specializers-safe (method)
  :documentation "Returns the list of method specializers for Method."
  :ticlos method-specializers
  :pcl method-type-specifiers
)

(defsafe-gf-simple compute-applicable-methods-safe (gf methods)
  :Documentation "Returns a list of the applicable methods for a generic
function GF from a list of methods Methods."
  :Both compute-applicable-methods
)

(defsafe-gf compute-effective-method-safe
	       (gf method-combination methods)
  :Documentation
"Computes the code for the effective method of a generic function, using a
 particular type of method combination and a list of effective methods.
"
  :Ticlos (ticlos:compute-effective-method gf method-combination methods)
  :Pcl (pcl:compute-effective-method-body gf methods)
)

(defsafe-simple class-prototype-safe (class)
  :Documentation "Returns the class prototype oc a class."
  :Both class-prototype
)

(defvar *method-combination-indent-increment* 2
"The number of spaces to indent by each time the method combination display
 throws a newline and tabs for a sublist.
"
)

(defun post-process-combined-method (form &optional (indent 0))
"Takes a form that (when ground) would appear as follows:
   (progn (sys:%apply-method before method 1)
          (sys:%apply-method before method 2)
          ...
          (sys:%apply-method before method n)
          (multiple-value-prog1
            (sys:%apply-method primary method)
            (sys:%apply-method after method 1)
            (sys:%apply-method after method 2)
            ...
            (sys:%apply-method after method n)))

 and transforms it into an inspector compatible item list that will
 look like:
   (progn <before method 1>
          <before method 2>
          ...
          <before method n>
          (multiple-value-prog1
            <primary method>
            <after method 1>
            <after method 2>
            ...
            <after method n>))

 where the things labeled <...> will appear as mouse-sensitive items of the
 type show-clos-method.m  The grindingf that this function does is very
 primitive.  Sub-lists are started on new lines indented by a few spaces
 (*method-combination-indent-increment*).
"
  (declare (special *this-line* *all-lines*))
  (if (consp form)
      (if (member (first form)
		  '(ticlos:call-method sys:%apply-method pcl:call-method)
	  )
	  (let ((meth
		  (if (equal (first form) 'sys:%apply-method)
		      (tv:method-from-method-function-safe
			(fdefinition (second (second form)))
		      )
		      (second form)
		  )
		)
	       )
	       (push (reverse *this-line*) *all-lines*)
	       (setq *this-line* nil)
	       (push `(,tv:*space-format* ,indent) *this-line*)
	       (push (if meth
			 `(:item1 tv:instance
				 ,(tv:allocate-data
				    'tv:show-clos-method
				    meth
				  )
				  tv:print-unpadded-method
			  )
			 `(:font 1
			   ,(format nil "~S"
			     (function-name
			       (fdefinition (second (second form)))
			     )
			    )
			  )
		     )
		     *this-line*)
	  )
	  (progn (push (reverse *this-line*) *all-lines*)
		 (setq *this-line* nil)
		 (push `(,tv:*space-format* ,indent) *this-line*)
		 (push '(:font 1 "(") *this-line*)
		 (mapcar #'(lambda (x)
			    (post-process-combined-method
			      x
			      (+ *method-combination-indent-increment* indent)
			    )
			   )
			   form
		 )
		 (push '(:font 1 ")") *this-line*)
	  )
      )
      (if form (push `(:font 1 ,(symbol-name form)) *this-line*) nil)
  )
)

(defun clean-up-method (method)
"Just makes sure that the method body doesn't have a boring definition for
 apply method at the front.
"
  (if (equal (first method) 'macrolet)
      (clean-up-method (rest (rest method)))
      method
  )
)

(defun method-combination-of-method-safe
       (method &optional (specializers (method-specializers-safe method)))
"Returns the list of method combination type for Method."
  (catch-error ;;; This catches things like eql specializers.
    (let ((methods (compute-applicable-methods-safe
		     (method-generic-function-safe method)
		     (mapcar #'(lambda (x)
				 (if (typep x 'ticlos:built-in-class)
				     t
				     (class-prototype-safe x)
				 )
			       )
			       specializers
		     )
		   )
	  )
	 )
	 (let ((code (compute-effective-method-safe
		       (method-generic-function-safe method)
		       (generic-function-method-combination-safe
			 (method-generic-function-safe method)
		       )
		       methods
		     )
	       )
	      )
	      (let ((form (clean-up-method code)))
		   (let ((*this-line* nil)
			 (*all-lines* nil)
			)
			(declare (special *this-line* *all-lines*))
			(Post-Process-Combined-Method
			  (if (equal (length form) 1)
			      (first form)
			      form
			  )
			)
			(push (reverse *this-line*) *all-lines*)
			(let ((result (reverse (remove nil *all-lines*))))
			     result
			)
		   )
	      )
	 )
    )
    nil
  )
)


(defsafe-simple slot-boundp-safe (instance slot)
  :documentation "Is true if the slot Slot is bound in the instance Instance."
  :both slot-boundp
)

(defun method-from-generic-function-using-method-function (fef gf)
"Given a method function and a generic function, returns the method object that
 has the method function Fef.
"
  (let ((methods (generic-function-methods-safe gf)))
       (find-if #'(lambda (method)
		    (equal fef (method-function-safe method))
		  )
		  methods
       )
  )
)


(defun method-from-method-function-safe (fef)
"Given a method function returns the method object for it."
  (let ((first-try (getf (sys:dbis-plist (sys:get-debug-info-struct
					   (1function-name* fef)
					 )
			 )
			 :method
		   )
	)
       )
       (or first-try
	   (method-from-generic-function-using-method-function
	     fef (function-generic-function-safe
		   (fdefinition-safe (second (function-name fef)))
		 )
	   )
	   (ferror nil "Can't get a method from ~S" fef)
       )
  )
)

;-------------------------------------------------------------------------------

(defun generic-object-foo-method (for-object)
"Given something (for-object) it returns the name of an :object-foo method that
 will know how to display it in the inspector.
"
  (cond ;;; These are the original-two data types
	;;; specified in this function by TI.
	((typep for-object 'stack-frame)     :object-stack-frame)
	((typep for-object 'any-sort-of-clos-instance) :object-clos-instance)
	((typep for-object 'named-structure) :object-named-structure)
	(T (case
	     (data-type for-object)
	     (dtp-instance :object-instance)
	     (dtp-array :object-array)
	     (dtp-list :object-list)
	     (dtp-stack-list :object-list)
	     (dtp-symbol :object-symbol)
	     (dtp-closure :object-closure)
	     (dtp-lexical-closure :object-lexical-closure)
	     (dtp-function :object-fef)
	     (dtp-locative :object-locative)
	     (dtp-stack-group :object-stack-group)  ;!
	     (otherwise :object-other)
	   )
	)
  )
)

;;; Abstracted out of TI definition of inspect-setup-object-display-list.
(defun inspect-object-display-list (object window)
"Given an object to inspect and the window in which it is to be inspected
 return the item list for the window.  If Object is an instance of
 encapsulation-for-generic-inspection, then this is taken as a directive to
 circumvent any specialised inspection behaviour added an use the really
 primitive stuff, like :object-named-structure.
"
  (send window (generic-object-foo-method object) object)
)

;;; Commented out by JPR on 09/14/90 11:48:14.
;;; Also defined in General-Inspector.lisp
;;; TI Code.
;(defun inspect-setup-object-display-list
;       (object window &optional top-item label &aux str)
;;;; A modified version of the TI one.  This is more general and extensible.
;  (multiple-value-bind
;    (display-list arg alt-print-fun first-top-item obj-label item-generator)
;      ;;; This part abstracted out by JPR.
;      (inspect-object-display-list object window)
;    (list object
;	  (or alt-print-fun 'inspect-printer)
;	  arg display-list (or top-item first-top-item 0)
;	  (or label
;	      obj-label
;	      (list nil nil nil nil (label-font (send window :label))
;		    (if (consp object)
;			"a list"
;			(nsubstring (setq str (format nil "~s~%" object))
;				    0 (position #\cr str)))))
;	  item-generator)))


;-------------------------------------------------------------------------------

(defun (:property clos-slot set-function)
       (item new-value object)
  (let ((slot (1if* (1typep* (third (second item)) 'inspection-data)
		  (slotd-name-safe (1send* (third (second item)) :Aux-Data))
		  (third (second item))
	      )
	)
       )
       (if (iwmc-class-p-safe object)
	   ;;; Use Eval here to make completely sure that there are no
	   ;;; macro-expansion dependencies for compilation of this file.
	   ;;; We don't want to have to load PCL just to compile this function.
	   (eval `(setf (pcl:slot-value    ,object ',slot) ',new-value))
	   (eval `(setf (ticlos:slot-value ,object ',slot) ',new-value))
       )
  )
)

(defprop clos-slot nil only-when-modify)

(defmethod (basic-inspect :object-clos-instance) (obj &aux result)
"The itemiser method for clos instances."
  (setq result (display-in-inspector self obj result))
  (values (nreverse result) obj 'inspect-printer)
)

(defparameter *CLOS-inspect-tab-width* 30.
"The default width to tab to when displaying clos instances.  This should give
 enough room for most slot names.
"
)

(defvar *show-plists-and-alists-for-show-slots-p* nil)
(defvar *sort-components-when-displaying-instances* nil)
(defvar *sort-ivs-when-displaying-instances* nil)
(defvar *separate-components-when-displaying-instances* nil)
(1defvar* *instance-component-sort-comparator* '2Symbol-Lessp*)

(defun symbol-lessp (a b)
  (let ((a (ucl:first-if-list a))
        (b (ucl:first-if-list b))
       )
       (if (symbolp a)
	   (if (symbolp b)
	       (or (string-lessp
		     (if (equal (symbol-package a) *package*)
			 ""
			 (package-name (symbol-package a))
		     )
		     (if (equal (symbol-package b) *package*)
			 ""
			 (package-name (symbol-package b))
		     )
		   )
		   (and	(string-equal
			  (if (equal (symbol-package a) *package*)
			      ""
			      (package-name (symbol-package a))
			  )
			  (if (equal (symbol-package b) *package*)
			      ""
			      (package-name (symbol-package b))
			  )
			)
			(string-lessp (symbol-name a) (symbol-name b))
		   )
	       )
	       t
	   )
	   (string-lessp (format nil "~S" a) (format nil "~S" b))
       )
  )
)

(defun 2Inspect-Slots*
       (window me title slots result &optional class &aux (maxlength 0))
"Generates an item list for the Slots of the instance Me, with title Title.
 Result is the list that we'll be collecting into.  It may already have
 had stuff put into it.
"
 (1let* ((cpl (class-precedence-list-safe (class-of-safe me) t)))
  (dolist (c slots)
    (Setq maxlength (max (flatsize c) maxlength)))
  (setq maxlength (min *CLOS-inspect-tab-width* maxlength))
  (if title
      (push `((:Font 1 ,title)
	      (3:Font* 1 " from class ")
	      (:Item1 instance ,(allocate-data 'show-clos-class class))
	     )
	    result
      )
      nil
  )
  (1if* title (push '("") result))
  (1if* 2*Sort-Ivs-When-Displaying-Instances**
      (1setq* slots (1sort* slots
		        #'(lambda (x y)
			    (1funcall* 2*Instance-Component-Sort-Comparator**
				     (2Slotd-Name-Safe* x)
				     (2Slotd-Name-Safe* y)
			    )
			  )
		 )
      )
      nil
  )
  (loop for c in slots
	for slot-name = (slotd-name-safe c)
	for (slot-boundp bound-p-error-p) = 
	  (multiple-value-list
	    (catch-error (slot-boundp-safe me (slotd-name-safe c)) nil)
	  )
	for (slot-value error-p) =
	  (if slot-boundp
	      (multiple-value-list
		(catch-error
		  (slot-value-safe me (slotd-name-safe c)) nil
		)
	      )
	      nil
	  )
	do
    (push `(,(1multiple-value-bind* (1ignore* class)
		 (1if* class
		     (1values* nil class)
		     (find-slot-in-classes (slotd-name-safe c) cpl)
		 )
	       (1if* class
		   `(:Item1 CLOS-SLOT
			    ,(allocate-data 'show-clos-instance-variable
				 class c
			     )
		    )
		   `(:Item1 CLOS-SLOT ,slot-name
		      ,#'(lambda (name stream) (format stream "~S" name))
		    )
	       )
	     )
	    (:colon ,(+ 2 maxlength))
	    ;;; Try to be really careful here.  Lots of catch-errors and such.
	    ;;; The last thing we want is for the Inspector to lock up
	    ;;; whilst looking at/printing slots and their values.
	   ,(if bound-p-error-p
	       '(:Font 2 "Error reading slot")
		(if slot-boundp
		    (if error-p
		       `(:font 2
			 "Some error happened whilst reading this slot."
			)
		       `(:Item1 Named-Structure-Value ,slot-value)
		    )
		   '(:Font 2 "unbound")
		)
	    )
	   )
	   result
    )
    (setq result
	  (maybe-expand-names-and-values-for-slot
	    result me slot-boundp slot-name slot-value window
	  )
    )
  )
  (1if* title (push '("") result))
  result
 )
)

;-------------------------------------------------------------------------------

(defun display-in-inspector (window me result)
"Generates a complete item list for the instance Me and collects it all into
 Result, which is returned.
"
  (let* ((class (class-of-safe me))
	 (instance-slots (class-instance-slots-grouped-safe class))
	 (non-instance-slots (class-non-instance-slots-grouped-safe class)))
    (push `(,(if (ticlos-instance-p me)
		 "CLOS instance of "
		 "CLOS (PCL) instance of "
	     )
	    (:item1 instance
		    ,(allocate-data 'show-clos-class (class-of-safe me))
	    )
	   )
	   result)
    (1if* 2*Sort-Components-When-Displaying-Instances**
        (1progn* (1setq* instance-slots
		     (1sortcar*
		       instance-slots
		       #'(lambda (x y)
			   (1funcall* 2*Instance-Component-Sort-Comparator**
				    (2Class-Name-Safe* x)
				    (2Class-Name-Safe* y)
			   )
			 )
		     )
	        )
	        (1setq* non-instance-slots
		     (1sortcar*
		       non-instance-slots
		       #'(lambda (x y)
			   (1funcall* 2*Instance-Component-Sort-Comparator**
				    (2Class-Name-Safe* x)
				    (2Class-Name-Safe* y)
			   )
			 )
		     )
	        )
	)
	nil
    )
    (Push '("") result)						
    (when (not (null non-instance-slots))
      (1if* *separate-components-when-displaying-instances*
	  (1loop* for (class . slotds) in non-instance-slots do
	     (setq result
		   (inspect-slots window me "Shared slots" slotds result class)
	     )
	  )
	  (1progn* (1push* `((:Font 1 "Shared Slots:")) result)
		  (1loop* for (class . slotds) in non-instance-slots do
		        (setq result
			      (inspect-slots window me nil slotds result class)
			)
		  )
	  )
      )
      (push '("") result)
    )
    (when (not (null instance-slots))
      (1if* *separate-components-when-displaying-instances*
	  (1loop* for (class . slotds) in instance-slots do
	     (setq result
		   (inspect-slots window me "Local slots" slotds result class)
	     )
	  )
	  (1progn* (1push* `((:Font 1 "Local Slots:")) result)
		  (1loop* for (class . slotds) in instance-slots do
		        (setq result
			      (inspect-slots window me nil slotds result class)
			)
		  )
	  )
      )
    )
    result
  )
)

;-------------------------------------------------------------------------------

;;; Temporary fixes?? !!!!! JPR.


(defun pcl:class-finalized-p (class)
  (ignore class)
  nil
)

(defun pcl:class-direct-slots (class)
  (pcl:class-slots class)
)

(defun pcl:slotd-accessors (slotd)
  (ignore slotd)
  nil
)

(defun pcl:generic-function-method-combination (gf)
  (ignore gf)
  :standard
)

(defun pcl:generic-function-argument-precedence-order (gf)
  (arglist (get-fef-from-object gf))
)

(defun pcl:generic-function-declare (gf)
  (ignore gf)
  nil
)

(defun pcl:class-of-generic-function (gf)
  (ignore gf)
  (class-named-safe 'pcl:standard-generic-function)
)

;-------------------------------------------------------------------------------

(eval-when (compile load eval)
  (if (not (get 'pcl:iwmc-class 'sys:defstruct-description))
      ;;; Make a dummy definition for iwmc-class so that we can defined methods
      ;;; on it, even if we don't have PCL loaded.
      (eval '(defstruct (pcl:iwmc-class)
	       (pcl:class-wrapper nil)
	       (pcl:static-slots nil)))
      nil))

(ticlos:defmethod documentation ((thing pcl:iwmc-class) &optional doc-type)
  (ignore doc-type)
  (typecase thing
    (pcl:standard-slot-description (pcl:slotd-documentation thing))
    (pcl:standard-class
     (second (assoc :documentation (pcl:class-options thing)))
    )
    (pcl:standard-method (pcl:method-documentation thing))
    (otherwise nil)
  )
)

(defun class-matches-p (specs1 specs2 so-far)
"Is true if the method spec spec1 for a method class-matches the method
 defined by spec2.  So-far accumulates whether we currently think that
 they match.
"
  (if specs1
      (if specs2
	  (let ((class-matches
		  (or (equal (first specs2) (first specs1))
		      (class-shadows (first specs2) (first specs1))
		  )
		)
	       )
	       (if (or (equal :disjoint class-matches) (not class-matches))
		   nil
		   (class-matches-p (rest specs1) (rest specs2)
				    (or class-matches so-far)
		   )
	       )
	  )
	  so-far
      )
      so-far
  )
)

;;; Supplied by David Gray.
(defun handlesp (generic-function &rest arguments)
  (let* ((defn generic-function)
	 (gfun
	   (cond
	     ((sys:typep-structure-or-flavor defn 'generic-function) defn)
	     ((ticlos:generic-function-p defn)
	      (ticlos:get-generic-function-object defn))
	     ((functionp defn) (return-from handlesp defn))
	     (t (error "~S is not a function." generic-function))))
	 (mloc
	   (apply #'ticlos:find-right-method
		  (or (ticlos:generic-function-method-hash-table gfun)
		      (progn (ticlos:build-method-hash-table gfun)
			     (ticlos:generic-function-method-hash-table gfun)))
		  (ticlos:reorder-parameter-specializers gfun arguments))))
    (or (contents mloc)
	(values (ticlos:find-handler
		  gfun
		  (mapcar #'ticlos:class-of
			  (ticlos:reorder-parameter-specializers
			    gfun arguments))) ))))

(defun handlers-for (gf &rest classes)
"Given a generic function and a list of classes it returns all of the methods
 that it thinks will handle the GF if called with those args.
"
  (let ((methods (generic-function-methods-safe
		   (function-generic-function-safe (get-fef-from-object gf))
		 )
	)
       )
       (remove (getf (sys:dbis-plist
		       (sys:get-debug-info-struct
			 #'(ticlos:method ticlos:print-object (t t))
		       )
		     )
		     :method
	       )
	       (remove-if-not
		#'(lambda (x)
		    (class-matches-p (unparse-specializers-safe x) classes t)
		  )
		  methods
	       )
       )
  )
)

;(handlers-for #'ticlos:print-object 'ticlos:standard-object)
;(handlers-for #'ticlos:print-object 'ticlos:standard-class)
;(handlers-for #'ticlos:print-object 'lexical-closure)
;(handlers-for #'ticlos:print-object 'closure)


(defun print-pointer (of stream)
  (let ((*print-base* 8.)
	(*print-radix* nil)
	(*nopoint t)
       )
       (format stream "~A" (%pointer of))
  )
)


;-------------------------------------------------------------------------------
;;; Redefine this so that we have :middle-button-result as a new required
;;; method.
(DEFFLAVOR 4inspection-data* (data (perspective-cache :unbound)) ()
  :SETTABLE-INSTANCE-VARIABLES
  (:REQUIRED-METHODS :format-concisely :handle-mouse-click 3:Who-Line-Doc* 3:Middle-Button-Result*)
  (:DOCUMENTATION :MIXIN
		  "3This mixin is used to define different types of information to inspect in custom Inspectors.
Flavors built on this mixin store some piece of data and provide operations which specify how to inspect the data.
Instance variable DATA is provided for storing the inspected data.  (See also flavor TV:AUXILIARY-DATA-MIXIN
which provides additional storage space.)  Flavors which mix in TV:INSPECTION-DATA are expected to define for
themselves a subset of the following methods:

:FORMAT-CONCISELY (stream)
  (required)  Returns a relatively short, one line string to be used in representing DATA in the
  inspection history, and in some cases, in the inspection pane itself.  The string is always made mouse-sensitive
  in either case.  Example: in the Flavor Inspector, \"TV:WINDOW's local methods\" is displayed in the history
  when the user inspects TV:WINDOW's local methods.  STREAM is the output stream.

:HANDLE-MOUSE-CLICK (blip inspector-instance)
  (required)  Handles mouse clicks on the mouse sensitive string (returned by :FORMAT-CONCISELY) which represents DATA.
  BLIP is the mouse blip, which contains the mouse button used and the input pane instance.  INSPECTOR-INSTANCE is
  the inspector constraint frame.  This method is called when the mouse-sensitive string is displayed either in an
  inspection pane or the history pane.  Usually this method should do the following:
    1. on MOUSE-L, (SEND inspector-instance :INSPECT-INFO-LEFT-CLICK)
    2. on MOUSE-M, (SEND inspector-instance :INSPECT-INFO-MIDDLE-CLICK)
    3. on MOUSE-R, pop up a menu of operations relevant to the data (or some other data-specific action)
  #1 will inspect the data, using the :GENERATE-ITEM method described below to generate a text scroll window item list.
  #2 will do the same, but will put (or leave) the contents of the clicked-on inspection pane in the middle inspection
  pane.  For an example of this, look at (:METHOD TV:FLAVOR-OPERATION-MIXIN :HANDLE-MOUSE-CLICK).

:GENERATE-ITEM ()
  (optional)  Provides text scroll window items to display in inspection panes when inspecting DATA.
  This method is optional because some pieces of data may be used only as mouse-sensitive items in another data's
  text scroll window item list (generated by *its* :GENERATE-ITEM method).  For instance, default instance variable
  values (instances of TV:SHOW-VALUE) are displayed in the item list for TV:SHOW-INSTANCE-VARIABLES.  When the user 
  clicks on them, their :HANDLE-MOUSE-CLICK is defined to pretty-print the value on a typeout window.  They are never
  \"inspected\", so they don't need a :GENERATE-ITEM method.

  :GENERATE-ITEM should return two values. The first value is a list of items to display in an inspection pane 
  (text scroll window).  The second value is a string to display in the pane's window label.  This string should
  probably be close to that returned by :FORMAT-CONCISELY, though you have more room to work with since the window
  label is much wider than the history pane.

:WHO-LINE-DOC (inspection-pane? &OPTIONAL no-sensitive-item?)
  (required)  Provides a string to display in the who-line appropriate for DATA.  The arguments provide information
  about where the mouse is.
  1. If NO-SENSITIVE-ITEM? is NIL, the mouse is over the mouse-sensitive string (returned by :FORMAT-CONCISELY) which
     represents DATA.  The string should describe what actions your :HANDLE-MOUSE-CLICK method
     does.  INSPECTION-PANE? is NIL when DATA is being \"moused over\" in the history pane; otherwise it is the inspection
     pane instance in which DATA is being \"moused over\".  Your :HANDLE-MOUSE-CLICK method may treat mouse clicks differently
     when DATA is moused in the history pane, so this argument is provided so that your mouse documentation can provide
     different strings for each case.  
  2. If NO-SENSITIVE-ITEM? is non-NIL, the mouse is NOT currently over a mouse sensitive item, but DATA is the currently
     inspected object of INSPECTION-PANE?.  If your inspector has commands which handle the mouse in this case,
     you should return a string which describes these commands.  For instance, in the flavor inspector, right click
     brings up a menu of operations on the flavor most relevant to DATA.*

:middle-button-result ()
  Returns the piece of data that you want the user to get a handle on if the
  instance is clicked on.  This is used in perspectives and such-like.
3:HELP ()
  (optional)  Returns a string to display which describes the inspection display of DATA (the text scroll window items
  returned by :GENERATE-ITEM).  The inspection code does not provide an automatic interface to this method: you can
  build whatever interface your prefer.  For an example, the Flavor Inspector defines a command on MOUSE-M which
  displays the help message returned by the object currently being inspected in the clicked-on inspection pane (do
  meta-point on (:METHOD TV:FLAVOR-INSPECTOR :HELP-ON-INSPECTED-DATA).*"))

(when (not (fboundp 'pcl:slotd-name))
  (defun pcl:slotd-name (&rest ignore))
  (defun pcl:slot-value (&rest ignore))
  (defun pcl:class-of (&rest ignore))
  (defun pcl:generic-function-p (&rest ignore))
  (defun pcl:generic-function-methods (&rest ignore))
  (defun pcl:generic-function-method-class (&rest ignore))
  (defun pcl:compute-applicable-methods (&rest ignore))
  (defun pcl:compute-effective-method-body (&rest ignore))
  (defun pcl:method-function (&rest ignore))
  (defun pcl:slotd-initform (&rest ignore))
  (defun pcl:slotd-readers (&rest ignore))
  (defun pcl:slotd-type (&rest ignore))
  (defun pcl:slotd-allocation (&rest ignore))
  (defun pcl:slotd-initargs (&rest ignore))
  (defun pcl:class-slots (&rest ignore))
  (defun pcl:class-precedence-list (&rest ignore))
  (defun pcl:method-type-specifiers (&rest ignore))
  (defun pcl:method-arglist (&rest ignore))
  (defun pcl:class-direct-methods (&rest ignore))
  (defun pcl:class-default-initargs (&rest ignore))
  (defun pcl:class-direct-generic-functions (&rest ignore))
  (defun pcl:method-qualifiers (&rest ignore))
  (defun pcl:unparse-specializers (&rest ignore))
  (defun pcl:find-class (&rest ignore))
  (defun pcl:class-name (&rest ignore))
  (defun pcl:subclassp (&rest ignore))
  (defun pcl:class-local-supers (&rest ignore))
  (defun pcl:method-generic-function (&rest ignore))
  (defun pcl:class-direct-subclasses (&rest ignore))
  (defun pcl:class-direct-superclasses (&rest ignore))
  (defun pcl:iwmc-class-p (&rest ignore))
  (defun pcl:generic-function-name (&rest ignore))
  (defun pcl:class-prototype (&rest ignore))
  (defun pcl:slot-boundp (&rest ignore))
  (defun pcl:slotd-documentation (&rest ignore))
  (defun pcl:class-options (&rest ignore))
  (defun pcl:method-documentation (&rest ignore))
)
