;;; -*- Mode:Common-Lisp; Package:COMMON-LISP; Fonts:(CPTFONT HL10B CPTFONTI CPTFONT CPTFONTB); Base:10; Patch-file: t -*-

(in-package "COMMON-LISP")

(defun 4defpackage-unstringify* 
       (symbol-names package-name &optional cerror-context cerror-packname)
  "2Return a list of symbols with print names SYMBOL-NAMES in the package
named by PACKAGE-NAME.  If CERROR-CONTEXT is non nil, signals a continuable
error before any new symbols are created.*"
  (let ((package (find-package package-name)))
    (unless package 
      (error "3DEFPACKAGE: package ~a not found.*" package-name))
    (loop for symbol-name in symbol-names
	  for symbol = (find-symbol symbol-name package)
	  for found-p = (sys:nth-value 1 (find-symbol symbol-name package))
	  unless found-p do
	    (when cerror-context
	      (cerror
		"Intern ~2*~s into package ~a and continue."
		"DEFPACKAGE: Error processing ~s clause for package ~a.~%~
                 Symbol ~s not found in package ~a."
		cerror-context cerror-packname symbol-name package-name))
	    (setf symbol (intern symbol-name package-name))
	  collect symbol)))


(eval-when (compile load eval)
  (when (eq (find-symbol "DEFPACKAGE" 'ticl) (find-symbol "DEFPACKAGE" "COMMON-LISP"))
    (unintern 'ticl:defpackage "COMMON-LISP")
    (shadow '("DEFPACKAGE") "COMMON-LISP")
    (export (find-symbol "DEFPACKAGE" "COMMON-LISP") "COMMON-LISP")))


(defmacro cl:4defpackage* (name &rest options)
  "2CLtL2 conformant DEFPACKAGE.  Defines (3creates or alters)* a package
which will be named NAME.  NAME may be a string or a symbol (if a symbol
only the print-name matters.)
  Each OPTION is a list of a keyword which defines the option and
arguments for that option.  No part of a DEFPACKAGE form is evaluated.
Except for the :SIZE (3and TI specific :AUTO-EXPORT-P and :PREFIX-NAME)* 
options, any option may occur any number of times.
  The supported options are as follows.  In every case, any argument
called <package-name> or <symbol-name> may be a string or a symbol;
if a symbol, only the print-name matters.  Options are processed in
the following order: :SHADOW and :SHADOWING-IMPORT-FROM, then :USE,
then :IMPORT-FROM and :INTERN, and finally :EXPORT.

 (3:SIZE <integer>)**  This specifies2 approximately the number of
symbols to allocate space for initially.

 (3:NICKNAMES <package-name>*)*  This specified names become 
nicknames of the package being defined.

 (3:SHADOW <symbol-name>*)*  Symbols with the specified names are
created as shadows in the package being defined (3see SHADOW.)*

 (3:SHADOWING-IMPORT-FROM <package-name> <symbol-name>*)*  Symbols
with the specified names are located in the specified package, and
imported into the package being defined, shadowing other symbols
if necessary.  See SHADOWING-IMPORT.

 (3:USE <package-name>*)*  The package being defined is made to 
\*"use2\*"2 (3inherit from)* the pacakges specified; see USE-PACKAGE.

 (3:IMPORT-FROM <pakage-name> <symbol-name>*)*  Symbols with the
specified names are located in the specified packages and 
imported into the package being defined.  See IMPORT.

 (3:INTERN <symbol-name>*)*  Symbols with the specified names are
located or created in the package being defined; see INTERN.

 (3:EXPORT <symbol-name>*)*  Symbols with the specified names are
located or created in the package being defined, and then
exported.  See EXPORT.


In addition, the following non-standard options are supported for
backwards compatibility with the old TI DEFPACKAGE:

 (3:IMPORT <symbol>*)*  The specified symbols will be imported
into the package being defined.  The arguments must be actual 
symbols, not strings, and the package of the symbols IS significant
in this case...

 (3:SHADOWING-IMPORT <symbol>*)*  The specified symbols will be
imported into the package being defined, shadowing existing 
symbols if necessary.  The arguments must be actual symbols,
not strings, and the package of the symbols IS significant in
this case...

 (3:PREFIX-NAME <package-name>)*  Sets the name to be used when
printing symbols which are owned by the package being defined.
This must be a name or nickname of the package being defined.

 (3:AUTO-EXPORT-P <boolean>)*  Specifies that all symbols placed
in the package being defined should be automatically exported
at that time.*"
  (let ((new-name (if (symbolp name) (symbol-name name) name))
	(option-alist (mapcar #'list '(:size :auto-export-p :import :shadowing-import :nicknames :shadow 
				       :shadowing-import-from :use :import-from :intern :export :prefix-name))))
    ;1; Collect option clauses*
    (loop for option in options
	  for entry = (assoc (first option) option-alist)
	  do (if entry
		 (setf (rest entry)
		       (if (member (first option) '(:shadowing-import-from :import-from))
			   (cons (rest option) (rest entry))
			   (append (rest option) (rest entry))))
		 (error "3Unrecognized DEFPACKAGE option for package ~a: ~s*" new-name (first option))))

    ;1; Quick error check...*
    (when (> (length (assoc :size option-alist)) 2)
      (error "3DEFPACKAGE: Multiple :SIZE options.*"))
    
    ;1; Check options and convert symbols to strings...*
    (loop for (option . args) in (nthcdr 4 option-alist) do
	  (if (member option '(:import-from :shadowing-import-from))
	      (loop for entry in args do
		    (loop for rest on entry
			  for head = (first rest) do
			  (typecase head
			    (symbol (setf (first rest) (symbol-name head)))
			    (string)
			    (t (error "3DEFPACKAGE: Invalid ~:[symbol~;package~] name in ~a option ~
                                       for package ~a: ~a~&Must be a symbol or string.*"
				      (eq head (first entry)) option new-name head)))))
	      (loop for rest on args
		    for head = (first rest) do
		    (typecase head
		      (symbol (setf (first rest) (symbol-name head)))
		      (string)
		      (t (error "3DEFPACKAGE: Invalid symbol name in ~a option for package ~a: ~a~&~
                                 Must be a symbol or string.*"
				option new-name head))))))

    ;1; Destructure options*
    (let ((nicknames (rest (assoc :nicknames option-alist)))
	  (shadows (rest (assoc :shadow option-alist)))
	  (shadow-imports (rest (assoc :shadowing-import-from option-alist)))
	  (imports (rest (assoc :import-from option-alist)))
	  (interns (rest (assoc :intern option-alist)))
	  (exports (rest (assoc :export option-alist)))
	  (size (second (assoc :size option-alist)))
	  (prefix-name (second (assoc :prefix-name option-alist)))
	  (auto-export-p (rest (assoc :auto-export-p option-alist)))
	  (old-style-imports (rest (assoc :import option-alist)))
	  (old-style-shadow-imports (rest (assoc :shadowing-import option-alist)))
	  (uses (cond 
		  ((rest (assoc :use option-alist)))
		  ((assoc :use options) nil)
		  (t '("TICL" "LISP")))))

      ;1; Support obsolete :IMPORT option*
      (when old-style-imports
	(warn "3DEFPACKAGE: :IMPORT is obsolete, use :IMPORT-FROM instead.*")
	(unless (every #'symbolp old-style-imports)
	  (error "3DEFPACKAGE: The import list for package ~a contains non-symbols: ~s*" 
		 new-name (remove-if #'symbolp old-style-imports)))
	(loop for sym in old-style-imports
	      for pname = (symbol-name sym) 
	      and packname = (package-name (symbol-package sym))
	      for entry = (assoc packname imports :test #'string=) do
	      (if entry
		  (setf (rest entry) (cons pname (rest entry)))
		  (push (list packname pname) imports))))

      ;1; Support obsolete :SHADOWING-IMPORT option*
      (when old-style-shadow-imports
	(warn "3DEFPACKAGE: :SHADOWING-IMPORT is obsolete, use :SHADOWING-IMPORT-FROM instead.*")
	(unless (every #'symbolp old-style-imports)
	  (error "3DEFPACKAGE: The shadowing import list for package ~a contains non-symbols: ~s*" 
		 new-name (remove-if #'symbolp old-style-shadow-imports)))
	(loop for sym in old-style-shadow-imports
	      for pname = (symbol-name sym) 
	      and packname = (package-name (symbol-package sym))
	      for entry = (assoc packname shadow-imports :test #'string=) do
	      (if entry
		  (setf (rest entry) (cons pname (rest entry)))
		  (push (list packname pname) shadow-imports))))

      ;1; Check for symbol name overlap between shadow, import and*
      ;1; intern options.  See CLtl2 p. 272.*
      (loop for (name1 . other-names) 
	    on '(:shadow :shadowing-import-from :import-from :intern)
	    and (val1 . other-vals) 
	    on (mapcar #'(lambda (entry)
			   (if (consp (first entry))
			       (mapcan #'(lambda (from) (copy-list (rest from))) entry)
			       entry))
		       (list shadows shadow-imports imports interns))		       
	    do (loop for name2 in other-names
		     and val2 in other-vals
		     for dups = (intersection val1 val2 :test #'string=)
		     when dups do
		     (error "DEFPACKAGE: The following symbol ~:[name appears~;names appear~]~&~
                             in both ~a and ~a options2 for package ~a*:~{~%  ~a~}~%"
			    (> (length dups) 1) name1 name2 new-name dups)))

      ;1; Check for symbol name overlap between intern and export*
      ;1; options.  See CLtL2 p. 272.*
      (let ((overlap (intersection interns exports :test #'string=)))
	(when overlap
	  (error "3DEFPACKAGE: The following symbol ~:[name appears~;names appear~]~%~*
		3  in both :INTERN and :EXPORT options for package ~a:~{~%  ~a~}~%*"
		 (> (length overlap) 1) new-name overlap)))
      
      ;1; The macro body...*
      `(eval-when (compile load eval)
	 (let* ((package-exists-p (find-package ,new-name))
		(new-package-sym (intern ,new-name "3USER*"))
		(new-package 
		  (funcall (if package-exists-p #'SI:ALTER-PACKAGE #'SI:MAKE-PACKAGE)
			   ,new-name 
			   :use (when package-exists-p (package-use-list ,new-name))
			   :nicknames ,(if nicknames
					   `(append ',nicknames 
						    (when package-exists-p (package-nicknames ,new-name)))
					   `(when package-exists-p (package-nicknames ,new-name)))
			   ,@(when size (list :size size))
			   ,@(when prefix-name (list :prefix-name prefix-name))
			   ,@(when auto-export-p (list :auto-export-p (first auto-export-p))))))
	   (si:record-source-file-name new-package-sym 'si:defpackage)
	   (setf (getf (si:pack-plist new-package) :source-file-name)
		 (second (assoc 'si:defpackage (get new-package-sym :source-file-name))))
	   ,@(when shadows
	       `((shadow (defpackage-unstringify ',(remove-duplicates shadows :test #'string=) new-package)
			 new-package)))
	   ,@(loop for shadow-import in shadow-imports
		   collect `(shadowing-import 
			      (defpackage-unstringify ',(remove-duplicates (rest shadow-import) :test #'string=) 
				,(first shadow-import) ":SHADOWING-IMPORT-FROM" ,new-name)
			      new-package))
	   ,@(when uses `((use-package ',(remove-duplicates uses :test #'string=) new-package)))
	   ,@(loop for import in imports
		   collect `(import 
			      (defpackage-unstringify ',(remove-duplicates (rest import) :test #'string=) 
				,(first import) ":IMPORT-FROM" ,new-name)
			      new-package))
	   ,@(when interns 
	       `((loop for symbol-name in ',(remove-duplicates interns :test #'string=)
		       do (intern symbol-name new-package))))
	   ,@(when exports
	       `((export (defpackage-unstringify ',(remove-duplicates exports :test #'string=) new-package)
			 new-package)))
	   new-package)))))


;;; Written 02/24/92 16:43:13 by RICE,
;;; while running on KSL-EXP-35 from band LOD1
;;; With SYSTEM 6.41, VIRTUAL-MEMORY 6.3, EH 6.8, MAKE-SYSTEM 6.5, MICRONET 6.0, LOCAL-FILE 6.2,
;;;  BASIC-PATHNAME 6.5, NETWORK-SUPPORT-COLD 6.2, BASIC-NAMESPACE 6.8, NETWORK-NAMESPACE 6.1,
;;;  DISK-IO 6.4, DISK-LABEL 6.1, BASIC-FILE 6.15, MAC-PATHNAME 6.0, NETWORK-PATHNAME 6.2,
;;;  COMPILER 6.18, TV 6.32, DATALINK 6.0, CHAOSNET 6.9, GC 6.7, MEMORY-AUX 6.0, NVRAM 6.4,
;;;  SYSLOG 6.2, STREAMER-TAPE 6.6, UCL 6.1, INPUT-EDITOR 6.1, METER 6.2, ZWEI 6.28,
;;;  DEBUG-TOOLS 6.5, NETWORK-SUPPORT 6.1, NETWORK-SERVICE 6.3, DATALINK-DISPLAYS 6.0,
;;;  FONT-EDITOR 6.1, SERIAL 6.0, PRINTER 6.8, MAC-PRINTER-TYPES 6.2, PRINTER-TYPES 6.2,
;;;  IMAGEN 6.1, SUGGESTIONS 6.1, MAIL-DAEMON 6.6, MAIL-READER 6.9, TELNET 6.1, VT100 6.0,
;;;  NAMESPACE-EDITOR 6.7, PROFILE 6.3, VISIDOC 6.7, TI-CLOS 6.53, CLEH 6.5, IP 3.50,
;;;  Experimental CLX 2.0,  microcode 488, Band Name: TI+IP+CLX+Defpackage

#!C
; From file PACKAGES.LISP#> KERNEL; KSL-EXP-7:
#10R SYSTEM#:
(lisp:COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "SYSTEM"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* *COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "KSL-EXP-7: KERNEL; PACKAGES.#"


(Defmacro sys:PARSE-PACKAGE-ARGUMENT (pkg)
;1; expands into code which attempts to produce a package object from the argument <pkg>*
;1; and default to *PACKAGE* if omitted.*
;1; Most package functions, e.g. intern, expect a package object as the second argument.*
  `(COND ((NULL ,pkg) *PACKAGE*)
	 ((FIND-PACKAGE ,pkg))1   ;; at this point, <pkg> should be a package object*
	 (T (PACKAGE-DOES-NOT-EXIST-ERROR  ,pkg))))

(Defun sys:EXPORT (symbols &OPTIONAL pkg)
1  "Makes SYMBOLS external in package PKG.
If the symbols are not already present in PKG, they are imported first.
Error if this causes a name conflict in any package that USEs PKG."*
  (LET ((pkg (PARSE-PACKAGE-ARGUMENT pkg))                          ;; verify package argument
	(export-list (IF (LISTP symbols) symbols (LIST symbols))))  ;; coerce <symbols> to a list
    (UNLESS (EVERY #'SYMBOLP export-list)                           ;; verify all are symbols - complain otherwise
	    (ERROR t 1"the export list contains non-symbols: ~s"* (REMOVE-IF #'SYMBOLP export-list)))
    (LET ((real-export-list    ;; prepare to punt symbols already exported
	    (REMOVE-IF         ;;   -- this is worthwhile since files with 'exports' are often re-compiled
	      #'(Lambda (sym) 
		  (MULTIPLE-VALUE-BIND (csym found) 
		      (FIND-SYMBOL (symbol-name sym) pkg)
		    (AND (EQ found :external) (EQ sym csym))))
	      export-list))
	  (used-by-list (PACK-USED-BY-LIST pkg)))
      (TAGBODY try-next-sym
	  (DOLIST (sym real-export-list)
		  (WHEN used-by-list
			(LET ((set-of-directly-conflicting-symbols nil)
			      (set-of-inherited-conflicting-symbols nil)
			      (name (SYMBOL-NAME sym)))
			  (DOLIST (p used-by-list)     ;; for each package p using pkg
				  (MULTIPLE-VALUE-BIND (csym found)  ;; look for a conflict
				      (FIND-SYMBOL name p)
				    (WHEN (AND found
					       (NEQ sym csym) 
					       (NOT (MEMBER csym (pack-shadowing-symbols p) :test #'eq)))
					  (IF (EQ found :inherited)
					      (PUSH (CONS csym p) set-of-inherited-conflicting-symbols)
					      (PUSH (CONS csym p) set-of-directly-conflicting-symbols)))))
			1  ;; Handle name conflicts*
			  (COND ((AND set-of-directly-conflicting-symbols
				      set-of-inherited-conflicting-symbols)
				 (signal-proceed-case ((ignore) 'eh:name-conflict
						       (format t "~%Attempting to export ~s from the ~a package ~
would introduce the following name conflicts:" sym (package-name pkg))
						       sym	
						       (package-name pkg)
						       :export
						       (cons set-of-directly-conflicting-symbols
							     set-of-inherited-conflicting-symbols)
						       (progn (dolist (pair set-of-directly-conflicting-symbols)
								      (format t "~&~10t~s is present in the ~a package."
									      (car pair) (package-name (cdr pair))))
							      (dolist (pair set-of-inherited-conflicting-symbols)
								      (format t "~&~10t~? is accessible by inheritance ~
in the ~a package."
									      "~a:~a"
									      `(,(multiple-value-bind (ignore ignore pack)
										     (find-symbol (symbol-name (car pair))
												  (cdr pair))
										   (package-name pack))
										,(car pair))
									      (package-name (cdr pair))))))
						      (:export-both-conflict-types nil)
						      (:skip (go try-next-sym))
						      (:skip-all (return-from export t))))
				(set-of-directly-conflicting-symbols
				  (signal-proceed-case ((ignore) 'eh:name-conflict
							(format t "~%Attempting to export ~s from the ~a package ~
would introduce the following name conflicts:" sym (package-name pkg))
							sym	
							(package-name pkg)
							:export
							set-of-directly-conflicting-symbols
							(dolist (pair set-of-directly-conflicting-symbols)
								(format t "~&~10t~s is present in the ~a package."
									(car pair) (package-name (cdr pair)))))
						       (:export-present nil)
						       (:unintern-all nil)
						       (:shadow-all nil)
						       (:skip (go try-next-sym))
						       (:skip-all (return-from export t))))
				(set-of-inherited-conflicting-symbols
				  (signal-proceed-case ((ignore) 'eh:name-conflict
							(format t "~%Attempting to export ~s from the ~a package ~
would introduce the following name conflicts:" sym (package-name pkg))
							sym	
							(package-name pkg)
							:export
							set-of-inherited-conflicting-symbols
							(dolist (pair set-of-inherited-conflicting-symbols)
								(format t "~&~10t~? is accessible by inheritance ~
in the ~a package."
									"~a:~a"
									`(,(multiple-value-bind (ignore ignore pack)
									       (find-symbol (symbol-name (car pair))
											    (cdr pair))
									     (package-name pack))
									  ,(car pair))
									(package-name (cdr pair)))))
						       (:export-accessible-by-inheritance nil)
						       (:skip (go try-next-sym))
						       (:skip-all (return-from export t))))
				(t nil))
			  ))
		  1;; If we get here, then proceed with exporting <sym>.*
		  (IMPORT (if sym sym '(nil)) pkg) ;; Allow for importing NIL.  JPR.  02/24/92 16:45:10
		  (EXTERNALIZE sym pkg)
		  try-next-sym
		  ))
      t)))

))
