;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONTB HL12B HL12I HL12BI); Vsp:0 -*-

;1;; File "3VOICEMAIL*"*
;1;; Code that allows buffers to have sounds stored in them using Diagram Lines.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   17 Apr 89*	1Jamie Zawinski*	1 Created.*
;1;;*   15 Apr 90*	1Jamie Zawinski *	 1Added a better file format, keeping compatibility with the old one.*
;1;;*

;1;; Defines two new editor commands,  5Record and Insert Sound Diagram* and5 Play Sound Buffer*.*
;1;;*
;1;; 7Record and Insert Sound Diagram**
;1;;*
;1;; prompts the user for the name and length of a sample, and then records with *TV:RECORD1.*  1It then inserts* 1a 6Diagram Line**
;1;; into the buffer at the* 1current point. * 6Diagram Lines1 are Zmacs' way of storing arbitrary** 1graphical* 1entities into buffers.*
;1;; This file defines a new kind of Diagram,* 1which holds a sound array, and* 1knows how to read and write itself* 1to and from*
;1;; a file.*
;1;;*
;1;; 7Play Sound Buffer**
;1;;*
;1;; will search the current buffer for Sound Diagrams and play the ones it finds.*  1If the current buffer is a* 1Mail Summary*
;1;; Buffer, then it searches the corresponding Sequence Buffer instead.*
;1;;*
;1;; 5How to Use:**
;1;;*
;1;; Though there is code to support6 Diagram Lines* in Zmacs, the code which writes them to files is broken. * 1A well-formed*
;1;; diagram can be read in and displayed by Zmacs as distributed with release 4.1, but if you try to write out a file with a*
;1;; diagram, the file will be written in an incorrect format.  therefore, to use this code, you also need my fix to the Zmacs*
;1;; code which writes diagram lines.  (That is, until TI fixes it.)  This code is in the file 3EPSILONS.LISP* in this directory.*
;1;;*
;1;; The TI mail reader does not interpret Zmacs font-change codes.  Since the same code parses font-changes and Diagram*
;1;; lines, Diagram lines cannot be sent in mail - unless of course, you load the file 3MAIL-FONT-PATCH.LISP*, which forces*
;1;; the mail reader to interpret font-changes (and as a side-effect, Diagram lines).*
;1;;*
;1;;*
;1;;*	5------------------------------------------------------------------------------*
;1;;*	5|  Warning!!1  **										5|*
;1;;*	5|*	2If someone tries to visit a file containing a Sound Diagram but does not have the code in1 **	5|*
;1;;*	5|*	2this file loaded, they will get an error.  If the diagram is in their mail file, they will be*	5|*
;1;;*	5|*	2unable to read their1 *mail until the offending diagram line is surgically removed (or this*	5|*
;1;;*	5|*	2code is loaded).1  *Therefore, discression is advised.*					5|*
;1;;*	5------------------------------------------------------------------------------*
;1;;*
;1;; If you're using this code, send voicemail to 5jwz@teak.berkeley.edu*.*
;1;;*
;1;; 5Problems:**
;1;;*
;1;; Only one problem.  A file with sound in it is absolutely huge.  This code does no compression at all, and it really 6really* should.*
;1;; The sound-data must be written to files using only the printable ASCII characters (32-126) or it might not make it through all*
;1;; mail delivery systems unharmed.  For a similar reason, we insert newlines every 70 or so characters.*
;1;;*
;1;; The old format was to write each 8 bits of sound data as two ASCII characters (in the range 50* to 5?*, or 548* to 563*).*
;1;; This format is exceptionally lame.  The file takes up exactly twice as much space as a straight binary dump would.*
;1;;*
;1;; The new format is to write each 32 bits of sound data as five ASCII characters (in the range of 5!* to 5v*, or 533* to 5117*).*
;1;; This format is a little less lame.  The file takes up exactly five fourths as much space as a straight binary dump would.*
;1;;*
;1;; This code knows how to read and write both formats, to maintain compatibility with people running the old version of this code.*
;1;;*


(defflavor 4sound-diagram*
	   ((sound-array nil)		;1 An element of the *TV:SOUND-ARRAY1 resource.*
	    (read-state nil)		;1 The state of the parser.*
	    (first-line nil)		;1 A *ZWEI:LINE1 which is the first line on which this diagram appears.*
	    (name nil)
	    (image-bit-array nil)	;1 The image used for this diagram - includes icon and text.*
	    (version 2)			;1 The version number of this sound diagram.*
	    )
	   (restorable-line-diagram-mixin)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(sound-diagram :print-self)* (stream &rest ignore)
  (format stream "3#<~S v~D, ~A ~,2F second~:P>*"
	  (type-of self) version (or name "3-noname-*")
	  (if sound-array (float (/ (length sound-array) 8000) 1.0s0) 0)
	  (if sound-array (length sound-array) 0)
	  ))


(defmethod 4(sound-diagram :number-of-lines)* ()
  "2Returns the number of lines necessary to write this diagram out to a file.*"
  (case version
    (1
     ;1; The output of a sound-diagram has two header lines, and then a representation of the sound data*
     ;1; with a newline every 70 characters.  The dump of the data is twice as large as the data itself... yuck.*
     (+ 2 (ceiling (* 2 (length sound-array))
		   70)))
    (2
     ;1; Slightly better - the output representation is only 5/4ths as large.*
     (+ 2 (ceiling (* 5/4 (length sound-array))
		   75)))
    (t
     (cerror "2Give up on this sound diagram.*" "2This is a version ~D sound diagram; current version is ~D.*"
	     version 2)
     0)))


(defmethod 4(sound-diagram :add-line)* (line &optional contents)
  "2Called by the Epsilon reader - given a line from the file, parse it into the diagram currently being built.*"
  (declare (ignore contents))
  (ecase read-state
    (NIL    (assert (string= line "3sound-array*" :end1 11) () "3Invalid sound-diagram.*")
	    (setf first-line line
		  (getf (line-plist line) :diagram-line-number) 0
		  read-state 'HEADER)
	    (cond ((> (length line) 12)
		   (multiple-value-bind (val end)
					(let* ((*package* *keyword-package*))
					  (read-from-string line nil nil :start 13))
		     (setq name val)
		     (setq version (read-from-string line nil 1 :start end))))
		  (t
		   (setq name nil version 1))))
    (HEADER (let* ((sample-length (parse-integer line :radix 10)))
	      (setf sound-array (allocate-resource 'tv:sound-array sample-length)
		    (fill-pointer sound-array) 0
		    (getf (line-plist line) :diagram-line-number) 1
		    read-state 'BODY)))
    (BODY   (let* ((length (length line)))
	      ;1;*
	1        *;1; The epsilon-reader has already inserted this line into the buffer.  This will not do.  The epsilon-reader assumes that we*
	      ;1; want a buffer-line for each file-line of the diagram, which is absolutely not the case for sound-diagrams (since we have*
	      ;1; hundreds of file-lines, and only four or five buffer-lines).  So, we remove it now (if we remove it right away, it might*
	      ;1; still be in generation zero, and be collected as soon as we are done parsing it).*
	      ;1;*
	      (let* ((p (line-previous line))
		     (n (line-next     line)))
		(and p (setf (line-next     p) n))
		(and n (setf (line-previous n) p))
		)
	      (case version
		(1	;1 The version 1 reader, with no compression.*
		 (do ((i 0 (+ 2 i)))
		     ((>= i length))
		   (let* ((c1 (char line i))
			  (c2 (char line (1+ i)))
			  (n (+ (dpb (- c1 #.(char-code 0)) (byte 4 4) 0)
				(dpb (- c2 #.(char-code 0)) (byte 4 0) 0))))
		     (unless (vector-push n sound-array)
		       ;1; When we reach the end of the vector, set 5read-state* to be 5EOF* so that attempts to read more lines won't work.*
		       (return (setq read-state 'EOF))))))
		
		(2	;1 The version 2 reader, which does 5btoa/atob.**
		 (let* ((start (fill-pointer sound-array)))
		   (setf (fill-pointer sound-array) (atob line sound-array 0 (length line) start))
		   (when (>= (fill-pointer sound-array) (array-dimension sound-array 0))
		     (setq read-state 'EOF))))
		
		(t (cerror "2Give up reading this sound diagram.*" "2This is a version ~D sound diagram; current version is ~D.*"
			   version 2)
		   (setq read-state 'EOF))
		)))))


(defun 4write-sound-array-as-ascii *(sound-array stream version &optional name)
  "2Write the sound array on* STREAM2 in a format which uses only printable ASCII characters.*"
  (when (eq stream t) (setq stream *standard-output*))
  (when (symbolp sound-array) (setq sound-array (get sound-array 'tv:sound-array)))
  (assert sound-array () "3not a sound array*")
  (case version
    (1		;1 For the version 1 reader, with no compression.*
     (format stream "3~&sound-array: ~A~%~D~%*" (or name "") (length sound-array))
     (let* ((j 0))
       (dotimes (i (length sound-array))
	 (let* ((sample (aref sound-array i))
		(ms-nibble (ldb (byte 4 4) sample))
		(ls-nibble (ldb (byte 4 0) sample)))
	   (send stream :tyo (+ ms-nibble #.(char-code #\0)))
	   (send stream :tyo (+ ls-nibble #.(char-code #\0))))
	 (cond ((= j 34) (terpri stream) (setq j 0))
	       (t (incf j))))))
    
    (2		;1 For the version 2 reader, which does 5btoa/atob.**
     (format stream "3~&sound-array: ~A ~D~%~D~%*" (or name "3-noname-*") 2 (length sound-array))
     (let* ((length (length sound-array))
	    (inc 60)	;1 <75 input characters> / <5/4 output characters>.*
	    (line (make-array 76 :element-type 'string-char :fill-pointer 0)))
       (declare (fixnum length inc)
		(string line)
		(optimize speed))
       (let* ((i 0))
	 (declare (fixnum i))
	 (loop
	   (let* ((j (min length (+ i inc))))
	     (declare (fixnum j))
	     (setf (fill-pointer line) 0)
	     (setf (fill-pointer line) (btoa sound-array line i j 0 75))
	     (when (plusp (fill-pointer line))
	       (send stream :string-out line)))
	   (when (> (incf i inc) length)
	     (return))))))
    
    (t (cerror "2Give up writing this sound diagram.*" "2This is a version ~D sound diagram; current version is ~D.*"
	       version 2)))
  
  (terpri stream))


;1;; Thanks to Paul Rutter (*philabs!per1) and Joe Orost (*petsd!joe1) for writing*
;1;; the C program on which these 5btoa* and 5atob* functions are based.*


(defun 4btoa *(8b-array into-8b-array &optional (start1 0) (end1 (length 8b-array)) (start2 0) (end2 (length into-8b-array)))
  "2Convert 8-bit binary into 7-bit ASCII, packing each 4 8-bit bytes into 5 ASCII bytes.*"
  (declare (type (array (unsigned-byte 8) 1)
		 8b-array into-8b-array)
	   (optimize speed))
  ;1;*
  ;1; The C btoa program optimizes occurences of four consecutive zeros to the one character, 'z'.*
  ;1; We don't do that because if we did, we could not compute exactly how many lines the output*
  ;1; will occupy before actually encoding it.  This could probably be circumvented, but I am not sure*
  ;1; how much of a win that would be.*
  ;1;*
  (let* ((inc 33)	;1 *(char-code #\!)
;	 (zero-tag 122)	;1 *(char-code #\z)
	 (out (1- start2))
	 (85^2 #.(* 85 85))
	 (85^3 #.(* 85 85 85))
	 (85^4 #.(* 85 85 85 85))
	 )
    (declare (fixnum inc out)
	     (optimize speed))
    (do* ((i start1 (+ i 4)))
	 ((>= i end1)
	  (when (> out (- end2 start2))
	    (error "2Wrote too much!  Wanted ~D, wrote ~D.*" (- end2 start2) out))
	  )
      (let* ((32b (dpb    (aref 8b-array i)       (byte 8 24)
		   (dpb   (aref 8b-array (+ i 1)) (byte 8 16)
		    (dpb  (aref 8b-array (+ i 2)) (byte 8  8)
		     (dpb (aref 8b-array (+ i 3)) (byte 8  0) 0))))))
	(declare (type (unsigned-byte 32) 32b))
	;1;*
	;1; Bad news.  I can't think of how to do the 32 bit -> 40 bit conversion*
	;1; without using 32 bit numbers.  This means we spend a whole bunch of*
	;1; time up in bignum-land, since fixnums are 24 bits...*
	;1;*
;	(cond ((zerop 32b)
;	       (setf (aref into-8b-array (incf out)) zero-tag))
;	      (t
	       (setf (aref into-8b-array (incf out)) (+ inc (floor 32b 85^4))
		     32b (mod 32b 85^4))
	       (setf (aref into-8b-array (incf out)) (+ inc (floor 32b 85^3))
		     32b (mod 32b 85^3))
	       (setf (aref into-8b-array (incf out)) (+ inc (floor 32b 85^2))
		     32b (mod 32b 85^2))
	       (setf (aref into-8b-array (incf out)) (+ inc (floor 32b 85))
		     32b (mod 32b 85))
	       (setf (aref into-8b-array (incf out)) (+ inc 32b))
;	       ))
	))
    (1+ out)))


(defun 4atob *(8b-array into-8b-array &optional (start1 0) (end1 (length 8b-array)) (start2 0))
  (declare (type (array (unsigned-byte 8) 1)
		 8b-array into-8b-array)
	   (optimize speed))
  (let* ((inc 33)	;1 *(char-code #\!)
	 (zero-tag 122)	;1 *(char-code #\z)
	 (out start2)
	 (bcount 0))
    (declare (fixnum inc zero-tag out bcount))
    (do* ((i start1 (1+ i))
	  (word 0))
	 ((>= i end1)
	  (assert (zerop bcount) () "2Premature end of data.  Read ~D bytes, need ~D more.*" i (- 5 bcount)))
      (declare (fixnum i)
	       (type (unsigned-byte 32) word))
      (let* ((c (aref 8b-array i)))
	(cond ((= c zero-tag)
	       (setf (aref into-8b-array out) 0
		     (aref into-8b-array (+ 1 out)) 0
		     (aref into-8b-array (+ 2 out)) 0
		     (aref into-8b-array (+ 3 out)) 0
		     out (+ out 4)
		     bcount 0))
	      (t
	       (cond ((= bcount 0)
		      (setf word (- c inc)
			    bcount (1+ bcount)))
		     ((< bcount 4)
		      (setf word (+ (* 85 word) (- c inc))
			    bcount (1+ bcount)))
		     (t
		      (setf word (+ (* 85 word) (- c inc))
			    (aref into-8b-array out)	   (ldb (byte 8 24) word)
			    (aref into-8b-array (+ 1 out)) (ldb (byte 8 16) word)
			    (aref into-8b-array (+ 2 out)) (ldb (byte 8  8) word)
			    (aref into-8b-array (+ 3 out)) (ldb (byte 8  0) word)
			    out (+ out 4)
			    bcount 0)))))))
    out))


(defmethod 4(sound-diagram :string-for-file*) (line)
  "2Returns the Epsilon-Hash format string necessary to write the diagram line to a file.  This string is extremely big...*"
  (let* ((lines (send self :number-of-lines))
	 (chars-per-line (case version (1 70) (2 75)))
	 (chars (+ 200 (* lines (1+ chars-per-line))))	;1 2roughly* how many characters.*
	 (string (make-array chars :element-type 'string-char :fill-pointer 0)))
    ;1;*
    ;1; We make the string ahead of time so that it doesn't have to be adjusted when we overflow it.*
    (with-output-to-string (stream string)
      (when (send self :first-line-p line)
	(let ((*package* nil))
	  (format stream "~C# ~D ~S~%" #\Epsilon lines (type-of self)))
	(send self :contents stream)))
    string))


(defmethod 4(sound-diagram :contents)* (&optional stream)
  "2Returns the string necessary to write the contents of the sound-array.  If STREAM is provided, then the data 
  is written there rather than being returned as a string.*"
  (setf read-state 'OUTPUT) ;1 Debugging.*
  (if stream
      (write-sound-array-as-ascii sound-array stream version name)
      (with-output-to-string (stream) (send self :contents stream))))


(defmethod 4(sound-diagram :draw*) (line-to-draw sheet)
  "2Invoked by Zmacs redisplay to draw one pline of the Diagram.*"
  (send self :update-lines)
  (let* ((i  (getf (line-plist line-to-draw) :diagram-line-number))
	 (sx (min (w:sheet-cursor-x sheet)
		  (w:sheet-inside-right sheet)))
	 (sy (w:sheet-cursor-y sheet))
	 (lh (tv:sheet-line-height sheet))
	 (y  (and i (* i lh)))
	 (dw (- (w:sheet-inside-right sheet) sx))
	 (dh (send self :height-in-pixels))
	 (screen-array (w:sheet-screen-array sheet)))
    
    (cond (image-bit-array
	   ;1;*
	   ;1; If there is a bit-array already, then we just need to blit the appropriate rows of the bit-array onto the screen.*
	   ;1;*
	   (when (> (* lh (1+ i)) dh)
	     (setq lh (1+ (rem dh lh))))
	   (when (< y dh)
	     (tv:prepare-sheet (sheet)
	       (bitblt W:ALU-SETA dw lh image-bit-array 0 (max 0 (1- y)) screen-array sx sy))))
	  (t
	   ;1;*
	   ;1; If there is not a bit-array, then we need to actually draw (and generate the bit array from what we drew).*
	   ;1;*
	   (when (zerop i)
	     (let* ((font FONTS:tr18)
		    (fh (tv:font-char-height font))
		    (fy (+ sy (floor (- (round (+ dh (* 3 fh)) 2) (* fh 4)))))
		    (fx (+ sx (* 3 dh)))
		    (*read-base* 10)
		    (*print-base* 10)
		    (sec-string (string-right-trim "30.*" (format nil "3~2,F*" (float (/ (length sound-array) 8000)))))
		    (string1 "3This is a Sound Array.*")
		    (string2 (format nil "3Name:    ~A*" (if name (string-capitalize (string name)) "3<untitled>*")))
		    (string3 (format nil "3Length:  ~A second~P*" sec-string (parse-integer sec-string :junk-allowed t))))
	       (case version
		 (1 (draw-horn sx sy dh dh sheet))
		 (2 (draw-horn-2 sx sy dh dh sheet))
		 (t (send sheet :string-out-explicit "3?*" (+ sx 5) (+ sy 15) dw (+ y dh) fonts:cmr18 w:alu-transp)))
	       (send sheet :string-out-explicit string1 fx (+ fy fh)       dw (+ y dh) fonts:tr12bi w:alu-transp)
	       (send sheet :string-out-explicit string2 fx (+ fy (* 2 fh)) dw (+ y dh) font w:alu-transp)
	       (send sheet :string-out-explicit string3 fx (+ fy (* 3 fh)) dw (+ y dh) font w:alu-transp)
	       )
	     (setq image-bit-array (make-array (list dh (array-dimension screen-array 1)) :type (array-type screen-array)))
	     (tv:prepare-sheet (sheet)
	       (bitblt W:ALU-SETA dw dh screen-array sx sy image-bit-array 0 0))
	     )))
    ))


(defun 4draw-horn *(x y w h sheet)
  "2Draw the Sound-Diagram icon for version 1.  *SHEET2 must include* W:GRAPHICS-MIXIN2.*"
  (let* ((thickness (float (/ w 15)))
	 (thick (floor thickness))
	 (thick2 (floor (* 2 thickness)))
	 (thick3 (floor (* 3 thickness)))
	 (thick/2 (round thickness 2))
	 (color-p (w:color-system-p sheet))
	 (bg (if color-p W:RED W:50%-GRAY-COLOR))
	 (fg W:BLACK)
	 (border (tv:sheet-foreground-color sheet)))
    (incf x thick/2)
    (incf y thick/2)
    (decf w thick)
    (decf h thick)
    (when (and (not color-p) (or (< w 30) (< h 30))) (setq bg W:WHITE))
    (send sheet :draw-filled-rectangle x y w h bg w:alu-seta)
    (send sheet :draw-rectangle x y w h thick border w:alu-transp)
    (send sheet :draw-triangle (+ x (floor w 2) thick) (+ y (floor h 2))
			       (+ x (- w thick2)) (+ y thick3)
			       (+ x (- w thick2)) (+ y (- h thick3))
			       thick fg w:alu-transp)
    (send sheet :draw-rectangle (+ x thick3) (+ y thick3) (round w 4) (- h thick3 thick3) thick fg w:alu-transp)
    ))

(defun 4draw-horn-2 *(x y w h sheet)
  "2Draw the Sound-Diagram icon for version 2.  *SHEET2 must include* W:GRAPHICS-MIXIN2.*"
  (let* ((thickness (float (/ w 15)))
	 (thick (floor thickness))
	 (thick/2 (round thickness 2))
	 (color-p (w:color-system-p sheet))
	 (bg (if color-p W:RED W:50%-GRAY-COLOR))
	 (fg W:BLACK)
	 (border (tv:sheet-foreground-color sheet)))
    (incf x thick/2)
    (incf y thick/2)
    (decf w thick)
    (decf h thick)
    (when (and (not color-p) (or (< w 30) (< h 30))) (setq bg W:WHITE))
    (send sheet :draw-filled-rectangle x y w h bg w:alu-seta)
    (decf x thick)
    (send sheet :draw-arc (+ x (round w 2)) (+ y (round h 2)) (+ x (floor w 2)) (+ y thick) -180 thick fg)
    (send sheet :draw-arc (+ x (round w 2)) (+ y (round h 2)) (+ x (floor w 2)) (+ y (* thick 3)) -180 thick fg)
    (send sheet :draw-arc (+ x (round w 2)) (+ y (round h 2)) (+ x (floor w 2)) (+ y (* thick 5)) -180 thick fg)
    (send sheet :draw-filled-circle (+ x (round w 2)) (+ y (round h 2)) thick/2 fg)
    (send sheet :draw-filled-triangle (+ x (round w 2)) (- (+ y (round h 2)) thick)
				      (+ x (round w 2)) y
				      (+ x (round (* w 4/5))) y
				      bg w:normal nil nil t)
    (send sheet :draw-filled-triangle (+ x (round w 2)) (+ (+ y (round h 2)) thick)
				      (+ x (round w 2)) (+ y h)
				      (+ x (round (* w 4/5))) (+ y h)
				      bg w:normal nil nil t)
    (send sheet :draw-arc (+ x (round w 3)) (+ y (round h 2)) (+ x (floor w 3) thick) (+ y (* thick 6)) 180 thick fg)
    (send sheet :draw-filled-circle (+ x (round w 3) thick/2) (+ y (round h 2)) thick/2 fg)
    (send sheet :draw-line (+ x (round w 3) thick/2) (+ y (round h 2))
			   (+ x (round w 2)) (+ y (- h (* 2 thick)))
			   thick fg w:normal nil)
    (send sheet :draw-filled-circle (+ x (round w 2)) (+ y (- h (* 2 thick))) thick/2 fg)
    (send sheet :draw-rectangle (+ x thick) y w h thick border w:alu-transp)
    ))


(defmethod 4(sound-diagram :height-in-pixels*) () 100)	;1 I like this number.*

(defmethod 4(sound-diagram :height-in-lines*) ()
  (1+ (ceiling (send self :height-in-pixels)
	       (+ (send *window* :vsp) (tv:sheet-line-height *window*)))))



;1;; The next few methods are more-or-less copied from the like-named methods of* DOX-DIAGRAM1.*  1I could just inherit*
;1;; *DOX-DIAGRAM1 to get these,* 1but then, sound-diagrams would be SHEETs as well.*  1There's no need for that - I assert that*
;1;; these should not be on *DOX-DIAGRAM1, but rather* 1should be* 1on an intermediate flavor between *DOX-DIAGRAM1 and*
;1;; *RESTORABLE-LINE-DIAGRAM-MIXIN1 - say,* MULTI-LINE-DISPLAYING-DIAGRAM-MIXIN1....*

(defmethod 4(sound-diagram :first-line-p*) (line)
  (zerop (getf (line-plist line) :diagram-line-number)))

(defmethod 4(sound-diagram :last-line-p*) (line)
  (cond ((and line (eq (getf (line-plist line) :diagram) self))
	 (or (null (setq line (line-next line)))
	     (neq (getf (line-plist line) :diagram) self)))))

(defmethod 4(sound-diagram :update-lines*) ()
  (when first-line
    (do ((i    (1- (send self :height-in-lines)) (1- i))
	 (line first-line next)
	 (next (line-next first-line) (line-next next)))
	((not (plusp i))
	 (do ((end line (line-next end)))
	     ((send self :last-line-p end)
	      (setq end (line-next end))
	      (setf (line-next line) end)
	      (when end (setf (line-previous end) line)))))
      (when (send self :last-line-p line)
	(dotimes (j i)
	  (let ((new (copy-line line (line-node line))))
	    (setf (line-next line) new)
	    (setf (line-previous new) line)
	    (setf (line-plist new) (copy-list (line-plist line)))
	    (incf (getf (line-plist new) :diagram-line-number))
	    (setq line new)))
	(setf (line-next line) next)
	(setf (line-previous next) line)
	(return)))
    (send self :mung)))

(defmethod 4(sound-diagram :mung*) ()
  (when first-line
    (do ((line first-line (line-next line)))
	((send self :last-line-p line)
	 (mung-line line))
      (mung-line line))))



;1;; The user interface - playing and recording Sound Diagrams.*
;1;;*

(defmethod 4(sound-diagram :play)* ()
  (setf (get :sound-diagram 'tv:sound-array) sound-array)
  (tv:play :sound-diagram))

(defun 4play-sound-diagram-line *(line)
  (send (getf (line-plist line) :diagram) :play))


(defcom 4com-play-sound-buffer*
	"2Play the sound-diagrams in the current buffer.
  If the current buffer is a Mail Summary Buffer, then play the sounds in
  the corresponding sequence buffer instead.*" ()
  (let* ((mail-summary-p (typep *interval* 'zwei:mail-summary-buffer))
	 (interval (if mail-summary-p
		       (send *interval* :sequence-buffer)
		       *interval*)))
    (do ((line (bp-line (interval-first-bp interval)) (line-next line))
	 (got-one nil))
	((null line)
	 (unless got-one (barf (if mail-summary-p
				   "3This message does not have any sounds.*"
				   "3There are no sounds in this buffer.*"))))
      (when (eql 0 (getf (line-plist line) :diagram-line-number))
	(setq got-one t)
	(format *query-io* "3~&Playing sound ~A...*" (or (send (getf (line-plist line) :diagram) :name) "3<untitled>*"))
	(play-sound-diagram-line line)
	(format *query-io* "3 done.~%*"))))
  DIS-NONE)


(defcom 4com-record-and-insert-sound-diagram*
	"2Records a sampled sound, and inserts a dagram-line of the sound into the buffer at point.*"
	()
  (let* ((name (intern (substitute #\- #\Space
				   (string-upcase (completing-read-from-mini-buffer "3Name of this sample*: " nil t nil
						3    *"3Type the name of what you are about to record.*")))
		       "3KEYWORD*"))
	 (length (ceiling (* 8000 (prompt-and-read :number "3How many seconds? *"))))
	 (symbol (gensym))
	 (line (bp-line (point))))
    (when (string= name "") (setq name nil))
    (unless (string= "" line)			;1 Can only insert diagrams on empty lines (else redisplay will crash...)*
      (insert-moving (point) #\Newline)
      (setq line (bp-line (point))))
    (let* ((ok (loop
		 (format *query-io* "3~&Type any character to begin recording...*")
		 (read-char)
		 (tv:record symbol (ceiling (/ length 8000)))
		 (format *query-io* "3  done.~%*")
		 (when
		   (loop
		     (tv:play symbol)
		     (case (fquery '(:choices (((:y  "3Yes.*") #\Y #\)
					       ((:n  "3No*.") #\N #\)
					       ((:?  "3I Don't Know.*") #\? #\Space)))
				3   *"3Is this sample acceptable? *" )
		       (:y (return t))
		       (:n (return nil))))
		   (return t)))))
      (when ok
	(let* ((new-line (create-line 'art-string 0 *interval*))
	       (diag (make-instance 'sound-diagram :name name :sound-array (get symbol 'tv:sound-array)
				    :first-line new-line)))
	  
	  ;1; Diagram lines can't come at the very front of the buffer.*
	  (when (eq line (bp-line (interval-first-bp *interval*)))
	    (insert (interval-first-bp *interval*) #\Newline)) ;1 This makes LINE be the second line of the interval.*
	  ;1;*
	  ;1; Install the diagram info on the line's plist.*
	  (setf (getf (line-plist new-line) :diagram) diag)
	  (setf (getf (line-plist new-line) :diagram-line-number) 0)
	  ;1;*
	  ;1; Insert the line into the buffer.*
	  (insert-line-with-leader new-line line)
	  ;1;*
	  ;1; For the diagram line to be read back properly, the buffer must have at least two fonts (hey, these aren't my rules!)*
	  ;1; So if the buffer doesn't have at least two fonts already, then pick a couple out of the air.*
	  (unless (cdr (send *interval* :get-attribute :fonts))
	    (send  *interval* :set-attribute :fonts '(:cptfont :hl12b) t))
	  ))))
  DIS-TEXT)


(set-comtab *zmacs-comtab* () '(("3Play Sound Buffer*" . com-play-sound-buffer)
				("3Record and Insert Sound Diagram*" . com-record-and-insert-sound-diagram)))
