;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER; VSP:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B) -*-

;1;; File "3VIDEO-TAPE*"*
;1;; Reading a human-readable list of videotape contents, and producing postscript output to print the labels.*
;1;; Operative functions are *READ-TAPE-FILE1, *LOOKUP-TAPE1, and *WRITE-LABEL-FILE1.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   16 Feb 88*	1Jamie Zawinski*	1Created.*
;1;;   24 Apr 88*	1Jamie Zawinski*	1Got it working.*
;1;;    7 Feb 89*	1Jamie Zawinski*	1Added *END-COUNTER1s.*
;1;;*  127 Nov 89*	1Jamie Zawinski *	1Documented file format.*
;1;;*

(in-package "3USER*")
(export '(lookup-tape read-tape-file write-label-file))


;1;; The format of the tape file is:*
;1;;*
;1;; 5Video Tape Entry:**	2<first-line>3 [ *<next-line>3 ]* [ *<blank-line>3 ]***
;1;;*
;1;; 5First Line:**		1[ 2<wspace>* ]  5'Tape '3 2<id>* ':' 2<wspace>* 2<name>* [ 2<wspace>* 2<counter>* [ 2<wspace>* 2<end-counter>* ]]***
;1;;*
;1;; 5Next Line:**		2<wspace> <name>3 [ *<wspace> <counter>3 [ *<wspace> <end-counter>3 ]]**
;1;;*
;1;; 5Whitespace:**		1one or more tabs, or two or more spaces.*
;1;;*
;1;;5 Name:**			1Any string; embedded spaces are allowed, so long as no two are consecutive.*
;1;; *			1This is the name of the movie.*
;1;;*
;1;; 5Counter:**		1The time or counter-position at which the movie begins.*
;1;; 5End Counter:**		1The time or counter-position at which the movie ends.  *
;1;; *			1This is useful only for the last movie on the tape.*
;1;;*
;1;; And, by example:*
;1;;*
;1;; Tape 17 has three movies, none of which have known counters.*
;1;; Tape 28 has four movies, all of which have known counters.*
;1;; Tape 45 has three movies, all of which have known counters.  The end-counter for the last one is known as well.*
;1;; Tape 48 only has one movie (so far), and it's end-counter is known.*
;1;;*
;1;;*       Tape 017:    If...
;1;;*                    Harold and Maude
;1;;*                    The Falcon and the Snowman
;1;;*
;1;;*       Tape 028:    The Thing                                        0
;1;;*                    Attack of the Killer Tomatoes                 1844
;1;;*                    Time Bandits                                  2840
;1;;*                    Max Headroom                                  3922
;1;;*
;1;;*       Tape 045:    Angel Heart                                      0
;1;;*                    Hellraiser                                    1909
;1;;*                    Prince of Darkness                            2974  3929
;1;;*
;1;;*       Tape 048:    Phantasm                                         0  1909
;1;;*
;1;;*
;1;; Blank lines are ignored, as are all lines before the first line matching the 5First-Line* template.*
;1;; So you can have whatever comments you like at the beginning of the file.*
;1;;*
;1;; The Postscript produced for some of the above tapes would produce side-labels like this:*
;1;; *
;1;; */--------------------------------------------\
;1;; *| 0 |  If...					|
;1;; *| 1 |  Harold and Maude			|
;1;; *| 7 |  The Falcon and the Snowman		|
;1;; *\--------------------------------------------/
;1;; *
;1;; */--------------------------------------------\
;1;; *| 0 |  The Thing			     0	|
;1;; *| 2 |  Attack of the Killer Tomatoes	   1844	|
;1;; *| 8 |  Time Bandits			   2840	|
;1;; *|   |  Max Headroom			   3922	|
;1;; *\--------------------------------------------/
;1;; *
;1;; */--------------------------------------------\
;1;; *| 0 |  Angel Heart			     0	|
;1;; *| 4 |  Hellraiser			   1909	|
;1;; *| 5 |  Prince of Darkness		   2974	|  ;1 note no end-counter*
;1;; *\--------------------------------------------/
;1;; *
;1;; */--------------------------------------------\
;1;; *| 0 |  Phantasm			     0	|
;1;; *| 4 |  				   1909	|  ;1 note use of end-counter, since there is spare room.*
;1;; *| 8 |						|3  ;1 Uses 3 lines minimum.**
;1;; *\--------------------------------------------/
;1;; *
;1;; Top-labels are produced in a similar manner; except that end-counters are always printed, since*
;1;; the top label always has room to burn.*


#-LISPM
(eval-when (load eval compile)
  (defun 4string-append *(&rest strings-or-chars) ;1 2quick and dirty.**
    (apply #'concatenate 'simple-string (mapcar #'string strings-or-chars))))


(defstruct 4(tape* (:print-function %print-tape)
		 (:constructor make-tape (id lines)))
  (id "" :type string)
  (lines () :type list))


(defstruct 4(tape-line* (:print-function %print-tl)
		      (:constructor make-line (string counter &optional end-counter)))
  (string      "" :type string)
  (counter     "" :type (or null string))
  (end-counter "" :type (or null string)))


(defun 4%print-tape* (struct stream depth)
  (declare (ignore depth))
  (format stream "3#<TAPE ~A ~A>*" (tape-id struct) (tape-lines struct)))


(defun 4%print-tl* (struct stream depth)
  (declare (ignore depth))
  (format stream "3#<TAPE-LINE ~A ~A*"
	  (or (tape-line-string struct) "")
	  (or (tape-line-counter struct) "3?*"))
  (when (tape-line-end-counter struct) (format stream "3 ~A*" (tape-line-end-counter struct)))
  (princ #\> stream)
  )

(defvar 4*video-tapes** nil)

(defun 4whitespace-p* (char) (or (char= char #\Space) (char= char #\Tab)))

(defun 4parse-tape-line* (line)
  (let* ((chomp-pos (search "3tape *" line :test #'char-equal)))
    (if chomp-pos
	(let* ((label-end (position #\: line :start chomp-pos :test #'char=))
	       (name-start (position-if-not #'whitespace-p line :start (1+ label-end)))
	       (name-end (search "3  *" line :start2 (or name-start 0) :test #'char=))
	       (counter-start (and name-end (position-if-not #'whitespace-p line :start (1+ name-end))))
	       (counter-end (and counter-start (position-if #'whitespace-p line :start (1+ counter-start))))
	       (end-counter-start (and counter-end (position-if-not #'whitespace-p line :start (1+ counter-end))))
	       (label (subseq line (+ chomp-pos 5) label-end))
	       (name (subseq line (or name-start 0) name-end))
	       (counter (and counter-start (subseq line counter-start counter-end)))
	       (end-counter (and end-counter-start (subseq line end-counter-start)))
	       )
	  (push (make-tape label (list (make-line name counter end-counter))) *video-tapes*))
	(when *video-tapes*
	  (let* ((name-start (position-if-not #'whitespace-p line))
		 (name-end (search "3  *" line :start2 (or name-start 0) :test #'char=))
		 (counter-start (and name-end (position-if-not #'whitespace-p line :start (1+ name-end))))
		 (counter-end (and counter-start (position-if #'whitespace-p line :start (1+ counter-start))))
		 (end-counter-start (and counter-end (position-if-not #'whitespace-p line :start (1+ counter-end))))
		 (name (subseq line (or name-start 0) name-end))
		 (counter (and counter-start (subseq line counter-start counter-end)))
		 (end-counter (and end-counter-start (subseq line end-counter-start)))
		 )
	    (setf (tape-lines (car *video-tapes*))
		  (append (tape-lines (car *video-tapes*))
			  (list (make-line name counter end-counter)))))))))

(defun 4read-tape-file* (&optional (file "3spice:/usr/jwz/misc/doc/tape-list.text*"))
  (setq *video-tapes* nil)
  (with-open-file (s file :direction :input)
    (do* ((line (read-line s nil nil) (read-line s nil nil)))
	 ((null line))
      (parse-tape-line line))))

(defun 4print-tape-readably* (tape &optional (stream *standard-output*))
  (let* ((id (tape-id tape))
	 (lines (tape-lines tape)))
    (format stream "3~&~7tTape ~A:*" id)
    (dolist (line lines)
      (let* ((string (tape-line-string line))
	     (counter (tape-line-counter line))
	     (end-counter (tape-line-end-counter line)))
	(format stream "3~18t  ~A~:[~;~48,8t  ~:*~@4A~]~:[~;  ~:*~@4A~]~%*" string counter end-counter)))))


;1;; writing PS*


(defun 4print-label* (tape x y &optional top-p (stream *standard-output*))
  (let* ((id (tape-id tape))
	 (lines (tape-lines tape))
	 (nlines (length lines))
	 (last-line (car (last (remove nil (tape-lines tape) :key #'tape-line-counter))))
	 )
    (unless (and last-line (tape-line-end-counter last-line)) (setq last-line nil))
    (when (and (not top-p) (>= nlines 3)) (setq last-line nil))
    (when (< nlines (if last-line 2 3))
      (setq lines (concatenate 'list lines (make-array (- 3 nlines) :initial-element (make-line "" nil)))
	    nlines 3))
    (format stream "3~&~%~D inch ~9t~D inch ~18t(~A)*" x y (or id ""))
    (dolist (line lines)
      (let* ((counter (tape-line-counter line))
	     (string (prin1-to-string (tape-line-string line)))
	     (no-dqs (subseq string 1 (- (length string) 1))))
	(format stream "3~24t(~A)  (~A)~%*" no-dqs (or counter ""))))
    (when last-line
      (format stream "3~24t()  (~A)~%*" (tape-line-end-counter last-line))
      (incf nlines))
    (format stream "3~18tdraw-~A-label-~D~%*"
	    (if top-p "3top*" "3side*") nlines)))


(defun 4draw-top-labels* (&optional (stream *standard-output*) (tapes *video-tapes*))
  (let* ((x 0.5) (y 10))
    (dolist (tape (reverse tapes))
      (print-label tape x y t stream)
      (cond ((= x 0.5)
	     (setq x 4.0))
	    (t (setq x 0.5)
	       (decf y 2)
	       (when (< y 2)
		 (format stream "3~&~%showpage~%~%*")
		 (setq y 10)))))))

(defun 4draw-side-labels* (&optional (stream *standard-output*) (tapes *video-tapes*))
  (let* ((x 0.5) (y 10))
    (dolist (tape (reverse tapes))
      (print-label tape x y nil stream)
      (decf y 1)
      (when (< y 1)
	(format stream "3~&~%showpage~%~%*")
	(setq y 10)))))

(defun 4draw-labels* (&optional (stream *standard-output*) (tapes *video-tapes*) (top-p t) (side-p t))
  (when top-p
    (draw-top-labels stream tapes)
    (format stream "3~&~%showpage~%~%*"))
  (when side-p
    (draw-side-labels stream tapes)
    (format stream "3~&~%showpage~%*"))
  )

(defun 4write-label-file* (file &key (tapes *video-tapes*) (top-p t) (side-p t))
  (unless (listp tapes) (setq tapes (list tapes)))
  (when (or top-p side-p)
    (with-open-file (s file :direction :output)
      (format s "3%!ps to print videotape labels.~%~%*")
      (write-prolog s)
      (when side-p
	(write-internal-side-proc s)
	(dotimes (i 4)
	  (write-side-proc (+ i 3) s)))
      (when top-p
	(write-internal-top-proc s)
	(dotimes (i 4)
	  (write-top-proc (+ i 3) s)))
      (format s "3~%~%% The actual labels...~%~%~%*")
      (draw-labels s tapes top-p side-p))))


(defun 4write-prolog* (&optional (stream *standard-output*))
  (format stream "3/rightshow { dup stringwidth pop neg 0 rmoveto show } def~%*")
  (format stream "3/centershow { dup stringwidth pop neg 2 div 0 rmoveto show } def~%*")
  (format stream "3/box { newpath /h exch def /w exch def /y exch def /x exch def x y moveto *")
  (format stream "3w 0 rlineto 0 h neg rlineto w neg 0 rlineto closepath } def~%*")
  (format stream "3/inch { 72 mul } def~%~%*"))


(defun 4write-internal-side-proc* (&optional (stream *standard-output*))
  (princ #.(string-append #\Newline
			  "3/internal-draw-side-label {*" #\Newline
			  "3     gsave*" #\Newline
			  "3     0 0 moveto*" #\Newline
			  "3     x y translate*" #\Newline
			  "3     1 setlinejoin*" #\Newline
			  "3     0.5 setgray*" #\Newline
			  "3     0 0 5.75 inch 0.75 inch  box fill*" #\Newline
			  "3     0 setgray*" #\Newline
			  "3     0 0 5.75 inch  0.75 inch  box stroke*" #\Newline
			  "3     1 inch -0.1 inch  4.25 inch 0.55 inch box fill*" #\Newline
			  "3     0.1 inch -0.1 inch  0.25 inch 0.55 inch box fill*" #\Newline
			  "3     1 setgray*" #\Newline
			  "3     1 inch -0.1 inch  4.25 inch 0.55 inch box stroke*" #\Newline
			  "3     0.1 inch -0.1 inch  0.25 inch 0.55 inch box stroke*" #\Newline
			  "3     gsave*" #\Newline
			  "3       /Helvetica-Bold findfont [21 0 0 18 0 0] makefont setfont*" #\Newline
			  "3       0.324 inch -0.375 inch moveto*" #\Newline
			  "3       90 rotate*" #\Newline
			  "3       number centershow*" #\Newline
			  "3     grestore*" #\Newline
			  "3 } def*" #\Newline
			  )
	 stream)
  nil)

(defun 4write-side-proc* (n &optional (stream *standard-output*))
  (format stream "3~&~%/draw-side-label-~D~%   {*" n)
  (dotimes (i n)
    (format stream "3~5t/mc~D exch def  /m~D exch def~%*" (- i n) (- i n)))
  (format stream "3~5t/number exch def~%~5t/y exch def  /x exch def~%*")
  (format stream "3~%~5t internal-draw-side-label~%*")
  (let* ((font-height (/ 13.7 72))
         (height 0.72) ;1 the height of the area we are finning N lines into.*
         (row (/ height n)))  ;1 the height of each row.*
    (format stream "3~%     /Helvetica-Bold findfont [11 0 0 ~D 0 0] makefont setfont~%*"
	    (case n (3 14)
	            (4 10)
		    (5 8)
		    (6 6)))
    (dotimes (i n)
      (let* ((num-str  (truncate-decimal (float (+ 0.02
						   (* height
						      (+ (* (1+ i) row)
							 (/ font-height 2)))))
					 4)))
        (format stream "3~%~5t1.05 inch -~A inch moveto  m-~D  show*" num-str (1+ i))
        (format stream "3~%~5t 5.2 inch -~A inch moveto  mc-~D rightshow*" num-str (1+ i)))))
  (format stream "3~%~%~5tgrestore~%~5t} def~%~%*")
  )

(defun 4write-internal-top-proc* (&optional (stream *standard-output*))
  (princ #.(string-append #\Newline
			  "3/internal-draw-top-label { *" #\Newline
			  "3     gsave*" #\Newline
			  "3     0 0 moveto*" #\Newline
			  "3     x y translate*" #\Newline
			  "3     *" #\Newline
			  "3     1 setlinejoin*" #\Newline
			  "3     *" #\Newline
			  "3     0.5 setgray*" #\Newline
			  "3     0 0 3.1 inch 1.75 inch box fill*" #\Newline
			  "3     0 setgray *" #\Newline
			  "3     0 0 3.1 inch 1.75 inch box stroke*" #\Newline
			  "3     0.1 inch  -0.5 inch  2.9 inch  1.15 inch  box fill*" #\Newline
			  "3     0.1 inch  -0.1 inch  2.9 inch  0.3 inch   box fill*" #\Newline
			  "3     1 setgray*" #\Newline
			  "3     0.1 inch  -0.5 inch  2.9 inch  1.15 inch  box stroke*" #\Newline
			  "3     0.1 inch  -0.1 inch  2.9 inch  0.3 inch   box stroke*" #\Newline
			  "3     *" #\Newline
			  "3     /Helvetica-Bold findfont [12 0 0 15 0 0] makefont setfont*" #\Newline
			  "3     0.2 inch -0.3 inch moveto   (Tape )show  number show*" #\Newline
			  "3     /Helvetica-Bold findfont [8 0 0 15 0 0] makefont setfont*" #\Newline
			  "3     2.9 inch -0.3 inch moveto   (JWZ)rightshow*" #\Newline
			  "3     /Helvetica-Bold findfont [9 0 0 10 0 0] makefont setfont*" #\Newline
			  "3 } def*" #\Newline
			  )
	 stream)
  nil)

(defun 4write-top-proc* (n &optional (stream *standard-output*))
  (format stream "3~&~%/draw-top-label-~D~%   {*" n)
  (dotimes (i n)
    (format stream "3~5t/mc~D exch def  /m~D exch def~%*" (- i n) (- i n)))
  (format stream "3~5t/number exch def~%~5t/y exch def  /x exch def~%*")
  (format stream "3~%~5t internal-draw-top-label ~%*")
  (dotimes (i n)
    (format stream "3~%~5t0.2 inch -0.75 inch ~2,D sub moveto m-~D  show*" (* i 10) (1+ i))
    (format stream "3~%~5t2.9 inch -0.75 inch ~2,D sub moveto mc-~D rightshow*" (* i 10) (1+ i)))
  (format stream "3~%~%~5tgrestore~%~5t} def~%~%*")
  )


(defun 4truncate-decimal* (float n)
  (let* ((string (princ-to-string float))
	 (dot-pos (position #\. string)))
    (subseq string 0 (if dot-pos
			 (min (length string) (+ dot-pos n 1))
			 nil))))


;1;; Some extremely complicated database stuff*

(defun 4lookup-tape* (string)
  "2Returns a list of tapes which have STRING in them somewhere.*"
  (let* ((result nil))
    (dolist (tape *video-tapes*)
      (when (or (search string (tape-id tape) :test #'char-equal)
		(dolist (line (tape-lines tape) nil)
		  (when (search string (tape-line-string line) :test #'char-equal)
		    (return T))))
	(push tape result)))
    (nreverse result)))


;1;; Here's how we do the big ones:*
;1;;*
;1;; *0.5 inch 4 inch   (026) ()() ()() ()() ()() ()() draw-top-label
;1;; *1 setgray    0.7 inch 2.45 inch moveto    /Helvetica-Bold findfont [60 0 0 95 0 0] makefont setfont     (Videos)show
;1;;*
;1;; *4.0 inch  6 inch   (035) ()() ()() ()() ()() ()() draw-top-label
;1;; *1 setgray    4.2 inch 4.65 inch moveto    /Helvetica-Bold findfont [48 0 0 75 0 0] makefont setfont   (Videos II)show
;1;; *             4.2 inch 4.45 inch moveto    /Helvetica-Bold findfont [15 0 0 15 0 0] makefont setfont   (   The Sequel.)show
;1;;*
;1;; *0.5 inch 4 inch   (026) ()() ()() ()() draw-side-label
;1;; *1 setgray    1.6 inch 3.4 inch moveto    /Helvetica-Bold findfont [25 0 0 45 0 0] makefont setfont     (Videos)show
;1;;*
;1;; *0.5 inch 5 inch   (035) ()() ()() ()() draw-side-label
;1;; *1 setgray    1.6 inch 4.4 inch moveto    /Helvetica-Bold findfont [25 0 0 45 0 0] makefont setfont
;1;; *            (Videos II:  the Sequel.)show
