;;; -*- Mode:Common-Lisp; Package:SYSTEM-INTERNALS; Fonts:(CPTFONT HL10B TR10BI); Base:10 -*-
;;; Copyright (c) 1986, Texas Instruments Incorporated.  All Rights Reserved.

;1;; Support for a Timer Queue, of the same functionality as the Symbolics facility*


;2TIMER QUEUE Facility*

;2The timer queue provides the capability of posting functions that are to be executed*
;2at specified times.  Optionally, these events can be repeated for a certain number*
;2of times or simply repeated forever.  Use of this facility should eliminate many*
;2existing uses of spawning off separate processes that wait for a specified interval*
;2of time and then become active.*

;2Typical uses of this tool are: to periodically monitor a situation (ie, the weather,*
;2hosts/users on the network, the availablity of a resource), or to set off*
;2alarms/notifications at specific times.*

;2When the specified time occurs, the specified function is called within its own*
;2process.  Thus, the timer queue process is analogous to the Explorer scehduler and*
;2the events to individual processes.  It doesn't matter how long the event execution*
;2occurs; timer queue servicing should proceed normally despite the number of timer*
;2events active.*

;2The user-accessible functions are:*

;2ADD-TIMER-QUEUE-ENTRY <Time> <Repeat> <Name> <Function> &rest <Args>*
;2REMOVE-TIMER-QUEUE-ENTRY <Timer-ID>*
;2PRINT-TIMER-QUEUE &Optional <Stream>*

;2The ID-Number that is returned by ADD-TIMER-QUEUE-ENTRY and is an*
;2argument to REMOVE-TIMER-QUEUE-ENTRY is simpy an integer that is used*
;2to identify this particular timer entry instance, and is sequentially incremented*
;2for new events.*

(PROVIDE 'timer-queue)

(EXPORT '(add-timer-queue-entry remove-timer-queue-entry print-timer-queue))

(DEFVAR *timer-queue-process* :unbound "2The process controlling the timer queue.*")

(DEFVAR *timer-queue-event-list* nil "2The list of timer events.*")
(PROCLAIM '(type list *timer-queue-event-list*)) ;1;for the compiler*

(DEFVAR *timer-queue-counter* 0 "2The current counter for timer queue events*")
(PROCLAIM '(type integer *timer-queue-counter*)) ;1;for the compiler*

(DEFSTRUCT (timer-queue-entry
	     (:print-function
	       (lambda (entry stream depth)
		 (DECLARE (IGNORE depth))
		 (si:printing-random-object (entry stream :typep)
		   (FORMAT stream "~d:~a" (timer-queue-entry-id entry) (timer-queue-entry-name entry)))))
	     )
 2 *"2Each event on the timer queue is an instance of this structure.*"
  time
  repeat
  name
  function
  arguments
  id
  )


(DEFRESOURCE timer-queue-entries ()
  :constructor (make-timer-queue-entry)
  :initial-copies 0)

(DEFUN add-timer-queue-entry (time repeat name function &rest args)
  "2Adds something to the timer queue.  
TIME is either (:ABSOLUTE <ut>) (or simply <ut>) (or simply a string), or (:RELATIVE <sec>) or (:DELTA <sec>)
Absolute time specifies at what time the function will be activated.
Relative time specifies the activation time relative to the current time.
REPEAT is :ONCE, (:FOREVER <sec>), or (<n-times> <sec>).
NAME is a string
FUNCTION is the thing to do when the event is activated (a symbol, fef, or closure).*"
  (DECLARE (VALUES id-number))
  (LET ((new-entry (ALLOCATE-RESOURCE 'timer-queue-entries))
	(entry-time (COND ((INTEGERP time) time)
			  ((STRINGP time)
			   (time:parse-universal-time time))
			  ((AND (CONSP time)
				(EQ :absolute (FIRST time)))
			   (SECOND time))
			  ((AND (CONSP time)
				(MEMBER (FIRST time) '(:relative :delta)))
			   (+ (SECOND time) (GET-UNIVERSAL-TIME))))))
    (SETF (timer-queue-entry-time new-entry) entry-time)
    (SETF (timer-queue-entry-repeat new-entry) repeat)
    (SETF (timer-queue-entry-name new-entry) name)
    (SETF (timer-queue-entry-function new-entry) function)
    (SETF (timer-queue-entry-arguments new-entry) (COPY-LIST args))
    (SETF (timer-queue-entry-id new-entry) (WITHOUT-INTERRUPTS (INCF *timer-queue-counter*)))
    (WITHOUT-INTERRUPTS
      (SETQ *timer-queue-event-list* (SORT (CONS new-entry *timer-queue-event-list*) #'< :key #'timer-queue-entry-time)))
    (timer-queue-entry-id new-entry)
    ))

(DEFUN (:property add-timer-queue-entry :undo-function) (form)
  "2Removes the timer-queue-entry of the specified NAME (the ID number is not used).
This is important for activation functions that conditionally add another timer-queue to the system.*"
  `(remove-timer-queue-entry ,(FOURTH form)))

(DEFUN remove-timer-queue-entry (timer-id &optional (free-entry-object-p t))
  "2Removes the entry from the timer queue.
If TIMER-ID is an integer, then the timer queue entry with that ID is removed.
If TIMER-ID is a string, then the timer queue entry with that NAME is removed.
TIMER-ID can also be a time-queue-entry object.*"
  (DECLARE (VALUES entry-removed-p))
  (LET ((entry (TYPECASE timer-id
		 (integer
		  (FIND (THE integer timer-id) *timer-queue-event-list* :key #'timer-queue-entry-id :test #'=))
		 (string
		  (FIND (THE string timer-id) *timer-queue-event-list* :key #'timer-queue-entry-name :test #'STRING-EQUAL))
		 (timer-queue-entry ;1;a special case used internally*
		  timer-id))))
    (WHEN entry
      (WITHOUT-INTERRUPTS
	(SETQ *timer-queue-event-list* (DELETE entry *timer-queue-event-list* :test #'EQ))
	)
      (WHEN free-entry-object-p
	(DEALLOCATE-RESOURCE 'timer-queue-entries entry))
      t))
  )

(DEFUN print-timer-queue (&optional (STREAM *standard-output*))
  "2Prints the contents of the timer queue.*"
  (COND
    ((NULL *timer-queue-event-list*)
     (FORMAT stream "~%The timer queue is empty."))
    (t
     (FORMAT stream "~%Event ~10t~a" (timer-queue-header-string))
     (LOOP for entry in (COPY-LIST *timer-queue-event-list*) ;1;make a copy because it may otherwise change from under us.*
	     for entry-count from 1
	     do
	     (FORMAT stream "~%[~2d]~10t~a"
		     entry-count
		     (timer-queue-entry-description-string entry)
		     ))))
  )

(DEFUN timer-queue-header-string ()
  (FORMAT nil "ID Num ~10tActivation Time ~30tName ~55tFunction to Call"))

(DEFPARAMETER timer-queue-max-function-name-length 50.
  "2Used for displaying information about name of the function.*")

(DEFUN timer-queue-entry-description-string (entry)
  "2Returns a one-line string describing the timer queue entry.*"
  (LABELS ((function-string
	     (funct)
	     ;1;handle the various types of functions and return a meaningful string name*
	     (TYPECASE funct
	       (STRING funct)
	       ((OR compiled-function microcode-function)
		(PRIN1-TO-STRING (function-name funct)))
	       (CLOSURE
		(function-string (CLOSURE-FUNCTION funct)))
	       (symbol
		(PRIN1-TO-STRING funct))
	       (t
		(PRIN1-TO-STRING funct))))
	   (one-line-string-of-width
	     (str)
	     ;1;Ensure that the string is short enough and doesn't span multiple lines*
	     (declare (string str))
	     (STRING-SUBST-CHAR #\space #\newline 
				(IF (> (LENGTH str) timer-queue-max-function-name-length)
				    (STRING-APPEND
				      (SUBSEQ (the string str) 0 timer-queue-max-function-name-length)
				      "...")
				    str))))
    (FORMAT nil "~4,'0d ~10t~a ~30t~20a ~55t~:A"
	    (timer-queue-entry-id entry)
	    (time:print-universal-time (timer-queue-entry-time entry) nil)
	    (timer-queue-entry-name entry)
	    (LET ((f (timer-queue-entry-function entry)))
	      (one-line-string-of-width (function-string f))))))

(DEFUN timer-queue-process-function ()
  ;1;wait for the *timer-queue-event-list* to be non-empty and the time of the next event to pass.*
  (loop
    (PROCESS-WAIT "Wait for Event" #'(lambda () (AND *timer-queue-event-list*
						     (<= (timer-queue-entry-time (CAR *timer-queue-event-list*))
							 (GET-UNIVERSAL-TIME)))))
    (LET ((entry (CAR *timer-queue-event-list*)))
      (PROCESS-RUN-FUNCTION (timer-queue-entry-name entry) #'run-entry entry)
      (remove-timer-queue-entry (timer-queue-entry-id entry) nil)))) ;1don't free resource here because it's needed in the new process*


(DEFUN run-entry (entry)
  (APPLY (timer-queue-entry-function entry) (timer-queue-entry-arguments entry))
  (LET ((entry-repeat (timer-queue-entry-repeat entry)))
    (WHEN (CONSP entry-repeat)
      (COND
	((EQ (CAR entry-repeat) :forever)
	 (APPLY #'add-timer-queue-entry
		(+ (SECOND entry-repeat) (GET-UNIVERSAL-TIME))
		entry-repeat
		(timer-queue-entry-name entry)
		(timer-queue-entry-function entry)
		(timer-queue-entry-arguments entry)))
	((AND (INTEGERP (CAR entry-repeat))
	      (> (CAR entry-repeat) 0))
	 (add-timer-queue-entry (+ (SECOND entry-repeat) (GET-UNIVERSAL-TIME))
				(IF (= (CAR entry-repeat) 1)
				    :once
				    (SETF (CAR entry-repeat) (1- (CAR entry-repeat)))
				    entry-repeat)
				(timer-queue-entry-name entry)
				(timer-queue-entry-function entry)
				(timer-queue-entry-arguments entry))))))
  (remove-timer-queue-entry entry) ;1we can finally get rid of this.  *
  (VALUES))

(DEFUN start-timer-queue ()
  "2Starts the timer queue process and bashes any existing timer queue processes.*"
  (LET ((timer-queue-process-name "Timer Queue"))
  (DOLIST (process (COPY-LIST all-processes))
    (WHEN (STRING-EQUAL (SEND process :name) timer-queue-process-name)
      (SEND process :kill)))
  (SETQ *timer-queue-process* (PROCESS-RUN-FUNCTION `(:name ,timer-queue-process-name
						      :restart-after-reset t
						      :priority 5
						      :restart-after-boot t)
						    #'timer-queue-process-function))))

(ADD-INITIALIZATION "Start Timer Queue"
		    '(start-timer-queue)
		    '(:now :warm))






;1;;*
;1;; FILE-Watching utility*
;1;; Notifies the user whenever the modification date for a file changes*
;1;;*

(DEFVAR *file-watch-alist* nil
  "2Alist of (file last-date message).*")

(DEFSTRUCT (watched-file (:type list))
  file
  last-date
  message)

(DEFVAR *file-watch-timer-interval* (* 60 5) ;15 minutes*
  "2Number of seconds between each check for changed files.*")

(DEFCONSTANT file-watch-timer-queue-name "File Watch")

(DEFUN check-for-any-changed-files ()
  (LOOP for file-alist-element in *file-watch-alist*
	for file = (watched-file-file file-alist-element)
	and last-date = (watched-file-last-date file-alist-element)
	for modification-date = (file-watch-file-modification-date file)
	when (< last-date modification-date)
	do
	(tv:notify nil (watched-file-message file-alist-element))
	(SETF (watched-file-last-date file-alist-element) modification-date))
  (si:add-timer-queue-entry `(:relative ,*file-watch-timer-interval*) :once file-watch-timer-queue-name
			    #'check-for-any-changed-files))

;1;;Be sure not to get screwed because of accessing files when no longer logged in.*
(ADD-INITIALIZATION "Clear File-Watch" '(SETQ *file-watch-alist* nil) :logout)

(si:add-timer-queue-entry '(:relative 0) :once file-watch-timer-queue-name
			  #'check-for-any-changed-files)

(DEFUN file-watch-file-modification-date (file)
  (LET ((file-property-list (IGNORE-ERRORS (fs:file-properties file))))
    (OR (GET file-property-list :modification-date)
	(GET file-property-list :creation-date)
	0)))

(DEFUN watch-file (file &optional message)
  "2Monitors FILE at regular intervals (si:*file-watch-timer-interval*), and notifies you
when the file has been changed.
The notification is MESSAGE, or if not provided, it is 'The file <name> has been changed.'*"
  (SETF file (MERGE-PATHNAMES file))
  (WHEN (NULL message)
    (SETF message (FORMAT nil "The file, ~a, has been changed." file)))
  (WITHOUT-INTERRUPTS
    (PUSH (make-watched-file :file file
			     :last-date (file-watch-file-modification-date file)
			     :message message)
	  *file-watch-alist*))
  )

(DEFUN (:property watch-file :undo-function) (form)
  `(delete-watched-file ,(SECOND form)))

(DEFUN delete-watched-file (file)
  "2Removes file from the set of watched files.*"
  (SETF file (MERGE-PATHNAMES file))
  (WITHOUT-INTERRUPTS
    (SETQ *file-watch-alist* (DELETE file *file-watch-alist* :key #'watched-file-file))))

(DEFUN display-watched-files (&optional (stream *standard-output*))
  nil)



;1;;;*
;1;;; Host watching utility*
;1;;;  Let's you know when a host (currently just Chaos) changes its up/down status*
;1;;;*

(EXPORT 'watch-host) ;1;A User Function*

(DEFSTRUCT (watched-host (:type list))
  host
  up-p
  function
  )

(DEFVAR *watched-hosts* nil
  "2List of watched hosts*")

(DEFUN host-up-p (host)
  (let ((host (si:parse-host host t)))
    (cond ((null host) nil)
	  ((send host :chaos-address)
	   (chaos:host-up host))
	  ((send host :ip-address)
	   (ip:icmp-echo host 1 ip:icmp-normal-echo-size nil))
	  )))

(DEFUN watch-host (host &optional (function 'host-watch-notify))
  (PUSH (make-watched-host :host host
			   :up-p (host-up-p host)
			   :function function)
	*watched-hosts*))

(DEFUN host-watch-notify (watched-host)
  (tv:notify nil "Host ~a is now ~:[down~;up~]"
	     (watched-host-host watched-host)
	     (watched-host-up-p watched-host)))

(DEFUN watch-hosts ()
  (LOOP for (host up-p function) in *watched-hosts* ;1;careful about the defstruct above*
	and watched-host in *watched-hosts*
	for host-up-p = (host-up-p host)
	unless (EQ host-up-p up-p)
	do
	(SETF (watched-host-up-p watched-host) host-up-p)
	(FUNCALL function watched-host)))

(si:add-timer-queue-entry '(:relative 0) '(:forever 60) "2Watch Hosts*"
			  'watch-hosts)

;1;;;*
;1;;; Patch Timer Queue into Peek Process display*
;1;;;*


(RECORD-SOURCE-FILE-NAME 'tv:peek-processes 'DEFUN t)

;1;; The function TV:PEEK-PROCESSES used to be redefined here, to include some *
;1;; additional code for displaying the timer queue.  I removed this redefinition,*
;1;; replacing it with Advice, which is more portable.  -- Jamie Zawinski.*

(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise tv:peek-processes :around peek-timer-queue nil
    ;1; Get the list that the real #'peek-processes would generate, and conc a list *
    ;1; describing the timer-queue onto the end of it.  This way, if the system code*
    ;1; for displaying processes changes, loading the timer-queue won't break Peek.*
    (nconc :DO-IT (tv:peek-processes-timer-queue))))

TV:
(defun peek-processes-timer-queue ()
  (list
	;1;We did a little extra hackery here because Peek only willingly updates changed entries and*
	;1;the timer queue stuff recycles old entries (new entries are EQ to old, discarded entries).*
	;1;To get around the problem, Peek uses timer-queue entries with cons-cell wrappers around.*
	;1;Peek will update the entry when a CONS cell changes.*
	;1;The cons consists of (<entry id number> . <entry>)*
	;1;These conses are only created when in Peek and then only once per Timer-Queue-Entry.*
	(SCROLL-PARSE-ITEM "")
	(SCROLL-PARSE-ITEM "Timer Queue")
	(scroll-parse-item (si:timer-queue-header-string))
	(scroll-maintain-list #'(lambda ()
				  (WITHOUT-INTERRUPTS 
				    (MAPCAR #'(lambda (entry)
						(OR (ASSOC (si:timer-queue-entry-id entry) *peek-timer-queue-cache-alist*
								:test #'=)
						    (LET ((new-cons (CONS (si:timer-queue-entry-id entry) entry)))
						      (SETQ *peek-timer-queue-cache-alist*
							    (DELETE entry *peek-timer-queue-cache-alist* :key #'CDR))
						      (PUSH new-cons *peek-timer-queue-cache-alist*)
						      new-cons)))
					    si:*timer-queue-event-list*)))
			      #'(lambda (timer-entry)
				  (scroll-parse-item
				    `(:mouse-item
				       (nil :eval (peek-timer-queue-menu ',timer-entry)
					    :documentation "Menu of things to do to this timer queue entry.")
				       :string ,(si:timer-queue-entry-description-string (CDR timer-entry))
				       )))
			      )
	))


tv:
(DEFUN peek-timer-queue-menu (timer-entry)
  (PROCESS-RUN-FUNCTION "Timer Queue Entry Menu"
			#'(lambda (timer-entry)
			    (LET* ((timer-entry-id (si:timer-queue-entry-id  timer-entry))
				   (funct (w:menu-choose
					   `(("Delete this entry" :value ,#'(lambda (entry)
									      (si:remove-timer-queue-entry
										(si:timer-queue-entry-id entry)))
					      :documentation "Remove this timer queue entry from the queue.")
					     ("Change activation time"
					      :value
					      ,#'(lambda (entry)
						   (LET* ((new-time (choose-new-timer-entry-time entry)))
						     (WHEN new-time
						       (WITHOUT-INTERRUPTS
							 (SETF (si:timer-queue-entry-time entry) new-time)
							 ;1;Be sure that Peek display doesn't use old information*
							 (SETQ *peek-timer-queue-cache-alist*
							    (DELETE entry *peek-timer-queue-cache-alist* :key #'CDR))
							 (SETQ si:*timer-queue-event-list* (SORT si:*timer-queue-event-list* #'<
										      :key #'si:timer-queue-entry-time)))
						       ))))
					     ,(LET ((repeat (si:timer-queue-entry-repeat timer-entry)))
						(IF (AND (CONSP repeat) (EQ (CAR repeat) :forever))
						    `("Change Repeat Interval"
						      :value
						      ,#'(lambda (entry)
							   (LET* ((new-time (choose-new-timer-entry-repeat entry)))
							     (WHEN new-time
							       (WITHOUT-INTERRUPTS
								 (SETF (SECOND (si:timer-queue-entry-repeat entry)) new-time)
							       )))))
						    '("" :no-select nil))))
					   :label
					   (FORMAT nil "Timer Entry #~d: ~a"
						   (si:timer-queue-entry-id timer-entry)
						   (si:timer-queue-entry-name timer-entry)))))
			      (WHEN funct
				(IF (= timer-entry-id (si:timer-queue-entry-id timer-entry))
				    (FUNCALL funct timer-entry)
				    (tv:notify nil "Timer Entry has already been removed.  No action taken.")))
			      ))
			(CDR timer-entry)))

tv:
(DEFUN choose-new-timer-entry-time (entry)
  "2Select a new activation time to this timer entry.*"
  (LET ((activation-time (si:timer-queue-entry-time entry)))
    (DECLARE (SPECIAL activation-time))
    (IF (CATCH 'choose
	  (tv:choose-variable-values
	    '((activation-time "Activation Time" :documentation "Set activation time for this timer queue entry" :date))
	    :label (FORMAT NIL "Activation Time for Timer Queue entry #~d: ~a"
			   (si:timer-queue-entry-id entry)
			   (si:timer-queue-entry-name entry))
	    :margin-choices '(("Do It" (THROW 'choose T)) "Abort")))
	activation-time
	nil)))

tv:
(DEFUN choose-new-timer-entry-repeat (entry)
  "2Select a new repeat time for this timer entry.*"
  (LET ((repeat-time (SECOND (si:timer-queue-entry-repeat entry))))
    (DECLARE (SPECIAL repeat-time))
    (IF (CATCH 'choose
	  (tv:choose-variable-values
	    '((repeat-time "Repeat Time" :documentation "Set repeat time for this timer queue entry (in seconds)." :fixnum))
	    :label (FORMAT NIL "Repeat Time for Timer Queue entry #~d: ~a"
			   (si:timer-queue-entry-id entry)
			   (si:timer-queue-entry-name entry))
	    :margin-choices '(("Do It" (THROW 'choose T)) "Abort")))
	repeat-time
	nil)))

