;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:(MKSYS use (LISP)) -*-

;;; File MKSYS.LISP
;;; Blah blah it does something else.
;;; Written and maintained by Jamie Zawinski;
;;; jwz@spice.cs.cmu.edu or jwz@teak.berkeley.edu.
;;;
;;; ChangeLog:
;;;
;;;  8 Feb 88  Jamie Zawinski    Created.
;;; 10 Feb 88  Jamie Zawinski    Made forward references possible.
;;; 13 Feb 88  Jamie Zawinski    Fixed bug where load-time was getting set in COMPILE-FILE-RECORD.
;;;                              Defined PPRINT-SYSTEM.
;;; 16 Feb 88  Jamie Zawinski    Made it such that parameter spec lists are evaluated while component specs are not.
;;;                              Previously neither was evaluated.
;;;                              Assured that there is a :DIRECTORY parameter supplied.
;;; 18 Feb 88  Jamie Zawinski    Added trapping of circular component dependancies.
;;; 19 Feb 88  Jamie Zawinski    Added REPORT-LOAD-COUNT.
;;; 26 Feb 88  Jamie Zawinski    Added binary type "fasl" for #+FRANZ-INC.
;;; 30 Apr 88  Jamie Zawinski    Changed the pretty-printer to be more accurate.
;;;                              Fixed the field-parser to always preserve the definition-order, even when there had been
;;;                               a forward-reference.
;;;                              Added *MKSYS-CACHE-FILE-DATES* after realizing how slow this was on an Explorer...
;;;  2 May 88  Jamie Zawinski    Added a NEEDS-COMPILED slot to COMPONENT, to properly propagate dependencies when a
;;;                               component doesn't have any files.
;;;                              Fixed LOOKUP-COMPONENT to sometimes create forward-referenced systems as well as
;;;                               components when given a descriptor of the form (<system> <component>).
;;;                              Fixed PARSE-COMPONENT-LIST to barf when the user passes in a fully-qualified pathname as
;;;                               the file of the component.
;;; 10 May 88  Jamie Zawinski    Defined DESCRIBE-SYSTEM.
;;;                              Made LOAD-FILE-RECORD use truenames, which means version numbers get printed.
;;; 12 May 88  Jamie Zawinski    Added caching of truenames, and defined a TI-specific way of getting the truename and
;;;                               write-date in one fell swoop.  This should speed things up quite a bit.
;;; 14 May 88  Jamie Zawinski    Added history tracking.
;;;                              Added code to deal with version numbers on file systems which support them.
;;;  8 Jun 88  Jamie Zawinski    Fixed the bug where redefining a system blasted forward dependancies.
;;;                              Introduced a new, less bad bug, described below.
;;;                              Made MKSYS gleefully accept quoted system names.
;;;                              Added some more history tracking.
;;; 24 Jun 88  Jamie Zawinski    Fixed a bug that only exhibited itself in TI rel 4.1 - FILE-RECORD-SOURCE-PATHNAME was not
;;;                               defaulting VERSION to :NEWEST.
;;; 29 Jun 88  Jamie Zawinski    Fixed the multiple dependancy system redefinition bug.
;;;                              Added #+EXPLORER source file recording.
;;;  7 Jul 88  Jamie Zawinski    Added a better error message when the source and binary don't exist.
;;; 27 Jul 88  Jamie Zawinski    The LAST-ONLY arg to HISTORY wasn't working right.  Fixed it.
;;; 29 Jul 88  Jamie Zawinski    Implemented EDIT-SYSTEM and COPY-SYSTEM.
;;;  7 Sep 88  Jamie Zawinski    Added a Resume case for the error signalled in COPY-SYSTEM-FILES when the destination
;;;                               directory of a subsystem couldn't be determined.
;;;  4 Oct 88  Jamie Zawinski    Added the :COPY-CHANGED option.
;;;                              Made the copying of file-versions be more sane - if the dest directory contains a newer
;;;                               or equal version as the source, copy a new version.  Otherwise, copy the original version
;;;                               number.
;;; 18 Jan 89  Jamie Zawinski 	 Added the NEVER-LOAD-P and BINARY-ONLY slots to COMPONENT, and allowed the 
;;;                               :NOLOAD and :BINARY-ONLY keywords as part of a DEFINE-SYSTEM component spec.
;;; 29 Mar 89  Jamie Zawinski 	 Made it be possible for DEVICE, DIRECTORY, and/or VERSION to be :UNSPECIFIC, since they
;;;				  get parsed that way ont the TI when the SYSTEM-DIRECTORY is a logical pathname.
;;; 25 Jan 90  Jamie Zawinski 	 Minor tweaking to make it compile under Lucid 4.0.

;;;
;;; KNOWN BUGS:
;;; -----------
;;;
;;; o  If a user goes outside of MKSYS to do compilation, dependancies are not updated properly.
;;;    For example - the user calls MKSYS to load a system.  The user then makes a change to some file in the system,
;;;    and calls COMPILE-FILE.  The user calls MKSYS again.  MKSYS will load the new binary, but will not mark components
;;;    which depend on that file as needing recompilation.  This is because of the way the write-date comparison is done.
;;;    The only solution I can think of is this:
;;;
;;;    Instead of having NEEDS-COMPILED flags and forward pointers to dependants, determine whether compilation is
;;;    by comparing the source-date of the current file with the write dates of EVERY file on which this file depends.
;;;    This means walking the tree orders of magnitude more, and means throwing out a lot of code.
;;;
;;;    The other solution is to convince people to never call COMPILE-FILE.
;;;
;;; o  If the user begins a MKSYS, and a file is compiled on which other files depend, then the only way to tell that the
;;;    dependant files need to be recompiled is that the corresponding FILE-RECORDs will be marked accordingly.
;;;    If the user aborts before the entire MKSYS is processed, and restarts the MKSYS, everything will work out fine,
;;;    because the data structures are intact.  But if the user reboots before restarting the MKSYS, or even reloads the
;;;    makefiles, those marked dependancies will be lost - there is no way to reconstruct them.  
;;;
;;;    This problem is solved by the same two methods as the previous problem...
;;;

;;; ## There should be a catcher for 'ABORT-FILE that one can throw to...  So when a file is not found, one can Resume, 
;;; ## and the file will not be prompted ofr in the future.



(in-package "MKSYS" :use '("LISP"))

#+EXPLORER (eval-when (load eval compile) (unuse-package '("TICL") "MKSYS"))	; just in case...


(export '(define-system mksys unload-system pprint-system describe-system edit-system *mksys-cache-file-dates*
	  history clear-history))

(defstruct (system (:print-function %print-system))
  (name         nil :type (or keyword null))
  (directory    nil :type (or pathname null))    ; This is a pathname of which only the DIRECTORY component is used.
  (package      nil :type (or package null))     ; The default package to load files into.
  (components   nil :type list)   ; A list of the FILE-RECORDs contained in this system.
  (dependancies nil :type list)   ; A list of the SYSTEMs on which this one depends.
  (dependants   nil :type list)   ; A list of the SYSTEMs which depend on this one.
  (forward-reference-p nil :type (member T NIL))
  )

(defstruct (component (:print-function %print-component))
  (name           nil :type (or keyword null))
  (system         nil :type (or system null))    ; The SYSTEM to which this component belongs.
  (file-records   nil :type list)                ; The FILE-RECORDs comprising this system.
  (dependancies   nil :type list)       ; A list of COMPONENTs which must be compiled and loaded before this.
  (dependants     nil :type list)       ; A list of COMPONENTs which must be compiled and loaded after this.
  (forward-reference-p nil :type (member T NIL))
  (needs-compiled nil :type (member T NIL))
  (never-load-p   nil :type (member T NIL))      ; If T, then this component is never loaded.
  (binary-only    nil :type (member T NIL))      ; If T, then the files of this component never have a source version.
  )

(defstruct (file-record (:print-function %print-file-record))
  (component        nil :type (or null component))        ; The COMPONENT to which this file belongs.
  (name             nil :type (or null string pathname))  ; Used as the NAME component of a pathname.
  (load-date        nil :type (or null integer))
  (source-file-date nil :type (or null integer))
  (binary-file-date nil :type (or null integer))
  (source-truename  nil :type (or null pathname))
  (binary-truename  nil :type (or null pathname))
  (needs-compiled   nil :type (member T NIL))
  )

(defun %print-system (struct stream depth)
  (declare (ignore depth))
  (format stream "#<~:[~;FORWARD-REFERENCED-~]SYSTEM ~A>"
	  (system-forward-reference-p struct)
	  (system-name struct)))

(defun %print-component (struct stream depth)
  (declare (ignore depth))
  (format stream "#<~:[~;FORWARD-REFERENCED-~]COMPONENT ~A of ~A: ~:S>"
	  (component-forward-reference-p struct)
	  (component-name struct)
	  (system-name (component-system struct))
	  (if (= (length (component-file-records struct)) 1)
	      (file-record-name (car (component-file-records struct)))
	      (mapcar #'file-record-name (component-file-records struct)))))

(defun %print-file-record (struct stream depth)
  (declare (ignore depth))
  (format stream "#<FILE-RECORD ~S of ~A:~A>"
	  (file-record-name struct)
	  (system-name (component-system (file-record-component struct)))
	  (component-name (file-record-component struct))))

(defun compile-component-p (component)
  "T if any of the file records in COMPONENT need to be compiled.
   SETFing this sets the NEEDS-COMPILED slot of all of the file records of this component."
  (unless (or (component-never-load-p component)
	      (component-binary-only component))
    (or (component-needs-compiled component)
	(some 'compile-file-record-p (component-file-records component)))))

(defsetf compile-component-p (component) (newval)
  `(progn (setf (component-needs-compiled ,component) ,newval)
	  (dolist (file-record (component-file-records ,component))
	    (setf (file-record-needs-compiled file-record) ,newval))
	  ,newval))

;;
;; Install some doc-strings on the more arbitrarily named slot-accessors.
;;
(setf (documentation 'system-dependancies 'function)
      "A list of the SYSTEMs on which this one depends - this system is a subsystem of these.")
(setf (documentation 'system-dependants 'function)
      "A list of the SYSTEMs which depend on this one - these are the subsystems of this system.")
(setf (documentation 'component-dependancies 'function)
      "A list of COMPONENTs which must be compiled and loaded before this.")
(setf (documentation 'component-dependants 'function)
      "A list of COMPONENTs which must be compiled and loaded after this, or more accurately,
a list of components before which this component must be compiled.")


(defvar *mksys-verbosity* nil)
(defvar *mksys-interactive-p* nil)
(defvar *mksys-print-only* nil)
(defvar *mksys-never-compile* nil)

;;; These are let-bound in MKSYS-1.
(defvar *load-count* 0 "Bound to a count of files loaded, for the user's benefit.")
(defvar *compile-count* 0 "Bound to a count of files compiled, for the user's benefit.")


(defconstant SOURCE-TYPE "lisp")
(defconstant BINARY-TYPE #+(and EXPLORER (not ELROY)) "xfasl"
                         #+(and EXPLORER ELROY)       "xld"
                         #+LUCID                      "lbin"
			 #+FRANZ-INC                  "fasl"
                         )


;;; MKSYS history code.

(defvar *mksys-history* () "This a list describing what MKSYS has done; see #'MKSYS:HISTORY.")

(defun clear-history ()
  "Throw away the MKSYS history record."
  (setq *mksys-history* nil))

(defun push-history (action data &optional (time (get-universal-time)))
  "Makes a MKSYS history entry."
  (check-type action keyword)
  (check-type time integer)
  (push (list action data time) *mksys-history*)
  action)

(defun history-mksys-tag ()
  "Push a marker into history saying that this is the beginning of a new MKSYS."
  (push :mksys *mksys-history*))

(defun history (&key (last-only t) (load t))
  "Show what MKSYS has done.  If LAST-ONLY is T, then show what was done the last time MKSYS was called.
  If it is NIL, then show what was done since the last time CLEAR-HISTORY was called.
  If LOAD is NIL, then LOADs will not be reported, only compilations and system definitions."
  (dolist (entry (if last-only
		     (let* ((pos (position :MKSYS *mksys-history* :from-end nil)))
		       (reverse (subseq *mksys-history*
					(or pos 0)
					(position :MKSYS *mksys-history* :start (1+ (or pos 0))))))
		     (reverse *mksys-history*)))
    (when (eq entry :mksys) (format t "~&~%-----------------~%~%"))
    (when (consp entry)
      (let* ((action (car entry))
	     (thing (second entry))
	     (time (third entry)))
	(flet ((report (string &rest args)
		 (format t "~&~A:~30t ~?~&" (pretty-time time) string args))
	       (component-verbose-name (comp)
		 (format nil "~A of ~A"
			     (component-name comp)
			     (system-name (component-system comp)))))
	  (case action
	    (:DEFINE     (report "defined system ~A" thing))
	    (:REDEFINE   (report "REdefined system ~A" thing))
	    (:PATCH-DEPENDANCIES (report "patched dependancies of ~A of ~A:~%~{~20t~A~%~}"
					 (component-name (second thing))
					 (system-name (first thing))
					 (mapcar #'component-verbose-name (cddr thing))))
	    (:LOAD       (and load (report "loaded ~A" (car thing))))
	    (:FAKE-LOAD  (and load (report "pretended to load ~A" (car thing))))
	    (:ABORT-LOAD (report "gave up on ~A" (car thing)))
	    (:COMPILE      (report "compiled ~A" (car thing)))
	    (:FAKE-COMPILE (report "pretended to compile ~A" (car thing)))
	    (:PROCESS-SYSTEM (report "start to process system ~A" (system-name thing)))
	    (:MUST-COMPILE   (report "marked for recompilation:~{~58t~A~%~}"
				     (mapcar #'component-verbose-name thing)))
	    (t (report "~A ~A" action thing)))))))
  nil)


;;; File system interface


(defparameter *mksys-cache-file-dates* #+EXPLORER t
                                       #-EXPLORER nil
  "This variable controls how frequently the dates of the source and binary files are updated.
  If it is T, then the cached dates will be updated only when they are not known, or when a file has been compiled.
  If it is NIL, then the cached dates will be updated quite frequently.
  Frequent updating has the effect of making MKSYS much more accurate - if a file has been edited after the MKSYS
  was begun, we will realize this and compile the file again.
  This variable should be T if you have a slow file-system.")


(defun file-record-source-pathname (file-record)
  "Returns the pathname to the source file that FILE-RECORD represents.  This file may or may not exist."
  (check-type file-record file-record)
  (let* ((component (file-record-component file-record))
	 (system (component-system component))
	 (directory (system-directory system))
	 (name (file-record-name file-record)))
    (if (pathnamep name)
	(merge-pathnames name (make-pathname :type SOURCE-TYPE :version :NEWEST :defaults directory))
	(make-pathname :name name :type SOURCE-TYPE :version :NEWEST :defaults directory))))

(defun file-record-binary-pathname (file-record)
  "Returns the pathname to the binary file that FILE-RECORD represents.  This file may or may not exist."
  (check-type file-record file-record)
  (let* ((component (file-record-component file-record))
	 (system (component-system component))
	 (directory (system-directory system))
	 (name (file-record-name file-record)))
    (if (pathnamep name)
	(merge-pathnames name (make-pathname :type BINARY-TYPE :version :NEWEST :defaults directory))
	(make-pathname :name name :type BINARY-TYPE :version :NEWEST :defaults directory))))


#-EXPLORER
(defun file-write-date-or-nil (filename)
  "If FILENAME does not exist, return NIL.
  Otherwise, return two values: the universal time on which it was written, and it's truename."
  (let* ((truename (probe-file filename)))
    (if truename
	(values (file-write-date truename) truename)
	nil)))

;;; On the Explorer (which, incidentally, has a really slow file system) it is possible
;;; to implement this much more efficiently.
;;;
#+EXPLORER
(defun file-write-date-or-nil (filename)
  "If FILENAME does not exist, return NIL.
  Otherwise, return two values: the universal time on which it was written, and it's truename.
  This is faster than calling PROBE-FILE as well as FILE-WRITE-DATE."
  (declare (values universal-time truename))
  (ticl:condition-case (stream)
      (open filename :direction nil)
    (fs::open-deleted-file   nil)
    (fs::file-not-found      nil)
    (fs::directory-not-found nil)
    (:no-error (multiple-value-prog1
		 (values (ticl:send stream :creation-date)
			 (ticl:send stream :truename))
		 (close stream)))))


(defun update-file-dates (file-record &optional force)
  "Go out to disk to find the write dates of the source and binary corresponding to FILE-RECORD.
  If FORCE is non-NIL, then we will not use cached information.  Specify this when you know the dates have changed."
  (check-type file-record file-record)
  ;; When we are in PRINT-ONLY mode, we sometimes need to lie about when the binary file was created;
  ;; Since in print-only, we don't actually write files, we NEVER update the dates once they have meaningful values.
  ;; 
  (unless (and (or *mksys-print-only*
		   (and (not force) *mksys-cache-file-dates*))
	       (or (file-record-source-file-date file-record)
		   (file-record-binary-file-date file-record)))
    (let* ((source (file-record-source-pathname file-record))
	   (binary (file-record-binary-pathname file-record))
	   source-date source-truename
	   binary-date binary-truename
	   (now (get-universal-time)))
      (multiple-value-setq (source-date source-truename) (file-write-date-or-nil source))
      (multiple-value-setq (binary-date binary-truename) (file-write-date-or-nil binary))
      
      (unless (or source-truename binary-truename)
	(cerror "Give up on this file." "Can't find source or binary corresponding to ~A." source)
	(format t "~&Giving up on ~A.~%" source)
	(push-history :ABORT-LOAD (cons binary file-record) now)
	(return-from UPDATE-FILE-DATES nil))

      ;; Random sanity checking.
      (when (and source-date (> source-date now)) (warn "~A claims to have been created in the future." source))
      (when (and binary-date (> binary-date now)) (warn "~A claims to have been created in the future." binary))
      ;; The actual do-it-up.
      (setf (file-record-source-file-date file-record) source-date
	    (file-record-binary-file-date file-record) binary-date
	    (file-record-source-truename file-record) source-truename
	    (file-record-binary-truename file-record) binary-truename)
      ))
  nil)



;;; Compilation and Loading

(defun any-compilation-warnings (pathname)
  "Returns T if the pathname had compilation warnings, NIL otherwise.  This is used so that the MKSYS output can be
  clever and preceed the ``compiled'' message with a newline and three semicolons if it will not be output on the same
  line as the ``compiling'' message.  What we really want is #'FRESH-LINE-P, but that's really hard to implement, so
  this is the next best thing."
  #+EXPLORER (not (null (member :COMPILE
				(sys:file-warnings-operations (ticl:send (pathname pathname) :generic-pathname))
				:test #'eq)))
  #-EXPLORER nil)




;;; SOURCE-FILE-NEWER-P duplicates code from PATHNAME-NEWER-P rather than calling it, because the file dates and 
;;; truenames can be gotten more efficiently from the cached information in a FILE-RECORD, when we have one.
;;;

(defun pathname-newer-p (pathname-a pathname-b)
  "Returns T if the first file is \"newer\" than the second.

  This is always true if the first has a later write date than the second, and is also true if the
  file system supports version numbers and the first has a higher version number than the second.

  If the second file is NIL, this returns T.  If the first file is NIL, this returns NIL.

  The second and third values are the truenames of the pathnames."
  (declare (values newer-p truename-a truename-b))
  (multiple-value-bind (date-a truename-a) (file-write-date-or-nil pathname-a)
    (multiple-value-bind (date-b truename-b) (file-write-date-or-nil pathname-b)
      (let* ((newer-p (cond ((null truename-b) T)
			    ((null truename-a) NIL)
			    (t (or (> date-a date-b) (file-version-> truename-a truename-b))))))
	(values newer-p truename-a truename-b)))))


(defun source-file-newer-p (file-record)
  "Returns T if the source file represented by FILE-RECORD is \"newer\" than the binary.

  This is always true if the source has a later write date than the binary, and is also true if the
  file system supports version numbers and the source has a higher version number than the binary.

  If there is no binary, this returns T.  If there is no source, this returns NIL."
  (update-file-dates file-record)
  (let* ((source (file-record-source-file-date file-record))
	 (binary (file-record-binary-file-date file-record)))
    (cond ((and source (not binary)) t)
	  ((and binary (not source)) nil)
	  (t (or (> source binary)
		 (file-version-> (file-record-source-truename file-record)
				 (file-record-binary-truename file-record)))))))


(defun file-version-> (pathname-1 pathname-2)
  "Returns T if the version number of PATHNAME-1 is greater than the version number of PATHNAME-2.
  This is only meaningful if the pathnames are both on filesystems which keep versions."
  (let* ((v1 (pathname-version pathname-1))
	 (v2 (pathname-version pathname-2)))
    (and (numberp v1)
	 (numberp v2)
	 (> v1 v2))))

(defun compile-file-record-p (file-record)
  "T if the file represented by FILE-RECORD needs to be compiled because of cached info or file dates."
  (if *mksys-never-compile*
      nil
      (or (file-record-needs-compiled file-record)
	  (setf (file-record-needs-compiled file-record)
		(source-file-newer-p file-record)))))


(defun load-file-record-p (file-record)
  "T if the file represented by FILE-RECORD needs to be loaded based on the file dates and last load date."
  (check-type file-record file-record)
  (let* ((load-date (file-record-load-date file-record))
	 (source-date (file-record-source-file-date file-record))
	 (binary-date (file-record-binary-file-date file-record)))
    (when (and source-date binary-date
	       (not *mksys-never-compile*)
	       (> source-date binary-date))
      (warn "The file ~A has changed since the MKSYS was begun.  The changes will not be reflected in this load."
	    (file-record-source-pathname file-record)))
    (or (null load-date)
	(and source-date (< load-date source-date))
	(and binary-date (< load-date binary-date)))))


(defun load-file-record (file-record)
  "Load the binary (or source if no binary) which FILE-RECORD represents, updating LOAD-DATE.
  Returns the pathname loaded, or NIL."
  (check-type file-record file-record)
  (update-file-dates file-record)
  (let* ((source (file-record-source-truename file-record))
	 (binary (file-record-binary-truename file-record))
	 (path (cond (binary binary)
		     (source source)
		     (t (cerror "Give up on this file." "Can't find source or binary corresponding to ~A." source)
			(format t "~&Giving up on ~A.~%" source)
			(push-history :ABORT-LOAD (cons (file-record-source-pathname file-record)
							file-record))
			nil)))
	 (ok t))
    (when path
      (if *mksys-interactive-p*
	  (setq ok (y-or-n-p "~&;;; Loading ~A, ok? " (namestring path)))
	  (format t "~&;;; Loading ~A... " (namestring path)))
      (when (and (if *mksys-interactive-p* ok t)
		 (not *mksys-print-only*))
	(force-output)
	(load path :verbose nil)
	(incf *load-count*)))
    (let* ((now (get-universal-time)))
      (setf (file-record-load-date file-record) now)
      (when path
	;;
	;; History
	;;
	(if *mksys-print-only*
	    (push-history :FAKE-LOAD (cons path file-record) now)
	    (push-history :LOAD      (cons path file-record) now))
	
	(format t "loaded.~%")))
    path))

(defun compile-file-record (file-record)
  "Compile the source file that FILE-RECORD represents.
   Updates cached file dates, and returns a pathname to the binary produced."
  (check-type file-record file-record)
  (let* ((source (file-record-source-pathname file-record))
	 (system (component-system (file-record-component file-record)))
	 (pack (or (system-package system) *package*))
	 (ok t)
	 binary)
    (if *mksys-interactive-p*
	(setq ok (y-or-n-p "~&;;; Compiling ~A, ok? " (namestring source)))
	(format t "~&;;; Compiling ~A... " (namestring source)))
    (when (and (if *mksys-interactive-p* ok t)
	       (not *mksys-print-only*))
      (let* ((*package* pack))
	(setq binary (compile-file source))
	(incf *compile-count*)))
    (force-output)
    (update-file-dates file-record (not *mksys-print-only*))  ; Force the update unless we're in print-only mode.
    (when *mksys-print-only*                                  ; If we're in print-only mode, LIE.  Fake the binary date.
      (setf (file-record-binary-file-date file-record) (get-universal-time)))
    (setf (file-record-needs-compiled file-record) nil)
    ;;
    ;; History.
    ;;
    (let* ((now (get-universal-time)))
      (if *mksys-print-only*
	  (push-history :FAKE-COMPILE (cons source file-record) now)
	  (push-history :COMPILE      (cons source file-record) now)))

    (if (and (not *mksys-print-only*) (any-compilation-warnings source))
	(format t "~&;;; compiled.~%")
	(format t "compiled.~%"))
    binary))

(defun compile-component (component)
  "Compile the file records of this component IF NECESSARY, and invalidate dependants."
  (check-type component component)
  ;;
  ;; Load (maybe compile) the components on which this component depends.
  (let* ((system (component-system component)))
    (dolist (comp (component-dependancies component))
      (let* ((system2 (component-system comp)))       ; It's possible for a component to depend on a component of another
	(unless (eq system system2)                   ; system.  In that case, we must process that other system before
	  (process-system system2 nil nil)))          ; compiling this component, to preserve that other system's load-order.
      (process-component comp)))
  (let* ((compile-dependants (compile-component-p component)))
    ;;
    ;; Maybe compile the files of this component.
    (dolist (file-record (component-file-records component))
      (when (compile-file-record-p file-record)
	(compile-file-record file-record)))
    (setf (compile-component-p component) nil)
    ;;
    ;; Mark the dependants of this component as needing to be compiled, if indeed we compiled something.
    (when compile-dependants
      (let* ((deps (component-dependants component)))
	(when deps
	  (push-history :MUST-COMPILE deps)
	  (dolist (dependant deps)
	    (setf (compile-component-p dependant) t))))))
  component)


(defvar *systems-being-processed* nil
  "Bound to a stack of the systems passed to PROCESS-SYSTEM, so that we can detect circular dependancies.")

(defun process-system (system &optional (children-too t) (parents-too t))
  "Load (perhaps compiling) the SYSTEM, if necessary."
  (check-type system system)
  (when (system-forward-reference-p system)
    (error "A forward reference to the system ~A was never resolved."
	   (system-name system)))
  (when (member system *systems-being-processed* :test #'eq)
    (cerror "Simply proceed." "A circular system dependancy was detected at ~A~{ <- ~A~}."
	    (system-name system)
	    (mapcar #'system-name *systems-being-processed*)))
  ;;
  ;; First process the systems this one depends on.
  (when parents-too
    (dolist (sys (system-dependancies system))
      (process-system sys nil t)))
  ;;
  ;; Then process the components of this system.
  (let* ((*systems-being-processed* (cons system *systems-being-processed*)))
    (dolist (component (system-components system))
      (process-component component))
    )
  ;;
  ;; Then process sub-systems of this one.
  (when children-too
    (dolist (sys (system-dependants system))
      (process-system sys t nil)))
  system)

(defvar *components-being-processed* nil
  "Bound to a stack of the components passed to PROCESS-COMPONENT, so that we can detect circular dependancies.")

(defun process-component (component)
  "Load (perhaps compiling) the files in the COMPONENT, if necessary."
  (check-type component component)
  (when (component-forward-reference-p component)
    (error "A forward reference to the component ~A of ~A was never resolved."
	   (component-name component) (system-name (component-system component))))
  (when (member component *components-being-processed* :test #'eq)
    (cerror "Simply proceed." "A circular component dependancy was detected at ~A~{ <- ~A~}."
	    (component-name component)
	    (mapcar #'component-name *components-being-processed*)))
  (let* ((*components-being-processed* (cons component *components-being-processed*)))
    (unless (component-never-load-p component)
      (unless (component-binary-only component)
	(compile-component component))
      (dolist (file-record (component-file-records component))
	(when (load-file-record-p file-record)
	  (dolist (comp2 (component-dependancies component))
	    (process-component comp2))
	  (load-file-record file-record)))))
  component)

(defun report-load-count (&optional system-name completed-normally)
  "Print how many files were loaded and compiled.  Do this at the end."
  (format t "~&;;;~%;;; ~A~[~:;~:*~D file~:P compiled, ~]~D file~:P loaded.~%;;;~%"
	  (if system-name
	      (if completed-normally
		  (format nil "System ~A processed.  " system-name)
		  (format nil "Abnormal exit of processing of system ~A.  " system-name))
	      "")
	  *compile-count* *load-count*))


;;; Defining the systems - user interface stuff.

(defun keywordify (symbol)
  "Returns a symbol with the same name as SYMBOL, but in the KEYWORD package."
  (if (keywordp symbol)
      symbol
      (intern (string symbol) "KEYWORD")))

(defvar *all-systems* nil)

(defun lookup-system (name &optional forward-p)
  "Return a system called NAME.
  If there is no such system, and FORWARD-P is T, create and return a forward referenced system.
  Otherwise signal an error."
  (setq name (keywordify name))
  (or (find name *all-systems* :test #'eq :key #'system-name)
      (if forward-p
	  (let* ((s (make-system :name name :forward-reference-p t)))
	    (push s *all-systems*)
	    s)
	  (error "No system called ~A." name))))

(defun lookup-component (comp-descriptor &optional system forward-p)
  "COMP-DESCRIPTOR is either a keyword, the name of a component of SYSTEM, or is a list of (<system-name> <component-name>)
  where SYSTEM-NAME and COMPONENT-NAME are both keywords.
  If FORWARD-P is T and such a component does not exist, a FORWARD-REFERENCED-COMPONENT is created and
  returned.  Otherwise NIL is returned."
  (check-type system (or system null))
  (cond ((symbolp comp-descriptor)
	 (setq comp-descriptor (keywordify comp-descriptor))
	 (let* ((comp (find comp-descriptor (system-components system) :key #'component-name :test #'eq)))
	   (when (and forward-p (null comp))
	     (setq comp (make-component :name comp-descriptor :system system :forward-reference-p t))
	     (push comp (system-components system)))
	   comp))
	((and (consp comp-descriptor)
	      (= (length comp-descriptor) 2))
	 (let* ((system-name (keywordify (car comp-descriptor)))
		(comp-name (keywordify (cadr comp-descriptor)))
		(system (lookup-system system-name forward-p)))
	   (if system
	       (lookup-component comp-name system forward-p)
	       nil)))
	(t (error "~S is not a legal component descriptor." comp-descriptor))))


(defun directory-specified (path)
  "T if the pathname has meaningful information in its DIRECTORY slot."
  ;;
  ;; In Lucid lisp, even a completely unspecified pathname has :RELATIVE in its directory slot.
  ;;
  #+LUCID (let* ((d (pathname-directory path)))
	    (and d
		 (not (eq d :RELATIVE))
		 (not (eq d :UNSPECIFIC))
		 (not (and (consp d)
			   (= (length d) 1)
			   (eq (car d) :RELATIVE)))))
  ;;
  ;; Hopefully other Lisps will be like LISPMs and use NIL.
  ;;
  #-LUCID (and (pathname-directory path)
	       (not (eq (pathname-directory path) :UNSPECIFIC)))
  )

(defun device-specified (path)
  (and (pathname-device path)
       (not (eq (pathname-device path) :UNSPECIFIC))))

(defun version-specified (path)
  (and (pathname-version path)
       (not (eq (pathname-version path) :UNSPECIFIC))))


;;;  ( <component-name> <file-or-list-of-files> <component-dependancies> &OPTIONAL <keyword> )
;;;
(defun parse-component-list (list system)
  (check-type list list)
  (check-type system system)
  (when (> (length list) 4) (error "~S should have four or fewer elements." list))
  (let* ((name (first list))
	 (files (second list))
	 (depend (third list))
	 (keyword (fourth list))
	 )
    (unless (listp files) (setq files (list files)))
    (unless (listp depend) (setq depend (list depend)))
    (setq name (keywordify name))
    (let* ((old (lookup-component name system t))
	   (component nil)
	   (file-records nil))
      (cond ((component-forward-reference-p old)
	     (setq component old)
	     (setf (component-forward-reference-p old) nil))
	    (t
	     (cerror "Allow the redefinition." "Component ~A of ~A is being redefined." name (system-name system))
	     (setq component old)
	     (setf (component-dependancies component) nil)
	     (setf (component-file-records component) nil)))
      ;; We want this component to be at the front of the list, even if it is a forward reference.
      (setf (system-components system)
	    (cons component (delete component (system-components system) :test #'eq)))
      ;;
      ;; Deal with the keyword.
      ;;
      (setf (component-binary-only component) nil)
      (setf (component-never-load-p component) nil)
      (ecase keyword
	((NIL) nil)
	(:BINARY-ONLY (setf (component-binary-only component) t))
	((:NO-LOAD :NOLOAD :NEVER-LOAD)
	 (setf (component-never-load-p component) t)))
      
      (dolist (file files)
	;;
	;; First do some error-checking on the file-names - make sure they have provided a file NAME but not a
	;; TYPE, DIRECTORY, etc. (unless this is a NO-LOAD component, in which case they may have a TYPE).
	;; If everything is OK, then make a file record and install it on the current component.
	;;
	(check-type file string)
	(let* ((path (parse-namestring file nil nil))
	       (noload (component-never-load-p component)))
	  (when (or (device-specified path)
		    (version-specified path)
		    (directory-specified path)
		    (and (pathname-type path) (not noload)))
	    (if noload
		(cerror "Use only the NAME and TYPE components of this pathname."
			"~S is parsable to a pathname which is too fully qualified.~%~
                         The file-name of a NO-LOAD component must be without extension, directory, device, or version."
			file)
		(cerror "Use only the NAME component of this pathname."
			"~S is parsable to a pathname which is too fully qualified.~%~
                         The file-name of a component must be without extension, directory, type, device, or version."
			file))
	    (if noload
		(setq file (make-pathname :name (pathname-name path) :type (pathname-type path)))
		(setq file (pathname-name path))))
	  (when noload (setq file (pathname file)))
	  (if file
	      (push (make-file-record :name file :component component) file-records)
	      (cerror "Ignore this bogus file." "The pathname ~S does not have a NAME component." path))))
      (setq file-records (nreverse file-records))
      (setf (component-file-records component) file-records)
      (dolist (dependancy depend)
	(let* ((comp (lookup-component dependancy system t)))
	  (push comp (component-dependancies component))
	  (push component (component-dependants comp))
	  ))
      (setf (component-dependancies component) (nreverse (component-dependancies component)))
      
      component)))



(defun define-system-1 (name fields)
  "Internal function of DEFINE-SYSTEM."
  (check-type name symbol)
  (setq name (keywordify name))
  
  #+EXPLORER (sys:record-source-file-name name 'mksys:system t)
  
  (let* ((system-object (lookup-system name t))
	 ;;
	 ;; MANDITORY-COMPONENTS is non-NIL only if this system is being redefined.
	 ;; It is a list of COMPONENT objects which existed in the previous incarnation of this system, and which
	 ;; must exist this time as well.  They must exist because there are still systems around which depend on them.
	 ;; When a component is parsed for this system, and there is a component of the same name on MANDITORY-COMPONENTS,
	 ;; the new component's DEPENDANTS list is set to that of the old component, so we don't lose these dependancies.
	 ;; When this happens, we remove the old component from the MANDITORY-COMPONENTS list.
	 ;; If MANDITORY-COMPONENTS is non-NIL when the entire system has been processed, then the system has changed
	 ;; incompatibly.
	 ;;
	 (manditory-components nil))
    (cond ((system-forward-reference-p system-object)
	   (push-history :DEFINE name)
	   (setf (system-forward-reference-p system-object) nil))
	  (t (warn "Redefining system ~A." name)
	     (push-history :REDEFINE name)
	     (setf (system-package system-object) nil)
	     (setf (system-directory system-object) nil)
	     ;;
	     ;; Note: do NOT set SYSTEM-DEPENDANCIES to NIL - this is wrong, because the dependancies list was set by
	     ;; the DEFINE-SYSTEMs of each of the systems on that list.  We are not redefining those here, so hands off.
	     ;;
	     ;; (setf (system-dependancies system-object) nil) ;; <-- don't do this!
	     
	     ;; Remove old dependants, that is, subsystems declared by the previous definition of this system.
	     ;; This entails removing the SYSTEM-OBJECT from the DEPENDANCIES list of all systems which *used* to be on
	     ;; this system's DEPENDANTS list.
	     ;; Also do a parallel thing for each of the COMPONENTs of this system.
	     ;;
	     (dolist (child-system (system-dependants system-object))
	       (setf (system-dependancies child-system)
		     (delete system-object (system-dependancies child-system) :test #'eq)))
	     (dolist (component (system-components system-object))
	       ;;
	       ;; If system A is being redefined, and component B1 of system B depended on component A1 of A, then
	       ;; the redefinition of system A must contain a definition of component A1.
	       ;;
	       (when (and (component-dependants component)
			  (some #'(lambda (comp) (not (eq system-object (component-system comp))))
				(component-dependants component)))
		 (push component manditory-components))
	       ;;
	       ;; Remove old dependancies.
	       ;;
	       (dolist (parent-component (component-dependancies component))
		 (setf (component-dependants parent-component)
		       (delete component (component-dependants parent-component))))
	       )
	     ;; This must be after the above DOLISTs.
	     (setf (system-dependants system-object) nil)
	     (setf (system-components system-object) nil)
	     ))
    (dolist (field fields)
      (case (car field)
	
	((:DEFAULT-PATHNAME :DEFAULT-DIRECTORY :PATHNAME :DIRECTORY)
	 (when (system-directory system-object)
	   (error "Duplicate DIRECTORY fields in DEFINE-SYSTEM."))
	 (unless (= 2 (length field)) (error "Invalid DIRECTORY field in DEFINE-SYSTEM."))
	 (let* ((path (second field)))
	   (etypecase path
	     ((or string pathname) (setq path (pathname path)))
	     (list                 (setq path (make-pathname :directory path :defaults nil))))
	   (setq path (make-pathname :name :UNSPECIFIC :type :UNSPECIFIC :version :UNSPECIFIC
				     :defaults path))
	   (setf (system-directory system-object) path)
	   ))
	
	((:DEFAULT-PACKAGE :PACKAGE)
	 (when (system-package system-object)
	   (error "Duplicate PACKAGE fields in DEFINE-SYSTEM."))
	 (unless (= 2 (length field)) (error "Invalid PACKAGE field in DEFINE-SYSTEM."))
	 (let* ((pack (second field)))
	   (etypecase pack
	     (string  (setf (system-package system-object)
			    (or (find-package pack) (error "No such package as ~S." pack))))
	     (package (setf (system-package system-object) pack)))))
	
	((:COMPONENT-SYSTEMS :SUBSYSTEMS)
	 (let* (subs)
	   (if (and (= 2 (length field))
		    (consp (second field)))
	       (setq subs (second field))
	       (setq subs (cdr field)))
	   (dolist (name subs)
	     (setq name (keywordify name))
	     (let* ((subsystem (lookup-system name t)))
	       ;; Make parent and child point to each other.
	       (pushnew system-object (system-dependancies subsystem) :test #'eq)
	       (push subsystem (system-dependants system-object))))
	   (setf (system-dependants system-object) (nreverse (system-dependants system-object)))))
	
	(t (when (keywordp (car field))
	     (error "~S is not a known keyword in DEFINE-SYSTEM." (car field)))
	   (let* ((new-component (parse-component-list field system-object))  ; This installs the component on the system.
		  (old-component (find (component-name new-component)
				       manditory-components
				       :key #'component-name :test #'eq)))
	     (when old-component
	       (push-history :PATCH-DEPENDANCIES (list* system-object new-component (component-dependants old-component)))
	       (setf (component-dependants new-component)
		     (component-dependants old-component))
	       (setq manditory-components (delete old-component manditory-components))
	       )))))
    ;;
    ;; See the comment up by the LET binding of MANDITORY-COMPONENTS for a rationale of this next bit.
    (when manditory-components
      ;; Blast the name so that it is recognisably bogus.
      (let* ((bogus-name (gentemp "OBSOLETE-SYSTEM-" "KEYWORD")))
	(setf (system-name system-object) bogus-name)
	(push-history :NUKE (cons name bogus-name)))
      (error "The redefinition of system ~A is incompatible with the current state.~%~
              The previous definition contained components on which components of other systems still depend.~%~
              The following components are missing:~{~%  ~A~}~%~
              To recover from this, you must reload all of your makefiles."
	     name
	     (mapcar #'component-name manditory-components)))
    ;;
    ;; Things were PUSHed, so we reverse the list at the end.
    (setf (system-components system-object) (nreverse (system-components system-object)))
    (assert (not (null (system-directory system-object)))
	    ((system-directory system-object))
	    "No directory was supplied in system ~A." name)
    system-object))




(defun munch-fields (list)
  "Return a list that when evaluated will evaluate elements in each sublist iff the sublist begins with a keyword."
  (cons 'LIST
	(mapcar #'(lambda (x)
		    (if (keywordp (car x))
			`(list ,@x)
			(list 'QUOTE x)))
		list)))

(defmacro define-system (name &body fields)
  "Syntax: DEFINE-SYSTEM <name> [ <component> | <parameter> ]*
  
  PARAMETERs are lists of a keyword and a value.  The value is evaluated.
  Legal keywords:
    :PACKAGE, :DEFAULT-PACKAGE        What *package* is to be bound to before
 			 	      loads.
    :COMPONENT-SYSTEMS, :SUBSYSTEMS   The names of other systems defined by
                                      DEFINE-SYSTEM which this includes.
    :DEFAULT-PATHNAME, :DEFAULT-DIRECTORY, :PATHNAME, DIRECTORY
   				      The directory which this system lives in.
				      Only one directory per system.
  
  COMPONENTs are lists of  ( <name> <file-or-list> <dependancies> )
    NAME is the name of this component, unique to this system.
    FILE-OR-LIST is a string or list of strings, the files which comprise
     this component.  No extensions - this is the NAME field of a pathname.
  DEPENDANCIES is a list of component names which this component depends on.
    What is \"depend\" you ask?  If a component is changed, then all components
    which depend on it must be recompiled.  So if you have a file which uses
    macros defined somewhere else, it should depend on it.  If you have a file
    which uses functions defined somewhere else, it need not depend on it.
  
    Elements of this list may be symbols, referring to components of this
    system, or list of ( <system-name> <component-name> ) referring to
    components of other systems.
  
    Component specification lists are never evaluated, though parameter
    specifications are.

  All NAMEs are symbols, whose package is irrelevant.  Only print-names are
  compared."
  `(define-system-1 ',name ,(munch-fields fields)))


#+(or LUCID HEMLOCK) (let* ((pack (or (find-package "ED")	; The editor might not be loaded - play it safe.
				      (find-package "EDITOR")
				      (find-package "HEMLOCK")))
			    (sym (and pack (find-symbol "DEFINDENT" pack))))
		       (when sym
			 (funcall sym "define-system" 1)
			 (funcall sym "mksys:define-system" 1)
			 (funcall sym "mksys::define-system" 1)))



;;; Implementation Dependent bashing.


;;; In Lucid 2.1, there is no nice, simple way to cause COMPILE-FILE to NOT print messages every time it opens an
;;; input file and closes an output file.  We want this so that output from MKSYS looks the same across systems.
;;; BASH it.

#+(and LUCID (not LCL3.0) (not LCL4.0) (not LCL5.0) (not LCL6.0))  ; If this is Lucid 2.1....
(progn
 (defvar sys::*compile-file-verbose* t "If this is NIL, COMPILE-FILE is quiet about opening and closing files.")
 
 (sys:defadvice (lucid::compile-file-internal message-hack) (&rest args)
   (if sys::*compile-file-verbose*
       (sys:apply-advice-continue args)
       (block NIL
	 (let* ((real-errout *error-output*))
	   (declare (special real-errout))
	   (with-output-to-string (*error-output*)
	     (return (sys::apply-advice-continue args)))))))
 
 (sys:defadvice (lucid::compile-file-loop message-hack) (&rest args)
   (if sys::*compile-file-verbose*
       (sys::apply-advice-continue args)
       (block NIL
	 (locally (declare (special real-errout))
	   (let* ((*error-output* real-errout))
	     (return (sys::apply-advice-continue args)))))))
 )


;;; SITE-COMPILER-WRAPPER expands differently in different implementations.
;;; It is used to do system dependent things wrapping the compilation of a system.

(defmacro site-compiler-wrapper (&body body)
  "Do implementation dependant compiler bashing."
  #+EXPLORER `(compiler:compiler-warnings-context-bind ,@body)

  #+(and LUCID (not LCL3.0) (not LCL4.0) (not LCL5.0) (not LCL6.0))  ; if this is Lucid 2.1...
  `(progn ; utils:delaying-undef-warnings ; This macro is defined in "~jwz/lisp/utils/delay-undef-warnings.lisp"
     (let* ((sys::*load-if-source-only* :load-source)
	    (sys::*load-if-source-newer* nil)
	    (sys::*compile-file-verbose* nil))
       ,@body))
  
  #+(or LCL3.0 LCL4.0 LCL5.0 LCL6.0)  ; if this is Lucid, but *not* Lucid 2.1...
  `(let* ((sys::*load-if-source-newer* nil)
	  (sys::*load-if-source-only* nil))
     ,@body)
  
  #-(or EXPLORER LUCID) `(progn ,@body)
  )


;;; Loading Systems


(defun mark-system-must-compile (system)
  "Recursively mark the components of SYSTEM and all of its subsystems as needing to be compiled."
  (let* ((components (system-components system)))
    (when components
      (push-history :MUST-COMPILE components)
      (dolist (component components)
	(unless (or (component-never-load-p component)
		    (component-binary-only component))
	  (setf (compile-component-p component) t)))))
  (dolist (sys2 (system-dependants system))
    (mark-system-must-compile sys2))
  nil)


(defun mksys-1 (name compile verbosity print-only interactive-p)
  "NAME is a symbol, the name of a system.
  COMPILE is whether we are allowed to compile files, or :ALL meaning all files should be recompiled.
  VERBOSITY is whether we should announce loading and compilation.
  INTERACTIVE-P is whether we should ask the user before each load or compile."
  (check-type name keyword)
  (check-type compile (member t nil :all))
  (let* ((*mksys-verbosity* verbosity)
	 (*mksys-interactive-p* interactive-p)
	 (*mksys-print-only* print-only)
	 (*mksys-never-compile* (eq compile nil))
	 (*load-count* 0)
	 (*compile-count* 0))
    (history-mksys-tag)
    (invalidate-file-dates name)
    (let* ((system (lookup-system name nil))
	   (completed-normally nil))
      (when (eq compile :all) (mark-system-must-compile system))
      (unwind-protect
	  (site-compiler-wrapper
	    (process-system system)
	    (setq completed-normally t))
	(report-load-count (system-name system) completed-normally))
      (system-name system))))


(defmacro mksys (name &optional mode print-only query (verbose t))
  "Make the system called NAME.
 NAME is a symbol in any package; it is not evaluated.
 MODE is one of:
    NIL           Compile those files that need to be compiled, and then load the system.  This is the default.
   :LOAD-ONLY     Don't compile any files, just load the system.
   :RECOMPILE     Compile ALL files, and then load the system.
   :EDIT          Place all of the files in this system in the editor.
   :COPY          Copy this system to another directory, which is prompted for.
   :COPY-CHANGED  Just like :COPY, but doesn't copy files if there is already an up-to-date version in the destination.

 If PRINT-ONLY is T, then no conpilation or loading will occur, but you will be shown what WOULD be compiled or loaded.
 If QUERY is T, then the user will be queried before each load or compile.  Be aware that by not answering Yes to every
      query, you can produce a nonconsistent system.
 VERBOSE controls how much noise is generated by MKSYS; it has no effect on noise generated by compilation.  T = loud.
  "
  (when (and (consp name)
	     (eq (car name) 'QUOTE)
	     (= (length name) 2))
    (setq name (second name))
    (warn "assuming that by (MKSYS '~A) you really meant (MKSYS ~A)." name name))
  (case mode
    (:EDIT
     `(edit-system ,(keywordify name)
		   :subsystems-too nil :parent-systems-too nil
		   :print-only ',print-only
		   :verbose ',verbose
		   :interactive-p ',query))
    (:COPY
     `(copy-system-files ,(keywordify name) nil
		   :print-only ',print-only
		   :verbose ',verbose
		   :interactive-p ',query))
    (:COPY-CHANGED
     `(copy-system-files ,(keywordify name) nil
		   :print-only ',print-only
		   :verbose ',verbose
		   :interactive-p ',query
		   :changed-files-only t))
    (t
     `(mksys-1 ,(keywordify name)
	       ,(case mode
		  ((nil)      t)
		  (:load-only nil)
		  (:recompile :all)
		  (t (error "~S is not one of NIL, :LOAD-ONLY, :RECOMPILE, or :EDIT." mode)))
	       ',verbose
	       ',print-only
	       ',query))))

(defun unload-system (system-name)
  "Mark all of the component files of the SYSTEM and its subsystems as never having been loaded."
  (setq system-name (keywordify system-name))
  (let* ((system (lookup-system system-name nil)))
    (dolist (comp (system-components system))
      (dolist (fr (component-file-records comp))
	(setf (file-record-load-date fr) nil)))
    (dolist (sys2 (system-dependants system))
      (unload-system (system-name sys2))))
  system-name)

(defun invalidate-file-dates (system-name)
  "Destroy all cached information about the file dates (but not the load dates) of SYSTEM."
  (setq system-name (keywordify system-name))
  (let* ((system (lookup-system system-name nil)))
    (dolist (comp (system-components system))
      (dolist (fr (component-file-records comp))
	(setf (file-record-source-file-date fr) nil
	      (file-record-binary-file-date fr) nil
	      (file-record-source-truename fr) nil
	      (file-record-binary-truename fr) nil)))
    (dolist (sys2 (system-dependants system))
      (invalidate-file-dates (system-name sys2))))
  system-name)



;;; Loading an entire system into the editor.

#+EXPLORER
(defun edit-system (system-name &key (subsystems-too t) (parent-systems-too nil) print-only interactive-p (verbose t)
		    zmacs-frame dont-select)
  "Load all of the files composing the system into ZMACS, and then select the editor.
  If SUBSYSTEMS-TOO is non-NIL, then the files composing all subsystems of this system will be visited as well.
  If PARENT-SYSTEMS-TOO is non-NIL, then all systems of which this one is a subsystem will be visited as well.
  ZMACS-FRAME, if non-NIL, should be a window of type ZWEI:ZMACS-FRAME.
  If DONT-SELECT is non-NIL, then the window will not be selected after the files are queued for visiting."
  (let ((*mksys-print-only* print-only)
	(*mksys-interactive-p* interactive-p)
	(*mksys-verbosity* verbose)
	(system (if (system-p system-name)
		    system-name
		    (lookup-system system-name)))
	(sheet (or zmacs-frame (zwei:find-or-create-idle-zmacs-window))))
    (when parent-systems-too
      (dolist (sys (system-dependancies system))
	(edit-system sys :subsystems-too nil :parent-systems-too t :zmacs-frame sheet :dont-select t
		     :print-only print-only :interactive-p interactive-p
		     )))
    (dolist (component (system-components system))
      (dolist (rec (component-file-records component))
	(let* ((file (file-record-source-pathname rec)))
	  (if (probe-file file)
	      (if print-only
		  (format t "~&;;; Would have edited ~A.~%" file)
		  (when (or (not interactive-p)
			    (y-or-n-p ";;; Edit the file ~A ?" file))
		    (ticl:send sheet :force-kbd-input `(:execute zwei:edit-thing ,file))))
	      (warn "file ~A does not exist.~%" file)))))
    (when subsystems-too
      (dolist (sys (system-dependants system))
	(edit-system sys  :subsystems-too t :parent-systems-too nil :zmacs-frame sheet :dont-select t
		     :print-only print-only :interactive-p interactive-p
		     )))
    (unless (or dont-select print-only)
      (ticl:send sheet :select)
      (w:await-window-exposure))
    system))


#-EXPLORER
(defun edit-system (system-name &key (subsystems-too t) (parent-systems-too nil) print-only interactive-p (verbose t))
  "Call the function ED on each of the files composing the system.
  If SUBSYSTEMS-TOO is non-NIL, then the files composing all subsystems of this system will be edited as well.
  If PARENT-SYSTEMS-TOO is non-NIL, then all systems of which this one is a subsystem will be edited as well."
  (let ((*mksys-print-only* print-only)
	(*mksys-interactive-p* interactive-p)
	(*mksys-verbosity* verbose)
	(system (if (system-p system-name)
		    system-name
		    (lookup-system system-name))))
    (when parent-systems-too
      (dolist (sys (system-dependancies system))
	(edit-system sys :subsystems-too nil :parent-systems-too t 
		         :print-only print-only :interactive-p interactive-p)))
    (dolist (component (system-components system))
      (dolist (rec (component-file-records component))
	(let* ((file (file-record-source-pathname rec)))
	  (if (probe-file file)
	      (if print-only
		  (format t "~&;;; Would have edited ~A.~%" file)
		  (when (or (not interactive-p)
			    (y-or-n-p ";;; Edit the file ~A ?" file))
		    (ed file)))
	      (warn "file ~A does not exist.~%" file)))))
    (when subsystems-too
      (dolist (sys (system-dependants system))
	(edit-system sys :subsystems-too t :parent-systems-too nil 
		         :print-only print-only :interactive-p interactive-p)))
    system))



;;; Copying a system to a different directory.


;;; This function goes to pains to follow the following algorithm for copying files on versioned file-systems:
;;;
;;; If the source file does not exist, signal an error.
;;; If the source file exists, and there is a file of the same name in the destination directory, and the version
;;;  number of the file in the destination is greater than or equal to the version number of the source, then copy
;;;  the source file to the destination directory with :NEW-VERSION.
;;; If there is no corresponding file in the destination directory, or if the file in the destination directory has
;;;  a smaller version number than the source file, then make the copied file have the same version number as the source.
;;;
;;; This causes copied systems to have the same version numbers as the place they were copied from, except when doing so
;;; would cause files to be written which have version numbers smaller than existing files - you don't want to write file
;;; version #5 when there is already a version #9...
;;;


(defun mksys-copy-file (file destination &optional only-if-changed)
  "This how we copy files on different implementations.  Knows about *MKSYS-PRINT-ONLY* and *MKSYS-INTERACTIVE-P*.
  If ONLY-IF-CHANGED is non-NIL, then the file will only be copied if the source is newer than the destination."
  (setq destination (make-pathname :defaults destination
				   :name (pathname-name file)
				   :type (pathname-type file)
				   :version :newest))
  
  (let* ((ok t)
	 source-newer-p source-truename dest-truename)
    
    (if only-if-changed
	(multiple-value-setq (source-newer-p source-truename dest-truename) (pathname-newer-p file destination))
	(setq source-newer-p t
	      source-truename (probe-file file)
	      dest-truename (probe-file destination)))
    
    (unless source-truename (error "The file ~A does not exist." file))
    (cond ((and dest-truename
		(file-version-> source-truename dest-truename))
	   ;;
	   ;; There is a file in the destination already, and it has a higher version number.
	   ;; Make the destination have the version :NEWEST.
	   ;;
	   (setq dest-truename (make-pathname :defaults dest-truename :version :newest)))
	  
	  (dest-truename
	   ;;
	   ;; There is a file in the destination already, and it has a lower or equal version number, or we are
	   ;; on a filesystem which doesn't use version numbers.
	   ;; Make the version of the destination be the same as that of the source.
	   ;;
	   (setq dest-truename (make-pathname :defaults dest-truename :version (pathname-version source-truename))))

	  (t
	   ;;
	   ;; The destination file doesn't yet exist.  Make it's version be the same as the source.
	   ;;
	   (setq dest-truename (make-pathname :defaults destination :version (pathname-version source-truename)))))
    
    (when source-newer-p
      (if *mksys-interactive-p*
	  (setq ok (y-or-n-p ";;; Copy ~A to ~A ?" source-truename (or dest-truename destination)))
	  (format t "~&;;; Copying ~A to ~A..."    source-truename (or dest-truename destination))))
    
    (cond ((and ok source-newer-p)
	   (unless *mksys-print-only*
	     
	     #+EXPLORER (multiple-value-bind (ignore ignore tr) (ticl:copy-file source-truename dest-truename :verbose nil)
			  (setq dest-truename (if (consp tr)
						  (car tr)
						  tr))
			  (when (eh:errorp tr) (eh:signal-condition tr)))
	     
	     #+LUCID    (if (lucid::copy-file source-truename dest-truename)
			    (error "Error copying ~A to ~A." source-truename dest-truename)
			    nil)
	     
	     #-(or EXPLORER LUCID) (error "Don't know how to copy files in this Lisp.")
	     )
	   (cond (*mksys-print-only* (format t " print-only.~%"))
		 ((numberp (pathname-version dest-truename))
		  (format t " copied - #~D.~%" (pathname-version dest-truename)))
		 (t
		  (format t " copied.~%")))
	   dest-truename)
	  
	  ((not ok) ; Not OK means the user has been asked and said "no."
	   (format t "~&;;; ~A not copied.~%" file))

	  (t nil))))


(defun copy-system-files (system-name destination-directory
			  &rest keys &key (subsystems-too t) (source t) (binary t)
			                  print-only interactive-p (verbose t)
					  changed-files-only)
  "Copy the system to another place.  If DESTINATION-DIRECTORY is NIL, the user will be prompted for it."
  (let ((*mksys-print-only* print-only)
	(*mksys-interactive-p* interactive-p)
	(*mksys-verbosity* verbose)
	(system (if (system-p system-name)
		    system-name
		    (lookup-system system-name))))
    ;;
    ;; If there are no subsystems, don't mention subsystems when prompting for dest directory.
    ;;
    (unless (system-dependants system) (setq subsystems-too nil))
    
    (unless destination-directory
      (format t "~&~%Type the name of the directory to which to copy the ~A system~
                 ~:[~;, and the ~%directories of its subsystems~]:  "
		system-name subsystems-too)
      (setq destination-directory (merge-pathnames (pathname (read-line))))
      (fresh-line) (terpri))
    ;;
    ;; Copy the files of this system.
    ;;
    (dolist (component (system-components system))
      (dolist (rec (component-file-records component))
	(let* ((source-file (file-record-source-pathname rec))
	       (binary-file (file-record-binary-pathname rec))
	       (binary-only (component-binary-only component))
	       (copy-source-p (and source (not binary-only)))
	       (copy-binary-p (and binary (not (equalp source-file binary-file))))
	       )
	  (when copy-source-p (mksys-copy-file source-file destination-directory changed-files-only))
	  (when copy-binary-p (mksys-copy-file binary-file destination-directory changed-files-only))
	  )))
    ;;
    ;; Here's where it gets complicated.
    ;;
    ;; To be able to copy subsystems too, the subsystems must lie in subdirectories of the parent system's directory -
    ;;
    ;; as in    system FOO     LM:NEW.FOO;
    ;;          system BAR     LM:NEW.FOO.BAR;
    ;;
    ;;  or      system BAR     LM:NEW.FOO.NEWER.BAR;
    ;;
    ;; We look at the difference between the directories of the parent-system and the child-system -
    ;; in the above example, it would be ("BAR") or ("NEWER" "BAR").
    ;; We signal an error if the subsystem is not under the system in the directory structure.
    ;;
    ;; We construct the destination directory for the subsystem by appending the above-gleaned difference to the
    ;; end of the passed-in destionation directory.  So, if we were copying the system FOO to the directory
    ;; "LM:OLD.JULY4.FOO;", then the system BAR would be copied to "LM:OLD.JULY4.FOO.BAR" or "LM:OLD.JULY4.FOO.NEWER.BAR".
    ;;
    ;; This code should work in all implementations of Lisp - it makes no TI-specific assumptions.
    ;;
    (when subsystems-too
      (let* ((parent-source-dir (pathname-directory (system-directory system)))
	     (parent-dest-dir (pathname-directory destination-directory)))
	(dolist (subsystem (system-dependants system))
	  (let* ((source-dir (pathname-directory (system-directory subsystem)))
		 (overlap '())) ; A list of the directories which the parent and child share.
	    ;;
	    ;; Build the list OVERLAP.  Iterate in parallel down the parent and child directory lists, and
	    ;; bug out at the first mismatch.
	    ;;
	    (do* ((parent-rest parent-source-dir (cdr parent-rest))
		  (child-rest source-dir (cdr child-rest)))
		 ((or (null parent-rest) (null child-rest))
		  (when parent-rest
		    ;;
		    ;; If the child directory ran out before the parent, then one is above the other.
		    ;;
		    (error
		      "While copying system ~A:  ~A is higher than ~A in the directory structure.  I can't cope with that."
		      (system-name subsystem)
		      (directory-namestring (system-directory subsystem))
		      (directory-namestring (system-directory system)))))
	      (if (equal (car parent-rest) (car child-rest))
		  (push (car parent-rest) overlap)
		  (return)))
	    
	    (cond ((not overlap)
		   ;;
		   ;; If OVERLAP is NIL, then the directories are completely discontinuous.
		   ;; To copy this sub-system, we must re-prompt.
		   ;;
		   (cerror
		     "Enter the pathname to which to copy the ~A system.~*~*"
		     "While copying the ~A system: the directories ~A and ~A have no common parent.  I can't cope with that."
		     (system-name subsystem)
		     (directory-namestring (system-directory system))
		     (directory-namestring (system-directory subsystem)))
		   (copy-system-files (system-name subsystem) nil
				      :subsystems-too subsystems-too
				      :source source :binary binary
				      :print-only print-only :interactive-p interactive-p :verbose verbose))
		  (t
		   ;;
		   ;; Otherwise, it is possible to surmise where the new directory should be.
		   ;;
		   (let* ((subsystem-directory-difference (subseq source-dir (length overlap)))
			  (dest-dir (append parent-dest-dir subsystem-directory-difference))
			  (dest-path (make-pathname :defaults destination-directory :directory dest-dir)))
		     (apply 'COPY-SYSTEM-FILES subsystem dest-path keys))))))))
    system))


;;; Pretty-printing and describing systems.


(defun pprint-system (system-or-name &optional (stream *standard-output*))
  "Write out an EVALable DEFINE-SYSTEM form."
  (let* ((system (if (system-p system-or-name)
		     system-or-name
		     (lookup-system (keywordify system-or-name) nil))))
    (format stream "~&(~(~S~) ~A~%" 'define-system (system-name system))
    (let* ((space-me nil))
      (when (system-dependants system)
	(setq space-me t)
	(format stream "  (:subsystems~{ ~(:~A~)~})~%" (mapcar #'system-name (system-dependants system))))
      (when (system-package system)
	(setq space-me t)
	(format stream "  (:default-package ~S)~%" (package-name (system-package system))))
      (when (system-directory system)
	(setq space-me t)
	(format stream "  (:default-directory \"~A:~A\")~%"
		(pathname-host (system-directory system))
		(directory-namestring (system-directory system))))
      (when space-me (terpri stream)))
    (let* ((components (system-components system))
	   (max-name-length 0)
	   (max-file-names-length 0))
      (dolist (comp components)
	(setq max-name-length (max max-name-length (length (symbol-name (component-name comp)))))
	(let* ((this-l 0)
	       (frs (component-file-records comp)))
	  (dolist (fr frs)
	    (let* ((f (file-record-name fr))
		   (name (if (pathnamep f) (format nil "~A.~A" (pathname-name f) (pathname-type f)) f)))
	      (incf this-l (+ 2 (length name)))))        ; two for the bracketing double-quotes
	  (unless (= (length frs) 1) (incf this-l 2))    ; if not a singleton, the two for bracketing parens
	  (setq max-file-names-length
		(max max-file-names-length (+ this-l (1- (length frs)))))))
      (dolist (component components)
	(let* ((files (mapcar #'(lambda (x)
				  (let* ((f (file-record-name x)))
				    (if (pathnamep f) (format nil "~A.~A" (pathname-name f) (pathname-type f)) f)))
			      (component-file-records component)))
	       (*print-case* :downcase)
	       (dependancies (mapcar #'(lambda (comp)
					 (if (eq (component-system comp) system)
					     (component-name comp)
					     (list (system-name (component-system comp))
						   (component-name comp))))
				     (component-dependancies component))))
	  (format stream "  (~A~vT ~:S~vT ~:A"
		  (component-name component)
		  (+ max-name-length 3)     ; three for the initial two spaces and open paren.
		  (if (= (length files) 1)
		      (car files)
		      files)
		  (+ max-name-length 4 max-file-names-length)  ; four for above plus one space.
		  dependancies)
	  (cond ((component-binary-only component)  (princ "   :BINARY-ONLY"))
		((component-never-load-p component) (princ "   :NOLOAD")))
	  (format t ")~%")
	  )))
    (format stream "~&  )~%")
    system))


(defun pretty-time (ut)
  (when ut
    (multiple-value-bind (sec min hour day month year dotw) (decode-universal-time ut)
      (format nil "~2,'0D:~2,'0D:~2,'0D ~2,D-~A-~D (~A)"
	      hour min sec
	      day (svref #("Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec") (1- month))
	      (- year 1900)
	      (svref #("Monday" "Tuesday" "Wednesday" "Thursday" "Friday" "Saturday" "Sunday") dotw)))))


(defun describe-system (sys-or-name)
  (let* ((system (if (system-p sys-or-name)
		     sys-or-name
		     (or (lookup-system sys-or-name nil) (error "~S is not a system or the name of one." sys-or-name)))))
    (format t "~&The ~A system:~% Directory:  ~S~% Package:    ~S~%"
	    (system-name system)
	    (namestring (system-directory system))
	    (if (packagep (system-package system))
		(package-name (system-package system))
		(system-package system)))
    (let* ((dependants (system-dependants system))
	   (dependancies (system-dependancies system)))
      (when dependants
	(format t " Subsystems:~{~<~%            ~1:; ~A~>~^,~}~%"
		(mapcar #'system-name dependants)))
      (when dependancies
	(format t " Parents:   ~{~<~%              ~1:; ~A~>~^,~}~%"
		(mapcar #'system-name dependancies)))
      (let* ((comps (system-components system)))
	(cond (comps
	       (terpri)
	       (dolist (comp comps)
		 (describe-component comp)
		 (terpri)))
	      (t (format t " This system has no components.~%")))))
    system))

(defun describe-component (component)
  (format t "~&   The ~A component:" (component-name component))
  (when (component-binary-only component) (princ "   Binary Only."))
  (when (component-never-load-p component) (princ "   Never Loaded."))
  (terpri)
  (let* ((system (component-system component)))
    (let* ((dependants (component-dependants component))
	   (dependancies (component-dependancies component)))
      (flet ((pretty-name (dep)
	       (if (eq system (component-system dep))
		   (component-name dep)
		   (format nil "~A of ~A" (component-name dep) (system-name (component-system dep))))))
	(when dependancies
	  (format t "   Depends on:    ~{~<~%                  ~1:; ~A~>~^,~}~%"
		(mapcar #'pretty-name dependancies)))
	(when dependants
	  (format t "   Depended on by:~{~<~%                  ~1:; ~A~>~^,~}~%"
		  (mapcar #'pretty-name dependants)))))
    (when (component-needs-compiled component)
      (format t "   This component is marked for recompilation.~%"))
    )
  (let* ((frs (component-file-records component)))
    (cond (frs
	   (terpri)
	   (dolist (fr frs)
	     (describe-file-record fr))
	   (terpri))
	  (t (format t "   This component has no files.~%"))))
  component)

(defun describe-file-record (fr)
  (let* ((name (file-record-name fr))
	 (load (file-record-load-date fr))
	 (src (file-record-source-file-date fr))
	 (bin (file-record-binary-file-date fr))
	 (srcp (file-record-source-truename fr))
	 (binp (file-record-binary-truename fr)))
    (when (pathnamep name) (setq name (format nil "~A.~A" (pathname-name name) (pathname-type name))))
    (format t "~&      The ~S file:~%" name)
    (format t "        Cached Source Truename: ~A~%" (or srcp "none"))
    (format t "        Cached Binary Truename: ~A~%" (or binp "none"))
    (format t "        Source Date: ~A~%" (if src (pretty-time src) "unknown"))
    (format t "        Binary Date: ~A~%" (if bin (pretty-time bin) "unknown"))
    (format t "        Last Loaded: ~A~%" (if load (pretty-time load) "never"))
    (when (and src bin)
      (if (> src bin)
	  (format t "        The source is newer than the binary.~%")
	  (if (file-version-> srcp binp)
	      (format t "        The binary is newer than the source, but the source has a higher version number.~%")
	      (format t "        The binary is newer than the source.~%")))
      (when load
	(cond ((and (< load src) (< load bin))
	       (format t "        Both the source and binary have changed since the last load.~%"))
	      ((< load src)
	       (format t "        The source has changed since the last load.~%"))
	      ((< load bin)
	       (format t "        The binary has changed since the last load.~%")))))
    (when (file-record-needs-compiled fr)
      (format t "        This file is marked for recompilation.~%")))
  fr)

#+LISPM
(defun (:property SYSTEM sys:named-structure-invoke) (op &optional self &rest args)
  (cond ((eq op :which-operations) '(:print-self :describe))
	((eq op :describe)
	 (describe-system self))
	((eq op :print-self)
	 (%print-system self (first args) (second args)))
	(t (error "unknown message - ~S" op))))
